/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

// defines basic XML types used in CIM Objects as well as a
// type<->string converter class
#ifndef __CIM_XML_TYPES_H
#define __CIM_XML_TYPES_H

#include <string>
#include <sstream>
#include <openwsman/cpp/Exception.h>

using namespace std;
using namespace WsmanExceptionNamespace;

// definitions used for type conversion
#define CONST_10 10
#define CHAR_0 '0'
#define CHAR_9 '9'
#define CHAR_MINUS '-'
#define NUMBER_STRING_SIZE 12

namespace CimXMLTypesNamespace
{
	// define HexBinary type
	typedef string HexBinary;

	// define DateTime type
	typedef string DateTime;

// define Int64 type
#ifndef _WIN32
	typedef long long Int64;
	typedef unsigned long long Uint64;
#else
	typedef __int64 Int64;
	typedef unsigned __int64 Uint64;
#endif

	// define Base64 type
	class Base64
	{
	public:
		unsigned char* data;
		unsigned int len;

		Base64();
		Base64(const Base64& b);
		Base64(unsigned char *buffer, unsigned int blen);
		Base64(const string& input);
		~Base64();
		string ToString() const;
		void SetData(const string& input);
		void SetData(unsigned char *buffer, unsigned int blen);
		Base64& operator =(const Base64& other);
	};

	class XMLTypeConvertionException : public ::GeneralWsmanException
	{
	public:
		XMLTypeConvertionException(const char* message)
			:GeneralWsmanException(message, WSMAN_TYPE_CONVERSION_ERROR){}
	};

	class XMLTypeConverter
	{
	public:
		// template functions for converting Integers
		template<class T>
		static bool StringToUInteger(const string& str, T& t)
		{
			T res = 0, tmp;
			char c;
			for (unsigned int i=0; i < str.size(); i++)
			{
				c = str.at(i);
				if (c < CHAR_0 || c > CHAR_9) // illegal char
				{
					return false;
				}
				tmp = res;
				for(unsigned int j = 1; j < CONST_10; j++)
				{
					res += tmp;
					if(res < tmp) // overflow
					{
						return false;
					}
				}
				res += c - CHAR_0;
				if (res < tmp) // overflow
				{
					return false;
				}
			}
			t = res;
			return true;
		}

		template<class T>
		static bool StringToInteger(const string& str, T& t)
		{
			bool minus = str.size() > 0 && str.at(0) == CHAR_MINUS;
			T res = 0, tmp;
			char c;
			unsigned int i = minus ? 1 : 0;
			for (; i < str.size(); i++)
			{
				c = str.at(i);
				if (c < CHAR_0 || c > CHAR_9)
				{
					return false;
				}
				tmp = res;
				for(unsigned int j = 1; j < CONST_10; j++)
				{
					res += tmp;
					if(res < 0) // check for overflow
					{
						if(minus)
						{
							if(res-1 < 0)
							{
								return false;
							}
						}
						else
						{
							return false;
						}
					}
				}
				res += c - CHAR_0;
				if(res < 0) // check for overflow
				{
					if(minus)
					{
						if((T)(res -1) < 0 || i+1 < str.size())
						{
							return false;
						}
					}
					else
					{
						return false;
					}
				}
			}
			if(minus)
			{
				t = res < 0 ? res : -res;
			}
			else
			{
				t = res;
			}
			return true;
		}

		template<class T>
		static string UIntegerToString(const T& number)
		{
			char buffer[NUMBER_STRING_SIZE];
			buffer[NUMBER_STRING_SIZE - 1] = 0;
			T res = number, rem, div = CONST_10;
			unsigned int i = NUMBER_STRING_SIZE - 1;
			if(res==0)
		  {
			    buffer[--i] = CHAR_0;
		  }
			else
			{
					while(res > 0)
					{
						i --;
						rem = res % div;
						res /= div;
						buffer[i] = CHAR_0 + rem;
					}
			}
			return string(buffer + i);
		}

		template<class T>
		static string IntegerToString(const T& number)
		{
			char buffer[NUMBER_STRING_SIZE];
			bool minus = (number < 0);
			buffer[NUMBER_STRING_SIZE - 1] = 0;
			T res = minus ? -number : number, rem, div = CONST_10;
			unsigned int i = NUMBER_STRING_SIZE - 1;
			while(res > 0)
			{
				i --;
				rem = res % div;
				res /= div;
				buffer[i] = CHAR_0 + rem;
			}
			if(minus)
			{
				i --;
				buffer[i] = CHAR_MINUS;
			}
			return string(buffer + i);
		}

		// template functions using streams (beware of use with char!)
		template <class T>
		static string ToString(const T& t)
		{
			stringstream tmp;
			tmp << t;
			return tmp.str();
		}
		template <class T>
		static void FromString(const string& str, T& t)
		{
			stringstream tmp(str);
			tmp >> t;
		}

		// type conversion function declarations
		static void StringToType(const string& str, int& t);
		static string TypeToString(const int& str);
		static void StringToType(const string& str, unsigned int& t);
		static string TypeToString(const unsigned int& str);
		static void StringToType(const string& str, char& t);
		static string TypeToString(const char& str);
		static void StringToType(const string& str, unsigned char& t);
		static string TypeToString(const unsigned char& str);
		static void StringToType(const string& str, short& t);
		static string TypeToString(const short& str);
		static void StringToType(const string& str, unsigned short& t);
		static string TypeToString(const unsigned short& str);
		static void StringToType(const string& str, Int64& t);
		static string TypeToString(const Int64& str);
		static void StringToType(const string& str, Uint64& t);
		static string TypeToString(const Uint64& str);
		static void StringToType(const string& str, bool& t);
		static string TypeToString(const bool& str);
		static void StringToType(const string& str, Base64& t);
		static string TypeToString(const Base64& buf);
		static void StringToType(const string& str, double& t);
		static string TypeToString(const double& str);
		static void StringToType(const string& str, string& t);
		static string TypeToString(const string& str);
	};
}//CimClassNamespace
#endif
