#! /usr/bin/python3
# -*- coding: utf-8 -*-
from gi.repository import Unity
from unittest import TestCase
import imp
import os


class ResultSet(Unity.ResultSet):
    def __init__(self):
        Unity.ResultSet.__init__(self)
        self.results = []

    def do_add_result(self, result):
        self.results.append({'uri':result.uri,
                             'title':result.title,
                             'comment':result.comment,
                             'icon':result.icon_hint})

class ScopeTestCase(TestCase):
    def init_scope(self, scope_path):
        self.scope_module = imp.load_source('scope', scope_path)
        self.scope = self.scope_module.load_scope()

    def perform_query(self, query, filter_set = Unity.FilterSet.new()):
        result_set = ResultSet()
        ctx = Unity.SearchContext.create(query, 0, filter_set,
                                         None, result_set, None)
        s = self.scope.create_search_for_query(ctx)
        s.run()
        return result_set


class TestYelpScope(ScopeTestCase):

    def setUp(self):
        self.init_scope('src/unity_yelp_daemon.py')
        self.scope_module.YELP_EXECUTABLE = os.path.join(
            os.path.dirname(__file__), "fake-yelp.py")

    def tearDown(self):
        self.scope = None
        self.scope_module = None

    def test_questions_search(self):
        self.scope_module.HELP_DIR = 'tests/data/'
        expected_results = ["tests/data/C/sample_help/mock_yelp_file.page",
                            "Welcome to Ubuntu"]
        results = []
        result_set = self.perform_query('unity')
        self.assertEqual(len(result_set.results), 1)
        result = result_set.results[0]
        self.assertEqual(
            result['uri'], "tests/data/C/sample_help/mock_yelp_file.page")
        self.assertEqual(result['title'], "Welcome to Ubuntu")
        self.assertEqual(
            result['comment'], "A visual introduction to the Unity desktop.")
        self.assertEqual(result['icon'], 'sample_help')

    def test_questions_failing_search(self):
        self.scope_module.HELP_DIR = 'tests/data/'
        result_set = self.perform_query('upnriitnyt')
        self.assertEqual(len(result_set.results), 0)

    def test_preview(self):
        result = Unity.ScopeResult()
        result.uri = "tests/data/C/sample_help/mock_yelp_file.page"
        result.title = "Welcome to Ubuntu"
        result.comment = "A visual introduction to the Unity desktop."
        result.icon_hint = 'sample_help'

        previewer = self.scope.create_previewer(result, Unity.SearchMetadata())
        preview = previewer.run()
        self.assertEqual(preview.props.title, "Welcome to Ubuntu")
        self.assertEqual(preview.props.description_markup,
                         "A visual introduction to the Unity desktop.")
        self.assertNotEqual(preview.props.image, None)
        self.assertEqual(preview.props.image.get_names(), ['sample_help'])

    def test_activation(self):
        result = Unity.ScopeResult()
        result.uri = "tests/data/C/sample_help/mock_yelp_file.page"
        activation = self.scope.activate(result, Unity.SearchMetadata(), None)
        self.assertEqual(activation.props.goto_uri, None)
        self.assertEqual(activation.props.handled, Unity.HandledType.HIDE_DASH)


if __name__ == '__main__':
    unittest.main()
