/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2017 Martin Brehm
                  2012-2017 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/


#ifndef CC_ENGINE_H
#define CC_ENGINE_H


// This must always be the first include directive
#include "config.h"

#include "cc_tools.h"
#include <stdio.h>
#include "cc_cubeframe.h"
#include "cc_hufftree.h"


extern std::vector<int> gc_iaAtomOrd;


class CCCEngine {
public:
	CCCEngine();
	~CCCEngine();

	bool MergeBQB(const char *outfile, const std::vector<const char*> &infile, bool verbose);

	bool SplitBQB(const char *infile, const char *outbase, int splitlength, int steps, bool verbose);

	void ResetStatistics();

	void BeginStatistics();

	void EndStatistics(bool n, bool c);

	bool CompressFile(
		const char *inp,
		const char *outp,
		int block,
		int tables,
		bool opttables,
		bool coderun,
		bool bw,
		bool mtf,
		bool preopt,
		int maxchunk,
		bool verbose
	);
	
	bool DecompressFile(const char *inp, const char *outp, const char *ref, bool verbose);

	void CompressString(const char *s, CBitSet *bs, bool verbose);

	void DecompressString(CBitSet *bs, char **s, bool verbose);

	bool CompressCube(
		const char *inp,
		const char *outp,
		const char *ref,
		int start,
		int steps,
		int stride,
		int corder,
		bool optcorder,
		int aorder,
		bool optaorder,
		int eps,
		int csigni,
		int asigni,
		int csplit,
		int asplit,
		int cblock,
		int ablock,
		int ctables,
		bool optctables,
		bool cpreopt,
		int cmaxchunk,
		int atables,
		bool optatables,
		bool hilbert,
		bool ccoderun,
		bool cbw,
		bool cmtf,
		bool acoderun,
		bool abw,
		bool amtf,
		bool apreopt,
		int amaxchunk,
		bool comment,
		bool compare,
		bool dummyread,
		bool verbose
	);


	bool DecompressCube(const char *inp, const char *outp, const char *readref, int steps, int stride, bool verbose);


	bool CompressXYZ(
		const char *inp, 
		const char *outp, 
		const char *ref,
		int start,
		int steps, 
		int stride, 
		int order, 
		bool optorder,
		int signi, 
		int split, 
		int block, 
		int tables, 
		bool opttables, 
		bool coderun, 
		bool bw, 
		bool mtf,
		bool preopt,
		bool comment, 
		bool compare,
		int maxchunk, 
		bool verbose
	);


	bool DecompressXYZ(const char *inp, const char *outp, const char *readref, int steps, int stride, bool verbose);

	double PredictFrame(int order, int eps, bool hilbert, bool coderun, bool bwmtf, int blocklength, int tables, bool verbose);

	bool CubeToIntegerArray(std::vector<int> &outp, int order, int eps, int signi, int split, bool hilbert, bool verbose);

	bool IntegerArrayToCube(std::vector<int> &inp, int order, int eps, int signi, int split, bool hilbert, bool verbose, bool second);

	void PerformDCT(CCubeFrame *cfr);
	void FFTTest();

	bool CompressCubeFrame(
		CBitSet *bs,
		int corder,
		int eps,
		int csigni,
		int csplit,
		int cblock,
		int ctables,
		bool opttables,
		bool hilbert,
		bool coderun,
		bool bw,
		bool mtf,
		bool preopt,
		bool atominfo,
		int maxchunk,
		bool verbose
	);

	bool DecompressCubeFrame(CBitSet *bs, bool verbose, bool second);

	bool CompressAtomFrame(
		CBitSet *bs,
		int order,
		int signi,
		int split,
		int block,
		int tables,
		bool opttables,
		bool coderun,
		bool bw,
		bool mtf,
		bool preopt,
		bool atominfo,
		bool comment,
		int maxchunk,
		bool verbose
	);

	bool DecompressAtomFrame(CBitSet *bs, bool verbose, bool second);

	bool PrepareDecompressCube(bool verbose);

	bool DecompressCubeStep(const std::vector<unsigned char> *inp, bool verbose, bool skipvol=false);


	void ExportCubeHeader(CBitSet *bs, int order, bool verbose);
	void ImportCubeHeader(CBitSet *bs, bool verbose, bool second);

	void AtomsToIntegerArray(std::vector<int> &outp, int order, int asigni, int esplit, bool verbose);
	void IntegerArrayToAtoms(const std::vector<int> &inp, int order, int asigni, int esplit, bool verbose, bool second);

	void PushInputCubeFrame(CCubeFrame *cfr);
	CCubeFrame* GetInputCubeFrame(int depth);

	void PushOutputCubeFrame(CCubeFrame *cfr);
	CCubeFrame* GetOutputCubeFrame(int depth);

	void PushOutput2CubeFrame(CCubeFrame *cfr);
	CCubeFrame* GetOutput2CubeFrame(int depth);

	void PushInputAtomFrame(CAtomSet *cfr);
	CAtomSet* GetInputAtomFrame(int depth);

	void PushOutputAtomFrame(CAtomSet *cfr);
	CAtomSet* GetOutputAtomFrame(int depth);

	void PushOutput2AtomFrame(CAtomSet *cfr);
	CAtomSet* GetOutput2AtomFrame(int depth);

	void PushInputAtomFrame_NoDelete(CAtomSet *cfr);
	void PushInputCubeFrame_NoDelete(CCubeFrame *cfr);

	void BuildHilbertIdx(int resx, int resy, int resz, bool verbose);

	double BlockMultiHuffman(std::vector<int> &inp, int blocksize, int multi, bool bwmtf, bool verbose=false);

	void BuildAtomSort(const CAtomSet *as);

	int m_iInputCubeBufPos;
	std::vector<CCubeFrame*> m_oaInputCubeBuf;

	int m_iOutputCubeBufPos;
	std::vector<CCubeFrame*> m_oaOutputCubeBuf;

	int m_iOutput2CubeBufPos;
	std::vector<CCubeFrame*> m_oaOutput2CubeBuf;

	int m_iInputAtomBufPos;
	std::vector<CAtomSet*> m_oaInputAtomBuf;

	int m_iOutputAtomBufPos;
	std::vector<CAtomSet*> m_oaOutputAtomBuf;

	int m_iOutput2AtomBufPos;
	std::vector<CAtomSet*> m_oaOutput2AtomBuf;

	std::vector<int> m_iaHilbertIdx;

	std::vector<double> m_faTempPred;
	std::vector<double> m_faTempPred2;

	std::vector<int> m_iaAtomSort;

};


#endif


