[section {Class ::httpd::reply}]

A class which shephards a request through the process of generating a
reply.

The socket associated with the reply is available at all times as the [arg chan]
variable.

The process of generating a reply begins with an [cmd httpd::server] generating a
[cmd http::class] object, mixing in a set of behaviors and then invoking the reply
object's [cmd dispatch] method.

In normal operations the [cmd dispatch] method:

[list_begin enumerated]

[enum]
Invokes the [cmd reset] method for the object to populate default headers.

[enum]
Invokes the [cmd HttpHeaders] method to stream the MIME headers out of the socket

[enum]
Invokes the [cmd {request parse}] method to convert the stream of MIME headers into a
dict that can be read via the [cmd request] method.

[enum]
Stores the raw stream of MIME headers in the [arg rawrequest] variable of the object.

[enum]
Invokes the [cmd content] method for the object, generating an call to the [cmd error]
method if an exception is raised.

[enum]
Invokes the [cmd output] method for the object
[list_end]

[para]

[section {Reply Method Ensembles}]

The [cmd http::reply] class and its derivatives maintain several variables as dictionaries
internally. Access to these dictionaries is managed through a dedicated ensemble. The
ensemble implements most of the same behaviors as the [cmd dict] command.

Each ensemble implements the following methods above, beyond, or modifying standard dicts:

[list_begin definitions]

[call method [cmd ENSEMBLE::add] [arg field] [arg element]]

Add [arg element] to a list stored in [arg field], but only if it is not already present om the list.

[call method [cmd ENSEMBLE::dump]]

Return the current contents of the data structure as a key/value list.

[call method [cmd ENSEMBLE::get] [arg field]]

Return the value of the field [arg field], or an empty string if it does not exist.

[call method [cmd ENSEMBLE::reset]]

Return a key/value list of the default contents for this data structure.

[call method [cmd ENSEMBLE::remove] [arg field] [arg element]]

Remove all instances of [arg element] from the list stored in [arg field].

[call method [cmd ENSEMBLE::replace] [arg keyvaluelist]]

Replace the internal dict with the contents of [arg keyvaluelist]

[call method [cmd ENSEMBLE::reset]]

Replace the internal dict with the default state.

[call method [cmd ENSEMBLE::set] [arg field] [arg value]]

Set the value of [arg field] to [arg value].

[list_end]

[section {Reply Method Ensemble: http_info}]

Manages HTTP headers passed in by the server.

Ensemble Methods:

[list_begin definitions]

[call method [cmd http_info::netstring]]

Return the contents of this data structure as a netstring encoded block.

[list_end]

[section {Reply Method Ensemble: request}]

Managed data from MIME headers of the request.

[list_begin definitions]

[call method  [cmd request::parse] [arg string]]

Replace the contents of the data structure with information encoded in a MIME
formatted block of text ([arg string]).

[list_end]

[section {Reply Method Ensemble: reply}]

Manage the headers sent in the reply.


[list_begin definitions]

[call method [cmd reply::output]]

Return the contents of this data structure as a MIME encoded block appropriate
for an HTTP response.

[list_end]

[section {Reply Methods}]

[list_begin definitions]
[call method [cmd close]]

Terminate the transaction, and close the socket.

[call method [cmd HttpHeaders] [arg sock] [arg ?debug?]]

Stream MIME headers from the socket [arg sock], stopping at an empty line. Returns
the stream as a block of text.

[call method [cmd dispatch] [arg newsock] [arg datastate]]

Take over control of the socket [arg newsock], and store that as the [arg chan] variable
for the object. This method runs through all of the steps of reading HTTP headers, generating
content, and closing the connection. (See class writetup).

[call method [cmd error] [arg code] [arg ?message?] [arg ?errorInfo?]]

Generate an error message of the specified [arg code], and display the [arg message] as the
reason for the exception. [arg errorInfo] is passed in from calls, but how or if it should be
displayed is a prerogative of the developer.

[call method [cmd content]]

Generate the content for the reply. This method is intended to be replaced by the mixin.

Developers have the option of streaming output to a buffer via the [cmd puts] method of the
reply, or simply populating the [arg reply_body] variable of the object.
The information returned by the [cmd content] method is not interpreted in any way.

If an exception is thrown (via the [cmd error] command in Tcl, for example) the caller will
auto-generate a 505 {Internal Error} message.

A typical implementation of [cmd content] look like:

[example {

tool::define ::test::content.file {
	superclass ::httpd::content.file
	# Return a file
	# Note: this is using the content.file mixin which looks for the reply_file variable
	# and will auto-compute the Content-Type
	method content {} {
	  my reset
    set doc_root [my http_info get doc_root]
    my variable reply_file
    set reply_file [file join $doc_root index.html]
	}
}
tool::define ::test::content.time {
  # return the current system time
	method content {} {
		my variable reply_body
    my reply set Content-Type text/plain
		set reply_body [clock seconds]
	}
}
tool::define ::test::content.echo {
	method content {} {
		my variable reply_body
    my reply set Content-Type [my request get Content-Type]
		set reply_body [my PostData [my request get Content-Length]]
	}
}
tool::define ::test::content.form_handler {
	method content {} {
	  set form [my FormData]
	  my reply set Content-Type {text/html; charset=UTF-8}
    my puts "<HTML><HEADER><TITLE>My Dynamic Page</TITLE></HEADER>"
    my puts "<BODY>"
    my puts "You Sent<p>"
    my puts "<TABLE>"
    foreach {f v} $form {
      my puts "<TR><TH>$f</TH><TD><verbatim>$v</verbatim></TD>"
    }
    my puts "</TABLE><p>"
    my puts "Send some info:<p>"
    my puts "<FORM action=/[my http_info get REQUEST_PATH] method POST>"
    my puts "<TABLE>"
    foreach field {name rank serial_number} {
      set line "<TR><TH>$field</TH><TD><input name=\"$field\" "
      if {[dict exists $form $field]} {
        append line " value=\"[dict get $form $field]\"""
      }
      append line " /></TD></TR>"
      my puts $line
    }
    my puts "</TABLE>"
    my puts "</BODY></HTML>"
	}
}

}]

[call method [cmd EncodeStatus] [arg status]]

Formulate a standard HTTP status header from he string provided.

[call method FormData]

For GET requests, converts the QUERY_DATA header into a key/value list.

For POST requests, reads the Post data and converts that information to
a key/value list for application/x-www-form-urlencoded posts. For multipart
posts, it composites all of the MIME headers of the post to a singular key/value
list, and provides MIME_* information as computed by the [cmd mime] package, including
the MIME_TOKEN, which can be fed back into the mime package to read out the contents.

[call method MimeParse [arg mimetext]]

Converts a block of mime encoded text to a key/value list. If an exception is encountered,
the method will generate its own call to the [cmd error] method, and immediately invoke
the [cmd output] method to produce an error code and close the connection.

[call method [cmd output]]

Schedules a call to [cmd DoOutput] when [arg chan] becomes writeable

[call method [cmd DoOutput]]

Generates the the HTTP reply, and streams that reply back across [arg chan].

[call method PostData [arg length]]

Stream [arg length] bytes from the [arg chan] socket, but only of the request is a
POST or PUSH. Returns an empty string otherwise.

[call method [cmd puts] [arg string]]

Appends the value of [arg string] to the end of [arg reply_body], as well as a trailing newline
character.

[call method [cmd reset]]

Clear the contents of the [arg reply_body] variable, and reset all headers in the [cmd reply]
structure back to the defaults for this object.

[call method [cmd timeOutCheck]]

Called from the [cmd http::server] object which spawned this reply. Checks to see
if too much time has elapsed while waiting for data or generating a reply, and issues
a timeout error to the request if it has, as well as destroy the object and close the
[arg chan] socket.

[call method [cmd timestamp]]

Return the current system time in the format: [example {%a, %d %b %Y %T %Z}]

[call method [cmd TransferComplete] [arg args]]

Intended to be invoked from [cmd {chan copy}] as a callback. This closes every channel
fed to it on the command line, and then destroys the object.

[example {
    ###
    # Output the body
    ###
    chan configure $sock -translation binary -blocking 0 -buffering full -buffersize 4096
    chan configure $chan -translation binary -blocking 0 -buffering full -buffersize 4096
    if {$length} {
      ###
      # Send any POST/PUT/etc content
      ###
      chan copy $sock $chan -command [namespace code [list my TransferComplete $sock]]
    } else {
      catch {close $sock}
      chan flush $chan
      my destroy
    }
}]

[call method [cmd Url_Decode] [arg string]]

De-httpizes a string.

[list_end]
