/*  RipOff - Plugin based CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <cdio/cdio.h>
#include "model.h"

/* helper function used to strings representing the length of a track, used by the model */
void track_length_string(CdIo_t *p_cdio, gchar *track_length, gint track_num);

/* reads the current CD and creates a GtkModel that represents it */
GtkTreeModel *create_and_fill_model(gchar *device_path);

void change_row(	gint track_number, 
			const gchar *artist_string, 
			const gchar *track_string,
			GtkTreeModel *model)
{
	GtkTreeIter iter;
	GtkTreePath *path;
	gchar path_string[6];

	/* creates a path string based on track_number */
	g_snprintf(path_string, 6, "%i", track_number); 

	/* create a path based on path_string */
	path = gtk_tree_path_new_from_string(path_string);

	/* creates a GtkTreeIter based on the path */
	gtk_tree_model_get_iter(GTK_TREE_MODEL (model), &iter, path);

	/* sets the artist and track names to the new values */
	if(artist_string != NULL)
	gtk_tree_store_set(	GTK_TREE_STORE (model),
				&iter,
				COL_ARTIST_NAME,
				artist_string,
				-1);

	if(track_string != NULL)
	gtk_tree_store_set(	GTK_TREE_STORE (model),
				&iter,
				COL_TRACK_NAME,
				track_string,
				-1);
}
				
void cell_toggled(	GtkCellRendererToggle *cell, 
			gchar *path_string, 
			gpointer data)
{
	GtkTreeModel *model = (GtkTreeModel *)data;
	GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
	GtkTreeIter iter;
	gboolean old_value;

	gint column = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (cell), "column"));
	gtk_tree_model_get_iter (GTK_TREE_MODEL (model), &iter, path);

	gtk_tree_model_get (	GTK_TREE_MODEL (model), 
				&iter, 
				column, 
				&old_value, 
				-1);

	/* handles the track IS to be extracted case */
	if(old_value)
		gtk_tree_store_set (	GTK_TREE_STORE (model), 
					&iter, 
					column, 
					FALSE, 
					-1);

	/* handles the track is NOT to be extracted case */
	else
		gtk_tree_store_set (	GTK_TREE_STORE (model), 
					&iter, 
					column, 
					TRUE, 
					-1);
}

void cell_edited(	GtkCellRendererText *cell, 
		 	gchar *path_string, 
			gchar *new_text, 
			gpointer data)
{
	GtkTreeModel *model = (GtkTreeModel *)data;
	GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
	GtkTreeIter iter;

	/* determines which column was edited */
	gint column = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (cell), "column"));
	
	/* sets up a GtkTreeIter according to the GtkTreePath */
	gtk_tree_model_get_iter(GTK_TREE_MODEL (model), 
				&iter,
				path);

	/* sets the new value of the cell */
	gtk_tree_store_set (	GTK_TREE_STORE (model), 
				&iter, 
				column, 
				new_text, 
				-1);
}

GtkWidget *create_view_and_model(gchar *device_path)
{
	GtkCellRenderer     *renderer;
  	GtkWidget           *view;
  	GtkTreeModel        *model;
 	GtkTreeViewColumn   *col;

  	view = gtk_tree_view_new();
  	model = GTK_TREE_MODEL(gtk_tree_store_new(	NUM_COLS, 
				       	G_TYPE_STRING, 
				       	G_TYPE_STRING, 
				      	G_TYPE_STRING, 
			               	G_TYPE_STRING,
				       	G_TYPE_BOOLEAN,
				       	G_TYPE_INT,
					G_TYPE_INT,
					G_TYPE_INT));

  	/* --- View Column #1 Extract Checkboxes EDITABLE---*/
  	renderer = gtk_cell_renderer_toggle_new();
  	g_signal_connect(	renderer, 
				"toggled", (GCallback) cell_toggled, 
				model);

  	col = gtk_tree_view_column_new_with_attributes(		"Extract", 
						  		renderer, 
						  		"active", COL_EXTRACT,
						 	 	NULL);

  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  	g_object_set_data(		G_OBJECT (renderer), 
					"column", (gint*) COL_EXTRACT);
  

  	/* --- View Column #2 Track Number Column NOT EDITABLE---*/


  	renderer = gtk_cell_renderer_text_new ();

  	col =   gtk_tree_view_column_new_with_attributes (	"Track",  
                                                   		renderer,
                                                    		"text", COL_TRACK_NUMBER,
						   		 NULL);

  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  	g_object_set_data(	G_OBJECT (renderer), 
				"column", (gint*)  COL_TRACK_NUMBER);

  	/* --- View Column #3 Track length NOT EDITABLE--- */
  	renderer = gtk_cell_renderer_text_new ();

  	col =   gtk_tree_view_column_new_with_attributes (	"Length",  
                                                    		renderer,
                                                   	 	"text", COL_TRACK_LENGTH_STRING,
						    		NULL);

 	 g_object_set(col, "resizable", TRUE, NULL);

  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  	g_object_set_data(	G_OBJECT (renderer), 
				"column", (gint*)  COL_TRACK_LENGTH_STRING);

  	/* --- Column #4 Artist Name EDITABLE--- */
  	renderer = gtk_cell_renderer_text_new ();

  	g_signal_connect(renderer, "edited", (GCallback) cell_edited, model);

  	col =   gtk_tree_view_column_new_with_attributes (	"Artist",  
                                                    		renderer,
                                                    		"text", COL_ARTIST_NAME,
						    		NULL);

  	g_object_set(col, "resizable", TRUE, NULL);

  	g_object_set(renderer, "editable", TRUE, NULL);

  	g_object_set_data(	G_OBJECT (renderer), 
				"column", (gint*)  COL_ARTIST_NAME);


  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  	/* --- Column #5 Track Name EDITABLE--- */
  	renderer = gtk_cell_renderer_text_new ();

  	g_signal_connect(renderer, "edited", (GCallback) cell_edited, model);

  	col =   gtk_tree_view_column_new_with_attributes (	"Track Name",  
                                                    		renderer,
                                                   		"text", COL_TRACK_NAME,
						   	 	NULL);

  	g_object_set(col, "resizable", TRUE, NULL);

  	g_object_set(renderer, "editable", TRUE, NULL);

  	g_object_set_data(	G_OBJECT (renderer), 
				"column", (gint*) COL_TRACK_NAME);

  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);


  	gtk_tree_view_set_model(GTK_TREE_VIEW(view), model); /*Connect tree and model here*/
  	g_object_unref(model); /* destroy model automatically with view */
  	gtk_tree_selection_set_mode(	gtk_tree_view_get_selection(GTK_TREE_VIEW(view)),
                             		GTK_SELECTION_NONE);

 	return view;
}

void track_length_string(CdIo_t *p_cdio, gchar *track_length, gint track_num)
{
	gint length = cdio_get_track_sec_count(p_cdio, track_num) - 150;

	
	/* Performs operations necessary to format the track second count into mm:ss format */
	g_snprintf(		track_length, 
				6, 
				"%i%i:%i%i", 
				((length/75)+2)/60/10, 
				((length/75)+2)/60%10, 
				((length/75)+2)%60/10,
				((length/75)+2)%60%10);
}

void select_all(GtkWidget *widget, gpointer model)
{
	GtkTreeIter iter;

	/* checks to see that model isn't empty */
	if(gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
		/* goes through model setting all extract checkboxes to TRUE */
		do{
		
			gtk_tree_store_set (		GTK_TREE_STORE (model), 
							&iter, 
							COL_EXTRACT, 
							TRUE, 
							-1);

		}while(gtk_tree_model_iter_next(GTK_TREE_MODEL (model), &iter));
}

void deselect_all(GtkWidget *widget, gpointer model)
{
	GtkTreeIter iter;

	/* checks to see that model isn't empty */
	if(gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
		/* goes through model setting all extract checkboxes to FALSE */
		do{
		
			gtk_tree_store_set (		GTK_TREE_STORE (model), 
							&iter, 
							COL_EXTRACT, 
							FALSE, 
							-1);

		}while(gtk_tree_model_iter_next(GTK_TREE_MODEL (model), &iter));
}

void ripoff_refill_model(gchar *device_path, GtkTreeStore *treestore)
{
	CdIo_t *p_cdio = cdio_open (device_path, DRIVER_DEVICE);
	GtkTreeIter iter;
	gshort tracks;
	gint j;
	gchar title_string[200];
	gchar track_num[5];
	gchar track_length[5];

	/* Checks to ensure that a CD was correctly detected and opened*/
	if(p_cdio !=NULL)
	{
		/* gets the number of tracks on the CD */
		tracks = cdio_get_num_tracks(p_cdio);

		/* Performs the following operations for each track detected by p_cdio */
		for(j = 1; j < tracks+1; ++j)
		{
			track_length_string(p_cdio, track_length, j);

			if(j >= 10)
			{
				/* Formats and title and track number strings for numbers < 10 */
				g_snprintf(title_string, 200, "Track %i", j);
				g_snprintf(track_num, 5, "%i", j);
			}
			else
			{
				/* Formats and title and track number strings for numbers > 10 */
				g_snprintf(title_string, 200, "Track 0%i", j);
				g_snprintf(track_num, 5, "0%i", j);
			}

			/* creates a new row and sets the values according to the tracks */
  			gtk_tree_store_append(treestore, &iter, NULL);

 			gtk_tree_store_set(	treestore, &iter,
		     				COL_TRACK_NUMBER, track_num,
                    				COL_TRACK_NAME, title_string,
                     				COL_ARTIST_NAME, "Unknown Artist",
                     			COL_TRACK_LENGTH_STRING, track_length,
		     				COL_EXTRACT, TRUE,
		     			COL_FRAME_OFFSET, cdio_get_track_lsn(p_cdio, j) + 150,
					COL_TRACK_LENGTH_INT, cdio_get_track_sec_count(p_cdio, j),
						-1);
		}

		/* destroys the now useless cdio object */
		cdio_destroy(p_cdio);
	}
}	
