/*  RipOff - Plugin based CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <sys/wait.h>
#include <unistd.h>
#include <cddb/cddb.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "RipOffCore.h"
#include "RipOffTrackList.h"
#include "RipOffExtractor.h"
#include "model.h"
#include "lib/ripoff.xpm"

enum
{
	COL_GENRE_LOOKUP=0,
	COL_TITLE_LOOKUP,
	COL_ARTIST_LOOKUP,
	COL_CATEGORY_LOOKUP,
	COL_ID_LOOKUP,
	COL_ARRAY_POS,
	NUM_COLS_LOOKUP
};

struct lookup_row_selection_handling_data_
{
	RipOffCore core;
	GtkWidget *lookup_view;
	GtkWidget *lookup_window;
};
typedef struct lookup_row_selection_handling_data_ * lookup_row_selection_handling_data;

cddb_disc_t *discs[10];
gint matches_viewed;

/* Helper function that traverses the GtkModel that represents the CD and forms a
   RipOffTrackList from the tracks that the user has selected to be extracted.  
   This list is then passed to a RipOffExtractor.  */
RipOffTrackList create_track_list(RipOffCore core);

/* Signal function called when the user requests a CDDB Lookup through the GUI widgets.  Data
   is a RipOffCore object*/
void perform_lookup(GtkWidget *widget, gpointer data);

/* Signal function called when the user requests to access RipOff's preferences window.  
   Data is NULL*/
void preferences_dialog(GtkWidget *widget, gpointer data);

/* Signal function called when the user requests RipOff's about window. Data is NULL*/
void about_dialog(GtkWidget *widget, gpointer data);

/* Signal function called when the user requests RipOff to perform an extraction operation
   Data is a RipOffCore object */
void perform_extraction(GtkWidget *widget, gpointer data);

/* creates a GtkTreeView that displays CDDB lookup results based on model */
GtkWidget *create_view_and_model_lookup(GtkTreeModel *model);

/* updates the artist column of the model when the user enters a new text in the artist entry */
gboolean artist_typing_handler(GtkWidget *widget, GdkEventKey *event, gpointer data);

void cd_selected_lookup_row_activation (	GtkTreeView *view,  
						GtkTreePath *path, 
						GtkTreeViewColumn *col,
						gpointer data);

/* handles when the lookup dialog apply button is clicked */
void cd_selected_lookup_apply_button(GtkButton *button, gpointer data);

/* handles when the lookup dialog is closed by a window event */
void close_lookup_window_event(GtkWidget *window, GdkEvent *event, gpointer data);

/* handles when the lookup dialog is closed by the close button on the dialog */
void close_lookup_window_button(GtkButton *button, gpointer data);

void ripoff_quit(GtkWidget *window, gpointer data);



RipOffCore ripoff_core_new(RipOffPreferences prefs)
{
	GtkWidget *label;
	GtkWidget *table;
	GtkWidget *scroll;
	GtkWidget *button;
	GtkWidget *cd_menu;
	GtkWidget *cd_item;
	GtkWidget *edit_menu;
	GtkWidget *edit_item;
	GtkWidget *help_menu;
	GtkWidget *help_item;
	GtkWidget *about_item;
	GtkWidget *select_all_item;
	GtkWidget *deselect_all_item;
	GtkWidget *preferences_item;
	GtkWidget *extract_item;
	GtkWidget *lookup_item;
	GtkWidget *quit_item;
	GtkWidget *menu_bar;
	GtkAccelGroup *accel_group;
	GtkWidget *sep;
	GtkWidget *sep2;
	GtkWidget *image;
        GdkPixbuf *icon;

	RipOffCore new = g_new(struct RipOffCore_, 1);
	new->prefs = prefs;

	/* Basic information about the window */
	new->main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_size_request (GTK_WIDGET (new->main_window), 600, 400);
	gtk_window_set_title (GTK_WINDOW (new->main_window), NAME);

	g_signal_connect_swapped (	G_OBJECT (new->main_window), 
					"delete_event",
                             		G_CALLBACK (gtk_widget_destroy), 
                              		G_OBJECT (new->main_window));
	
	table = gtk_table_new(6, 7, FALSE);
	gtk_container_add(GTK_CONTAINER(new->main_window), table);

	/* Beginning of RipOff's menu creation code */
	cd_menu = gtk_menu_new();
	edit_menu = gtk_menu_new();
	help_menu = gtk_menu_new();

	accel_group = gtk_accel_group_new ();
	gtk_window_add_accel_group (GTK_WINDOW (new->main_window), accel_group);
	

	image = gtk_image_new_from_stock(GTK_STOCK_CDROM, 1);

	extract_item = gtk_image_menu_item_new_with_mnemonic("E_xtract CD");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(extract_item), image);

	help_item = gtk_menu_item_new_with_mnemonic("_Help");
	
	gtk_widget_add_accelerator(	extract_item, 
					"activate", 
					accel_group, 
					88, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);

	image = gtk_image_new_from_stock(GTK_STOCK_NETWORK, 1);

	lookup_item = gtk_image_menu_item_new_with_mnemonic("CDDB _Lookup");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(lookup_item), image);

	gtk_widget_add_accelerator(	lookup_item, 
					"activate", 
					accel_group, 
					108, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);

	image = gtk_image_new_from_stock(GTK_STOCK_QUIT, 1);

	quit_item = gtk_image_menu_item_new_with_mnemonic("_Quit");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(quit_item), image);

	gtk_widget_add_accelerator(	quit_item, 
					"activate", 
					accel_group, 
					81, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);

	select_all_item = gtk_menu_item_new_with_mnemonic("_Select all");

	gtk_widget_add_accelerator(	select_all_item, 
					"activate", 
					accel_group, 
					83, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);

	deselect_all_item = gtk_menu_item_new_with_mnemonic("_Deselect all");

	gtk_widget_add_accelerator(	deselect_all_item, 
					"activate", 
					accel_group, 
					68, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);
	
	image = gtk_image_new_from_stock(GTK_STOCK_PREFERENCES, 1);

	preferences_item = gtk_image_menu_item_new_with_mnemonic("_Preferences");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(preferences_item), image);

	gtk_widget_add_accelerator(	preferences_item, 
					"activate", 
					accel_group, 
					80, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);

	#if defined(GTK_MINOR_VERSION) && GTK_MINOR_VERSION > 6
        image = gtk_image_new_from_stock(GTK_STOCK_ABOUT, 1);
	#else
	image = gtk_image_new_from_stock(GTK_STOCK_DIALOG_QUESTION, 1);
	#endif
	

	about_item = gtk_image_menu_item_new_with_mnemonic("_About");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(about_item), image);

	gtk_widget_add_accelerator(	about_item, 
					"activate", 
					accel_group, 
					65, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);
	
	sep = gtk_separator_menu_item_new();
	sep2 = gtk_separator_menu_item_new();

	gtk_menu_shell_append(GTK_MENU_SHELL(cd_menu), extract_item);
	gtk_menu_shell_append(GTK_MENU_SHELL(cd_menu), lookup_item);
	gtk_menu_shell_append(GTK_MENU_SHELL(cd_menu), sep);
	gtk_menu_shell_append(GTK_MENU_SHELL(cd_menu), quit_item);

	gtk_menu_shell_append(GTK_MENU_SHELL(edit_menu), select_all_item);
	gtk_menu_shell_append(GTK_MENU_SHELL(edit_menu), deselect_all_item);
	gtk_menu_shell_append(GTK_MENU_SHELL(edit_menu), sep2);
	gtk_menu_shell_append(GTK_MENU_SHELL(edit_menu), preferences_item);

	gtk_menu_shell_append(GTK_MENU_SHELL(help_menu), about_item);

	menu_bar = gtk_menu_bar_new();
	
	gtk_table_attach (	GTK_TABLE(table), 
				menu_bar, 0, 6, 0, 1, 
				GTK_FILL | GTK_EXPAND, 
				GTK_SHRINK, 
				0, 5);

	cd_item = gtk_menu_item_new_with_mnemonic("_CD");

	gtk_widget_add_accelerator(	cd_item, 
					"activate", 
					accel_group, 
					67, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);

	edit_item = gtk_menu_item_new_with_mnemonic("_Edit");

	gtk_widget_add_accelerator(	edit_item, 
					"activate", 
					accel_group, 
					67, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);

	help_item = gtk_menu_item_new_with_mnemonic("_Help");

	gtk_widget_add_accelerator(	help_item, 
					"activate", 
					accel_group, 
					72, 
					GDK_CONTROL_MASK, 
					GTK_ACCEL_VISIBLE);

	gtk_menu_item_set_submenu(GTK_MENU_ITEM(cd_item), cd_menu);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(edit_item), edit_menu);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(help_item), help_menu);

	gtk_menu_bar_append(GTK_MENU_BAR(menu_bar), cd_item);
	gtk_menu_bar_append(GTK_MENU_BAR(menu_bar), edit_item);
	gtk_menu_bar_append(GTK_MENU_BAR(menu_bar), help_item);

	/* end of RipOff's menu creation code */

	/* Creates the label and entry fields for information on the entire CD */
	label = gtk_label_new("Album:  ");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach(	GTK_TABLE(table), 
				label, 
				0, 1, 
				1, 2, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				0, 5);

	new->album = gtk_entry_new();
	gtk_entry_set_max_length (GTK_ENTRY(new->album), 200);
	/* gtk_entry_set_text (GTK_ENTRY (new->album), "Untitled Album"); */
	gtk_table_attach(	GTK_TABLE(table), 
				new->album, 
				1, 2, 
				1, 2, 
				GTK_FILL | GTK_EXPAND, 
				GTK_SHRINK, 
				0, 5);

	label = gtk_label_new("Artist:  ");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach(	GTK_TABLE(table), 
				label, 
				0, 1, 
				2, 3, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				0, 5);

	new->artist = gtk_entry_new();
	gtk_entry_set_max_length (GTK_ENTRY(new->artist), 200);
	/* gtk_entry_set_text (GTK_ENTRY (new->artist), "Untitled Artist"); */
	gtk_table_attach(	GTK_TABLE(table), 
				new->artist, 
				1, 2, 
				2, 3, 
				GTK_FILL | GTK_EXPAND, 
				GTK_SHRINK, 
				0, 5);

	label = gtk_label_new("Genre:  ");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach(	GTK_TABLE(table), 
				label, 
				0, 1, 
				3, 4, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				0, 5);

	new->genre = gtk_entry_new();
	gtk_entry_set_max_length (GTK_ENTRY(new->genre), 50);
	/* gtk_entry_set_text (GTK_ENTRY (new->genre), "Genre"); */
	gtk_table_attach(	GTK_TABLE(table), 
				new->genre, 
				1, 2, 
				3, 4, 
				GTK_FILL | GTK_EXPAND, 
				GTK_SHRINK, 
				0, 5);

	label = gtk_label_new("Year:  ");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach(	GTK_TABLE(table), 
				label, 
				2, 3, 
				3, 4, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				5, 5);

	new->year = gtk_entry_new();
	gtk_entry_set_max_length (GTK_ENTRY(new->year), 4);
	/* gtk_entry_set_text (GTK_ENTRY (new->year), "Year"); */
	gtk_table_attach(	GTK_TABLE(table), 
				new->year, 
				3, 4, 
				3, 4, 
				GTK_SHRINK, 
				GTK_SHRINK, 
				0, 5);
	gtk_widget_set_size_request(new->year, 60, 30);

	/* creates the view used for looking at a CD's track list, and puts it in a scrolled
	   window */
	new->view = create_view_and_model(ripoff_preferences_get_cd_device_path(new->prefs)); 

	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (new->view), TRUE);

	gtk_tree_selection_set_mode (	gtk_tree_view_get_selection (GTK_TREE_VIEW (new->view)),
				   	GTK_SELECTION_SINGLE);

	scroll = gtk_scrolled_window_new(NULL, NULL);
	
	gtk_scrolled_window_set_policy(		GTK_SCROLLED_WINDOW(scroll), 
				       		GTK_POLICY_AUTOMATIC, 
				       		GTK_POLICY_AUTOMATIC);

	gtk_container_add(GTK_CONTAINER(scroll), new->view);
	
	gtk_table_attach(	GTK_TABLE(table), 
				scroll, 0,4,
				4, 5, 
				GTK_EXPAND | GTK_FILL, 
				GTK_EXPAND | GTK_FILL, 
				5, 5);
	
	/* attaches CDDB lookup button to table and connects to signal */

	button = gtk_button_new_with_mnemonic("CDDB _Lookup");
	
	#if defined(GTK_MINOR_VERSION) && GTK_MINOR_VERSION > 6
	image = gtk_image_new_from_stock(GTK_STOCK_NETWORK, 1); 
	gtk_button_set_image(GTK_BUTTON(button), image);
	#endif
	
	gtk_table_attach(	GTK_TABLE(table), 
				button, 
				2,3, 5,6, 
				0, 
				0,
				5,5);


  	g_signal_connect(	button, "clicked", 
				(GCallback) perform_lookup, 
				/*gtk_tree_view_get_model(GTK_TREE_VIEW(new->view))*/ new
				);
	/* attaches extract button to table and connects to signal */
	button = gtk_button_new_with_mnemonic("E_xtract");

	#if defined(GTK_MINOR_VERSION) && GTK_MINOR_VERSION > 6
	image = gtk_image_new_from_stock(GTK_STOCK_CDROM, 1);
	gtk_button_set_image(GTK_BUTTON(button), image);
	#endif

	gtk_table_attach(	GTK_TABLE(table), 
				button, 
				3, 4, 5,6, 
				0, 
				0,
				5,5);

  	g_signal_connect(	button, "clicked", 
				G_CALLBACK(perform_extraction), 
				new);

	/* attaches the status bar to table */
	gtk_table_attach(	GTK_TABLE(table),
				status_bar,
				0, 5, 6, 7,
				GTK_FILL | GTK_EXPAND,
				0,
				0,0);

	/* connects extract menu item to signal */
	g_signal_connect(	G_OBJECT(extract_item), 
				"activate", 
				G_CALLBACK(perform_extraction),
				new);
	
	/* connects lookup menu item to signal */
	g_signal_connect(	G_OBJECT(lookup_item),
				"activate",
				G_CALLBACK(perform_lookup),
				/*gtk_tree_view_get_model(GTK_TREE_VIEW(new->view))*/ new
				);

	/* connects quit menu item to signal. This needs to be changed */
	g_signal_connect(	G_OBJECT(quit_item),
				"activate",
				G_CALLBACK(ripoff_quit),
				new);

	/* connects select all item to signal */
	g_signal_connect(	G_OBJECT(select_all_item),
				"activate",
				G_CALLBACK(select_all), 
				GTK_TREE_MODEL(gtk_tree_view_get_model(GTK_TREE_VIEW(new->view)))
				);
	
	/* connects deselect all menu item to signal */
	g_signal_connect(	G_OBJECT(deselect_all_item),
				"activate",
				G_CALLBACK(deselect_all),
				GTK_TREE_MODEL(gtk_tree_view_get_model(GTK_TREE_VIEW(new->view)))
				);

	/* connects preferences menu item to signal */
	g_signal_connect(	G_OBJECT(preferences_item),
				"activate",
				G_CALLBACK(preferences_dialog),
				new);

	/* connects aboutmenu item to signal */
	g_signal_connect(	G_OBJECT(about_item),
				"activate",
				G_CALLBACK(about_dialog),
				new->main_window);

	/* connects window destruction signal to a proper exit function */
	g_signal_connect (	G_OBJECT (new->main_window), 
				"destroy",
                      		G_CALLBACK (ripoff_quit), 
				new);

	/* connects artist name changes to the model so that all cd track artist entries get updated 
	   accordingly when the artist field changes */
	g_signal_connect(	G_OBJECT (new->artist),
				"key-release-event",
				G_CALLBACK (artist_typing_handler),
				new->view);

        /* Set icon for program*/
        icon = gdk_pixbuf_new_from_xpm_data((const gchar **) ripoff_xpm);
        gtk_window_set_icon(GTK_WINDOW(new->main_window), icon);
        g_object_unref(icon);

	/* shows all item contained in menu */	
	gtk_widget_show_all(new->main_window);

	return new;
}

gboolean artist_typing_handler(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	const gchar *entry = gtk_entry_get_text(GTK_ENTRY(widget));
	GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(data));
	GtkTreeIter iter;

	/* checks to see that model isn't empty */
	if(gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
	{

		/* goes through model setting all of the artist column to whatever
		   is in the artist entry*/
		do{
			gtk_tree_store_set(	GTK_TREE_STORE (model),
						&iter,
						COL_ARTIST_NAME,
						entry,
						-1);

		}while(gtk_tree_model_iter_next(GTK_TREE_MODEL (model), &iter));
	}

	return FALSE;
	
}

gchar *ripoff_core_get_album(RipOffCore core)
{
	return (gchar *) gtk_entry_get_text(GTK_ENTRY(core->album));
}

gchar *ripoff_core_get_year(RipOffCore core)
{
	return (gchar *) gtk_entry_get_text(GTK_ENTRY(core->year));
}

gchar *ripoff_core_get_genre(RipOffCore core)
{
	return (gchar *) gtk_entry_get_text(GTK_ENTRY(core->genre));
}

void ripoff_core_set_album(RipOffCore core, gchar *album_string)
{
	gtk_entry_set_text (GTK_ENTRY(core->album), album_string);
}

void ripoff_core_set_artist(RipOffCore core, gchar *artist_string)
{
	gtk_entry_set_text (GTK_ENTRY(core->artist), artist_string);
}

void ripoff_core_set_year(RipOffCore core, gchar *year_string)
{
	gtk_entry_set_text (GTK_ENTRY(core->year), year_string);
}

void ripoff_core_set_genre(RipOffCore core, gchar *genre_string)
{
	gtk_entry_set_text (GTK_ENTRY(core->genre), genre_string);
}

void ripoff_core_update_model(RipOffCore core)
{
	
	ripoff_refill_model(ripoff_preferences_get_cd_device_path(core->prefs),
			    GTK_TREE_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(core->view))));
}

void ripoff_core_clear_model(RipOffCore core)
{
   gtk_tree_store_clear(GTK_TREE_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(core->view))));
}

void perform_lookup(GtkWidget *widget, gpointer data)
{
	GtkTreeIter iter;
	GtkWidget *view;
	GtkWidget *apply_button;
	GtkWidget *close_button;
	GtkWidget *lookup_window;
	GtkWidget *table;
	GtkWidget *scroll;
	GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(((RipOffCore)data)->view));
	GtkTreeStore *treestore;
	cddb_conn_t *conn = NULL;
	cddb_disc_t *disc = NULL;
	cddb_track_t *track = NULL;
	gint second_count = 0;
	gint frame_offset;
	gint seconds;
	gint matches;
        gchar *proxy_string;
        gchar *proxy_string_copy;
	gboolean cddb_connection_initialized = FALSE;
	lookup_row_selection_handling_data handling_data;

	#ifdef DEBUG
	g_print("RipOffCore: CDDB Lookup Activated\n");
	#endif

	disc = cddb_disc_new();

	/* checks that the model is not empty */

	if(gtk_tree_model_get_iter_first (model, &iter))
	{
		/* This loops goes through the model and stores the track length and frame offset
		   of each track. The frame offset is stored in a cddb_track_t and then that
		   that track is added to a cddb_disc_t. Iterates until it gets to the end of the
		   model */
		do{

			track = cddb_track_new();

			gtk_tree_model_get(	model,
						&iter,
						COL_FRAME_OFFSET,
						&frame_offset,
						-1);

			gtk_tree_model_get(	model,
						&iter,
						COL_TRACK_LENGTH_INT,
						&seconds,
						-1);

			second_count += seconds;

			cddb_track_set_frame_offset(track, frame_offset);

			cddb_disc_add_track(disc, track);
			
		} while(gtk_tree_model_iter_next(model, &iter));

		/* second_count is divided by 75 and has 2 added to it. This is the formula
		   for convertaing a disc sector count into its length in seconds. */
		second_count = second_count / 75 + 2;

		/* Sets the length of the cddb_disc equal to second_count */
		cddb_disc_set_length(disc, second_count);

		/* Queries the CDDB for a matching CDDB entry for our particular disc. Matches
		   stores the total number of matching discs found whereas disc is now the
		   first disc the CDDB found */
		conn = cddb_new();
		cddb_cache_disable(conn);
		cddb_set_timeout(conn, 10);

                #ifdef DEBUG
                printf("RipOffCore: Checking if HTTP_PROXY is set in environment\n");
                #endif

                proxy_string = getenv("HTTP_PROXY");

                if(proxy_string != NULL)
                {
                        #ifdef DEBUG
                        g_printf("RipOffCore: HTTP_PROXY set\n");
                        #endif

                        proxy_string_copy = g_malloc(strlen(proxy_string) + 1);
                        strcpy(proxy_string_copy, proxy_string);

                        cddb_http_proxy_enable(conn);
                        cddb_set_server_port(conn, 80); 
                        
                        proxy_string = strtok(proxy_string_copy, "http://");
                        
                        if(proxy_string != NULL)
                        {
                                cddb_set_http_proxy_server_name(conn, proxy_string);
                                #ifdef DEBUG
                                g_printf("RipOffCore: Proxy server name %s\n", proxy_string);
                                #endif

                                proxy_string = strtok(NULL, ":");

                                if(proxy_string != NULL)
                                {
                                        cddb_set_http_proxy_server_port(conn, atoi(proxy_string));
                                                
                                        #ifdef DEBUG
                                        g_printf("RipOffCore: Proxy server port %i\n", atoi(proxy_string));
                                        #endif
                                }
                        }
                        g_free(proxy_string_copy);
                }
                #ifdef DEBUG
                else 
                {
                        g_printf("RipOffCore: HTTP_PROXY not set");
                }
                #endif
		cddb_connection_initialized = TRUE;

		handling_data = g_new(struct lookup_row_selection_handling_data_, 1);

		gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
		gtk_statusbar_push(GTK_STATUSBAR(status_bar), 0, "Querying CDDB.....");

		matches = cddb_query(conn, disc);

		if(matches > 0)
		{
			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);

			treestore = gtk_tree_store_new(		NUM_COLS_LOOKUP, 
								G_TYPE_STRING,
								G_TYPE_STRING,
								G_TYPE_STRING,
								G_TYPE_INT,
								G_TYPE_INT,
								G_TYPE_INT);
			matches_viewed = 0;

			do{
				/* creates a new row and sets the values 
				   according to the tracks */
				cddb_read(conn, disc);
				discs[matches_viewed] = cddb_disc_clone(disc);
  				gtk_tree_store_append(treestore, &iter, NULL);

 				gtk_tree_store_set(	treestore, 
							&iter,
		     					COL_GENRE_LOOKUP, 
							cddb_disc_get_genre(disc),
                    					COL_TITLE_LOOKUP, 
							cddb_disc_get_title(disc),
                     					COL_ARTIST_LOOKUP, 
							cddb_disc_get_artist(disc),
							COL_CATEGORY_LOOKUP,
							cddb_disc_get_category(disc),
							COL_ID_LOOKUP,
							cddb_disc_get_discid(disc),
							COL_ARRAY_POS,
							matches_viewed,
							-1);

				cddb_query_next(conn, disc);
				--matches;
				++matches_viewed;

			} while(matches > 0  && matches_viewed < 10);
			cddb_destroy(conn);
			cddb_disc_destroy(disc);

			view = create_view_and_model_lookup(GTK_TREE_MODEL(treestore));

			scroll = gtk_scrolled_window_new(NULL, NULL);
	
			gtk_scrolled_window_set_policy(		GTK_SCROLLED_WINDOW(scroll), 
				       				GTK_POLICY_AUTOMATIC, 
				       				GTK_POLICY_AUTOMATIC);

			gtk_container_add(GTK_CONTAINER(scroll), view);
			apply_button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
			close_button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);

			lookup_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
			gtk_window_set_type_hint(GTK_WINDOW(lookup_window), GDK_WINDOW_TYPE_HINT_DIALOG);
			gtk_window_set_decorated(GTK_WINDOW(lookup_window), TRUE);
			gtk_window_set_skip_taskbar_hint(GTK_WINDOW(lookup_window), TRUE);
			gtk_widget_set_size_request(GTK_WIDGET(lookup_window), 525, 165);
			gtk_window_set_title (GTK_WINDOW (lookup_window), "CDDB Lookup Results");
			gtk_window_set_transient_for(GTK_WINDOW (lookup_window), 
						     GTK_WINDOW (((RipOffCore)data)->main_window));
			gtk_container_set_border_width(GTK_CONTAINER (lookup_window), 6);

			table = gtk_table_new(3, 3, FALSE);

			gtk_table_attach (	GTK_TABLE(table), 
						scroll, 0, 3, 0, 2, 
						GTK_EXPAND | GTK_FILL, 
						GTK_EXPAND | GTK_FILL, 
						3, 2);

			gtk_table_attach (	GTK_TABLE(table), 
						apply_button, 0, 1, 2, 3, 
						GTK_SHRINK, 
						GTK_SHRINK, 
						3, 2);

			gtk_table_attach (	GTK_TABLE(table), 
						close_button, 2, 3, 2, 3, 
						GTK_SHRINK, 
						GTK_SHRINK, 
						3, 2);
			

			gtk_container_add(GTK_CONTAINER(lookup_window), table);

			/*gtk_window_set_keep_above(GTK_WINDOW(lookup_window), TRUE);
			gtk_window_set_transient_for(		GTK_WINDOW(lookup_window), 
						GTK_WINDOW(((RipOffCore)data)->main_window));*/

			handling_data->core = (RipOffCore)data;
			handling_data->lookup_view = view;
			handling_data->lookup_window = lookup_window;

			/*g_signal_connect_swapped (	lookup_window, 
							"delete_event",
                             				G_CALLBACK (gtk_widget_destroy), 
                              				lookup_window);

			g_signal_connect_swapped (	close_button, 
							"clicked",
                             				G_CALLBACK (gtk_widget_destroy), 
                              				lookup_window);*/

			g_signal_connect(	lookup_window,
						"delete_event",
						G_CALLBACK(close_lookup_window_event),
						handling_data);

			g_signal_connect(	close_button,
						"clicked",
						G_CALLBACK(close_lookup_window_button),
						handling_data);

			g_signal_connect(	apply_button,
						"clicked",
						G_CALLBACK(cd_selected_lookup_apply_button),
						handling_data);

			g_signal_connect(	view,
						"row-activated",
						G_CALLBACK(cd_selected_lookup_row_activation),
						handling_data);
									
			gtk_widget_show_all(lookup_window);

		}

		else if(matches == 0)
		{
			cddb_destroy(conn);
			cddb_disc_destroy(disc);

			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
			gtk_statusbar_push(	GTK_STATUSBAR(status_bar), 
						0, 
						"No matches found for CD");
		}

		else
		{
			cddb_destroy(conn);
			cddb_disc_destroy(disc);
			libcddb_shutdown();

			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
			gtk_statusbar_push(	GTK_STATUSBAR(status_bar), 
						0, 
						"Could not connect to CDDB");
		}
	}
}


void preferences_dialog(GtkWidget *widget, gpointer data)
{
	#ifdef DEBUG
	g_print("RipOffCore: Preferences Activated\n");
	#endif

	ripoff_preferences_draw_window(((RipOffCore)data)->prefs, ((RipOffCore)data)->main_window);
}

void about_dialog(GtkWidget *widget, gpointer data)
{
	gtk_widget_show_all(about_window_draw_with_logo
                                        (NAME, 
					NAME" "VERSION, 
					NAME" is a modular, intuitive, GTK+ based CD Ripper",
					"Copyright © 2006 Bobby Ryan Newberry",
					"http://ripoffc.sourceforge.net"));
}

void perform_extraction(GtkWidget *widget, gpointer data)
{
	RipOffPlugin plugin = ripoff_preferences_get_plugin(((RipOffCore)data)->prefs);
	RipOffTrackList list = create_track_list((RipOffCore)data);
	const gchar *cd_device_path = ripoff_preferences_get_cd_device_path(((RipOffCore)data)->prefs);
	const gchar *output_folder = ripoff_preferences_get_output_folder(((RipOffCore)data)->prefs);
	const gchar *file_pattern = ripoff_preferences_get_file_pattern(((RipOffCore)data)->prefs);

	#ifdef DEBUG
	g_print("RipOffCore: Creating a new RipOffExtractor....\n");
	#endif

	RipOffExtractor extractor = ripoff_extractor_new(	plugin,
								list,
								cd_device_path,
								output_folder,
								file_pattern,
								((RipOffCore)data)->main_window);

	#ifdef DEBUG
	g_print("RipOffCore: New RipOffExtractor created succesfully. Performing operation....\n");
	#endif

	ripoff_extractor_perform_operation(extractor);

	#ifdef DEBUG
	g_print("RipOffCore: Operation exited successfully. Thread should be running....\n");
	#endif
}

RipOffTrackList create_track_list(RipOffCore core)
{
	GtkTreeModel *model;
	gchar *album_title_string;
	gchar *genre_string;
	gchar *year_string;
	RipOffTrackList list;
	RipOffTrack track;
	GtkTreeIter iter;
	gboolean extraction;
	gint counter = 1;
	gchar *buffer;
	list = ripoff_track_list_new();
	album_title_string = ripoff_core_get_album(core);
	genre_string = ripoff_core_get_genre(core);
	year_string = ripoff_core_get_year(core);
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(core->view));

	/* Checks to see if the model is not empty, then goes through it if it is
	   forming a RipOffTrackList from the data contained in it */
	if(gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
	{
		gtk_tree_model_get (		GTK_TREE_MODEL (model), 
						&iter, 
						COL_EXTRACT, 
						&extraction, 
						-1);

	/* This case handles if the very first track in the model has been selected */
		if(extraction)
		{
			track = ripoff_track_new();
			ripoff_track_set_album_title(track, album_title_string);
			ripoff_track_set_genre(track, genre_string);
			ripoff_track_set_year(track, year_string);
			
			gtk_tree_model_get (		GTK_TREE_MODEL (model),
							&iter,
							COL_ARTIST_NAME,
							&buffer,
							-1);
		
			ripoff_track_set_artist(track, buffer);

			gtk_tree_model_get (		GTK_TREE_MODEL (model),
							&iter,
							COL_TRACK_NAME,
							&buffer,
							-1);

			ripoff_track_set_track_title(track, buffer);

			ripoff_track_set_track_num(track, 1);
			ripoff_track_list_add_track(list, track);
		}

		/* Traverses through rest of model adding selected tracks to list */
		while(gtk_tree_model_iter_next(GTK_TREE_MODEL (model), &iter))
		{
			++counter;
			gtk_tree_model_get (		GTK_TREE_MODEL (model), 
							&iter, 
							COL_EXTRACT, 
							&extraction, 
							-1);

			if(extraction)
			{
					track = ripoff_track_new();
					ripoff_track_set_album_title(track, album_title_string);
					ripoff_track_set_genre(track, genre_string);
					ripoff_track_set_year(track, year_string);
			
					gtk_tree_model_get (		GTK_TREE_MODEL (model),
									&iter,
									COL_ARTIST_NAME,
									&buffer,
									-1);
					
					ripoff_track_set_artist(track, buffer);

					gtk_tree_model_get (		GTK_TREE_MODEL (model),
									&iter,
									COL_TRACK_NAME,
									&buffer,
									-1);

					ripoff_track_set_track_title(track, buffer);

					ripoff_track_set_track_num(track, counter);
					ripoff_track_list_add_track(list, track);
			}
			

		}
	}

	return list;
}

GtkWidget *create_view_and_model_lookup(GtkTreeModel *model)
{
	GtkCellRenderer     *renderer;
  	GtkWidget           *view;
 	GtkTreeViewColumn   *col;

  	view = gtk_tree_view_new();

  	/* --- View Column #1 Genre NOT EDITABLE ---*/
  	renderer = gtk_cell_renderer_text_new();

  	col = gtk_tree_view_column_new_with_attributes(		"Genre", 
						  		renderer, 
						  		"text", COL_GENRE_LOOKUP,
						 	 	NULL);

	g_object_set(col, "resizable", TRUE, NULL);
	g_object_set(renderer, "editable", FALSE, NULL);
 	g_object_set_data(		G_OBJECT (renderer), 
					"column", (gint*) COL_GENRE_LOOKUP);

  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  	/* --- View Column #2 Title NOT EDITABLE---*/

  	renderer = gtk_cell_renderer_text_new ();

  	col =   gtk_tree_view_column_new_with_attributes (	"Title",  
                                                   		renderer,
                                                    		"text", COL_TITLE_LOOKUP,
						   		 NULL);

	g_object_set(col, "resizable", TRUE, NULL);
	g_object_set(renderer, "editable", FALSE, NULL);
  	g_object_set_data(	G_OBJECT (renderer), 
				"column", (gint*)  COL_TITLE_LOOKUP);

  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  	/* --- View Column #3 Artist NOT EDITABLE--- */

  	renderer = gtk_cell_renderer_text_new ();

  	col =   gtk_tree_view_column_new_with_attributes (	"Artist",  
                                                    		renderer,
                                                   	 	"text", COL_ARTIST_LOOKUP,
						    		NULL);

 	g_object_set(col, "resizable", TRUE, NULL);
	g_object_set(renderer, "editable", FALSE, NULL);
  	g_object_set_data(	G_OBJECT (renderer), 
				"column", (gint*)  COL_ARTIST_LOOKUP);

  	gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);


  	gtk_tree_view_set_model(GTK_TREE_VIEW(view), model); /*Connect tree and model here*/
  	g_object_unref(model); /* destroy model automatically with view */
  	gtk_tree_selection_set_mode(	gtk_tree_view_get_selection(GTK_TREE_VIEW(view)),
                             		GTK_SELECTION_NONE);

  	gtk_tree_selection_set_mode(	gtk_tree_view_get_selection(GTK_TREE_VIEW(view)),
                             		GTK_SELECTION_BROWSE);

 	return view;
}

void cd_selected_lookup_row_activation (	GtkTreeView *view,  
						GtkTreePath *path, 
						GtkTreeViewColumn *col,
						gpointer data)
{
	GtkTreeIter iter;
	gchar year[50];
	GtkTreeModel *model;
	gint j, num_tracks, array_pos;
	cddb_track_t *track = NULL;

	lookup_row_selection_handling_data handling = (lookup_row_selection_handling_data) data;
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(handling->core->view));

	/* gets the row selected in the model */
	if(gtk_tree_model_get_iter(gtk_tree_view_get_model(view), &iter, path))
	{

		g_snprintf(year, 50, "%i", cddb_disc_get_year(discs[array_pos]));

		/* gets the id and category of selected disc */
		gtk_tree_model_get (	gtk_tree_view_get_model(view), 
					&iter, 
					COL_ARRAY_POS,
					&array_pos,
					-1);

		g_snprintf(year, 50, "%i", cddb_disc_get_year(discs[array_pos]));

		/* sets the info displayed in the ripoff core to the selected discs info */
		ripoff_core_set_album(handling->core, cddb_disc_get_title(discs[array_pos]));
		ripoff_core_set_artist(	handling->core, 
						cddb_disc_get_artist(discs[array_pos]));
		ripoff_core_set_genre(handling->core, cddb_disc_get_genre(discs[array_pos]));
		ripoff_core_set_year(handling->core, year);

		num_tracks = cddb_disc_get_track_count(discs[array_pos]);

			/* sets the information stored in the tree model to the correct values */
		for(j = 0; j < num_tracks; ++j)
		{
			track = cddb_disc_get_track(discs[array_pos], j);
			change_row(	j, 
					cddb_track_get_artist(track), 
					cddb_track_get_title(track), 
					model);
			}
		}
}

void cd_selected_lookup_apply_button(GtkButton *button, gpointer data)
{
	GtkTreeIter iter;
	gchar year[50];
	GtkTreeModel *model;
	GtkTreeModel *lookup_model;
	gint j, num_tracks, array_pos;
	cddb_track_t *track = NULL;

	lookup_row_selection_handling_data handling = (lookup_row_selection_handling_data) data;
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(handling->core->view));
	lookup_model = gtk_tree_view_get_model(GTK_TREE_VIEW(handling->lookup_view));

	/* gets the row selected in the model */
	gtk_tree_selection_get_selected(
			gtk_tree_view_get_selection(GTK_TREE_VIEW(handling->lookup_view)), 
						&lookup_model,
						&iter);

	/* gets the id and category of selected disc */
	gtk_tree_model_get (	lookup_model, 
				&iter, 
				COL_ARRAY_POS, 
				&array_pos,
				-1);

	g_snprintf(year, 50, "%i", cddb_disc_get_year(discs[array_pos]));

	/* sets the info displayed in the ripoff core to the selected discs info */
	ripoff_core_set_album(handling->core, cddb_disc_get_title(discs[array_pos]));
	ripoff_core_set_artist(handling->core, cddb_disc_get_artist(discs[array_pos]));
	ripoff_core_set_genre(handling->core, cddb_disc_get_genre(discs[array_pos]));
	ripoff_core_set_year(handling->core, year);

	num_tracks = cddb_disc_get_track_count(discs[array_pos]);

	/* sets the information stored in the tree model to the correct values */
	for(j = 0; j < num_tracks; ++j)
	{
		track = cddb_disc_get_track(discs[array_pos], j);
		change_row(j, cddb_track_get_artist(track), cddb_track_get_title(track), model);
	}
}

void close_lookup_window_event(GtkWidget *window, GdkEvent *event, gpointer data)
{
	close_lookup_window_button(NULL, data);
}

void close_lookup_window_button(GtkButton *button, gpointer data)
{
	gint j;
	lookup_row_selection_handling_data handling = (lookup_row_selection_handling_data) data;

	gtk_widget_destroy(handling->lookup_window);
	g_free(handling);

	for(j = 0; j < matches_viewed; ++j)
	{
		cddb_disc_destroy(discs[j]);
	}
}

void ripoff_quit(GtkWidget *window, gpointer data)
{
	/*g_print("Attempting to save preferences to disc....\n");
	g_print("Test String Value: %s", ((RipOffCore) data)->test_string);*/
	RipOffPreferences prefs= ((RipOffCore) data)->prefs;

	ripoff_preferences_save(prefs);
	libcddb_shutdown();

	gtk_main_quit();
}
