//! Load the test definitions.
/*
 * Copyright (c) 2022  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

use std::fs;
use std::path::Path;

use anyhow::{bail, Context, Result};
use config_diag::ConfigDiag;
use serde::Deserialize;

use crate::defs::{Config, TestCase, TestData};

const PATH_TESTS: &str = "tests/test-data/tests.json";

/// The full test definitions data file.
#[derive(Debug, Deserialize)]
struct TestDataTop {
    /// The test cases themselves.
    tests: Vec<TestCase>,
}

/// Load the test definitions from the JSON file.
pub fn load_defs(cfg: &Config) -> Result<TestData> {
    let test_defs = Path::new(PATH_TESTS)
        .canonicalize()
        .with_context(|| format!("Could not canonicalize the test defs path {}", PATH_TESTS))?;
    cfg.diag(|| format!("about to read {}", test_defs.display()));
    let contents = fs::read_to_string(&test_defs).with_context(|| {
        format!(
            "Could not read the test definitions from {}",
            test_defs.display()
        )
    })?;
    let fver = typed_format_version::get_version_from_str(&contents, serde_json::from_str)
        .with_context(|| {
            format!(
                "Could not get the format version from {}",
                test_defs.display()
            )
        })?;
    if (fver.major(), fver.minor()) != (0, 1) {
        bail!(format!(
            "Unexpected test definitions format version {}.{}",
            fver.major(),
            fver.minor()
        ));
    }

    let test_data = serde_json::from_str::<TestDataTop>(&contents).with_context(|| {
        format!(
            "Could not load the test definitions from {}",
            test_defs.display()
        )
    })?;
    Ok(TestData {
        tests: test_data.tests,
    })
}
