\name{plot.dmat}
\alias{plot.dmat}
\title{ Plot Distance Matrix }
\description{
  Plot a distance matrix (DM) or a difference distance matrix (DDM).
}
\usage{
\method{plot}{dmat}(x, key = TRUE, resnum.1 = c(1:ncol(x)), resnum.2 = resnum.1,
          axis.tick.space = 20, zlim = range(x, finite = TRUE),
          nlevels = 20, levels = pretty(zlim, nlevels),
          color.palette = bwr.colors,
          col = color.palette(length(levels) - 1),
          axes = TRUE, key.axes, xaxs = "i", yaxs = "i", las = 1,
          grid = TRUE, grid.col = "yellow", grid.nx = floor(ncol(x)/30),
          grid.ny = grid.nx, center.zero = TRUE, flip=TRUE, ...)
}
\arguments{
  \item{x}{ a numeric distance matrix generated by the function
    \code{\link{dm}}. }
  \item{key}{ logical, if TRUE a color key is plotted. }
  \item{resnum.1}{ a vector of residue numbers for annotating the x axis. }
  \item{resnum.2}{ a vector of residue numbers for annotating the y axis. }
  \item{axis.tick.space}{ the separation between each axis tick mark. }
  \item{zlim}{ z limits for the distances to be plotted. }
  \item{nlevels}{ if \code{levels} is not specified, the range of 'z' values
    is divided into approximately this many levels. }
  \item{levels}{ a set of levels used to partition the range of 'z'.
    Must be *strictly* increasing (and finite).  Areas with 'z' values
    between consecutive levels are painted with the same color. }
  \item{color.palette}{ a color palette function, used to assign
    colors in the plot. }
  \item{col}{ an explicit set of colors to be used in the plot. This
    argument overrides any palette function specification. }
  \item{axes}{ logical, if TRUE plot axes are drawn. }
  \item{key.axes}{ statements which draw axes on the plot key. It
    overrides the default axis. }
  \item{xaxs}{ the x axis style.  The default is to use internal labeling. }
  \item{yaxs}{ the y axis style.  The default is to use internal labeling. }
  \item{las}{ the style of labeling to be used.  The default is to use
    horizontal labeling. }
  \item{grid}{ logical, if TRUE overlaid grid is drawn. }
  \item{grid.col}{ color of the overlaid grid. }
  \item{grid.nx}{ number of grid cells in the x direction.  }
  \item{grid.ny}{ number of grid cells in the y direction.  }
  \item{center.zero}{ logical, if TRUE levels are forced to be
    equidistant around zero, assuming that zlim ranges from less than to
    more than zero. }
  \item{flip}{ logical, indicating whether the second axis should be
    fliped. }
  \item{\dots}{ additional graphical parameters for image. }
}
\value{
  Called for its effect.
}
\references{
  Grant, B.J. et al. (2006) \emph{Bioinformatics} \bold{22}, 2695--2696.T

  Much of this function is based on the \code{filled.contour} function
  by Ross Ihaka.
 }
\author{ Barry Grant }
\note{
  This function is based on the \code{layout} and legend key code in the
  function \code{filled.contour} by Ross Ihaka.  As with
  \code{filled.contour} the output is a combination of two plots: the
  legend and (in this case) \code{image} (rather than a contour plot).
}
\seealso{ \code{\link{dm}}, \code{\link{filled.contour}},
  \code{\link{contour}}, \code{\link{image}} }
\examples{

# Read PDB file
pdb <- read.pdb( system.file("examples/1hel.pdb", package="bio3d") )

# DM
d <- dm(pdb,"calpha")

# Plot DM
##filled.contour(d, nlevels = 4)
##plot(d)
plot(d,
     resnum.1 = pdb$atom[pdb$calpha,"resno"],
     color.palette = mono.colors,
     xlab="Residue Number", ylab="Residue Number")

\dontrun{
# Download and align two PDB files
pdbs <- pdbaln( get.pdb( c( "4q21", "521p"), path=tempdir(), overwrite=TRUE))

# Get distance matrix
a <- dm.xyz(pdbs$xyz[1,])
b <- dm.xyz(pdbs$xyz[2,])

# Calculate DDM
c <- a - b

# Plot DDM
plot(c,key=FALSE, grid=FALSE)

plot(c, axis.tick.space=10,
     resnum.1=pdbs$resno[1,],
     resnum.2=pdbs$resno[2,],
     grid.col="black",
     xlab="Residue No. (4q21)", ylab="Residue No. (521p)")

}
}
\keyword{ hplot }
