/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmlinearalgebra.c
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>

/* test vec3 */
PGM_START_TEST (test_linearalgebra_vec3)
{
  PgmVec3 *v1, *v2, *v3;
  gfloat a;

  /* pgm_vec3_new */
  v1 = pgm_vec3_new ();
  fail_if (v1->v[0] != 0.0f || v1->v[1] != 0.0f || v1->v[2] != 0.0f,
           "pgm_vec3_new error");
  pgm_vec3_free (v1);

  /* pgm_vec3_new_from_scalars */
  v1 = pgm_vec3_new_from_scalars (1.0f, 2.0f, 3.0f);
  fail_if (v1->v[0] != 1.0f || v1->v[1] != 2.0f || v1->v[2] != 3.0f,
           "pgm_vec3_new_from_scalars error");
  pgm_vec3_free (v1);

  /* pgm_vec3_copy */
  v1 = pgm_vec3_new_from_scalars (1.0f, 2.0f, 3.0f);
  v2 = pgm_vec3_copy (v1);
  fail_if (v2->v[0] != 1.0f || v2->v[1] != 2.0f || v2->v[2] != 3.0f,
           "pgm_vec3_copy error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);

  /* pgm_vec3_set_from_scalars */
  v1 = pgm_vec3_new ();
  pgm_vec3_set_from_scalars (v1, 5.0f, 0.5f, 3.5f);
  fail_if (v1->v[0] != 5.0f || v1->v[1] != 0.5f || v1->v[2] != 3.5f,
           "pgm_vec3_set_from_scalars error");
  pgm_vec3_free (v1);

  /* pgm_vec3_set_from_vec3 */
  v1 = pgm_vec3_new ();
  v2 = pgm_vec3_new_from_scalars (5.0f, 0.5f, 3.5f);
  pgm_vec3_set_from_vec3 (v1, v2);
  fail_if (v1->v[0] != 5.0f || v1->v[1] != 0.5f || v1->v[2] != 3.5f,
           "pgm_vec3_set_from_vec3 error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);

  /* pgm_vec3_length */
  v1 = pgm_vec3_new_from_scalars (5.0f, 0.5f, 3.5f);
  a = pgm_vec3_length (v1);
  fail_if (a < 6.12372 || a > 6.12373, "pgm_vec3_length error");
  pgm_vec3_free (v1);

  /* pgm_vec3_normalize */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_normalize (v1);
  fail_if (v2->v[0] < 0.91265f || v2->v[0] > 0.91266f ||
           v2->v[1] < 0.16319f || v2->v[1] > 0.1632f  ||
           v2->v[2] < 0.37473f || v2->v[2] > 0.37474f,
           "pgm_vec3_normalize error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);

  /* pgm_vec3_dot_product */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_new_from_scalars (5.2f, -5.6f, 16.9f);
  a = pgm_vec3_dot_product (v1, v2);
  fail_if (a < 168.18f || a > 168.19f, "pgm_vec3_dot_product error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);

  /* pgm_vec3_cross_product */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_new_from_scalars (5.2f, -5.6f, 16.9f);
  v3 = pgm_vec3_cross_product (v1, v2);
  fail_if (v3->v[0] < 80.34f || v3->v[0] > 80.36f ||
           v3->v[1] < -222.96f || v3->v[1] > -222.94f ||
           v3->v[2] < -98.7f || v3->v[2] > -98.5f,
           "pgm_vec3_cross_product error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);
  pgm_vec3_free (v3);

  /* pgm_vec3_add_scalar */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_add_scalar (v1, 12.0f);
  fail_if (v2->v[0] != 27.1f || v2->v[1] != 14.7f || v2->v[2] != 18.2f,
           "pgm_vec3_add_scalar error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);

  /* pgm_vec3_add_vec3 */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_new_from_scalars (5.2f, -5.6f, 16.9f);
  v3 = pgm_vec3_add_vec3 (v1, v2);
  fail_if (v3->v[0] < 20.29f || v3->v[0] > 20.3f ||
           v3->v[1] < -2.95f || v3->v[1] > -2.85f ||
           v3->v[2] < 23.09f || v3->v[2] > 23.1f,
           "pgm_vec3_add_vec3 error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);
  pgm_vec3_free (v3);

  /* pgm_vec3_substract_scalar */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_substract_scalar (v1, 1.2f);
  fail_if (v2->v[0] < 13.89f || v2->v[0] > 13.91f ||
           v2->v[1] < 1.49f || v2->v[1] > 1.51f ||
           v2->v[2] < 4.99f || v2->v[2] > 5.01f,
           "pgm_vec3_substract_scalar error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);

  /* pgm_vec3_substract_vec3 */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_new_from_scalars (5.2f, -5.6f, 16.9f);
  v3 = pgm_vec3_substract_vec3 (v1, v2);
  fail_if (v3->v[0] < 9.89f || v3->v[0] > 9.91f ||
           v3->v[1] < 8.29f || v3->v[1] > 8.31f ||
           v3->v[2] < -10.71f || v3->v[2] > -10.69f,
           "pgm_vec3_substract_vec3 error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);
  pgm_vec3_free (v3);

  /* pgm_vec3_multiply_scalar */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_multiply_scalar (v1, 2.5f);
  fail_if (v2->v[0] != 37.75f || v2->v[1] != 6.75f || v2->v[2] != 15.5f,
           "pgm_vec3_multiply_scalar error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);

  /* pgm_vec3_multiply_vec3 */
  v1 = pgm_vec3_new_from_scalars (15.1f, 2.7f, 6.2f);
  v2 = pgm_vec3_new_from_scalars (5.2f, -5.6f, 16.9f);
  v3 = pgm_vec3_multiply_vec3 (v1, v2);
  fail_if (v3->v[0] < 78.519f || v3->v[0] > 78.52 ||
           v3->v[1] < -15.13f || v3->v[1] > -15.11f ||
           v3->v[2] < 104.779f || v3->v[2] > 104.78f,
           "pgm_vec3_multiply_vec3 error");
}
PGM_END_TEST;

/* test vec4 */
PGM_START_TEST (test_linearalgebra_vec4)
{
  PgmVec4 *v1, *v2, *v3;
  gfloat a;

  /* pgm_vec4_new */
  v1 = pgm_vec4_new ();
  fail_if (v1->v[0] != 0.0f || v1->v[1] != 0.0f || v1->v[2] != 0.0f
           || v1->v[3] != 0.0f,
           "pgm_vec4_new error");
  pgm_vec4_free (v1);

  /* pgm_vec4_new_from_scalars */
  v1 = pgm_vec4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f);
  fail_if (v1->v[0] != 1.0f || v1->v[1] != 2.0f || v1->v[2] != 3.0f
           || v1->v[3] != 4.0f,
           "pgm_vec4_new_from_scalars error");
  pgm_vec4_free (v1);

  /* pgm_vec4_copy */
  v1 = pgm_vec4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f);
  v2 = pgm_vec4_copy (v1);
  fail_if (v2->v[0] != 1.0f || v2->v[1] != 2.0f || v2->v[2] != 3.0f
           || v1->v[3] != 4.0f,
           "pgm_vec4_copy error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);

  /* pgm_vec4_set_from_scalars */
  v1 = pgm_vec4_new ();
  pgm_vec4_set_from_scalars (v1, 5.0f, 0.5f, 3.5f, 6.8f);
  fail_if (v1->v[0] != 5.0f || v1->v[1] != 0.5f || v1->v[2] != 3.5f
           || v1->v[3] != 6.8f,
           "pgm_vec4_set_from_scalars error");
  pgm_vec4_free (v1);

  /* pgm_vec4_set_from_vec4 */
  v1 = pgm_vec4_new ();
  v2 = pgm_vec4_new_from_scalars (5.0f, 0.5f, 3.5f, 6.8);
  pgm_vec4_set_from_vec4 (v1, v2);
  fail_if (v1->v[0] != 5.0f || v1->v[1] != 0.5f || v1->v[2] != 3.5f
           || v1->v[3] != 6.8f,
           "pgm_vec4_set_from_vec4 error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);

  /* pgm_vec4_length */
  v1 = pgm_vec4_new_from_scalars (5.0f, 0.5f, 3.5f, 2.7f);
  a = pgm_vec4_length (v1);
  fail_if (a < 6.69253 || a > 6.69254, "pgm_vec4_length error");
  pgm_vec4_free (v1);

  /* pgm_vec4_normalize */
  v1 = pgm_vec4_new_from_scalars (15.1f, 2.7f, 6.2f, 3.9f);
  v2 = pgm_vec4_normalize (v1);
  fail_if (v2->v[0] < 0.88831f || v2->v[0] > 0.88832f ||
           v2->v[1] < 0.15883f || v2->v[1] > 0.15884f ||
           v2->v[2] < 0.36473f || v2->v[2] > 0.36474f ||
           v2->v[3] < 0.22943f || v2->v[3] > 0.22944f,
           "pgm_vec4_normalize error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);

  /* pgm_vec4_add_scalar */
  v1 = pgm_vec4_new_from_scalars (15.1f, 2.7f, 6.2f, 5.8f);
  v2 = pgm_vec4_add_scalar (v1, 12.0f);
  fail_if (v2->v[0] != 27.1f || v2->v[1] != 14.7f || v2->v[2] != 18.2f ||
           v2->v[3] != 17.8f, "pgm_vec4_add_scalar error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);

  /* pgm_vec4_add_vec4 */
  v1 = pgm_vec4_new_from_scalars (15.1f, 2.7f, 6.2f, 4.7f);
  v2 = pgm_vec4_new_from_scalars (5.2f, -5.6f, 16.9f, 6.2f);
  v3 = pgm_vec4_add_vec4 (v1, v2);
  fail_if (v3->v[0] < 20.29f || v3->v[0] > 20.3f ||
           v3->v[1] < -2.95f || v3->v[1] > -2.85f||
           v3->v[2] < 23.09f || v3->v[2] > 23.1f ||
           v3->v[3] < 10.85f || v3->v[3] > 10.95f,
           "pgm_vec4_add_vec4 error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);
  pgm_vec4_free (v3);

  /* pgm_vec4_substract_scalar */
  v1 = pgm_vec4_new_from_scalars (15.1f, 2.7f, 6.2f, 5.8f);
  v2 = pgm_vec4_substract_scalar (v1, 1.2f);
  fail_if (v2->v[0] < 13.89f || v2->v[0] > 13.91f ||
           v2->v[1] < 1.49f || v2->v[1] > 1.51f ||
           v2->v[2] < 4.99f || v2->v[2] > 5.01f ||
           v2->v[3] < 4.59f || v2->v[3] > 4.61f,
           "pgm_vec4_substract_scalar error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);

  /* pgm_vec4_substract_vec4 */
  v1 = pgm_vec4_new_from_scalars (15.1f, 2.7f, 6.2f, 4.7f);
  v2 = pgm_vec4_new_from_scalars (5.2f, -5.6f, 16.9f, 6.2f);
  v3 = pgm_vec4_substract_vec4 (v1, v2);
  fail_if (v3->v[0] < 9.89f || v3->v[0] > 9.91f ||
           v3->v[1] < 8.29f || v3->v[1] > 8.31f ||
           v3->v[2] < -10.71f || v3->v[2] > -10.69f ||
           v3->v[3] < -1.51f || v3->v[3] > -1.49f,
           "pgm_vec4_substract_vec4 error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);
  pgm_vec4_free (v3);

  /* pgm_vec4_multiply_scalar */
  v1 = pgm_vec4_new_from_scalars (15.1f, 2.7f, 6.2f, 4.7f);
  v2 = pgm_vec4_multiply_scalar (v1, 2.5f);
  fail_if (v2->v[0] != 37.75f || v2->v[1] != 6.75f || v2->v[2] != 15.5f
           || v2->v[3] != 11.75f,
           "pgm_vec4_multiply_scalar error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);

  /* pgm_vec4_multiply_vec4 */
  v1 = pgm_vec4_new_from_scalars (15.1f, 2.7f, 6.2f, 4.7f);
  v2 = pgm_vec4_new_from_scalars (5.2f, -5.6f, 16.9f, 6.2f);
  v3 = pgm_vec4_multiply_vec4 (v1, v2);
  fail_if (v3->v[0] < 78.519f || v3->v[0] > 78.52 ||
           v3->v[1] < -15.13f || v3->v[1] > -15.11f ||
           v3->v[2] < 104.779f || v3->v[2] > 104.78f ||
           v3->v[3] < 29.135f || v3->v[3] > 29.145f,
           "pgm_vec4_multiply_vec4 error");
}
PGM_END_TEST;

/* test mat3x3 */
PGM_START_TEST (test_linearalgebra_mat3x3)
{
  PgmMat3x3 *m1, *m2, *m3;
  PgmVec3 *v1, *v2, *v3;

  /* pgm_mat3x3_new */
  m1 = pgm_mat3x3_new ();
  fail_if (m1->m[0] != 0.0f || m1->m[1] != 0.0f || m1->m[2] != 0.0f ||
           m1->m[3] != 0.0f || m1->m[4] != 0.0f || m1->m[5] != 0.0f ||
           m1->m[6] != 0.0f || m1->m[7] != 0.0f || m1->m[8] != 0.0f,
           "pgm_mat3x3_new error");
  pgm_mat3x3_free (m1);

  /* pgm_mat3x3_new_from_scalars */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  fail_if (m1->m[0] != 1.0f || m1->m[1] != 2.0f || m1->m[2] != 3.0f ||
           m1->m[3] != 4.0f || m1->m[4] != 5.0f || m1->m[5] != 6.0f ||
           m1->m[6] != 7.0f || m1->m[7] != 8.0f || m1->m[8] != 9.0f,
           "pgm_mat3x3_new_from_scalars error");
  pgm_mat3x3_free (m1);

  /* pgm_mat3x3_new_from_vectors */
  v1 = pgm_vec3_new_from_scalars (1.0f, 2.0f, 3.0f);
  v2 = pgm_vec3_new_from_scalars (4.0f, 5.0f, 6.0f);
  v3 = pgm_vec3_new_from_scalars (7.0f, 8.0f, 9.0f);
  m1 = pgm_mat3x3_new_from_vec3 (v1, v2, v3);
  fail_if (m1->m[0] != 1.0f || m1->m[1] != 2.0f || m1->m[2] != 3.0f ||
           m1->m[3] != 4.0f || m1->m[4] != 5.0f || m1->m[5] != 6.0f ||
           m1->m[6] != 7.0f || m1->m[7] != 8.0f || m1->m[8] != 9.0f,
           "pgm_mat3x3_new_from_vec3 error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);
  pgm_vec3_free (v3);
  pgm_mat3x3_free (m1);

  /* pgm_mat3x3_new_identity */
  m1 = pgm_mat3x3_new_identity ();
  fail_if (m1->m[0] != 1.0f || m1->m[1] != 0.0f || m1->m[2] != 0.0f ||
           m1->m[3] != 0.0f || m1->m[4] != 1.0f || m1->m[5] != 0.0f ||
           m1->m[6] != 0.0f || m1->m[7] != 0.0f || m1->m[8] != 1.0f,
           "pgm_mat3x3_new_identity error");
  pgm_mat3x3_free (m1);

  /* pgm_mat3x3_copy */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m2 = pgm_mat3x3_copy (m1);
  fail_if (m2->m[0] != 1.0f || m2->m[1] != 2.0f || m2->m[2] != 3.0f ||
           m2->m[3] != 4.0f || m2->m[4] != 5.0f || m2->m[5] != 6.0f ||
           m2->m[6] != 7.0f || m2->m[7] != 8.0f || m2->m[8] != 9.0f,
           "pgm_mat3x3_copy error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);

  /* pgm_mat3x3_set_from_scalars */
  m1 = pgm_mat3x3_new ();
  pgm_mat3x3_set_from_scalars (m1,
                               1.0f, 2.0f, 3.0f,
                               4.0f, 5.0f, 6.0f,
                               7.0f, 8.0f, 9.0f);
  fail_if (m1->m[0] != 1.0f || m1->m[1] != 2.0f || m1->m[2] != 3.0f ||
           m1->m[3] != 4.0f || m1->m[4] != 5.0f || m1->m[5] != 6.0f ||
           m1->m[6] != 7.0f || m1->m[7] != 8.0f || m1->m[8] != 9.0f,
           "pgm_mat3x3_set_from_scalars error");
  pgm_mat3x3_free (m1);

  /* pgm_mat3x3_set_from_vec3 */
  v1 = pgm_vec3_new_from_scalars (1.0f, 2.0f, 3.0f);
  v2 = pgm_vec3_new_from_scalars (4.0f, 5.0f, 6.0f);
  v3 = pgm_vec3_new_from_scalars (7.0f, 8.0f, 9.0f);
  m1 = pgm_mat3x3_new ();
  pgm_mat3x3_set_from_vec3 (m1, v1, v2, v3);
  fail_if (m1->m[0] != 1.0f || m1->m[1] != 2.0f || m1->m[2] != 3.0f ||
           m1->m[3] != 4.0f || m1->m[4] != 5.0f || m1->m[5] != 6.0f ||
           m1->m[6] != 7.0f || m1->m[7] != 8.0f || m1->m[8] != 9.0f,
           "pgm_mat3x3_set_from_vec3 error");
  pgm_vec3_free (v1);
  pgm_vec3_free (v2);
  pgm_vec3_free (v3);
  pgm_mat3x3_free (m1);

  /* pgm_mat3x3_set_from_mat3x3 */
  m1 = pgm_mat3x3_new ();
  m2 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  pgm_mat3x3_set_from_mat3x3 (m1, m2);
  fail_if (m1->m[0] != 1.0f || m1->m[1] != 2.0f || m1->m[2] != 3.0f ||
           m1->m[3] != 4.0f || m1->m[4] != 5.0f || m1->m[5] != 6.0f ||
           m1->m[6] != 7.0f || m1->m[7] != 8.0f || m1->m[8] != 9.0f,
           "pgm_mat3x3_set_from_mat3x3 error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);

  /* pgm_mat3x3_is_identity */
  m1 = pgm_mat3x3_new_identity ();
  fail_if (!pgm_mat3x3_is_identity (m1), "1st pgm_mat3x3_is_identity error");
  pgm_mat3x3_free (m1);
  m1 = pgm_mat3x3_new ();
  fail_if (pgm_mat3x3_is_identity (m1), "2nd pgm_mat3x3_is_identity error");
  pgm_mat3x3_free (m1);

  /* FIXME: pgm_mat3x3_inverse */

  /* pgm_mat3x3_transpose */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m2 = pgm_mat3x3_transpose (m1);
  fail_if (m2->m[0] != 1.0f || m2->m[1] != 4.0f || m2->m[2] != 7.0f ||
           m2->m[3] != 2.0f || m2->m[4] != 5.0f || m2->m[5] != 8.0f ||
           m2->m[6] != 3.0f || m2->m[7] != 6.0f || m2->m[8] != 9.0f,
           "pgm_mat3x3_transpose error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);

  /* pgm_mat3x3_add_scalar */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m2 = pgm_mat3x3_add_scalar (m1, 3.5f);
  fail_if (m2->m[0] != 4.5f || m2->m[1] != 5.5f || m2->m[2] != 6.5f ||
           m2->m[3] != 7.5f || m2->m[4] != 8.5f || m2->m[5] != 9.5f ||
           m2->m[6] != 10.5f || m2->m[7] != 11.5f || m2->m[8] != 12.5f,
           "pgm_mat3x3_add_scalar error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);

  /* pgm_mat3x3_add_mat3x3 */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m2 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m3 = pgm_mat3x3_add_mat3x3 (m1, m2);
  fail_if (m3->m[0] != 2.0f || m3->m[1] != 4.0f || m3->m[2] != 6.0f ||
           m3->m[3] != 8.0f || m3->m[4] != 10.0f || m3->m[5] != 12.0f ||
           m3->m[6] != 14.0f || m3->m[7] != 16.0f || m3->m[8] != 18.0f,
           "pgm_mat3x3_add_mat3x3 error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);
  pgm_mat3x3_free (m3);

  /* pgm_mat3x3_substract_scalar */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m2 = pgm_mat3x3_substract_scalar (m1, 0.5f);
  fail_if (m2->m[0] != 0.5f || m2->m[1] != 1.5f || m2->m[2] != 2.5f ||
           m2->m[3] != 3.5f || m2->m[4] != 4.5f || m2->m[5] != 5.5f ||
           m2->m[6] != 6.5f || m2->m[7] != 7.5f || m2->m[8] != 8.5f,
           "pgm_mat3x3_substract_scalar error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);

  /* pgm_mat3x3_substract_mat3x3 */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m2 = pgm_mat3x3_new_from_scalars (0.5f, 0.5f, 0.5f,
                                    0.5f, 0.5f, 0.5f,
                                    0.5f, 0.5f, 0.5f);
  m3 = pgm_mat3x3_substract_mat3x3 (m1, m2);
  fail_if (m3->m[0] != 0.5f || m3->m[1] != 1.5f || m3->m[2] != 2.5f ||
           m3->m[3] != 3.5f || m3->m[4] != 4.5f || m3->m[5] != 5.5f ||
           m3->m[6] != 6.5f || m3->m[7] != 7.5f || m3->m[8] != 8.5f,
           "pgm_mat3x3_substract_mat3x3 error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);
  pgm_mat3x3_free (m3);

  /* pgm_mat3x3_multiply_scalar */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m2 = pgm_mat3x3_multiply_scalar (m1, 3.5f);
  fail_if (m2->m[0] != 3.5f || m2->m[1] != 7.0f || m2->m[2] != 10.5f ||
           m2->m[3] != 14.0f || m2->m[4] != 17.5f || m2->m[5] != 21.0f ||
           m2->m[6] != 24.5f || m2->m[7] != 28.0f || m2->m[8] != 31.5f,
           "pgm_mat3x3_multiply_scalar error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);

  /* pgm_mat3x3_multiply_vec3 */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  v1 = pgm_vec3_new_from_scalars (4.0f, 5.0f, 6.0f);
  v2 = pgm_mat3x3_multiply_vec3 (m1, v1);
  fail_if (v2->v[0] != 32.0f || v2->v[1] != 77.0f || v2->v[2] != 122.0f,
           "pgm_mat3x3_multiply_vec3 error");
  pgm_mat3x3_free (m1);
  pgm_vec3_free (v2);
  pgm_vec3_free (v1);

  /* pgm_mat3x3_multiply_mat3x3 */
  m1 = pgm_mat3x3_new_from_scalars (1.0f, 2.0f, 3.0f,
                                    4.0f, 5.0f, 6.0f,
                                    7.0f, 8.0f, 9.0f);
  m2 = pgm_mat3x3_new_from_scalars (10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f,
                                    16.0f, 17.0f, 18.0f);
  m3 = pgm_mat3x3_multiply_mat3x3 (m1, m2);
  fail_if (m3->m[0] != 84.0f || m3->m[1] != 90.0f || m3->m[2] != 96.0f ||
           m3->m[3] != 201.0f || m3->m[4] != 216.0f || m3->m[5] != 231.0f ||
           m3->m[6] != 318.0f || m3->m[7] != 342.0f || m3->m[8] != 366.0f,
           "pgm_mat3x3_multiply_mat3x3 error");
  pgm_mat3x3_free (m1);
  pgm_mat3x3_free (m2);
  pgm_mat3x3_free (m3);
}
PGM_END_TEST;

/* test mat4x4 */
PGM_START_TEST (test_linearalgebra_mat4x4)
{
  PgmMat4x4 *m1, *m2, *m3;
  PgmVec4 *v1, *v2, *v3, *v4;
  PgmVec3 *v5;

  /* pgm_mat4x4_new */
  m1 = pgm_mat4x4_new ();
  fail_if
    (m1->m[0]!=0.0f || m1->m[1]!=0.0f || m1->m[2]!=0.0f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.0f || m1->m[5]!=0.0f || m1->m[6]!=0.0f || m1->m[7]!=0.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.0f || m1->m[10]!=0.0f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=0.0f,
     "pgm_mat4x4_new error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_from_scalars */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=2.0f || m1->m[2]!=3.0f || m1->m[3]!=4.0f ||
     m1->m[4]!=5.0f || m1->m[5]!=6.0f || m1->m[6]!=7.0f || m1->m[7]!=8.0f ||
     m1->m[8]!=9.0f || m1->m[9]!=10.0f || m1->m[10]!=11.0f || m1->m[11]!=12.0f ||
     m1->m[12]!=13.0f || m1->m[13]!=14.0f || m1->m[14]!=15.0f || m1->m[15]!=16.0f,
     "pgm_mat4x4_new_from_scalars error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_from_vec4 */
  v1 = pgm_vec4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f);
  v2 = pgm_vec4_new_from_scalars (5.0f, 6.0f, 7.0f, 8.0f);
  v3 = pgm_vec4_new_from_scalars (9.0f, 10.0f, 11.0f, 12.0f);
  v4 = pgm_vec4_new_from_scalars (13.0f, 14.0f, 15.0f, 16.0f);
  m1 = pgm_mat4x4_new_from_vec4 (v1, v2, v3, v4);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=2.0f || m1->m[2]!=3.0f || m1->m[3]!=4.0f ||
     m1->m[4]!=5.0f || m1->m[5]!=6.0f || m1->m[6]!=7.0f || m1->m[7]!=8.0f ||
     m1->m[8]!=9.0f || m1->m[9]!=10.0f || m1->m[10]!=11.0f || m1->m[11]!=12.0f ||
     m1->m[12]!=13.0f || m1->m[13]!=14.0f || m1->m[14]!=15.0f || m1->m[15]!=16.0f,
     "pgm_mat4x4_new_from_vec4 error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);
  pgm_vec4_free (v3);
  pgm_vec4_free (v4);
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_identity */
  m1 = pgm_mat4x4_new_identity ();
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=0.0f || m1->m[2]!=0.0f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.0f || m1->m[5]!=1.0f || m1->m[6]!=0.0f || m1->m[7]!=0.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.0f || m1->m[10]!=1.0f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_new_identity error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_translate_from_vec3 */
  v5 = pgm_vec3_new_from_scalars (1.0f, 2.0f, 3.0f);
  m1 = pgm_mat4x4_new_translate_from_vec3 (v5);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=0.0f || m1->m[2]!=0.0f || m1->m[3]!=1.0f ||
     m1->m[4]!=0.0f || m1->m[5]!=1.0f || m1->m[6]!=0.0f || m1->m[7]!=2.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.0f || m1->m[10]!=1.0f || m1->m[11]!=3.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_new_translate_from_vec3 error");
  pgm_vec3_free (v5);
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_translate_from_scalars */
  m1 = pgm_mat4x4_new_translate_from_scalars (1.0f, 2.0f, 3.0f);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=0.0f || m1->m[2]!=0.0f || m1->m[3]!=1.0f ||
     m1->m[4]!=0.0f || m1->m[5]!=1.0f || m1->m[6]!=0.0f || m1->m[7]!=2.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.0f || m1->m[10]!=1.0f || m1->m[11]!=3.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_new_translate_from_scalars error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_scale_from_vec3 */
  v5 = pgm_vec3_new_from_scalars (1.0f, 2.0f, 3.0f);
  m1 = pgm_mat4x4_new_scale_from_vec3 (v5);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=0.0f || m1->m[2]!=0.0f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.0f || m1->m[5]!=2.0f || m1->m[6]!=0.0f || m1->m[7]!=0.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.0f || m1->m[10]!=3.0f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_new_scale_from_vec3 error");
  pgm_vec3_free (v5);
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_scale_from_scalars */
  m1 = pgm_mat4x4_new_scale_from_scalars (1.0f, 2.0f, 3.0f);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=0.0f || m1->m[2]!=0.0f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.0f || m1->m[5]!=2.0f || m1->m[6]!=0.0f || m1->m[7]!=0.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.0f || m1->m[10]!=3.0f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_new_scale_from_scalars error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_rotate_x */
  m1 = pgm_mat4x4_new_rotate_x (PGM_DEGREES_TO_RADIANS (30.0f));
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=0.0f || m1->m[2]!=0.0f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.0f || m1->m[5]<0.866f||m1->m[5]>0.867f || m1->m[6]!=-0.5f || m1->m[7]!=0.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.5f || m1->m[10]<0.866f||m1->m[10]>0.867f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_new_rotate_x error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_rotate_y */
  m1 = pgm_mat4x4_new_rotate_y (PGM_DEGREES_TO_RADIANS (30.0f));
  fail_if
    (m1->m[0]<0.866f||m1->m[0]>0.867f || m1->m[1]!=0.0f || m1->m[2]!=0.5f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.0f || m1->m[5]!=1.0f || m1->m[6]!=0.0f || m1->m[7]!=0.0f ||
     m1->m[8]!=-0.5f || m1->m[9]!=0.0f || m1->m[10]<0.866f||m1->m[10]>0.867f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_new_rotate_y error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_rotate_z */
  m1 = pgm_mat4x4_new_rotate_z (PGM_DEGREES_TO_RADIANS (30.0f));
  fail_if
    (m1->m[0]<0.866f||m1->m[0]>0.867f || m1->m[1]!=-0.5f || m1->m[2]!=0.0f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.5f || m1->m[5]<0.866f||m1->m[5]>0.867f || m1->m[6]!=0.0f || m1->m[7]!=0.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.0f || m1->m[10]!=1.0f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_new_rotate_z error");
  pgm_mat4x4_free (m1);

  /* FIXME: pgm_mat4x4_new_rotate_axis_from_scalars
   *        pgm_mat4x4_new_rotate_axis_from_vec3 */

  /* pgm_mat4x4_copy */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m2 = pgm_mat4x4_copy (m1);
  fail_if
    (m2->m[0]!=1.0f || m2->m[1]!=2.0f || m2->m[2]!=3.0f || m2->m[3]!=4.0f ||
     m2->m[4]!=5.0f || m2->m[5]!=6.0f || m2->m[6]!=7.0f || m2->m[7]!=8.0f ||
     m2->m[8]!=9.0f || m2->m[9]!=10.0f || m2->m[10]!=11.0f || m2->m[11]!=12.0f ||
     m2->m[12]!=13.0f || m2->m[13]!=14.0f || m2->m[14]!=15.0f || m2->m[15]!=16.0f,
     "pgm_mat4x4_copy error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);

  /* pgm_mat4x4_set_from_scalars */
  m1 = pgm_mat4x4_new ();
  pgm_mat4x4_set_from_scalars (m1,
                               1.0f, 2.0f, 3.0f, 4.0f,
                               5.0f, 6.0f, 7.0f, 8.0f,
                               9.0f, 10.0f, 11.0f, 12.0f,
                               13.0f, 14.0f, 15.0f, 16.0f);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=2.0f || m1->m[2]!=3.0f || m1->m[3]!=4.0f ||
     m1->m[4]!=5.0f || m1->m[5]!=6.0f || m1->m[6]!=7.0f || m1->m[7]!=8.0f ||
     m1->m[8]!=9.0f || m1->m[9]!=10.0f || m1->m[10]!=11.0f || m1->m[11]!=12.0f ||
     m1->m[12]!=13.0f || m1->m[13]!=14.0f || m1->m[14]!=15.0f || m1->m[15]!=16.0f,
     "pgm_mat4x4_set_from_scalars error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_set_from_vec4 */
  v1 = pgm_vec4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f);
  v2 = pgm_vec4_new_from_scalars (5.0f, 6.0f, 7.0f, 8.0f);
  v3 = pgm_vec4_new_from_scalars (9.0f, 10.0f, 11.0f, 12.0f);
  v4 = pgm_vec4_new_from_scalars (13.0f, 14.0f, 15.0f, 16.0f);
  m1 = pgm_mat4x4_new ();
  pgm_mat4x4_set_from_vec4 (m1, v1, v2, v3, v4);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=2.0f || m1->m[2]!=3.0f || m1->m[3]!=4.0f ||
     m1->m[4]!=5.0f || m1->m[5]!=6.0f || m1->m[6]!=7.0f || m1->m[7]!=8.0f ||
     m1->m[8]!=9.0f || m1->m[9]!=10.0f || m1->m[10]!=11.0f || m1->m[11]!=12.0f ||
     m1->m[12]!=13.0f || m1->m[13]!=14.0f || m1->m[14]!=15.0f || m1->m[15]!=16.0f,
     "pgm_mat4x4_set_from_vec4 error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);
  pgm_vec4_free (v3);
  pgm_vec4_free (v4);
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_set_from_mat4x4 */
  m1 = pgm_mat4x4_new ();
  m2 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  pgm_mat4x4_set_from_mat4x4 (m1, m2);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=2.0f || m1->m[2]!=3.0f || m1->m[3]!=4.0f ||
     m1->m[4]!=5.0f || m1->m[5]!=6.0f || m1->m[6]!=7.0f || m1->m[7]!=8.0f ||
     m1->m[8]!=9.0f || m1->m[9]!=10.0f || m1->m[10]!=11.0f || m1->m[11]!=12.0f ||
     m1->m[12]!=13.0f || m1->m[13]!=14.0f || m1->m[14]!=15.0f || m1->m[15]!=16.0f,
     "pgm_mat4x4_set_from_mat4x4 error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);

  /* pgm_mat4x4_is_identity */
  m1 = pgm_mat4x4_new_identity ();
  fail_if (!pgm_mat4x4_is_identity (m1), "1st pgm_mat4x4_is_identity error");
  pgm_mat4x4_free (m1);
  m1 = pgm_mat4x4_new ();
  fail_if (pgm_mat4x4_is_identity (m1), "2nd pgm_mat4x4_is_identity error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_transpose */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m2 = pgm_mat4x4_transpose (m1);
  fail_if
    (m2->m[0]!=1.0f || m2->m[1]!=5.0f || m2->m[2]!=9.0f || m2->m[3]!=13.0f ||
     m2->m[4]!=2.0f || m2->m[5]!=6.0f || m2->m[6]!=10.0f || m2->m[7]!=14.0f ||
     m2->m[8]!=3.0f || m2->m[9]!=7.0f || m2->m[10]!=11.0f || m2->m[11]!=15.0f ||
     m2->m[12]!=4.0f || m2->m[13]!=8.0f || m2->m[14]!=12.0f || m2->m[15]!=16.0f,
     "pgm_mat4x4_transpose error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);

  /* pgm_mat4x4_translate_from_vec3 */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  v5 = pgm_vec3_new_from_scalars (2.0f, 3.0f, 4.0f);
  pgm_mat4x4_translate_from_vec3 (m1, v5);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=2.0f || m1->m[2]!=3.0f || m1->m[3]!=24.0f ||
     m1->m[4]!=5.0f || m1->m[5]!=6.0f || m1->m[6]!=7.0f || m1->m[7]!=64.0f ||
     m1->m[8]!=9.0f || m1->m[9]!=10.0f || m1->m[10]!=11.0f || m1->m[11]!=104.0f ||
     m1->m[12]!=13.0f || m1->m[13]!=14.0f || m1->m[14]!=15.0f || m1->m[15]!=144.0f,
     "pgm_mat4x4_translate_from_vec3 error");
  pgm_mat4x4_free (m1);
  pgm_vec3_free (v5);

  /* pgm_mat4x4_translate_from_scalars */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  pgm_mat4x4_translate_from_scalars (m1, 2.0f, 3.0f, 4.0f);
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=2.0f || m1->m[2]!=3.0f || m1->m[3]!=24.0f ||
     m1->m[4]!=5.0f || m1->m[5]!=6.0f || m1->m[6]!=7.0f || m1->m[7]!=64.0f ||
     m1->m[8]!=9.0f || m1->m[9]!=10.0f || m1->m[10]!=11.0f || m1->m[11]!=104.0f ||
     m1->m[12]!=13.0f || m1->m[13]!=14.0f || m1->m[14]!=15.0f || m1->m[15]!=144.0f,
     "pgm_mat4x4_translate_from_scalars error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_scale_from_vec3 */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  v5 = pgm_vec3_new_from_scalars (2.0f, 3.0f, 4.0f);
  pgm_mat4x4_scale_from_vec3 (m1, v5);
  fail_if
    (m1->m[0]!=2.0f || m1->m[1]!=6.0f || m1->m[2]!=12.0f || m1->m[3]!=4.0f ||
     m1->m[4]!=10.0f || m1->m[5]!=18.0f || m1->m[6]!=28.0f || m1->m[7]!=8.0f ||
     m1->m[8]!=18.0f || m1->m[9]!=30.0f || m1->m[10]!=44.0f || m1->m[11]!=12.0f ||
     m1->m[12]!=26.0f || m1->m[13]!=42.0f || m1->m[14]!=60.0f || m1->m[15]!=16.0f,
     "pgm_mat4x4_scale_from_vec3 error");
  pgm_mat4x4_free (m1);
  pgm_vec3_free (v5);

  /* pgm_mat4x4_scale_from_vec3 */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  pgm_mat4x4_scale_from_scalars (m1, 2.0f, 3.0f, 4.0f);
  fail_if
    (m1->m[0]!=2.0f || m1->m[1]!=6.0f || m1->m[2]!=12.0f || m1->m[3]!=4.0f ||
     m1->m[4]!=10.0f || m1->m[5]!=18.0f || m1->m[6]!=28.0f || m1->m[7]!=8.0f ||
     m1->m[8]!=18.0f || m1->m[9]!=30.0f || m1->m[10]!=44.0f || m1->m[11]!=12.0f ||
     m1->m[12]!=26.0f || m1->m[13]!=42.0f || m1->m[14]!=60.0f || m1->m[15]!=16.0f,
     "pgm_mat4x4_scale_from_scalars error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_rotate_x */
  m1 = pgm_mat4x4_new_identity ();
  pgm_mat4x4_rotate_x (m1, PGM_DEGREES_TO_RADIANS (30.0f));
  fail_if
    (m1->m[0]!=1.0f || m1->m[1]!=0.0f || m1->m[2]!=0.0f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.0f || m1->m[5]<0.866f||m1->m[5]>0.867f || m1->m[6]!=-0.5f || m1->m[7]!=0.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.5f || m1->m[10]<0.866f||m1->m[10]>0.867f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_rotate_x error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_rotate_y */
  m1 = pgm_mat4x4_new_identity ();
  pgm_mat4x4_rotate_y (m1, PGM_DEGREES_TO_RADIANS (30.0f));
  fail_if
    (m1->m[0]<0.866f||m1->m[0]>0.867f || m1->m[1]!=0.0f || m1->m[2]!=0.5f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.0f || m1->m[5]!=1.0f || m1->m[6]!=0.0f || m1->m[7]!=0.0f ||
     m1->m[8]!=-0.5f || m1->m[9]!=0.0f || m1->m[10]<0.866f||m1->m[10]>0.867f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_rotate_y error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_new_rotate_z */
  m1 = pgm_mat4x4_new_identity ();
  pgm_mat4x4_rotate_z (m1, PGM_DEGREES_TO_RADIANS (30.0f));
  fail_if
    (m1->m[0]<0.866f||m1->m[0]>0.867f || m1->m[1]!=-0.5f || m1->m[2]!=0.0f || m1->m[3]!=0.0f ||
     m1->m[4]!=0.5f || m1->m[5]<0.866f||m1->m[5]>0.867f || m1->m[6]!=0.0f || m1->m[7]!=0.0f ||
     m1->m[8]!=0.0f || m1->m[9]!=0.0f || m1->m[10]!=1.0f || m1->m[11]!=0.0f ||
     m1->m[12]!=0.0f || m1->m[13]!=0.0f || m1->m[14]!=0.0f || m1->m[15]!=1.0f,
     "pgm_mat4x4_rotate_z error");
  pgm_mat4x4_free (m1);

  /* FIXME: pgm_mat4x4_rotate_axis_from_scalars
   *        pgm_mat4x4_rotate_axis_from_vec3 */

  /* pgm_mat4x4_add_scalar */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m2 = pgm_mat4x4_add_scalar (m1, 3.5f);
  fail_if (m2->m[0] != 4.5f || m2->m[1] != 5.5f || m2->m[2] != 6.5f ||
           m2->m[3] != 7.5f || m2->m[4] != 8.5f || m2->m[5] != 9.5f ||
           m2->m[6] != 10.5f || m2->m[7] != 11.5f || m2->m[8] != 12.5f ||
           m2->m[9] != 13.5f || m2->m[10] != 14.5f || m2->m[11] != 15.5f ||
           m2->m[12] != 16.5f || m2->m[13] != 17.5f || m2->m[14] != 18.5f ||
           m2->m[15] != 19.5f, "pgm_mat4x4_add_scalar error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);

  /* pgm_mat4x4_add_mat4x4 */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m2 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m3 = pgm_mat4x4_add_mat4x4 (m1, m2);
  fail_if (m3->m[0] != 2.0f || m3->m[1] != 4.0f || m3->m[2] != 6.0f ||
           m3->m[3] != 8.0f || m3->m[4] != 10.0f || m3->m[5] != 12.0f ||
           m3->m[6] != 14.0f || m3->m[7] != 16.0f || m3->m[8] != 18.0f ||
           m3->m[9] != 20.0f || m3->m[10] != 22.0f || m3->m[11] != 24.0f ||
           m3->m[12] != 26.0f || m3->m[13] != 28.0f || m3->m[14] != 30.0f ||
           m3->m[15] != 32.0f, "pgm_mat4x4_add_mat4x4 error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);
  pgm_mat4x4_free (m3);

  /* pgm_mat4x4_substract_scalar */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m2 = pgm_mat4x4_substract_scalar (m1, 0.5f);
  fail_if (m2->m[0] != 0.5f || m2->m[1] != 1.5f || m2->m[2] != 2.5f ||
           m2->m[3] != 3.5f || m2->m[4] != 4.5f || m2->m[5] != 5.5f ||
           m2->m[6] != 6.5f || m2->m[7] != 7.5f || m2->m[8] != 8.5f ||
           m2->m[9] != 9.5f || m2->m[10] != 10.5f || m2->m[11] != 11.5f ||
           m2->m[12] != 12.5f || m2->m[13] != 13.5f || m2->m[14] != 14.5f ||
           m2->m[15] != 15.5f,
           "pgm_mat4x4_substract_scalar error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);

  /* pgm_mat4x4_substract_mat4x4 */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m2 = pgm_mat4x4_new_from_scalars (0.5f, 0.5f, 0.5f, 0.5f,
                                    0.5f, 0.5f, 0.5f, 0.5f,
                                    0.5f, 0.5f, 0.5f, 0.5f,
                                    0.5f, 0.5f, 0.5f, 0.5f);
  m3 = pgm_mat4x4_substract_mat4x4 (m1, m2);
  fail_if (m3->m[0] != 0.5f || m3->m[1] != 1.5f || m3->m[2] != 2.5f ||
           m3->m[3] != 3.5f || m3->m[4] != 4.5f || m3->m[5] != 5.5f ||
           m3->m[6] != 6.5f || m3->m[7] != 7.5f || m3->m[8] != 8.5f ||
           m3->m[9] != 9.5f || m3->m[10] != 10.5f || m3->m[11] != 11.5f ||
           m3->m[12] != 12.5f || m3->m[13] != 13.5f || m3->m[14] != 14.5f ||
           m3->m[15] != 15.5f,
           "pgm_mat4x4_substract_mat4x4 error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);
  pgm_mat4x4_free (m3);

  /* pgm_mat4x4_multiply_scalar */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m2 = pgm_mat4x4_multiply_scalar (m1, 5.0f);
  fail_if
    (m2->m[0]!=5.0f || m2->m[1]!=10.0f || m2->m[2]!=15.0f || m2->m[3]!=20.0f ||
     m2->m[4]!=25.0f || m2->m[5]!=30.0f || m2->m[6]!=35.0f || m2->m[7]!=40.0f ||
     m2->m[8]!=45.0f || m2->m[9]!=50.0f || m2->m[10]!=55.0f || m2->m[11]!=60.0f ||
     m2->m[12]!=65.0f || m2->m[13]!=70.0f || m2->m[14]!=75.0f || m2->m[15]!=80.0f,
     "pgm_mat4x4_multiply_scalar error");
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_multiply_vec4 */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  v1 = pgm_vec4_new_from_scalars (2.0f, 3.0f, 4.0f, 5.0f);
  v2 = pgm_mat4x4_multiply_vec4 (m1, v1);
  fail_if
    (v2->v[0]!=40.0f || v2->v[1]!=96.0f || v2->v[2]!=152.0f || v2->v[3]!=208.0f,
     "pgm_mat4x4_multiply_vec4 error");
  pgm_vec4_free (v1);
  pgm_vec4_free (v2);
  pgm_mat4x4_free (m1);

  /* pgm_mat4x4_multiply_mat4x4 */
  m1 = pgm_mat4x4_new_from_scalars (1.0f, 2.0f, 3.0f, 4.0f,
                                    5.0f, 6.0f, 7.0f, 8.0f,
                                    9.0f, 10.0f, 11.0f, 12.0f,
                                    13.0f, 14.0f, 15.0f, 16.0f);
  m2 = pgm_mat4x4_new_from_scalars (12.0f, 22.0f, 32.0f, 42.0f,
                                    52.0f, 62.0f, 72.0f, 82.0f,
                                    92.0f, 102.0f, 112.0f, 122.0f,
                                    132.0f, 142.0f, 152.0f, 162.0f);
  m3 = pgm_mat4x4_multiply_mat4x4 (m1, m2);
  fail_if
    (m3->m[0]!=920.0f || m3->m[1]!=1020.0f || m3->m[2]!=1120.0f || m3->m[3]!=1220.0f ||
     m3->m[4]!=2072.0f || m3->m[5]!=2332.0f || m3->m[6]!=2592.0f || m3->m[7]!=2852.0f ||
     m3->m[8]!=3224.0f || m3->m[9]!=3644.0f || m3->m[10]!=4064.0f || m3->m[11]!=4484.0f ||
     m3->m[12]!=4376.0f || m3->m[13]!=4956.0f || m3->m[14]!=5536.0f || m3->m[15]!=6116.0f,
     "pgm_mat4x4_multiply_mat4x4 error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);
  pgm_mat4x4_free (m3);

  /* pgm_mat4x4_inverse
   * M . M^1 = I */
  m1 = pgm_mat4x4_new_from_scalars (763.94f, 0.0f, 0.0f, 0.0f,
                                    0.0f, 763.94f, 0.0f, 0.0f,
                                    0.0f, 0.0f, -1.1f, -553.22f,
                                    0.0f, 0.0f, -1.0f, 0.0f);
  m2 = pgm_mat4x4_inverse (m1);
  m3 = pgm_mat4x4_multiply_mat4x4 (m1, m2);
  fail_if
    (m3->m[0]<0.99f || m3->m[0]>1.01f || m3->m[1]<-0.01f || m3->m[1]>0.01f ||
     m3->m[2]<-0.01f || m3->m[2]>0.01f || m3->m[3]<-0.01f || m3->m[3]>0.01f ||
     m3->m[4]<-0.01f || m3->m[4]>0.01f || m3->m[5]<0.99f || m3->m[5]>1.01f ||
     m3->m[6]<-0.01f || m3->m[6]>0.01f || m3->m[7]<-0.01f || m3->m[7]>0.01f ||
     m3->m[8]<-0.01f || m3->m[8]>0.01f || m3->m[9]<-0.01f || m3->m[9]>0.01f ||
     m3->m[10]<0.99f || m3->m[10]>1.01f || m3->m[11]<-0.01f || m3->m[11]>0.01f ||
     m3->m[12]<-0.01f || m3->m[12]>0.01f || m3->m[13]<-0.01f || m3->m[13]>0.01f ||
     m3->m[14]<-0.01f || m3->m[14]>0.01f || m3->m[15]<0.99f || m3->m[15]>1.01f,
     "pgm_mat4x4_inverse error");
  pgm_mat4x4_free (m1);
  pgm_mat4x4_free (m2);
  pgm_mat4x4_free (m3);
}
PGM_END_TEST;

/* test utilities */
PGM_START_TEST (test_linearalgebra_utilities)
{
  PgmVec3 *l1, *l2, *p, *pu, *pv, *i;
  gboolean belongs;

  /* 1st pgm_intersection_line_plane test */
  l1 = pgm_vec3_new_from_scalars (0.0f, 0.0f, 0.0f);
  l2 = pgm_vec3_new_from_scalars (0.0f, 0.0f, 1.0f);
  p = pgm_vec3_new_from_scalars (-0.5f, -0.5f, 1.0f);
  pu = pgm_vec3_new_from_scalars (1.0f, 0.0f, 0.0f);
  pv = pgm_vec3_new_from_scalars (0.0f, 1.0f, 0.0f);
  i = pgm_intersection_line_plane (l1, l2, p, pu, pv);
  fail_if (!i || i->v[0]!=0.0f || i->v[1]!=0.0f || i->v[2]!=1.0f,
           "1st pgm_intersection_line_plane test error");
  pgm_vec3_free (l1);
  pgm_vec3_free (l2);
  pgm_vec3_free (p);
  pgm_vec3_free (pu);
  pgm_vec3_free (pv);
  pgm_vec3_free (i);

  /* 2nd pgm_intersection_line_plane test */
  l1 = pgm_vec3_new_from_scalars (1.0f, 1.0f, 0.5f);
  l2 = pgm_vec3_new_from_scalars (1.0f, 0.0f, 0.5f);
  p = pgm_vec3_new_from_scalars (0.0f, 0.0f, 0.0f);
  pu = pgm_vec3_new_from_scalars (2.0f, 0.0f, 0.0f);
  pv = pgm_vec3_new_from_scalars (0.0f, 0.0f, 1.0f);
  i = pgm_intersection_line_plane (l1, l2, p, pu, pv);
  fail_if (!i || i->v[0]!=1.0f || i->v[1]!=0.0f || i->v[2]!=0.5f,
           "2nd pgm_intersection_line_plane test error");
  pgm_vec3_free (l1);
  pgm_vec3_free (l2);
  pgm_vec3_free (p);
  pgm_vec3_free (pu);
  pgm_vec3_free (pv);
  pgm_vec3_free (i);

  /* 3rd pgm_intersection_line_plane test */
  l1 = pgm_vec3_new_from_scalars (1.0f, 1.0f, 0.5f);
  l2 = pgm_vec3_new_from_scalars (1.0f, 0.0f, 0.5f);
  p = pgm_vec3_new_from_scalars (0.0f, 0.0f, 0.0f);
  pu = pgm_vec3_new_from_scalars (20.0f, 0.0f, 0.0f);
  pv = pgm_vec3_new_from_scalars (0.0f, 0.0f, 5.0f);
  i = pgm_intersection_line_plane (l1, l2, p, pu, pv);
  fail_if (!i || i->v[0]!=1.0f || i->v[1]!=0.0f || i->v[2]!=0.5f,
           "3rd pgm_intersection_line_plane test error");
  pgm_vec3_free (l1);
  pgm_vec3_free (l2);
  pgm_vec3_free (p);
  pgm_vec3_free (pu);
  pgm_vec3_free (pv);
  pgm_vec3_free (i);

  /* pgm_point_belongs_rectangle */
  p = pgm_vec3_new_from_scalars (-0.5f, -0.5f, 0.0f);
  pu = pgm_vec3_new_from_scalars (1.0f, 0.0f, 0.0f);
  pv = pgm_vec3_new_from_scalars (0.0f, 1.0f, 0.0f);

  i = pgm_vec3_new_from_scalars (0.0f, 0.0f, 0.0f);
  belongs = pgm_point_belongs_rectangle (i, p, pu, pv);
  fail_if (belongs != TRUE, "1st pgm_point_belongs_rectangle test error");

  pgm_vec3_set_from_scalars (i, 1.0f, 0.0f, 0.0f);
  belongs = pgm_point_belongs_rectangle (i, p, pu, pv);
  fail_if (belongs != FALSE, "2nd pgm_point_belongs_rectangle test error");

  pgm_vec3_set_from_scalars (i, 0.5f, 0.0f, 0.0f);
  belongs = pgm_point_belongs_rectangle (i, p, pu, pv);
  fail_if (belongs != TRUE, "3rd pgm_point_belongs_rectangle test error");

  pgm_vec3_set_from_scalars (i, 0.5f, 0.5f, 0.0f);
  belongs = pgm_point_belongs_rectangle (i, p, pu, pv);
  fail_if (belongs != TRUE, "4th pgm_point_belongs_rectangle test error");

  pgm_vec3_set_from_scalars (p, 1.0f, -1.0f, 0.0f);
  pgm_vec3_set_from_scalars (pu, 0.0f, 0.0f, 2.0f);
  pgm_vec3_set_from_scalars (pv, 0.0f, 1.0f, 0.0f);

  pgm_vec3_set_from_scalars (i, 1.0f, 0.0f, 0.0f);
  belongs = pgm_point_belongs_rectangle (i, p, pu, pv);
  fail_if (belongs != TRUE, "5th pgm_point_belongs_rectangle test error");

  pgm_vec3_set_from_scalars (i, 1.0f, -0.5f, 1.9f);
  belongs = pgm_point_belongs_rectangle (i, p, pu, pv);
  fail_if (belongs != TRUE, "6th pgm_point_belongs_rectangle test error");

  pgm_vec3_set_from_scalars (i, 1.0f, -0.5f, 2.1f);
  belongs = pgm_point_belongs_rectangle (i, p, pu, pv);
  fail_if (belongs != FALSE, "7th pgm_point_belongs_rectangle test error");

  pgm_vec3_set_from_scalars (i, 1.0f, -0.5f, 1.0f);
  belongs = pgm_point_belongs_rectangle (i, p, pu, pv);
  fail_if (belongs != TRUE, "8th pgm_point_belongs_rectangle test error");
}
PGM_END_TEST;

Suite*
pgm_linearalgebra_suite (void)
{
  Suite *s = suite_create ("PgmLinearAlgebra");
  TCase *tc_chain = tcase_create ("pgmlinearalgebra tests");

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_linearalgebra_vec3);
  tcase_add_test (tc_chain, test_linearalgebra_vec4);
  tcase_add_test (tc_chain, test_linearalgebra_mat3x3);
  tcase_add_test (tc_chain, test_linearalgebra_mat4x4);
  tcase_add_test (tc_chain, test_linearalgebra_utilities);

  return s;
}

GST_CHECK_MAIN (pgm_linearalgebra);
