///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "MultiFileParser.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(MultiFileParser, AbstractFileColumnParser)

/******************************************************************************
 * Sets the name of the input file for this parser.
 *****************************************************************************/
bool MultiFileParser::setInputFile(const QString& filename)
{
	// Clear cached file info when the input filename has changed.
	if(filename != inputFile()) _timeSteps.clear();

	// Use filename as default wild-card pattern.
	if(wildcardFilename().isEmpty()) {
		// Replace last sequence of numbers in the filename with the wild-card.
		QString wcname = QFileInfo(filename).fileName();
		int startIndex, endIndex;
		for(endIndex = wcname.length()-1; endIndex >= 0; endIndex--)
			if(wcname.at(endIndex).isNumber()) break;
		if(endIndex >= 0) {
			for(startIndex = endIndex-1; startIndex >= 0; startIndex--)
				if(!wcname.at(startIndex).isNumber()) break;
			wcname = wcname.left(startIndex+1) + '*' + wcname.mid(endIndex+1);
		}
		setWildcardFilename(wcname);
	}

	// Maybe the filename already contains some wild-card pattern.
	// In this case it has to be replaced by a real filename since
	// the base class is not able to handle this case.
	QString realFilename = filename;
	if(filename.contains('*') || filename.contains('%')) {
		QFileInfo fileInfo(filename);
		QDir dir = fileInfo.dir();
		// Use the first file that matches the file pattern.
		QStringList entries = dir.entryList(QStringList(fileInfo.fileName()), QDir::Files|QDir::NoDotAndDotDot, QDir::Name);
		if(!entries.empty()) {
			realFilename = dir.filePath(entries.first());
		}
	}

	return AbstractFileColumnParser::setInputFile(realFilename);
}


/******************************************************************************
 * Scans the input file for multiple time steps before it is actually loaded.
 *****************************************************************************/
bool MultiFileParser::prepareInputFile(bool suppressDialogs)
{
	if(!AbstractFileColumnParser::prepareInputFile(suppressDialogs)) return false;

	_timeSteps.clear();

	// Find all files that match the specified wild-card pattern.
	QStringList filenames;
	if(useWildcardFilename() && !wildcardFilename().isEmpty()) {

		// Look for additional files in the directory that match the pattern.
		QDir dir = QFileInfo(inputFile()).dir();
		QStringList entries = dir.entryList(QStringList(wildcardFilename()), QDir::Files|QDir::NoDotAndDotDot, QDir::Name);
		if(entries.empty())
			throw Exception(tr("No files in directory '%1' that match the given wild-card pattern: %2").arg(dir.path()).arg(wildcardFilename()));

		// Now the file names have to be sorted.
		// This is a little bit tricky since a file called "abc9.xyz" must come before
		// a file "abc10.xyz" which is not the default ordering.
		QMap<QString, QString> sortedFilenames;
		Q_FOREACH(QString oldName, entries) {
			// Generate a new name from the original filename that gives us the correct ordering.
			QString newName;
			QString number;
			for(int index = 0; index < oldName.length(); index++) {
				QChar c = oldName[index];
				if(!c.isDigit()) {
					if(!number.isEmpty()) {
						newName.append(number.rightJustified(10, '0'));
						number.clear();
					}
					newName.append(c);
				}
				else number.append(c);
			}
			if(!number.isEmpty()) newName.append(number.rightJustified(10, '0'));
			sortedFilenames[newName] = oldName;
		}

		for(QMap<QString, QString>::const_iterator iter = sortedFilenames.constBegin(); iter != sortedFilenames.constEnd(); ++iter)
			filenames.append(dir.absoluteFilePath(iter.value()));

	}
	else {
		// Load only the selected filename.
		if(inputFile().isEmpty() == false)
			filenames.push_back(inputFile());
	}

	if(filenames.empty())
		throw Exception(tr("No input file found."));

	// Now process each file in the list.
	Q_FOREACH(QString filename, filenames) {

		if(movieFileEnabled()) {
			if(!scanFileForTimeSteps(filename, suppressDialogs))
				return false;
		}
		else {
			// Record only one time step for this file.
			addTimeStep(filename);
		}
	}

	if(_timeSteps.empty())
		throw Exception(tr("No atomic dataset found in input file."));

	return true;
}

/******************************************************************************
 * Records the storage location of single time step for random access at a later time.
 *****************************************************************************/
void MultiFileParser::addTimeStep(const QString& filepath, streampos byteOffset, int lineNumber)
{
	TimeStep step;
	step.filename   = filepath;
	step.byteOffset = byteOffset;
	step.lineNumber = lineNumber;
	step.lastModificationTime = QFileInfo(step.filename).lastModified();

	VerboseLogger() << "Found timestep at line " << lineNumber << " (byte offset " << byteOffset << ") of input file." << endl;

	_timeSteps.push_back(step);
}

/******************************************************************************
 * Reads an atomic data set from the input file.
 *****************************************************************************/
EvaluationStatus MultiFileParser::loadAtomsFile(AtomsObject* destination, int movieFrame, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);
	OVITO_ASSERT_MSG(!inputFile().isEmpty(), "MultiFileParser::loadAtomsFile()", "Input filename has not been set. AtomsFileParser::setInputFile() must be called first.");
	OVITO_ASSERT_MSG(!_timeSteps.empty(), "MultiFileParser::loadAtomsFile()", "No input files. MultiFileParser::prepareInputFile() must be called first.");

	if(_timeSteps.empty())
		throw Exception(tr("Cannot load atoms file. No input file available."));

	if(movieFrame < 0) movieFrame = 0;
	else if(movieFrame >= _timeSteps.size()) movieFrame = _timeSteps.size() - 1;

	TimeStep& timestep = _timeSteps[movieFrame];

	// Check if the file still exists.
	QFileInfo fileinfo(timestep.filename);
	if(!fileinfo.exists())
		throw Exception(tr("Cannot load frame %1 because the file '%2' no longer exists.").arg(movieFrame+1).arg(timestep.filename));

	// Check if the file has been modified in the meantime.
	if(fileinfo.lastModified() != timestep.lastModificationTime) {
		if(timestep.byteOffset != streampos(0)) {
			throw Exception(tr("Cannot load frame %1 because the file '%2' has been modified by an external program and it is now out of sync. Press \"Reload\" to rescan the file.").arg(movieFrame+1).arg(timestep.filename));
		}
		else {
			timestep.lastModificationTime = fileinfo.lastModified();
		}
	}

	setSourceFile(timestep.filename);

	return loadTimeStep(destination, movieFrame, timestep.filename, timestep.byteOffset, timestep.lineNumber, suppressDialogs);
}

/******************************************************************************
 * Saves the class' contents to the given stream.
 *****************************************************************************/
void MultiFileParser::saveToStream(ObjectSaveStream& stream)
{
	AbstractFileColumnParser::saveToStream(stream);

	stream.beginChunk(0x3374BB);

	stream << _scanMovieFile;
	stream << _useWildcardFilename;
	stream << _wildcardFilename;

	stream << (quint32)_timeSteps.size();
	for(QVector<TimeStep>::const_iterator step = _timeSteps.begin(); step != _timeSteps.end(); ++step) {
		stream << step->filename;
		stream << (qint64)step->byteOffset;
		stream << step->lineNumber;
		stream << step->lastModificationTime;
	}
	stream.endChunk();
}

/******************************************************************************
 * Loads the class' contents from the given stream.
 *****************************************************************************/
void MultiFileParser::loadFromStream(ObjectLoadStream& stream)
{
	AbstractFileColumnParser::loadFromStream(stream);

	stream.expectChunk(0x3374BB);
	stream >> _scanMovieFile;
	stream >> _useWildcardFilename;
	stream >> _wildcardFilename;

	quint32 nFrames;
	stream >> nFrames;
	_timeSteps.resize(nFrames);
	for(QVector<TimeStep>::iterator step = _timeSteps.begin(); step != _timeSteps.end(); ++step) {
		stream >> step->filename;
		qint64 byteOffset;
		stream >> byteOffset;
		step->byteOffset = (streampos)byteOffset;
		stream >> step->lineNumber;
		stream >> step->lastModificationTime;
	}
	stream.closeChunk();
}

/******************************************************************************
 * Creates a copy of this object.
 *****************************************************************************/
RefTarget::SmartPtr MultiFileParser::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	MultiFileParser::SmartPtr clone = static_object_cast<MultiFileParser>(AbstractFileColumnParser::clone(deepCopy, cloneHelper));

	// Copy internal data.
	clone->_scanMovieFile = this->_scanMovieFile;
	clone->_useWildcardFilename = this->_useWildcardFilename;
	clone->_wildcardFilename = this->_wildcardFilename;
	clone->_timeSteps = this->_timeSteps;

	return clone;
}

};	// End of namespace AtomViz
