/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _AST_EXPRESSION_H_
#define _AST_EXPRESSION_H_

#include "GTLCore/String.h"
#include "GTLCore/AST/Statement.h"
#include "GTLCore/AST/ExpressionResult.h"

namespace LLVMBackend{
  class ExpressionGenerationContext;
  class ExpressionResult;
}

namespace GTLCore {
  class Type;
  class Function;
  class Value;

  namespace AST {
    class GenerationVisitor;
    /**
     * @internal
     * Base class for Expression node in the
     * @ingroup GTLCore_AST
     */
    class GTLCORE_EXPORT Expression : public Statement {
      public:
        virtual ~Expression() {}
        virtual const GTLCore::Type* type() const = 0;
        virtual bool isConstant() const = 0;
        virtual llvm::BasicBlock* generateStatement( LLVMBackend::GenerationContext&, llvm::BasicBlock* ) const;
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext&, LLVMBackend::ExpressionGenerationContext& ) const = 0;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const = 0;
        virtual bool isReturnStatement() const { return false; }
        /**
         * This function is called inside a \ref ReturnStatement to warn the \ref Expression
         * that the result will be returned.
         * This is mostly used to mark Array and Structure to be allocated in memory
         * and not on the stack.
         */
        virtual void markAsReturnExpression() = 0;
      public:
        /**
         * @param _declarationType is used for structure whose Value's type is different from their actual type
         * @return an \ref AST::Expression corresponding to the value @p _val given in parameter.
         */
        static AST::Expression* fromValue( const GTLCore::Value& _val, const GTLCore::Type* _declarationType = 0);
    };

    /**
     * @internal
     * A ProxyExpression is used when you want to clone some part of
     * the AST tree. The main reason behind this is that, usually
     * the Expression object is owned by the parent, this means
     * when that Expression objects are deleted by the parent.
     * While the proxy Expression won't delete the Expression it
     * got in its constructor.
     * 
     * The following construct is wrong, because the expression
     * will leak:
     * @code
     * ProxyExpression pe(new NumberExpression\<int\>(10) );
     * @endcode
     * @ingroup GTLCore_AST
     */
    class ProxyExpression : public Expression {
      public:
        ProxyExpression( Expression* expr );
        virtual const GTLCore::Type* type() const { return m_clone->type(); }
        virtual bool isConstant() const { return m_clone->isConstant(); }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual void markAsReturnExpression() { m_clone->markAsReturnExpression(); }
      private:
        Expression* m_clone;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     *
     * Store a value in the global area of the program memory.
     */
    class GlobalDataExpression : public Expression {
      public:
        GlobalDataExpression( Expression* expression );
        ~GlobalDataExpression();
        virtual const GTLCore::Type* type() const { return m_expression->type(); }
        virtual bool isConstant() const { return true; }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual void markAsReturnExpression() { m_expression->markAsReturnExpression(); m_isreturn = true; }
      private:
        Expression* m_expression;
        bool m_isreturn;
    };
    /**
     * @internal
     * This represent a constant value in the tree. _T_ can be a float, int or bool.
     * @ingroup GTLCore_AST
     */
    template<typename _T_>
    class GTLCORE_EXPORT NumberExpression : public Expression {
      public:
        explicit NumberExpression(_T_ val) : m_val(val) { }
        virtual const GTLCore::Type* type() const;
        virtual bool isConstant() const { return true; }
        _T_ value() const { return m_val; }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual void markAsReturnExpression() { }
      private:
        _T_ m_val;
    };
    
    /**
     * @internal
     * This expression call a function.
     * @ingroup GTLCore_AST
     */
    class FunctionCallExpression : public Expression {
      public:
        /**
         * @param _function a pointer to the function that will be called by this expression
         * @param _arguments list of expression used as argument of the function
         */
        FunctionCallExpression( GTLCore::Function* _function, std::list<Expression*> _arguments ) : m_function(_function), m_arguments(_arguments)
        {
        }
        ~FunctionCallExpression();
        virtual const GTLCore::Type* type() const;
        virtual bool isConstant() const { return false; }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual void markAsReturnExpression();
      private:
        GTLCore::Function* m_function;
        std::list<Expression*> m_arguments;
    };
    /**
     * @internal
     * This \ref Expression is used to hold a string (i.e. "my string")
     *
     * In OpenGTL Strings are not really uses except for debug usage in a print statement
     *
     * @ingroup GTLCore_AST
     */
    class StringExpression : public Expression {
      public:
        /**
         * Construct a string expression.
         * @param _string string value
         */
        explicit StringExpression( const GTLCore::String& _string ) : m_string(_string)
        {
        }
        /**
         */
        virtual const GTLCore::Type* type() const { return 0; }
        virtual bool isConstant() const { return true; }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual void markAsReturnExpression() {}
      private:
        GTLCore::String m_string;
    };
  }

}

#endif
