/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var ENV = require( '@stdlib/process/env' );
var kstest = require( '@stdlib/stats/kstest' );
var now = require( '@stdlib/time/now' );
var isnan = require( '@stdlib/math/base/assert/is-nan' );
var isUint32Array = require( '@stdlib/assert/is-uint32array' );
var UINT32_MAX = require( '@stdlib/constants/uint32/max' );
var Uint32Array = require( '@stdlib/array/uint32' );
var typedarray2json = require( '@stdlib/array/to-json' );
var minstd = require( './../../../base/minstd' );
var factory = require( './../lib/factory.js' );


// VARIABLES //

var opts = {
	'skip': ( ENV.TEST_MODE === 'coverage' )
};


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof factory, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function throws an error if minimum support `a` is not a number primitive', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		null,
		true,
		false,
		void 0,
		NaN,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( value, 2.0 );
		};
	}
});

tape( 'the function throws an error if maximum support `b` is not a number primitive', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		null,
		true,
		false,
		void 0,
		NaN,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( 2.0, value );
		};
	}
});

tape( 'the function throws an error if minimum support `a` is greater than or equal to maximum support `b`', function test( t ) {
	var values;
	var i;

	values = [
		[ 0.0, 0.0 ],
		[ -2.0, -4.0 ],
		[ 2.0, 1.0 ]
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( arr ) {
		return function badValue() {
			factory( arr[0], arr[1] );
		};
	}
});

tape( 'the function throws an error if provided an options argument which is not an object (no other arguments)', function test( t ) {
	var values;
	var i;

	values = [
		'abc',
		5,
		null,
		true,
		false,
		void 0,
		NaN,
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( value );
		};
	}
});

tape( 'the function throws an error if provided an options argument which is not an object (other arguments)', function test( t ) {
	var values;
	var i;

	values = [
		'abc',
		5,
		null,
		true,
		false,
		void 0,
		NaN,
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( 0.0, 1.0, value );
		};
	}
});

tape( 'if provided a `prng` option which is not a function, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'prng': value
			});
		};
	}
});

tape( 'if provided a `prng` option which is not a function, the function throws an error (other arguments)', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( 0.0, 1.0, {
				'prng': value
			});
		};
	}
});

tape( 'if provided a `copy` option which is not a boolean, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'copy': value
			});
		};
	}
});

tape( 'if provided a `seed` which is not a positive integer or a non-empty array-like object, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		0.0,
		-5.0,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'seed': value
			});
		};
	}
});

tape( 'the function throws a range error if provided a `seed` which is an integer greater than the maximum unsigned 32-bit integer', function test( t ) {
	var values;
	var i;

	values = [
		UINT32_MAX + 1,
		UINT32_MAX + 2,
		UINT32_MAX + 3
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws a range error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'seed': value
			});
		};
	}
});

tape( 'if provided a `state` option which is not a Uint32Array, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'state': value
			});
		};
	}
});

tape( 'if provided an invalid `state` option, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		new Uint32Array( 0 ),
		new Uint32Array( 10 ),
		new Uint32Array( 100 )
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'state': value
			});
		};
	}
});

tape( 'the function returns a pseudorandom number generator (no seed)', function test( t ) {
	var uniform;
	var r;
	var i;

	// When binding distribution parameters...
	uniform = factory( -10.0, 10.0 );
	for ( i = 0; i < 100; i++ ) {
		r = uniform();
		t.equal( typeof r, 'number', 'returns a number' );
	}

	// Without binding distribution parameters...
	uniform = factory();
	for ( i = 0; i < 100; i++ ) {
		r = uniform( -10.0, 10.0 );
		t.equal( typeof r, 'number', 'returns a number' );
	}
	t.end();
});

tape( 'the function returns a seeded pseudorandom number generator (integer seed)', function test( t ) {
	var uniform1;
	var uniform2;
	var seed;
	var r1;
	var r2;
	var i;

	seed = now();

	uniform1 = factory( 0.0, 4.0, {
		'seed': seed
	});
	uniform2 = factory( 0.0, 4.0, {
		'seed': seed
	});

	t.notEqual( uniform1, uniform2, 'separate generators' );

	for ( i = 0; i < 100; i++ ) {
		r1 = uniform1();
		r2 = uniform2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'the function returns a seeded pseudorandom number generator (array seed)', function test( t ) {
	var uniform1;
	var uniform2;
	var seed;
	var r1;
	var r2;
	var i;

	seed = [ now()+1, now()+2, now()+3 ];

	uniform1 = factory( 0.0, 4.0, {
		'seed': seed
	});
	uniform2 = factory( 0.0, 4.0, {
		'seed': seed
	});

	t.notEqual( uniform1, uniform2, 'separate generators' );

	for ( i = 0; i < 100; i++ ) {
		r1 = uniform1();
		r2 = uniform2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'attached to the returned function is the generator name', function test( t ) {
	var uniform = factory();
	t.equal( uniform.NAME, 'uniform', 'has property' );
	t.end();
});

tape( 'attached to the returned function is the underlying PRNG', function test( t ) {
	var uniform = factory();
	t.equal( typeof uniform.PRNG, 'function', 'has property' );

	uniform = factory({
		'prng': minstd.normalized
	});
	t.equal( uniform.PRNG, minstd.normalized, 'has property' );
	t.end();
});

tape( 'attached to the returned function is the generator seed (integer seed)', function test( t ) {
	var uniform = factory({
		'seed': 12345
	});
	t.equal( isUint32Array( uniform.seed ), true, 'has property' );
	t.equal( uniform.seed[ 0 ], 12345, 'equal to provided seed' );

	uniform = factory({
		'seed': 12345,
		'prng': minstd.normalized
	});
	t.equal( uniform.seed, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator seed (array seed)', function test( t ) {
	var actual;
	var rand;
	var seed;
	var i;

	seed = [ 1234, 5678 ];
	rand = factory({
		'seed': seed
	});

	actual = rand.seed;
	t.equal( isUint32Array( actual ), true, 'has property' );
	for ( i = 0; i < seed.length; i++ ) {
		t.equal( actual[ i ], seed[ i ], 'returns expected value for word '+i );
	}
	t.end();
});

tape( 'attached to the returned function is the generator seed length', function test( t ) {
	var uniform = factory();
	t.equal( typeof uniform.seedLength, 'number', 'has property' );

	uniform = factory({
		'prng': minstd.normalized
	});
	t.equal( uniform.seedLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state', function test( t ) {
	var uniform = factory();
	t.equal( isUint32Array( uniform.state ), true, 'has property' );

	uniform = factory({
		'prng': minstd.normalized
	});
	t.equal( uniform.state, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state length', function test( t ) {
	var uniform = factory();
	t.equal( typeof uniform.stateLength, 'number', 'has property' );

	uniform = factory({
		'prng': minstd.normalized
	});
	t.equal( uniform.stateLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state size', function test( t ) {
	var uniform = factory();
	t.equal( typeof uniform.byteLength, 'number', 'has property' );

	uniform = factory({
		'prng': minstd.normalized
	});
	t.equal( uniform.byteLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is a method to serialize the generator as a JSON object', function test( t ) {
	var uniform;
	var o;

	uniform = factory();
	t.equal( typeof uniform.toJSON, 'function', 'has method' );

	o = uniform.toJSON();
	t.equal( o.type, 'PRNG', 'has property' );
	t.equal( o.name, uniform.NAME, 'has property' );
	t.deepEqual( o.state, typedarray2json( uniform.state ), 'has property' );
	t.deepEqual( o.params, [], 'has property' );

	uniform = factory( 1.0, 2.0 );
	o = uniform.toJSON();

	t.deepEqual( o.params, [ 1.0, 2.0 ], 'has property' );

	uniform = factory({
		'prng': minstd.normalized
	});
	t.equal( typeof uniform.toJSON, 'function', 'has method' );
	t.equal( uniform.toJSON(), null, 'returns expected value' );

	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if `a` is `NaN`', function test( t ) {
	var uniform;
	var r;

	uniform = factory();
	r = uniform( NaN, 5.0 );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if `b` is `NaN`', function test( t ) {
	var uniform;
	var r;

	uniform = factory();
	r = uniform( 0.0, NaN );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if both `a` and `b` are `NaN`', function test( t ) {
	var uniform;
	var r;

	uniform = factory();
	r = uniform( NaN, NaN );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if minimum support `a` is greater than or equal to maximum support `b`', function test( t ) {
	var uniform;
	var values;
	var r;
	var i;

	uniform = factory();
	values = [
		[ 2.0, 1.0 ],
		[ -3.0, -4.0 ],
		[ 0.0, 0.0 ]
	];

	for ( i = 0; i < values.length; i++ ) {
		r = uniform( values[ i ][ 0 ], values[ i ][ 1 ] );
		t.strictEqual( isnan( r ), true, 'returns `NaN` when provided '+values[ i ] );
	}
	t.end();
});

tape( 'the function supports specifying the underlying PRNG', function test( t ) {
	var uniform;
	var r;
	var i;

	uniform = factory( 1.0, 2.0, {
		'prng': minstd.normalized
	});

	for ( i = 0; i < 1e2; i++ ) {
		r = uniform();
		t.equal( typeof r, 'number', 'returns a number' );
	}
	t.end();
});

tape( 'the function supports specifying the underlying PRNG (parameters)', function test( t ) {
	var uniform;
	var r;
	var i;

	uniform = factory({
		'prng': minstd.normalized
	});

	for ( i = 0; i < 1e2; i++ ) {
		r = uniform( 1.0, 2.0 );
		t.equal( typeof r, 'number', 'returns a number' );
	}
	t.end();
});

tape( 'the function supports providing a seeded underlying PRNG', function test( t ) {
	var uniform1;
	var uniform2;
	var randu;
	var seed;
	var r1;
	var r2;
	var i;

	seed = now();

	randu = minstd.factory({
		'seed': seed
	});
	uniform1 = factory( 1.0, 2.0, {
		'prng': randu.normalized
	});

	randu = minstd.factory({
		'seed': seed
	});
	uniform2 = factory( 1.0, 2.0, {
		'prng': randu.normalized
	});

	t.notEqual( uniform1, uniform2, 'separate generators' );

	for ( i = 0; i < 1e2; i++ ) {
		r1 = uniform1();
		r2 = uniform2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'the function supports providing a seeded underlying PRNG (parameters)', function test( t ) {
	var uniform1;
	var uniform2;
	var randu;
	var seed;
	var r1;
	var r2;
	var i;

	seed = now();

	randu = minstd.factory({
		'seed': seed
	});
	uniform1 = factory({
		'prng': randu.normalized
	});

	randu = minstd.factory({
		'seed': seed
	});
	uniform2 = factory({
		'prng': randu.normalized
	});

	t.notEqual( uniform1, uniform2, 'separate generators' );

	for ( i = 0; i < 1e2; i++ ) {
		r1 = uniform1( 1.0, 2.0 );
		r2 = uniform2( 1.0, 2.0 );
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'the function returns a PRNG for generating random numbers from a uniform distribution', opts, function test( t ) {
	var threshold;
	var count;
	var npass;
	var N;
	var a;
	var b;
	var x;

	threshold = 0.10;

	a = 2.0;
	b = 4.0;

	x = new Array( 1e3 );
	N = 500;

	count = -1;
	npass = 0;

	gof();

	function gof() {
		var rejected;
		var uniform;
		var pValue;
		var bool;
		var i;
		var j;

		count += 1;
		rejected = 0;
		for ( i = 0; i < N; i++ ) {
			uniform = factory( a, b );
			t.ok( true, 'seed: '+uniform.seed );
			for ( j = 0; j < x.length; j++ ) {
				x[ j ] = uniform();
				if ( x[ j ] < a || x[ j ] > b ) {
					t.ok( false, 'returned a number outside support: '+x[ j ] );
				}
			}
			// Test using Kolmogorov-Smirnov goodness-of-fit test:
			pValue = kstest( x, 'uniform', a, b ).pValue;
			t.equal( typeof pValue, 'number', 'returns a p-value: '+pValue );
			if ( pValue < 0.05 ) {
				rejected += 1;
			}
		}
		// Account for small sample size and few repeats...
		bool = ( rejected / N < threshold );

		// If we succeed the first time, we are done...
		if ( count === 0 && bool ) {
			return done( bool, rejected );
		}
		// Retry mode...
		if ( bool ) {
			npass += 1;
		}
		// Retry twice...
		if ( count < 2 ) {
			return gof();
		}
		// Both retries must succeed for test to pass:
		bool = ( npass >= 2 );
		return done( bool, rejected );
	}

	function done( bool, rejected ) {
		t.ok( bool, 'null hypothesis (i.e., that numbers are drawn from Uniform('+a+','+b+')) is rejected in less than '+(threshold*100)+'% of cases ('+rejected+' of '+N+'). Repeats: '+npass+' of '+count+'.' );
		t.end();
	}
});

tape( 'the function supports specifying the generator state', function test( t ) {
	var uniform;
	var state;
	var arr;
	var i;

	uniform = factory( 0.0, 3.14 );

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		uniform();
	}
	// Capture the current state:
	state = uniform.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( uniform() );
	}

	// Create another PRNG using the captured state:
	uniform = factory( 0.0, 3.14, {
		'state': state
	});

	// Replay previously generated values...
	for ( i = 0; i < 100; i++ ) {
		t.equal( uniform(), arr[ i ], 'returns expected value. i: '+i+'.' );
	}
	t.end();
});

tape( 'the function supports specifying a shared generator state', function test( t ) {
	var uniform;
	var shared;
	var state;
	var rand1;
	var rand2;
	var arr;
	var v1;
	var v2;
	var i;
	var j;

	uniform = factory( 0.0, 3.14 );

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		uniform();
	}
	// Capture the current state:
	state = uniform.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( uniform() );
	}

	// Create a copy of the state (to prevent mutation) which will be shared by more than one PRNG:
	shared = new Uint32Array( state );

	// Create PRNGs using the captured state:
	rand1 = factory( 0.0, 3.14, {
		'state': shared,
		'copy': false
	});
	rand2 = factory( 0.0, 3.14, {
		'state': shared,
		'copy': false
	});

	// Replay previously generated values...
	j = 0;
	for ( i = 0; i < 25; i++ ) {
		v1 = rand1();
		v2 = rand2();
		t.equal( v1, arr[ j ], 'returns expected value. i: '+j+'.' );
		t.equal( v2, arr[ j+1 ], 'returns expected value. i: '+(j+1)+'.' );
		j += 2; // stride
	}

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		v2 = rand2();
	}

	// Reset the (shared) state:
	rand1.state = state;

	// Replay previously generated values...
	j = 0;
	for ( i = 0; i < 25; i++ ) {
		v1 = rand1();
		v2 = rand2();
		t.equal( v1, arr[ j ], 'returns expected value. i: '+j+'.' );
		t.equal( v2, arr[ j+1 ], 'returns expected value. i: '+(j+1)+'.' );
		j += 2; // stride
	}
	t.end();
});

tape( 'the returned function supports setting the generator state', function test( t ) {
	var uniform;
	var state;
	var arr;
	var i;

	uniform = factory( 0.0, 3.14 );

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		uniform();
	}
	// Capture the current state:
	state = uniform.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( uniform() );
	}
	// Set the state:
	uniform.state = state;

	// Replay previously generated values...
	for ( i = 0; i < 100; i++ ) {
		t.equal( uniform(), arr[ i ], 'returns expected value. i: '+i+'.' );
	}
	t.end();
});
