/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * NAME
 * 	getpgid02.c
 *
 * DESCRIPTION
 *	Testcase to check the basic functionality of getpgid().
 *
 * ALGORITHM
 * 	test 1: Does getpgid(-99) and expects ESRCH.
 * 	test 2: Searches an unused pid and expects ESRCH.
 *
 * USAGE:  <for command-line>
 *  getpgid02 [-c n] [-e] [-i n] [-I x] [-P x] [-t]
 *     where,  -c n : Run n copies concurrently.
 *             -e   : Turn on errno logging.
 *             -i n : Execute test n times.
 *             -I x : Execute test for x seconds.
 *             -P x : Pause for x seconds between iterations.
 *             -t   : Turn on syscall timing.
 *
 * HISTORY
 *	07/2001 Ported by Wayne Boyer
 *
 * RESTRICTIONS
 *	none
 */

#include <sys/types.h>
#include <errno.h>
#include <sys/wait.h>
#include <test.h>
#include <usctest.h>

void setup(void);
void cleanup(void) __attribute__((noreturn));

const char *TCID = "getpgid01";
int TST_TOTAL = 2;
extern int Tst_count;

int 	pgid_0, pgid_1;
#define BADPID -99

int exp_enos[]={ESRCH, 0};

struct test_case_t {
        int *id;
        int error;
} TC[] = {
	/* The pid value is negative */
        {&pgid_0, ESRCH},

	/* The pid value does not match any process */
        {&pgid_1, ESRCH}
};

int
main(int ac, char **av)
{
	int lc;				/* loop counter */
	int i;
	const char *msg;		/* message returned by parse_opts */

	/* parse standard options */
	if ((msg = parse_opts(ac, av, (option_t *)NULL, NULL)) != (char *)NULL){
		tst_brkm(TBROK, cleanup, "OPTION PARSING ERROR - %s", msg);
	}

	setup();

	/* set up the expected errnos */
	TEST_EXP_ENOS(exp_enos);

	/* check looping state if -i option given */
	for (lc = 0; TEST_LOOPING(lc); lc++) {
		/* reset Tst_count in case we are looping */
		Tst_count = 0;


		/* loop through the test cases */
		for (i = 0; i < TST_TOTAL; i++) {

			TEST(getpgid(*TC[i].id));

                        if (TEST_RETURN != -1) {
                                tst_resm(TFAIL, "call succeeded unexpectedly");
                                continue;
                        }

                        TEST_ERROR_LOG(TEST_ERRNO);

                        if (TEST_ERRNO == TC[i].error) {
                                tst_resm(TPASS, "expected failure - "
                                         "errno = %d : %s", TEST_ERRNO,
                                         strerror(TEST_ERRNO));
                        } else {
                                tst_resm(TFAIL, "unexpected error - %d : %s - "
                                         "expected %d", TEST_ERRNO,
                                         strerror(TEST_ERRNO), TC[i].error);
			}
		}
	}
	cleanup();

	/*NOTREACHED*/
}

/*
 * setup() - performs all ONE TIME setup for this test.
 */
void
setup()
{
	/* capture signals */
	tst_sig(NOFORK, DEF_HANDLER, cleanup);

	/* Pause if that option was specified */
	TEST_PAUSE;

	pgid_0 = BADPID;

	/*
	 * Find a pid that isn't currently being used.  Start
	 * at 'our pid - 1' and go backwards until we find one.
	 * In this way, we can be reasonably sure that the pid
	 * we find won't be reused for a while as new pids are
	 * allocated counting up to PID_MAX.
	 */
	for (pgid_1 = getpid() - 1; --pgid_1 > 0; ) {
		if (kill(pgid_1, 0) < 0 && errno == ESRCH) {
			break;
		}
	}
}


/*
 * cleanup() - performs all ONE TIME cleanup for this test at
 *	       completion or premature exit.
 */
void
cleanup()
{
	/*
	 * print timing stats if that option was specified.
	 * print errno log if that option was specified.
	 */
	TEST_CLEANUP;

	/* exit with return code appropriate for results */
	tst_exit();
}

