/*
 * Contributed by Stephane Eranian <eranian@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 *
 * PMU: icx_unc_imc (IcelakeX Uncore IMC)
 * Based on Intel JSON event table version   : 1.21
 * Based on Intel JSON event table published : 06/06/2023
 */
static const intel_x86_umask_t icx_unc_m_act_count[]={
  { .uname   = "ALL",
    .udesc   = "All Activates",
    .ucode   = 0x0b00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "BYP",
    .udesc   = "Activate due to Bypass (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_cas_count[]={
  { .uname   = "ALL",
    .udesc   = "TBD",
    .ucode   = 0x3f00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "RD",
    .udesc   = "TBD",
    .ucode   = 0x0f00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD_PRE_REG",
    .udesc   = "DRAM RD_CAS commands w/auto-pre (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD_PRE_UNDERFILL",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD_REG",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD_UNDERFILL",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WR",
    .udesc   = "TBD",
    .ucode   = 0x3000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WR_NONPRE",
    .udesc   = "DRAM WR_CAS commands w/o auto-pre (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WR_PRE",
    .udesc   = "DRAM WR_CAS commands w/ auto-pre (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_dram_refresh[]={
  { .uname   = "HIGH",
    .udesc   = "TBD",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OPPORTUNISTIC",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PANIC",
    .udesc   = "TBD",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_pcls[]={
  { .uname   = "RD",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "TOTAL",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WR",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_pmm_cmd1[]={
  { .uname   = "ALL",
    .udesc   = "All",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "MISC",
    .udesc   = "Misc Commands (error, flow ACKs) (experimental)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MISC_GNT",
    .udesc   = "Misc GNTs (experimental)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD",
    .udesc   = "Reads - RPQ",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RPQ_GNTS",
    .udesc   = "RPQ GNTs (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "UFILL_RD",
    .udesc   = "Underfill reads",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WPQ_GNTS",
    .udesc   = "Underfill GNTs (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WR",
    .udesc   = "Writes",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_pmm_cmd2[]={
  { .uname   = "NODATA_EXP",
    .udesc   = "Expected No data packet (ERID matched NDP encoding) (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NODATA_UNEXP",
    .udesc   = "Unexpected No data packet (ERID matched a Read, but data was a NDP) (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OPP_RD",
    .udesc   = "Opportunistic Reads (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM_ECC_ERROR",
    .udesc   = "ECC Errors (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM_ERID_ERROR",
    .udesc   = "ERID detectable parity error (experimental)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM_ERID_STARVED",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REQS_SLOT0",
    .udesc   = "Read Requests - Slot 0 (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REQS_SLOT1",
    .udesc   = "Read Requests - Slot 1 (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_pmm_rpq_occupancy[]={
  { .uname   = "ALL",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "GNT_WAIT",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NO_GNT",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_pmm_wpq_occupancy[]={
  { .uname   = "ALL",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "CAS",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PWR",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_power_cke_cycles[]={
  { .uname   = "LOW_0",
    .udesc   = "DIMM ID (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LOW_1",
    .udesc   = "DIMM ID (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LOW_2",
    .udesc   = "DIMM ID (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LOW_3",
    .udesc   = "DIMM ID (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_power_throttle_cycles[]={
  { .uname   = "SLOT0",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SLOT1",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_pre_count[]={
  { .uname   = "ALL",
    .udesc   = "TBD",
    .ucode   = 0x1c00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "PAGE_MISS",
    .udesc   = "Precharge due to page miss (experimental)",
    .ucode   = 0x0c00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PGT",
    .udesc   = "Precharge due to page table",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD",
    .udesc   = "Precharge due to read",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WR",
    .udesc   = "Precharge due to write",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_rpq_inserts[]={
  { .uname   = "PCH0",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PCH1",
    .udesc   = "TBD",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_accesses[]={
  { .uname   = "ACCEPTS",
    .udesc   = "Scoreboard Accesses Accepted (experimental)",
    .ucode   = 0x0500ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMRD_CMPS",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMWR_CMPS",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_RD_CMPS",
    .udesc   = "Write Accepts (experimental)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_WR_CMPS",
    .udesc   = "Write Rejects (experimental)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NMRD_CMPS",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NMWR_CMPS",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_RD_CMPS",
    .udesc   = "FM read completions (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_WR_CMPS",
    .udesc   = "FM write completions (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD_ACCEPTS",
    .udesc   = "Read Accepts (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD_REJECTS",
    .udesc   = "Read Rejects (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REJECTS",
    .udesc   = "Scoreboard Accesses Rejected (experimental)",
    .ucode   = 0x0a00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WR_ACCEPTS",
    .udesc   = "NM read completions (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WR_REJECTS",
    .udesc   = "NM write completions (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_canary[]={
  { .uname   = "ALLOC",
    .udesc   = "Alloc (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEALLOC",
    .udesc   = "Dealloc (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMRD_STARVED",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMTGRWR_STARVED",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMWR_STARVED",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_RD_STARVED",
    .udesc   = "Near Mem Write Starved (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_TGR_WR_STARVED",
    .udesc   = "Far Mem Write Starved (experimental)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_WR_STARVED",
    .udesc   = "Far Mem Read Starved (experimental)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NMRD_STARVED",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NMWR_STARVED",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_RD_STARVED",
    .udesc   = "Valid (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_WR_STARVED",
    .udesc   = "Near Mem Read Starved (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "VLD",
    .udesc   = "Reject (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_inserts[]={
  { .uname   = "BLOCK_RDS",
    .udesc   = "Block region reads (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "BLOCK_WRS",
    .udesc   = "Block region writes (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM_RDS",
    .udesc   = "Persistent Mem reads (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM_WRS",
    .udesc   = "Persistent Mem writes (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RDS",
    .udesc   = "Reads (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WRS",
    .udesc   = "Writes (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_occupancy[]={
  { .uname   = "BLOCK_RDS",
    .udesc   = "Block region reads (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "BLOCK_WRS",
    .udesc   = "Block region writes (experimental)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM_RDS",
    .udesc   = "Persistent Mem reads (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM_WRS",
    .udesc   = "Persistent Mem writes (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RDS",
    .udesc   = "Reads (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_pref_inserts[]={
  { .uname   = "ALL",
    .udesc   = "All (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "DDR",
    .udesc   = "DDR4 (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM",
    .udesc   = "Persistent Mem (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_pref_occupancy[]={
  { .uname   = "ALL",
    .udesc   = "All (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "DDR",
    .udesc   = "DDR4 (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMEM",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM",
    .udesc   = "Persistent Mem (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_reject[]={
  { .uname   = "CANARY",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DDR_EARLY_CMP",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_ADDR_CNFLT",
    .udesc   = "FM requests rejected due to full address conflict (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_SET_CNFLT",
    .udesc   = "NM requests rejected due to set conflict (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PATROL_SET_CNFLT",
    .udesc   = "Patrol requests rejected due to set conflict (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_strv_dealloc[]={
  { .uname   = "FMRD",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMTGR",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMWR",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_RD",
    .udesc   = "Far Mem Read - Set (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_TGR",
    .udesc   = "Near Mem Read - Clear (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_WR",
    .udesc   = "Far Mem Write - Set (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NMRD",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NMWR",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_RD",
    .udesc   = "Near Mem Read - Set (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_WR",
    .udesc   = "Near Mem Write - Set (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_strv_occ[]={
  { .uname   = "FMRD",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMTGR",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FMWR",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_RD",
    .udesc   = "Far Mem Read (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_TGR",
    .udesc   = "Near Mem Read - Clear (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FM_WR",
    .udesc   = "Far Mem Write (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NMRD",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NMWR",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_RD",
    .udesc   = "Near Mem Read (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_WR",
    .udesc   = "Near Mem Write (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_sb_tagged[]={
  { .uname   = "DDR4_CMP",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NEW",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OCC",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM0_CMP",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM1_CMP",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PMM2_CMP",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD_HIT",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RD_MISS",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_tagchk[]={
  { .uname   = "HIT",
    .udesc   = "Hit in Near Memory Cache",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MISS_CLEAN",
    .udesc   = "Miss, no data in this line",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MISS_DIRTY",
    .udesc   = "Miss, existing data may be evicted to Far Memory",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_RD_HIT",
    .udesc   = "Read Hit in Near Memory Cache",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NM_WR_HIT",
    .udesc   = "Write Hit in Near Memory Cache",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_wpq_inserts[]={
  { .uname   = "PCH0",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PCH1",
    .udesc   = "TBD",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t icx_unc_m_wpq_write_hit[]={
  { .uname   = "PCH0",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PCH1",
    .udesc   = "TBD (experimental)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_entry_t intel_icx_unc_imc_pe[]={
  { .name   = "UNC_M_ACT_COUNT",
    .desc   = "DRAM Activate Count",
    .code   = 0x0001,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_act_count),
    .umasks = icx_unc_m_act_count,
  },
  { .name   = "UNC_M_CAS_COUNT",
    .desc   = "All DRAM read CAS commands issued (including underfills)",
    .code   = 0x0004,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_cas_count),
    .umasks = icx_unc_m_cas_count,
  },
  { .name   = "UNC_M_CLOCKTICKS",
    .desc   = "DRAM Clockticks",
    .code   = 0x0000,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_CLOCKTICKS_FREERUN",
    .desc   = "Free running counter that increments for the Memory Controller (experimental)",
    .code   = 0x0000,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0x10ull,
  },
  { .name   = "UNC_M_DRAM_PRE_ALL",
    .desc   = "DRAM Precharge All Commands (experimental)",
    .code   = 0x0044,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_DRAM_REFRESH",
    .desc   = "Number of DRAM Refreshes Issued",
    .code   = 0x0045,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_dram_refresh),
    .umasks = icx_unc_m_dram_refresh,
  },
  { .name   = "UNC_M_HCLOCKTICKS",
    .desc   = "Half clockticks for IMC",
    .code   = 0x0000,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0x1ull,
  },
  { .name   = "UNC_M_PARITY_ERRORS",
    .desc   = "UNC_M_PARITY_ERRORS (experimental)",
    .code   = 0x002c,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PCLS",
    .desc   = "UNC_M_PCLS.RD",
    .code   = 0x00a0,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_pcls),
    .umasks = icx_unc_m_pcls,
  },
  { .name   = "UNC_M_PMM_CMD1",
    .desc   = "PMM Commands",
    .code   = 0x00ea,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_pmm_cmd1),
    .umasks = icx_unc_m_pmm_cmd1,
  },
  { .name   = "UNC_M_PMM_CMD2",
    .desc   = "PMM Commands - Part 2",
    .code   = 0x00eb,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_pmm_cmd2),
    .umasks = icx_unc_m_pmm_cmd2,
  },
  { .name   = "UNC_M_PMM_RPQ_CYCLES_FULL",
    .desc   = "PMM Read Queue Cycles Full (experimental)",
    .code   = 0x00e2,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PMM_RPQ_CYCLES_NE",
    .desc   = "PMM Read Queue Cycles Not Empty (experimental)",
    .code   = 0x00e1,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PMM_RPQ_INSERTS",
    .desc   = "PMM Read Queue Inserts",
    .code   = 0x00e3,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PMM_RPQ_OCCUPANCY",
    .desc   = "PMM Read Pending Queue Occupancy",
    .code   = 0x00e0,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_pmm_rpq_occupancy),
    .umasks = icx_unc_m_pmm_rpq_occupancy,
  },
  { .name   = "UNC_M_PMM_WPQ_CYCLES_FULL",
    .desc   = "PMM Write Queue Cycles Full (experimental)",
    .code   = 0x00e6,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PMM_WPQ_CYCLES_NE",
    .desc   = "PMM Write Queue Cycles Not Empty (experimental)",
    .code   = 0x00e5,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PMM_WPQ_FLUSH",
    .desc   = "UNC_M_PMM_WPQ_FLUSH (experimental)",
    .code   = 0x00e8,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PMM_WPQ_FLUSH_CYC",
    .desc   = "UNC_M_PMM_WPQ_FLUSH_CYC (experimental)",
    .code   = 0x00e9,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PMM_WPQ_INSERTS",
    .desc   = "PMM Write Queue Inserts",
    .code   = 0x00e7,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_PMM_WPQ_OCCUPANCY",
    .desc   = "PMM Write Pending Queue Occupancy",
    .code   = 0x00e4,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_pmm_wpq_occupancy),
    .umasks = icx_unc_m_pmm_wpq_occupancy,
  },
  { .name   = "UNC_M_POWER_CHANNEL_PPD",
    .desc   = "Channel PPD Cycles (experimental)",
    .code   = 0x0085,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_POWER_CKE_CYCLES",
    .desc   = "CKE_ON_CYCLES by Rank",
    .code   = 0x0047,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_power_cke_cycles),
    .umasks = icx_unc_m_power_cke_cycles,
  },
  { .name   = "UNC_M_POWER_CRIT_THROTTLE_CYCLES",
    .desc   = "Throttle Cycles for Rank 0",
    .code   = 0x0086,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_power_throttle_cycles), /* shared */
    .umasks = icx_unc_m_power_throttle_cycles,
  },
  { .name   = "UNC_M_POWER_SELF_REFRESH",
    .desc   = "Clock-Enabled Self-Refresh (experimental)",
    .code   = 0x0043,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_POWER_THROTTLE_CYCLES",
    .desc   = "Throttle Cycles for Rank 0",
    .code   = 0x0046,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_power_throttle_cycles),
    .umasks = icx_unc_m_power_throttle_cycles,
  },
  { .name   = "UNC_M_PRE_COUNT",
    .desc   = "DRAM Precharge commands.",
    .code   = 0x0002,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_pre_count),
    .umasks = icx_unc_m_pre_count,
  },
  { .name   = "UNC_M_RDB_FULL",
    .desc   = "Read Data Buffer Full (experimental)",
    .code   = 0x0019,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_RDB_INSERTS",
    .desc   = "Read Data Buffer Inserts (experimental)",
    .code   = 0x0017,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_RDB_NOT_EMPTY",
    .desc   = "Read Data Buffer Not Empty (experimental)",
    .code   = 0x0018,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_RDB_OCCUPANCY",
    .desc   = "Read Data Buffer Occupancy (experimental)",
    .code   = 0x001a,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_RPQ_CYCLES_FULL_PCH0",
    .desc   = "Read Pending Queue Full Cycles (experimental)",
    .code   = 0x0012,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_RPQ_CYCLES_FULL_PCH1",
    .desc   = "Read Pending Queue Full Cycles (experimental)",
    .code   = 0x0015,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_RPQ_CYCLES_NE",
    .desc   = "Read Pending Queue Not Empty",
    .code   = 0x0011,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_rpq_inserts), /* shared */
    .umasks = icx_unc_m_rpq_inserts,
  },
  { .name   = "UNC_M_RPQ_INSERTS",
    .desc   = "Read Pending Queue Allocations",
    .code   = 0x0010,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_rpq_inserts),
    .umasks = icx_unc_m_rpq_inserts,
  },
  { .name   = "UNC_M_RPQ_OCCUPANCY_PCH0",
    .desc   = "Read Pending Queue Occupancy",
    .code   = 0x0080,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_RPQ_OCCUPANCY_PCH1",
    .desc   = "Read Pending Queue Occupancy",
    .code   = 0x0081,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_SB_ACCESSES",
    .desc   = "Scoreboard Accesses",
    .code   = 0x00d2,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_accesses),
    .umasks = icx_unc_m_sb_accesses,
  },
  { .name   = "UNC_M_SB_CANARY",
    .desc   = "TBD",
    .code   = 0x00d9,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_canary),
    .umasks = icx_unc_m_sb_canary,
  },
  { .name   = "UNC_M_SB_CYCLES_FULL",
    .desc   = "Scoreboard Cycles Full (experimental)",
    .code   = 0x00d1,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_SB_CYCLES_NE",
    .desc   = "Scoreboard Cycles Not-Empty (experimental)",
    .code   = 0x00d0,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_SB_INSERTS",
    .desc   = "Scoreboard Inserts",
    .code   = 0x00d6,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_inserts),
    .umasks = icx_unc_m_sb_inserts,
  },
  { .name   = "UNC_M_SB_OCCUPANCY",
    .desc   = "Scoreboard Occupancy",
    .code   = 0x00d5,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_occupancy),
    .umasks = icx_unc_m_sb_occupancy,
  },
  { .name   = "UNC_M_SB_PREF_INSERTS",
    .desc   = "Scoreboard Prefetch Inserts",
    .code   = 0x00da,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_pref_inserts),
    .umasks = icx_unc_m_sb_pref_inserts,
  },
  { .name   = "UNC_M_SB_PREF_OCCUPANCY",
    .desc   = "Scoreboard Prefetch Occupancy",
    .code   = 0x00db,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_pref_occupancy),
    .umasks = icx_unc_m_sb_pref_occupancy,
  },
  { .name   = "UNC_M_SB_REJECT",
    .desc   = "Number of Scoreboard Requests Rejected",
    .code   = 0x00d4,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_reject),
    .umasks = icx_unc_m_sb_reject,
  },
  { .name   = "UNC_M_SB_STRV_ALLOC",
    .desc   = "This event is deprecated. Refer to new event UNC_M_SB_STRV_ALLOC.NM_RD",
    .code   = 0x00d7,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_strv_dealloc), /* shared */
    .umasks = icx_unc_m_sb_strv_dealloc,
  },
  { .name   = "UNC_M_SB_STRV_DEALLOC",
    .desc   = "This event is deprecated. Refer to new event UNC_M_SB_STRV_DEALLOC.NM_RD",
    .code   = 0x00de,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_strv_dealloc),
    .umasks = icx_unc_m_sb_strv_dealloc,
  },
  { .name   = "UNC_M_SB_STRV_OCC",
    .desc   = "This event is deprecated. Refer to new event UNC_M_SB_STRV_OCC.NM_RD",
    .code   = 0x00d8,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_strv_occ),
    .umasks = icx_unc_m_sb_strv_occ,
  },
  { .name   = "UNC_M_SB_TAGGED",
    .desc   = "UNC_M_SB_TAGGED.NEW",
    .code   = 0x00dd,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_sb_tagged),
    .umasks = icx_unc_m_sb_tagged,
  },
  { .name   = "UNC_M_TAGCHK",
    .desc   = "2LM Tag Check",
    .code   = 0x00d3,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_tagchk),
    .umasks = icx_unc_m_tagchk,
  },
  { .name   = "UNC_M_WPQ_CYCLES_FULL_PCH0",
    .desc   = "Write Pending Queue Full Cycles (experimental)",
    .code   = 0x0022,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_WPQ_CYCLES_FULL_PCH1",
    .desc   = "Write Pending Queue Full Cycles (experimental)",
    .code   = 0x0016,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_WPQ_CYCLES_NE",
    .desc   = "Write Pending Queue Not Empty",
    .code   = 0x0021,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_wpq_inserts), /* shared */
    .umasks = icx_unc_m_wpq_inserts,
  },
  { .name   = "UNC_M_WPQ_INSERTS",
    .desc   = "Write Pending Queue Allocations",
    .code   = 0x0020,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_wpq_inserts),
    .umasks = icx_unc_m_wpq_inserts,
  },
  { .name   = "UNC_M_WPQ_OCCUPANCY_PCH0",
    .desc   = "Write Pending Queue Occupancy",
    .code   = 0x0082,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_WPQ_OCCUPANCY_PCH1",
    .desc   = "Write Pending Queue Occupancy",
    .code   = 0x0083,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
  },
  { .name   = "UNC_M_WPQ_READ_HIT",
    .desc   = "Write Pending Queue CAM Match",
    .code   = 0x0023,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_wpq_write_hit), /* shared */
    .umasks = icx_unc_m_wpq_write_hit,
  },
  { .name   = "UNC_M_WPQ_WRITE_HIT",
    .desc   = "Write Pending Queue CAM Match",
    .code   = 0x0024,
    .modmsk = ICX_UNC_IMC_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(icx_unc_m_wpq_write_hit),
    .umasks = icx_unc_m_wpq_write_hit,
  },
};
/* 59 events available */
