/*
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Some functions in this file use the VSOP87 solution by
 *  Messrs. Bretagnon and Francou.
 * 
 *  Copyright (C) 2000 - 2005 Liam Girdwood <lgirdwood@gmail.com>
 */

#include <math.h>
#include <libnova/venus.h>
#include <libnova/vsop87.h>
#include <libnova/solar.h>
#include <libnova/earth.h>
#include <libnova/transform.h>
#include <libnova/rise_set.h>
#include <libnova/utility.h>

#define LONG_L0 416
#define LONG_L1 235
#define LONG_L2 72
#define LONG_L3 7
#define LONG_L4 4
#define LONG_L5 2
#define LAT_B0 210
#define LAT_B1 121
#define LAT_B2 51
#define LAT_B3 12
#define LAT_B4 4
#define LAT_B5 4
#define RADIUS_R0 323
#define RADIUS_R1 174
#define RADIUS_R2 62
#define RADIUS_R3 8
#define RADIUS_R4 3
#define RADIUS_R5 2

/* cache variables */
static double cJD = 0.0, cL = 0.0, cB = 0.0, cR = 0.0;

static const struct ln_vsop ALIGN32 venus_longitude_l0[LONG_L0] = {
    {     3.17614666774,  0.00000000000,        0.00000000000}, 
    {     0.01353968419,  5.59313319619,    10213.28554621100}, 
    {     0.00089891645,  5.30650047764,    20426.57109242200}, 
    {     0.00005477194,  4.41630661466,     7860.41939243920}, 
    {     0.00003455741,  2.69964447820,    11790.62908865880}, 
    {     0.00002372061,  2.99377542079,     3930.20969621960}, 
    {     0.00001317168,  5.18668228402,       26.29831979980}, 
    {     0.00001664146,  4.25018630147,     1577.34354244780}, 
    {     0.00001438387,  4.15745084182,     9683.59458111640}, 
    {     0.00001200521,  6.15357116043,    30639.85663863300}, 
    {     0.00000761380,  1.95014701047,      529.69096509460}, 
    {     0.00000707676,  1.06466702668,      775.52261132400}, 
    {     0.00000584836,  3.99839888230,      191.44826611160}, 
    {     0.00000769314,  0.81629615196,     9437.76293488700}, 
    {     0.00000499915,  4.12340212820,    15720.83878487840}, 
    {     0.00000326221,  4.59056477038,    10404.73381232260}, 
    {     0.00000429498,  3.58642858577,    19367.18916223280}, 
    {     0.00000326967,  5.67736584311,     5507.55323866740}, 
    {     0.00000231937,  3.16251059356,     9153.90361602180}, 
    {     0.00000179695,  4.65337908917,     1109.37855209340}, 
    {     0.00000128263,  4.22604490814,       20.77539549240}, 
    {     0.00000155464,  5.57043891690,    19651.04848109800}, 
    {     0.00000127907,  0.96209781904,     5661.33204915220}, 
    {     0.00000105547,  1.53721203088,      801.82093112380}, 
    {     0.00000085722,  0.35589247720,     3154.68708489560}, 
    {     0.00000099121,  0.83288208931,      213.29909543800}, 
    {     0.00000098804,  5.39389623302,    13367.97263110660}, 
    {     0.00000082094,  3.21597037872,    18837.49819713819}, 
    {     0.00000088031,  3.88868864136,     9999.98645077300}, 
    {     0.00000071577,  0.11145736657,    11015.10647733480}, 
    {     0.00000056122,  4.24039842051,        7.11354700080}, 
    {     0.00000070239,  0.67458825333,    23581.25817731760}, 
    {     0.00000050796,  0.24531639097,    11322.66409830440}, 
    {     0.00000046111,  5.31576442737,    18073.70493865020}, 
    {     0.00000044576,  6.06281108312,    40853.14218484400}, 
    {     0.00000042594,  5.32873395426,     2352.86615377180}, 
    {     0.00000042635,  1.79955442721,     7084.89678111520}, 
    {     0.00000041177,  0.36241012200,      382.89653222320}, 
    {     0.00000035749,  2.70448479527,    10206.17199921020}, 
    {     0.00000033893,  2.02347385644,     6283.07584999140}, 
    {     0.00000029170,  3.59117396909,    22003.91463486980}, 
    {     0.00000028479,  2.22375430133,     1059.38193018920}, 
    {     0.00000029850,  4.02177029338,    10239.58386601080}, 
    {     0.00000033252,  2.10025580495,    27511.46787353720}, 
    {     0.00000030172,  4.94191918273,    13745.34623902240}, 
    {     0.00000029252,  3.51392387787,      283.85931886520}, 
    {     0.00000024424,  2.70177487840,     8624.21265092720}, 
    {     0.00000020274,  3.79493777545,    14143.49524243060}, 
    {     0.00000024322,  4.27814493315,        5.52292430740}, 
    {     0.00000026260,  0.54067510171,    17298.18232732620}, 
    {     0.00000020492,  0.58547075036,       38.02767263580}, 
    {     0.00000018988,  4.13811517967,     4551.95349705880}, 
    {     0.00000023739,  4.82870820701,     6872.67311951120}, 
    {     0.00000015953,  1.50376176156,     8635.94200376320}, 
    {     0.00000019069,  6.12025555817,    29050.78374334920}, 
    {     0.00000018269,  3.04740409161,    19999.97290154599}, 
    {     0.00000013656,  4.41336264990,     3532.06069281140}, 
    {     0.00000017118,  3.51922693724,    31441.67756975680}, 
    {     0.00000010955,  2.84562940868,    18307.80723204360}, 
    {     0.00000011048,  2.58361219121,     9786.68735533500}, 
    {     0.00000009904,  1.08737985358,     7064.12138562280}, 
    {     0.00000010576,  0.85419798194,    10596.18207843420}, 
    {     0.00000009235,  5.52461085424,    12566.15169998280}, 
    {     0.00000011599,  5.81007484555,    19896.88012732740}, 
    {     0.00000011807,  1.91250004145,    21228.39202354580}, 
    {     0.00000010105,  2.34270729521,    10742.97651130560}, 
    {     0.00000008154,  1.92331359797,       15.25247118500}, 
    {     0.00000008893,  1.97291419659,    10186.98722641120}, 
    {     0.00000009352,  4.94508838657,    35371.88726597640}, 
    {     0.00000006821,  4.39733528050,     8662.24032356300}, 
    {     0.00000006688,  1.55309955053,    14945.31617355440}, 
    {     0.00000006413,  2.17677578364,    10988.80815753500}, 
    {     0.00000005802,  1.93461898145,     3340.61242669980}, 
    {     0.00000005950,  2.96578177047,     4732.03062734340}, 
    {     0.00000005275,  5.01875399411,    28286.99048486120}, 
    {     0.00000007047,  1.00111452053,      632.78373931320}, 
    {     0.00000005048,  4.27886655804,    29580.47470844380}, 
    {     0.00000006305,  0.35506330531,      103.09277421860}, 
    {     0.00000005959,  5.04792949123,      245.83164622940}, 
    {     0.00000004651,  0.85216995524,     6770.71060124560}, 
    {     0.00000005580,  0.48723420248,      522.57741809380}, 
    {     0.00000005327,  3.03115799765,    10021.83728009940}, 
    {     0.00000005010,  5.77374296245,    28521.09277825460}, 
    {     0.00000004608,  1.93302031704,     4705.73230754360}, 
    {     0.00000005526,  3.36797150122,    25158.60171976540}, 
    {     0.00000003863,  4.89351765621,    25934.12433108940}, 
    {     0.00000005303,  0.08161035601,    39302.09696219600}, 
    {     0.00000004254,  5.36046525146,    21535.94964451540}, 
    {     0.00000003763,  1.05304597315,       19.66976089979}, 
    {     0.00000004407,  4.02575372996,       74.78159856730}, 
    {     0.00000004145,  1.14356412295,     9676.48103411560}, 
    {     0.00000004318,  4.38289970585,      316.39186965660}, 
    {     0.00000003642,  6.11733531450,     3128.38876509580}, 
    {     0.00000003238,  5.39551036769,      419.48464387520}, 
    {     0.00000003909,  4.05263635330,     9690.70812811720}, 
    {     0.00000003173,  0.74132026653,    16496.36139620240}, 
    {     0.00000003496,  0.72415151509,     3723.50895892300}, 
    {     0.00000003755,  3.80208523566,    19786.67380610799}, 
    {     0.00000003016,  1.57249073681,    17277.40693183380}, 
    {     0.00000002858,  3.31246198673,    32217.20018108080}, 
    {     0.00000003825,  0.19612307327,      426.59819087600}, 
    {     0.00000003798,  0.45524571743,    10316.37832042960}, 
    {     0.00000002570,  1.20813798183,    13936.79450513400}, 
    {     0.00000002796,  3.65128969074,      206.18554843720}, 
    {     0.00000002466,  3.61988425580,     1551.04522264800}, 
    {     0.00000003108,  1.50324979041,    43232.30665841560}, 
    {     0.00000002976,  4.79414174980,    29088.81141598500}, 
    {     0.00000002196,  3.57357339264,    24356.78078864160}, 
    {     0.00000002227,  4.96059090976,      536.80451209540}, 
    {     0.00000002397,  3.45249688427,    19374.30270923360}, 
    {     0.00000002462,  0.53295346740,    19360.07561523199}, 
    {     0.00000002205,  2.70398708223,    12592.45001978260}, 
    {     0.00000002230,  3.01413989795,    18875.52586977400}, 
    {     0.00000001858,  4.06129093893,     2379.16447357160}, 
    {     0.00000001807,  3.15086244011,     9573.38825989700}, 
    {     0.00000002238,  5.52216925076,    10138.50394764370}, 
    {     0.00000002195,  2.32046407873,     8094.52168583260}, 
    {     0.00000002101,  2.90421302974,     9967.45389998160}, 
    {     0.00000001916,  4.56513913980,     2218.75710418680}, 
    {     0.00000001467,  2.42640162464,    10234.06094170340}, 
    {     0.00000001726,  5.59790615295,    20452.86941222180}, 
    {     0.00000001455,  2.44757225671,     1589.07289528380}, 
    {     0.00000001991,  4.04623282256,    31749.23519072640}, 
    {     0.00000001406,  2.71737807375,    16983.99614745660}, 
    {     0.00000001658,  0.11252373292,      153.77881048480}, 
    {     0.00000001851,  2.92897863746,    47162.51635463520}, 
    {     0.00000001492,  1.07515446511,     9103.90699411760}, 
    {     0.00000001247,  2.48433636140,    17778.11626694899}, 
    {     0.00000001549,  4.20555105766,     3442.57494496540}, 
    {     0.00000001243,  3.95452438599,      170.67287061920}, 
    {     0.00000001694,  6.20691533902,    33019.02111220460}, 
    {     0.00000001221,  4.77932060567,    30110.16567353840}, 
    {     0.00000001206,  0.30531254531,    29864.33402730900}, 
    {     0.00000001238,  5.05581030425,    20213.27199698400}, 
    {     0.00000001152,  3.26229919481,       11.72935283600}, 
    {     0.00000001179,  1.69491078238,    20400.27277262220}, 
    {     0.00000001165,  2.88995128147,      574.34479833480}, 
    {     0.00000001306,  0.14519568603,     9146.79006902100}, 
    {     0.00000001113,  1.52599723495,    10426.58464164900}, 
    {     0.00000001045,  0.57538516899,    15874.61759536320}, 
    {     0.00000001105,  2.54593760294,    18849.22754997420}, 
    {     0.00000001360,  2.41974967161,    38734.37832446560}, 
    {     0.00000000981,  4.37930757970,      110.20632121940}, 
    {     0.00000001146,  4.54241415687,    10220.39909321180}, 
    {     0.00000001091,  0.48630333703,    51066.42773105500}, 
    {     0.00000000981,  1.65915064734,    10103.07922499160}, 
    {     0.00000001270,  4.69374306132,     9050.81084180320}, 
    {     0.00000001065,  4.41646204804,    22805.73556599360}, 
    {     0.00000000854,  2.34437927295,     6681.22485339960}, 
    {     0.00000001104,  0.49781459714,        1.48447270830}, 
    {     0.00000001075,  1.09857593161,      377.37360791580}, 
    {     0.00000001114,  4.35024212108,    51092.72605085480}, 
    {     0.00000000829,  5.41196376472,    27991.40181316000}, 
    {     0.00000000900,  2.74195213638,    41962.52073693740}, 
    {     0.00000001010,  2.96092073452,      135.62532501000}, 
    {     0.00000000768,  3.98260860494,    18844.61174413899}, 
    {     0.00000001018,  1.36889990282,    36949.23080842420}, 
    {     0.00000000725,  1.67729445754,    21202.09370374600}, 
    {     0.00000000727,  0.89048212541,      467.96499035440}, 
    {     0.00000000869,  2.93235455158,    10192.51015071860}, 
    {     0.00000000696,  5.35698039414,    10063.72234907640}, 
    {     0.00000000920,  4.17128923588,    18734.40542291960}, 
    {     0.00000000691,  1.50594847085,    27197.28169366760}, 
    {     0.00000000835,  0.48049677664,    20618.01935853360}, 
    {     0.00000000711,  0.19750098222,    18830.38465013739}, 
    {     0.00000000811,  0.16685071959,    12432.04265039780}, 
    {     0.00000000756,  3.79022449760,     9161.01716302260}, 
    {     0.00000000622,  5.33660452826,     9411.46461508720}, 
    {     0.00000000862,  5.72705356405,    10175.15251057320}, 
    {     0.00000000853,  0.10404194066,     2107.03450754240}, 
    {     0.00000000742,  3.96365892051,      813.55028395980}, 
    {     0.00000000705,  0.71229660616,      220.41264243880}, 
    {     0.00000000584,  1.71900692700,       36.02786667740}, 
    {     0.00000000612,  0.36418385449,      949.17560896980}, 
    {     0.00000000587,  1.58648949290,        6.62855890001}, 
    {     0.00000000581,  5.49288908804,     6309.37416979120}, 
    {     0.00000000581,  4.80362317972,    24150.08005134500}, 
    {     0.00000000516,  6.07328802561,       38.13303563780}, 
    {     0.00000000627,  5.47281424955,     9580.50180689780}, 
    {     0.00000000601,  1.40500080384,     1162.47470440780}, 
    {     0.00000000620,  4.00681042667,     9992.87290377220}, 
    {     0.00000000611,  3.62010998629,     7255.56965173440}, 
    {     0.00000000697,  2.22359630727,      348.92442044800}, 
    {     0.00000000569,  0.23482601786,    37410.56723987860}, 
    {     0.00000000693,  5.77432072851,    55022.93574707440}, 
    {     0.00000000494,  0.29762226375,     7058.59846131540}, 
    {     0.00000000487,  5.86917216517,    10137.01947493540}, 
    {     0.00000000493,  2.04534833854,      735.87651353180}, 
    {     0.00000000636,  2.79707415908,    40879.44050464380}, 
    {     0.00000000519,  4.13931655130,    16522.65971600220}, 
    {     0.00000000535,  4.60569597820,    19573.37471066999}, 
    {     0.00000000555,  5.88120469821,    26735.94526221320}, 
    {     0.00000000440,  5.61490952812,    23958.63178523340}, 
    {     0.00000000541,  0.62494922735,    10007.09999777380}, 
    {     0.00000000427,  4.02335620501,       14.22709400160}, 
    {     0.00000000434,  0.29028429049,     9264.10993724120}, 
    {     0.00000000451,  1.66321968415,    26087.90314157420}, 
    {     0.00000000422,  3.38413582674,    10787.63034454580}, 
    {     0.00000000568,  5.14001584538,    27490.69247804480}, 
    {     0.00000000421,  4.23407420790,    39793.76025465480}, 
    {     0.00000000458,  5.28786368820,       49.75702547180}, 
    {     0.00000000418,  5.69090817000,    14765.23904326980}, 
    {     0.00000000475,  0.97544690438,     1052.26838318840}, 
    {     0.00000000387,  4.41665162999,       21.85082932640}, 
    {     0.00000000523,  2.90512729055,    20235.12282631040}, 
    {     0.00000000506,  5.26999240626,    29999.95935231900}, 
    {     0.00000000530,  0.71374608018,    33794.54372352860}, 
    {     0.00000000434,  2.49892103222,    20956.26205751660}, 
    {     0.00000000382,  1.92119365480,        3.93215326310}, 
    {     0.00000000365,  3.81713155389,    20419.45754542119}, 
    {     0.00000000426,  2.06384189772,    38204.68735937100}, 
    {     0.00000000496,  0.44077356179,     9835.91193829520}, 
    {     0.00000000410,  4.93346330078,    19264.09638801420}, 
    {     0.00000000380,  3.79573339943,     8521.11987670860}, 
    {     0.00000000334,  5.51158557800,    10251.31321884680}, 
    {     0.00000000412,  2.56129670728,       77.67377042800}, 
    {     0.00000000418,  2.37865963521,       32.53255079140}, 
    {     0.00000000325,  6.03020523465,    18947.70451835760}, 
    {     0.00000000400,  0.91999360201,      227.47613278900}, 
    {     0.00000000437,  0.91420135162,    58953.14544329400}, 
    {     0.00000000360,  0.82477639126,       22.77520145080}, 
    {     0.00000000413,  4.22381925464,    44809.65020086340}, 
    {     0.00000000375,  3.15657291896,    19992.85935454519}, 
    {     0.00000000371,  6.05370874275,    20007.08644854680}, 
    {     0.00000000361,  5.44371227904,    19470.28193645139}, 
    {     0.00000000386,  5.28104804025,    47623.85278608960}, 
    {     0.00000000389,  0.73216672240,    19050.79729257620}, 
    {     0.00000000320,  2.84811591194,    10199.05845220940}, 
    {     0.00000000386,  3.88754165531,     1975.49254585600}, 
    {     0.00000000279,  0.50274101065,     9830.38901398780}, 
    {     0.00000000276,  4.33978638152,    20809.46762464520}, 
    {     0.00000000309,  3.79299100668,    18204.71445782499}, 
    {     0.00000000377,  0.73768790585,    11506.76976979360}, 
    {     0.00000000322,  0.96113438814,    30666.15495843280}, 
    {     0.00000000363,  1.30472406690,     9367.20271145980}, 
    {     0.00000000366,  2.79974205585,    11272.66747640020}, 
    {     0.00000000271,  4.66141338193,      846.08283475120}, 
    {     0.00000000259,  0.42024142711,    39264.06928956020}, 
    {     0.00000000285,  0.40546033634,       30.91412563500}, 
    {     0.00000000247,  4.80676152850,    36147.40987730040}, 
    {     0.00000000264,  2.71608177583,       11.04570026390}, 
    {     0.00000000233,  2.76423842887,      187.92514776260}, 
    {     0.00000000248,  1.60765612335,    10497.14486507620}, 
    {     0.00000000271,  0.82348919630,    19793.78735310880}, 
    {     0.00000000225,  3.80080911298,     8631.32619792800}, 
    {     0.00000000263,  1.92319161528,    37724.75341974820}, 
    {     0.00000000214,  5.01663561403,      639.89728631400}, 
    {     0.00000000289,  0.12342601246,    20277.00789528740}, 
    {     0.00000000210,  0.12771800254,       29.82143814880}, 
    {     0.00000000227,  4.18036609801,    17468.85519794540}, 
    {     0.00000000200,  3.67516531895,    30213.25844775700}, 
    {     0.00000000274,  2.34929343000,    62883.35513951360}, 
    {     0.00000000260,  5.65254501655,    48739.85989708300}, 
    {     0.00000000271,  4.95325404028,     4214.06901508480}, 
    {     0.00000000219,  2.08775228014,      194.97138446060}, 
    {     0.00000000191,  2.49267248333,      568.82187402740}, 
    {     0.00000000231,  0.85116556400,    52670.06959330260}, 
    {     0.00000000250,  1.52909737354,     6037.24420376200}, 
    {     0.00000000231,  5.23674429498,      491.66329245880}, 
    {     0.00000000182,  4.98046042571,    18418.01355326299}, 
    {     0.00000000188,  2.82273639603,     1385.89527633620}, 
    {     0.00000000204,  4.09938646222,    14919.01785375460}, 
    {     0.00000000176,  3.82400982460,     9360.08916445900}, 
    {     0.00000000198,  2.76491873243,    10217.21769947410}, 
    {     0.00000000168,  5.19268690225,     1066.49547719000}, 
    {     0.00000000199,  1.95274291865,     7564.83072073800}, 
    {     0.00000000172,  5.29332134227,    11764.33076885900}, 
    {     0.00000000164,  3.25435371801,     3914.95722503460}, 
    {     0.00000000200,  3.82443218090,    18314.92077904440}, 
    {     0.00000000203,  5.93253927885,     8617.09910392640}, 
    {     0.00000000169,  1.78341902846,    31022.75317085620}, 
    {     0.00000000160,  2.84901702400,    10207.76262190360}, 
    {     0.00000000159,  2.09236390117,    17248.42530185440}, 
    {     0.00000000179,  0.90840065587,     7880.08915333899}, 
    {     0.00000000163,  2.79665037814,       41.55079098480}, 
    {     0.00000000153,  0.07463240782,    28528.20632525540}, 
    {     0.00000000199,  2.48348609726,    34596.36465465240}, 
    {     0.00000000149,  2.59213969874,    43071.89928903080}, 
    {     0.00000000149,  2.17259986337,     9929.42622734580}, 
    {     0.00000000156,  5.20953676299,    22645.32819660879}, 
    {     0.00000000148,  2.94331271107,    41654.96311596780}, 
    {     0.00000000149,  4.49800344900,    30831.30490474460}, 
    {     0.00000000159,  2.11138069179,    19317.19254032860}, 
    {     0.00000000154,  2.76536164654,    28513.97923125379}, 
    {     0.00000000140,  4.94595038686,     9256.99639024040}, 
    {     0.00000000141,  2.57248608474,    13553.89797291080}, 
    {     0.00000000137,  1.66482327575,     2636.72547263700}, 
    {     0.00000000136,  4.14497573316,     2333.19639287200}, 
    {     0.00000000133,  3.91457156721,    32858.61374281979}, 
    {     0.00000000140,  2.90644630691,    48947.66387067660}, 
    {     0.00000000156,  6.01143377008,    29057.89729034999}, 
    {     0.00000000134,  5.75241675118,    68050.42387851159}, 
    {     0.00000000154,  3.66827363753,      276.74577186440}, 
    {     0.00000000176,  3.77298381177,    66813.56483573320}, 
    {     0.00000000126,  5.00217740223,    27461.71084806540}, 
    {     0.00000000135,  1.34807013920,    53285.18483524180}, 
    {     0.00000000150,  0.25029475344,      290.97286586600}, 
    {     0.00000000152,  3.13035647340,    29043.67019634839}, 
    {     0.00000000169,  5.04348109430,       73.29712585900}, 
    {     0.00000000166,  5.39219948035,    41236.03871706720}, 
    {     0.00000000115,  2.02657557658,    37674.99639427640}, 
    {     0.00000000163,  5.59796070948,     7576.56007357400}, 
    {     0.00000000126,  0.77391784606,       49.99662190420}, 
    {     0.00000000163,  0.44241846674,    20350.30502114640}, 
    {     0.00000000136,  3.09066368912,      418.92439890060}, 
    {     0.00000000154,  0.47086190960,    28418.00000403600}, 
    {     0.00000000125,  5.46344374133,    22779.43724619380}, 
    {     0.00000000136,  4.17495174196,    42430.48572729180}, 
    {     0.00000000120,  0.88537349448,    29573.36116144300}, 
    {     0.00000000132,  1.48003093721,    17085.95866572220}, 
    {     0.00000000126,  1.39497760964,      966.97087743560}, 
    {     0.00000000126,  0.16302255819,    67589.08744705719}, 
    {     0.00000000111,  2.50223464050,    20405.79569692960}, 
    {     0.00000000143,  3.84026797958,    14128.24277124560}, 
    {     0.00000000108,  4.57687521213,    30426.55754319500}, 
    {     0.00000000147,  2.11609860260,    34363.36559755600}, 
    {     0.00000000133,  1.07792835035,    20639.87018786000}, 
    {     0.00000000106,  1.43873202489,    27682.14074415640}, 
    {     0.00000000149,  0.09286508794,     8144.27871130440}, 
    {     0.00000000103,  0.01992041470,    18300.69368504279}, 
    {     0.00000000116,  5.27605692179,     8734.41897214660}, 
    {     0.00000000121,  3.57602835443,       45.14121963660}, 
    {     0.00000000125,  0.11630302078,      149.56319713460}, 
    {     0.00000000108,  5.23732413902,    14169.79356223040}, 
    {     0.00000000113,  4.57267032764,    20447.34648791440}, 
    {     0.00000000099,  1.51324741656,    10419.47109464820}, 
    {     0.00000000105,  1.70350302928,    19580.48825767080}, 
    {     0.00000000133,  3.02183293676,    76251.32777062019}, 
    {     0.00000000136,  4.17517197268,     3646.35037735440}, 
    {     0.00000000097,  6.25332492425,    10198.03307502600}, 
    {     0.00000000123,  0.44045588682,      515.46387109300}, 
    {     0.00000000129,  3.95296865191,    38526.57435087200}, 
    {     0.00000000113,  5.69261397722,    10110.19277199240}, 
    {     0.00000000098,  6.23797900467,      202.25339517410}, 
    {     0.00000000099,  3.75627530197,    59728.66805461800}, 
    {     0.00000000101,  4.62832512136,    65236.22129328540}, 
    {     0.00000000111,  1.25947267592,    10846.06928552420}, 
    {     0.00000000126,  4.45075374981,    80181.53746683979}, 
    {     0.00000000110,  5.87454529350,    38500.27603107220}, 
    {     0.00000000093,  0.03768705119,    90695.75207512038}, 
    {     0.00000000128,  6.01024562160,    90394.82301305079}, 
    {     0.00000000091,  1.77665981007,     1539.31586981200}, 
    {     0.00000000092,  0.99804571578,       95.97922721780}, 
    {     0.00000000117,  2.24143299549,    56600.27928952220}, 
    {     0.00000000118,  6.09121325940,    29786.66025688100}, 
    {     0.00000000098,  4.60938156207,    11787.10597030980}, 
    {     0.00000000097,  3.92727733144,    11794.15220700780}, 
    {     0.00000000085,  5.30575031807,    19624.75016129820}, 
    {     0.00000000106,  0.82812501368,    24383.07910844140}, 
    {     0.00000000085,  5.69642646462,    37703.97802425580}, 
    {     0.00000000114,  5.19676285428,    70743.77453195279}, 
    {     0.00000000081,  5.51324815184,      412.37109687440}, 
    {     0.00000000080,  5.77520799089,    10632.77019008620}, 
    {     0.00000000089,  2.13409771828,    44768.09940987860}, 
    {     0.00000000094,  2.18120919704,    28313.28880466100}, 
    {     0.00000000088,  0.57046254096,     3956.50801601940}, 
    {     0.00000000078,  4.47358603432,      114.43928868521}, 
    {     0.00000000104,  4.94045302010,    45585.17281218740}, 
    {     0.00000000097,  4.02223363553,    10218.80847051840}, 
    {     0.00000000094,  5.81137122373,    69166.43098950500}, 
    {     0.00000000076,  0.78247116213,     9793.80090233580}, 
    {     0.00000000079,  4.73339536112,    24978.52458948080}, 
    {     0.00000000101,  4.81437377872,    58177.62283197000}, 
    {     0.00000000097,  3.61746670975,    60530.48898574180}, 
    {     0.00000000088,  4.87010387338,    63658.87775083760}, 
    {     0.00000000088,  5.17408120336,    48417.97290558199}, 
    {     0.00000000078,  5.81927313665,      567.71863773040}, 
    {     0.00000000077,  6.16012067704,     1573.82042409880}, 
    {     0.00000000084,  0.62512782571,    18100.00325845000}, 
    {     0.00000000076,  5.12867208321,    49515.38250840700}, 
    {     0.00000000067,  0.24541021814,    17271.88400752640}, 
    {     0.00000000069,  0.29569499496,    10175.25787357520}, 
    {     0.00000000088,  5.43871184563,    42456.78404709160}, 
    {     0.00000000079,  0.30211728530,    63498.47038145279}, 
    {     0.00000000080,  2.69571879625,     3149.16416058820}, 
    {     0.00000000061,  4.80385228047,    19889.76658032659}, 
    {     0.00000000084,  0.22956807093,    88817.47947060299}, 
    {     0.00000000060,  0.46406912136,    36301.18868778519}, 
    {     0.00000000071,  0.76007659495,    57837.13833230060}, 
    {     0.00000000081,  5.24314123043,    36173.70819710020}, 
    {     0.00000000057,  3.07470109636,     2118.76386037840}, 
    {     0.00000000075,  0.66947402578,    10735.86296430480}, 
    {     0.00000000058,  4.94877232192,    47888.28194048740}, 
    {     0.00000000068,  0.96147256965,    24341.52831745660}, 
    {     0.00000000076,  0.15485592669,    68390.90837818099}, 
    {     0.00000000064,  3.61826420613,    32243.49850088060}, 
    {     0.00000000062,  1.59487827272,    39999.94580309200}, 
    {     0.00000000063,  0.16823121009,    53445.59220462660}, 
    {     0.00000000052,  2.85553976522,    10323.49186743040}, 
    {     0.00000000058,  1.07613515145,    11692.15212027540}, 
    {     0.00000000052,  2.34807495268,    26709.64694241340}, 
    {     0.00000000062,  4.09802218900,    34554.81386366760}, 
    {     0.00000000048,  0.76057327599,    44783.35188106360}, 
    {     0.00000000067,  0.07870600216,    47964.33728575900}, 
    {     0.00000000048,  5.30855519348,    48733.23133818299}, 
    {     0.00000000047,  3.09314000663,    70269.18098269838}, 
    {     0.00000000053,  5.53372176617,    19903.99367432819}, 
    {     0.00000000057,  2.43412854215,    10228.53801739600}, 
    {     0.00000000043,  6.00071261364,    47938.03896595920}, 
    {     0.00000000058,  1.23233304711,    59754.96637441780}, 
    {     0.00000000055,  0.99068063794,    21000.91589075680}, 
    {     0.00000000060,  5.94539126752,    69968.25192062880}, 
    {     0.00000000049,  3.83996757537,    38631.28555024700}, 
    {     0.00000000042,  4.73460925987,    37895.42629036740}, 
    {     0.00000000044,  1.75905995399,    71519.29714327680}, 
    {     0.00000000043,  6.08746206685,     2957.71589447660}, 
    {     0.00000000040,  1.31794872194,    20529.66386664059}, 
    {     0.00000000046,  4.41749158726,    34570.06633485260}, 
    {     0.00000000045,  2.60413561993,    58101.56748669840}, 
    {     0.00000000037,  4.69673401538,    44007.82926973960}, 
    {     0.00000000039,  2.32823034676,    55798.45835839840}, 
    {     0.00000000051,  1.76846419002,    78604.19392439199}, 
    {     0.00000000038,  1.28879356308,    73096.64068572459}, 
    {     0.00000000048,  1.18664500846,    28306.66024576099}, 
    {     0.00000000036,  0.64770877031,     1478.86657406440}, 
    {     0.00000000048,  0.40666475573,    40103.91789331980}, 
};


static const struct ln_vsop ALIGN32 venus_longitude_l1[LONG_L1] = {
    { 10213.28554621638,  0.00000000000,        0.00000000000}, 
    {     0.00095617813,  2.46406511110,    10213.28554621100}, 
    {     0.00007787201,  0.62478482220,    20426.57109242200}, 
    {     0.00000151666,  6.10638559291,     1577.34354244780}, 
    {     0.00000141694,  2.12362986036,    30639.85663863300}, 
    {     0.00000173908,  2.65539499463,       26.29831979980}, 
    {     0.00000082235,  5.70231469551,      191.44826611160}, 
    {     0.00000069732,  2.68128549229,     9437.76293488700}, 
    {     0.00000052292,  3.60270736876,      775.52261132400}, 
    {     0.00000038313,  1.03371309443,      529.69096509460}, 
    {     0.00000029630,  1.25050823203,     5507.55323866740}, 
    {     0.00000025056,  6.10650638660,    10404.73381232260}, 
    {     0.00000017772,  6.19369679929,     1109.37855209340}, 
    {     0.00000016510,  2.64360813203,        7.11354700080}, 
    {     0.00000014231,  5.45125927817,     9153.90361602180}, 
    {     0.00000011627,  4.97604433638,      213.29909543800}, 
    {     0.00000012563,  1.88122194951,      382.89653222320}, 
    {     0.00000008877,  0.95245393457,    13367.97263110660}, 
    {     0.00000007374,  4.39476352550,    10206.17199921020}, 
    {     0.00000006550,  2.28168331756,     2352.86615377180}, 
    {     0.00000006444,  1.41156299643,    40853.14218484400}, 
    {     0.00000006269,  4.08365791523,     3154.68708489560}, 
    {     0.00000006702,  5.05916048534,      801.82093112380}, 
    {     0.00000004099,  4.12235848112,    18837.49819713819}, 
    {     0.00000004883,  3.44513438224,    11015.10647733480}, 
    {     0.00000003549,  6.19934345402,        5.52292430740}, 
    {     0.00000003448,  1.77404222413,    11322.66409830440}, 
    {     0.00000004290,  0.08136342862,     6283.07584999140}, 
    {     0.00000003699,  2.48455675849,     5661.33204915220}, 
    {     0.00000003671,  1.48751015918,     1059.38193018920}, 
    {     0.00000003018,  2.23849523296,    18073.70493865020}, 
    {     0.00000003000,  0.39169917698,       15.25247118500}, 
    {     0.00000002776,  1.45644767378,    10239.58386601080}, 
    {     0.00000002463,  0.36768007007,    22003.91463486980}, 
    {     0.00000002954,  5.34806371393,     7084.89678111520}, 
    {     0.00000002350,  2.36543900882,    17298.18232732620}, 
    {     0.00000002405,  2.36085521516,    10596.18207843420}, 
    {     0.00000002471,  2.11327686375,     8635.94200376320}, 
    {     0.00000001717,  4.72125604095,    10186.98722641120}, 
    {     0.00000002135,  4.46350370275,     8624.21265092720}, 
    {     0.00000001626,  0.86147271429,    12566.15169998280}, 
    {     0.00000001523,  0.67189933187,    14143.49524243060}, 
    {     0.00000001473,  2.59350470099,     7064.12138562280}, 
    {     0.00000001474,  5.92233777117,     9786.68735533500}, 
    {     0.00000001237,  2.59741927942,     4551.95349705880}, 
    {     0.00000001219,  2.83617320089,     9676.48103411560}, 
    {     0.00000001242,  0.82080840456,    10742.97651130560}, 
    {     0.00000001007,  4.25205211647,      426.59819087600}, 
    {     0.00000001097,  3.84455455465,    21228.39202354580}, 
    {     0.00000001150,  2.35531987378,     9690.70812811720}, 
    {     0.00000001219,  2.27326005102,      522.57741809380}, 
    {     0.00000001101,  3.74248862749,    18307.80723204360}, 
    {     0.00000001031,  2.03889374176,       38.02767263580}, 
    {     0.00000000971,  6.10589743884,     3532.06069281140}, 
    {     0.00000000887,  4.72636674421,    10988.80815753500}, 
    {     0.00000000927,  1.05402087347,    10021.83728009940}, 
    {     0.00000000821,  2.60456032773,       19.66976089979}, 
    {     0.00000000780,  0.18364413336,     4705.73230754360}, 
    {     0.00000000810,  1.32033186808,    19896.88012732740}, 
    {     0.00000000744,  3.33129743110,      536.80451209540}, 
    {     0.00000000797,  2.23891817601,     3723.50895892300}, 
    {     0.00000000709,  5.95126881058,       20.77539549240}, 
    {     0.00000000894,  1.08522968560,    11790.62908865880}, 
    {     0.00000000660,  0.29717009501,     7860.41939243920}, 
    {     0.00000000733,  2.22147883292,    19360.07561523199}, 
    {     0.00000000702,  1.76206343944,    19374.30270923360}, 
    {     0.00000000633,  2.15202166788,    16496.36139620240}, 
    {     0.00000000575,  2.38792087791,     6770.71060124560}, 
    {     0.00000000532,  4.41576130890,      574.34479833480}, 
    {     0.00000000518,  1.52546538858,    25934.12433108940}, 
    {     0.00000000479,  0.30473233584,    28286.99048486120}, 
    {     0.00000000576,  5.41170044566,      206.18554843720}, 
    {     0.00000000482,  0.40816741074,     3340.61242669980}, 
    {     0.00000000537,  0.22055767322,    19367.18916223280}, 
    {     0.00000000574,  4.07672246337,    19651.04848109800}, 
    {     0.00000000501,  3.08578363577,      245.83164622940}, 
    {     0.00000000488,  5.22318430105,    25158.60171976540}, 
    {     0.00000000450,  0.21279844600,       11.72935283600}, 
    {     0.00000000432,  1.32004964493,      103.09277421860}, 
    {     0.00000000434,  5.91094755233,    19786.67380610799}, 
    {     0.00000000421,  2.71057839701,    13936.79450513400}, 
    {     0.00000000564,  6.13266504618,     3930.20969621960}, 
    {     0.00000000478,  4.70225377704,    14945.31617355440}, 
    {     0.00000000408,  2.93857115699,    10220.39909321180}, 
    {     0.00000000359,  0.72354778897,      419.48464387520}, 
    {     0.00000000449,  1.44520508753,     8662.24032356300}, 
    {     0.00000000353,  2.85616596224,    29864.33402730900}, 
    {     0.00000000333,  1.41991315364,    29580.47470844380}, 
    {     0.00000000443,  1.93864343247,     9146.79006902100}, 
    {     0.00000000310,  1.08363376581,    20618.01935853360}, 
    {     0.00000000324,  1.80146948625,    18830.38465013739}, 
    {     0.00000000370,  6.16895004656,     2218.75710418680}, 
    {     0.00000000278,  2.20429108375,    18844.61174413899}, 
    {     0.00000000286,  3.08458231517,    17277.40693183380}, 
    {     0.00000000383,  0.13900678695,     4732.03062734340}, 
    {     0.00000000292,  0.43530118043,    29088.81141598500}, 
    {     0.00000000273,  5.84415383782,     9573.38825989700}, 
    {     0.00000000324,  2.14135439540,     9999.98645077300}, 
    {     0.00000000264,  5.20407029554,      220.41264243880}, 
    {     0.00000000254,  0.34417665274,    28521.09277825460}, 
    {     0.00000000300,  3.76015126903,     8094.52168583260}, 
    {     0.00000000303,  1.85088655195,     1589.07289528380}, 
    {     0.00000000206,  2.49682541324,    51066.42773105500}, 
    {     0.00000000206,  0.97144383633,    10234.06094170340}, 
    {     0.00000000212,  0.24173677600,       36.02786667740}, 
    {     0.00000000208,  5.19345603406,    24356.78078864160}, 
    {     0.00000000216,  5.88607943223,    18875.52586977400}, 
    {     0.00000000258,  6.27611237676,     1551.04522264800}, 
    {     0.00000000200,  2.09314510849,     9683.59458111640}, 
    {     0.00000000217,  5.79530885159,     9103.90699411760}, 
    {     0.00000000188,  0.39123199129,    19573.37471066999}, 
    {     0.00000000187,  5.49670351645,      170.67287061920}, 
    {     0.00000000233,  3.85755192539,    20400.27277262220}, 
    {     0.00000000178,  4.90042854659,    10787.63034454580}, 
    {     0.00000000188,  1.62614657498,     9161.01716302260}, 
    {     0.00000000177,  1.88170417337,    33019.02111220460}, 
    {     0.00000000209,  2.66033526608,     3442.57494496540}, 
    {     0.00000000164,  5.04773317971,    32217.20018108080}, 
    {     0.00000000162,  4.92736966968,    10426.58464164900}, 
    {     0.00000000186,  5.13678812068,     7255.56965173440}, 
    {     0.00000000177,  5.70206821967,     9992.87290377220}, 
    {     0.00000000177,  2.40042123486,    20452.86941222180}, 
    {     0.00000000212,  2.73881400176,     3128.38876509580}, 
    {     0.00000000208,  3.38876526854,    17778.11626694899}, 
    {     0.00000000173,  4.09215943674,    33794.54372352860}, 
    {     0.00000000147,  4.25008910034,    16983.99614745660}, 
    {     0.00000000189,  1.43553862242,     2379.16447357160}, 
    {     0.00000000139,  2.99154379541,      110.20632121940}, 
    {     0.00000000159,  5.23851679605,    10007.09999777380}, 
    {     0.00000000169,  2.67970563071,    15720.83878487840}, 
    {     0.00000000136,  0.88942869764,    22805.73556599360}, 
    {     0.00000000155,  5.90500835975,    12592.45001978260}, 
    {     0.00000000151,  0.03422618975,    27991.40181316000}, 
    {     0.00000000111,  0.77661072477,     6681.22485339960}, 
    {     0.00000000109,  2.25388616761,    26735.94526221320}, 
    {     0.00000000122,  4.84805105466,    19992.85935454519}, 
    {     0.00000000112,  3.31796669604,    36949.23080842420}, 
    {     0.00000000106,  3.34507236765,    10103.07922499160}, 
    {     0.00000000114,  4.36384000196,    20007.08644854680}, 
    {     0.00000000121,  5.42385293514,    37724.75341974820}, 
    {     0.00000000100,  0.05731448574,    30110.16567353840}, 
    {     0.00000000098,  5.07711736751,      135.62532501000}, 
    {     0.00000000109,  0.54849921587,    21535.94964451540}, 
    {     0.00000000104,  3.67555714699,    21202.09370374600}, 
    {     0.00000000103,  2.65873403125,    20213.27199698400}, 
    {     0.00000000087,  5.04316754485,     9830.38901398780}, 
    {     0.00000000103,  2.01542881355,    45585.17281218740}, 
    {     0.00000000088,  2.62613816931,       21.85082932640}, 
    {     0.00000000084,  3.50355880173,      639.89728631400}, 
    {     0.00000000099,  0.61072730211,    41654.96311596780}, 
    {     0.00000000081,  0.46468679835,       77.67377042800}, 
    {     0.00000000092,  4.82530051729,    29043.67019634839}, 
    {     0.00000000090,  4.34075776744,    29057.89729034999}, 
    {     0.00000000081,  0.01915973453,    24150.08005134500}, 
    {     0.00000000095,  2.42613786510,    23581.25817731760}, 
    {     0.00000000101,  4.27032438869,    15874.61759536320}, 
    {     0.00000000093,  1.79250830018,    12432.04265039780}, 
    {     0.00000000087,  5.25157021446,    14128.24277124560}, 
    {     0.00000000089,  5.65756996753,      377.37360791580}, 
    {     0.00000000076,  1.32618111784,    18300.69368504279}, 
    {     0.00000000097,  5.67942873241,      227.47613278900}, 
    {     0.00000000076,  2.93363913259,    38204.68735937100}, 
    {     0.00000000091,  2.60544242067,     1052.26838318840}, 
    {     0.00000000084,  5.65355105405,    17468.85519794540}, 
    {     0.00000000082,  0.14972353998,    29050.78374334920}, 
    {     0.00000000086,  1.42751065635,    11272.66747640020}, 
    {     0.00000000073,  4.75280755154,    40879.44050464380}, 
    {     0.00000000068,  3.56578829793,    30666.15495843280}, 
    {     0.00000000068,  5.80655405181,    20809.46762464520}, 
    {     0.00000000079,  3.84440385450,    27511.46787353720}, 
    {     0.00000000067,  3.74236733721,    49515.38250840700}, 
    {     0.00000000064,  4.24772678145,      153.77881048480}, 
    {     0.00000000064,  2.99454749109,    27197.28169366760}, 
    {     0.00000000070,  4.03868009742,    56600.27928952220}, 
    {     0.00000000069,  1.73648747605,    37410.56723987860}, 
    {     0.00000000065,  1.08206062736,    68050.42387851159}, 
    {     0.00000000062,  4.77698454650,     3914.95722503460}, 
    {     0.00000000068,  4.28465793234,    39264.06928956020}, 
    {     0.00000000061,  4.96121014691,    34596.36465465240}, 
    {     0.00000000058,  3.74010494151,     1066.49547719000}, 
    {     0.00000000063,  0.15331622244,    57375.80190084620}, 
    {     0.00000000057,  5.42590958930,    20419.45754542119}, 
    {     0.00000000057,  3.59399518494,      735.87651353180}, 
    {     0.00000000065,  2.10322000074,       74.78159856730}, 
    {     0.00000000065,  4.21150522641,       49.75702547180}, 
    {     0.00000000052,  0.03409255574,    18849.22754997420}, 
    {     0.00000000052,  1.95168272115,    18314.92077904440}, 
    {     0.00000000055,  2.63906959481,    52670.06959330260}, 
    {     0.00000000057,  1.25081330234,    10218.80847051840}, 
    {     0.00000000053,  2.14346912907,    13745.34623902240}, 
    {     0.00000000049,  0.77417108584,    35371.88726597640}, 
    {     0.00000000048,  6.00565977593,      283.85931886520}, 
    {     0.00000000050,  4.45949921607,    40077.61957352000}, 
    {     0.00000000047,  2.63299859494,    51868.24866217880}, 
    {     0.00000000046,  0.05105081843,    38526.57435087200}, 
    {     0.00000000050,  4.37549274002,    28513.97923125379}, 
    {     0.00000000046,  2.93422086586,    27682.14074415640}, 
    {     0.00000000062,  0.41777498261,    30831.30490474460}, 
    {     0.00000000053,  1.55276873128,    36301.18868778519}, 
    {     0.00000000051,  5.45979584751,    60530.48898574180}, 
    {     0.00000000051,  2.02307685593,    59728.66805461800}, 
    {     0.00000000045,  5.59492908223,      467.96499035440}, 
    {     0.00000000046,  5.16141298965,    53445.59220462660}, 
    {     0.00000000045,  5.79863400674,    39793.76025465480}, 
    {     0.00000000044,  2.35499883188,     9411.46461508720}, 
    {     0.00000000043,  5.62725673544,       14.22709400160}, 
    {     0.00000000047,  3.73567275749,    64460.69868196140}, 
    {     0.00000000041,  4.11417818861,    14765.23904326980}, 
    {     0.00000000039,  0.99375127466,    94138.32702008578}, 
    {     0.00000000040,  3.06358586355,      813.55028395980}, 
    {     0.00000000039,  6.21378797479,    44809.65020086340}, 
    {     0.00000000038,  5.90441887648,    14919.01785375460}, 
    {     0.00000000037,  1.08711907413,    17085.95866572220}, 
    {     0.00000000039,  1.31034025695,    48739.85989708300}, 
    {     0.00000000047,  1.46478876042,    38500.27603107220}, 
    {     0.00000000045,  5.84053368365,    37674.99639427640}, 
    {     0.00000000036,  1.43280677914,    42456.78404709160}, 
    {     0.00000000039,  0.94257393865,    16522.65971600220}, 
    {     0.00000000035,  3.00235738584,    36147.40987730040}, 
    {     0.00000000031,  1.04908493403,    20235.12282631040}, 
    {     0.00000000029,  2.40482238520,    22779.43724619380}, 
    {     0.00000000032,  0.41152109043,    10207.76262190360}, 
    {     0.00000000028,  4.55982574192,    69166.43098950500}, 
    {     0.00000000026,  5.74236795898,    31441.67756975680}, 
    {     0.00000000027,  0.29541194003,    19999.97290154599}, 
    {     0.00000000033,  4.76470012256,    32858.61374281979}, 
    {     0.00000000032,  2.73060531796,    42430.48572729180}, 
    {     0.00000000031,  6.16859793641,    10192.51015071860}, 
    {     0.00000000022,  5.00575021321,    38741.49187146640}, 
    {     0.00000000023,  1.31610007722,    41962.52073693740}, 
    {     0.00000000025,  3.71847476472,    55798.45835839840}, 
    {     0.00000000020,  3.45607652624,    51092.72605085480}, 
    {     0.00000000022,  4.07064236572,    77026.85038194420}, 
    {     0.00000000020,  3.00992711288,    24383.07910844140}, 
    {     0.00000000023,  2.62973172963,    17248.42530185440}, 
};


static const struct ln_vsop ALIGN32 venus_longitude_l2[LONG_L2] = {
    {     0.00003894209,  0.34823650721,    10213.28554621100}, 
    {     0.00000595403,  2.01456107998,    20426.57109242200}, 
    {     0.00000287868,  0.00000000000,        0.00000000000}, 
    {     0.00000023838,  2.04588223604,       26.29831979980}, 
    {     0.00000009964,  3.97089333901,      775.52261132400}, 
    {     0.00000007196,  3.65730119531,    30639.85663863300}, 
    {     0.00000007043,  1.52107808192,     1577.34354244780}, 
    {     0.00000006014,  1.00039990357,      191.44826611160}, 
    {     0.00000003167,  4.36138169912,     9437.76293488700}, 
    {     0.00000001934,  3.39260216059,      382.89653222320}, 
    {     0.00000001459,  6.05311371882,      529.69096509460}, 
    {     0.00000001345,  2.94746266562,     5507.55323866740}, 
    {     0.00000001024,  1.40825326249,    10404.73381232260}, 
    {     0.00000001224,  3.73276078401,     3154.68708489560}, 
    {     0.00000001033,  3.52850062173,    11015.10647733480}, 
    {     0.00000000767,  2.69606070058,    40853.14218484400}, 
    {     0.00000000954,  5.11160150203,      801.82093112380}, 
    {     0.00000000742,  1.49195106907,     1109.37855209340}, 
    {     0.00000000525,  3.31953730020,      213.29909543800}, 
    {     0.00000000574,  0.92286899335,    10239.58386601080}, 
    {     0.00000000602,  5.19233174414,     7084.89678111520}, 
    {     0.00000000430,  2.67149013569,    13367.97263110660}, 
    {     0.00000000414,  4.12761290944,     8635.94200376320}, 
    {     0.00000000354,  1.12184896586,     9153.90361602180}, 
    {     0.00000000301,  3.90093359463,    10596.18207843420}, 
    {     0.00000000212,  5.32697962605,    18837.49819713819}, 
    {     0.00000000260,  0.22263894789,     2352.86615377180}, 
    {     0.00000000238,  4.67681098719,     6283.07584999140}, 
    {     0.00000000201,  4.08110855678,    11790.62908865880}, 
    {     0.00000000194,  6.02724392286,     7860.41939243920}, 
    {     0.00000000142,  2.26680009846,    12566.15169998280}, 
    {     0.00000000134,  4.10542992676,    17298.18232732620}, 
    {     0.00000000132,  4.94076356419,    14143.49524243060}, 
    {     0.00000000120,  3.39720208207,    11322.66409830440}, 
    {     0.00000000122,  5.92478855457,      574.34479833480}, 
    {     0.00000000123,  0.09583563880,    18073.70493865020}, 
    {     0.00000000093,  0.40064005550,     1059.38193018920}, 
    {     0.00000000092,  5.49232517605,    10021.83728009940}, 
    {     0.00000000084,  4.89826807621,    18307.80723204360}, 
    {     0.00000000074,  2.35249523388,      426.59819087600}, 
    {     0.00000000093,  4.99308589840,    14945.31617355440}, 
    {     0.00000000082,  5.40423993574,    10186.98722641120}, 
    {     0.00000000077,  3.75197709593,     3723.50895892300}, 
    {     0.00000000073,  4.51758491759,    22003.91463486980}, 
    {     0.00000000063,  5.40172448808,    21228.39202354580}, 
    {     0.00000000056,  4.11553639116,     7064.12138562280}, 
    {     0.00000000060,  4.93370008569,    19367.18916223280}, 
    {     0.00000000049,  3.25379905760,     4705.73230754360}, 
    {     0.00000000050,  3.77184609210,    16496.36139620240}, 
    {     0.00000000046,  3.47336429841,    24356.78078864160}, 
    {     0.00000000039,  4.90344944830,    10988.80815753500}, 
    {     0.00000000043,  5.95093813100,    15720.83878487840}, 
    {     0.00000000046,  0.17122496786,    18875.52586977400}, 
    {     0.00000000036,  0.01596424782,    32217.20018108080}, 
    {     0.00000000034,  4.09532663146,    51066.42773105500}, 
    {     0.00000000035,  5.21393402819,    29088.81141598500}, 
    {     0.00000000038,  5.09836948659,    20400.27277262220}, 
    {     0.00000000031,  4.74821153380,    28521.09277825460}, 
    {     0.00000000034,  0.23500259595,     1551.04522264800}, 
    {     0.00000000028,  6.22734256170,    19896.88012732740}, 
    {     0.00000000027,  2.91232520372,    20452.86941222180}, 
    {     0.00000000024,  0.49535690639,    21202.09370374600}, 
    {     0.00000000022,  0.66358494216,    25158.60171976540}, 
    {     0.00000000021,  4.95420122738,    28286.99048486120}, 
    {     0.00000000023,  2.15949646745,    20618.01935853360}, 
    {     0.00000000017,  5.75476455324,    19651.04848109800}, 
    {     0.00000000014,  4.88424354153,    10742.97651130560}, 
    {     0.00000000016,  1.22245134655,     9683.59458111640}, 
    {     0.00000000012,  3.40005378661,     9830.38901398780}, 
    {     0.00000000012,  6.06704683094,    25934.12433108940}, 
    {     0.00000000011,  3.73596829532,     4732.03062734340}, 
    {     0.00000000010,  5.50483756065,    29580.47470844380}, 
};


static const struct ln_vsop ALIGN32 venus_longitude_l3[LONG_L3] = {
    {     0.00000136328,  4.79698723753,    10213.28554621100}, 
    {     0.00000030661,  3.71663788064,    20426.57109242200}, 
    {     0.00000003041,  3.14159265359,        0.00000000000}, 
    {     0.00000000506,  5.34186957078,    30639.85663863300}, 
    {     0.00000000071,  4.27707588774,    40853.14218484400}, 
    {     0.00000000008,  1.76653383282,    10186.98722641120}, 
    {     0.00000000006,  5.61707828538,    10239.58386601080}, 
};


static const struct ln_vsop ALIGN32 venus_longitude_l4[LONG_L4] = {
    {     0.00000001636,  2.50540811485,    10213.28554621100}, 
    {     0.00000001080,  5.10106236574,    20426.57109242200}, 
    {     0.00000000018,  0.88315856739,    30639.85663863300}, 
    {     0.00000000005,  5.76650226003,    40853.14218484400}, 
};


static const struct ln_vsop ALIGN32 venus_longitude_l5[LONG_L5] = {
    {     0.00000000122,  1.88711724630,    10213.28554621100}, 
    {     0.00000000043,  0.42125909290,    20426.57109242200}, 
};


static const struct ln_vsop ALIGN32 venus_latitude_b0[LAT_B0] = {
    {     0.05923638472,  0.26702775812,    10213.28554621100}, 
    {     0.00040107978,  1.14737178112,    20426.57109242200}, 
    {     0.00032814918,  3.14159265359,        0.00000000000}, 
    {     0.00001011392,  1.08946119730,    30639.85663863300}, 
    {     0.00000149458,  6.25390268112,    18073.70493865020}, 
    {     0.00000137788,  0.86020095586,     1577.34354244780}, 
    {     0.00000129973,  3.67152480061,     9437.76293488700}, 
    {     0.00000119507,  3.70468787104,     2352.86615377180}, 
    {     0.00000107971,  4.53903678347,    22003.91463486980}, 
    {     0.00000092029,  1.53954519783,     9153.90361602180}, 
    {     0.00000052982,  2.28138198002,     5507.55323866740}, 
    {     0.00000045617,  0.72319646289,    10239.58386601080}, 
    {     0.00000038855,  2.93437865147,    10186.98722641120}, 
    {     0.00000043491,  6.14015779106,    11790.62908865880}, 
    {     0.00000041700,  5.99126840013,    19896.88012732740}, 
    {     0.00000039644,  3.86842103668,     8635.94200376320}, 
    {     0.00000039175,  3.94960158566,      529.69096509460}, 
    {     0.00000033320,  4.83194901518,    14143.49524243060}, 
    {     0.00000023711,  2.90647469167,    10988.80815753500}, 
    {     0.00000023501,  2.00771051056,    13367.97263110660}, 
    {     0.00000021809,  2.69701690731,    19651.04848109800}, 
    {     0.00000020653,  0.98666980431,      775.52261132400}, 
    {     0.00000016976,  4.13711781587,    10021.83728009940}, 
    {     0.00000017835,  5.96267283261,    25934.12433108940}, 
    {     0.00000014949,  5.61073907363,    10404.73381232260}, 
    {     0.00000018579,  1.80529274878,    40853.14218484400}, 
    {     0.00000015408,  3.29564350192,    11015.10647733480}, 
    {     0.00000012936,  5.42651380854,    29580.47470844380}, 
    {     0.00000011961,  3.57602108535,    10742.97651130560}, 
    {     0.00000011827,  1.19069755007,     8624.21265092720}, 
    {     0.00000011466,  5.12780356163,     6283.07584999140}, 
    {     0.00000009485,  2.75168410372,      191.44826611160}, 
    {     0.00000013129,  5.70734244216,     9683.59458111640}, 
    {     0.00000008583,  0.43182249198,     9786.68735533500}, 
    {     0.00000009762,  0.14614751233,    20618.01935853360}, 
    {     0.00000008149,  1.30548940012,    15720.83878487840}, 
    {     0.00000006050,  6.26541666310,    11322.66409830440}, 
    {     0.00000005955,  4.92234246004,     1059.38193018920}, 
    {     0.00000006983,  3.44922389028,    17298.18232732620}, 
    {     0.00000006228,  1.13312066423,    29864.33402730900}, 
    {     0.00000006186,  4.92496803755,    19367.18916223280}, 
    {     0.00000006155,  2.42413936688,     4705.73230754360}, 
    {     0.00000005204,  3.42528894645,     9103.90699411760}, 
    {     0.00000006000,  3.57638441137,     3154.68708489560}, 
    {     0.00000004795,  3.86669268805,     7860.41939243920}, 
    {     0.00000005290,  4.99182440452,     7084.89678111520}, 
    {     0.00000004070,  5.58794939725,    12566.15169998280}, 
    {     0.00000003942,  5.68758787835,    10206.17199921020}, 
    {     0.00000003797,  3.89520601076,    10192.51015071860}, 
    {     0.00000003798,  6.06410995916,    10234.06094170340}, 
    {     0.00000003579,  0.73789669232,     4551.95349705880}, 
    {     0.00000003641,  2.61501257209,    15874.61759536320}, 
    {     0.00000003266,  0.97519046577,    23581.25817731760}, 
    {     0.00000002813,  0.29952168976,     9411.46461508720}, 
    {     0.00000003048,  2.51084795752,    33794.54372352860}, 
    {     0.00000002559,  4.58049120677,      801.82093112380}, 
    {     0.00000002462,  5.05797190589,    29050.78374334920}, 
    {     0.00000002593,  5.73113176750,    20213.27199698400}, 
    {     0.00000002625,  4.24272906572,      213.29909543800}, 
    {     0.00000002246,  0.82114581737,    28286.99048486120}, 
    {     0.00000002229,  2.22457598233,    10426.58464164900}, 
    {     0.00000001742,  1.48397982399,     7058.59846131540}, 
    {     0.00000001660,  5.42775872173,    32217.20018108080}, 
    {     0.00000001491,  4.64880136152,     1109.37855209340}, 
    {     0.00000002010,  0.75702888130,     9999.98645077300}, 
    {     0.00000001563,  3.93962296253,    37724.75341974820}, 
    {     0.00000001538,  2.17314207854,    21535.94964451540}, 
    {     0.00000001546,  4.70759129230,    14945.31617355440}, 
    {     0.00000001200,  1.48266337604,     9830.38901398780}, 
    {     0.00000001224,  5.55090394425,     5661.33204915220}, 
    {     0.00000001111,  1.20281402179,     9573.38825989700}, 
    {     0.00000001064,  1.98891375551,       26.29831979980}, 
    {     0.00000001041,  5.38535116069,        7.11354700080}, 
    {     0.00000001036,  1.16727012915,     8662.24032356300}, 
    {     0.00000001143,  3.20596958339,     3532.06069281140}, 
    {     0.00000001201,  0.81922498884,     8094.52168583260}, 
    {     0.00000001005,  2.38430401606,    27511.46787353720}, 
    {     0.00000001047,  4.56525030764,    20419.45754542119}, 
    {     0.00000000968,  6.18496285828,    25158.60171976540}, 
    {     0.00000001044,  1.98033274400,    10596.18207843420}, 
    {     0.00000000962,  0.48573513747,    23958.63178523340}, 
    {     0.00000000846,  0.01577122167,     3128.38876509580}, 
    {     0.00000000792,  5.39683788914,    24356.78078864160}, 
    {     0.00000000858,  5.34721623594,    41654.96311596780}, 
    {     0.00000000758,  6.25923864937,    20452.86941222180}, 
    {     0.00000000801,  4.62406152514,     9929.42622734580}, 
    {     0.00000000802,  5.37234892520,    10497.14486507620}, 
    {     0.00000000750,  3.85221291360,    21228.39202354580}, 
    {     0.00000000701,  1.98087454891,     3930.20969621960}, 
    {     0.00000000719,  6.11596800207,    10218.80847051840}, 
    {     0.00000000672,  6.23429601211,    14765.23904326980}, 
    {     0.00000000639,  5.37595576896,     1589.07289528380}, 
    {     0.00000000605,  2.42330391120,    10251.31321884680}, 
    {     0.00000000726,  6.16679714176,    18875.52586977400}, 
    {     0.00000000613,  5.99731180692,     4732.03062734340}, 
    {     0.00000000720,  3.84286345199,    10207.76262190360}, 
    {     0.00000000637,  6.17053891151,    10220.39909321180}, 
    {     0.00000000515,  1.03004255543,    22779.43724619380}, 
    {     0.00000000574,  0.43826866390,    17085.95866572220}, 
    {     0.00000000510,  1.41066951048,     9161.01716302260}, 
    {     0.00000000569,  3.34606383399,     3340.61242669980}, 
    {     0.00000000608,  1.25236241968,    10175.25787357520}, 
    {     0.00000000524,  2.39794248664,    26087.90314157420}, 
    {     0.00000000543,  1.34669062643,    29088.81141598500}, 
    {     0.00000000527,  4.02005650673,    18849.22754997420}, 
    {     0.00000000568,  1.65449906239,    39264.06928956020}, 
    {     0.00000000518,  4.96996115441,    30213.25844775700}, 
    {     0.00000000515,  5.78424359731,    12592.45001978260}, 
    {     0.00000000538,  4.56198503482,    10063.72234907640}, 
    {     0.00000000484,  4.18570967634,    14919.01785375460}, 
    {     0.00000000493,  4.79953014808,     9146.79006902100}, 
    {     0.00000000427,  3.76882042266,    11272.66747640020}, 
    {     0.00000000495,  0.49176119118,    45585.17281218740}, 
    {     0.00000000495,  3.74294537743,    31441.67756975680}, 
    {     0.00000000524,  0.97969838972,    30110.16567353840}, 
    {     0.00000000483,  1.87898083461,    51066.42773105500}, 
    {     0.00000000505,  3.70047474052,    20400.27277262220}, 
    {     0.00000000351,  4.34026574490,    10137.01947493540}, 
    {     0.00000000356,  5.56743301907,    18837.49819713819}, 
    {     0.00000000328,  3.78427378911,     6681.22485339960}, 
    {     0.00000000349,  4.20513745358,    20956.26205751660}, 
    {     0.00000000333,  4.44934899983,    28521.09277825460}, 
    {     0.00000000296,  2.83205515647,    17277.40693183380}, 
    {     0.00000000311,  2.57313811169,    20809.46762464520}, 
    {     0.00000000294,  0.75089224483,     3149.16416058820}, 
    {     0.00000000377,  3.98143310344,    21202.09370374600}, 
    {     0.00000000272,  5.56146064111,    16496.36139620240}, 
    {     0.00000000314,  0.02584607106,    13745.34623902240}, 
    {     0.00000000263,  0.55359290511,    36147.40987730040}, 
    {     0.00000000286,  5.16408902152,      426.59819087600}, 
    {     0.00000000279,  4.29871616015,    19999.97290154599}, 
    {     0.00000000280,  1.92923790310,    49515.38250840700}, 
    {     0.00000000265,  4.81161712090,    20235.12282631040}, 
    {     0.00000000273,  5.12739677980,    35371.88726597640}, 
    {     0.00000000307,  5.28914601501,      382.89653222320}, 
    {     0.00000000223,  2.50591295336,    26709.64694241340}, 
    {     0.00000000235,  5.96522395118,    10198.03307502600}, 
    {     0.00000000234,  3.52866583267,    10228.53801739600}, 
    {     0.00000000224,  6.24561979791,     7064.12138562280}, 
    {     0.00000000251,  2.84752569358,    33019.02111220460}, 
    {     0.00000000196,  1.50610823821,    31749.23519072640}, 
    {     0.00000000192,  1.69322019350,    13341.67431130680}, 
    {     0.00000000180,  6.19353102835,    39793.76025465480}, 
    {     0.00000000199,  1.16466425386,    22805.73556599360}, 
    {     0.00000000180,  3.72630445822,     1551.04522264800}, 
    {     0.00000000172,  3.35254224852,    53445.59220462660}, 
    {     0.00000000195,  1.51901264133,    43232.30665841560}, 
    {     0.00000000174,  2.84049662693,     9967.45389998160}, 
    {     0.00000000163,  4.29008955007,    36949.23080842420}, 
    {     0.00000000169,  0.37000676558,    10459.11719244040}, 
    {     0.00000000137,  5.61149803116,    10529.67741586760}, 
    {     0.00000000139,  0.87990954673,    16522.65971600220}, 
    {     0.00000000139,  4.12576475412,    36301.18868778519}, 
    {     0.00000000127,  5.14447758617,     5481.25491886760}, 
    {     0.00000000131,  3.11317801589,     9896.89367655440}, 
    {     0.00000000131,  0.89697384734,     3442.57494496540}, 
    {     0.00000000121,  1.32914080245,    38734.37832446560}, 
    {     0.00000000122,  1.59017183044,    10110.19277199240}, 
    {     0.00000000123,  2.33714216061,    10316.37832042960}, 
    {     0.00000000133,  2.90682399304,     9793.80090233580}, 
    {     0.00000000111,  2.51956905782,    13936.79450513400}, 
    {     0.00000000120,  0.36015902371,      536.80451209540}, 
    {     0.00000000115,  2.53305863643,    26735.94526221320}, 
    {     0.00000000108,  2.65839634325,    10232.95530711079}, 
    {     0.00000000108,  0.55230439694,    10193.61578531121}, 
    {     0.00000000138,  1.06919239242,    65236.22129328540}, 
    {     0.00000000101,  3.17003240566,    19317.19254032860}, 
    {     0.00000000127,  5.63110477712,    10288.06714477830}, 
    {     0.00000000127,  3.86278127025,    10138.50394764370}, 
    {     0.00000000137,  2.93350659462,    47162.51635463520}, 
    {     0.00000000095,  5.03917884339,    52175.80628314840}, 
    {     0.00000000094,  0.71305001399,    38500.27603107220}, 
    {     0.00000000093,  5.46196838829,    11764.33076885900}, 
    {     0.00000000096,  1.52914774412,     9690.70812811720}, 
    {     0.00000000101,  0.83318284424,     6489.77658728800}, 
    {     0.00000000115,  3.76443612240,      522.57741809380}, 
    {     0.00000000089,  2.53312656681,    10735.86296430480}, 
    {     0.00000000082,  0.85729618993,     2379.16447357160}, 
    {     0.00000000103,  5.22683237626,      103.09277421860}, 
    {     0.00000000090,  2.12423586627,    28313.28880466100}, 
    {     0.00000000090,  0.39668501734,     9580.50180689780}, 
    {     0.00000000074,  6.02798429860,     3723.50895892300}, 
    {     0.00000000081,  5.25045057985,    10419.47109464820}, 
    {     0.00000000080,  4.23724598221,    10007.09999777380}, 
    {     0.00000000091,  2.48874147948,    10846.06928552420}, 
    {     0.00000000085,  3.82491067322,    51868.24866217880}, 
    {     0.00000000081,  2.26186773193,     3903.91137641980}, 
    {     0.00000000097,  0.77436917916,    18307.80723204360}, 
    {     0.00000000094,  0.17064745052,     6872.67311951120}, 
    {     0.00000000080,  5.62388508292,    29999.95935231900}, 
    {     0.00000000068,  2.71544542018,    16983.99614745660}, 
    {     0.00000000076,  0.36189761470,    39302.09696219600}, 
    {     0.00000000066,  0.76731351843,       20.77539549240}, 
    {     0.00000000075,  2.27327165974,     8521.11987670860}, 
    {     0.00000000058,  2.14482855875,     8631.32619792800}, 
    {     0.00000000064,  5.83567991327,     2118.76386037840}, 
    {     0.00000000058,  2.98359709521,    19889.76658032659}, 
    {     0.00000000054,  1.78252843890,    40077.61957352000}, 
    {     0.00000000055,  4.70485939801,      639.89728631400}, 
    {     0.00000000060,  5.89548057045,    41962.52073693740}, 
    {     0.00000000066,  2.24746237992,       74.78159856730}, 
    {     0.00000000061,  3.40726181591,    27490.69247804480}, 
    {     0.00000000051,  3.07546032535,    24150.08005134500}, 
    {     0.00000000057,  2.30081371235,    20529.66386664059}, 
    {     0.00000000051,  2.37255859198,    29573.36116144300}, 
    {     0.00000000052,  4.76676085839,    57375.80190084620}, 
    {     0.00000000054,  5.89684197257,    19903.99367432819}, 
    {     0.00000000047,  1.61776744350,    30831.30490474460}, 
    {     0.00000000040,  5.32238959879,    42430.48572729180}, 
    {     0.00000000051,  5.29366964725,    29587.58825544460}, 
};


static const struct ln_vsop ALIGN32 venus_latitude_b1[LAT_B1] = {
    {     0.00287821243,  1.88964962838,    10213.28554621100}, 
    {     0.00003499578,  3.71117560516,    20426.57109242200}, 
    {     0.00001257844,  0.00000000000,        0.00000000000}, 
    {     0.00000096152,  2.74240664188,    30639.85663863300}, 
    {     0.00000013051,  2.27549606211,     9437.76293488700}, 
    {     0.00000008052,  5.55049163175,     1577.34354244780}, 
    {     0.00000006792,  1.60704519868,    18073.70493865020}, 
    {     0.00000007521,  0.28931916342,     9153.90361602180}, 
    {     0.00000005612,  1.59167143282,    11790.62908865880}, 
    {     0.00000006460,  5.23429470615,    10186.98722641120}, 
    {     0.00000005405,  6.15384515120,    22003.91463486980}, 
    {     0.00000004194,  5.40627767662,     2352.86615377180}, 
    {     0.00000005097,  4.17404177420,    10239.58386601080}, 
    {     0.00000004085,  0.73618047930,     5507.55323866740}, 
    {     0.00000003349,  1.34464128958,    10404.73381232260}, 
    {     0.00000002941,  4.40460505149,    19651.04848109800}, 
    {     0.00000002670,  3.92292198547,    40853.14218484400}, 
    {     0.00000002746,  5.04771993757,    10988.80815753500}, 
    {     0.00000002216,  1.74777661827,     8635.94200376320}, 
    {     0.00000001934,  1.35730576040,    19896.88012732740}, 
    {     0.00000001960,  0.38471810829,    13367.97263110660}, 
    {     0.00000001859,  5.60931743743,      529.69096509460}, 
    {     0.00000001586,  2.31959766692,    10021.83728009940}, 
    {     0.00000001733,  3.34619758004,     6283.07584999140}, 
    {     0.00000001227,  1.71667580691,    20618.01935853360}, 
    {     0.00000001223,  6.18665596267,     7860.41939243920}, 
    {     0.00000000980,  2.99089645869,    15720.83878487840}, 
    {     0.00000000961,  5.06770207004,      775.52261132400}, 
    {     0.00000000976,  1.28004342015,    25934.12433108940}, 
    {     0.00000000785,  1.77460186481,    11322.66409830440}, 
    {     0.00000000727,  0.60317628984,     8624.21265092720}, 
    {     0.00000000756,  0.68622306427,    19367.18916223280}, 
    {     0.00000000625,  0.74326657001,    29580.47470844380}, 
    {     0.00000000525,  5.08155157418,     9786.68735533500}, 
    {     0.00000000553,  1.74144182497,    17298.18232732620}, 
    {     0.00000000503,  2.68460156951,    23581.25817731760}, 
    {     0.00000000467,  0.17554338129,    11015.10647733480}, 
    {     0.00000000400,  0.33304670031,     4705.73230754360}, 
    {     0.00000000438,  2.30926644329,    10742.97651130560}, 
    {     0.00000000356,  1.05880387658,    14143.49524243060}, 
    {     0.00000000359,  4.84089408623,     3930.20969621960}, 
    {     0.00000000431,  3.73519308368,    10596.18207843420}, 
    {     0.00000000312,  6.24028343022,     9830.38901398780}, 
    {     0.00000000321,  3.83963668925,    21228.39202354580}, 
    {     0.00000000329,  2.71641787634,    29864.33402730900}, 
    {     0.00000000267,  1.90575325749,     9103.90699411760}, 
    {     0.00000000256,  4.22499119257,      191.44826611160}, 
    {     0.00000000259,  1.81366693404,     1059.38193018920}, 
    {     0.00000000348,  2.27789745461,     9999.98645077300}, 
    {     0.00000000258,  2.61400704653,     9411.46461508720}, 
    {     0.00000000242,  3.18580218709,    12566.15169998280}, 
    {     0.00000000294,  4.38485988087,    15874.61759536320}, 
    {     0.00000000274,  4.80172249298,    10206.17199921020}, 
    {     0.00000000243,  6.15371511696,    20419.45754542119}, 
    {     0.00000000216,  2.14070499134,      801.82093112380}, 
    {     0.00000000195,  3.43788708589,    28286.99048486120}, 
    {     0.00000000203,  4.40417469081,    10220.39909321180}, 
    {     0.00000000204,  1.39050905939,    10234.06094170340}, 
    {     0.00000000177,  6.13411796219,    10426.58464164900}, 
    {     0.00000000174,  6.13034607143,     9161.01716302260}, 
    {     0.00000000186,  3.77010906124,     7058.59846131540}, 
    {     0.00000000167,  5.56979067201,    10192.51015071860}, 
    {     0.00000000197,  0.07905532678,    29050.78374334920}, 
    {     0.00000000186,  4.07019341948,    33794.54372352860}, 
    {     0.00000000155,  3.66480557020,        7.11354700080}, 
    {     0.00000000138,  5.96011224678,     3154.68708489560}, 
    {     0.00000000158,  5.50203206635,    37724.75341974820}, 
    {     0.00000000147,  3.62347099998,    21535.94964451540}, 
    {     0.00000000141,  1.64907514042,    32217.20018108080}, 
    {     0.00000000168,  6.17942898657,    18837.49819713819}, 
    {     0.00000000149,  4.71613322834,    25158.60171976540}, 
    {     0.00000000124,  4.14558329896,    27511.46787353720}, 
    {     0.00000000117,  1.26060116043,     7084.89678111520}, 
    {     0.00000000112,  2.64211360535,    22779.43724619380}, 
    {     0.00000000111,  1.56882986044,     9683.59458111640}, 
    {     0.00000000125,  5.85773130047,      213.29909543800}, 
    {     0.00000000106,  1.60206102260,    10207.76262190360}, 
    {     0.00000000106,  1.60857575234,    10218.80847051840}, 
    {     0.00000000107,  0.06029270673,    20400.27277262220}, 
    {     0.00000000105,  5.17582412260,     8662.24032356300}, 
    {     0.00000000115,  0.14229628995,     9146.79006902100}, 
    {     0.00000000118,  0.93060644045,    20213.27199698400}, 
    {     0.00000000134,  5.76506264779,       26.29831979980}, 
    {     0.00000000100,  0.63855627551,    41654.96311596780}, 
    {     0.00000000093,  2.39421056035,     3128.38876509580}, 
    {     0.00000000083,  0.24054840895,    14919.01785375460}, 
    {     0.00000000086,  1.99704954707,    10228.53801739600}, 
    {     0.00000000086,  1.21365119312,    10198.03307502600}, 
    {     0.00000000076,  6.15164147653,    21202.09370374600}, 
    {     0.00000000076,  2.50538996449,     4551.95349705880}, 
    {     0.00000000081,  3.54317093055,    51066.42773105500}, 
    {     0.00000000070,  3.62977580498,    20452.86941222180}, 
    {     0.00000000069,  3.33801158730,    11272.66747640020}, 
    {     0.00000000066,  2.40044362019,    16496.36139620240}, 
    {     0.00000000074,  0.81937069238,      382.89653222320}, 
    {     0.00000000084,  1.84615505638,    45585.17281218740}, 
    {     0.00000000058,  5.54500123157,     5661.33204915220}, 
    {     0.00000000062,  6.22847666967,    29088.81141598500}, 
    {     0.00000000048,  5.23337244147,     3149.16416058820}, 
    {     0.00000000045,  4.23847970134,    20809.46762464520}, 
    {     0.00000000043,  3.13312030807,    30110.16567353840}, 
    {     0.00000000042,  2.05265738782,    24356.78078864160}, 
    {     0.00000000048,  1.26595147651,    33019.02111220460}, 
    {     0.00000000056,  3.26535945634,    49515.38250840700}, 
    {     0.00000000049,  4.20217286778,    13341.67431130680}, 
    {     0.00000000042,  2.07085732347,      426.59819087600}, 
    {     0.00000000044,  1.20392434392,    12592.45001978260}, 
    {     0.00000000036,  5.57938191798,    31441.67756975680}, 
    {     0.00000000038,  4.98083686092,      536.80451209540}, 
    {     0.00000000043,  1.90863602359,     3532.06069281140}, 
    {     0.00000000035,  3.83053326183,    18849.22754997420}, 
    {     0.00000000032,  5.81052626816,    28521.09277825460}, 
    {     0.00000000028,  3.80386218448,     4732.03062734340}, 
    {     0.00000000031,  5.41631373906,      522.57741809380}, 
    {     0.00000000026,  2.11998839024,    23958.63178523340}, 
    {     0.00000000023,  4.45861699215,    17277.40693183380}, 
    {     0.00000000022,  1.46296031303,     8094.52168583260}, 
    {     0.00000000024,  6.20898418337,    26087.90314157420}, 
    {     0.00000000020,  4.66182643911,    19889.76658032659}, 
    {     0.00000000020,  0.98405518609,    19999.97290154599}, 
    {     0.00000000023,  3.22003976882,    39264.06928956020}, 
};


static const struct ln_vsop ALIGN32 venus_latitude_b2[LAT_B2] = {
    {     0.00012657745,  3.34796457029,    10213.28554621100}, 
    {     0.00000151225,  0.00000000000,        0.00000000000}, 
    {     0.00000037476,  5.34638962141,    20426.57109242200}, 
    {     0.00000010627,  3.81894300538,    30639.85663863300}, 
    {     0.00000000942,  0.01311905561,    10186.98722641120}, 
    {     0.00000000747,  4.13620174126,    10239.58386601080}, 
    {     0.00000000434,  3.26791015348,    11790.62908865880}, 
    {     0.00000000439,  6.04066783494,    10988.80815753500}, 
    {     0.00000000331,  6.20632270120,     9437.76293488700}, 
    {     0.00000000332,  2.82983937642,    10404.73381232260}, 
    {     0.00000000340,  5.14190117554,     9153.90361602180}, 
    {     0.00000000244,  6.07029755311,    19651.04848109800}, 
    {     0.00000000274,  0.56238407651,     2352.86615377180}, 
    {     0.00000000273,  3.09551287480,    18073.70493865020}, 
    {     0.00000000268,  4.02642385642,     1577.34354244780}, 
    {     0.00000000217,  5.37498456751,    40853.14218484400}, 
    {     0.00000000213,  1.22633958472,    22003.91463486980}, 
    {     0.00000000141,  5.42727181668,     5507.55323866740}, 
    {     0.00000000130,  5.26806549972,    13367.97263110660}, 
    {     0.00000000101,  4.64089661780,    15720.83878487840}, 
    {     0.00000000097,  0.79512785540,    10021.83728009940}, 
    {     0.00000000095,  0.22330149810,     8635.94200376320}, 
    {     0.00000000090,  2.70556129424,    19896.88012732740}, 
    {     0.00000000090,  0.51306700906,      529.69096509460}, 
    {     0.00000000084,  2.50344831127,     9683.59458111640}, 
    {     0.00000000070,  3.27104614873,    20618.01935853360}, 
    {     0.00000000058,  3.30656672462,    11322.66409830440}, 
    {     0.00000000075,  5.25593390437,    10596.18207843420}, 
    {     0.00000000045,  4.74613910444,     7058.59846131540}, 
    {     0.00000000039,  4.73196017252,     9830.38901398780}, 
    {     0.00000000046,  3.81232018036,      775.52261132400}, 
    {     0.00000000034,  3.64209909537,    14143.49524243060}, 
    {     0.00000000031,  4.03554572158,     6283.07584999140}, 
    {     0.00000000036,  2.08483265333,      801.82093112380}, 
    {     0.00000000034,  2.85714821360,     9411.46461508720}, 
    {     0.00000000023,  0.53259218563,    10206.17199921020}, 
    {     0.00000000023,  5.81970120815,    10220.39909321180}, 
    {     0.00000000029,  0.34238212730,    11015.10647733480}, 
    {     0.00000000021,  4.61555386478,     9999.98645077300}, 
    {     0.00000000020,  4.88064054230,    10426.58464164900}, 
    {     0.00000000026,  2.20763733076,    18849.22754997420}, 
    {     0.00000000022,  3.98619109775,    10742.97651130560}, 
    {     0.00000000019,  4.86026864659,     4705.73230754360}, 
    {     0.00000000018,  4.37664030659,    23581.25817731760}, 
    {     0.00000000018,  2.57254812330,    19367.18916223280}, 
    {     0.00000000018,  5.52181260638,    21228.39202354580}, 
    {     0.00000000021,  2.75902377111,    25934.12433108940}, 
    {     0.00000000019,  2.22080338031,    29580.47470844380}, 
    {     0.00000000017,  2.48643394548,     3128.38876509580}, 
    {     0.00000000014,  6.20079529340,      191.44826611160}, 
    {     0.00000000015,  5.74322714868,       26.29831979980}, 
};


static const struct ln_vsop ALIGN32 venus_latitude_b3[LAT_B3] = {
    {     0.00000376505,  4.87650249694,    10213.28554621100}, 
    {     0.00000012587,  3.14159265359,        0.00000000000}, 
    {     0.00000004809,  0.43423918018,    20426.57109242200}, 
    {     0.00000000835,  5.57179521329,    30639.85663863300}, 
    {     0.00000000046,  1.54914240166,    10186.98722641120}, 
    {     0.00000000034,  5.78743368814,    10239.58386601080}, 
    {     0.00000000019,  0.76227312580,    40853.14218484400}, 
    {     0.00000000013,  4.63920619996,    11790.62908865880}, 
    {     0.00000000014,  5.47124304598,     9437.76293488700}, 
    {     0.00000000014,  0.87652853636,    10988.80815753500}, 
    {     0.00000000010,  1.84595947891,     2352.86615377180}, 
    {     0.00000000005,  1.36825703657,    19651.04848109800}, 
};


static const struct ln_vsop ALIGN32 venus_latitude_b4[LAT_B4] = {
    {     0.00000008558,  0.17181972054,    10213.28554621100}, 
    {     0.00000000114,  2.50366130090,    20426.57109242200}, 
    {     0.00000000115,  3.14159265359,        0.00000000000}, 
    {     0.00000000041,  0.74061432691,    30639.85663863300}, 
};


static const struct ln_vsop ALIGN32 venus_latitude_b5[LAT_B5] = {
    {     0.00000000149,  1.67437168506,    10213.28554621100}, 
    {     0.00000000023,  0.00000000000,        0.00000000000}, 
    {     0.00000000008,  3.73924477319,    20426.57109242200}, 
    {     0.00000000002,  2.28783748701,    30639.85663863300}, 
};


static const struct ln_vsop ALIGN32 venus_radius_r0[RADIUS_R0] = {
    {     0.72334820891,  0.00000000000,        0.00000000000}, 
    {     0.00489824182,  4.02151831717,    10213.28554621100}, 
    {     0.00001658058,  4.90206728031,    20426.57109242200}, 
    {     0.00001632096,  2.84548795207,     7860.41939243920}, 
    {     0.00001378043,  1.12846591367,    11790.62908865880}, 
    {     0.00000498395,  2.58682193892,     9683.59458111640}, 
    {     0.00000373958,  1.42314832858,     3930.20969621960}, 
    {     0.00000263615,  5.52938716941,     9437.76293488700}, 
    {     0.00000237454,  2.55136053886,    15720.83878487840}, 
    {     0.00000221985,  2.01346696541,    19367.18916223280}, 
    {     0.00000119466,  3.01975080538,    10404.73381232260}, 
    {     0.00000125896,  2.72769850819,     1577.34354244780}, 
    {     0.00000076176,  1.59574968674,     9153.90361602180}, 
    {     0.00000085337,  3.98598666191,    19651.04848109800}, 
    {     0.00000074347,  4.11957779786,     5507.55323866740}, 
    {     0.00000041902,  1.64282225331,    18837.49819713819}, 
    {     0.00000042494,  3.81864493274,    13367.97263110660}, 
    {     0.00000039437,  5.39018702243,    23581.25817731760}, 
    {     0.00000029042,  5.67739528728,     5661.33204915220}, 
    {     0.00000027555,  5.72392434415,      775.52261132400}, 
    {     0.00000027288,  4.82140494620,    11015.10647733480}, 
    {     0.00000031274,  2.31806719544,     9999.98645077300}, 
    {     0.00000019700,  4.96157560246,    11322.66409830440}, 
    {     0.00000019811,  0.53189302682,    27511.46787353720}, 
    {     0.00000013569,  3.75536825122,    18073.70493865020}, 
    {     0.00000012921,  1.13381083556,    10206.17199921020}, 
    {     0.00000016214,  0.56446585474,      529.69096509460}, 
    {     0.00000011828,  5.09037966560,     3154.68708489560}, 
    {     0.00000011729,  0.23450811362,     7084.89678111520}, 
    {     0.00000013066,  5.24354222739,    17298.18232732620}, 
    {     0.00000013180,  3.37207825651,    13745.34623902240}, 
    {     0.00000009097,  3.07004839111,     1109.37855209340}, 
    {     0.00000010818,  2.45024714924,    10239.58386601080}, 
    {     0.00000011434,  4.56780914249,    29050.78374334920}, 
    {     0.00000008377,  5.78327641089,    30639.85663863300}, 
    {     0.00000008193,  1.95023341446,    22003.91463486980}, 
    {     0.00000009319,  1.61646997033,     2352.86615377180}, 
    {     0.00000010653,  1.95585283247,    31441.67756975680}, 
    {     0.00000010357,  1.20234990063,    15874.61759536320}, 
    {     0.00000009585,  1.46639856227,    19999.97290154599}, 
    {     0.00000006504,  2.17386891309,    14143.49524243060}, 
    {     0.00000007555,  1.13845893425,     8624.21265092720}, 
    {     0.00000006438,  0.84494385894,     6283.07584999140}, 
    {     0.00000005895,  0.01147034372,     8635.94200376320}, 
    {     0.00000005633,  3.94955705099,    12566.15169998280}, 
    {     0.00000005521,  1.27351436322,    18307.80723204360}, 
    {     0.00000004488,  2.47835713175,      191.44826611160}, 
    {     0.00000004524,  4.73049812703,    19896.88012732740}, 
    {     0.00000006182,  3.25901855473,     6872.67311951120}, 
    {     0.00000006076,  0.35368446866,    21228.39202354580}, 
    {     0.00000004315,  2.59737099527,     4551.95349705880}, 
    {     0.00000006001,  3.37877140835,    35371.88726597640}, 
    {     0.00000003852,  1.01162850363,     9786.68735533500}, 
    {     0.00000004033,  0.00058119465,      801.82093112380}, 
    {     0.00000003913,  5.56523869375,    10596.18207843420}, 
    {     0.00000002709,  5.80195530113,     7064.12138562280}, 
    {     0.00000003216,  0.39767254849,    10186.98722641120}, 
    {     0.00000003089,  6.26174762875,    14945.31617355440}, 
    {     0.00000002983,  4.21306731485,    28521.09277825460}, 
    {     0.00000003280,  0.70603484350,    10742.97651130560}, 
    {     0.00000003495,  4.79876624499,    39302.09696219600}, 
    {     0.00000003172,  1.80518718365,    25158.60171976540}, 
    {     0.00000002463,  0.68708155159,    10988.80815753500}, 
    {     0.00000002374,  3.77948972590,    21535.94964451540}, 
    {     0.00000002199,  2.82965458882,     8662.24032356300}, 
    {     0.00000001958,  5.41763443097,    16496.36139620240}, 
    {     0.00000001876,  2.63426835834,    29580.47470844380}, 
    {     0.00000001902,  2.85782199151,     3532.06069281140}, 
    {     0.00000001706,  3.67573010387,       26.29831979980}, 
    {     0.00000001820,  0.41906719561,     4705.73230754360}, 
    {     0.00000001857,  1.50350423721,    10021.83728009940}, 
    {     0.00000002086,  6.22149960504,    43232.30665841560}, 
    {     0.00000001945,  2.21581377123,    19786.67380610799}, 
    {     0.00000001497,  0.00134773818,    17277.40693183380}, 
    {     0.00000001831,  3.23180044003,    29088.81141598500}, 
    {     0.00000001422,  5.85815118404,     9676.48103411560}, 
    {     0.00000001223,  5.55818994329,     6770.71060124560}, 
    {     0.00000001135,  5.91731919731,    13936.79450513400}, 
    {     0.00000001483,  2.47551114520,    31749.23519072640}, 
    {     0.00000001185,  1.42087628352,     4732.03062734340}, 
    {     0.00000001323,  2.48821075422,     9690.70812811720}, 
    {     0.00000001237,  1.88503842890,    19374.30270923360}, 
    {     0.00000001402,  5.17536780118,    10316.37832042960}, 
    {     0.00000001257,  5.24711268081,    19360.07561523199}, 
    {     0.00000001041,  3.05620764519,    25934.12433108940}, 
    {     0.00000001174,  1.42916822908,    18875.52586977400}, 
    {     0.00000001277,  1.35812964004,    47162.51635463520}, 
    {     0.00000000919,  6.26767378713,    20618.01935853360}, 
    {     0.00000000907,  1.12515741342,    12592.45001978260}, 
    {     0.00000001093,  4.64451665343,    33019.02111220460}, 
    {     0.00000001012,  1.09063299331,     1059.38193018920}, 
    {     0.00000000783,  2.02117411797,    24356.78078864160}, 
    {     0.00000000779,  0.41587370509,     3340.61242669980}, 
    {     0.00000000702,  1.15494723276,    16983.99614745660}, 
    {     0.00000000878,  0.87851267907,    38734.37832446560}, 
    {     0.00000000620,  0.90736173080,    17778.11626694899}, 
    {     0.00000000608,  1.58478313439,     9573.38825989700}, 
    {     0.00000000800,  3.94213003073,    10138.50394764370}, 
    {     0.00000000758,  1.31882722638,     9967.45389998160}, 
    {     0.00000000802,  2.78173370208,    51092.72605085480}, 
    {     0.00000000664,  4.45866629009,     3128.38876509580}, 
    {     0.00000000668,  5.10542540579,      382.89653222320}, 
    {     0.00000000530,  0.85392938403,    10234.06094170340}, 
    {     0.00000000509,  3.56810277681,    28286.99048486120}, 
    {     0.00000000600,  4.25927697166,    41962.52073693740}, 
    {     0.00000000601,  5.78144137927,      213.29909543800}, 
    {     0.00000000595,  2.83043640070,    22805.73556599360}, 
    {     0.00000000673,  6.06080293558,    36949.23080842420}, 
    {     0.00000000537,  5.85435364687,     9103.90699411760}, 
    {     0.00000000544,  5.44804869552,     3723.50895892300}, 
    {     0.00000000493,  3.84940583750,    27991.40181316000}, 
    {     0.00000000635,  0.76493226364,     8094.52168583260}, 
    {     0.00000000435,  6.21370952619,    27197.28169366760}, 
    {     0.00000000459,  3.55062885409,    20213.27199698400}, 
    {     0.00000000398,  6.16269975784,    10426.58464164900}, 
    {     0.00000000378,  2.41665947591,    18844.61174413899}, 
    {     0.00000000421,  4.86558476805,     9146.79006902100}, 
    {     0.00000000500,  4.20351458644,    55022.93574707440}, 
    {     0.00000000404,  4.96112893536,    37410.56723987860}, 
    {     0.00000000402,  2.97963246945,    10220.39909321180}, 
    {     0.00000000464,  2.59869499733,    18734.40542291960}, 
    {     0.00000000353,  0.08229680434,    10103.07922499160}, 
    {     0.00000000348,  4.90260339364,    18830.38465013739}, 
    {     0.00000000339,  3.23485348169,    24150.08005134500}, 
    {     0.00000000431,  1.21298886779,    40879.44050464380}, 
    {     0.00000000375,  6.17532088119,    26087.90314157420}, 
    {     0.00000000409,  3.12070020561,     9050.81084180320}, 
    {     0.00000000385,  1.94284690176,      283.85931886520}, 
    {     0.00000000331,  4.89790702632,    12432.04265039780}, 
    {     0.00000000330,  4.29239717956,    26735.94526221320}, 
    {     0.00000000310,  0.50879975679,    38204.68735937100}, 
    {     0.00000000329,  3.88427322515,    29864.33402730900}, 
    {     0.00000000313,  1.36138752543,    10192.51015071860}, 
    {     0.00000000347,  3.58439807220,    27490.69247804480}, 
    {     0.00000000251,  3.78618457047,    10063.72234907640}, 
    {     0.00000000245,  3.84553750369,     9411.46461508720}, 
    {     0.00000000281,  4.50895217144,    32217.20018108080}, 
    {     0.00000000237,  0.87748812264,     6681.22485339960}, 
    {     0.00000000315,  5.62657778233,    58953.14544329400}, 
    {     0.00000000311,  4.15626121491,    10175.15251057320}, 
    {     0.00000000251,  2.53793253681,    16522.65971600220}, 
    {     0.00000000218,  5.08472965959,     7058.59846131540}, 
    {     0.00000000291,  3.72573045764,    29999.95935231900}, 
    {     0.00000000267,  2.97685503991,    19573.37471066999}, 
    {     0.00000000280,  3.70200084294,    47623.85278608960}, 
    {     0.00000000239,  3.94545782067,     9580.50180689780}, 
    {     0.00000000246,  2.18261064090,     9161.01716302260}, 
    {     0.00000000269,  2.62767323960,    44809.65020086340}, 
    {     0.00000000254,  2.69185768875,     3442.57494496540}, 
    {     0.00000000194,  4.79747763686,    33794.54372352860}, 
    {     0.00000000203,  2.54900468678,     2379.16447357160}, 
    {     0.00000000210,  1.90935455520,     1551.04522264800}, 
    {     0.00000000187,  3.65645008095,    20452.86941222180}, 
    {     0.00000000224,  2.43601863127,     9992.87290377220}, 
    {     0.00000000190,  6.19594324465,    36147.40987730040}, 
    {     0.00000000176,  4.29837616553,    10137.01947493540}, 
    {     0.00000000175,  2.71993327300,    20809.46762464520}, 
    {     0.00000000186,  2.55096983527,    14919.01785375460}, 
    {     0.00000000161,  4.13272566951,    23958.63178523340}, 
    {     0.00000000197,  5.70241397428,    30666.15495843280}, 
    {     0.00000000160,  4.46270605493,    18947.70451835760}, 
    {     0.00000000156,  1.80543158863,    10787.63034454580}, 
    {     0.00000000192,  5.34144333446,    10007.09999777380}, 
    {     0.00000000190,  2.99208130371,     2218.75710418680}, 
    {     0.00000000209,  4.83588196833,    20277.00789528740}, 
    {     0.00000000145,  5.92559556225,    37724.75341974820}, 
    {     0.00000000198,  0.77846666692,    62883.35513951360}, 
    {     0.00000000144,  5.00261963924,     9264.10993724120}, 
    {     0.00000000171,  2.05212624568,     7255.56965173440}, 
    {     0.00000000188,  4.08173534559,    48739.85989708300}, 
    {     0.00000000146,  3.93370103580,     6309.37416979120}, 
    {     0.00000000145,  5.06295543750,    39264.06928956020}, 
    {     0.00000000139,  2.81266025896,       20.77539549240}, 
    {     0.00000000177,  5.16224804657,     9835.91193829520}, 
    {     0.00000000120,  0.21443767468,    31022.75317085620}, 
    {     0.00000000118,  1.36333433479,    40077.61957352000}, 
    {     0.00000000128,  2.92458887987,        7.11354700080}, 
    {     0.00000000109,  5.19050570589,     9830.38901398780}, 
    {     0.00000000150,  5.73646272556,      632.78373931320}, 
    {     0.00000000106,  0.62215793949,    11272.66747640020}, 
    {     0.00000000114,  2.63301326520,    17468.85519794540}, 
    {     0.00000000123,  6.22518843711,    53285.18483524180}, 
    {     0.00000000107,  1.17258978900,    43071.89928903080}, 
    {     0.00000000116,  2.20934021321,     8521.11987670860}, 
    {     0.00000000104,  1.01322416672,    41654.96311596780}, 
    {     0.00000000109,  2.01412666864,    20419.45754542119}, 
    {     0.00000000102,  4.23406964348,    10251.31321884680}, 
    {     0.00000000116,  1.27731728606,    10199.05845220940}, 
    {     0.00000000112,  2.24436894064,    18204.71445782499}, 
    {     0.00000000118,  0.23754207200,    10497.14486507620}, 
    {     0.00000000123,  0.88054816668,    34596.36465465240}, 
    {     0.00000000102,  4.39438646620,    18300.69368504279}, 
    {     0.00000000131,  6.01711652115,     9367.20271145980}, 
    {     0.00000000100,  5.00532389609,    10175.25787357520}, 
    {     0.00000000107,  0.41270205378,    40853.14218484400}, 
    {     0.00000000132,  5.45008342761,    11506.76976979360}, 
    {     0.00000000098,  1.07722950958,    13553.89797291080}, 
    {     0.00000000094,  2.91719764259,    44007.82926973960}, 
    {     0.00000000097,  1.04004223634,    68050.42387851159}, 
    {     0.00000000127,  2.20215372683,    66813.56483573320}, 
    {     0.00000000111,  1.57823839032,    29043.67019634839}, 
    {     0.00000000118,  2.33268176890,    18314.92077904440}, 
    {     0.00000000090,  2.42353056125,    32858.61374281979}, 
    {     0.00000000109,  3.82796787296,    19470.28193645139}, 
    {     0.00000000111,  4.47666957576,    29057.89729034999}, 
    {     0.00000000101,  3.41528493660,    19264.09638801420}, 
    {     0.00000000092,  3.66289799512,    22645.32819660879}, 
    {     0.00000000094,  6.07530805791,    10846.06928552420}, 
    {     0.00000000114,  4.02718653431,     7576.56007357400}, 
    {     0.00000000087,  6.01880724674,    17085.95866572220}, 
    {     0.00000000109,  5.46886607309,    52670.06959330260}, 
    {     0.00000000108,  5.44460610707,    19050.79729257620}, 
    {     0.00000000106,  0.55746535085,    34363.36559755600}, 
    {     0.00000000076,  6.15177368654,    27682.14074415640}, 
    {     0.00000000107,  4.80525404063,     8144.27871130440}, 
    {     0.00000000071,  2.26530392323,     8631.32619792800}, 
    {     0.00000000097,  5.13538849186,    22779.43724619380}, 
    {     0.00000000095,  5.25816059997,      522.57741809380}, 
    {     0.00000000069,  1.57003462107,    20956.26205751660}, 
    {     0.00000000068,  6.22807515295,    10419.47109464820}, 
    {     0.00000000091,  4.28652743953,    10110.19277199240}, 
    {     0.00000000089,  1.79703857734,    45585.17281218740}, 
    {     0.00000000067,  2.25900071584,     9360.08916445900}, 
    {     0.00000000085,  0.67062144972,    56600.27928952220}, 
    {     0.00000000080,  1.58278081077,    19992.85935454519}, 
    {     0.00000000085,  4.52011215904,    29786.66025688100}, 
    {     0.00000000068,  4.48235266554,    10632.77019008620}, 
    {     0.00000000063,  0.36743160711,    17248.42530185440}, 
    {     0.00000000064,  4.33495700921,    47938.03896595920}, 
    {     0.00000000071,  3.03858484137,    11787.10597030980}, 
    {     0.00000000067,  3.64786869696,    11764.33076885900}, 
    {     0.00000000087,  4.81823064076,     2107.03450754240}, 
    {     0.00000000070,  2.35648061034,    11794.15220700780}, 
    {     0.00000000080,  2.33248094128,    38526.57435087200}, 
    {     0.00000000070,  3.70454061100,     8734.41897214660}, 
    {     0.00000000077,  4.49569185467,    20007.08644854680}, 
    {     0.00000000072,  1.19410424468,    10217.21769947410}, 
    {     0.00000000068,  2.01841060183,    14128.24277124560}, 
    {     0.00000000064,  5.39293951654,     7880.08915333899}, 
    {     0.00000000066,  3.20467070973,    14765.23904326980}, 
    {     0.00000000080,  3.41620457770,    48417.97290558199}, 
    {     0.00000000066,  5.85414440204,     9793.80090233580}, 
    {     0.00000000082,  3.62592908644,    70743.77453195279}, 
    {     0.00000000058,  4.95163459260,    30110.16567353840}, 
    {     0.00000000079,  6.24161471033,     6037.24420376200}, 
    {     0.00000000069,  5.50183658445,    19793.78735310880}, 
    {     0.00000000076,  3.40005875207,      245.83164622940}, 
    {     0.00000000056,  1.24148350566,    10207.76262190360}, 
    {     0.00000000070,  2.45123308846,    10218.80847051840}, 
    {     0.00000000064,  5.53983104501,    10735.86296430480}, 
    {     0.00000000054,  3.62259713240,    27461.71084806540}, 
    {     0.00000000073,  1.75920522146,     1589.07289528380}, 
    {     0.00000000075,  3.38244819846,     4214.06901508480}, 
    {     0.00000000054,  0.64971567468,     9929.42622734580}, 
    {     0.00000000054,  3.40959637230,    18418.01355326299}, 
    {     0.00000000056,  3.65815006538,    14169.79356223040}, 
    {     0.00000000056,  0.71243223808,     9896.89367655440}, 
    {     0.00000000052,  1.33348131832,    20400.27277262220}, 
    {     0.00000000067,  3.12806595410,     5481.25491886760}, 
    {     0.00000000058,  0.54482893546,    28313.28880466100}, 
    {     0.00000000054,  0.15603935681,    19580.48825767080}, 
    {     0.00000000051,  3.37515473510,     9256.99639024040}, 
    {     0.00000000063,  3.38894120617,    49515.38250840700}, 
    {     0.00000000069,  4.90917651401,    63498.47038145279}, 
    {     0.00000000057,  5.07437742030,    18521.10632748160}, 
    {     0.00000000050,  1.59156823654,    18631.31264870099}, 
    {     0.00000000054,  6.25816208666,    37674.99639427640}, 
    {     0.00000000057,  5.48066346720,    24383.07910844140}, 
    {     0.00000000045,  1.10466490660,    10408.25693067160}, 
    {     0.00000000051,  3.61196470338,      426.59819087600}, 
    {     0.00000000058,  4.40545155507,     8617.09910392640}, 
    {     0.00000000057,  2.09567711267,    60530.48898574180}, 
    {     0.00000000060,  5.94659889997,    13897.66359620120}, 
    {     0.00000000051,  5.47238517720,    57837.13833230060}, 
    {     0.00000000051,  2.32438478428,    19779.56025910719}, 
    {     0.00000000052,  3.23766328818,    18940.59097135679}, 
    {     0.00000000048,  1.12206254877,     9779.57380833420}, 
    {     0.00000000058,  3.08646083897,    12074.48840752400}, 
    {     0.00000000046,  4.07536026888,     7863.94251078820}, 
    {     0.00000000045,  4.75746520642,     7856.89627409019}, 
    {     0.00000000050,  3.70569982975,    42456.78404709160}, 
    {     0.00000000044,  1.29248911155,    69166.43098950500}, 
    {     0.00000000046,  0.41229872114,     7564.83072073800}, 
    {     0.00000000039,  1.25238820180,    10323.49186743040}, 
    {     0.00000000044,  6.17937388307,    13341.67431130680}, 
    {     0.00000000053,  4.71379706388,    53445.59220462660}, 
    {     0.00000000041,  3.48003037828,    37895.42629036740}, 
    {     0.00000000040,  1.23305546260,    10228.53801739600}, 
    {     0.00000000053,  5.04979874661,    74673.98422817240}, 
    {     0.00000000039,  2.15376025201,     8947.71806758460}, 
    {     0.00000000039,  0.70253732683,    18093.37469954999}, 
    {     0.00000000052,  1.18164377451,    10211.80107350270}, 
    {     0.00000000047,  1.78672260794,    10401.21069397360}, 
    {     0.00000000040,  3.66961416802,    10198.03307502600}, 
    {     0.00000000051,  2.71698589018,    94325.03270927040}, 
    {     0.00000000038,  1.36497153223,    21202.09370374600}, 
    {     0.00000000049,  1.21335959420,     9721.62225375220}, 
    {     0.00000000042,  6.05968230173,   105460.99111839019}, 
    {     0.00000000040,  1.97642764685,    32243.49850088060}, 
    {     0.00000000036,  4.96702216961,    36301.18868778519}, 
    {     0.00000000037,  5.29642935562,       38.02767263580}, 
    {     0.00000000035,  2.34112124655,    58946.51688439399}, 
    {     0.00000000034,  5.77533382849,    51868.24866217880}, 
    {     0.00000000034,  1.82989750626,    17675.02349273040}, 
    {     0.00000000034,  0.76493664110,    55798.45835839840}, 
    {     0.00000000035,  1.09353675147,    69159.80243060499}, 
    {     0.00000000032,  3.32960781870,    71519.29714327680}, 
    {     0.00000000031,  0.68615213145,    10202.23984594710}, 
    {     0.00000000030,  4.42039942947,    10459.11719244040}, 
    {     0.00000000029,  1.30367701539,    20103.06567576459}, 
    {     0.00000000031,  4.51793347997,     2957.71589447660}, 
    {     0.00000000036,  5.67202610552,    24341.52831745660}, 
    {     0.00000000029,  3.36506645849,    10288.06714477830}, 
    {     0.00000000029,  1.40019042576,     9988.94075050910}, 
    {     0.00000000039,  5.15481059724,    20350.30502114640}, 
    {     0.00000000034,  5.22945947227,     8673.96967639900}, 
    {     0.00000000039,  4.50883171158,    16004.69810374360}, 
    {     0.00000000028,  2.32945945641,    11392.48008525060}, 
    {     0.00000000032,  5.46972716255,    64607.84893354619}, 
    {     0.00000000029,  0.59723613788,    26709.64694241340}, 
    {     0.00000000029,  6.16476830510,    65236.22129328540}, 
    {     0.00000000026,  5.36096904409,     1478.86657406440}, 
    {     0.00000000036,  1.29565084626,    90394.82301305079}, 
};


static const struct ln_vsop ALIGN32 venus_radius_r1[RADIUS_R1] = {
    {     0.00034551041,  0.89198706276,    10213.28554621100}, 
    {     0.00000234203,  1.77224942363,    20426.57109242200}, 
    {     0.00000233998,  3.14159265359,        0.00000000000}, 
    {     0.00000023867,  1.11270233944,     9437.76293488700}, 
    {     0.00000010571,  4.59152848465,     1577.34354244780}, 
    {     0.00000009124,  4.53540895241,    10404.73381232260}, 
    {     0.00000006600,  5.97725159435,     5507.55323866740}, 
    {     0.00000004665,  3.87732289579,     9153.90361602180}, 
    {     0.00000003840,  5.66215379445,    13367.97263110660}, 
    {     0.00000002662,  2.82393816664,    10206.17199921020}, 
    {     0.00000002194,  2.05324060020,      775.52261132400}, 
    {     0.00000002093,  2.54944827541,    18837.49819713819}, 
    {     0.00000001781,  2.64889239766,    30639.85663863300}, 
    {     0.00000001844,  1.87568236008,    11015.10647733480}, 
    {     0.00000001303,  0.20613045604,    11322.66409830440}, 
    {     0.00000001168,  0.79442892853,    17298.18232732620}, 
    {     0.00000001002,  6.16544615317,    10239.58386601080}, 
    {     0.00000000915,  4.59854496963,     1109.37855209340}, 
    {     0.00000000882,  0.66800567417,    18073.70493865020}, 
    {     0.00000000846,  5.58765716729,    12566.15169998280}, 
    {     0.00000001080,  4.94678408464,     6283.07584999140}, 
    {     0.00000000888,  2.47876529003,     3154.68708489560}, 
    {     0.00000000904,  0.81413053841,    10596.18207843420}, 
    {     0.00000000820,  0.89897968675,     5661.33204915220}, 
    {     0.00000000845,  5.48503535862,      529.69096509460}, 
    {     0.00000000824,  3.74890493484,     7084.89678111520}, 
    {     0.00000000655,  5.06545516257,    22003.91463486980}, 
    {     0.00000000847,  0.44120492340,     8635.94200376320}, 
    {     0.00000000638,  4.10125783744,      191.44826611160}, 
    {     0.00000000618,  3.15185635462,    10186.98722641120}, 
    {     0.00000000530,  5.86995223147,     2352.86615377180}, 
    {     0.00000000576,  2.25211985760,    21228.39202354580}, 
    {     0.00000000517,  5.33412478409,    14143.49524243060}, 
    {     0.00000000664,  2.86641799035,     8624.21265092720}, 
    {     0.00000000552,  2.16807468414,    18307.80723204360}, 
    {     0.00000000518,  4.34059510806,     9786.68735533500}, 
    {     0.00000000501,  5.56479595613,    10742.97651130560}, 
    {     0.00000000426,  1.02161443120,     7064.12138562280}, 
    {     0.00000000416,  1.26242491254,     9676.48103411560}, 
    {     0.00000000393,  0.78006660798,     9690.70812811720}, 
    {     0.00000000336,  3.17756997609,    10988.80815753500}, 
    {     0.00000000375,  0.66142254036,    19360.07561523199}, 
    {     0.00000000361,  0.19448210327,    19374.30270923360}, 
    {     0.00000000314,  1.09753065455,     4551.95349705880}, 
    {     0.00000000329,  0.58926282996,    16496.36139620240}, 
    {     0.00000000339,  5.76768761424,    10021.83728009940}, 
    {     0.00000000289,  3.67224020150,    25158.60171976540}, 
    {     0.00000000223,  4.33581625553,    19786.67380610799}, 
    {     0.00000000266,  3.57410277687,      801.82093112380}, 
    {     0.00000000273,  5.72814567551,    11790.62908865880}, 
    {     0.00000000275,  5.65863344812,    19896.88012732740}, 
    {     0.00000000212,  4.27040662743,     4705.73230754360}, 
    {     0.00000000230,  6.13409928801,     1059.38193018920}, 
    {     0.00000000203,  4.88193121349,     7860.41939243920}, 
    {     0.00000000241,  1.12684737375,       26.29831979980}, 
    {     0.00000000215,  3.13976767933,    14945.31617355440}, 
    {     0.00000000206,  0.32759945446,      382.89653222320}, 
    {     0.00000000216,  2.54929765288,    19651.04848109800}, 
    {     0.00000000160,  1.13642774778,    13936.79450513400}, 
    {     0.00000000153,  0.81251420624,     6770.71060124560}, 
    {     0.00000000150,  5.11133177708,    28521.09277825460}, 
    {     0.00000000150,  1.35720240642,    10220.39909321180}, 
    {     0.00000000149,  5.03755185727,    29088.81141598500}, 
    {     0.00000000127,  4.49298610266,     3532.06069281140}, 
    {     0.00000000121,  6.27033352540,    29580.47470844380}, 
    {     0.00000000146,  6.16157870651,     8662.24032356300}, 
    {     0.00000000117,  3.56101830556,    24356.78078864160}, 
    {     0.00000000114,  0.00568663739,    25934.12433108940}, 
    {     0.00000000127,  0.66789441339,     3723.50895892300}, 
    {     0.00000000148,  0.36877913428,     9146.79006902100}, 
    {     0.00000000106,  4.26220077003,     9573.38825989700}, 
    {     0.00000000136,  5.09588289539,    19367.18916223280}, 
    {     0.00000000099,  1.51389536350,    17277.40693183380}, 
    {     0.00000000117,  0.57647358272,     9999.98645077300}, 
    {     0.00000000097,  3.71741720558,      213.29909543800}, 
    {     0.00000000092,  0.22936081655,    18830.38465013739}, 
    {     0.00000000112,  4.04771058126,     9103.90699411760}, 
    {     0.00000000083,  5.86195695833,    10234.06094170340}, 
    {     0.00000000079,  0.64440357793,    18844.61174413899}, 
    {     0.00000000084,  0.62796944327,     9683.59458111640}, 
    {     0.00000000107,  1.77067111589,    17778.11626694899}, 
    {     0.00000000081,  6.19048048769,    20618.01935853360}, 
    {     0.00000000087,  0.15771136594,    33019.02111220460}, 
    {     0.00000000085,  4.77733091614,     3930.20969621960}, 
    {     0.00000000086,  2.21506820700,     8094.52168583260}, 
    {     0.00000000074,  0.82723399048,     3128.38876509580}, 
    {     0.00000000063,  2.70493265779,    16983.99614745660}, 
    {     0.00000000059,  3.32160695003,    10787.63034454580}, 
    {     0.00000000061,  0.04044699966,     9161.01716302260}, 
    {     0.00000000064,  4.13127333938,     9992.87290377220}, 
    {     0.00000000063,  4.47517295578,    18875.52586977400}, 
    {     0.00000000060,  6.24567706571,    32217.20018108080}, 
    {     0.00000000076,  4.84321073833,     4732.03062734340}, 
    {     0.00000000054,  3.38449893196,    10426.58464164900}, 
    {     0.00000000054,  5.15928227000,    28286.99048486120}, 
    {     0.00000000063,  4.32344259630,    12592.45001978260}, 
    {     0.00000000059,  3.65953929456,    10007.09999777380}, 
    {     0.00000000049,  5.10267262491,    19573.37471066999}, 
    {     0.00000000047,  5.79444960738,    68050.42387851159}, 
    {     0.00000000052,  3.56658420552,     7255.56965173440}, 
    {     0.00000000050,  1.61783309819,    36949.23080842420}, 
    {     0.00000000053,  2.64129868542,    15874.61759536320}, 
    {     0.00000000042,  2.02654328254,     7058.59846131540}, 
    {     0.00000000051,  0.80486467278,    23581.25817731760}, 
    {     0.00000000040,  3.89322312104,    20419.45754542119}, 
    {     0.00000000050,  1.07461899460,     3442.57494496540}, 
    {     0.00000000038,  1.77428239418,    10103.07922499160}, 
    {     0.00000000040,  5.22873972349,    21535.94964451540}, 
    {     0.00000000038,  0.10029856174,    11272.66747640020}, 
    {     0.00000000038,  1.16414419860,        7.11354700080}, 
    {     0.00000000044,  4.54118898535,    10192.51015071860}, 
    {     0.00000000047,  0.05589432390,    12432.04265039780}, 
    {     0.00000000034,  3.45481114998,     9830.38901398780}, 
    {     0.00000000037,  4.97010610007,     3340.61242669980}, 
    {     0.00000000044,  0.70533597362,    20213.27199698400}, 
    {     0.00000000034,  2.16487642765,    64460.69868196140}, 
    {     0.00000000031,  1.57612397319,    36147.40987730040}, 
    {     0.00000000034,  1.06669952696,    29864.33402730900}, 
    {     0.00000000028,  2.56454760402,    94138.32702008578}, 
    {     0.00000000029,  0.59718407064,    59728.66805461800}, 
    {     0.00000000031,  3.27727318906,    19992.85935454519}, 
    {     0.00000000027,  5.68943276986,    17085.95866572220}, 
    {     0.00000000030,  3.03241502538,    40879.44050464380}, 
    {     0.00000000032,  2.64260788260,    41962.52073693740}, 
    {     0.00000000035,  0.26995532429,     1589.07289528380}, 
    {     0.00000000028,  4.93395364921,    29050.78374334920}, 
    {     0.00000000028,  3.58869616799,    40853.14218484400}, 
    {     0.00000000029,  2.79705093386,    20007.08644854680}, 
    {     0.00000000033,  0.93875241519,    15720.83878487840}, 
    {     0.00000000024,  2.74970637101,    18947.70451835760}, 
    {     0.00000000024,  4.38966861409,    46386.99374331120}, 
    {     0.00000000024,  0.73362177921,     9411.46461508720}, 
    {     0.00000000028,  4.19559784013,    37674.99639427640}, 
    {     0.00000000024,  1.18299920855,      426.59819087600}, 
    {     0.00000000025,  4.14802726798,    14919.01785375460}, 
    {     0.00000000030,  1.24986033487,    27461.71084806540}, 
    {     0.00000000022,  2.95789824506,    40077.61957352000}, 
    {     0.00000000027,  3.94986823486,    17468.85519794540}, 
    {     0.00000000025,  4.75875623888,    27991.40181316000}, 
    {     0.00000000022,  4.11184201321,    19779.56025910719}, 
    {     0.00000000027,  3.72446446080,    19793.78735310880}, 
    {     0.00000000027,  6.12859780299,    38500.27603107220}, 
    {     0.00000000022,  4.99040169444,    31022.75317085620}, 
    {     0.00000000021,  0.63975130342,    13745.34623902240}, 
    {     0.00000000023,  1.33505132122,    65236.22129328540}, 
    {     0.00000000020,  0.91294578294,    22779.43724619380}, 
    {     0.00000000021,  4.46897353468,    53285.18483524180}, 
    {     0.00000000020,  4.15140915983,     2218.75710418680}, 
    {     0.00000000019,  1.43653410349,    27197.28169366760}, 
    {     0.00000000027,  1.22597317413,    42430.48572729180}, 
    {     0.00000000024,  4.66952456612,     1551.04522264800}, 
    {     0.00000000021,  4.16155332491,    44007.82926973960}, 
    {     0.00000000018,  5.45345109729,     6681.22485339960}, 
    {     0.00000000023,  2.19259117717,    27511.46787353720}, 
    {     0.00000000018,  4.24146756742,    31441.67756975680}, 
    {     0.00000000019,  3.65054338893,    49515.38250840700}, 
    {     0.00000000018,  2.29853355765,    19264.09638801420}, 
    {     0.00000000020,  3.68116637773,    14128.24277124560}, 
    {     0.00000000021,  4.30316190532,    44809.65020086340}, 
    {     0.00000000017,  3.02735393984,    28528.20632525540}, 
    {     0.00000000019,  5.92656850674,    22805.73556599360}, 
    {     0.00000000020,  5.80568617231,    10218.80847051840}, 
    {     0.00000000020,  0.75829381378,    18314.92077904440}, 
    {     0.00000000017,  5.63315744126,    16522.65971600220}, 
    {     0.00000000015,  5.27016880041,    53445.59220462660}, 
    {     0.00000000015,  6.06411037636,    18300.69368504279}, 
    {     0.00000000020,  5.38470522864,    10207.76262190360}, 
    {     0.00000000015,  5.81110284451,    43071.89928903080}, 
    {     0.00000000015,  5.98462571207,    10228.53801739600}, 
    {     0.00000000014,  1.44966591677,      536.80451209540}, 
    {     0.00000000015,  5.20122735812,    10198.03307502600}, 
    {     0.00000000014,  3.48144272414,    29786.66025688100}, 
    {     0.00000000018,  1.99790369768,    30831.30490474460}, 
    {     0.00000000017,  2.36905439219,    33794.54372352860}, 
};


static const struct ln_vsop ALIGN32 venus_radius_r2[RADIUS_R2] = {
    {     0.00001406587,  5.06366395112,    10213.28554621100}, 
    {     0.00000015529,  5.47321056992,    20426.57109242200}, 
    {     0.00000013059,  0.00000000000,        0.00000000000}, 
    {     0.00000001096,  2.78919545899,     9437.76293488700}, 
    {     0.00000000487,  6.27655636902,     1577.34354244780}, 
    {     0.00000000361,  6.11959389556,    10404.73381232260}, 
    {     0.00000000310,  1.39073645837,     5507.55323866740}, 
    {     0.00000000385,  1.95564555688,    11015.10647733480}, 
    {     0.00000000371,  2.33232050485,      775.52261132400}, 
    {     0.00000000207,  5.63406721595,    10239.58386601080}, 
    {     0.00000000167,  1.11639732890,    13367.97263110660}, 
    {     0.00000000175,  6.16674649733,    30639.85663863300}, 
    {     0.00000000168,  3.64042170990,     7084.89678111520}, 
    {     0.00000000120,  5.85861348966,     9153.90361602180}, 
    {     0.00000000161,  2.21564443685,     3154.68708489560}, 
    {     0.00000000118,  2.62362056521,     8635.94200376320}, 
    {     0.00000000112,  2.36235956804,    10596.18207843420}, 
    {     0.00000000093,  0.74251193058,    12566.15169998280}, 
    {     0.00000000067,  3.76089669118,    18837.49819713819}, 
    {     0.00000000065,  2.47990173302,    11790.62908865880}, 
    {     0.00000000048,  4.26821029269,     2352.86615377180}, 
    {     0.00000000049,  5.51819305769,      191.44826611160}, 
    {     0.00000000048,  2.51039041484,    17298.18232732620}, 
    {     0.00000000046,  3.40293459332,    14143.49524243060}, 
    {     0.00000000041,  1.84035920887,    11322.66409830440}, 
    {     0.00000000035,  6.20303344591,     1109.37855209340}, 
    {     0.00000000039,  4.76600160325,    18073.70493865020}, 
    {     0.00000000034,  3.14302158146,     4705.73230754360}, 
    {     0.00000000034,  0.47584552790,       26.29831979980}, 
    {     0.00000000046,  3.30090415967,     6283.07584999140}, 
    {     0.00000000034,  3.92299418240,    10021.83728009940}, 
    {     0.00000000034,  3.23795065286,    22003.91463486980}, 
    {     0.00000000042,  3.39360926939,    14945.31617355440}, 
    {     0.00000000044,  4.39723735555,     7860.41939243920}, 
    {     0.00000000037,  3.63183908790,      801.82093112380}, 
    {     0.00000000034,  2.16381407025,    16496.36139620240}, 
    {     0.00000000034,  1.82177984232,      382.89653222320}, 
    {     0.00000000025,  3.32908650295,    18307.80723204360}, 
    {     0.00000000026,  3.64313769818,    29088.81141598500}, 
    {     0.00000000029,  3.82967178810,    10186.98722641120}, 
    {     0.00000000022,  3.17741520378,    28521.09277825460}, 
    {     0.00000000025,  5.71510731610,    21202.09370374600}, 
    {     0.00000000021,  5.30608699641,    19896.88012732740}, 
    {     0.00000000019,  4.65302013202,    19651.04848109800}, 
    {     0.00000000022,  2.42894070342,      529.69096509460}, 
    {     0.00000000021,  3.82207126131,    21228.39202354580}, 
    {     0.00000000016,  2.96175040524,    19367.18916223280}, 
    {     0.00000000016,  3.36250300274,    28286.99048486120}, 
    {     0.00000000015,  5.05571633205,    33019.02111220460}, 
    {     0.00000000014,  2.83786355803,    19786.67380610799}, 
    {     0.00000000014,  1.79922718553,     9830.38901398780}, 
    {     0.00000000014,  3.49994470578,    10988.80815753500}, 
    {     0.00000000013,  3.06303088617,    10742.97651130560}, 
    {     0.00000000013,  5.43981998532,    25158.60171976540}, 
    {     0.00000000015,  4.83166312889,    18875.52586977400}, 
    {     0.00000000012,  2.54141086214,     7064.12138562280}, 
    {     0.00000000010,  1.87933121728,    24356.78078864160}, 
    {     0.00000000011,  4.26593805644,    15720.83878487840}, 
    {     0.00000000010,  2.17901309900,     3723.50895892300}, 
    {     0.00000000010,  2.50084772448,     9103.90699411760}, 
    {     0.00000000007,  3.48063149205,     1059.38193018920}, 
    {     0.00000000007,  2.38734958588,    20618.01935853360}, 
};


static const struct ln_vsop ALIGN32 venus_radius_r3[RADIUS_R3] = {
    {     0.00000049582,  3.22264415899,    10213.28554621100}, 
    {     0.00000000831,  3.21255590531,    20426.57109242200}, 
    {     0.00000000112,  3.14159265359,        0.00000000000}, 
    {     0.00000000013,  3.77454760284,    30639.85663863300}, 
    {     0.00000000008,  4.29674209391,    10239.58386601080}, 
    {     0.00000000006,  0.39133521368,    10186.98722641120}, 
    {     0.00000000004,  4.77456526708,     9437.76293488700}, 
    {     0.00000000003,  0.14604782945,    10988.80815753500}, 
};


static const struct ln_vsop ALIGN32 venus_radius_r4[RADIUS_R4] = {
    {     0.00000000573,  0.92253525592,    10213.28554621100}, 
    {     0.00000000039,  0.95696787303,    20426.57109242200}, 
    {     0.00000000006,  3.14159265359,        0.00000000000}, 
};


static const struct ln_vsop ALIGN32 venus_radius_r5[RADIUS_R5] = {
    {     0.00000000045,  0.30037014808,    10213.28554621100}, 
    {     0.00000000002,  5.33215705373,    20426.57109242200}, 
};

/*! \fn void ln_get_venus_equ_coords(double JD, struct ln_equ_posn *position);
* \param JD Julian Day
* \param position Pointer to store position
*
* Calculates Venus's equatorial position for given julian day.
* This function includes calculations for planetary aberration and refers
* to the FK5 reference frame.
*
* To get the complete equatorial coordinates, corrections for nutation
* have to be applied.
*
* The position returned is accurate to within 0.1 arcsecs..
*/ 
void ln_get_venus_equ_coords(double JD, struct ln_equ_posn *position)
{
	struct ln_helio_posn h_sol, h_venus;
	struct ln_rect_posn g_sol, g_venus;
	double a,b,c;
	double ra, dec, delta, diff, last, t = 0;
	
	/* need typdef for solar heliocentric coords */
	ln_get_solar_geom_coords(JD, &h_sol);
	ln_get_rect_from_helio(&h_sol,  &g_sol);
	
	do {
		last = t;
		ln_get_venus_helio_coords(JD - t, &h_venus);
		ln_get_rect_from_helio(&h_venus, &g_venus);

		/* equ 33.10 pg 229 */
		a = g_sol.X + g_venus.X;
		b = g_sol.Y + g_venus.Y;
		c = g_sol.Z + g_venus.Z;
	
		delta = a * a + b * b + c * c;
		delta = sqrt(delta);
		t = delta * 0.0057755183;
		diff = t - last;
	} while (diff > 0.0001 || diff < -0.0001);
		
	ra = atan2(b, a);
	dec = c / delta;
	dec = asin(dec);

	/* back to hours, degrees */
	position->ra = ln_range_degrees(ln_rad_to_deg(ra));
	position->dec = ln_rad_to_deg(dec);
}
	
/*! \fn void ln_get_venus_helio_coords(double JD, struct ln_helio_posn *position)
* \param JD Julian Day
* \param position Pointer to store new heliocentric position
*
* Calculate Venus heliocentric (referred to the centre of the Sun) coordinates
* in the FK5 reference frame for the given julian day.
* Longitude and Latitude are in degrees, whilst radius vector is in AU.
*/ 
/* Chapter 31 Pg 206-207 Equ 31.1 31.2 , 31.3 using VSOP 87 
*/
void ln_get_venus_helio_coords(double JD, struct ln_helio_posn *position)
{
	double t, t2, t3, t4, t5;
	double L0, L1, L2, L3, L4, L5;
	double B0, B1, B2, B3, B4, B5;
	double R0, R1, R2, R3, R4, R5;
           	
	/* check cache first */
	if(JD == cJD) {
		/* cache hit */
		position->L = cL;
		position->B = cB;
		position->R = cR;
		return;
	}
	
	/* get julian ephemeris day */
	t =(JD - 2451545.0) / 365250.0;
	t2 = t * t;
	t3 = t2 * t;
	t4 = t3 * t;
	t5 = t4 * t;
	
	/* calc L series */
	L0 = ln_calc_series(venus_longitude_l0, LONG_L0, t);
	L1 = ln_calc_series(venus_longitude_l1, LONG_L1, t);
	L2 = ln_calc_series(venus_longitude_l2, LONG_L2, t);
	L3 = ln_calc_series(venus_longitude_l3, LONG_L3, t);
	L4 = ln_calc_series(venus_longitude_l4, LONG_L4, t);
	L5 = ln_calc_series(venus_longitude_l5, LONG_L5, t);
	position->L = (L0 + L1 * t + L2 * t2 + L3 * t3 + L4 * t4 + L5 * t5);

	/* calc B series */
	B0 = ln_calc_series(venus_latitude_b0, LAT_B0, t);
	B1 = ln_calc_series(venus_latitude_b1, LAT_B1, t);
	B2 = ln_calc_series(venus_latitude_b2, LAT_B2, t);
	B3 = ln_calc_series(venus_latitude_b3, LAT_B3, t);
	B4 = ln_calc_series(venus_latitude_b4, LAT_B4, t);
	B5 = ln_calc_series(venus_latitude_b5, LAT_B5, t);
	position->B = (B0 + B1 * t + B2 * t2 + B3 * t3 + B4 * t4 + B5 * t5);


	/* calc R series */
	R0 = ln_calc_series(venus_radius_r0, RADIUS_R0, t);
	R1 = ln_calc_series(venus_radius_r1, RADIUS_R1, t);
	R2 = ln_calc_series(venus_radius_r2, RADIUS_R2, t);
	R3 = ln_calc_series(venus_radius_r3, RADIUS_R3, t);
	R4 = ln_calc_series(venus_radius_r4, RADIUS_R4, t);
	R5 = ln_calc_series(venus_radius_r5, RADIUS_R5, t);
	position->R = (R0 + R1 * t + R2 * t2 + R3 * t3 + R4 * t4 + R5 * t5);
	
	/* change to degrees in correct quadrant */
	position->L = ln_rad_to_deg(position->L);
	position->B = ln_rad_to_deg(position->B);
	position->L = ln_range_degrees(position->L);
	
	/* change to fk5 reference frame */
	ln_vsop87_to_fk5(position, JD);
	
	/* save cache */
	cJD = JD;
	cL = position->L;
	cB = position->B;
	cR = position->R;
}

/*! \fn double ln_get_venus_earth_dist(double JD);
* \param JD Julian day
* \brief Calculate the distance between Venus and the Earth in AU
* \return Distance in AU
*
* Calculates the distance in AU between the Earth and Venus for the
* given julian day.
*/
double ln_get_venus_earth_dist(double JD)
{
	struct ln_helio_posn h_venus, h_earth;
	struct ln_rect_posn g_venus, g_earth;
	double x, y, z;
	
	/* get heliocentric positions */
	ln_get_venus_helio_coords(JD, &h_venus);
	ln_get_earth_helio_coords(JD, &h_earth);
	
	/* get geocentric coords */
	ln_get_rect_from_helio(&h_venus, &g_venus);
	ln_get_rect_from_helio(&h_earth, &g_earth);
	
	/* use pythag */
	x = g_venus.X - g_earth.X;
	y = g_venus.Y - g_earth.Y;
	z = g_venus.Z - g_earth.Z;

	x = x * x;
	y = y * y;
	z = z * z;
	return sqrt(x + y + z);
}

/*! \fn double ln_get_venus_solar_dist(double JD);
* \param JD Julian day
* \brief Calculate the distance between Venus and the Sun in AU
* \return Distance in AU
*
* Calculates the distance in AU between the Sun and Venus for
* the given julian day.
*/ 
double ln_get_venus_solar_dist(double JD)
{
	struct ln_helio_posn h_venus;

	/* get heliocentric position */
	ln_get_venus_helio_coords(JD, &h_venus);

	return h_venus.R;
}
	
/*! \fn double ln_get_venus_magnitude(double JD);
* \param JD Julian day
* \brief Calculate the visible magnitude of Venus
* \return Visible magnitude of venus
*
* Calculate the visible magnitude of Venus for the 
* given julian day.
*/ 
double ln_get_venus_magnitude(double JD)
{
	double delta, r, i, i2, i3;
	
	/* get distances */
	r = ln_get_venus_solar_dist(JD);
	delta = ln_get_venus_earth_dist(JD);
	
	/* get phase */
	i = ln_get_venus_phase(JD);
	i2 = i * i;
	i3 = i2 * i;
	
	return -4.40 + 5.0 * log10(r * delta) + 0.0009 * i +
		0.000239 * i2 - 0.00000065 * i3;
}

/*! \fn double ln_get_venus_disk(double JD);
* \param JD Julian day
* \brief Calculate the illuminated fraction of Venus disk
* \return Illuminated fraction of venus disk
*
* Calculate the illuminated fraction of Venus's disk for the given Julian
* day
*/ 
/* Chapter 41 */
double ln_get_venus_disk(double JD)
{
	double r, delta, R;
	
	/* get distances */
	R = ln_get_earth_solar_dist(JD);
	r = ln_get_venus_solar_dist(JD);
	delta = ln_get_venus_earth_dist(JD);
	
	/* calc fraction angle */
	return (((r + delta) * (r + delta)) - R * R) / (4.0 * r * delta);
}

/*! \fn double ln_get_venus_phase(double JD);
* \param JD Julian day
* \brief Calculate the phase angle of Venus (Sun - Venus - Earth)
* \return Phase angle of Venus (degrees)
*
* Calculates the phase angle of Venus, that is, the angle Sun -
* Venus - Earth for the given Julian day.
*/ 
/* Chapter 41 */
double ln_get_venus_phase(double JD)
{
	double i, r, delta, R;
	
	/* get distances */
	R = ln_get_earth_solar_dist(JD);
	r = ln_get_venus_solar_dist(JD);
	delta = ln_get_venus_earth_dist(JD);

	/* calc phase */
	i = (r * r + delta * delta - R * R) / (2.0 * r * delta);
	i = acos(i);

	return ln_rad_to_deg(i);
}


/*! \fn double ln_get_venus_rst(double JD, struct ln_lnlat_posn *observer, struct ln_rst_time *rst);
* \param JD Julian day
* \param observer Observers position
* \param rst Pointer to store Rise, Set and Transit time in JD
* \return 0 for success, else 1 for circumpolar.
*
* Calculate the time the rise, set and transit (crosses the local meridian at upper culmination)
* time of Venus for the given Julian day.
*
* Note: this functions returns 1 if Venus is circumpolar, that is it remains the whole
* day either above or below the horizon.
*/
int ln_get_venus_rst(double JD, struct ln_lnlat_posn *observer,
	struct ln_rst_time *rst)
{
	return ln_get_body_rst_horizon(JD, observer, ln_get_venus_equ_coords,
		LN_STAR_STANDART_HORIZON, rst);
}


/*! \fn double ln_get_venus_sdiam(double JD)
* \param JD Julian day
* \return Semidiameter in arc seconds
*
* Calculate the semidiameter of Venus in arc seconds for the 
* given julian day.
*/
double ln_get_venus_sdiam(double JD)
{
	double So = 8.41; /* at 1 AU, using atmosphere value, not crust (8.34) */
	double dist;
	
	dist = ln_get_venus_earth_dist(JD);
	return So / dist;
}
	
/*! \fn void ln_get_venus_rect_helio(double JD, struct ln_rect_posn *position)
* \param JD Julian day.
* \param position pointer to return position
*
* Calculate Venus rectangular heliocentric coordinates for the
* given Julian day. Coordinates are in AU.
*/
void ln_get_venus_rect_helio(double JD, struct ln_rect_posn *position)
{
	struct ln_helio_posn venus;
	
	ln_get_venus_helio_coords(JD, &venus);
	ln_get_rect_from_helio(&venus, position);
}
