/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
// A small Set of common functionality for all IFrIT viewization objects
//

#ifndef IVIEWSUBJECT_H
#define IVIEWSUBJECT_H


#include "iobject.h"
#include "idataconsumer.h"
#include "ireplicatedelement.h"


#include "iarray.h"
#include "iposition.h"


class iDataHelper;
class iDataLimits;
class iDataReader;
class iDataSubject;
class iEventObserver;
class iMarker;
class iViewSubjectObserver;
class iViewSubjectPipeline;


namespace iParameter
{
	namespace ViewSubject
	{
		namespace Flag
		{
			const unsigned int AlwaysShaded =	  1U;
			const unsigned int NoReplicating =	  2U;
			const unsigned int HasPosition =	  4U;
			const unsigned int NoColor =		  8U;
			const unsigned int NoOpacity =		 16U;
			const unsigned int NoPalette =		 32U;
			const unsigned int SameColor =		 64U;
			const unsigned int SameOpacity =	128U;
			const unsigned int SamePalette =	256U;
		};

		namespace Location
		{
			const int PickedPoint =	0;
			const int FocalPoint =	1;
			const int BoxCenter =	2;
		};

		namespace Id
		{
			const int Undefined			= -1;
			const int CrossSection		= 0;
			const int Particles			= 1;
			const int Surface			= 2;
			const int TensorField		= 3;
			const int VectorField		= 4;
			const int Volume			= 5;
		};
	};
};


//
//  Useful macro to declare all keys that have to be present in children too
//
#define IVIEWSUBJECT_DECLARE_INHERITED_KEYS \
	static const iObjectKey& KeyDataType(); \
	static const iObjectKey& KeyClipPlane(); \
	static const iObjectKey& KeyShading(); \
	static const iObjectKey& KeyAmbient(); \
	static const iObjectKey& KeyDiffuse(); \
	static const iObjectKey& KeySpecular(); \
	static const iObjectKey& KeySpecularPower(); \
	static const iObjectKey& KeyNumReplicas(); \
	static const iObjectKey& KeyDirectionIsPeriodic(); \
	static const iObjectKey& KeyVisible(); \
	static const iObjectKey& KeyDataPresent(); \
	static const iObjectKey& KeyPosition(bool opengl = false); \
	static const iObjectKey& KeyMoveTo(); \
	static const iObjectKey& KeyAttachToMarker(); \
	static const iObjectKey& KeyReady()


class iViewSubject : public iObject, public iDataConsumer, public iReplicatedElement
{
	
	friend class iControlModule;
	friend class iViewObject;

public:
	
	vtkTypeMacro(iViewSubject,iObject);

	virtual void Delete();
	static const iObjectType& Type();

	IVIEWSUBJECT_DECLARE_INHERITED_KEYS;

	inline bool GetClipPlane() const { return mClipPlaneOn; }
		
	void SetShading(bool);
	inline bool GetShading() const { return mShading; }
	
	void SetAmbient(float v);
	inline float GetAmbient() const { return mAmbient; }

	void SetDiffuse(float v);
	inline float GetDiffuse() const { return mDiffuse; }

	void SetSpecular(float v);
	inline float GetSpecular() const { return mSpecular; }

	void SetSpecularPower(float v);
	inline float GetSpecularPower() const { return mSpecularPower; }
	
	virtual void UpdateMaterialProperties() = 0;

	virtual void ShowClipPlane(bool s) = 0;
	virtual void ShowColorBars(bool s) = 0;

	virtual bool IsInitialized() const { return mIsInitialized; }
	virtual bool IsVisible() const = 0;
	virtual const iObjectType& RealType();
	bool IsReady() const;

	void Show(bool s);
	virtual void Reset() = 0;

	virtual float GetMemorySize() const;

	inline int GetSubjectId() const { return mSubjectId; }
	inline const iString& GetSubjectName() const { return mSubjectName; }

	void SetPosition(int location, bool detach = true);
	void SetPosition(const iPosition &p, bool detach = true);
	inline const iPosition& GetPosition() const { return mPosition; }

	virtual void AttachToMarker(int m);
	virtual void UpdateAfterMarkerMove();
	virtual void UpdateAfterMarkerAdd(iMarker * ){}
	virtual void UpdateAfterMarkerDelete(iMarker *m);

	virtual bool CanBeShown() const = 0;

	iDataLimits* GetLimits() const;

	//
	//  Registry for replicated components
	//
	void RegisterReplicatedElement(iReplicatedElement *r, bool isData); 
	void UnRegisterReplicatedElement(iReplicatedElement *r); 
	const iReplicatedElement* GetDataReplicated() const { return (mDataReplicated.Size() > 0) ? mDataReplicated[0] : 0; }

	//
	//  Pipeline operation
	//
	inline bool IsCreatingMainPipeline() const { return mCreatingMainPipeline; }
	virtual iViewSubjectPipeline* CreatePipeline(int id);

	void RemoveInternalData();	
	void Configure();

protected:
	
	iViewSubject(iViewModule *vm, const iDataType &type, const iString &name, unsigned int flags);
	virtual ~iViewSubject();
	virtual void ConfigureBody() = 0; // Create and attach pipelines here. This will be called automatically by the factory.
	virtual void ConfigureMainPipeline(iViewSubjectPipeline *p, int id);

	void AddMainPipeline(int numInputs);
	virtual iViewSubjectPipeline* CreateMainPipeline(int numInputs, int id); // can be overwritten to create custom pipelines
	inline iViewSubjectPipeline* Pipeline(int n = 0) const { return mPipelines[n]; } // will automatically return zero if out-of-bounds
	virtual void RemoveInternalDataForExtraComponents();	

	virtual void ViewSubjectPackStateBody(iString &s) const = 0;
	virtual void ViewSubjectUnPackStateBody(const iString &s) = 0;

	virtual void FinishInitialization(){}
	virtual void SyncWithDataBody(const iDataSyncRequest &r);
    virtual void ViewSubjectSyncWithData(const iDataSyncRequest &r) = 0;
	virtual iDataHelper* CreateDataHelper(vtkDataSet *ds) const;

	virtual void UpdateReplicasHead();
	virtual void UpdateReplicasBody();
	virtual void UpdatePosition(const iPosition &oldPos);
	
	virtual void ShowBody(bool s) = 0;

	virtual float GetExtraMemorySize() const;

	iDataSubject* GetSubject() const;

	//
	//  helpers
	//
	void DetachFromMarker();

	//
	//  Members
	//
	bool mCreatingMainPipeline;

	iViewSubjectObserver *mObjectObserver;
	iString mSubjectName;
	int mSubjectId, mAttachedMarkerIndex;
	iPosition mPosition;

	bool mIsInitialized, mIsConfigured;
	bool mClipPlaneOn;
	const bool mAlwaysShaded, mNoReplicating, mHasPosition;

	iMarker *mAttachedMarker;

	//
	//  ReplicationElement members
	//
	iSearchableArray<iReplicatedElement*> mDataReplicated, mPropReplicated;

	//
	//  Material properties
	//
	bool mShading;
	float mAmbient;
	float mDiffuse;
	float mSpecular;
	float mSpecularPower;

	iColor mDefaultColor;

private:

	//
	//  make it private so that it can only be created by AddMainPipeline calls
	//
	iArray<iViewSubjectPipeline*> mPipelines;

	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);
};


//
//  Useful macros to declare all members that have to be overwritten in children
//
#define IVIEWSUBJECT_DEFINE_TYPE(_type_,_fname_,_sname_) \
	IOBJECT_DEFINE_TYPE(_type_,_fname_,_sname_,iObjectType::_View); \
	const iObjectType& _type_::RealType(){ return _type_::Type(); }

#define IVIEWSUBJECT_DECLARE_INHERITED_MEMBERS \
	virtual void UpdateMaterialProperties(); \
	virtual void Reset(); \
	virtual void ShowClipPlane(bool s); \
	virtual void ShowColorBars(bool s); \
	virtual float GetMemorySize() const; \
	virtual bool IsVisible() const; \
	virtual bool CanBeShown() const; \
	protected: \
	virtual void ViewSubjectPackStateBody(iString &s) const; \
	virtual void ViewSubjectUnPackStateBody(const iString &s); \
	virtual void ViewSubjectSyncWithData(const iDataSyncRequest &r); \
	virtual void ShowBody(bool s); \
	public: \
	virtual const iObjectType& RealType()

#define IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COMMON(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,DataType,dt,Int,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,ClipPlane,cp,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,Ambient,ma,Float,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,Diffuse,md,Float,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,Specular,ms,Float,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,SpecularPower,mp,Float,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,Visible,vis,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,DataPresent,dp,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,Ready,r,Bool,1)

#define IVIEWSUBJECT_DEFINE_INHERITED_KEYS_SHADING(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,Shading,mw,Bool,1)

#define IVIEWSUBJECT_DEFINE_INHERITED_KEYS_REPLICATING(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,NumReplicas,nr,Int,6); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,DirectionIsPeriodic,-dp,Bool,3)

#define IVIEWSUBJECT_DEFINE_INHERITED_KEYS_POSITION(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,MoveTo,to,Int,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,_type_,AttachToMarker,am,OffsetInt,1); \
	IOBJECT_DEFINE_INHERITED_POSITION_KEY(iViewSubject,_type_,Position,x)

#endif // IVIEWSUBJECT_H

