//**************************************************************************************************
//                                           NetList.hpp                                           *
//                                          -------------                                          *
// Description : This class provides an interface to the circuit description (netlist) file. The   *
//               raw netlist data is loaded from file into memory and stored as a string array.    *
//               The individual netlist types (component definitions, models, etc.) are extracted  *
//               and stored as object attributes.                                                  *
// Started     : 2003-09-01                                                                        *
// Last Update : 2015-03-08                                                                        *
// Copyright   : (C) 2003 by MSWaters                                                              *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#ifndef NETLIST_HPP
#define NETLIST_HPP

// Application Includes

#include "Version.hpp"
#include "TypeDefs.hpp"
#include "utility/StrUtils.hpp"
#include "netlist/Component.hpp"

// wxWidgets Includes

#include <wx/filename.h>
#include <wx/textfile.h>
#include <wx/tokenzr.h>

//**************************************************************************************************

class NetList
{
  protected :

    // File names
    static   wxFileName      m_ofnLoadFile;
    static   wxFileName      m_ofnSaveFile;
    static   ArrayFileName   m_oaSchemFiles;

  public :

    // String array containing the netlist data
    static   wxArrayString   m_osaNetLst;

    // String arrays containing data extracted from the netlist
    static   wxArrayString   m_osaTitle;
    static   wxArrayString   m_osaIncludes;
    static   ArrayComponent  m_oaCpnts;
    static   wxArrayString   m_osaModels;
    static   wxArrayString   m_osaSubCcts;

    // String arrays containing node labels only
    static   wxArrayString   m_osaNodeLbls;

  private :

    static   bool  m_bIsValid;

             // Functions to transfer circuit description information to or from m_osaNetLst
             bool  bLoadTitle     ( void );
             bool  bLoadIncludes  ( void );
             bool  bLoadCpnts     ( void );
             bool  bLoadModels    ( void );
             bool  bLoadSubCcts   ( void );
             bool  bLoadNodeLbls  ( void );
             bool  bLoadSchemFiles( void );

             bool  bIsSubCkt( wxString & roLine );

  public :

                   NetList( void );
                  ~NetList( );

    static   bool  bClear   ( void );
             bool  bIsEmpty ( void ) { return( m_osaNetLst.IsEmpty( ) ); }

    virtual  bool  bValidate( void );
    virtual  bool  bIsValid ( void ) { return( m_bIsValid ); }

    static   bool  bLoadFile( const wxString & rosFName=wxT("") );
    static   bool  bSaveFile( const wxString & rosFName=wxT("") );

    virtual  bool  bLoad    ( void );
    virtual  bool  bSave    ( void );

    static   bool  bSetLoadFile  ( const wxString      &  rosFName  );
    static   bool  bSetSaveFile  ( const wxString      &  rosFName  );
             bool  bSetSchemFiles( const wxArrayString & rosaFNames );
             bool  bSetSchemFiles( const wxString      &  rosFNames );

    static  const  wxFileName    &  roGetLoadFile  ( void ) { return( m_ofnLoadFile  ); }
    static  const  wxFileName    &  roGetSaveFile  ( void ) { return( m_ofnSaveFile  ); }
    static  const  ArrayFileName &  roGetSchemFiles( void ) { return( m_oaSchemFiles ); }
    static  const  wxString      & rosGetSchemFiles( void );
    static  const  Component     &  roGetCpnt      ( const wxString & rosName );

    virtual  void  Print( const wxString & rosPrefix=wxT("  ") );

    friend  class  PrcGnuCap;
    friend  class  PrcNgSpice;
};

//**************************************************************************************************

#endif // NETLIST_HPP
