/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "minorfeed.h"


MinorFeed::MinorFeed(PumpController *pumpController,
                     QWidget *parent) : QFrame(parent)
{
    this->pController = pumpController;
    this->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);

    // Layout for the items
    itemsLayout = new QVBoxLayout();
    itemsLayout->setAlignment(Qt::AlignTop);

    // Button to get more items (older stuff)
    getMoreButton = new QPushButton(QIcon::fromTheme("list-add"),
                                    tr("Get More"));
    getMoreButton->setFlat(true);
    getMoreButton->setToolTip(tr("Get older activities"));
    connect(getMoreButton, SIGNAL(clicked()),
            this, SLOT(getMoreActivities()));

    // Set disabled initially; will be enabled when contents are set
    getMoreButton->setDisabled(true);

    // Main layout
    mainLayout = new QVBoxLayout();
    mainLayout->setAlignment(Qt::AlignTop);
    mainLayout->addLayout(itemsLayout);
    mainLayout->addWidget(getMoreButton);
    this->setLayout(mainLayout);



    // Demo activity, stating that there's nothing to show
    QVariantMap demoGenerator;
    demoGenerator.insert("displayName", "Dianara");

    QVariantMap demoActivityMap;
    demoActivityMap.insert("published", QDateTime::currentDateTimeUtc()
                                        .toString(Qt::ISODate));
    demoActivityMap.insert("generator", demoGenerator);
    demoActivityMap.insert("content",   tr("There are no activities to show yet."));

    ASActivity *demoActivity = new ASActivity(demoActivityMap, this);

    MinorFeedItem *demoFeedItem = new MinorFeedItem(demoActivity,
                                                    "",
                                                    this->pController);
    this->itemsLayout->addWidget(demoFeedItem);
    this->itemsInFeed.append(demoFeedItem);

    this->feedOffset = 0;
    this->newItemsCount = 0;

    QSettings settings;
    settings.beginGroup("MinorFeedState");
    this->previousNewestActivityId = settings.value("previousNewestItemId", "").toString();
    settings.endGroup();

    qDebug() << "MinorFeed created";
}


MinorFeed::~MinorFeed()
{
    QSettings settings;

    settings.beginGroup("MinorFeedState");
    settings.setValue("previousNewestItemId", this->previousNewestActivityId);
    settings.endGroup();

    qDebug() << "MinorFeed destroyed";
}


void MinorFeed::clearContents()
{
    qDebug() << "MinorFeed::clearContents()";
    foreach (MinorFeedItem *feedItem, itemsInFeed)
    {
        this->itemsLayout->removeWidget(feedItem);
        delete feedItem;
    }

    itemsInFeed.clear();
}


void MinorFeed::markAllAsRead()
{
    foreach (MinorFeedItem *feedItem, itemsInFeed)
    {
        feedItem->setItemAsNew(false);
    }
}



/*******************************************************************************/
/********************************** SLOTS **************************************/
/*******************************************************************************/


/*
 * Get the latest activities
 *
 */
void MinorFeed::updateFeed()
{
    this->feedOffset = 0;
    this->pController->getMinorFeed();
}

/*
 * Get additional older activities
 *
 */
void MinorFeed::getMoreActivities()
{
    this->feedOffset += 50;
    this->pController->getMinorFeed(feedOffset);
}



void MinorFeed::setFeedContents(QVariantList activitiesList)
{
    this->getMoreButton->setDisabled(true);

    if (feedOffset == 0) // Not asking for more, but a reload
    {
        this->clearContents();
        this->newItemsCount = 0;
    }

    // Process event queue, so GUI can get updated
    qApp->processEvents(); // This also acts as "scroll to top", incidentally


    QStringList filteredContent;
    QStringList filteredAuthor;
    QStringList filteredGenerator;

    // Define filters
    foreach (QVariant filter, pController->getCurrentFilters())
    {
        int filterType = filter.toMap().value("type").toInt();
        QString filterText = filter.toMap().value("text").toString();

        switch (filterType)
        {
        case 0: // content
            filteredContent.append(filterText);
            break;

        case 1: // author
            filteredAuthor.append(filterText);
            break;

        case 2: // application (generator)
            filteredGenerator.append(filterText);
            break;

        default:
            break;
        }
    }



    QString activityActorAvatarUrl;
    QString activityActorAvatarFilename;

    QString activityContent;
    QString activityAuthor;
    QString activityGenerator;

    bool filtered;

    bool allNewItemsCounted = false;
    QString newestActivityId; // To store the activity ID for the newest item in the feed
                              // so we can know how many new items we receive next time


    foreach (QVariant activityVariant, activitiesList)
    {
        ASActivity *activity = new ASActivity(activityVariant.toMap(), this);

        ////////////////////////////////////////////////////////////// Filtering Start
        filtered = false;  // Innocent until proven guilty!

        activityContent = activity->getContent();
        foreach (QString content, filteredContent)
        {
            if (activityContent.contains(content))
            {
                qDebug() << "Filtering item because of Content:" << content;
                filtered = true;
            }
        }

        activityAuthor = activity->getAuthorName();
        foreach (QString author, filteredAuthor)
        {
            if (activityAuthor.contains(author))
            {
                qDebug() << "Filtering item because of Author:" << author;
                filtered = true;
            }
        }

        activityGenerator = activity->getGenerator();
        foreach (QString generator, filteredGenerator)
        {
            if (activityGenerator.contains(generator))
            {
                qDebug() << "Filtering item because of Application (generator):" << generator;
                filtered = true;
            }
        }
        ////////////////////////////////////////////////////////////// Filtering End

        bool itemIsNew;

        // If there is no reason to filter out the item, add it to the feed
        if (!filtered)
        {
            itemIsNew = false;

            // Determine which activities are new
            if (newestActivityId.isEmpty()) // Only first time, for newest item
            {
                if (this->feedOffset == 0)
                {
                    newestActivityId = activity->getId();
                }
                else
                {
                    newestActivityId = this->previousNewestActivityId;
                    allNewItemsCounted = true;
                }
            }

            if (!allNewItemsCounted)
            {
                if (activity->getId() == this->previousNewestActivityId)
                {
                    allNewItemsCounted = true;
                }
                else
                {
                    // If activity is not ours, add it to the count
                    if (activity->getAuthorId() != pController->currentUserId())
                    {
                        ++newItemsCount;
                        itemIsNew = true;
                    }
                }
            }



            activityActorAvatarUrl = activity->getAuthorAvatar();
            activityActorAvatarFilename = MiscHelpers::getCachedAvatarFilename(activityActorAvatarUrl);
            if (!QFile::exists(activityActorAvatarFilename))
            {
                this->pController->getAvatar(activityActorAvatarUrl);
            }

            MinorFeedItem *newFeedItem = new MinorFeedItem(activity,
                                                           activityActorAvatarFilename,
                                                           this->pController);
            if (itemIsNew)
            {
                newFeedItem->setItemAsNew(true);
                connect(newFeedItem, SIGNAL(itemRead()),
                        this, SLOT(decreaseNewItemsCount()));
            }
            this->itemsLayout->addWidget(newFeedItem);
            this->itemsInFeed.append(newFeedItem);
        }
        else
        {
            // Since the item is not added to the feed, we need to delete the activity
            delete activity;
        }
    }


    this->previousNewestActivityId = newestActivityId;
    emit newItemsCountChanged(newItemsCount);
    qDebug() << "Meanwhile feed updated; New items:" << newItemsCount;

    this->getMoreButton->setEnabled(true);
}


void MinorFeed::decreaseNewItemsCount()
{
    --newItemsCount;

    emit newItemsCountChanged(newItemsCount);
}
