/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _GNU_SOURCE /* asprintf() */

#include <inttypes.h>
#include <stdlib.h>
#include <unistd.h> /* sleep() and usleep() on Alpine Linux */

#include "cdw_file_manager.h"
#include "gettext.h"
#include "cdw_create_image.h"
#include "cdw_utils.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_mkisofs.h"
#include "cdw_xorriso.h"
#include "cdw_mkudffs.h"
#include "cdw_mkudffs_helpers.h"
#include "cdw_main_window.h"
#include "cdw_processwin.h"
#include "cdw_image_wizard.h"
#include "cdw_logging.h"
#include "cdw_config.h"
#include "cdw_string.h"




extern cdw_config_t global_config;





/* Pause between each step of "make UDF" recipe, in microseconds. This
   pause is for improved user experience, so that - for some steps -
   messages in process window don't "flash" too quickly. */
#define CDW_CREATE_IMAGE_PAUSE       1000000  /* [us] */

/* Additional wait time after call of sync(1) utility returns, to give
   a hard disc some more time to finalize syncing contents of mounted
   UDF file system. Otherwise we may attempt to umount UDF filesystem
   too early, when copied data is still flushed to the filesystem. */
#define CDW_CREATE_IMAGE_SYNC_DELAY       5  /* [s] */

static cdw_id_t cdw_create_image_get_initial_task_id(void);
//static cdw_rv_t cdw_create_image_check_boot_image(void);
static cdw_rv_t cdw_create_image_udf(cdw_task_t *task);





/**
   \brief Check preconditions for creating image and invoke process creating image

   Check if there are any files selected by user, do any other
   necessary checks and then call run_command_*() to create ISO
   image from the files.

   \return CDW_NO if some of preconditions were not met
   \return CDW_ERROR on errors
   \return value returned by cdw_mkisofs_run_task()
*/
cdw_rv_t cdw_create_image(void)
{
	if (!cdw_selected_files_get_number()) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("No files selected"),
				   /* 2TRANS: this is message in dialog window, user
				      wants to write files or create image, but no
				      files from hdd are selected yet */
				   _("No files selected. Please use 'Add files'"),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		return CDW_NO;
	}


	cdw_id_t task_id = cdw_create_image_get_initial_task_id();
	if (task_id == CDW_TASK_NONE) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("No tools available"),
				   /* 2TRANS: this is message in dialog window, user
				      wants to write files or create image, but no
				      tools for this task are available. */
				   _("Cannot create image file. No tools for this task are available. Check your configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		return CDW_NO;
	}
	cdw_vdm ("INFO: initial task id = %lld\n", task_id);


	cdw_task_t *task = cdw_task_new(task_id, (cdw_disc_t *) NULL);
	if (!task) {
		cdw_vdm ("ERROR: failed to create a task\n");
		return CDW_ERROR;
	}


	int rv = cdw_image_wizard(task);
	if (rv == CDW_IMAGE_FORMAT_NONE) {
		cdw_vdm ("INFO: cancelled in wizard\n");
		cdw_task_delete(&task);
		cdw_file_manager_delete_graftpoints_file();
		return CDW_NO;
	} else if (rv == CDW_IMAGE_FORMAT_ERROR) {
		cdw_vdm ("ERROR: error in wizard\n");
		cdw_task_delete(&task);
		cdw_file_manager_delete_graftpoints_file();
		return CDW_ERROR;
	} else {
		; /* Pass, CDW_IMAGE_FORMAT_ISO9660 || CDW_IMAGE_FORMAT_UDF. */
	}

	/* 2TRANS: this is title of dialog window */
	cdw_processwin_create(_("Create image"),
			      task->id == CDW_TASK_CREATE_IMAGE_ISO9660 ?
			      /* 2TRANS: this is message in dialog
				 window - creating ISO9660 image is in
				 progress */
			      _("Creating ISO9660 image...") :
			      /* 2TRANS: this is message in dialog
				 window - creating UDF image is in
				 progress */
			      _("Creating UDF image..."),
			      false);


	if (rv == CDW_IMAGE_FORMAT_ISO9660) {
		task->id = CDW_TASK_CREATE_IMAGE_ISO9660;
		/* create graftpoints file for mkisofs; first checks if there
		   are any files that will be used as source data */

		/* 2TRANS: this is message in dialog window - cdw is putting file names of selected files on a list. */
		cdw_processwin_display_sub_info(_("Preparing list of files..."));

		cdw_rv_t crv = cdw_file_manager_create_graftpoints_file();
		if (crv != CDW_OK) {
			cdw_task_delete(&task);
			cdw_processwin_destroy(_("Error"), true);
			return CDW_NO;
		}
		cdw_processwin_display_sub_info(""); /* Graftpoins file has been created, clear message about creating list of files. */
	} else {
		task->id = CDW_TASK_CREATE_IMAGE_UDF;
	}

	/* The function has been called in cdw_task_new(), but we need
	   to call it again after user has selected image format in
	   wizard. */
	cdw_rv_t crv = cdw_task_select_tools_for_task(task, task->disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to select tools for task\n");
		cdw_task_delete(&task);
		cdw_processwin_destroy(_("Error"), true);
		return CDW_NO;
	}

#if 0
	/* check if user specified boot image, and if so,
	   if the boot image exists */
	crv = cdw_create_image_check_boot_image();
	if (crv != CDW_OK) {
		cdw_task_delete(&task);
		return CDW_NO;
	}
#endif

	cdw_logging_write_separator();

	/* go! */
	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		cdw_vdm ("INFO: path at call to run task: %s\n", task->image_file_fullpath);
		cdw_processwin_add_progress_bar();
		crv = cdw_mkisofs_run_task(task, (cdw_disc_t *) NULL);
	} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
		cdw_processwin_add_progress_bar();
		crv = cdw_xorriso_run_task(task, (cdw_disc_t *) NULL);
	} else if (task->create_image.tool.id == CDW_TOOL_MKUDFFS) {
		crv = cdw_create_image_udf(task);
	} else {
		cdw_vdm ("ERROR: tool id is invalid: %lld\n", task->create_image.tool.id);
		crv = CDW_ERROR;
	}

	if (crv == CDW_OK) {
		/* 2TRANS: this is message in process window:
		   operation finished with unknown result. */
		cdw_processwin_destroy(_("Finished"), true);
	} else {
		/* 2TRANS: this is message in process window:
		   operation finished with error. */
		cdw_processwin_destroy(_("Error"), true);
	}

	if (task->id == CDW_TASK_CREATE_IMAGE_ISO9660) {
		cdw_file_manager_delete_graftpoints_file();
	}

	/* this function sets task.success according to task->tool_status,
	   and resets task->tool_status; displays error messages in case
	   of (some) problems reported by mkisofs */
	cdw_task_check_tool_status(task);

	if (crv == CDW_OK) {
		cdw_task_save_options(task);
		/* 2TRANS: this is title of dialog window, window shows
		   messages from program writing selected files to iso
		   image (creating iso image from selected files) */
		after_event(_("\"Create image\" log"), 1);
	} else {
		cdw_sdm ("ERROR: failed to create image\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      probably some malloc() call failed when
				      program was preparing call of mkisofs;
				      'command' is command-line command. 'Image'
				      means ISO9660 or UDF image file. */
				   _("Some error occurred when creating image. Image not created or the image is invalid. Please consult log file ('L' hotkey in main window)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	}

	cdw_task_delete(&task);

	return crv;
}





cdw_id_t cdw_create_image_get_initial_task_id(void)
{
	bool can_do_iso = cdw_ext_tools_iso9660_sa_tool_available();
	bool can_do_udf = cdw_ext_tools_udf_sa_tools_tools_available();

	if (can_do_iso && global_config.task_id == CDW_TASK_CREATE_IMAGE_ISO9660) {

		/* Previously selected image format was ISO9660. Use
		   (initially) the same format this time. */
		return CDW_TASK_CREATE_IMAGE_ISO9660;
	}

	if (can_do_udf && global_config.task_id == CDW_TASK_CREATE_IMAGE_UDF) {

		/* Previously selected image format was UDF. Use
		   (initially) the same format this time. */
		return CDW_TASK_CREATE_IMAGE_UDF;
	}


	/* Previous task was not "create image" task. Function is free
	   to select any image format as default/initial one. Let's
	   use ISO9660 (if there are tools for it). */

	if (can_do_iso) {
		cdw_vdm ("INFO: can do ISO9660\n");
		return CDW_TASK_CREATE_IMAGE_ISO9660;
	}

	if (can_do_udf) {
		cdw_vdm ("INFO: can do UDF\n");
		return CDW_TASK_CREATE_IMAGE_UDF;
	}

	cdw_vdm ("INFO: cannot do neither ISO9660 nor UDF\n");
	return CDW_TASK_NONE;
}





#if 0
cdw_rv_t cdw_create_image_check_boot_image(void)
{
	if (strlen(global_config.boot_image_path)) {
		int rv = cdw_fs_check_existing_path(global_config.boot_image_path, R_OK, CDW_FS_FILE);
		if (rv != 0) {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Boot image error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Boot image file doesn't exist or has wrong permissions! Check \"Boot image\" option in Configuration."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_NO;
		}
	}
	return CDW_OK;
}
#endif





cdw_rv_t cdw_create_image_udf(cdw_task_t *task)
{
	cdw_processwin_display_main_info("Getting volume size");

	usleep(2 * CDW_CREATE_IMAGE_PAUSE);

	/* Ensure that we have some value of UDF file size before
	   doing anything else. */
	uint64_t volume_size = 0;
	if (global_config.general.volume_size_id == CDW_CONFIG_VOLUME_SIZE_AUTO) {
		volume_size = cdw_disc_get_total_capacity_bytes();
	} else {
		/* TODO: use function that returns value in bytes. */
		long int size = cdw_config_get_current_volume_size_value_megabytes();
		cdw_vdm ("INFO: size = %ld\n", size);
		if (size < 0) {
			volume_size = 0;
		} else {
			volume_size = ((uint64_t) size) * 1024 * 1024;
		}
	}

	if (volume_size == 0) {
		if (global_config.general.volume_size_id == CDW_CONFIG_VOLUME_SIZE_AUTO) {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is message in dialog window */
					   _("To create UDF image, cdw needs to get size of UDF volume.\ncdw failed to get total capacity from optical disc. Enter disc into tray, or change settings in Configuration -> Log and misc -> Volume size."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is message in dialog window */
					   _("To create UDF image, cdw needs to get size of UDF volume.\nVolume size configured in Configuration -> Log and misc -> Volume size is invalid."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		}
		cdw_main_window_wrefresh();
		cdw_processwin_wrefresh();

		return CDW_ERROR;
	}

	uint64_t selected_size = (uint64_t) cdw_selected_files_get_size();
	cdw_vdm ("INFO: selected size: %"PRIu64"\n", selected_size);
	cdw_vdm ("INFO: volume size:   %"PRIu64" bytes (%f sectors)\n", volume_size, volume_size / 2048.0);
	if (volume_size < selected_size) {
		/* 2TRANS: this is title of dialog window */
		cdw_rv_t crv = cdw_buttons_dialog(_("Warning"),
						  /* 2TRANS: this is message in dialog window */
						  _("Current volume size is / may be too small for selected files. Expect problems with this UDF image. Press OK to continue, press Cancel to cancel."),
						  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);
		if (crv != CDW_OK) {
			cdw_main_window_wrefresh();
			cdw_processwin_wrefresh();
			return CDW_NO;
		}
	}

	task->create_image.udf.volume_size_bytes = volume_size;


	cdw_main_window_wrefresh();

	/* Clear messages that may have been used when getting volume
	   size information from optical disc. */
	cdw_processwin_display_main_info("");
	cdw_processwin_display_sub_info("");
	cdw_processwin_wrefresh();



	/* No progress bar while calling truncate, mkudffs and mount. */
	//cdw_processwin_delete_progress_bar();

	cdw_logging_write("\n\nCreating UDF file: 1/6: attempting to create initial empty file:");
	cdw_processwin_display_main_info("Creating initial .udf file");
	usleep(2 * CDW_CREATE_IMAGE_PAUSE);
	cdw_rv_t crv = cdw_truncate_run_task(task);
	if (crv != CDW_OK) {
		cdw_logging_write("Creating UDF file: 1/6: attempting to create initial empty file FAILED\n");
		cdw_logging_write("Creating UDF file: 1/6: \"truncate\" exit status is %d.\n", task->tool_status.child_exit_status);
		return CDW_ERROR;
	}


	usleep(CDW_CREATE_IMAGE_PAUSE);


	cdw_logging_write("\n\nCreating UDF file: 2/6: attempting to create initial UDF file system:\n");
	cdw_processwin_display_main_info("Creating initial UDF file system");
	crv = cdw_mkudffs_run_task(task);
	if (crv != CDW_OK) {
		cdw_logging_write("Creating UDF file: 2/6: attempting to create initial UDF file system FAILED\n");
		cdw_logging_write("Creating UDF file: 2/6: \"mkudffs\" exit status is %d.\n", task->tool_status.child_exit_status);
		return CDW_ERROR;
	}


	usleep(CDW_CREATE_IMAGE_PAUSE);


	cdw_logging_write("\n\nCreating UDF file: 3/6: attempting to mount UDF file system:\n");
	cdw_processwin_display_main_info("Mounting UDF file system");
	crv = cdw_mount_run_task(task);
	if (crv != CDW_OK) {
		cdw_logging_write("Creating UDF file: 3/6: attempting to mount UDF file system FAILED.\n");
		cdw_logging_write("Creating UDF file: 3/6: \"sudo mount\" exit status is %d.\n", task->tool_status.child_exit_status);
		return CDW_ERROR;
	} else {
		;
	}
	/* A file system has been mounted. It will be unmounted at the
	   end of this recipe, but it also has to be unmounted if -
	   because of errors in one of next steps - the recipe will be
	   terminated. From now on always attempt to unmount the file
	   system in error handling code. */


	usleep(CDW_CREATE_IMAGE_PAUSE);


	cdw_logging_write("\n\nCreating UDF file: 4/6: attempting to copy (rsync) files into UDF file system:\n");
	cdw_processwin_display_main_info("Copying files to UDF file system");
	cdw_processwin_add_progress_bar();
	crv = cdw_rsync_run_task(task);
	if (crv != CDW_OK) {
		cdw_logging_write("Creating UDF file: 4/6: attempting to copy (rsync) files into UDF file system FAILED.\n");
		cdw_logging_write("Creating UDF file: 4/6: \"rsync\" exit status is %d.\n", task->tool_status.child_exit_status);

		crv = cdw_umount_run_task(task);
		if (crv != CDW_OK) {
			cdw_logging_write("Creating UDF file: 4/6: failed to unmount mounted UDF file system.\n");
		}

		return CDW_ERROR;
	}


	usleep(CDW_CREATE_IMAGE_PAUSE);


	cdw_logging_write("\n\nCreating UDF file: 5/6: syncing devices:\n");
	cdw_processwin_display_main_info("Syncing...");
	cdw_processwin_display_sub_info("This may take a while");
	crv = cdw_sync_run_task(task);
	if (crv != CDW_OK) {
		cdw_logging_write("Creating UDF file: 5/6: syncing devices FAILED.\n");
		cdw_logging_write("Creating UDF file: 5/6: \"sync\" exit status is %d.\n", task->tool_status.child_exit_status);

		crv = cdw_umount_run_task(task);
		if (crv != CDW_OK) {
			cdw_logging_write("Creating UDF file: 5/6: failed to unmount mounted UDF file system.\n");
		}
		return CDW_ERROR;
	}



	/* Give "sync" command enough time to completely flush data
	   into UDF file system. */
	int sync_delay = CDW_CREATE_IMAGE_SYNC_DELAY;
	/* Get number of digits in value of volume size. */
	{
		char *size_string = (char *) NULL;
		int n = asprintf(&size_string, "%"PRIu64"", task->create_image.udf.volume_size_bytes);
		if (n > 0) {
			sync_delay += n;
		} else {
			/* A minimal effort to increase sync delay. */
			sync_delay += 5;
		}
		cdw_string_delete(&size_string);
	}
	sleep((unsigned int) sync_delay);



	cdw_processwin_delete_progress_bar();
	cdw_processwin_wrefresh();

	cdw_logging_write("\n\nCreating UDF file: 6/6: attempting to unmount UDF file system:\n");
	cdw_processwin_display_main_info("Unmounting UDF filesystem");
	cdw_processwin_display_sub_info("");
	crv = cdw_umount_run_task(task);
	if (crv != CDW_OK) {
		cdw_logging_write("Creating UDF file: 6/6: attempting to unmount UDF file system FAILED.\n");
		cdw_logging_write("Creating UDF file: 6/6: \"sudo umount\" exit status is %d.\n", task->tool_status.child_exit_status);
		return CDW_ERROR;
	}


	usleep(CDW_CREATE_IMAGE_PAUSE);


	return CDW_OK;
}
