//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/MaterialModel.h
//! @brief     Defines class MaterialModel
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_SAMPLE_MATERIALMODEL_H
#define BORNAGAIN_GUI_MODEL_SAMPLE_MATERIALMODEL_H

#include <QMap>
#include <QObject>
#include <QVector>
#include <QXmlStreamReader>

//! Materials created by default
enum class DefaultMaterials { Default, Vacuum, Particle, Core, Substrate };

extern QMap<QString, DefaultMaterials> materialMap;

class MaterialItem;

class MaterialModel : public QObject {
    Q_OBJECT

public:
    MaterialModel();
    ~MaterialModel();

    void clear();

    //! Add the material and take ownership of it.
    MaterialItem* addMaterialItem(MaterialItem* materialItem);

    MaterialItem* addRefractiveMaterialItem(const QString& name, double delta, double beta);
    MaterialItem* addSLDMaterialItem(const QString& name, double sld, double abs_term);

    MaterialItem* materialItemFromName(const QString& name) const;
    MaterialItem* materialItemFromIdentifier(const QString& identifier) const;

    //! Inserts a copy of the given material and returns the newly inserted item.
    //!
    //! The copy will have a different material identifier and a different name.
    MaterialItem* copyMaterialItem(const MaterialItem& materialItem);

    const QVector<MaterialItem*>& materialItems() const;

    MaterialItem* defaultMaterialItem() const;
    MaterialItem* defaultCoreMaterialItem() const;
    MaterialItem* defaultParticleMaterialItem() const;

    void removeMaterialItem(MaterialItem* materialItem);

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    //! Copies the complete content, emitting signals for existing and changed materials
    void initFrom(const MaterialModel& from);

signals:
    void materialAddedOrRemoved();
    void materialChanged();

private:
    //! Add the material and take ownership of it.
    //!
    //! In the given flag the signal "materialAddedOrRemoved" can be suppressed.
    MaterialItem* addMaterialItem(MaterialItem* material, bool signalAdding);

    MaterialItem* addMaterialItem();

    // TODO OwningVector
    QVector<MaterialItem*> m_materials; //!< all materials (owned by this class)
};

#endif // BORNAGAIN_GUI_MODEL_SAMPLE_MATERIALMODEL_H
