# Copyright 2012-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Test the factory where appropriate.  Don't overdo this."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = []

from datetime import datetime
from itertools import count
import os.path
from random import randint
import subprocess

from maastesting.factory import factory
from maastesting.testcase import MAASTestCase
from netaddr import (
    IPAddress,
    IPNetwork,
    )
from testtools.matchers import (
    Contains,
    FileContains,
    FileExists,
    MatchesAll,
    Not,
    StartsWith,
    )


class TestFactory(MAASTestCase):

    def test_getRandomString_respects_size(self):
        sizes = [1, 10, 100]
        random_strings = [factory.getRandomString(size) for size in sizes]
        self.assertEqual(sizes, [len(string) for string in random_strings])

    def test_getRandomBoolean_returns_bool(self):
        self.assertIsInstance(factory.getRandomBoolean(), bool)

    def test_getRandomPort_returns_int(self):
        self.assertIsInstance(factory.getRandomPort(), int)

    def test_getRandomIPAddress(self):
        ip_address = factory.getRandomIPAddress()
        self.assertIsInstance(ip_address, unicode)
        octets = ip_address.split('.')
        self.assertEqual(4, len(octets))
        for octet in octets:
            self.assertTrue(0 <= int(octet) <= 255)

    def test_getRandomIPAddress_but_not(self):
        # We want to look for clashes between identical IPs and/or netmasks.
        # Narrow down the range of randomness so we have a decent chance of
        # triggering a clash, but not so far that we'll loop for very long
        # trying to find a network we haven't seen already.
        self.patch(
            factory, 'getRandomIPAddress',
            lambda: '10.%d.0.0' % randint(1, 200))
        networks = []
        for _ in range(100):
            networks.append(factory.getRandomNetwork(but_not=networks))
        self.assertEquals(len(networks), len(set(networks)))

    def test_getRandomUUID(self):
        uuid = factory.getRandomUUID()
        self.assertIsInstance(uuid, unicode)
        self.assertEqual(36, len(uuid))

    def test_getRandomNetwork(self):
        network = factory.getRandomNetwork()
        self.assertIsInstance(network, IPNetwork)

    def test_getRandomIPInNetwork(self):
        network = factory.getRandomNetwork()
        ip = factory.getRandomIPInNetwork(network)
        self.assertTrue(
            network.first <= IPAddress(ip).value <= network.last)

    def test_make_ip_range_returns_IPs(self):
        low, high = factory.make_ip_range()
        self.assertIsInstance(low, IPAddress)
        self.assertIsInstance(high, IPAddress)
        self.assertLess(low, high)

    def test_make_ip_range_obeys_network(self):
        network = factory.getRandomNetwork()
        low, high = factory.make_ip_range(network)
        self.assertIn(low, network)
        self.assertIn(high, network)

    def test_make_ip_range_returns_low_and_high(self):
        # Make a very very small network, to maximise the chances of exposure
        # if the method gets this wrong e.g. by returning identical addresses.
        low, high = factory.make_ip_range(factory.getRandomNetwork(slash=31))
        self.assertLess(low, high)

    def test_make_ip_range_obeys_but_not(self):
        # Make a very very small network, to maximise the chances of exposure
        # if the method gets this wrong.
        network = factory.getRandomNetwork(slash=30)
        first_low, first_high = factory.make_ip_range(network)
        second_low, second_high = factory.make_ip_range(
            network, but_not=(first_low, first_high))
        self.assertNotEqual((first_low, first_high), (second_low, second_high))

    def test_getRandomDate_returns_datetime(self):
        self.assertIsInstance(factory.getRandomDate(), datetime)

    def test_getRandomMACAddress(self):
        mac_address = factory.getRandomMACAddress()
        self.assertIsInstance(mac_address, unicode)
        self.assertEqual(17, len(mac_address))
        for hex_octet in mac_address.split(":"):
            self.assertTrue(0 <= int(hex_octet, 16) <= 255)

    def test_getRandomMACAddress_alternative_delimiter(self):
        self.patch(factory, "random_octets", count(0x3a))
        mac_address = factory.getRandomMACAddress(delimiter="-")
        self.assertEqual("3a-3b-3c-3d-3e-3f", mac_address)

    def test_make_random_leases_maps_ips_to_macs(self):
        [(ip, mac)] = factory.make_random_leases().items()
        self.assertEqual(
            4, len(ip.split('.')),
            "IP address does not look like an IP address: '%s'" % ip)
        self.assertEqual(
            6, len(mac.split(':')),
            "MAC address does not look like a MAC address: '%s'" % mac)

    def test_make_random_leases_randomizes_ips(self):
        self.assertNotEqual(
            factory.make_random_leases().keys(),
            factory.make_random_leases().keys())

    def test_make_random_leases_randomizes_macs(self):
        self.assertNotEqual(
            factory.make_random_leases().values(),
            factory.make_random_leases().values())

    def test_make_random_leases_returns_requested_number_of_leases(self):
        num_leases = randint(0, 3)
        self.assertEqual(
            num_leases,
            len(factory.make_random_leases(num_leases)))

    def test_make_file_creates_file(self):
        self.assertThat(factory.make_file(self.make_dir()), FileExists())

    def test_make_file_writes_contents(self):
        contents = factory.getRandomString().encode('ascii')
        self.assertThat(
            factory.make_file(self.make_dir(), contents=contents),
            FileContains(contents))

    def test_make_file_makes_up_contents_if_none_given(self):
        with open(factory.make_file(self.make_dir())) as temp_file:
            contents = temp_file.read()
        self.assertNotEqual('', contents)

    def test_make_file_uses_given_name(self):
        name = factory.getRandomString()
        self.assertEqual(
            name,
            os.path.basename(factory.make_file(self.make_dir(), name=name)))

    def test_make_file_uses_given_dir(self):
        directory = self.make_dir()
        name = factory.getRandomString()
        self.assertEqual(
            (directory, name),
            os.path.split(factory.make_file(directory, name=name)))

    def test_make_name_returns_unicode(self):
        self.assertIsInstance(factory.make_name(), unicode)

    def test_make_name_includes_prefix_and_separator(self):
        self.assertThat(factory.make_name('abc'), StartsWith('abc-'))

    def test_make_name_includes_random_text_of_requested_length(self):
        size = randint(1, 99)
        self.assertEqual(
            len('prefix') + len('-') + size,
            len(factory.make_name('prefix', size=size)))

    def test_make_name_includes_random_text(self):
        self.assertNotEqual(
            factory.make_name(size=100), factory.make_name(size=100))

    def test_make_name_uses_configurable_separator(self):
        sep = 'SEPARATOR'
        prefix = factory.getRandomString(3)
        self.assertThat(
            factory.make_name(prefix, sep=sep),
            StartsWith(prefix + sep))

    def test_make_name_does_not_require_prefix(self):
        size = randint(1, 99)
        unprefixed_name = factory.make_name(sep='-', size=size)
        self.assertEqual(size, len(unprefixed_name))
        self.assertThat(unprefixed_name, Not(StartsWith('-')))

    def test_make_name_does_not_include_weird_characters(self):
        self.assertThat(
            factory.make_name(size=100),
            MatchesAll(*[Not(Contains(char)) for char in '/ \t\n\r\\']))

    def test_make_names_calls_make_name_with_each_prefix(self):
        self.patch(factory, "make_name", lambda prefix: prefix + "-xxx")
        self.assertSequenceEqual(
            ["abc-xxx", "def-xxx", "ghi-xxx"],
            list(factory.make_names("abc", "def", "ghi")))

    def test_make_tarball_writes_tarball(self):
        filename = factory.make_name()
        contents = {filename: factory.getRandomString()}

        tarball = factory.make_tarball(self.make_dir(), contents)

        dest = self.make_dir()
        subprocess.check_call(['tar', '-xzf', tarball, '-C', dest])
        self.assertThat(
            os.path.join(dest, filename),
            FileContains(contents[filename]))

    def test_make_tarball_makes_up_content_if_None(self):
        filename = factory.make_name()
        tarball = factory.make_tarball(self.make_dir(), {filename: None})

        dest = self.make_dir()
        subprocess.check_call(['tar', '-xzf', tarball, '-C', dest])
        self.assertThat(os.path.join(dest, filename), FileExists())
        with open(os.path.join(dest, filename), 'rb') as unpacked_file:
            contents = unpacked_file.read()
        self.assertGreater(len(contents), 0)
