/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
Taylor series expansion for J0
Generated by SageMath:
```python
def compute_intervals(zeros):
    intervals = []
    for i in range(0, len(zeros)):
        if i == 0:
            a = (zeros[i]) / 2 - 0.05 - zeros[i]
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        elif i + 1 > len(zeros) - 1:
            a = (zeros[i - 1] + zeros[i]) / 2 - 0.05 - zeros[i]
            b = (zeros[i]) + 0.83 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        else:
            a = (zeros[i - 1] + zeros[i]) / 2 - zeros[i] - 0.05
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05  - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
    return intervals

intervals = compute_intervals(j0_zeros)
# print(intervals)

def build_sollya_script(a, b, zero, deg):
    return f"""
prec = 500;
bessel_j0 = library("./notes/bessel_sollya/cmake-build-release/libbessel_sollya.dylib");
f = bessel_j0(x + {zero});
d = [{a}, {b}];
pf = remez(f, {deg}, d);
for i from 0 to degree(pf) do {{
    write(coeff(pf, i)) >> "coefficients.txt";
    write("\\n") >> "coefficients.txt";
}};
"""

def load_coefficients(filename):
    with open(filename, "r") as f:
        return [RealField(500)(line.strip()) for line in f if line.strip()]

def call_sollya_on_interval(a, b, zero, degree=12):
    sollya_script = build_sollya_script(a, b, zero, degree)
    with open("tmp_interval.sollya", "w") as f:
        f.write(sollya_script)
    import subprocess
    if os.path.exists("coefficients.txt"):
        os.remove("coefficients.txt")
    try:
        result = subprocess.run(
            ["sollya", "tmp_interval.sollya"],
            check=True,
            capture_output=True,
            text=True
        )
    except subprocess.CalledProcessError as e:
        return

def print_remez_coeffs(poly):
    print("J0TaylorExtendedSeries {")
    print_double_double("a0: ", poly[0])
    print_double_double("a1: ", poly[1])
    print_double_double("a2: ", poly[2])
    print_double_double("a3: ", poly[3])
    print_double_double("a4: ", poly[4])
    print_double_double("a5: ", poly[5])
    print_double_double("a6: ", poly[6])
    print_double_double("a7: ", poly[7])
    print_double_double("a8: ", poly[8])
    print_double_double("a9: ", poly[9])
    print_double_double("a10: ", poly[10])
    print_double_double("a11: ", poly[11])
    print_double_double("a12: ", poly[12])
    print_double_double("a13: ", poly[13])
    print("c: [")
    for i in range(14, len(poly)):
        coeff = poly[i]
        print(f"{double_to_hex(coeff)},")
    print("],")
    print("},")


degree = 21

print(f"pub(crate) static J0_COEFFS: [J0TaylorExtendedSeries; {len(intervals)}] = [")
for i in range(0, len(intervals)):
    interval = intervals[i]
    call_sollya_on_interval(interval[0], interval[1], interval[2], degree)
    coeffs = load_coefficients(f"coefficients.txt")
    print_remez_coeffs(coeffs)
print("];")
```
see notes/bessel_j0_taylor.ipynb
**/
pub(crate) static J0_COEFFS_REMEZ: [[(u64, u64); 24]; 47] = [
    [
        (0x35cabb65251ee9e1, 0xb940904225a2e689),
        (0xbc8ac8cc3d6baf81, 0xbfe09cdb36551280),
        (0xbc5b50b1160eaa90, 0x3fbba1deea029494),
        (0x3c4bc98bcfdb9552, 0x3facfae864368d70),
        (0xbc1a87a5513baa19, 0xbf81bb1cbe1a4071),
        (0x3bcf770f17b53b67, 0xbf61f992590d12bd),
        (0x3ba05111ecddc158, 0x3f315382ba06cc47),
        (0x3ba2b3fa2a3da72a, 0x3f06ed3b9f07eb28),
        (0xbb46f7b4858b68c3, 0xbed232c77d228ab6),
        (0x3b47b12392b6b437, 0xbea1cce302821846),
        (0xbab50abcf1097a30, 0x3e67ff99166c20b8),
        (0xbad63d3a07c5005a, 0x3e32951bd4726a93),
        (0xba9950044640bb2e, 0xbdf5c2c38b2a2783),
        (0xba541f8018d200d7, 0xbdbbdc468c1a7eee),
        (0xba165a9b127054c4, 0x3d7cd41cf2489ea1),
        (0x39d72c6649bda0f9, 0x3d3f70b02016ca26),
        (0xb993b90d2d9ac399, 0xbcfd22d8ef92bec0),
        (0xb957753210682ee6, 0xbcbbaa35275a6222),
        (0xb8d21bc46d19becc, 0x3c77361304109810),
        (0xb8d349b7e68b934a, 0x3c33829044877cd2),
        (0xb885327d8bda1ea2, 0xbbede8805081c87c),
        (0x382f2276708fc7d3, 0xbba68d801b3bad5d),
        (0xb8067b24c3fb14b1, 0x3b601bea84e3ff3f),
        (0x37a9569e06a99bfe, 0x3b19ff405c2e53c4),
    ],
    [
        (0x3c62de1143765a96, 0xbfd9c6cf582cbf7f),
        (0xb5f60db51e8c7541, 0x395df78c3799704c),
        (0xbc52de1143765a6f, 0x3fc9c6cf582cbf7f),
        (0xbc3b84c5dde44b4f, 0xbf91f06d14e11e02),
        (0x3c27767d96985ea8, 0xbf8b589d1da13905),
        (0x3beae1afe730c7c9, 0x3f50f9103cf5b152),
        (0x3bd3e8ae8ef24234, 0x3f386445621cc085),
        (0x3b8a8903a2c6108a, 0xbefa2a033ccf2705),
        (0xbb6859eebd6af12b, 0xbed83a06e30c4109),
        (0xbb3d51d443e4c446, 0x3e96a4fd997104b3),
        (0x3af99b61d1c76464, 0x3e6ec03c7b7d1357),
        (0xbac247e4733f0637, 0xbe295db03343bc40),
        (0xba9b3084e57a7079, 0xbdfb1e242e3fafb5),
        (0x3a505dc534b67fe8, 0x3db3fa9bccb27d21),
        (0x3a20ef6f1892b081, 0x3d8195cae4f67f76),
        (0xb9ddf4b73fd307e9, 0xbd3762afe2a6aee9),
        (0x398a6dd1d504dc6d, 0xbd017a40c9409365),
        (0xb90b1f08ce463e38, 0x3cb52a10c56acb7f),
        (0xb8f458d9f512adee, 0x3c7b78e3226aab83),
        (0xb88362e957f09f13, 0xbc2e864870efa32e),
        (0x3883c3c8d8c9548c, 0xbbf180323daae355),
        (0xb84ee7453edcaae1, 0x3ba1f821fbac48d2),
        (0x38094473de9ddfb1, 0x3b62580a3d2321cb),
        (0x37afb06968799ff5, 0xbb13009d6caab011),
    ],
    [
        (0x35dbe52d2b337a1c, 0xb9486fd64fd4d5ac),
        (0xbc6af17f78e5834c, 0x3fd5c6e60a097823),
        (0xbc22c1940b6591bc, 0xbf9f8f72e7a848e0),
        (0x3c39226a69688cd8, 0xbfab2150cb41e8c1),
        (0xbc0ead94b778456a, 0x3f72f7ffe90256bb),
        (0xbbdc44bdbd19f356, 0x3f627e31fe9a9779),
        (0xbbc64599a5779ae3, 0xbf26f641f41956f7),
        (0xbb948617d08510cb, 0xbf0863f481a43036),
        (0x3b680f7f67daa761, 0x3ecad77d748a06db),
        (0x3b3cc529a50aa60d, 0x3ea32e6d99c6af7d),
        (0xbac1cce367cee3c7, 0xbe62da37e38435b9),
        (0xbada187c686c04e7, 0xbe341d72d9392e0e),
        (0xba9d2e19dd36e939, 0x3df1d0433d9a0e48),
        (0x3a31b2df296572e6, 0x3dbe2f3389aa5fae),
        (0x39eecfa6ba4ba8c8, 0xbd78498ffdebd97b),
        (0x39d03604efc229e3, 0xbd410477aeae8e02),
        (0xb9773b9fe3142404, 0x3cf911a8d777928f),
        (0xb957c6801c088821, 0x3cbde68af5af2e98),
        (0xb90138809c41f3e6, 0xbc744b38708ab5a6),
        (0x38db3cf268b04601, 0xbc350be960e7059b),
        (0xb88759fc2b6b9b1a, 0x3bea7a1734099f03),
        (0x3834f08a90df154d, 0x3ba84749424dc4da),
        (0xb7d2d06b0965e481, 0xbb5c42f208f4edf8),
        (0xb7bf8d11d7307b88, 0xbb1812bbeb085f69),
    ],
    [
        (0x3c7af22d033ee0a4, 0x3fd33518b3874e8a),
        (0xb565f8b72bba82bc, 0xb8cc87a125947e43),
        (0xbc6af22d033ee078, 0xbfc33518b3874e8a),
        (0xbc006437c8a4159e, 0x3f7d34125d59d8ff),
        (0x3c276d8715dea5f6, 0x3f880c83bdeee5b0),
        (0x3be6843190a13b37, 0xbf4483c20f1cb1cf),
        (0xbbb05a9f2f74161a, 0xbf36ffa5fc8cad74),
        (0x3b913c314acf400c, 0x3ef2ccf7b21fff81),
        (0x3b3475a5e179bf64, 0x3ed796a751f89051),
        (0x3b28cd72cc09b73f, 0xbe91e850e4c28d2e),
        (0xbb0d842a5a3f172c, 0xbe6e6a49ae425df4),
        (0x3ac77dfa876dbcd5, 0x3e254c4387ef6820),
        (0xba4ce9bca6cbfb11, 0x3dfb084d1fe8eb86),
        (0xba46d58ee64093b4, 0xbdb177e06f295811),
        (0x3a2849b6d182b267, 0xbd8198853f279199),
        (0xb9d57dcf1658bea4, 0x3d350c2b4db30b56),
        (0x39abd4acfcd6f54a, 0x3d0183edbb87358c),
        (0xb95012e0c95b80e9, 0xbcb376d168f148ae),
        (0x390c67518d849338, 0xbc7b8a6525f06ee6),
        (0xb8cd24af389f1494, 0x3c2c8b0bedec138a),
        (0x3894158db0bcbb7a, 0x3bf189a730b0f950),
        (0xb84a2fbef2679eb4, 0xbba1058c9711525e),
        (0xb80049fcdfa04e9b, 0xbb62579aecb57942),
        (0x37bdf8b8eca85db7, 0x3b11906f2331b44b),
    ],
    [
        (0xb5f59a445e01eb06, 0x3950ae6da0547fff),
        (0x3c70b85158068ef5, 0xbfd15f7977a772d4),
        (0x3c1371b46642a67b, 0x3f900f7fcf183e0d),
        (0xbc44d97f9cced450, 0x3fa68b984ec6493c),
        (0x3c042ebf64ae5867, 0xbf648e63600d8418),
        (0xbbeee6fe2679d992, 0xbf60e0d60385a6f0),
        (0x3baeef0b8f617ca9, 0x3f1d796052775aab),
        (0xbba7aac03828df95, 0x3f07800bc550673c),
        (0xbb5a87dc43504bb3, 0xbec3324842f7d51c),
        (0xbb4a6095f3f5ebca, 0xbea30e8cc35f2086),
        (0x3ae3bca6d31bd279, 0x3e5ceda479a13c54),
        (0xbad0ae4da6548e0c, 0x3e34582cb217a0ff),
        (0xba7adb1d4f626b31, 0xbdecade19f5ddb0d),
        (0xba5f5d3513abe02f, 0xbdbedafbed1c0060),
        (0x3a0b34ecc7b06112, 0x3d743c230e74eaef),
        (0x39da767fb508fb46, 0x3d41820d4a511a84),
        (0xb98edf1089e10b5b, 0xbcf56d4abf96e195),
        (0xb93ff26a999ae6a3, 0xbcbee1c9645e9dcf),
        (0xb8e367fc57f02855, 0x3c71b0595f0e1dc1),
        (0xb8d180382e6bf1c1, 0x3c35c9767e2c08b3),
        (0x387cd5296e5b56a9, 0xbbe76fd0e549a473),
        (0xb84bc9a6b0df08ca, 0xbba9294c38de66a6),
        (0xb7e02c9e93087d88, 0x3b595008e255e71b),
        (0x37b5a3d651bbe212, 0x3b188b36ce9fc633),
    ],
    [
        (0xbc5052a3a2541c56, 0xbfcff654544ebcd1),
        (0xb60528028a348c29, 0x396d15232f36db8f),
        (0x3c4052a3a2541b46, 0x3fbff654544ebcd1),
        (0xbc1261ff874dd101, 0xbf70c17ff72afae7),
        (0x3c123a275592e8f3, 0xbf84b0c5d5da6789),
        (0xbbda98843b2d7643, 0x3f394154be70ed46),
        (0x3bd7d4055820b3e5, 0x3f34e12c3067bef4),
        (0x3b83b9b8d3c54cc7, 0xbee9f32fc25ad134),
        (0x3b7dd34ebfa238e0, 0xbed63c5475439cb2),
        (0xbb2e2ce103e7ab56, 0x3e8adbafdf1415b1),
        (0xbb04326873ff6a93, 0x3e6d601d278516f5),
        (0xbaaec485a6d4e24e, 0xbe20eedcab593552),
        (0xba8c720226ba7a69, 0xbdfa84fedb589fe6),
        (0xba416138f5e356cd, 0x3dacf8e343fe22e2),
        (0xba24b409ac564a10, 0x3d81702cfaaee181),
        (0x39c7d31ba4d6f537, 0xbd320271d8cdc0c2),
        (0x39a6d8840aab3531, 0xbd01797e5ea36c33),
        (0xb92966390929c005, 0x3cb10ed3b69b52d3),
        (0x391516ca621c4097, 0x3c7b98f43bd2851c),
        (0x38b9ad988a829a99, 0xbc297cc12a1feee1),
        (0x389015ff3d9a5854, 0xbbf19fe694606f79),
        (0xb83de6f88813ef23, 0x3b9edac0e4d9d135),
        (0x37c618b3f375f1a3, 0x3b627614b4015fcf),
        (0xb797a11ae6442380, 0xbb0fd9f6fd427743),
    ],
    [
        (0x36182629090c00b9, 0xb981570c6a6c1920),
        (0xbc56d72d40e790a3, 0x3fcdc13e66ac2e77),
        (0xbc17a0058564d1fe, 0xbf842ff0cdc58463),
        (0xbc4ea7eea5885fab, 0xbfa38d1dd8992e04),
        (0xbbe0000f328e352e, 0x3f5a55e9b346edc1),
        (0xbbf1ad5da345eb7e, 0x3f5e2e16f97d0a81),
        (0x3bba0bf2e3bea8c6, 0xbf13dfc3782af205),
        (0x3baf57787ba2c01a, 0xbf05ce7f496656d0),
        (0x3b554db180ff9e7b, 0x3ebbb178da9c4ac0),
        (0x3b4faac66e76c3d4, 0x3ea2346d74940be8),
        (0x3afdf2da5b6281cc, 0xbe5612f2e799b732),
        (0x3aa72b6634c3ae5c, 0xbe33d79882e5df9f),
        (0x3a865e4a580e28bc, 0x3de6dbc112bdb275),
        (0x3a5f226492347489, 0x3dbe88c4d898d67e),
        (0xba18523fedb5deb4, 0xbd70ad880fc1dc07),
        (0x39ce811e88c6c354, 0xbd4180d64e2c6148),
        (0xb9932884c5f7b3a4, 0x3cf220941372b04a),
        (0x395486256f555b5a, 0x3cbf18a851090945),
        (0x3906d76d19d83863, 0xbc6e8dc104a832cd),
        (0x38de3ac4d7a1c28e, 0xbc360ce0fa8619a8),
        (0xb8723deefb433289, 0x3be494552b1ee19e),
        (0x38260b2fc3fa5df7, 0x3ba98e84df559883),
        (0x37de2732117b2742, 0xbb56861d006ce503),
        (0xb727a6bd23f6ede0, 0xbb18d0646a86c4e8),
    ],
    [
        (0x3c6c8c66d2e42062, 0x3fcbf3337873a7d8),
        (0xb60f0cbcd3444aa6, 0xb964840aa1183331),
        (0xbc5c8c66d2e41fc8, 0xbfbbf3337873a7d8),
        (0x3bfaea5f363a1a4a, 0x3f66604d91f926ee),
        (0xbc24192692d91168, 0x3f8251858011816b),
        (0xbbdcdb1275879ace, 0xbf314bc11a32c246),
        (0x3bd505487959e0b7, 0xbf32e7decd1f73a5),
        (0xbb847a9b71aaa4b6, 0x3ee293b4c9a24966),
        (0x3b6e3a2461d917f1, 0x3ed4a6704d05ad0b),
        (0x3b124de00e846cdf, 0xbe843ca9b71b6996),
        (0x3afc563d5074b0c6, 0xbe6bddfbf35630ba),
        (0xbabc60e5bca9317e, 0x3e1aae76737061db),
        (0xba871cc918e40357, 0x3df992424643ff62),
        (0xba3975292cf7e7a6, 0xbda7aad4b0989b20),
        (0xba0b3d3cb2ccb405, 0xbd81051e84a7a409),
        (0x39cd31eea0333123, 0x3d2e49ae1a0f5538),
        (0xb9a4374f38b86f7b, 0x3d013585a9ad29c6),
        (0xb943f3b2e472171f, 0xbcad5c6822190eea),
        (0xb91a3dcc41f8b30f, 0xbc7b5bcfa3070c5e),
        (0xb8b8722028e41e05, 0x3c2659e1ea7b0928),
        (0xb89ccb60dab570a5, 0x3bf18ed1ab836080),
        (0xb82cd51c1659b017, 0xbb9b79f386c69ed4),
        (0xb80e73ef8d65d07f, 0xbb6274a94c2329fb),
        (0x37966592010226ae, 0x3b0c90cd312dc7b4),
    ],
    [
        (0x35b57b124ca891a6, 0xb9753e82348b36e7),
        (0x3c32010996eec73f, 0xbfca701d0f967500),
        (0xbc1331fb2bff5f5f, 0x3f7c54b930fef892),
        (0x3c442403558772be, 0x3fa17798aa09f11f),
        (0xbbf91cbfe50364c6, 0xbf52a2151407dd09),
        (0xbbeaf2596c181ca5, 0xbf5b541f829bfb41),
        (0x3b96785653f397a8, 0x3f0cc0bda19ffe34),
        (0x3b7cc73c0281eea2, 0x3f041f3b0662f49e),
        (0x3b50b097e3ffae6e, 0xbeb4b230b114b149),
        (0xbb39f756c517f389, 0xbea1223e2bd0689f),
        (0xbadb05bfc44ceed1, 0x3e511963968a437f),
        (0xbadcfd5e302d5caa, 0x3e32ffb8dedaa0e5),
        (0xba7ba82c85c63aed, 0xbde24c47f1c46f85),
        (0x3a56063e275fb1ee, 0xbdbda527540ac8ab),
        (0x3a042a1749b80056, 0x3d6b75eeb2587187),
        (0x39cf0477925a6771, 0x3d412dcf9e14ca72),
        (0x3965d39c1b93d637, 0xbcee8dc94e4a47ce),
        (0xb94ec20106752176, 0xbcbec6e9eef76922),
        (0xb90db7e15209bac8, 0x3c6a411a8f127544),
        (0x38d4829603e3c8d8, 0x3c35f7179d3c30ad),
        (0x38752bac2edc6ca6, 0xbbe1f8b6ea290920),
        (0xb82284d049652280, 0xbba99593860b4ea4),
        (0xb7dada6a0b2440d5, 0x3b53ef32d1c66177),
        (0xb7baf92eb88ac861, 0x3b18d6a7f6c008d1),
    ],
    [
        (0x3c6e9557ccd1703f, 0xbfc925c6fca08f55),
        (0x35fab5fb9c5380cd, 0x397dd9d8f9235378),
        (0xbc5e9557ccd170e3, 0x3fb925c6fca08f55),
        (0xbbf0bd4a46186c99, 0xbf6049377403bcb4),
        (0xbc1dac1b118906a5, 0xbf809463bbd0367f),
        (0xbbbd1ccb59bc25f7, 0x3f297b354706c53c),
        (0xbbdfefc73f41d900, 0x3f314dd4404e5fcc),
        (0xbb768a017e0a8626, 0xbedbf665d5afcb1e),
        (0xbb5562ef116833bd, 0xbed32cb00ee8c1f3),
        (0x3b1692f6b84439ce, 0x3e7f50fa815b542a),
        (0xbaec0ca08efbeecb, 0x3e6a4339e5f906ee),
        (0x3aa7971bb866b080, 0xbe1541c07333f998),
        (0xba57a16f0ea87b71, 0xbdf86c0bb112e482),
        (0xba36e61435a1cd00, 0x3da35fba37d1f734),
        (0x3a1f20691ce23042, 0x3d80700770cc1156),
        (0xb9b992380621d60f, 0xbd2963b2030905ef),
        (0x3994c48a74316e13, 0xbd00c5877018b95c),
        (0x3934c66edb676868, 0x3ca91e0b94a51ae6),
        (0x38d882f89d4daaf7, 0x3c7adb6f4a466e65),
        (0xb8ce178a661e3146, 0xbc2373e4dad053be),
        (0x388de38db781365d, 0xbbf155ead1bcc6e1),
        (0xb838caf4a70bed71, 0x3b9843ad83497bad),
        (0x380a61496dc04a9e, 0x3b624dcba30b4ca7),
        (0xb7a04207f42b18e2, 0xbb0975abfc72009f),
    ],
    [
        (0x35e086b1188ee76d, 0xb9659deaf309da6c),
        (0x3c4a4f96a2520bb6, 0x3fc8077f56c9b782),
        (0xbbf97f76587fbf04, 0xbf75467eb535deaa),
        (0x3c28681fd3a1f1d9, 0xbf9fd7c3ad6f59e0),
        (0x3be0e0f0ccefc77f, 0x3f4c1b47c809c5eb),
        (0x3bf6295d17ac5a65, 0x3f59166c7d3eaa7c),
        (0xbba4cf4874f894a7, 0xbf05f0152478f5f6),
        (0x3ba861e7bdca105b, 0xbf02aa939fd9fc9b),
        (0xbb5e0f4929f81ac1, 0x3eb0129708ec2410),
        (0x3b439423523353b8, 0x3ea01716dc9f2e1b),
        (0x3ad0e7c543b4566f, 0xbe4b227eef409a9c),
        (0xbadfc64557b91f43, 0xbe32101c49d611c3),
        (0x3a7be06eedb88596, 0x3dddb1e8b7adba58),
        (0x3a037d51b581b88f, 0x3dbc80aab1d93238),
        (0x3a04fb2f6b0c475b, 0xbd66c2b17ab70f1a),
        (0xb9e88a8587ab97a1, 0xbd40ad5f60b5515a),
        (0x398a5248c0478c2e, 0x3ce9d02ffd43b4b6),
        (0xb92666b9cd4e8f2b, 0x3cbe1eda1d7f5a91),
        (0xb8f1b085dd83117e, 0xbc668e7624b57aed),
        (0xb8d7b1a60f9f9ddd, 0xbc35a3dd86c4185f),
        (0xb803b25228593cc6, 0x3bdf559780a6bb0e),
        (0x382586b97e4f1ff9, 0x3ba957aa7a9d5cb7),
        (0xb7ed5614f18b6bf3, 0xbb5198d0c1cebd10),
        (0xb7a8382c4740132b, 0xbb18a72411fe5ccf),
    ],
    [
        (0x3c62da0057f84d3b, 0x3fc70c511227d5aa),
        (0x360a5b28d6094082, 0x397cbdb54d06802e),
        (0xbc52da0057f84c9a, 0xbfb70c511227d5aa),
        (0x3bfe726f7024dd2d, 0x3f5910ebe1f1cbcd),
        (0xbc0e61277df378de, 0x3f7e7dc08e70e99a),
        (0x3bb740aae8c42ea9, 0xbf23bd7d159e09a5),
        (0x3bc9abf1335e8790, 0xbf300357a187375c),
        (0x3b7cab6494ed67eb, 0x3ed5ec73302455b1),
        (0xbb76a67dea84dca0, 0x3ed1e5d2836c8d99),
        (0xbad181196d149707, 0xbe78f41b94dfba1f),
        (0xbafeb3021d837aa6, 0xbe68c11aada79ae8),
        (0xba8510c3ac160a0a, 0x3e1141423452a28d),
        (0xba892fee369915f4, 0x3df73ef6f4772770),
        (0x3a3b488b50613a1c, 0xbda007403efd2ccd),
        (0x39f5bf6b3205778c, 0xbd7f93ab666786fa),
        (0x39cec570d9db2fc3, 0x3d25638d0e0e19b5),
        (0x3992830e7334f62d, 0x3d003d05920debc7),
        (0x3928fcb1f53c1973, 0xbca5832b6cec0d11),
        (0x3905c27fab0794dc, 0xbc7a2e88dfea664c),
        (0xb888ccbe8e25c424, 0x3c20e8847ee36bf7),
        (0xb88b102ba2ef2315, 0x3bf0ff53969fee86),
        (0x380e372a38bcf693, 0xbb955dc67acbb8c1),
        (0xb7f237d40b294b7a, 0xbb62083f7d27b5d9),
        (0x377966f003d521c9, 0x3b06a16c733cdfe2),
    ],
    [
        (0x36043fe2f5037bf7, 0x39714ccaaa6fb0d5),
        (0xbc6444d3d89ac012, 0xbfc62d93aa9d05bb),
        (0xbc1afb8f729bef14, 0x3f70ba9ce88929f2),
        (0xbc2f64ee23828443, 0x3f9d7073daebb02c),
        (0x3bd6ad884b6c9520, 0xbf462813c7f58733),
        (0xbbc1c5da6149803a, 0xbf574a948d05638e),
        (0x3ba7816473a6692a, 0x3f01695765f13bbf),
        (0x3b9157b6b65ca494, 0x3f0170ab5eeb0e4e),
        (0x3b2716b17d28be20, 0xbea9c8e418a20315),
        (0x3b394c2374daaa73, 0xbe9e4d893dce335e),
        (0xbaef0ae11c684392, 0x3e460f73a6ccb16f),
        (0x3ad9ae1edaadae80, 0x3e312782f0f2448a),
        (0x3a51af0d933062c5, 0xbdd881d5842ec66b),
        (0xba56bb5ed05dc669, 0xbdbb4c521a31613b),
        (0xb9fae678b88ed7b7, 0x3d63132da6b9e700),
        (0x39d7ed04f36de95b, 0x3d4019364d866ae6),
        (0xb97cfaa457ac4250, 0xbce5f4ab6157e7be),
        (0x394e31d3157f8b40, 0xbcbd47ca9a1247cb),
        (0xb90d596ff8cf14c9, 0x3c6373719443eb4a),
        (0x38d15089179a9756, 0x3c352a81aa5252ec),
        (0x387db4c0fd37289a, 0xbbdb5bf6bb6be367),
        (0x384ae6b921aecd05, 0xbba8eb04a93528a2),
        (0xb7e3e44a16c09699, 0x3b4f12507c54bef3),
        (0xb7b95db3cb024c97, 0x3b184f56b858ed01),
    ],
    [
        (0x3c6a47ab4241a9f5, 0xbfc5664e13b70622),
        (0xb5f760d6e0df5525, 0xb96a27b55db95023),
        (0xbc5a47ab4241aa99, 0x3fb5664e13b70622),
        (0x3bddc134f1d56911, 0xbf540ee3940b092f),
        (0xbbffe75afd61d48d, 0xbf7c5e1ad9fb2f40),
        (0xbbab8661007349b1, 0x3f1fb8a98ef4a243),
        (0xbbc018bee0f9b515, 0x3f2de9be58a373e2),
        (0xbb75089c8e88074b, 0xbed1bec95415e630),
        (0x3b44800f434e8417, 0xbed0cf264341409e),
        (0x3ac78de205cc4cd6, 0x3e74677d9e214cd4),
        (0x3b022bc30f064717, 0x3e6766cc63507104),
        (0xbaa2f651c24f2d57, 0xbe0c905882b94f65),
        (0x3a9112b75a40b2a8, 0xbdf62157963573da),
        (0xba20399ceae10199, 0x3d9ae2e0cbb67b5c),
        (0x39ff975d6442b115, 0x3d7e456a1c8db625),
        (0xb9a84aef2f5c2b19, 0xbd222e473663eea9),
        (0x399ffbe46660f083, 0xbcff56a09d8cd619),
        (0x393107311db89703, 0x3ca28607cd9d833f),
        (0x391393d18c89b7c0, 0x3c796a2f1da7d934),
        (0x38935c8a666a0b0e, 0xbc1d78e9ab2b3109),
        (0x386087d39d98f4be, 0xbbf096321470c83b),
        (0xb831ea93ce0d9c89, 0x3b92d4333620b1e7),
        (0x37b11cc0e6a17b7a, 0x3b61ad4cac5f830d),
        (0xb7932d42d34590f1, 0xbb041e3c367ed3b2),
    ],
    [
        (0x35fb617f2a7d3e44, 0x39534fd9c53767f3),
        (0xbc6e5d93454f99e3, 0x3fc4b2a2ebf61ece),
        (0x3bf37b02bd5f62b0, 0xbf6b3297fdae7902),
        (0x3c2eb354a631372f, 0xbf9b8105d59b114c),
        (0x3bb9bab613407fc9, 0x3f420a3f8c12a1ff),
        (0xbbf48ec1e2f4d8eb, 0x3f55d18d69de6cfb),
        (0x3b0ad3357483176b, 0xbefc79db4b341e8e),
        (0x3b70c0983bb2f771, 0xbf00679c92c303b2),
        (0xbb4af53c2daf664b, 0x3ea53ac4ecd4be2d),
        (0x3b3bd8bfa73be7aa, 0x3e9ca7507840c04b),
        (0xbadae384e855ecf8, 0xbe4253415f09767a),
        (0x3a835115b77c34b2, 0xbe305141505c9f00),
        (0x3a6c4c5d652bb599, 0x3dd4919940f05334),
        (0xba58b3871fdc4174, 0x3dba20de95780e2b),
        (0xba0a69548bda3e7f, 0xbd602fff3daab710),
        (0x39d417d75702d45d, 0xbd3f01d18581c5a9),
        (0xb9753674cf72aa69, 0x3ce2d78e0d8458c3),
        (0xb9203e6bf52424e4, 0x3cbc5cc76fd7614d),
        (0xb8f39cb8f8a8d557, 0xbc60e0077908e478),
        (0xb8cd5838286ea7a4, 0xbc349c9cfabd7bb7),
        (0x381968d8765f42f8, 0x3bd7fb4989ff942e),
        (0x3844cfbad0863c18, 0x3ba861a8f70eebaa),
        (0x37deeff465dec1db, 0xbb4b7ea330f8d706),
        (0x37bb0df82dbf52ea, 0xbb17dc3e59fbd2b3),
    ],
    [
        (0x3c6316f8ffd294bc, 0x3fc40f90793605bb),
        (0xb5bc63171f8ad7f8, 0xb95cae2e73fe7fb0),
        (0xbc5316f8ffd29417, 0xbfb40f90793605bb),
        (0x3bf7017919c19ee3, 0x3f5085775a5529c9),
        (0xbbe51eb6f0b3a2c8, 0x3f7aa0ce0421d1a8),
        (0x3bb34ca5a9d8eee5, 0xbf1a32a28e4bc82e),
        (0x3b829f8e74c703e7, 0xbf2c26ebca0e46de),
        (0xbb655ef820b8ed31, 0x3ecd7400876206d0),
        (0x3b4cb530be3f4c97, 0x3ecfc1bbf57e3ae2),
        (0xbb1b043cc6acf5d6, 0xbe710c7090487d3e),
        (0x3af031f8178d9744, 0xbe6634db39e4a305),
        (0xbaad7fb323c3df7f, 0x3e0810d7e4efab4a),
        (0xba9ee3d784feb6b0, 0x3df51b513b3c4cef),
        (0x3a2790b6bc250dd4, 0xbd96dd877beb37bf),
        (0xba19b6fbdfe4281c, 0xbd7d06057acf8fa0),
        (0x39a09d743019f4e9, 0x3d1f3b007c5421d0),
        (0xb994cbeda4661eeb, 0x3cfe3418121e09f1),
        (0x394f465df58cfe59, 0xbca0119136910e8f),
        (0xb91cf9a0016f1fe5, 0xbc789daf5f18a5af),
        (0xb8a09684e33e70bf, 0x3c19d09d89e3d4e7),
        (0x389e2b54b001cc6c, 0x3bf023a06c8fea08),
        (0x382b986bb50f84de, 0xbb90a4fdf6b47d26),
        (0xb7fdb60c46ef0002, 0xbb614563ff0bb2b8),
        (0x37a20bcc61b90c31, 0x3b01ebc30518711a),
    ],
    [
        (0x35f3364fd9d34cb1, 0x397482cca2804d57),
        (0x3c5948539688f9cb, 0xbfc37aac8c1aeabb),
        (0x3c0e1b9871576171, 0x3f66ac0d2e2f2f87),
        (0xbc3e2c9f8f0de37f, 0x3f99e74e754ea71f),
        (0x3bc834f162077a01, 0xbf3e1c0589e32bb1),
        (0x3bfe8b15bd171deb, 0xbf5496158dc5f7ff),
        (0x3b66422ba8f591b3, 0x3ef7d55405348ca9),
        (0x3b913888244cb67e, 0x3eff0b30f4506228),
        (0xbb49c57ec3fb6664, 0xbea1d9e3629b98ec),
        (0x3b31cbbb1f5801f4, 0xbe9b35ef28e9de24),
        (0x3ac3a4c0891c58de, 0x3e3f0200e2650870),
        (0xbacbfc59e2173110, 0x3e2f1fb225e849cb),
        (0xba74d0e4dbfc0552, 0xbdd187928a64d03e),
        (0x3a28ff960c8419c5, 0xbdb908ba84482af5),
        (0xb9fd78b0c3058f80, 0x3d5bd0a97427d7c2),
        (0x39b4f3422c873b72, 0x3d3dd98a1888d3c4),
        (0x398676253f0dc67e, 0xbce05368fda435c7),
        (0x3954262b278fb382, 0xbcbb6e7eba338096),
        (0x38f1da524082e61d, 0x3c5d7e5c6064299d),
        (0x38dbb3c7ca9d7c4a, 0x3c34063c1ddae8bd),
        (0x387e4fa2050cb586, 0xbbd521e70bcccf02),
        (0xb83b4d316015135a, 0xbba7c8f868314c63),
        (0xb7a6a57c7316cadd, 0x3b486ba7bc735663),
        (0xb76268b35420b1c6, 0x3b1758996990291d),
    ],
    [
        (0x3c689d1f48185c7e, 0xbfc2f2072e638cf4),
        (0xb5ed38c4f6890952, 0x394461aeca52479d),
        (0xbc589d1f48185d20, 0x3fb2f2072e638cf4),
        (0x3be64a78bbba5f88, 0xbf4bd42b64fc5bed),
        (0x3c082c4cf0185189, 0xbf792bb5e1e159fc),
        (0x3ba0f7052d2216a4, 0x3f161ace3386dfd7),
        (0xbbc911bb7e9afeac, 0x3f2aa8d1cf8db852),
        (0x3b53e6d52ff3d031, 0xbec8ef624c36fc32),
        (0x3b54f2165ec886a8, 0xbece26d3747fe829),
        (0x3b0ba49dcf5f6fd0, 0x3e6d010d2bdb6fa2),
        (0xbb091b80aed15c31, 0x3e65272828ae4057),
        (0xba8eb72fbfdc3a74, 0xbe0497b03c4482e4),
        (0xba9363f2627d714a, 0xbdf42e35495a0b9c),
        (0xba32072f1fd852bb, 0x3d93b2e62efa9f34),
        (0xba1e4f176dd7ecab, 0x3d7bdc7867dad736),
        (0xb990e20c6ed58c22, 0xbd1b1acb454d4a67),
        (0xb99fc3000cfbbd47, 0xbcfd1ce799670eb9),
        (0xb92e5f75bbb7af1c, 0x3c9c1b1fc9d584de),
        (0xb91578e77423719c, 0x3c77d2b95397d836),
        (0xb850bafe09573a49, 0xbc16c0325b1f55bb),
        (0xb870fccd3a2e4892, 0xbbef5c231492c633),
        (0xb81afdc499b9ae20, 0x3b8d8f129e22a179),
        (0xb80d2a72cf226d5a, 0x3b60d71423839ff2),
        (0x37a376b0e7e390a2, 0xbb0004820cb21317),
    ],
    [
        (0xb5fe7034a738c57f, 0xb965d89919797608),
        (0xbc389c717cff1eb2, 0x3fc27407dfadee6d),
        (0x3c01b3998e256aba, 0xbf6346950bfd91f1),
        (0x3c3dd532243e7bdf, 0xbf988d48d1d4eb7b),
        (0x3bcf03d11acc3cef, 0x3f399e6923ada922),
        (0x3bf2c9a985d24428, 0x3f538984b76cdf4a),
        (0x3b96fc0cd00a1e02, 0xbef452194b75f3c7),
        (0xbb929abb89c807a0, 0xbefd855d7b8f0243),
        (0x3b33da403d28aa47, 0x3e9e8a88601ff296),
        (0x3aebbce16fe837d2, 0x3e99f1bd69b16e6b),
        (0x3ac9d47900ce737a, 0xbe3aa483fce054e3),
        (0x3ac5507ea35951eb, 0xbe2dc4c02391c504),
        (0x3a50d4172d38a515, 0x3dce48dc1b4db78b),
        (0xba5adebc7a9a6eae, 0x3db807112e663700),
        (0x39be45131e2d0e10, 0xbd582c0e5f51a78c),
        (0x39d352a7d1eff8f8, 0xbd3cc1592707ba1f),
        (0xb97498b39d0e69d4, 0x3cdc8f29e77a7dbc),
        (0xb95f8164e5aa870e, 0x3cba864ef2011d61),
        (0x38d4d82995e917f3, 0xbc59f854a17b0eb3),
        (0x38c2b397d698e08f, 0xbc336efc952890f2),
        (0xb85e61d1aea7560f, 0x3bd2bbab562fd018),
        (0x3844a8bee87edfd3, 0x3ba72a2cd621b87c),
        (0x37b0634af3b51b0d, 0xbb45ca2b0c668e28),
        (0xb786436234762b46, 0xbb16cc8b2df45666),
    ],
    [
        (0x3c51f9b16832f362, 0x3fc1ff5eec6a01cd),
        (0x3608538a42387313, 0x39703f488735c3d2),
        (0xbc41f9b16832f226, 0xbfb1ff5eec6a01cd),
        (0xbbe5f2ed43d916df, 0x3f47daf64983af9d),
        (0xbc19a4b7b3f004ae, 0x3f77ed5fffc1c774),
        (0xbbafa9d6f36c74e4, 0xbf12f947962314a1),
        (0x3bba27856fb2a3d9, 0xbf296027ea1d6e5c),
        (0xbb585d62c36dd030, 0x3ec57486c67fbc78),
        (0x3b6dbac7c03738c9, 0x3eccc11a59e13739),
        (0x3b07e8342468c4ca, 0xbe690ade515567ae),
        (0x3aff414d51e7ab0e, 0xbe6438a7e22c9734),
        (0xbaac64a5bfb76d87, 0x3e01db6d29a7d049),
        (0xba7a7ab522e094b3, 0x3df3588cd2990098),
        (0xba353be57bfdf484, 0xbd912b3d3f46a156),
        (0xba024df787236d46, 0xbd7aca95934e0225),
        (0x39b4eea5a63388b9, 0x3d17c19e5c2accea),
        (0x399da5ae5c22cafa, 0x3cfc15e96b11e6a8),
        (0xb91064ad2d23f659, 0xbc98c7e06936a2ce),
        (0x391a3907b000a135, 0xbc770ecfa1a29de4),
        (0xb8a29f60e2d525d6, 0x3c142e96cdb6412a),
        (0xb84a16ff92e47bb9, 0x3bee735a654a68e5),
        (0xb8292e46bf7abeaa, 0xbb8a61d3cc01d06e),
        (0x37f75c146bd04a11, 0xbb6066ff09d84644),
        (0x3799d2dbdc6509c3, 0x3afcc034864d659f),
    ],
    [
        (0xb5eb961d9912b9a7, 0xb9682c6f4767341f),
        (0x3c6f5f4b08a76fd4, 0xbfc192f23ce3e051),
        (0x3bdee957b8601e17, 0x3f60a668185c01b1),
        (0x3c39de5fb6906b20, 0x3f9764141d652089),
        (0x3bd0a73bf174fe85, 0xbf3624437a2fe76a),
        (0xbbe29f0dad78a87f, 0xbf52a184be0d9891),
        (0x3b9224756b415cd4, 0x3ef196de0eeef190),
        (0x3b54bbcaa68c5293, 0x3efc317f854112ad),
        (0xbb3c48e804e50119, 0xbe9a8019ef772196),
        (0x3b1fa8a6e09c6db9, 0xbe98d38497beea33),
        (0x3ad0d50082f534e7, 0x3e37318410813eeb),
        (0xbaa0ab33bf6dee66, 0x3e2c8d9d45d76323),
        (0xba5370b8aa12ac53, 0xbdca782c1acc5d79),
        (0x3a50c665c12ff6de, 0xbdb71bbb74f830ec),
        (0x39efa2e983abd5aa, 0x3d5539502b6b020f),
        (0x39c79df656360a67, 0x3d3bbc22cd6efb9f),
        (0xb97de689090afaa9, 0xbcd933a6400afa8f),
        (0xb9538df7643bec5c, 0xbcb9a8f9bcb96916),
        (0xb8f3ea053f8219b5, 0x3c5709ca7b2dbe67),
        (0x38aaa5e94cc9a9ff, 0x3c32db59b4212a49),
        (0x3875dd42a8ae8eb2, 0xbbd0b55ba2bce2af),
        (0x384f96f41944caf5, 0xbba68b427859d5fc),
        (0x37ed698b1a87de93, 0x3b438a193d38c988),
        (0x37becfa942bd3543, 0x3b163dbcc0f4e249),
    ],
    [
        (0x3c6e71c482be67be, 0xbfc12dd57bf18ada),
        (0xb56ade05c8d02fe1, 0xb8f86dd1ee1cd4b8),
        (0xbc5e71c482be685a, 0x3fb12dd57bf18ada),
        (0x3bd11b1ac52c9e63, 0xbf44bebeff7b7f02),
        (0xbc1286f932bc06e0, 0xbf76d9afe88301fa),
        (0x3b859ae05ee9b9e1, 0x3f10842d50687949),
        (0xbbc5025e9c48ddf7, 0x3f2841d86b9b92f4),
        (0x3b45bc8a5bed0a88, 0xbec2b5caad1f2b9a),
        (0xbb3770fdeb5f4fdb, 0xbecb86bad42fc220),
        (0x3ac419e306d0b4f6, 0x3e65e5117a965bcf),
        (0x3b0715f2ca682ea9, 0x3e6364a25cc7309d),
        (0xba8789a8788b4eb8, 0xbdff53dcc9459e76),
        (0xba7a9cb0ee97c39b, 0xbdf297f421bb27b9),
        (0x3a21251ff8e8e5fd, 0x3d8e3c9c7289c6c0),
        (0xba04a1e2968b5e32, 0x3d79cfbae2bdb794),
        (0x39a8e2edbbc6a796, 0xbd1502858f3b96b9),
        (0xb99b52142daa6d4d, 0xbcfb20c42e50d24c),
        (0xb93206530eead3e5, 0x3c9604b999fb92e6),
        (0xb91d665f1b08d083, 0x3c7654b9ed136fab),
        (0x38820031c752db33, 0xbc120524ce30eaa4),
        (0x387abcba8637daf2, 0xbbed91d6a951e05e),
        (0xb811548ed562e5dd, 0x3b87ac7727dd5787),
        (0xb7df152483a30c1d, 0x3b5ff0620f1479de),
        (0x37930d38ec1cefd3, 0xbaf9eb53c9bb31d6),
    ],
    [
        (0xb5b3453ff73e0bdd, 0x393fd7d1a10c14b9),
        (0x3c53f099a5f56db3, 0x3fc0cf3ed059c573),
        (0xbbff3cb9b807f03b, 0xbf5d242aa529931c),
        (0x3beb544962a1902a, 0xbf96613d93b0180b),
        (0x3bde039e228ed279, 0x3f33627f261f5116),
        (0x3bf7fb39963e2efc, 0x3f51d69ca0d88394),
        (0xbb384fc08b8a0bbe, 0xbeeed574afab70fd),
        (0xbb95a4431994f7ed, 0xbefb06384e48ee5b),
        (0xbb387c5180387ee8, 0x3e97452c65235728),
        (0xbaf8807a182da1e9, 0x3e97d51f133b6843),
        (0xbad80ae2bd8213ae, 0xbe346ac67a7e0c7c),
        (0x3ac7244b550b1780, 0xbe2b75d66a8d7fbb),
        (0x3a669cb8e5bcff01, 0x3dc75f5a5db478e9),
        (0xba5afd0a5423da82, 0x3db64531ca4ef465),
        (0xb9fa4a2410291e1f, 0xbd52ceac14a86241),
        (0xb9d685e38c55ad6c, 0xbd3aca497ff207d0),
        (0x39542d374b775a5a, 0x3cd66b96eb099cc5),
        (0xb9597fd736e29864, 0x3cb8d889b2e1f475),
        (0x38e51f4f682fb00a, 0xbc5494fef4e26442),
        (0xb8cc652bba3e525d, 0xbc324dc25f106529),
        (0x380c5c0722c461e5, 0x3bcdfc3c890260b7),
        (0x38473d4459e77144, 0x3ba5efe49281d368),
        (0x37d76e1dde8d0fdc, 0xbb419c5b516958d8),
        (0xb7a6d50e7f01d17c, 0xbb15aff74e510abf),
    ],
    [
        (0x3c61a13e2fee5687, 0x3fc076826cc2c191),
        (0x35fbed7a3ebfda21, 0xb970a6aa5adc8d66),
        (0xbc51a13e2fee55ee, 0xbfb076826cc2c191),
        (0x3be79f62aa4912e7, 0x3f4241b03eaaf5d9),
        (0xbc15dbe9d7239a1e, 0x3f75e7f53001e4b1),
        (0x3ba5f4ab60853145, 0xbf0d17978e2d0336),
        (0xbbb79a682db866ff, 0xbf2745b0df80666a),
        (0x3b63b65170c03680, 0x3ec0803f7f7fe323),
        (0xbb5016bca9a839cb, 0x3eca7006e6ad9cfe),
        (0xbb09b83b761c8dad, 0xbe63590d57d48525),
        (0xbad1e26a947a9c23, 0xbe62a7084b42b890),
        (0x3a87e1ccce8475a5, 0x3dfbc0dd5a22c9a6),
        (0x3a7031ad9111bbe1, 0x3df1e9e4e20477ca),
        (0x3a2e94263a9c2435, 0xbd8ade1de2460f0e),
        (0x3a12379dba208e28, 0xbd78ea3bc5e5158b),
        (0xb9b0319572048e91, 0x3d12bb86a88d447a),
        (0x399493366981bd01, 0x3cfa3d6bcabdd7ff),
        (0xb936ae36a8b153bc, 0xbc93b47615e29aea),
        (0x391e0fe4bfd67f75, 0xbc75a597f7583751),
        (0x38b24cb1a71df570, 0x3c1030bba99bb5cc),
        (0x387b1b9ba6d62368, 0x3becba423ca5f9ce),
        (0x37e762b30a8a5a4c, 0xbb855b595b5871b2),
        (0x37f25ff71c04c351, 0xbb5f1916ebc1bb1a),
        (0xb79220ba6f90a9af, 0x3af7791a566a7a85),
    ],
    [
        (0xb60c6a70e175e648, 0x39735f9916d9837d),
        (0x3c665439df5bb54b, 0xbfc0230b9797a7b3),
        (0x3bd7d914863d8b85, 0x3f59c8083b2b753a),
        (0xbc3c455b2fe15157, 0x3f957d3203befd90),
        (0xbbd3de5214d61226, 0xbf3127cba22892de),
        (0x3bf0c5ff9d68abde, 0xbf51234471455a6c),
        (0xbb8a9c283025b635, 0x3eeb4fe26ec3e489),
        (0xbb9cd17c61f89721, 0x3ef9fc5254f1086c),
        (0xbb3a464f70c3c823, 0xbe94a44c6506c43f),
        (0x3b3f504de9997bea, 0xbe96f17dd184ad59),
        (0x3a90aecf258a5953, 0x3e3225640a6a9328),
        (0xbab85c7b0680994b, 0x3e2a7943505d15ed),
        (0xba6cf77c6abfe37c, 0xbdc4d296ecea7b5b),
        (0xba3e0d08073ac67f, 0xbdb58177059b1ee7),
        (0x39fe64bae160ea42, 0x3d50cd71be541f6c),
        (0xb9de66ddc4c43f8f, 0x3d39eafac486df97),
        (0x397f8cdc2afe705d, 0xbcd417df1a7faedc),
        (0xb9481c4ab05db957, 0xbcb81583d8826b2f),
        (0x38d9d79064997308, 0x3c52822e1fc6fcbb),
        (0x38d8493a5a515ee9, 0x3c31c75f9746b7f8),
        (0xb85d87e0134de425, 0xbbcb0fab39bb8bfe),
        (0xb830a586be8798a7, 0xbba55a2deec91831),
        (0xb7d2c2b65fcec4b3, 0x3b3fe74d126342bb),
        (0xb796cef3b221776c, 0x3b15259d7309762d),
    ],
    [
        (0x3c5d7cc4171715a0, 0xbfbfa8b41711c83a),
        (0xb60ba9b9f91fb8b5, 0xb966589e13c65233),
        (0xbc4d7cc4171716ca, 0x3fafa8b41711c83a),
        (0x3bca510f3e4d422a, 0xbf403a8d0f110fe1),
        (0x3c195ccf34ff059f, 0xbf7511c6dadaaa12),
        (0xbb6fad55d0eb473e, 0x3f09e040fc62c87e),
        (0x3bb45c7340135a79, 0x3f266582f66d8d4c),
        (0xbb2c8a3f0d0d8730, 0xbebd62a18e287536),
        (0x3b667f97c31286c8, 0xbec976fb023f0f79),
        (0x3b0f3da3b757dc00, 0x3e6141188eda6cd5),
        (0x3afa50d6da64db1a, 0x3e61fc77546c2a70),
        (0xba85cd99748ec679, 0xbdf8ccadf7842b28),
        (0x3a8f0a159918c910, 0xbdf14c0515097ba8),
        (0x3a267994041e70e6, 0x3d8810b7fe5b7b00),
        (0x3a19f29d974623dc, 0x3d781814784424c6),
        (0xb98d0230bc946bce, 0xbd10d30b1ed751e4),
        (0xb97b8f89dea8e8b4, 0xbcf96afe8202dbff),
        (0x39286258642b03cd, 0x3c91c038ce19ab04),
        (0x38e5014a0308a5f1, 0x3c750195ff9477a1),
        (0x3877348a77be6ba1, 0xbc0d437455c82ee5),
        (0xb8854f0f0926b218, 0xbbebedd520bd61e5),
        (0x3802b652644462a2, 0x3b835da91ec0fb79),
        (0xb7f3b739a177a033, 0x3b5e4a41b6b72b80),
        (0x376f5eb22e1b1ed0, 0xbaf55a9e0c921524),
    ],
    [
        (0x361a1ce3816c12e9, 0xb970a7cb3e4926ce),
        (0x3c581bdf89b0a8b3, 0x3fbf13faf32c8e0a),
        (0xbbee91046256adc2, 0xbf570558dddb7d46),
        (0xbc326d6d43908d73, 0xbf94b24d7a933972),
        (0xbbb5423c2dd21ed6, 0x3f2ea52a21487a11),
        (0xbbf33353f1356cea, 0x3f50834d8f3fdd5b),
        (0x3b8f2ed187ec8c27, 0xbee86941a4b43bea),
        (0xbb93d844185ba20f, 0xbef90e32cef3e900),
        (0xbb24c412e1beda17, 0x3e92785f6385b273),
        (0xbaf58ee8d31f36a8, 0x3e962482bf9d2bb8),
        (0xbaaf8eeab1cf99c8, 0xbe3043125386ac84),
        (0x3acbaa01c1b653ac, 0xbe29943c303e31c2),
        (0xba66220f22c18b8e, 0x3dc2b2457a1921af),
        (0x3a5711c244de4ccd, 0x3db4ce82afcbf543),
        (0xb9d04d4e5a872347, 0xbd4e3dcde42d3546),
        (0xb9df1485d764cf0a, 0xbd391cdf33ca545f),
        (0x397020766d9e2bc7, 0x3cd220c06f47b251),
        (0x3941d73392b94dde, 0x3cb75f9d03c7dab4),
        (0xb8e752a11f1ff8cf, 0xbc50be9727e162d0),
        (0x38c8623cee5ad9d9, 0xbc31489750bef0f5),
        (0x384f066486611d86, 0x3bc88de9469da922),
        (0x3839422334460fd8, 0x3ba4cb353920bd3f),
        (0xb7d635c8739b1cc3, 0xbb3d0958e2b20819),
        (0x37b805c0dbe5c052, 0xbb14a01af94d0b34),
    ],
    [
        (0x3c0020b4016594a1, 0x3fbe8727daa3daed),
        (0x35e429e6f0925013, 0x3954c091fcd570df),
        (0xbbf020b401657018, 0xbfae8727daa3daed),
        (0xbbae628434cc3c70, 0x3f3d19c52e070d9f),
        (0xbc1361836c55971f, 0x3f74524d4813cc25),
        (0xbb7cb5b9b349156d, 0xbf0735f790b535f3),
        (0xbbcfed1fa3bfcdb7, 0xbf259c8f9f0a3484),
        (0xbb45b8645b0be816, 0x3eba619ffc5a3ad0),
        (0x3b695223039dbc70, 0x3ec896d7dc819faf),
        (0xbaad000c2e45e4dd, 0xbe5f04efbdfeacf4),
        (0x3aff55bb37d4ad1a, 0xbe6162253f3024f4),
        (0x3a774121f038cc79, 0x3df653d736c3ef75),
        (0x3a97cad085bd78e6, 0x3df0bc406f716b3d),
        (0xba1b1362e426bbf6, 0xbd85b5420be0cc8e),
        (0xba06a1118f4cd431, 0xbd77573e801c920a),
        (0x399a08aaec9f5f5a, 0x3d0e6b725d3bea39),
        (0x399f56e3dc7cb7dc, 0x3cf8a83e6e3a037e),
        (0xb932a9c0fdf0215f, 0xbc901633974b8728),
        (0xb90da8cd34f16ed9, 0xbc74685bc5b412ab),
        (0xb89a26475fca50a1, 0x3c0a9719b6c2c317),
        (0x3884ebc520e017d9, 0x3beb2ce5154261a5),
        (0x381bbed6afb225d9, 0xbb81a55f76501009),
        (0x37e01b88fc0ca8c4, 0xbb5d84f510d0e5db),
        (0xb77f816f7a76ffd2, 0x3af38293e0fae553),
    ],
    [
        (0x35f7b4ea1935a568, 0xb956dc89f8b0297a),
        (0xbc5a45a53b37a59e, 0xbfbe018d99f5da1b),
        (0x3be001e9c4dcec6c, 0x3f54b85897b36265),
        (0xbc32db4ef382192e, 0x3f93fc442153435d),
        (0xbbcf201dea1f5a71, 0xbf2b9694d71486e3),
        (0xbbef8e2381bad736, 0xbf4fe6fdc644ddde),
        (0x3b87d6d0d3615a87, 0x3ee5fd096e4523fb),
        (0xbb6b106322d36ddc, 0x3ef83770c9a84498),
        (0x3b3a40e2b2746cab, 0xbe90a6f6f7e05f1b),
        (0xbb3f771adc91215c, 0xbe956ad4a35eb0ad),
        (0x3ab3b86f94e0c858, 0x3e2d5bea474ca54f),
        (0xbac5585b134469a5, 0x3e28c39f68d21b3c),
        (0xba690d5e5a6b3a35, 0xbdc0e73408ce7d83),
        (0x3a327165e3e97d3a, 0xbdb42a6dedaa1a20),
        (0xb9dd142a729ed644, 0x3d4b640f953ca9d2),
        (0x39d7f71ecb3fc409, 0x3d385e74fde5cbdb),
        (0xb97b8a515f767ea6, 0xbcd07409455800ac),
        (0xb95ec4cbe35c9396, 0xbcb6b62242767fd2),
        (0xb8e317bb80a9622e, 0x3c4e76f4a5aa2684),
        (0x38b554cbd013807d, 0x3c30d15ee339d044),
        (0xb8101586ff6d60a8, 0xbbc664451d8e4e4f),
        (0xb849c3a261353029, 0xbba4436d1959e4e5),
        (0x37b47984bdada57e, 0x3b3a8bfe9a4ae941),
        (0x37b807503fdd6126, 0x3b1420356bcc7e9f),
    ],
    [
        (0xbc5cb1f28997ca38, 0xbfbd8293aa55d18f),
        (0xb61515229b393b76, 0xb9729d22385eacde),
        (0x3c4cb1f28997c91b, 0x3fad8293aa55d18f),
        (0x3bb680f417a90d4b, 0xbf3a48fe4afedcc8),
        (0xbc16c091c5e058be, 0xbf73a5ccbc12a67b),
        (0xbba33c90d0f9c0c9, 0x3f04f91e41eee9bc),
        (0xbb9b30874f0a2926, 0x3f24e72224db2c0e),
        (0xbb169858252daef6, 0xbeb7dac8202ad4fb),
        (0x3b63726cbb3fc903, 0xbec7cbd49c315be0),
        (0x3aee4734322f57a4, 0x3e5c1396b62b0f84),
        (0x3af3c6385c4eb2c7, 0x3e60d5c64a9c427f),
        (0x3a70141b0643212e, 0xbdf43c4a5d5a74a1),
        (0xba8fd98f49242c47, 0xbdf038cb3f5e3248),
        (0xba158fe9d08d69ff, 0x3d83b473da8d9cc0),
        (0xba029334408e97a4, 0x3d76a5d59f9a7a17),
        (0x39ad9590d69212ea, 0xbd0ba9a1977eefff),
        (0xb971f0fe79a4cdf4, 0xbcf7f3d211c66187),
        (0x391950769136ef0f, 0x3c8d51075a5e7bbb),
        (0xb916720d8484ea73, 0x3c73d94e734e8879),
        (0xb892e80fae65fb53, 0xbc0848485fe35579),
        (0x38777f0c4d7b3dfa, 0xbbea774325d98b75),
        (0x382276ef09c2242b, 0x3b8026f3af966377),
        (0xb7f67847065ac7b4, 0x3b5cc9941483fe56),
        (0x377a99585b78b747, 0xbaf1e6750f0747ba),
    ],
    [
        (0xb6025c847eb79bd4, 0xb964158ebf13126e),
        (0x3c5f215e77086bf5, 0x3fbd09b210b30217),
        (0x3bfc1f6fd559ba97, 0xbf52c74f6d120291),
        (0x3c0587bb3b3049ff, 0xbf9357bfc2be5860),
        (0xbbca26f6fb3da340, 0x3f2901e4c495acea),
        (0x3bdfa22468c1def8, 0x3f4ee2a36979f905),
        (0xbb819f063106cdcb, 0xbee3f0cb93a497a4),
        (0x3b845c631b76c8a1, 0xbef7748921871bcc),
        (0x3af9b751dac720d1, 0x3e8e39a085fc522a),
        (0x3b168e67fb0cb203, 0x3e94c1b7a6b2509f),
        (0xbaca52721df6c6b6, 0xbe2aabe92ddd7d73),
        (0x3ac84b97b1182cc0, 0xbe2804c5ad3234f3),
        (0xba54edd49d1edf2d, 0x3dbec02b4af63baf),
        (0xba503f24cfdd22c9, 0x3db3938387c8ea5e),
        (0xb9ee5637ec8cd309, 0xbd48f3df4c86043e),
        (0xb97c878aca76d363, 0xbd37ae3cd67c3185),
        (0x3953bf1f530af427, 0x3cce075e0947f84e),
        (0x394a5c410c455b26, 0x3cb61833a4ed09fb),
        (0xb8d6421dd6abfed8, 0xbc4bda71bc7ca99f),
        (0x38d6cb9f7285a2f3, 0xbc30616da73cfd68),
        (0xb860f21809293998, 0x3bc483adcba31a64),
        (0xb819a874856663c4, 0x3ba3c2e3aaaa861b),
        (0x37d51466fd699b53, 0xbb385faa94578a21),
        (0x379f3946f24c4995, 0xbb13a642c5cb638d),
    ],
    [
        (0xbc49df1f0f8d2108, 0x3fbc96700bf039e2),
        (0xb5aa78cc35dda805, 0x393def6cb4dcf261),
        (0x3c39df1f0f8d2339, 0xbfac96700bf039e2),
        (0xbbb8cb9fe6e3ff6e, 0x3f37e5647d30fea8),
        (0x3c063977044d6f5c, 0x3f73095734a24496),
        (0xbb56346fb6ed118c, 0xbf0312a4db537d5b),
        (0x3bc2ab35d69fc6e2, 0xbf24424a96e62373),
        (0xbb5dc2fb5a697d64, 0x3eb5b4a6639fb7be),
        (0xbb533dbdb81ff3d8, 0x3ec712e4d44c4a74),
        (0xba8ccfbfe857180a, 0xbe59917dedf003d7),
        (0x3b0b894b3c8075c7, 0xbe6055757b098917),
        (0x3a95a6cc3450560a, 0x3df2728cdc02e18c),
        (0xba805244323472ea, 0x3def80393fe2fc7c),
        (0x3a2d37e6a4cca504, 0xbd81fcaa37a8b3b3),
        (0xba1bcda846633c8d, 0xbd760225681cce6f),
        (0x39a7e3fa90942e91, 0x3d0949ef9ee67213),
        (0xb995a2c090d0fa89, 0x3cf74c654a150fde),
        (0xb912f420e2959c3d, 0xbc8ad88e52941a98),
        (0x39086b9adf654b32, 0xbc7353b6bf507102),
        (0x38a54a053e9cfe0f, 0x3c0646e8a6aa6dbe),
        (0x38174c918b573c6f, 0x3be9cc769d138c5f),
        (0xb80b75e9f287b65e, 0xbb7db1eb4b8463d9),
        (0x37f46072c5cf6b31, 0xbb5c181474bdc517),
        (0x379310eb69ece94f, 0x3af07ca4c538ae49),
    ],
    [
        (0xb6025d7555eb3333, 0xb96e6cb57bbffe78),
        (0xbc37a2663626dca6, 0xbfbc28612a3bc18b),
        (0xbbf422f2d193783c, 0x3f511f52577ff6ba),
        (0x3c3976e0041aa223, 0x3f92c21da135f56c),
        (0xbbc90eddb53bb0bc, 0xbf26ce18f8229e00),
        (0xbbed07e9f5028d13, 0xbf4df586d8b786e1),
        (0xbb8f298c930f3daa, 0x3ee230fede9c5ad4),
        (0xbb94e2ce32d56eeb, 0x3ef6c2a7558fc928),
        (0xbb04521d6df510c8, 0xbe8b97329e667f58),
        (0x3b301d874878e4d4, 0xbe9426ecfd66cd08),
        (0xbab8e3f15d7488a6, 0x3e285e012a388a8c),
        (0x3aca2aaf6142964c, 0x3e27557143798fcd),
        (0xba467e3999dd049e, 0xbdbc1f9f6fa068b6),
        (0x3a5bd65c11af362e, 0xbdb30842b25be60a),
        (0x39cac535c84bd033, 0x3d46d975726392d2),
        (0xb9d9799d506b6536, 0x3d370aced616e29d),
        (0xb960a1d7a205223b, 0xbccb8986f84a30fa),
        (0xb8fca075da3aafd8, 0xbcb584e49cb6db11),
        (0x38e10f7fdc1a5917, 0x3c4995407ad5e053),
        (0xb8cee1114e53e0aa, 0x3c2ff0b6f99d000a),
        (0x38430648734f6eb9, 0xbbc2dffde43760ea),
        (0x3835046e458663d2, 0xbba3496b757a3add),
        (0xb7dc717d402f56b2, 0x3b36778e7fdb6444),
        (0xb7b8d1b6dcf93aee, 0x3b13324afffd82fa),
    ],
    [
        (0x3c58fff4515190b5, 0xbfbbbf246914235f),
        (0x35d982a917cde59c, 0x3950d75272d68d70),
        (0xbc48fff4515191c8, 0x3fabbf246914235f),
        (0x3b9cd1da148e225e, 0xbf35d923e8470178),
        (0xbbfbef9e89616fef, 0xbf727a96f174b6d1),
        (0xbb885ae82b0f986d, 0x3f01715e4bbb00e7),
        (0xbbcb5280334e8030, 0x3f23abacdb5106b5),
        (0xbb5ee327f9eba23d, 0xbeb3dc30d27849d7),
        (0x3b6dfe7950a9c090, 0xbec6698d6ee99eb9),
        (0x3aeb757878040b38, 0x3e576911a4642dff),
        (0xbaee8dc651ba599a, 0x3e5fbf415682210b),
        (0x3a81620ce1ff0741, 0xbdf0e7d3674631fb),
        (0xba8353972b806230, 0xbdeea1cbf2d9d28a),
        (0xba25d1ac6266a88d, 0x3d80805ad6510660),
        (0xba069055b8b763ed, 0x3d756aaba5459ef1),
        (0xb9a9f31ddbf5fa91, 0xbd073ab529b3b8ec),
        (0x398c43d0fb4ec9a1, 0xbcf6b0b8fe628d9e),
        (0xb8c99b2526d558a3, 0x3c88b1cfe95984a1),
        (0x38c9b40ec3d6b48d, 0x3c72d6d66d90227d),
        (0xb8a48afcea90991a, 0xbc04861250a2137e),
        (0xb88e608ec89832fa, 0xbbe92be16535de20),
        (0xb805a233b6080776, 0x3b7b6761d996b09c),
        (0xb7f3fb75ac66c54a, 0x3b5b702a10c5d8d6),
        (0x3787fbc0b82d70c7, 0xbaee7b19f87b38e8),
    ],
    [
        (0x35ff73655c8feec1, 0x39576c36a9ae2d5c),
        (0x3c5d0edcbac85111, 0x3fbb5a6219b35e14),
        (0x3be54c0ba940867f, 0xbf4f645fdb1a8c89),
        (0x3c37a2e3fd651672, 0xbf923940d01de8e9),
        (0x3bb0148b185376f2, 0x3f24e86a1e6384ff),
        (0xbbe17bbc96492f6b, 0x3f4d1c6a18c7ed95),
        (0xbb8c8ebdf6811341, 0xbee0aeec61621923),
        (0xbb99391cbc0f757c, 0xbef61f7d30244338),
        (0xbaa9ea01a9b8fa8c, 0x3e8950f781c1c41c),
        (0x3b3c5364d9ae9a38, 0x3e93989942e07175),
        (0x3acc3081ebda4ff1, 0xbe265fd114ee2251),
        (0xbab6323d2dc56e9e, 0xbe26b3bd4dfd4b81),
        (0x3a5ff68cf4ae06aa, 0x3db9d8a1bbe00791),
        (0xba462dc328f651f0, 0x3db2875c16e4487f),
        (0x39a13905c77f8bc2, 0xbd4505775bc7e4de),
        (0x39ca895460eb8613, 0xbd3672e2d4f29271),
        (0x39541d93187b0343, 0x3cc95d609db7ab8a),
        (0xb94983f2e6b2775d, 0x3cb4fb4d36ab3e60),
        (0xb8c9c87d6b8b88ce, 0xbc4798ba8aedf4d9),
        (0xb8c9a4efd28a4426, 0xbc2f2b65d0eb351a),
        (0x3864f45927d43933, 0x3bc16f636e45b56a),
        (0xb84852ca897a91ce, 0x3ba2d6b5ad521072),
        (0xb7a77d76082f0cb2, 0xbb34c92434bd9c18),
        (0xb7baea41b7cf079e, 0xbb12c43424dc0933),
    ],
    [
        (0xbc5024304247ada3, 0x3fbaf9cb49c4f935),
        (0xb606a30b444ceba4, 0xb9708319890a5e6d),
        (0x3c4024304247aeb1, 0xbfaaf9cb49c4f935),
        (0xbbdc12274808197b, 0x3f3413b75ce0cc1b),
        (0x3c06b1ae60010808, 0x3f71f7a8fec6eba8),
        (0xbbac4403af34826f, 0xbf0008442739ebfc),
        (0xbbc9d6333d2b3d56, 0xbf23215dab7537c6),
        (0xbb3addf7c6694661, 0x3eb242e9e6bab199),
        (0xbb4ca73ddba08b13, 0x3ec5cdc48f5d75eb),
        (0x3afbb2f24513857c, 0xbe5589cf32f43f3a),
        (0xbafb14e3e9852288, 0xbe5ee5f0d63c1125),
        (0x3a8b9bbc9f549992, 0x3def216eedb0f95e),
        (0x3a7b918a152ee6ef, 0x3dedd414a7d8c97b),
        (0xba0a1e2e293b285d, 0xbd7e69fdff8447cd),
        (0xba130096e56c8b4c, 0xbd74de15be68829d),
        (0x39845da07a4cf814, 0x3d056e1418ae157c),
        (0xb9994fbaa41d7396, 0x3cf61fa98590ddb6),
        (0x392ed1bc75d499ea, 0xbc86cf2506f44167),
        (0x391952212e39899e, 0xbc7261f3fbe2abde),
        (0x38af858de9bd8291, 0x3c02fb5ec2e4f362),
        (0xb884c724ba966327, 0x3be894d61148c89d),
        (0x3814ecb4513b97db, 0xbb7961b08a806964),
        (0x37f74e1e809c4297, 0xbb5ad164c25d22e1),
        (0x378b5dec7edc635c, 0x3aec45b6d88eaf60),
    ],
    [
        (0x35fc420ae9aceac9, 0xb9652ed980617fa4),
        (0x3c508b7cc7933a76, 0xbfba9d1835947d70),
        (0xbbcfb3814e43f8ba, 0x3f4cea253049a85b),
        (0x3c3e993cf753c70d, 0x3f91bb71f665dcdf),
        (0x3bc6de474ee9719c, 0xbf23427f4797ae90),
        (0x3bab43046c68dd4b, 0xbf4c54a7bd6f30c1),
        (0xbb74fee51586a24b, 0x3edebe9e65809919),
        (0xbb914661f197da7a, 0x3ef58924fa089950),
        (0xbb1b1c449258996b, 0xbe87564501e7bcdc),
        (0xbb32d1ffd0fc3508, 0xbe9315306e5bb42b),
        (0x3ab9cf483ad9233b, 0x3e24a32382dea2a5),
        (0xbacfa4096d44c807, 0x3e261e0ee5367a22),
        (0x3a4c5e7f996c4e59, 0xbdb7db853758262e),
        (0x3a56a3ad86030da8, 0xbdb20facb4a9bb8a),
        (0xb9e5ff48c852945f, 0x3d436bd6f000149c),
        (0x39d6c37ca2f2d752, 0x3d35e552618cbb0c),
        (0x396867fd75ecd168, 0xbcc7754c8d2d22bc),
        (0xb9220db8a9e7d56d, 0xbcb47a92b936acfa),
        (0xb8edde52032032d2, 0x3c45d9205d867e02),
        (0x38c44f9ceec60e46, 0x3c2e71f7334b0fac),
        (0x3864418502fa183f, 0xbbc029e4f35c0aa8),
        (0xb84306e130ba75b2, 0xbba26a627f026785),
        (0xb79943e923428172, 0x3b334bc2432e2477),
        (0x37b67d0793319e79, 0x3b125bc6a2301ad4),
    ],
    [
        (0xbc55d35a88f1e0a3, 0xbfba4407e04298d1),
        (0x35c8ff9720aff2d4, 0xb952ea1cdd28e494),
        (0x3c45d35a88f1df9c, 0x3faa4407e04298d1),
        (0xbbdfcf49ead42a08, 0xbf3288694b34ab21),
        (0x3c1c3cb8ccc5cfb9, 0xbf717f0266db2149),
        (0x3b7370def0b046b9, 0x3efd9a9a1d05433b),
        (0xbbc76ae4901c1264, 0x3f22a1c916a2d5a1),
        (0xbb55855d21f5ed77, 0xbeb0dd9e92661e07),
        (0xbb0d68a20388319a, 0xbec53dd972d8f232),
        (0x3af0f257afd1ac40, 0x3e53e6e553aac740),
        (0xbadedfaa1b67f332, 0x3e5e1cc65033998a),
        (0xba7a37ed01b9dc68, 0xbdecc8a3f36b4e82),
        (0xba7e73855d5a6fce, 0xbded152ceac33738),
        (0x3a0403bc50b56bec, 0x3d7c249e1f30a046),
        (0x3a07ab8d362d1a04, 0x3d745b3c1804b70e),
        (0x396e1c149d462be8, 0xbd03d90bb02406e2),
        (0xb973199796754cc5, 0xbcf598302aa37892),
        (0xb916ac374a93ff56, 0x3c85259cd94ec2fe),
        (0x3902b130fb62197d, 0x3c71f460935c9412),
        (0x38a7f1bf80e44c1a, 0xbc019e603f7b6fda),
        (0x386260160040e9ca, 0xbbe806a4e2d693da),
        (0xb81630b9badcde1d, 0x3b77968008016bca),
        (0x37d07b22ab7a0d45, 0x3b5a3b41f167ae95),
        (0xb76d821cbcda7119, 0xbaea4e94d614d85d),
    ],
    [
        (0xb594a4d849ef503c, 0x3938d6c331cb03ad),
        (0x3c2748c82bad8c50, 0x3fb9ee5ee937fc89),
        (0x3bebaa9c093c243c, 0xbf4abf28ad5bf6da),
        (0x3bf098a110e15588, 0xbf9147481084ae0f),
        (0xbbcb58642b3fc6f6, 0x3f21d137345cfee5),
        (0x3bdbab7a25ea6636, 0x3f4b9c10ddf62464),
        (0x3b707cd24e8374d7, 0xbedc72c9d49c6d7a),
        (0x3b9799bc070c59b2, 0xbef4fe0b363ccc66),
        (0x3b2f6fb041c061d5, 0x3e859a1c336b7506),
        (0xbb38437a08d58d5c, 0x3e929b6632cc324d),
        (0xba9f406b3224c41e, 0xbe231cd6e7991df9),
        (0xbab2457dd0d0e718, 0xbe25930872001f69),
        (0x3a548f54ac84016e, 0x3db61befbaaf9fff),
        (0xba589a9303d7d25f, 0x3db1a038446ec46b),
        (0xb9db6b1cfe9b31f4, 0xbd42030319ffb5eb),
        (0x39bf4002924f6ae5, 0xbd356117967009ca),
        (0x394da21091709c76, 0x3cc5c664f861ad05),
        (0xb93ec733758fe60c, 0x3cb401eb6a584504),
        (0xb8c585bd91ee5a94, 0xbc444cf483677683),
        (0x38b16263dda87c68, 0xbc2dc380ac38062e),
        (0xb85553427dd612fd, 0x3bbe11fffb02da4f),
        (0xb8433d91b3bab077, 0x3ba2040b40b2538d),
        (0xb7df7b4517c493ac, 0xbb31f841ca6c686b),
        (0xb7b3136588d9a93e, 0xbb11f8bc69e97ebb),
    ],
    [
        (0x3c5728ab934a269f, 0x3fb99be744018c90),
        (0xb60a99831efba67e, 0xb968c3fcaae6f521),
        (0xbc4728ab934a259d, 0xbfa99be744018c90),
        (0xbbc84fc914744b2d, 0x3f312d4e1c1ca4c3),
        (0x3c0f281d89c5c387, 0x3f710f5ca51f98b0),
        (0x3b9fa3d9c31f9653, 0xbefb71417476a570),
        (0xbb82784b5921b117, 0xbf222b9fa8dfd762),
        (0xbb2378a1f5ba83c5, 0x3eaf471061b7dd7e),
        (0x3b688d2d785bb65e, 0x3ec4b862279de756),
        (0x3ae3ff1d8fa9486f, 0xbe52763f34a53566),
        (0xbafc90858b5fda2a, 0xbe5d61e605e8e69a),
        (0xba8db88bf0b8b3a0, 0x3deab7331a682503),
        (0x3a83becc2585c0e6, 0x3dec637428f3fda0),
        (0xb9f91cdb1c054c59, 0xbd7a236ad0b3eba5),
        (0x3a130d91a2571fc8, 0xbd73e11cef6ea796),
        (0x39ab3e18bb4b1cc4, 0x3d0272cbaef4a6ce),
        (0xb992fbf6044183a9, 0x3cf519623b3e6ff3),
        (0x392d2f84f2aa9293, 0xbc83ac610a98a111),
        (0x390f58be5a588034, 0xbc718d7aab4e57b8),
        (0xb89f4ff0d5411205, 0x3c006837647ec51a),
        (0xb880cf6db8b6e7c8, 0x3be780a339ad4fe5),
        (0xb80c4294a291cdbe, 0xbb75fd4c1cf239b2),
        (0x37a40abc824fc196, 0xbb59ad37b5365c90),
        (0xb78678af8405295a, 0x3ae88d6e7fc415df),
    ],
    [
        (0x35f8589681278026, 0x39644f6712268762),
        (0xbc4e31356cad466e, 0xbfb94c6f54aef04b),
        (0xbbe68528f64c7b1b, 0x3f48d6371f018ef0),
        (0xbbf705a06948b19e, 0x3f90db975fd7dc47),
        (0x3bc06e812fcafb21, 0xbf208bd1634353c9),
        (0x3beeb410edf2c87a, 0xbf4af0d3d4cd140d),
        (0x3b6d81ec11b0f0a1, 0x3eda6c8e7c9d023e),
        (0xbb94f5f0c72b68d7, 0x3ef47cddcdac15d7),
        (0xbaee0517040c548e, 0xbe8412394fc50e0a),
        (0xbb3fb62ad27acc53, 0xbe922a21cb1edfe9),
        (0xbac19c23bc37cb83, 0x3e21c417dedcf50f),
        (0xbaae59d2170adaa5, 0x3e25117f3061daaa),
        (0xba5e45da99f1d381, 0xbdb49009a6c8ef21),
        (0x3a587572c3e6ea79, 0xbdb13823dda87c56),
        (0xb9e347d6a1129f4a, 0x3d40c350b604f8b8),
        (0xb9cc2ad178b32695, 0x3d34e54a89ae35f1),
        (0xb96a80e6a3444766, 0xbcc447dde867e136),
        (0xb958d6073cf4ba2f, 0xbcb3909fa79d8804),
        (0xb8d64b27142de74c, 0x3c42ec7e360503ed),
        (0x38c6d01b41c0980d, 0x3c2d1f243bdfcc89),
        (0xb85a84d31e1eccc9, 0xbbbc0eb8ec6b686f),
        (0x383387e43fd2ce16, 0xbba1a3489d347b47),
        (0x37dde5e850e9f781, 0x3b30c8b4a21c4541),
        (0x379dea6fe0f54701, 0x3b119ac8a8b69886),
    ],
    [
        (0x3c5e213a1a4b3671, 0xbfb8ffc9bd24fe08),
        (0xb6087102bb621f14, 0xb968aedf9eaeb722),
        (0xbc4e213a1a4b376e, 0x3fa8ffc9bd24fe08),
        (0xbbcd363274122e3f, 0xbf2ff51b38eef42c),
        (0xbc0f532ddb1fa0a0, 0xbf70a7a725d3fbc4),
        (0xbb834bf08647ec89, 0x3ef988128728122a),
        (0xbbb5bd6096bda7b8, 0x3f21bdc846a09e43),
        (0x3b30137f6ede612f, 0xbead1b60a4812ac4),
        (0xbb5a77139f0b51d5, 0xbec43c2d8e698c10),
        (0x3ad810635b7d0ab5, 0x3e512fd6ccdafecc),
        (0xbaf69212c4f321ef, 0x3e5cb3bb8802e835),
        (0x3a8f78dd26a56680, 0xbde8e1dd09a92325),
        (0x3a8e97521eebd870, 0xbdebbd84404b6233),
        (0x3a0668a211ed0544, 0x3d785bcfd7db3e06),
        (0xba1ac0b0871b3637, 0x3d736ed7607396ae),
        (0xb955951b23d30299, 0xbd01343464758a13),
        (0xb9870af9e1cfe573, 0xbcf4a26ef2120ad4),
        (0x392f6ff8929d9ec8, 0x3c825c400013c92d),
        (0xb91624b068d7998e, 0x3c712caecd60a253),
        (0xb89eef1d95657c36, 0xbbfea68103e8017a),
        (0xb85dbca3eb7791dd, 0xbbe7022f94b51f88),
        (0x381743e0f343b5ae, 0x3b748f053210dc4e),
        (0xb7fa879e893fa8e0, 0x3b5926bc9b441cf3),
        (0x378b8478cb5f337e, 0xbae6f9f3f5bcbb92),
    ],
    [
        (0x3601868fee3dfc1f, 0xb967b581e11caa4e),
        (0x3c35e8738ef1b9cc, 0x3fb8b5ccad12d632),
        (0xbbd18507f867652e, 0xbf4724d0185980d2),
        (0x3c1f6ab95e49bea7, 0xbf907764ae2b1eeb),
        (0x3bbea971beb0d4c5, 0x3f1ed6acc18fdf29),
        (0x3bca7600bd6cd00f, 0x3f4a51693df2efbb),
        (0x3b5424c646ef6906, 0xbed8a0ec61e4df7b),
        (0xbb2fcebd0b42c3a3, 0xbef4047f31fae045),
        (0xbb244d840b8d48b4, 0x3e82b668af276950),
        (0x3b3cad3ad53887a6, 0x3e91c07467ed6403),
        (0xbac9698769c60aee, 0xbe2091d32eea7f0f),
        (0xbacf39eabd09fce6, 0xbe24987294de7d1e),
        (0xba4e5986bfe40615, 0x3db32fe6ab220650),
        (0x3a4ecd045a4261a7, 0x3db0d6b127f8ac20),
        (0x39d762ccc3d86511, 0xbd3f4d16b8caee52),
        (0xb9d08617d1af539c, 0xbd34711e3d50f4ef),
        (0xb96508919fc82bc0, 0x3cc2f28e5dd490f1),
        (0x395bb058a146ba42, 0x3cb326098ea185f3),
        (0xb8e94f0af2b5ad80, 0xbc41b16976a3d442),
        (0x38c176e23d72ae5d, 0xbc2c84134caa87ba),
        (0x385559d5c6add4b3, 0x3bba412149720827),
        (0xb840ccac5bb143cd, 0x3ba147b6307ccbc3),
        (0xb7b2120a01259b99, 0xbb2f70570b14782d),
        (0x37b4facf0ef3cce3, 0xbb11419a7914e3d3),
    ],
    [
        (0x3c5b1c9821974148, 0x3fb86e51be0a9153),
        (0xb5f75cf6d696fc83, 0xb95827c8ffd9b363),
        (0xbc4b1c982197404f, 0xbfa86e51be0a9153),
        (0xbbcc79e50a2cd67c, 0x3f2dd3c244b53279),
        (0xbbe9cfc13650412e, 0x3f7046fc5a218a86),
        (0x3b473390e469f8a4, 0xbef7d51accbfba76),
        (0x3bbac27d2877fb16, 0xbf2157556c0e1b1a),
        (0xbb2e687ae53690de, 0x3eab2d01c12c8227),
        (0xbb685c4f9e1e413b, 0x3ec3c838897d0a1e),
        (0xbaf6af7b27e224bc, 0xbe500d37aa536a8b),
        (0x3afe75b9f5dba876, 0xbe5c10ecf088eac6),
        (0x3a7e762d28ae6967, 0x3de73f8d1af57f4c),
        (0xba76c817fc84ee3f, 0x3deb2227fe940228),
        (0x3a0d6d1da90d883a, 0xbd76c5390bcf01cc),
        (0x39fafe0f0f7cf1bb, 0xbd7303a6ded9e74a),
        (0x39a769658d5dfcab, 0x3d0017772c8c67d0),
        (0x39816dc9a19e88df, 0x3cf4329cf31c85bc),
        (0xb9187bba5059be09, 0xbc812f535bf22ca0),
        (0x39028bb4a0afb991, 0xbc70d1775336e3dd),
        (0x388f6ab178fe4eee, 0x3bfcb5a6e861125e),
        (0xb8764727329b2f6d, 0x3be68ab377a5b3ae),
        (0x380ffcd7837774b1, 0xbb7345ce122d1040),
        (0x37f98513cc233bda, 0xbb58a74b58fe9d86),
        (0x37810ff858a703fb, 0x3ae58ea451ed4ee5),
    ],
    [
        (0x3609fbc30a1ed1d3, 0xb9625183186cb588),
        (0x3c40ee4d5b1b8051, 0xbfb829356999a097),
        (0x3be3831592c6b0a5, 0x3f45a280e033eb59),
        (0x3c215c7646576687, 0x3f9019dba8336e48),
        (0x3b986e85a68d172a, 0xbf1cd4559d95e619),
        (0xbbc0173a940c218c, 0xbf49bc857750335e),
        (0xbb63dae6bf1f89ae, 0x3ed70706580693f9),
        (0x3b8384969ed36c54, 0x3ef393fc6cf8481f),
        (0xbaf56ba053870856, 0xbe81800af4665cb3),
        (0xbb2c21dbefaffb7f, 0xbe915d91a2289a30),
        (0x3a9c9f08edce59cf, 0x3e1f009c83572407),
        (0x3ab89826637a4616, 0x3e2427054565b344),
        (0xba34b49867a6db4f, 0xbdb1f5167f464cc1),
        (0x3a578760251f40d0, 0xbdb07b3a2cb21728),
        (0xb9d9c43afa0fe4e6, 0x3d3d4f453474c223),
        (0x39d2dacef909b870, 0x3d3403dd864cdceb),
        (0x394afae738fa0ade, 0xbcc1c09693482adf),
        (0xb955a0b7487878c3, 0xbcb2c193ec7672bc),
        (0xb8e991d90439ccb6, 0x3c40967d711ba719),
        (0xb8bd41e60b473ee2, 0x3c2bf18fb3b9d25e),
        (0x38559b44bb41d0b5, 0xbbb8a1e44dd80d7b),
        (0xb8300b6984de9d94, 0xbba0f0f498e20346),
        (0xb7be574e735a8cb7, 0x3b2d850a1c84b4b5),
        (0xb791e712d7e8c434, 0x3b10ece621e460f6),
    ],
    [
        (0x3c21907f595a082c, 0xbfb7e656efb009ae),
        (0xb5d70ebcecd9fbaa, 0xb9348631c28e1012),
        (0xbc11907f595a0fca, 0x3fa7e656efb009ae),
        (0x3bcce5fd1ce60611, 0xbf2bec6b33efcb49),
        (0x3bcf8f1c97773b2d, 0xbf6fd932c26aad94),
        (0x3b80b8964e3bc289, 0x3ef6504d7e054d50),
        (0xbbb3c0e78ce81095, 0x3f20f77ce6105150),
        (0x3b3d95f66e8bcdc7, 0xbea972e573db9593),
        (0x3b4458e98ac8a7d0, 0xbec35ba58bf2f993),
        (0x3ad1d824fbf1c841, 0x3e4e1246a53112c6),
        (0xbacbe1b18e34adb3, 0x3e5b7850b3bc8175),
        (0x3a8ea247bf864770, 0xbde5c8dcec477a4f),
        (0xba8c8c74de40c8c9, 0xbdea905348b49249),
        (0xba0bd92b052944e0, 0x3d7558a1f12cdd8b),
        (0xba1c00a1d58c8cfb, 0x3d729edf3b285f5a),
        (0xb992a01268eb295d, 0xbcfe2f9cdddcaae8),
        (0xb996b7f5d81cfb6a, 0xbcf3c947b927c569),
        (0xb925ba6791e41b2f, 0x3c8020bb87a79ea9),
        (0xb90e7d0294d49287, 0x3c707b5b9c74c0fe),
        (0x38890499436f4660, 0xbbfaf62358105eee),
        (0xb88dc52f6583904b, 0xbbe619a4100326eb),
        (0x37f0135f77fb4dbd, 0x3b721cbe450a51a8),
        (0xb7e6ee90d9e1aa8a, 0x3b582e655a55f7e3),
        (0xb7348781a2c50334, 0xbae44663af3f368d),
    ],
    [
        (0xb6020e4a1b2a8f8d, 0x396ab0500de5301b),
        (0xbc529270d2a71e93, 0x3fb7a597e9550934),
        (0xbbd42472048d4d2c, 0xbf44486c0b012534),
        (0xbc2ac52ac43275af, 0xbf8f848eec0e0e6b),
        (0x3b684f3433759ef3, 0x3f1b077fae06adc6),
        (0xbbce518d0ac3eb8f, 0x3f49310d6e6842c6),
        (0x3b230b8e78861165, 0xbed597a5befbd70e),
        (0x3b9e1e122fd16318, 0xbef32a855e4e113b),
        (0xbb1d74885153e029, 0x3e8069ba0b1a612d),
        (0xbb3a50c31550125c, 0x3e9100c97ba8e22d),
        (0xbaaea70ea0155098, 0xbe1d15b5350404fa),
        (0x3ac3cb5036af1144, 0xbe23bc775cca4413),
        (0xba5b1e821a17c11c, 0x3db0da51f94c98eb),
        (0xba50d67d4e601934, 0x3db0252dc33a4d71),
        (0xb9c1e859398b6f43, 0xbd3b84d751a3586e),
        (0xb9df2f53d258d2cf, 0xbd339ce8224cadca),
        (0xb9601e564eee8b92, 0x3cc0ad1ba9bada71),
        (0x393bd1c6476589ac, 0x3cb262b8db685d38),
        (0x38d88290cd473175, 0xbc3f2ec74017a50b),
        (0x38cc155000aaf19d, 0xbc2b66eb4fd1abd4),
        (0xb84d6b52f7a6d523, 0x3bb72adeccad55cf),
        (0xb81b307d97df219d, 0x3ba09e84d1e5cbd9),
        (0x37cfdfe2f1e86ba7, 0xbb2bc6df8963705f),
        (0x37bd47c432afc7ac, 0xbb10899f2f554f23),
    ],
];
