/* Copyright (c) 2003-2004 Ecole centrale de Lyon
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "fdtd3d.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <complex.h>
#ifdef HAVE_FFTW3
#include <fftw3.h>
#endif
#ifdef HAVE_MPI
#include <mpi.h>
#endif
#include "utils.h"
#include "arrayh5.h"
#include "block.h"

int output_datfile_const(const double *values, int number, const char *name, double interval, double constant)
{
  FILE *thefile;
  int i;
  char filename[FILENAME_BUF];

  if((i=snprintf(filename,FILENAME_BUF-1,"%s.dat",name))<0)
  {
    perror(name);
    return -1;
  }
  if(i>=FILENAME_BUF-1)
  {
    fprintf(stderr,"%s: too long filename for array output.",name);
    return -1;
  }

  if(values==NULL)
  {
    fprintf(stderr,"%s: cannot output empty array!\n",name);
    return -5;
  }
  
  thefile=fopen(filename,"w");
  if(thefile==NULL)
  {
    perror(filename);
    return -1;
  }
  for(i=0;i<number;i++)
    if(fprintf(thefile,"%9e %9e\n",i*interval,values[i]*constant)<0)
    {
      perror(filename);
      fclose(thefile);
      return -2;
    }
  
  fclose(thefile);
  return 0;
}

int output_datfile(const double *values, int number, const char *name, double interval)
{
  return output_datfile_const(values,number,name,interval,1);
}


int output_datfile_const_fft(double *values, int number, const char *name, double interval, double constant)
{
  int retval=0;

#ifdef HAVE_FFTW3
  int i;
  int n2=number/2+1;
  double *fft_values=myalloc(sizeof(double)*n2);
  complex *transform=fftw_malloc(sizeof(complex)*n2);
  char filename[FILENAME_BUF];
  fftw_plan p=fftw_plan_dft_r2c_1d(number,values,transform,FFTW_ESTIMATE);

  if(transform==NULL)
    panic("not enough memory for allocation");
#endif // HAVE_FFTW3
  
  retval=output_datfile_const(values,number,name,interval,constant);

#ifdef HAVE_FFTW3
  fftw_execute(p);
  fftw_destroy_plan(p);

  // Use the square module for output
  for(i=0;i<n2;i++)
    fft_values[i]=SQUARE(creal(transform[i]))+SQUARE(cimag(transform[i]));

  if(retval==0)
    {
      if(snprintf(filename,FILENAME_BUF-1,"%s_dft",name)<0)
	perror("snprintf");
      else
	retval=output_datfile_const(fft_values,n2,filename,1/(number*interval),constant);
    }
  fftw_free(transform);
  free(fft_values);
#endif // HAVE_FFTW3

  return retval;
}

int output_datfile_fft(double *values, int number, const char *name, double interval)
{
  return output_datfile_const_fft(values,number,name,interval,1);
}

int output_datfile3(const double *valx, const double *valy, const double *valz, int number, const char *name, double interval)
{
  FILE *thefile;
  int i;
  char filename[FILENAME_BUF];

  if((i=snprintf(filename,FILENAME_BUF-1,"%s.dat",name))<0)
  {
    perror(name);
    return -1;
  }
  if(i>=FILENAME_BUF-1)
  {
    fprintf(stderr,"%s: too long filename for array output.",name);
    return -1;
  }
  
  if(valx==NULL || valy==NULL || valz==NULL)
  {
    fprintf(stderr,"%s: cannot output empty array!\n",name);
    return -5;
  }
  
  thefile=fopen(filename,"w");
  if(thefile==NULL)
  {
    perror(filename);
    return -1;
  }
  for(i=0;i<number;i++)
    if(fprintf(thefile,"%9e %9e %9e %9e\n",i*interval,valx[i],valy[i],valz[i])<0)
    {
      perror(filename);
      fclose(thefile);
      return -2;
    }
  
  fclose(thefile);
  return 0;
}

int output_datfile3_fft(const double *valx, const double *valy, const double *valz, int number, const char *name, double interval)
{
  int retval=0;

#ifdef HAVE_FFTW3
  int i;
  int n2=number/2+1;
  double *values=fftw_malloc(sizeof(double)*number);
  double *fft_x=myalloc(sizeof(double)*n2);
  double *fft_y=myalloc(sizeof(double)*n2);
  double *fft_z=myalloc(sizeof(double)*n2);
  complex *transform=fftw_malloc(sizeof(complex)*n2);
  char filename[FILENAME_BUF];
  fftw_plan p=fftw_plan_dft_r2c_1d(number,values,transform,FFTW_ESTIMATE);
  
  if(values==NULL || transform==NULL)
    panic("not enough memory for allocation");
#endif // HAVE_FFTW3

  retval=output_datfile3(valx,valy,valz,number,name,interval);
  
#ifdef HAVE_FFTW3
  memcpy(values,valx,number*sizeof(double));
  fftw_execute(p);
  for(i=0;i<n2;i++)
    fft_x[i]=SQUARE(creal(transform[i]))+SQUARE(cimag(transform[i]));
  memcpy(values,valy,number*sizeof(double));
  fftw_execute(p);
  for(i=0;i<n2;i++)
    fft_y[i]=SQUARE(creal(transform[i]))+SQUARE(cimag(transform[i]));
  memcpy(values,valz,number*sizeof(double));
  fftw_execute(p);
  for(i=0;i<n2;i++)
    fft_z[i]=SQUARE(creal(transform[i]))+SQUARE(cimag(transform[i]));
  fftw_destroy_plan(p);

  if(retval==0)
    {
      if(snprintf(filename,FILENAME_BUF-1,"%s_dft",name)<0)
	perror("snprintf");
      else
	retval=output_datfile3(fft_x,fft_y,fft_z,n2,filename,1/(number*interval));
    }
  fftw_free(transform);
  free(fft_x);
  free(fft_y);
  free(fft_z);
  fftw_free(values);
#endif // HAVE_FFTW3

  return retval;
}

void output_carto(const varBlockList *b, const varSpace *space, int iteration, int max_iteration, const char *name, int field, double constant, double *temp_field, double *temp_bfield)
{
  arrayh5 a;
  int size[3];
  char d[15];
  char filename[FILENAME_BUF];
  
  if(max_iteration>1000000000) // Well, that would be quite a long simulation
    panic("no, really, that's too many iterations");

  if(snprintf(d,14,"%s%i%s","%s%0",(int) ceil(log10(max_iteration)),"d.h5")<0)
    panic("whoa ?");

  if(snprintf(filename,FILENAME_BUF-1,d,name,iteration)<0)
    panic("filename too long for field output");

  size[0]=space->nx;
  size[1]=space->ny;
  size[2]=space->nz;
  a.N=space->nx*space->ny*space->nz;
  a.rank=3;
  a.dims=size;
  a.data=temp_field;

  block_extract_field(b,space,field,DIR_X,constant,temp_field,temp_bfield);
#ifdef HAVE_MPI
  if(b->mpi_node==0)
#endif
    arrayh5_write(a,filename,"x_comp",0);
  block_extract_field(b,space,field,DIR_Y,constant,temp_field,temp_bfield);
#ifdef HAVE_MPI
  if(b->mpi_node==0)
#endif
    arrayh5_write(a,filename,"y_comp",1);
  block_extract_field(b,space,field,DIR_Z,constant,temp_field,temp_bfield);
#ifdef HAVE_MPI
  if(b->mpi_node==0)
#endif
    arrayh5_write(a,filename,"z_comp",1);
}
