/*
 * Attempt to extract torus-specific information from BASIL Inventory.
 *
 * WARNING: as of Basil 3.1, this is pure guesswork and will not work
 *          reliably. Maybe in future Cray has more informative tags.
 */
#include "basil_alps.h"
#include "basil_torus.h"

/**
 * get_gemini_cabling  -  Determine Gemini torus information
 * WARNING: This can also not distinguish 2D/3D torus.
 * FIXME: no support at all for 3D torus!
 */
static struct torus_info *get_gemini_cabling(struct torus_info *ti)
{
	uint32_t total_chassis;

	assert(ti != NULL);
	total_chassis = XT_CHASSIS_PER_CABINET * (ti->cabs - 1) + ti->chassis;

	switch (ti->cabling) {
	case CABLING_CLASS0:
		ti->x_max = 1;	/* XXX 2D Torus */

		/* Gemini XE6m (XT5m) 2D Torus variants
		 *
		 * +----------+---------+------------------+
		 * | Cabinets | Chassis | Torus Dimensions |
		 * +----------+---------+------------------+
		 * |        1 |       1 |   1 x  4 x  8    |
		 * |        1 |       2 |   1 x  8 x  8    |
		 * |        1 |       3 |   1 x 12 x  8    |
		 * |        2 |       4 |   1 x  8 x 16    |
		 * |        2 |       6 |   1 x 12 x 16    |
		 * |        3 |       8 |   1 x 16 x 16    |
		 * |        3 |       9 |   1 x 12 x 24    |
		 * |        4 |      12 |   1 x 16 x 24    |
		 * |        5 |      15 |   1 x 20 x 24    |
		 * |        6 |      18 |   1 x 24 x 24    |
		 * +----------+---------+------------------+
		 */
		if (ti->cabs == 1) {
 			ti->y_max = total_chassis * 4;
			ti->z_max = 8;
		} else if (ti->cabs == 2) {
			if (total_chassis == 4)
				ti->y_max = 8;
			else if(total_chassis == 6)
				ti->y_max = 12;
			else
				errx(1, "can not handle 2-cabinet Gemini system with %d chassis",
					total_chassis);
			ti->z_max = 16;
		} else if (ti->cabs == 3) {
			if (total_chassis == 8) {
				ti->y_max = 16;
				ti->z_max = 16;
			} else if(total_chassis == 9) {
				ti->y_max = 12;
				ti->z_max = 24;
			} else {
				errx(1, "can not handle 3-cabinet Gemini system with %d chassis",
					total_chassis);
			}
		} else if (ti->cabs == 4) {
			if (total_chassis == 12) {
				ti->y_max = 16;
				ti->z_max = 24;
			} else {
				errx(1, "can not handle 4-cabinet Gemini system with %d chassis",
					total_chassis);
			}
		} else if (ti->cabs == 5) {
			if (total_chassis == 15) {
				ti->y_max = 20;
				ti->z_max = 24;
			} else {
				errx(1, "can not handle 5-cabinet Gemini system with %d chassis",
					total_chassis);
			}
		} else if (ti->cabs == 6) {
			if (total_chassis == 18) {
				ti->y_max = 24;
				ti->z_max = 24;
			} else {
				errx(1, "can not handle 6-cabinet Gemini system with %d chassis",
					total_chassis);
			}
		} else {
			errx(1, "can not handle Gemini system with %d cabinets",
				ti->cabs);
		}
		break;
	case CABLING_CLASS1:
	case CABLING_CLASS2:
	case CABLING_CLASS3:
		errx(1, "can not handle Gemini cabling class %d", ti->cabling);
	default:
		errx(1, "unknown Gemini torus topology");
	}
	return ti;
}

/**
 * torus_from_inventory - Derive some torus information from inventory.
 *
 * NOTE: This variant can not determine whether the torus is 2D or 3D!
 * Return result is allocated, caller must free().
 */
struct torus_info *torus_from_inventory(const struct basil_inventory *inv)
{
	const struct basil_node *node;
	struct torus_info *ti;
	uint8_t max_cab = 0, max_row = 0, max_chassis = 0, max_router_id = 0;

	ti = calloc(1, sizeof(*ti));
	if (ti == NULL)
		return NULL;

	for (node = inv->f->node_head; node; node = node->next) {
		struct nodecoord nc;

		if (node->arch != BNA_XT) {
			warnx("nid%05u (%s) has unsupported architecture %s",
			      node->node_id, node->name, nam_arch[node->arch]);
		} else if (!string2nodecoord(node->name, &nc)) {
			warnx("can not decode '%s' of nid%05u",
			      node->name, node->node_id);
		} else {
			if (nc.cab > max_cab)
				max_cab = nc.cab;
			if (nc.row > max_row)
				max_row = nc.row;
			if (nc.cage > max_chassis)
				max_chassis = nc.cage;
			if (node->router_id > max_router_id)
				max_router_id = node->router_id;
		}
	}

	/* The 'router_id' attribute is only set on Basil 3.1/XE systems */
	ti->network   = max_router_id ? ASIC_XE_GEMINI : ASIC_XT_SEASTAR;
	ti->cabling   = torus_from_placement(max_cab, max_row);
	ti->cabs      = max_cab + 1;
	ti->rows      = max_row + 1;
	ti->chassis   = max_chassis + 1;
	/*
	 * WARNING: If the system has more than a single chassis, the
	 * assignment below MAY BE WRONG, since by looking at the coordinates
	 * we have no way of seeing whether the cabling is 2D or 3D.
	 * Single-chassis systems have a 2D Torus due to the absence of X cables.
	 */
	ti->dimension = max_chassis ? TORUS_3D : TORUS_2D;

	if (ti->network == ASIC_XE_GEMINI)
		return get_gemini_cabling(ti);
	/*
	 * SeaStar cabling
	 */
	switch (ti->cabling) {
	case CABLING_CLASS0:
		/*
		 * Class 0 topology:
		 * - Torus is N x 4 x 8
		 * - where N = total number of chassis
		 */
		ti->x_max = XT_CHASSIS_PER_CABINET * max_cab + ti->chassis;
 		ti->y_max = 4;
		ti->z_max = 8;
		break;
	case CABLING_CLASS1:
		/*
		 * Class 1 topology:
		 * - Torus is N x 12 x 8
		 * - where N = number of cabinets in the row
		 */
		ti->x_max = ti->cabs;
 		ti->y_max = 12;
		ti->z_max = 8;
		break;
	case CABLING_CLASS2:
		/*
		 * Class 2 topology:
		 * - Torus is N x 12 x 16
		 * - where N = number of cabinets in a row (i.e., 8 <= N <= 24)
		 * - Y = 12, which is 4 SeaStars/blade x 3 cages/cabinet
		 */
		ti->x_max = ti->cabs;
 		ti->y_max = 12;
		ti->z_max = 16;
		break;
	case CABLING_CLASS3:
		/*
		 * Class 3 topology:
		 * - 48..576 cabinets (TR-XTS-S says 48..320)
		 *   o  48 cabinets is  4 rows of 12 cabinets per row
 		 *   o 320 cabinets is 10 rows of 32 cabinets per row
		 *   o 576 cabinets is 12 rows of 48 cabinets per row
		 * - Torus is N x (4 * R) x 24
 		 * - where N = number of cabinets in a row
		 */
		ti->x_max = ti->cabs;
 		ti->y_max = 4 * ti->rows;
		ti->z_max = 24;
		break;
	default:
		warnx("unknown torus topology");
		free(ti);
		return NULL;
	}
	return ti;
}
