/***************************************************************************
 *   Copyright (C) 2007-2025 by Ilya Kotov                                 *
 *   forkotov02@ya.ru                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/
#include <QTimer>
#include <QSettings>
#include <QPainter>
#include <QMenu>
#include <QActionGroup>
#include <qmmp/buffer.h>
#include <qmmp/qmmp.h>
#include <math.h>
#include <stdlib.h>
#include "skin.h"
#include "fft.h"
#include "inlines.h"
#include "skinnedvisualization.h"

SkinnedVisualization *SkinnedVisualization::m_instance = nullptr;

SkinnedVisualization *SkinnedVisualization::instance()
{
    if(!m_instance)
         qCFatal(plugin) << "this object is not created!";
    return m_instance;
}

SkinnedVisualization::SkinnedVisualization(QWidget *parent) : Visual(parent)
{
    m_skin = Skin::instance();
    m_ratio = m_skin->ratio();
    connect(m_skin, &Skin::skinChanged, this, &SkinnedVisualization::readSettings);
    m_timer = new QTimer(this);
    connect(m_timer, &QTimer::timeout, this, &SkinnedVisualization::timeout);
    m_instance = this;
    createMenu();
    readSettings();
}

SkinnedVisualization::~SkinnedVisualization()
{
    writeSettings();
    if(m_vis)
    {
        delete m_vis;
        m_vis = nullptr;
    }
    m_instance = nullptr;
}

void SkinnedVisualization::setVisual(SkinnedVisualBase *newvis)
{
    m_timer->stop();
    delete m_vis;
    m_vis = newvis;
    if(m_vis)
        m_timer->start();
    else
    {
        m_pixmap.fill(Qt::transparent);
        update();
    }
}

void SkinnedVisualization::clear()
{
    if(m_vis)
        m_vis->clear();
    m_pixmap = m_bg;
    update();
}

void SkinnedVisualization::timeout()
{
    if(m_vis && takeData(m_buffer))
    {
        m_vis->process(m_buffer);
        m_pixmap = m_bg;
        QPainter p(&m_pixmap);
        m_vis->draw (&p);
        update();
    }
}

void SkinnedVisualization::paintEvent (QPaintEvent *)
{
    QPainter painter (this);
    painter.drawPixmap (0,0, m_pixmap);
}

void SkinnedVisualization::hideEvent (QHideEvent *)
{
    m_timer->stop();
}

void SkinnedVisualization::showEvent (QShowEvent *)
{
    if(m_vis && m_running)
        m_timer->start();
}

void SkinnedVisualization::mousePressEvent (QMouseEvent *e)
{
    if(e->button() == Qt::RightButton)
    {
        m_menu->exec(e->globalPosition().toPoint());
        return;
    }

    m_pixmap = m_bg;
    if(!m_vis)
        setVisual(new mainvisual::Analyzer);
    else if(m_vis->name() == "Analyzer"_L1)
        setVisual(new mainvisual::Scope);
    else if(m_vis->name() == "Scope"_L1)
        setVisual(nullptr);

    QString str = m_vis ? m_vis->name() : "Off"_L1;
    for(QAction *act : m_visModeGroup->actions())
    {
        if(str == act->data().toString())
        {
            act->setChecked(true);
            break;
        }
    }
    writeSettings();
}

void SkinnedVisualization::start()
{
    m_running = true;
    if(isVisible())
        m_timer->start();
}

void SkinnedVisualization::stop()
{
    m_running = false;
    m_timer->stop();
    clear();
}

void SkinnedVisualization::drawBackGround()
{
    m_bg = QPixmap (76 * m_ratio, 16 * m_ratio);
    if(m_transparentAction->isChecked())
    {
        m_bg.fill (Qt::transparent);
        return;
    }
    QPainter painter(&m_bg);
    for(int x = 0; x < 76 * m_ratio; x += 2)
    {
        painter.setPen(m_skin->getVisColor(0));
        painter.drawLine(x + 1, 0, x + 1, 16 *m_ratio);
        for(int y = 0; y < 16 *m_ratio; y += 2)
        {
            painter.setPen(m_skin->getVisColor(0));
            painter.drawPoint(x, y);
            painter.setPen(m_skin->getVisColor(1));
            painter.drawPoint(x, y + 1);
        }
    }
}

void SkinnedVisualization::writeSettings()
{
    QSettings settings;
    settings.beginGroup("Skinned"_L1);
    QAction *act = m_peaksFalloffGroup->checkedAction ();
    settings.setValue("vis_peaks_falloff"_L1, act ? act->data().toDouble() : 0.2);
    act = m_analyzerFalloffGroup->checkedAction ();
    settings.setValue("vis_analyzer_falloff"_L1, act ? act->data().toDouble() : 2.2);

    settings.setValue("vis_show_peaks"_L1, m_peaksAction->isChecked());

    act = m_analyzerModeGroup->checkedAction();
    settings.setValue("vis_analyzer_mode"_L1, act ? act->data().toInt() : 0);
    act = m_analyzerTypeGroup->checkedAction();
    settings.setValue("vis_analyzer_type"_L1, act ? act->data().toInt() : 1);
    settings.setValue("vis_transparent_bg"_L1, m_transparentAction->isChecked());

    act = m_visModeGroup->checkedAction ();
    settings.setValue("vis_type"_L1, act ? act->data().toString() : u"Off"_s);

    act = m_fpsGroup->checkedAction();
    settings.setValue("vis_rate"_L1, act ? act->data().toInt() : 25);
}

void SkinnedVisualization::createMenu()
{
    m_menu = new QMenu (this);
    connect(m_menu, &QMenu::triggered, this, &SkinnedVisualization::writeSettings);
    connect(m_menu, &QMenu::triggered, this, &SkinnedVisualization::readSettings);
    QMenu *visMode = m_menu->addMenu(tr("Visualization Mode"));
    m_visModeGroup = new QActionGroup(this);
    m_visModeGroup->setExclusive(true);
    m_visModeGroup->addAction(tr("Analyzer"))->setData(u"Analyzer"_s);
    m_visModeGroup->addAction(tr("Scope"))->setData(u"Scope"_s);
    m_visModeGroup->addAction(tr("Off"))->setData(u"Off"_s);
    for(QAction *act : m_visModeGroup->actions())
    {
        act->setCheckable(true);
        visMode->addAction(act);
    }

    QMenu *analyzerMode = m_menu->addMenu(tr("Analyzer Mode"));
    m_analyzerModeGroup = new QActionGroup(this);
    m_analyzerTypeGroup = new QActionGroup(this);
    m_analyzerModeGroup->addAction(tr("Normal"))->setData(0);
    m_analyzerModeGroup->addAction(tr("Fire"))->setData(1);
    m_analyzerModeGroup->addAction(tr("Vertical Lines"))->setData(2);
    m_analyzerTypeGroup->addAction(tr("Lines"))->setData(0);
    m_analyzerTypeGroup->addAction(tr("Bars"))->setData(1);
    for(QAction *act : m_analyzerModeGroup->actions())
    {
        act->setCheckable(true);
        analyzerMode->addAction(act);
    }
    analyzerMode->addSeparator ();
    for(QAction *act : m_analyzerTypeGroup->actions())
    {
        act->setCheckable(true);
        analyzerMode->addAction(act);
    }
    analyzerMode->addSeparator ();
    m_peaksAction = analyzerMode->addAction(tr("Peaks"));
    m_peaksAction->setCheckable(true);


    QMenu *refreshRate = m_menu->addMenu(tr("Refresh Rate"));
    m_fpsGroup = new QActionGroup(this);
    m_fpsGroup->setExclusive(true);
    m_fpsGroup->addAction(tr("50 fps"))->setData(50);
    m_fpsGroup->addAction(tr("25 fps"))->setData(25);
    m_fpsGroup->addAction(tr("10 fps"))->setData(10);
    m_fpsGroup->addAction(tr("5 fps"))->setData(5);
    for(QAction *act : m_fpsGroup->actions())
    {
        act->setCheckable(true);
        refreshRate->addAction(act);
    }

    QMenu *analyzerFalloff = m_menu->addMenu(tr("Analyzer Falloff"));
    m_analyzerFalloffGroup = new QActionGroup(this);
    m_analyzerFalloffGroup->setExclusive(true);
    m_analyzerFalloffGroup->addAction(tr("Slowest"))->setData(1.2);
    m_analyzerFalloffGroup->addAction(tr("Slow"))->setData(1.8);
    m_analyzerFalloffGroup->addAction(tr("Medium"))->setData(2.2);
    m_analyzerFalloffGroup->addAction(tr("Fast"))->setData(2.4);
    m_analyzerFalloffGroup->addAction(tr("Fastest"))->setData(2.8);
    for(QAction *act : m_analyzerFalloffGroup->actions())
    {
        act->setCheckable(true);
        analyzerFalloff->addAction(act);
    }

    QMenu *peaksFalloff = m_menu->addMenu(tr("Peaks Falloff"));
    m_peaksFalloffGroup = new QActionGroup(this);
    m_peaksFalloffGroup->setExclusive(true);
    m_peaksFalloffGroup->addAction(tr("Slowest"))->setData(0.05);
    m_peaksFalloffGroup->addAction(tr("Slow"))->setData(0.1);
    m_peaksFalloffGroup->addAction(tr("Medium"))->setData(0.2);
    m_peaksFalloffGroup->addAction(tr("Fast"))->setData(0.4);
    m_peaksFalloffGroup->addAction(tr("Fastest"))->setData(0.8);
    for(QAction *act : m_peaksFalloffGroup->actions())
    {
        act->setCheckable(true);
        peaksFalloff->addAction(act);
    }
    QMenu *background = m_menu->addMenu(tr("Background"));
    m_transparentAction = background->addAction(tr("Transparent"));
    m_transparentAction->setCheckable(true);
    update();
}


void SkinnedVisualization::readSettings()
{
    QSettings settings;
    settings.beginGroup("Skinned"_L1);
    QString vis_name = settings.value(u"vis_type"_s, u"Analyzer"_s).toString();
    if(!m_update)
    {
        m_update = true;

        for(QAction *act : m_visModeGroup->actions())
        {
            if(vis_name == act->data().toString())
                act->setChecked(true);
        }
        m_peaksAction->setChecked(settings.value("vis_show_peaks"_L1, true).toBool());
        int fps = settings.value("vis_rate"_L1, 25).toInt();
        for(QAction *act : m_fpsGroup->actions())
        {
            if(fps == act->data().toInt())
                act->setChecked(true);
        }
        int mode = settings.value("vis_analyzer_mode"_L1, 0).toInt();
        for(QAction *act : m_analyzerModeGroup->actions())
        {
            if(mode == act->data().toInt())
                act->setChecked(true);
        }
        int type = settings.value("vis_analyzer_type"_L1, 1).toInt();
        for(QAction *act : m_analyzerTypeGroup->actions())
        {
            if(type == act->data().toInt())
                act->setChecked(true);
        }
        double speed = settings.value("vis_peaks_falloff"_L1, 0.2).toDouble();
        for(QAction *act : m_peaksFalloffGroup->actions())
        {
            if(speed == act->data().toDouble())
                act->setChecked(true);
        }
        speed = settings.value("vis_analyzer_falloff"_L1, 2.2).toDouble();
        for(QAction *act : m_analyzerFalloffGroup->actions())
        {
            if(speed == act->data().toDouble())
                act->setChecked(true);
        }
        m_transparentAction->setChecked(settings.value("vis_transparent_bg"_L1, false).toBool());

        //update settings from previous version
        if(!m_analyzerFalloffGroup->checkedAction() || !m_peaksFalloffGroup->checkedAction())
        {
            m_analyzerFalloffGroup->actions().at(2)->setChecked(true);
            m_peaksFalloffGroup->actions().at(2)->setChecked(true);
            writeSettings();
        }
    }
    m_ratio = m_skin->ratio();
    drawBackGround();
    m_pixmap = m_bg;
    QAction *act = m_fpsGroup->checkedAction ();
    m_timer->setInterval (act ? 1000 / act->data().toInt() : 25);
    if(vis_name == "Analyzer"_L1)
        setVisual(new mainvisual::Analyzer);
    else if(vis_name == "Scope"_L1)
        setVisual(new mainvisual::Scope);
    else
        setVisual(nullptr);
    resize(76 * m_ratio, 16 * m_ratio);
    update();
}

using namespace mainvisual;

Analyzer::Analyzer()
{
    Analyzer::clear();
    m_skin = Skin::instance();
    m_size = QSize(76 * m_skin->ratio(), 16 * m_skin->ratio());

    QSettings settings;
    settings.beginGroup("Skinned"_L1);
    m_peaks_falloff = settings.value("vis_peaks_falloff"_L1, 0.2).toDouble();
    m_analyzer_falloff = settings.value("vis_analyzer_falloff"_L1, 2.2).toDouble();
    m_show_peaks = settings.value("vis_show_peaks"_L1, true).toBool();

    m_lines = settings.value("vis_analyzer_type"_L1, 1).toInt() == 0;
    m_mode = settings.value("vis_analyzer_mode"_L1, 0).toInt();
}

void Analyzer::clear()
{
    for(int i = 0; i < 75; ++i)
    {
        m_intern_vis_data[i] = 0;
        m_peaks[i] = 0;
    }
}

bool Analyzer::process(float *l)
{
    static fft_state *state = nullptr;
    if(!state)
        state = fft_init();
    short dest[256];

    static const int xscale_long[] =
    {
        0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34,
        35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
        52, 53, 54, 55, 56, 57, 58, 61, 66, 71, 76, 81, 87, 93, 100, 107,
        114, 122, 131, 140, 150, 161, 172, 184, 255
    };

    static const int xscale_short[] =
    {
        0, 1, 2, 3, 4, 5, 6, 7, 8, 11, 15, 20, 27,
        36, 47, 62, 82, 107, 141, 184, 255
    };

    calc_freq (dest, l);

    const double y_scale = 3.60673760222;   /* 20.0 / log(256) */
    int max = m_lines ? 75 : 19;

    for(int i = 0; i < max; i++)
    {
        int y = 0;

        if(m_lines)
        {
            for(int j = xscale_long[i]; j < xscale_long[i + 1]; j++)
            {
                y = qMax(dest[j] >> 7, y); //128
            }
        }
        else
        {
            for(int j = xscale_short[i]; j < xscale_short[i + 1]; j++)
            {
                y = qMax(dest[j] >> 7, y);
            }
        }

        int magnitude = 0;
        if(y > 0)
        {
            magnitude = qBound(0, int(log(y) * y_scale), 15);
        }

        m_intern_vis_data[i] -= m_analyzer_falloff;
        m_intern_vis_data[i] = magnitude > m_intern_vis_data[i] ? magnitude : m_intern_vis_data[i];
        if(m_show_peaks)
        {
            m_peaks[i] -= m_peaks_falloff;
            m_peaks[i] = magnitude > m_peaks[i] ? magnitude : m_peaks[i];
        }
    }
    return true;
}

void Analyzer::draw (QPainter *p)
{
    int r = m_skin->ratio();
    if(m_lines)
        for(int j = 0; j < 75; ++j)
        {
            for(int i = 0; i <= m_intern_vis_data[j]; ++i)
            {
                if(m_mode == 0)
                    p->setPen(m_skin->getVisColor(18 - i));
                else if(m_mode == 1)
                    p->setPen(m_skin->getVisColor(3 + (int(m_intern_vis_data[j]) - i)));
                else
                    p->setPen(m_skin->getVisColor(18 - int(m_intern_vis_data[j])));
                p->drawPoint(j * r, m_size.height() - r * i);
                if(r == 2)
                    p->drawPoint(j * r + 1, m_size.height() - r * i);
            }
            p->setPen(m_skin->getVisColor(23));
            if(m_show_peaks)
            {
                p->drawPoint(j * r, m_size.height() - r * m_peaks[j]);
                if(r == 2)
                    p->drawPoint (j * r + 1, m_size.height() - r * m_peaks[j]);
            }
        }
    else
        for(int j = 0; j < 19; ++j)
        {
            for(int i = 0; i <= m_intern_vis_data[j]; ++i)
            {
                if(m_mode == 0)
                    p->setPen(m_skin->getVisColor(18 - i));
                else if(m_mode == 1)
                    p->setPen(m_skin->getVisColor(3 + (int(m_intern_vis_data[j]) - i)));
                else
                    p->setPen(m_skin->getVisColor(18 - int(m_intern_vis_data[j])));

                p->drawLine(j * 4 * r, m_size.height() - r * i, (j * 4 + 2) * r, m_size.height() - r * i);
                if(r == 2)
                    p->drawLine(j * 4 * r, m_size.height() - r * i +1, (j * 4 + 2) * r, m_size.height() - r * i + 1);
            }
            p->setPen (m_skin->getVisColor (23));
            if(m_show_peaks)
            {
                p->drawLine(j * 4 * r, m_size.height() - r * m_peaks[j],
                             (j * 4 + 2) * r, m_size.height() - r * m_peaks[j]);
                if(r == 2)
                    p->drawLine(j * 4 * r, m_size.height() - r * m_peaks[j] + 1,
                                 (j * 4 + 2) * r, m_size.height() - r * m_peaks[j] + 1);
            }
        }
}

const QString Analyzer::name()
{
    return u"Analyzer"_s;
}

Scope::Scope()
{
    Scope::clear();
    m_skin = Skin::instance();
    m_ratio = m_skin->ratio();
}

void Scope::clear()
{
    for(int i = 0; i < 76; ++i)
        m_intern_vis_data[i] = 5;
}

bool Scope::process(float *l)
{
    int step = (QMMP_VISUAL_NODE_SIZE << 8) / 76;
    int pos = 0;

    for(int i = 0; i < 76; ++i)
    {
        pos += step;
        m_intern_vis_data[i] = int(l[pos >> 8] * 8.0);
        m_intern_vis_data[i] = qBound(-4, m_intern_vis_data[i], 4);
    }
    return true;
}

void Scope::draw(QPainter *p)
{
    for(int i = 0; i < 75; ++i)
    {
        int h1 = 8 - m_intern_vis_data[i];
        int h2 = 8 - m_intern_vis_data[i+1];
        if(h1 > h2)
            qSwap(h1, h2);
        p->setPen(m_skin->getVisColor(18 + qAbs(8 - h2)));
        p->drawLine(i * m_ratio, h1 * m_ratio, (i + 1) * m_ratio, h2 * m_ratio);
    }
    for(int i = 0; i < 76; ++i)
        m_intern_vis_data[i] = 0;
}

const QString Scope::name()
{
    return u"Scope"_s;
}
