///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting/Scripting.h>
#include <core/plugins/PluginClassDescriptor.h>
#include <core/plugins/Plugin.h>
#include <core/plugins/NativePlugin.h>
#include <core/plugins/PluginManager.h>
#include <core/utilities/PathManager.h>
#include <core/viewport/ViewportManager.h>
#include "ScriptEngine.h"

namespace Scripting {

using namespace boost::python;

/// Array of char* pointers to the registered module names.
QVector< shared_array<char> > ScriptEngine::moduleNames;

/// The initXXX() functions for each of the registered plugins.
PythonPluginRegistration* PythonPluginRegistration::instances = NULL;

/// The master script engine that runs in the global scope.
ScriptEngine* ScriptEngine::masterEngine = NULL;

/// This cleanup handler is used to delete the master engine when the program ends.
QObjectCleanupHandler ScriptEngine::masterEngineCleanupHandler;

/******************************************************************************
* Constructor
******************************************************************************/
ScriptEngine::ScriptEngine(QObject* parent)
	: QObject(parent), failedToInitialize(true)
{
	// Create a copy of the master engine.
	try {
		// Make a copy of the global main namespace. The original namespace dictionary
		// is not touched by the script.
		main_namespace = boost::python::object(boost::python::handle<>(PyDict_Copy(master().main_namespace.ptr())));

		// Register the output redirector objects to make this engine receive
		// the script output.
		setupOutputRedirector();
	}
	catch(const error_already_set&) {
		PyErr_Print();
		throw Exception(tr("Could not prepare Python interpreter. See console output for error details."));
	}
	// Everything went well.
	failedToInitialize = false;
}

/******************************************************************************
* Returns a reference to the global master engine.
******************************************************************************/
ScriptEngine& ScriptEngine::master()
{
	if(masterEngine == NULL) {

		// Create the engine instance.
		masterEngine = new ScriptEngine(true);
		masterEngineCleanupHandler.add(masterEngine);

		// Forward all messages from the master script engine to the console of the host program,
		connect(masterEngine, SIGNAL(scriptOutput(const QString&)), masterEngine, SLOT(consoleOutput(const QString&)));
		connect(masterEngine, SIGNAL(scriptError(const QString&)), masterEngine, SLOT(consoleError(const QString&)));

		// Initialize the Python interpreter, load scriptable plugins etc.
		masterEngine->initInterpreter();
	}
	if(masterEngine->failedToInitialize)
		throw Exception(tr("The Python interpreter failed to initialize."));
	return *masterEngine;
}

/******************************************************************************
* Initializes the python interpreter.
******************************************************************************/
void ScriptEngine::initInterpreter()
{
	try {
		// Initialize the Python interpreter.
		Py_IsInitialized();

		VerboseLogger() << logdate << "Initializing embedded Python interpreter." << endl;
		VerboseLogger() << "Python program name: " << Py_GetProgramName() << endl;
		VerboseLogger() << "Python program full path: " << Py_GetProgramFullPath() << endl;
		VerboseLogger() << "Python prefix: " << Py_GetPrefix() << endl;
		VerboseLogger() << "Python exec prefix: " << Py_GetExecPrefix() << endl;
		VerboseLogger() << "Python path: " << Py_GetPath() << endl;

		// Import the Scripting module itself.
		QSet<Plugin*> registeredScriptablePlugins;
		registerPluginWithInterpreter(PLUGIN_MANAGER.plugin("Scripting"), registeredScriptablePlugins);
		// Import all other scriptable plugins.
		Q_FOREACH(Plugin* plugin, PLUGIN_MANAGER.plugins()) {
			registerPluginWithInterpreter(plugin, registeredScriptablePlugins);
		}

		VerboseLogger() << "Initalizing Python interpreter." << endl;
		Py_Initialize();

		// Pass command line parameters to the script.
		VerboseLogger() << "Passing command line parameters to script." << endl;
		QStringList args = qApp->arguments();
		for(int i=0; i<args.size()-1; i++) {
			if(args[i] == "--script") {

				// Convert to C array.
				int argc = args.size() - i - 1;
				char** argv = new char*[argc];
				QVector<QByteArray> argumentList(argc);
				for(int j=0; j<argc; j++) {
					argumentList[j] = args[i + j + 1].toLocal8Bit();
					argv[j] = argumentList[j].data();
				}

				// Pass all remaining parameters after the --script command to the script.
				PySys_SetArgv(argc, argv);

				break;
			}
		}

		VerboseLogger() << "Installing output redirector." << endl;
		boost::python::object main_module = object(boost::python::handle< >(borrowed(PyImport_AddModule("__main__"))));
		main_namespace = object(boost::python::handle< >(borrowed(PyModule_GetDict(main_module.ptr()))));

		// Register the output redirector class.
		object redirectorClass = class_<InterpreterOutputRedirector, auto_ptr<InterpreterOutputRedirector>, noncopyable>("_StdOutWriterClass", no_init)
			.def("write", &InterpreterOutputRedirector::write)
		;
		main_namespace["__OutputRedirectorClass"] = redirectorClass;
		setupOutputRedirector();

		// Import the Scripting classes into the main namespace.
		VerboseLogger() << "Importing Python based script classes into namespace." << endl;
		exec("from Scripting import *", main_namespace, main_namespace);

		// Import the third-party plugin module libraries.
		for(int i=0; i<moduleNames.size(); i++) {
			string importStatement("import ");
			importStatement += moduleNames[i].get();
			exec(importStatement.c_str(), main_namespace, main_namespace);
		}

		try {
			// Import the third-party Python source modules.

			// Add scripts directory to python path.
			QByteArray command("sys.path.append(\"");
			command += PATH_MANAGER.scriptsDirectory();
			command += "\")";
			exec(command.constData(), main_namespace, main_namespace);

			// Iterate over all subdirectories in the scripts directory.
			Q_FOREACH(QString moduleName, QDir(PATH_MANAGER.scriptsDirectory()).entryList(QDir::Dirs|QDir::NoDotAndDotDot)) {
				QDir packageDir(PATH_MANAGER.scriptsDirectory() + '/' + moduleName);
				if(packageDir.exists("__init__.py")) {
					VerboseLogger() << logdate << "Importing Python source module" << moduleName << "from scripts directory." << endl;

					QByteArray importStatement("import ");
					importStatement += moduleName;
					exec(importStatement.constData(), main_namespace, main_namespace);
				}
			}
			failedToInitialize = false;
		}
		catch(error_already_set) {
			PyErr_Print();
			throw Exception(tr("Python interpreter has exited with an error."));
		}
	}
	catch(const Exception&) {
		throw;
	}
	catch(const std::exception& ex) {
		throw Exception(tr("Script execution error: %1").arg(QString(ex.what())));
	}
	catch(error_already_set) {
		PyErr_Print();
		throw Exception(tr("Python interpreter has exited with an error."));
	}
	catch(...) {
		throw Exception(tr("Unhandled exception thrown by Python interpreter."));
	}
}

/******************************************************************************
* This registers the output redirector objects
* to catch the Python output to stdout/stderr.
******************************************************************************/
void ScriptEngine::setupOutputRedirector()
{
	boost::python::object sys_module = object(boost::python::handle< >(PyImport_ImportModule("sys")));
	boost::python::object sys_namespace = object(boost::python::handle< >(borrowed(PyModule_GetDict(sys_module.ptr()))));
	main_namespace["sys"] = sys_module;
	sys_namespace["stdout"] = ptr(new InterpreterOutputRedirector(this, false));
	sys_namespace["stderr"] = ptr(new InterpreterOutputRedirector(this, true));
}

/******************************************************************************
* Executes an existing Python script file.
* Throws an exception on error.
******************************************************************************/
int ScriptEngine::executeScriptFile(const QString& scriptFile)
{
	if(failedToInitialize)
		throw Exception(tr("The Python interpreter failed to initialize."));

	try {
		// Do not allow viewport updates while the script is being executed.
		ViewportSuspender noVPUpdates;

		// Run the user script.
		VerboseLogger() << logdate << "Running script file" << scriptFile << endl << flush;
		exec_file(QDir::toNativeSeparators(scriptFile).toAscii().constData(), main_namespace, main_namespace);

		return 0;
	}
	catch(error_already_set) {
		// Handle call to sys.exit()
		if(PyErr_Occurred() && PyErr_ExceptionMatches(PyExc_SystemExit)) {
			return handleSystemExit();
		} 
		else {
			PyErr_Print();
			throw Exception(tr("Python interpreter has exited with an error. See interpreter output for details."));
		}
	}
	catch(const Exception&) {
		throw;
	}
	catch(const std::exception& ex) {
		throw Exception(tr("Script execution error: %1").arg(QString(ex.what())));
	}
	catch(...) {
		throw Exception(tr("Unhandled exception thrown by Python interpreter."));
	}
}

/******************************************************************************
* Executes a script.
******************************************************************************/
int ScriptEngine::executeScript(const QByteArray& script)
{
	if(failedToInitialize)
		throw Exception(tr("The Python interpreter failed to initialize."));

	try {
		// Do not allow viewport updates while the script is being executed.
		ViewportSuspender noVPUpdates;

		// Run the user script.
		exec(script.constData(), main_namespace, main_namespace);

		return 0;
	}
	catch(error_already_set) {
		// Handle call to sys.exit()
		if(PyErr_Occurred() && PyErr_ExceptionMatches(PyExc_SystemExit)) {
			return handleSystemExit();
		} 
		else {
			PyErr_Print();
			throw Exception(tr("Python interpreter has exited with an error. See interpreter output for details."));
		}
	}
	catch(const Exception&) {
		throw;
	}
	catch(const std::exception& ex) {
		throw Exception(tr("Script execution error: %1").arg(QString(ex.what())));
	}
	catch(...) {
		throw Exception(tr("Unhandled exception thrown by Python interpreter."));
	}
}

/******************************************************************************
* Register the Python classes exposed by a plugin with the Python interpreter.
******************************************************************************/
void ScriptEngine::registerPluginWithInterpreter(Plugin* plugin, QSet<Plugin*>& listOfAlreadyLoadedPlugins)
{
	CHECK_POINTER(plugin);

	// Check if this plugin has already been registered.
	if(listOfAlreadyLoadedPlugins.contains(plugin)) return;
	listOfAlreadyLoadedPlugins.insert(plugin);

	// Register all plugins required for this plugin first.
	Q_FOREACH(Plugin* p, plugin->dependencies()) {
		registerPluginWithInterpreter(p, listOfAlreadyLoadedPlugins);
	}

	// Check if it is a native scriptable plugin.
	NativePlugin* nativePlugin = qobject_cast<NativePlugin*>(plugin);
	if(!nativePlugin || !nativePlugin->isScriptable()) return;

	// Load plugin.
	plugin->loadPlugin();

	// Lookup plugin's initXXX() function.
	QByteArray qModuleName = plugin->pluginId().toAscii();
#ifndef OVITO_MONOLITHIC_BUILD
	QByteArray initFunctionName("init" + qModuleName);
	void (*initFunc)() = (void (*)())nativePlugin->library()->resolve(initFunctionName.constData());
#else
	void (*initFunc)() = NULL;
	PythonPluginRegistration* r = PythonPluginRegistration::instances;
	while(r) {
		if(qModuleName == r->_pluginName) {
			initFunc = r->_initFunc;
			break;
		}
		r = r->next;
	}
#endif
	if(!initFunc) return;

	// Allocate memory for the module name string.
	char* moduleName = new char[qModuleName.size() + 1];
	memcpy(moduleName, qModuleName.constData(), qModuleName.size());
	moduleName[qModuleName.size()] = '\0';
	moduleNames.push_back(shared_array<char>(moduleName));

	// Pass plugin's init function to the Python interpreter.
	try {
		PyImport_AppendInittab(moduleName, initFunc);
		VerboseLogger() << logdate << "Registered scriptable native plugin" << plugin->pluginId() << "with Python interpreter." << endl;
	}
	catch(const Exception& ex) {
		MsgLogger() << "Warning: Failed to load scriptable plugin:" << plugin->pluginId() << endl;
		MsgLogger() << "Reason:" << ex.message() << endl;
		moduleNames.remove(moduleNames.size() - 1);
	}
}


/******************************************************************************
* Handles a call to sys.exit() in the Python interpreter.
******************************************************************************/
int ScriptEngine::handleSystemExit()
{
	PyObject *exception, *value, *tb;
	int exitcode = 0;

	PyErr_Fetch(&exception, &value, &tb);
	if(Py_FlushLine())
		PyErr_Clear();
	if(value == NULL || value == Py_None)
		goto done;
#ifdef PyExceptionInstance_Check
	if(PyExceptionInstance_Check(value)) {		// Python 2.6.x or newer
#else
	if(PyInstance_Check(value)) {			// Python 2.4.x
#endif
		/* The error code should be in the code attribute. */
		PyObject *code = PyObject_GetAttrString(value, "code");
		if (code) {
			Py_DECREF(value);
			value = code;
			if (value == Py_None)
				goto done;
		}
		/* If we failed to dig out the 'code' attribute,
		   just let the else clause below print the error. */
	}
	if(PyInt_Check(value))
		exitcode = (int)PyInt_AsLong(value);
	else {
		PyObject *s = PyObject_Str(value);
		QString errorMsg;
		if(s) {
			const char* s2 = PyString_AsString(s);
			if(s2) {
				errorMsg = QString(s2) + '\n';
			}
		}
		Py_XDECREF(s);
		if(errorMsg.isEmpty() == false)
			scriptError(errorMsg);
		exitcode = 1;
	}
 done:
	PyErr_Restore(exception, value, tb);
	PyErr_Clear();
	return exitcode;
}

};
