/*
 * Copyright 1997-2024 Optimatika
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.ojalgo.optimisation.integer;

import java.math.BigDecimal;
import java.math.RoundingMode;

import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import org.ojalgo.TestUtils;
import org.ojalgo.function.aggregator.Aggregator;
import org.ojalgo.function.constant.BigMath;
import org.ojalgo.function.constant.PrimitiveMath;
import org.ojalgo.matrix.MatrixR064;
import org.ojalgo.matrix.decomposition.Eigenvalue;
import org.ojalgo.matrix.store.MatrixStore;
import org.ojalgo.matrix.store.PhysicalStore;
import org.ojalgo.matrix.store.R064Store;
import org.ojalgo.matrix.store.RawStore;
import org.ojalgo.netio.BasicLogger;
import org.ojalgo.optimisation.Expression;
import org.ojalgo.optimisation.ExpressionsBasedModel;
import org.ojalgo.optimisation.Optimisation;
import org.ojalgo.optimisation.Optimisation.Result;
import org.ojalgo.optimisation.Variable;
import org.ojalgo.random.SampleSet;
import org.ojalgo.structure.Access1D;
import org.ojalgo.structure.Access2D;
import org.ojalgo.type.context.NumberContext;

public class NextGenSysModTest {

    public static final class Case010A extends CaseData {

        public static final double[][] covarianceMtrx = {
                { 0.07152680941594794, 0.007853246279673355, 0.013893689282343354, 0.013207960586582078, 0.014606418480005646, 0.010232228938801623,
                        0.025802891886875352, 0.009909963551243848, 0.0057891645799561745, 0.016572750681393696 },
                { 0.007853246279673355, 0.06545536904213232, 0.011788770189230618, 0.02069862233221937, 0.009701337711031054, 0.012825181519248805,
                        0.03306713990594811, 3.876524551038414E-4, 0.0029015393830402907, 0.019797189007735083 },
                { 0.013893689282343354, 0.011788770189230618, 0.05107412429782171, 0.014481439953126243, 0.009569900297446286, 0.017420675978535728,
                        0.02922165116888207, -0.001868396127039013, 0.0015014633763758113, 0.015562592869480252 },
                { 0.013207960586582078, 0.02069862233221937, 0.014481439953126243, 0.03868323186216877, 0.012363646457175585, 0.013953647050572403,
                        0.026686590357644764, 0.004692546639365153, 0.004776952477106753, 0.022046298023845272 },
                { 0.014606418480005646, 0.009701337711031054, 0.009569900297446286, 0.012363646457175585, 0.051660196459506966, 0.011674857280291604,
                        0.017035602731414894, 0.005005116437727502, 6.813938897308552E-4, 0.022621376440059772 },
                { 0.010232228938801623, 0.012825181519248805, 0.017420675978535728, 0.013953647050572403, 0.011674857280291604, 0.025036862968080657,
                        0.01837538927744351, 0.005804645222440367, 0.0050198986900658665, 0.01364529284043886 },
                { 0.025802891886875352, 0.03306713990594811, 0.02922165116888207, 0.026686590357644764, 0.017035602731414894, 0.01837538927744351,
                        0.08298207390686048, 0.008267339447806964, 0.003962445153184218, 0.024499027467744947 },
                { 0.009909963551243848, 3.876524551038414E-4, -0.001868396127039013, 0.004692546639365153, 0.005005116437727502, 0.005804645222440367,
                        0.008267339447806964, 0.022756650635823328, 0.01710285736388286, 0.011543351051513556 },
                { 0.0057891645799561745, 0.0029015393830402907, 0.0015014633763758113, 0.004776952477106753, 6.813938897308552E-4, 0.0050198986900658665,
                        0.003962445153184218, 0.01710285736388286, 0.0215237061398265, 0.011662166620020871 },
                { 0.016572750681393696, 0.019797189007735083, 0.015562592869480252, 0.022046298023845272, 0.022621376440059772, 0.01364529284043886,
                        0.024499027467744947, 0.011543351051513556, 0.011662166620020871, 0.07551465164636477 } };
        public static final double[][] dataSetMtrx = { { 0.05, 0.05, 0.07, 0.04, 0.09, 0.04, 0.14, 0.05, 0.05, 0.11 },
                { 0.08, 0.62, 0.64, 0.2, 0.35, 0.1, 0.71, 0.38, 0.49, 0.35 }, { 1.04, 1.13, 1.12, 1.1, 0.98, 0.91, 1.88, 0.31, 0.17, 1.03 } };
        public static final double[] optimisationSolution = { 0.0, 0.0, 0.0, 0.0, 0.1752712392223518, 0.324728760777648, 0.0, 0.0, 0.5, 0.0, 0.0, 0.0, 0.0, 0.0,
                1.0, 1.0, 0.0, 0.0, 1.0, 0.0 };

        Case010A() {
            super(covarianceMtrx, dataSetMtrx, optimisationSolution, -0.5805630399168837, -0.5805630399168837);
        }

    }

    public static final class Case020A extends CaseData {

        public static final double[][] covarianceMtrx = {
                { 0.09117796973644532, 0.0019031315462784144, 0.017590569558430334, 0.01672103191085612, 0.01755632403955279, 0.012262007474096198,
                        0.03483071434392497, 0.01307003508704781, 0.005202737936835605, 0.019212532773270445, 0.0032373487733811435, 0.02742784802402331,
                        0.01469847057079009, 0.016954568683183992, 0.01210386425386745, 0.011041948656750589, 0.030678267401908477, 0.004527800043582497,
                        0.026694382890454317, 0.006961437730164128 },
                { 0.0019031315462784144, 0.06622730393922867, 0.012494152072132197, 0.01891563129681168, 0.013894089770211386, 0.013408278126136618,
                        0.03336665227352387, 0.002430241846797234, 0.0035228540494572727, 0.009724515161045723, 0.015555669454596842, 0.021940278452343226,
                        0.008899920180129696, 0.003981848183691244, 0.010261816942671307, 0.01318203584761964, 0.007924711770956756, 0.017508075858796414,
                        0.00754009693790208, 0.02396586902906051 },
                { 0.017590569558430334, 0.012494152072132197, 0.06505015179612904, 0.010661347990588136, 0.005865624490709825, 0.01706014323531787,
                        0.033077339240295295, -0.00545492067459816, -0.004257968627745464, 0.005753167641120909, 0.021769507844239328, 0.02053550525218241,
                        0.008115846426667728, 0.007030917918202513, 0.011403974271592995, -1.1968542425288143E-4, 0.034187953223910356, 0.014059473032420408,
                        0.01765778947389243, 0.007488386888784544 },
                { 0.01672103191085612, 0.01891563129681168, 0.010661347990588136, 0.025908760714290772, 0.01181965443694628, 0.010285757376829598,
                        0.02417538638259604, 0.003982022177760808, 5.649112511131918E-4, 0.01078495204115518, 0.019567814831825632, 0.01572802612300913,
                        0.010262395763195575, 0.014228888439641704, 0.011045638308162718, 0.00508217802122697, 0.008890206419396685, 0.01069663043546007,
                        0.01523762573844658, 0.020696322027017818 },
                { 0.01755632403955279, 0.013894089770211386, 0.005865624490709825, 0.01181965443694628, 0.04387138117868266, 0.011036201036393327,
                        0.01566143317741111, -0.0017589255719603067, -0.005272928346857749, 0.017996287827954132, 8.671496540116026E-4, 0.016914486925711292,
                        0.0048007221182065326, 0.005261882412570885, 0.010052376671615942, 0.016659286051292577, -0.004428915437565974, 0.0074091709094127375,
                        0.007205803949615143, 0.012918531207864934 },
                { 0.012262007474096198, 0.013408278126136618, 0.01706014323531787, 0.010285757376829598, 0.011036201036393327, 0.029307520343741927,
                        0.0191901669847533, 0.006033007872740154, 0.0036059357703206504, 0.009949006260278406, 0.009517288863072777, 0.010260707780304752,
                        0.009430523432626849, 0.008246402583601598, 0.010550995236378848, 0.009595714479419879, 0.010409963539959625, 0.011047523101924736,
                        0.00939938810060992, 0.011683362866361038 },
                { 0.03483071434392497, 0.03336665227352387, 0.033077339240295295, 0.02417538638259604, 0.01566143317741111, 0.0191901669847533,
                        0.08968768565121141, 0.004549978039767166, -0.0011075972828496884, 0.009985737893322532, 0.030278712700881318, 0.04087939766119492,
                        0.01682797115962829, 0.015581996190322998, 0.02048149693990213, 0.03384577542062634, 0.03602951860237752, 0.01867672444585223,
                        0.03634263058620054, 0.03463901608655406 },
                { 0.01307003508704781, 0.002430241846797234, -0.00545492067459816, 0.003982022177760808, -0.0017589255719603067, 0.006033007872740154,
                        0.004549978039767166, 0.023709648931160177, 0.020495754079972485, 0.009098091744889007, -9.311015941926178E-4, 0.00541658477850558,
                        0.013123457114001903, 0.006231161126801938, 0.0029997191341117266, 0.0038815389652371293, 0.008028768372252833, 0.009945996448204816,
                        0.007087987964531582, 0.0035076640762001475 },
                { 0.005202737936835605, 0.0035228540494572727, -0.004257968627745464, 5.649112511131918E-4, -0.005272928346857749, 0.0036059357703206504,
                        -0.0011075972828496884, 0.020495754079972485, 0.02357089990619829, 0.006306182100948804, -0.006124098827426066, 0.0034995594693629783,
                        0.013531450641355029, 0.0036378818196341487, -1.3325202264541727E-4, -1.3051295019579227E-4, 0.007828169941982996, 0.009461083625657948,
                        0.002210064019302967, -0.0031363208956410264 },
                { 0.019212532773270445, 0.009724515161045723, 0.005753167641120909, 0.01078495204115518, 0.017996287827954132, 0.009949006260278406,
                        0.009985737893322532, 0.009098091744889007, 0.006306182100948804, 0.04485390256742983, 0.0037658097368951384, 0.011904447476562344,
                        0.007844524086767562, 0.004222410256638925, 0.01563336914077596, 0.005248367848712948, -0.006251007700809657, 0.008237442260460446,
                        0.01126235298099557, 0.014930019733989969 },
                { 0.0032373487733811435, 0.015555669454596842, 0.021769507844239328, 0.019567814831825632, 8.671496540116026E-4, 0.009517288863072777,
                        0.030278712700881318, -9.311015941926178E-4, -0.006124098827426066, 0.0037658097368951384, 0.12054413753720775, -0.007788688021656533,
                        -0.003636435944943654, 8.053710502302343E-4, 0.012906875269653405, -4.720704176235305E-4, 0.007410732044651898, 0.0057267246840456195,
                        0.011884035452223034, 0.02361858284123764 },
                { 0.02742784802402331, 0.021940278452343226, 0.02053550525218241, 0.01572802612300913, 0.016914486925711292, 0.010260707780304752,
                        0.04087939766119492, 0.00541658477850558, 0.0034995594693629783, 0.011904447476562344, -0.007788688021656533, 0.08552019201267644,
                        0.022078679057397525, 0.012668190001884981, 0.008049423563410513, 0.02272983644849825, 0.023782529097678827, 0.013116745919009099,
                        0.03278157777009603, 0.01959696739753192 },
                { 0.01469847057079009, 0.008899920180129696, 0.008115846426667728, 0.010262395763195575, 0.0048007221182065326, 0.009430523432626849,
                        0.01682797115962829, 0.013123457114001903, 0.013531450641355029, 0.007844524086767562, -0.003636435944943654, 0.022078679057397525,
                        0.027475323023698726, 0.011259464553565357, 0.005162914488124204, 0.0018817860303838724, 0.020846561439125576, 0.010073434900466327,
                        0.01290954000467054, 0.007526190125097093 },
                { 0.016954568683183992, 0.003981848183691244, 0.007030917918202513, 0.014228888439641704, 0.005261882412570885, 0.008246402583601598,
                        0.015581996190322998, 0.006231161126801938, 0.0036378818196341487, 0.004222410256638925, 8.053710502302343E-4, 0.012668190001884981,
                        0.011259464553565357, 0.039533169745603294, 0.006169802828992263, -0.008077749651298382, 0.013648948999977383, 0.007309167256855088,
                        0.009825369792204802, 0.009991458603752586 },
                { 0.01210386425386745, 0.010261816942671307, 0.011403974271592995, 0.011045638308162718, 0.010052376671615942, 0.010550995236378848,
                        0.02048149693990213, 0.0029997191341117266, -1.3325202264541727E-4, 0.01563336914077596, 0.012906875269653405, 0.008049423563410513,
                        0.005162914488124204, 0.006169802828992263, 0.018593876970330925, 0.0077550886284123415, 0.007200656814036332, 0.009443676098385572,
                        0.015472820227027667, 0.021299198360080023 },
                { 0.011041948656750589, 0.01318203584761964, -1.1968542425288143E-4, 0.00508217802122697, 0.016659286051292577, 0.009595714479419879,
                        0.03384577542062634, 0.0038815389652371293, -1.3051295019579227E-4, 0.005248367848712948, -4.720704176235305E-4, 0.02272983644849825,
                        0.0018817860303838724, -0.008077749651298382, 0.0077550886284123415, 0.0954688433617368, 0.017431762058874794, 0.010137259063522776,
                        0.010905195306460426, 0.009479342970761851 },
                { 0.030678267401908477, 0.007924711770956756, 0.034187953223910356, 0.008890206419396685, -0.004428915437565974, 0.010409963539959625,
                        0.03602951860237752, 0.008028768372252833, 0.007828169941982996, -0.006251007700809657, 0.007410732044651898, 0.023782529097678827,
                        0.020846561439125576, 0.013648948999977383, 0.007200656814036332, 0.017431762058874794, 0.08530699452145411, 0.013499940818289344,
                        0.015214953591500251, 0.013933728739300818 },
                { 0.004527800043582497, 0.017508075858796414, 0.014059473032420408, 0.01069663043546007, 0.0074091709094127375, 0.011047523101924736,
                        0.01867672444585223, 0.009945996448204816, 0.009461083625657948, 0.008237442260460446, 0.0057267246840456195, 0.013116745919009099,
                        0.010073434900466327, 0.007309167256855088, 0.009443676098385572, 0.010137259063522776, 0.013499940818289344, 0.022906101356054396,
                        0.008027443861606297, 0.012701516990467458 },
                { 0.026694382890454317, 0.00754009693790208, 0.01765778947389243, 0.01523762573844658, 0.007205803949615143, 0.00939938810060992,
                        0.03634263058620054, 0.007087987964531582, 0.002210064019302967, 0.01126235298099557, 0.011884035452223034, 0.03278157777009603,
                        0.01290954000467054, 0.009825369792204802, 0.015472820227027667, 0.010905195306460426, 0.015214953591500251, 0.008027443861606297,
                        0.03978297426898423, 0.027828143533949194 },
                { 0.006961437730164128, 0.02396586902906051, 0.007488386888784544, 0.020696322027017818, 0.012918531207864934, 0.011683362866361038,
                        0.03463901608655406, 0.0035076640762001475, -0.0031363208956410264, 0.014930019733989969, 0.02361858284123764, 0.01959696739753192,
                        0.007526190125097093, 0.009991458603752586, 0.021299198360080023, 0.009479342970761851, 0.013933728739300818, 0.012701516990467458,
                        0.027828143533949194, 0.09177753896957352 } };
        public static final double[][] dataSetMtrx = {
                { 0.05, 0.05, 0.07, 0.04, 0.09, 0.04, 0.14, 0.05, 0.05, 0.11, 0.09, 0.12, 0.06, 0.1, 0.05, 0.05, 0.11, 0.11, 0.06, 0.09 },
                { 0.08, 0.62, 0.64, 0.2, 0.35, 0.1, 0.71, 0.38, 0.49, 0.35, 0.5, 0.33, 0.55, 0.48, 0.04, 0.19, 0.58, 0.55, 0.67, 0.63 },
                { 1.04, 1.13, 1.12, 1.1, 0.98, 0.91, 1.88, 0.31, 0.17, 1.03, 0.78, 1.43, 0.71, 0.61, 0.87, 0.69, 0.88, 0.82, 1.21, 1.69 } };
        public static final double[] optimisationSolution = { 0.0, 0.0, 0.0, 0.0, 0.15864260150899884, 0.0, 0.0, 0.0, 0.38265285466215915, 0.0, 0.0, 0.0, 0.0,
                0.12046798151975475, 0.3382365623090873, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0,
                0.0, 0.0, 0.0, 0.0, 0.0 };

        Case020A() {
            super(covarianceMtrx, dataSetMtrx, optimisationSolution, -0.37262238039708406, -0.37262238039708406);
        }

    }

    public static final class Case030B extends CaseData {

        public static final double[][] covarianceMtrx = {
                { 1.7968364332038715E-4, 0.006745983514862486, -0.0024043171847082574, -0.0040372524330148166, -0.0021126786727520396, 0.0013346448517579253,
                        -7.198923964332896E-4, 0.0, -0.003974804206902136, -2.1483922231064307E-4, -0.002687790324504905, -0.0024490119785168926,
                        -0.004951512922120711, 9.507864041874528E-4, -0.0018879465916660238, 1.085480140112422E-4, 3.64111570949766E-4, -0.006049445893625436,
                        -6.157857964463406E-4, -0.0033126840961996357, -0.004673896123587855, -0.0017915302633958576, -0.004825858505530162,
                        0.0027870132680284563, -0.003222655728513482, -0.0075442652074308916, 0.0024831331111637715, -0.0038318402854690572,
                        -7.998585739727498E-4, -0.0017335916654738826 },
                { 0.006745983514862486, 0.253268982873707, -0.09026689237162301, -0.15157327542548926, -0.07931771214798021, 0.05010746666629673,
                        -0.027027402990458003, 0.0, -0.14922873979551324, -0.008065852991801147, -0.10090951455267896, -0.09194489898737056,
                        -0.1858979700600548, 0.035696011558310814, -0.0708804449251434, 0.004075290881013756, 0.013670084876997314, -0.2271184038698722,
                        -0.023118859094516993, -0.12437032046964466, -0.17547521642625724, -0.0672606220572691, -0.18118044203565875, 0.10463470805909346,
                        -0.12099032508966526, -0.28323940777588275, 0.09322593155155257, -0.14386134942326884, -0.030029626818176648, -0.06508539442255903 },
                { -0.0024043171847082574, -0.09026689237162301, 0.03217177155282864, 0.05402180868736533, 0.028269404742685494, -0.0178586625541775,
                        0.009632761379393642, 0.0, 0.05318619927727815, 0.0028747281867488113, 0.03596487886533468, 0.03276982521444928, 0.0662553774454164,
                        -0.01272231600123333, 0.025262301844916884, -0.001452463065810792, -0.004872116854980944, 0.08094663738575343, 0.008239728141839237,
                        0.044326479321219525, 0.06254063286910524, 0.02397217086436296, 0.06457401642913034, -0.037292564701521994, 0.043121824587270825,
                        0.10094856798890744, -0.03322639446064562, 0.05127322263264312, 0.0107027756071814, 0.023196904045038057 },
                { -0.0040372524330148166, -0.15157327542548926, 0.05402180868736533, 0.09071169142992717, 0.04746907928919385, -0.02998769434657283,
                        0.01617502447803162, 0.0, 0.08930856286378755, 0.0048271515256071325, 0.060391073035471504, 0.05502604124694933, 0.11125390838325681,
                        -0.021362905674941167, 0.042419648387332476, -0.0024389294738665353, -0.008181102623151233, 0.13592300167735805, 0.013835887668896575,
                        0.07443160479189825, 0.10501622823268192, 0.04025330175333745, 0.10843062080000856, -0.06262048058058163, 0.07240878713434892,
                        0.16950960310672628, -0.05579269770632557, 0.08609635373348912, 0.01797175810867395, 0.03895149853350358 },
                { -0.0021126786727520396, -0.07931771214798021, 0.028269404742685494, 0.04746907928919385, 0.02484038664745226, -0.01569244513246884,
                        0.00846432810753456, 0.0, 0.046734827506330054, 0.002526029830311389, 0.031602416282743045, 0.028794915779292128, 0.05821874242480725,
                        -0.01117912638763554, 0.022198038874332846, -0.0012762824146560548, -0.004281139541879111, 0.07112798407944884, 0.007240266810575097,
                        0.038949772557354205, 0.05495458838930457, 0.02106439801155139, 0.05674132688957899, -0.032769056677807336, 0.03789123985599932,
                        0.08870372344857765, -0.029196104156270154, 0.04505389082114657, 0.009404551907025785, 0.020383169392757996 },
                { 0.0013346448517579253, 0.05010746666629673, -0.0178586625541775, -0.02998769434657283, -0.01569244513246884, 0.009913405847118799,
                        -0.00534717942582118, 0.0, -0.029523844649724847, -0.0015957716390540427, -0.019964229647820266, -0.018190644226825994,
                        -0.036778591015865926, 0.007062201967975322, -0.014023191829810185, 8.062673117682387E-4, 0.0027045290525810383, -0.04493376062905798,
                        -0.004573901818917056, -0.0246057832131655, -0.03471652335029253, -0.013307035624530375, -0.03584526165374811, 0.020701232684395214,
                        -0.02393707517038473, -0.056036902042555826, 0.01844408740719443, -0.028461944644794714, -0.005941148054214921, -0.01287668231019582 },
                { -7.198923964332896E-4, -0.027027402990458003, 0.009632761379393642, 0.01617502447803162, 0.00846432810753456, -0.00534717942582118,
                        0.0028842083389771954, 0.0, 0.015924829177454896, 8.607412435493791E-4, 0.010768480547603114, 0.00981182855339133, 0.019837957632682528,
                        -0.0038092721761337723, 0.007563951682508585, -4.348915042605954E-4, -0.0014587925007327376, 0.024236764242866506,
                        0.0024671111098460274, 0.013272082247282784, 0.018725701565892948, 0.007177665093862634, 0.019334530289991655, -0.011166011682181516,
                        0.012911388662920173, 0.030225673629189473, -0.009948532949496996, 0.015352052278555704, 0.003204588325261457, 0.00694553736463096 },
                { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                        0.0, 0.0 },
                { -0.003974804206902136, -0.14922873979551324, 0.05318619927727815, 0.08930856286378755, 0.046734827506330054, -0.029523844649724847,
                        0.015924829177454896, 0.0, 0.08792713789221311, 0.004752485139258291, 0.05945694383580432, 0.05417489836626768, 0.10953303277740596,
                        -0.02103246422097036, 0.041763501208511064, -0.0024012041094549855, -0.008054557316674055, 0.13382054450295652, 0.01362187413965791,
                        0.07328029662796878, 0.10339183707186511, 0.03963066362719055, 0.10675341581032682, -0.06165186690198123, 0.0712887679166915,
                        0.16688762833583617, -0.05492969618248991, 0.08476461521710128, 0.017693770929860166, 0.038348996701327694 },
                { -2.1483922231064307E-4, -0.008065852991801147, 0.0028747281867488113, 0.0048271515256071325, 0.002526029830311389, -0.0015957716390540427,
                        8.607412435493791E-4, 0.0, 0.004752485139258291, 2.5687308310385867E-4, 0.003213663594415705, 0.0029281676349130524,
                        0.005920289492086537, -0.001136810273236422, 0.0022573283245060807, -1.2978571940440003E-4, -4.3535096067522084E-4,
                        0.007233035946847038, 7.36265912543757E-4, 0.003960819481600613, 0.0055883562343062385, 0.0021420478871728072, 0.005770050457295393,
                        -0.0033322997686831817, 0.003853176826197067, 0.009020320604139244, -0.0029689646572059656, 0.004581549949323406, 9.563530147504985E-4,
                        0.0020727734496708275 },
                { -0.002687790324504905, -0.10090951455267896, 0.03596487886533468, 0.060391073035471504, 0.031602416282743045, -0.019964229647820266,
                        0.010768480547603114, 0.0, 0.05945694383580432, 0.003213663594415705, 0.04020520006721454, 0.036633444084375744, 0.07406700063403623,
                        -0.01422229898404945, 0.02824077077073716, -0.0016237109645168067, -0.005446547829031152, 0.09049038544098541, 0.009211206290518437,
                        0.04955264762764535, 0.06991428127000286, 0.026798531124100185, 0.07218740425618905, -0.041689422351687995, 0.04820596202440236,
                        0.11285057813457103, -0.03714384363157272, 0.05731842394784793, 0.01196465129696267, 0.025931859010643567 },
                { -0.0024490119785168926, -0.09194489898737056, 0.03276982521444928, 0.05502604124694933, 0.028794915779292128, -0.018190644226825994,
                        0.00981182855339133, 0.0, 0.05417489836626768, 0.0029281676349130524, 0.036633444084375744, 0.0333789963298163, 0.06748702460597834,
                        -0.012958816116134943, 0.02573191266801095, -0.0014794634706051155, -0.004962686543385585, 0.0824513861312471, 0.00839289967539603,
                        0.045150481605995045, 0.06370322518784102, 0.024417798937373676, 0.06577440811124761, -0.03798581079256633, 0.0439234330733883,
                        0.10282514045623302, -0.03384405292054767, 0.05222636064955927, 0.010901733694735523, 0.023628120379507456 },
                { -0.004951512922120711, -0.1858979700600548, 0.0662553774454164, 0.11125390838325681, 0.05821874242480725, -0.036778591015865926,
                        0.019837957632682528, 0.0, 0.10953303277740596, 0.005920289492086537, 0.07406700063403623, 0.06748702460597834, 0.13644803591950877,
                        -0.026200666234914335, 0.05202583703314419, -0.002991239959938095, -0.010033763314987718, 0.16670359616733021, 0.016969108996336215,
                        0.09128709662230665, 0.12879779497419253, 0.04936890796319032, 0.13298539760712624, -0.07680127114385495, 0.08880619954276234,
                        0.20789608877137752, -0.06842729510637756, 0.10559339925655882, 0.022041572575603213, 0.04777230344761578 },
                { 9.507864041874528E-4, 0.035696011558310814, -0.01272231600123333, -0.021362905674941167, -0.01117912638763554, 0.007062201967975322,
                        -0.0038092721761337723, 0.0, -0.02103246422097036, -0.001136810273236422, -0.01422229898404945, -0.012958816116134943,
                        -0.026200666234914335, 0.005031035489278383, -0.009989968580431321, 5.743760200777738E-4, 0.0019266769354686901, -0.03201031992806928,
                        -0.0032583976612104546, -0.01752889100995455, -0.0247316717691886, -0.00947978672766528, -0.025535772599006825, 0.014747331891565903,
                        -0.01705251070952532, -0.039920076509244895, 0.013139366267593544, -0.02027597826445218, -0.00423240893468553, -0.009173207730468114 },
                { -0.0018879465916660238, -0.0708804449251434, 0.025262301844916884, 0.042419648387332476, 0.022198038874332846, -0.014023191829810185,
                        0.007563951682508585, 0.0, 0.041763501208511064, 0.0022573283245060807, 0.02824077077073716, 0.02573191266801095, 0.05202583703314419,
                        -0.009989968580431321, 0.019836765701750105, -0.0011405203573217428, -0.0038257416571582633, 0.06356188323704987, 0.006470097523146726,
                        0.03480656632465298, 0.049108901028875905, 0.018823713678901018, 0.050705578699407505, -0.029283312064862312, 0.03386063297877093,
                        0.07926803754528317, -0.026090425412376873, 0.04026137089095329, 0.008404161005652104, 0.018214949428291625 },
                { 1.085480140112422E-4, 0.004075290881013756, -0.001452463065810792, -0.0024389294738665353, -0.0012762824146560548, 8.062673117682387E-4,
                        -4.348915042605954E-4, 0.0, -0.0024012041094549855, -1.2978571940440003E-4, -0.0016237109645168067, -0.0014794634706051155,
                        -0.002991239959938095, 5.743760200777738E-4, -0.0011405203573217428, 6.55745349329076E-5, 2.1996207987967826E-4, -0.0036545081426841283,
                        -3.720000553494861E-4, -0.00200121320474304, -0.002823529913658766, -0.001082274649707047, -0.002915331340096025, 0.001683652165979573,
                        -0.0019468265041150863, -0.004557537849900779, 0.001500076260485098, -0.002314838709354824, -4.83199572816961E-4,
                        -0.0010472735799223456 },
                { 3.64111570949766E-4, 0.013670084876997314, -0.004872116854980944, -0.008181102623151233, -0.004281139541879111, 0.0027045290525810383,
                        -0.0014587925007327376, 0.0, -0.008054557316674055, -4.3535096067522084E-4, -0.005446547829031152, -0.004962686543385585,
                        -0.010033763314987718, 0.0019266769354686901, -0.0038257416571582633, 2.1996207987967826E-4, 7.378369764192337E-4,
                        -0.012258618575404022, -0.0012478305179556108, -0.006712834780274624, -0.00947119965160738, -0.0036303632682135245,
                        -0.009779136760360359, 0.005647613553061476, -0.0065304009772723165, -0.015287725724936094, 0.005031829727378619, -0.007764854720153791,
                        -0.0016208362459990996, -0.003512956288267211 },
                { -0.006049445893625436, -0.2271184038698722, 0.08094663738575343, 0.13592300167735805, 0.07112798407944884, -0.04493376062905798,
                        0.024236764242866506, 0.0, 0.13382054450295652, 0.007233035946847038, 0.09049038544098541, 0.0824513861312471, 0.16670359616733021,
                        -0.03201031992806928, 0.06356188323704987, -0.0036545081426841283, -0.012258618575404022, 0.20366792961031555, 0.020731786092643874,
                        0.11152881159528125, 0.1573570147487227, 0.060315815034323164, 0.16247316327754166, -0.09383094453228624, 0.10849780816533794,
                        0.25399431654520466, -0.0836001492682927, 0.12900734898071387, 0.026929001863418595, 0.05836518443264647 },
                { -6.157857964463406E-4, -0.023118859094516993, 0.008239728141839237, 0.013835887668896575, 0.007240266810575097, -0.004573901818917056,
                        0.0024671111098460274, 0.0, 0.01362187413965791, 7.36265912543757E-4, 0.009211206290518437, 0.00839289967539603, 0.016969108996336215,
                        -0.0032583976612104546, 0.006470097523146726, -3.720000553494861E-4, -0.0012478305179556108, 0.020731786092643874, 0.002110332026321009,
                        0.011352751852410645, 0.016017700853489185, 0.0061396734266784, 0.016538484352003703, -0.00955124881288902, 0.011044219650625686,
                        0.025854614663376803, -0.00850983468658759, 0.013131928863638606, 0.0027411596287593726, 0.00594111464299016 },
                { -0.0033126840961996357, -0.12437032046964466, 0.044326479321219525, 0.07443160479189825, 0.038949772557354205, -0.0246057832131655,
                        0.013272082247282784, 0.0, 0.07328029662796878, 0.003960819481600613, 0.04955264762764535, 0.045150481605995045, 0.09128709662230665,
                        -0.01752889100995455, 0.03480656632465298, -0.00200121320474304, -0.006712834780274624, 0.11152881159528125, 0.011352751852410645,
                        0.06107331595924338, 0.08616889701135015, 0.03302901533908532, 0.08897050630966172, -0.051381941941332554, 0.05941353471076902,
                        0.13908760368135833, -0.04577954572877716, 0.0706445847728989, 0.014746354917146827, 0.03196084759523562 },
                { -0.004673896123587855, -0.17547521642625724, 0.06254063286910524, 0.10501622823268192, 0.05495458838930457, -0.03471652335029253,
                        0.018725701565892948, 0.0, 0.10339183707186511, 0.0055883562343062385, 0.06991428127000286, 0.06370322518784102, 0.12879779497419253,
                        -0.0247316717691886, 0.049108901028875905, -0.002823529913658766, -0.00947119965160738, 0.1573570147487227, 0.016017700853489185,
                        0.08616889701135015, 0.12157648058781866, 0.04660093817468899, 0.1255292966303167, -0.07249524925649228, 0.0838270965504647,
                        0.1962399651784509, -0.06459077748065775, 0.09967308724104412, 0.020805766285826276, 0.04509385058881016 },
                { -0.0017915302633958576, -0.0672606220572691, 0.02397217086436296, 0.04025330175333745, 0.02106439801155139, -0.013307035624530375,
                        0.007177665093862634, 0.0, 0.03963066362719055, 0.0021420478871728072, 0.026798531124100185, 0.024417798937373676, 0.04936890796319032,
                        -0.00947978672766528, 0.018823713678901018, -0.001082274649707047, -0.0036303632682135245, 0.060315815034323164, 0.0061396734266784,
                        0.03302901533908532, 0.04660093817468899, 0.01786239763037501, 0.04811607445040403, -0.027787830444065, 0.03213139025594802,
                        0.07521986522778476, -0.02475800264556113, 0.03820524622642901, 0.007974965418269926, 0.01728472261400825 },
                { -0.004825858505530162, -0.18118044203565875, 0.06457401642913034, 0.10843062080000856, 0.05674132688957899, -0.03584526165374811,
                        0.019334530289991655, 0.0, 0.10675341581032682, 0.005770050457295393, 0.07218740425618905, 0.06577440811124761, 0.13298539760712624,
                        -0.025535772599006825, 0.050705578699407505, -0.002915331340096025, -0.009779136760360359, 0.16247316327754166, 0.016538484352003703,
                        0.08897050630966172, 0.1255292966303167, 0.04811607445040403, 0.1296106305785008, -0.07485228725331904, 0.08655256689175628,
                        0.20262031505193095, -0.06669081739124344, 0.10291375826842361, 0.021482224152951927, 0.04655998692672384 },
                { 0.0027870132680284563, 0.10463470805909346, -0.037292564701521994, -0.06262048058058163, -0.032769056677807336, 0.020701232684395214,
                        -0.011166011682181516, 0.0, -0.06165186690198123, -0.0033322997686831817, -0.041689422351687995, -0.03798581079256633,
                        -0.07680127114385495, 0.014747331891565903, -0.029283312064862312, 0.001683652165979573, 0.005647613553061476, -0.09383094453228624,
                        -0.00955124881288902, -0.051381941941332554, -0.07249524925649228, -0.027787830444065, -0.07485228725331904, 0.043228436448813655,
                        -0.0499855418539141, -0.11701659005847705, 0.03851505233981973, -0.05943440103519157, -0.012406340482720189, -0.026889164109413773 },
                { -0.003222655728513482, -0.12099032508966526, 0.043121824587270825, 0.07240878713434892, 0.03789123985599932, -0.02393707517038473,
                        0.012911388662920173, 0.0, 0.0712887679166915, 0.003853176826197067, 0.04820596202440236, 0.0439234330733883, 0.08880619954276234,
                        -0.01705251070952532, 0.03386063297877093, -0.0019468265041150863, -0.0065304009772723165, 0.10849780816533794, 0.011044219650625686,
                        0.05941353471076902, 0.0838270965504647, 0.03213139025594802, 0.08655256689175628, -0.0499855418539141, 0.05779886111282119,
                        0.1353076386858501, -0.044535401205578085, 0.06872468644626202, 0.014345595223811005, 0.031092251962417508 },
                { -0.0075442652074308916, -0.28323940777588275, 0.10094856798890744, 0.16950960310672628, 0.08870372344857765, -0.056036902042555826,
                        0.030225673629189473, 0.0, 0.16688762833583617, 0.009020320604139244, 0.11285057813457103, 0.10282514045623302, 0.20789608877137752,
                        -0.039920076509244895, 0.07926803754528317, -0.004557537849900779, -0.015287725724936094, 0.25399431654520466, 0.025854614663376803,
                        0.13908760368135833, 0.1962399651784509, 0.07521986522778476, 0.20262031505193095, -0.11701659005847705, 0.1353076386858501,
                        0.31675636395332657, -0.104257763198677, 0.16088509121862998, 0.03358316371473106, 0.07278723340669742 },
                { 0.0024831331111637715, 0.09322593155155257, -0.03322639446064562, -0.05579269770632557, -0.029196104156270154, 0.01844408740719443,
                        -0.009948532949496996, 0.0, -0.05492969618248991, -0.0029689646572059656, -0.03714384363157272, -0.03384405292054767,
                        -0.06842729510637756, 0.013139366267593544, -0.026090425412376873, 0.001500076260485098, 0.005031829727378619, -0.0836001492682927,
                        -0.00850983468658759, -0.04577954572877716, -0.06459077748065775, -0.02475800264556113, -0.06669081739124344, 0.03851505233981973,
                        -0.044535401205578085, -0.104257763198677, 0.034315588964119556, -0.05295401024662924, -0.011053623315832553, -0.023957321803076578 },
                { -0.0038318402854690572, -0.14386134942326884, 0.05127322263264312, 0.08609635373348912, 0.04505389082114657, -0.028461944644794714,
                        0.015352052278555704, 0.0, 0.08476461521710128, 0.004581549949323406, 0.05731842394784793, 0.05222636064955927, 0.10559339925655882,
                        -0.02027597826445218, 0.04026137089095329, -0.002314838709354824, -0.007764854720153791, 0.12900734898071387, 0.013131928863638606,
                        0.0706445847728989, 0.09967308724104412, 0.03820524622642901, 0.10291375826842361, -0.05943440103519157, 0.06872468644626202,
                        0.16088509121862998, -0.05295401024662924, 0.08171584069654508, 0.017057369551226498, 0.03696967770445088 },
                { -7.998585739727498E-4, -0.030029626818176648, 0.0107027756071814, 0.01797175810867395, 0.009404551907025785, -0.005941148054214921,
                        0.003204588325261457, 0.0, 0.017693770929860166, 9.563530147504985E-4, 0.01196465129696267, 0.010901733694735523, 0.022041572575603213,
                        -0.00423240893468553, 0.008404161005652104, -4.83199572816961E-4, -0.0016208362459990996, 0.026929001863418595, 0.0027411596287593726,
                        0.014746354917146827, 0.020805766285826276, 0.007974965418269926, 0.021482224152951927, -0.012406340482720189, 0.014345595223811005,
                        0.03358316371473106, -0.011053623315832553, 0.017057369551226498, 0.0035605563563566225, 0.007717052769931537 },
                { -0.0017335916654738826, -0.06508539442255903, 0.023196904045038057, 0.03895149853350358, 0.020383169392757996, -0.01287668231019582,
                        0.00694553736463096, 0.0, 0.038348996701327694, 0.0020727734496708275, 0.025931859010643567, 0.023628120379507456, 0.04777230344761578,
                        -0.009173207730468114, 0.018214949428291625, -0.0010472735799223456, -0.003512956288267211, 0.05836518443264647, 0.00594111464299016,
                        0.03196084759523562, 0.04509385058881016, 0.01728472261400825, 0.04655998692672384, -0.026889164109413773, 0.031092251962417508,
                        0.07278723340669742, -0.023957321803076578, 0.03696967770445088, 0.007717052769931537, 0.016725729771862437 } };
        public static final double[][] dataSetMtrx = {
                { 0.05, 0.12, 0.08, 0.07, 0.08, 0.08, 0.05, 0.19, 0.11, 0.16, 0.08, 0.12, 0.05, 0.05, 0.19, 0.05, 0.05, 0.1, 0.08, 0.06, 0.07, 0.11, 0.05, 0.07,
                        0.04, 0.09, 0.05, 0.07, 0.07, 0.17 },
                { 0.16, 0.61, 0.23, 0.47, 0.05, 0.34, 0.5, 0.38, 0.85, 0.46, 0.35, 0.59, 0.6, 0.29, 0.76, 0.37, 0.11, 0.38, 0.28, 0.05, 0.54, 0.4, 0.3, 0.67,
                        0.21, 0.7, 0.21, 0.38, 0.37, 0.83 },
                { 1.11, 1.48, 1.66, 0.64, 1.75, 0.85, 0.09, 2.11, 1.38, 0.91, 1.27, 1.11, 0.0, 1.2, 0.85, 0.23, 0.8, 1.23, 1.77, 1.57, 1.14, 1.67, 0.19, 0.91,
                        0.47, 0.62, 0.43, 0.74, 1.44, 2.32 } };
        public static final double[] optimisationSolution = { 0.0, 0.12062708652718095, 0.0, 0.0, 0.0, 0.0, 0.0, 0.43968645673640950, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.43968645673640950, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0,
                0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };

        Case030B() {
            super(covarianceMtrx, dataSetMtrx, optimisationSolution, 0.18148166043548658, 0.18148166043548658);
        }

    }

    public static final class Case040B extends CaseData {

        public static final double[][] covarianceMtrx = {
                { 0.11423759083528717, 0.03529574204583191, 0.044048052988533405, 0.05796803304912017, 0.03024614795882693, 0.047677339681368495,
                        0.022300803815387335, 0.0945069813846437, 0.014826297451926613, -0.0042194666225330805, 0.03456090469873314, 0.013129932123324166,
                        0.10098228446904794, 0.03172896936207796, 0.04065114217427125, 0.018708278498595608, 0.014970831633909321, 0.022127475397152634,
                        0.026275696337464093, -0.003948643803924303, 0.07082107462033305, 0.0658366601288068, 0.050725663259258466, 0.024822645154161157,
                        0.041683611229193514, 0.06458034419192545, 0.028530735004718547, 0.03298807719882751, 0.05200394751545241, 0.016728980309436047,
                        0.046616864765327205, 0.05941143749220734, 0.07483002326526854, 0.021470119593532108, 0.042083419362192044, -0.029111890890972332,
                        0.05021019316054823, 0.002656326252966717, 0.04725639883263942, 0.049567735395679144 },
                { 0.03529574204583191, 0.12445530256212692, 0.03211006607706391, 0.03197540822603182, 0.008921454200339735, 0.02986453977024333,
                        0.02499508069979818, 0.07089820403868295, -0.0016292321260703102, -0.00732608301061315, -0.002876376325930557, 0.032213528241629284,
                        0.030923211666380047, -0.006234760845690931, 0.01527095115288592, 5.102620126248635E-4, 0.02662182422108269, 0.0380112081398036,
                        -0.010373359603696125, 0.016422990483992875, 0.022117239596419914, 0.009691181453777868, -0.021945635303864786, 0.01361349121480444,
                        0.011958692354888122, 0.02269693257165875, 2.9719693177579937E-4, 0.0036152679315530544, 0.0020106538529165888, -0.0056283741407999145,
                        0.02656977111291494, 0.03095384346196263, 0.03115847494092476, 0.09138587802305595, 0.0028115584151911454, -0.01655623912849832,
                        -0.005296165729409121, -0.014922728428734586, 0.011638614972424643, 0.01922773249088486 },
                { 0.044048052988533405, 0.03211006607706391, 0.08934999703340155, 0.03705482148499526, 0.017299066247381284, 0.036724650110265836,
                        0.04050012053511281, 0.05881993438530615, 0.0034953957066222877, -0.002005605541746385, 0.02359795560114026, 0.0723123262546953,
                        0.036303444297770306, 0.015937381641559327, -0.008596731648094097, 0.022172056727430375, 0.011051612599715517, 0.05191110385554176,
                        0.04851472012499791, 0.01259368060321454, 0.03366019334221206, 0.05972331400841309, 0.0064452206192276516, 0.02166929892310226,
                        0.03814907665959447, 0.039275477054221644, 0.04144855090481069, 0.03877637999156372, 0.038962578010259076, 0.01898917950869441,
                        0.013716128297221158, 0.03351501230894848, 0.07942808649836595, 0.02287207176388129, 0.017620534332066423, 0.007845232991027098,
                        0.02173381687935026, 0.005282021074004339, 0.046825194752539, 0.03953802848278956 },
                { 0.05796803304912017, 0.03197540822603182, 0.03705482148499526, 0.03874330648034364, 0.02081719703057394, 0.028809226357834054,
                        0.019097974642012617, 0.06468136040907116, 0.006156177055611507, -0.004993799769628678, 0.017355184433952456, 0.04158118860009863,
                        0.049203214103827476, 0.013844417534584336, 0.02235123640006018, 0.011676500577512787, 0.014860670675792154, 0.025227968994455438,
                        0.02689039716603544, -1.3704601315645373E-4, 0.037928855695355236, 0.057113120935497204, 0.03338769930328975, 0.017778504617330073,
                        0.0301435066162404, 0.04569030910463451, 0.01953583159695776, 0.0233627729259551, 0.0341760184116546, 0.009285185941754532,
                        0.02912326270484259, 0.04235564016214425, 0.042682166709852816, 0.027263527278438802, 0.019129899885334672, -0.00971908852928216,
                        0.03516100894179494, 0.00254392172776378, 0.03531404919000957, 0.03527967997107477 },
                { 0.03024614795882693, 0.008921454200339735, 0.017299066247381284, 0.02081719703057394, 0.04771480208367859, 0.014849990668564485,
                        0.01634760973230179, 0.03080207265112083, -8.78639361326244E-4, -0.006231876867518344, -0.0014558219647955437, 0.005220260615982615,
                        0.044939875442752485, 0.014737992216484098, 0.0182626096492644, 0.0023653627721559422, 0.015667535931036404, 0.0035879025637721535,
                        0.021124443568772616, -0.005177333134917284, 0.02696541575882995, 0.0511518908670498, 0.03671713255675548, 0.03781242949121847,
                        0.02548023892891345, 0.03587739845307688, -0.007814837930681249, 0.009109405263370285, 0.050414064203654185, 0.016115077305646222,
                        0.011847508709543146, 0.024987183164790004, 0.019413844326668876, 0.028922306654112627, 0.016220919717171108, 0.027085690893309367,
                        0.036466036425669146, 0.0028567251966046554, 0.016849486672585515, 0.020488434921039467 },
                { 0.047677339681368495, 0.02986453977024333, 0.036724650110265836, 0.028809226357834054, 0.014849990668564485, 0.03784318258425451,
                        0.02150790000919776, 0.04964019021956377, 0.008501170188146083, 4.837462184201102E-4, 0.023006299856337682, -0.02380650415984551,
                        0.05729937055851521, 0.014996846152706921, 0.02492061477258955, 0.010111784687892774, 0.019244763913463154, 0.024523939577038523,
                        0.015059945219934912, 0.01211590943139201, 0.0273412600094722, 0.04424546873330098, 0.010071328652315727, 0.018016028767881283,
                        0.019137865144769835, 0.03428254115459666, 0.016409659886959575, 0.020344238302234554, 0.03255614116521642, 0.011358229160216487,
                        0.019240907644382715, 0.03764622060389949, 0.04611298150742911, 0.03534667295223163, 0.019946492797574274, -0.015322402803329367,
                        0.01847377124541714, 0.0042783428957608935, 0.02718932296363568, 0.02577509127226036 },
                { 0.022300803815387335, 0.02499508069979818, 0.04050012053511281, 0.019097974642012617, 0.01634760973230179, 0.02150790000919776,
                        0.04827698371946098, 0.03925497273135471, 0.009257819996529007, 0.006147484982078632, 0.02466107423434494, 0.01808896282299442,
                        0.028104290227842568, 0.012303342353806902, 0.006687978797468554, 0.011413981480112105, 0.027673301958123585, 0.03699300103325232,
                        0.025276675459190344, 0.014949904162634543, 0.01724399352386856, 0.03410530227596977, -0.003111499046483724, 0.008884900822379032,
                        0.019778318946012053, 0.029133246246129167, 0.03314910985426561, 0.02348556639689422, 0.011183285326402405, 0.004432874083613435,
                        0.012701210048847559, 0.026776226361946077, 0.05936641819148798, 0.02588484889486488, 0.014954715120237137, -0.004683203605893482,
                        0.010936576822281642, 0.008813737055580913, 0.027572153989607752, 0.02282236486394741 },
                { 0.0945069813846437, 0.07089820403868295, 0.05881993438530615, 0.06468136040907116, 0.03080207265112083, 0.04964019021956377,
                        0.03925497273135471, 0.11730283079720126, 0.013201217219896571, -0.005363471484700739, 0.031924226772419524, 0.0756876991278218,
                        0.08530236841703563, 0.0225042003135513, 0.04526208056205902, 0.016067993316525725, 0.029543146387987093, 0.05371355782763075,
                        0.038639360683430386, 0.007256886182511498, 0.06019215129895734, 0.09207647340633393, 0.03936523989223399, 0.025004689946515425,
                        0.04826198605225175, 0.07921658230544397, 0.03505198729563819, 0.03681000640478329, 0.0472229871395588, 0.012227707641585279,
                        0.05318046481542839, 0.07490700029477473, 0.07824501788687714, 0.06378515655245237, 0.03299631120353101, -0.021181826438642196,
                        0.05543620660483181, 0.005086821279949997, 0.05746904155057801, 0.05591004503153245 },
                { 0.014826297451926613, -0.0016292321260703102, 0.0034953957066222877, 0.006156177055611507, -8.78639361326244E-4, 0.008501170188146083,
                        0.009257819996529007, 0.013201217219896571, 0.015080506648818118, 0.012814342429560387, 0.014534641883219027, -0.024577761735934366,
                        0.01694278533069256, 0.017244926517318782, 0.02285627247267461, 9.99836487243132E-4, 0.0046899020588221825, 0.021048522934857813,
                        0.022773641899567723, 0.009031513125518007, 0.006855539071104791, 0.014213745488325092, -6.185255981476717E-4, -0.0019163874246696656,
                        0.002563815173634479, 0.01289690129752545, 0.009619424247633328, 0.0075196145097158685, -0.0042892129468472965, 0.0012104835302798936,
                        0.0038502214814615894, 6.698329372330263E-4, 0.016949689501183234, -0.02676579377758475, 0.020272571503783324, -0.011402675902423573,
                        -8.29949259254447E-4, 0.017348921987971493, 0.015570583821289027, 0.0030120166682332753 },
                { -0.0042194666225330805, -0.00732608301061315, -0.002005605541746385, -0.004993799769628678, -0.006231876867518344, 4.837462184201102E-4,
                        0.006147484982078632, -0.005363471484700739, 0.012814342429560387, 0.014656912996283317, 0.008399627692590951, -0.03561826155632546,
                        6.638040649468997E-4, 0.013438486271057107, 0.013109581986486995, -0.0017767031333220554, 0.0012174122098295894, 0.017456903839315654,
                        0.018093861340230766, 0.010696158215762626, -0.003868916637659621, -0.002846668680941308, -0.012004263678554493, -0.004608203002229871,
                        -0.006039491561324153, -6.432747059442158E-4, 0.004515530436644883, 0.0019374524570577875, -0.01403139789236411, -0.0013429726308514967,
                        -0.006494477730471548, -0.013567632726882516, 0.00802580760344967, -0.03510199799206229, 0.014751089010379536, -0.006284093973439056,
                        -0.013763809666256146, 0.016781781533798255, 0.006264469295454043, -0.006771266334106357 },
                { 0.03456090469873314, -0.002876376325930557, 0.02359795560114026, 0.017355184433952456, -0.0014558219647955437, 0.023006299856337682,
                        0.02466107423434494, 0.031924226772419524, 0.014534641883219027, 0.008399627692590951, 0.032907613599285554, -0.014606513156765784,
                        0.03572886593500993, 0.016170947636307498, 0.018782861481715308, 0.010817362138441864, 0.013664531694155244, 0.0251581236481739,
                        0.01847182921430062, 0.011158410922471993, 0.01832696130668024, 0.03160108447203802, 0.002123815787306502, -0.004142434862006321,
                        0.01257632906392454, 0.026080599162536168, 0.03138050677615261, 0.022835396931805, 0.006896364814518656, 0.003954023369421305,
                        0.014786838068377418, 0.02571166691262626, 0.04657305585151127, 0.0013339754567227866, 0.01910808939777914, -0.022295310362234862,
                        0.012160025490758715, 0.012070779844287963, 0.02563875987292235, 0.017301210250680762 },
                { 0.013129932123324166, 0.032213528241629284, 0.0723123262546953, 0.04158118860009863, 0.005220260615982615, -0.02380650415984551,
                        0.01808896282299442, 0.0756876991278218, -0.024577761735934366, -0.03561826155632546, -0.014606513156765784, 0.5210706358631969,
                        -0.0629125248959329, -0.03204829009101253, -0.04522114543420099, 0.015089857373055519, -0.04393937633609057, 0.03341458154478393,
                        0.03700948692516811, -0.03905290023401894, -0.003902960427117434, 0.06380416681071675, 0.016192821874924254, -0.03741018548156246,
                        0.0796393026031978, 0.029078751836244143, 0.06096022712407209, 0.010293849793353162, 0.009920987588152234, 0.018205666934178303,
                        0.045367848450588895, 0.047933809556820486, 0.012640488728025725, 0.045037701272249174, -0.03402902753784182, 0.045855164114116886,
                        0.08843349481632931, -0.02007244730552257, 0.04561031106664142, 0.050596875757000295 },
                { 0.10098228446904794, 0.030923211666380047, 0.036303444297770306, 0.049203214103827476, 0.044939875442752485, 0.05729937055851521,
                        0.028104290227842568, 0.08530236841703563, 0.01694278533069256, 6.638040649468997E-4, 0.03572886593500993, -0.0629125248959329,
                        0.12598636740790697, 0.03819565974088615, 0.05878253020522255, 0.010921756285095256, 0.027126013615866855, 0.02417891891903383,
                        0.017695739321827428, 0.009969434625075786, 0.06343703165601249, 0.0752993121482678, 0.036411698797323123, 0.04299566574394415,
                        0.03492515752114659, 0.06951048406658122, 0.012518304206139278, 0.027912603176972292, 0.06924284991304819, 0.023900966493056203,
                        0.03894217923450521, 0.06345282855424528, 0.07749673009125944, 0.052334130471718726, 0.04795648465578124, -0.01597501305300018,
                        0.0491476311627496, 0.00818064867347355, 0.03835617178471043, 0.03936815173706582 },
                { 0.03172896936207796, -0.006234760845690931, 0.015937381641559327, 0.013844417534584336, 0.014737992216484098, 0.014996846152706921,
                        0.012303342353806902, 0.0225042003135513, 0.017244926517318782, 0.013438486271057107, 0.016170947636307498, -0.03204829009101253,
                        0.03819565974088615, 0.029448803166982822, 0.02808297039918179, 0.0032216445566134854, 0.0034831771677052673, 0.02627852457795584,
                        0.037027205388486056, 0.00815934195455226, 0.022279015199860724, 0.031227880493592326, 0.011005581644748368, 0.013849551914424496,
                        0.012025107348250064, 0.026168684242878694, 0.007932758553023606, 0.01414799689219929, 0.015736794045720804, 0.010089318888050378,
                        0.004944710258536939, 0.0031139719744871343, 0.03249021987345137, -0.032678729706193614, 0.03287123082218088, 6.177358493974885E-4,
                        0.009566137463423413, 0.021657475109794103, 0.023965293037087054, 0.008907385037788441 },
                { 0.04065114217427125, 0.01527095115288592, -0.008596731648094097, 0.02235123640006018, 0.0182626096492644, 0.02492061477258955,
                        0.006687978797468554, 0.04526208056205902, 0.02285627247267461, 0.013109581986486995, 0.018782861481715308, -0.04522114543420099,
                        0.05878253020522255, 0.02808297039918179, 0.06832350285136185, -0.005447252162487134, 0.016218934763394425, 0.02828947837846918,
                        0.028095754067927206, 0.011772444267577753, 0.01826502460851273, 0.05603513553141615, 0.0168853555878979, 0.01190117824636687,
                        0.012448851168357599, 0.042977505590523005, -0.0036728753981352014, 0.0063170689833977264, 0.019710635235747717, 0.007514771380440481,
                        0.02199261774265877, 0.0273686806342315, 0.011713627068513541, 0.003893499050943835, 0.035554412597114886, -0.013242610060189314,
                        0.025189473493481492, 0.02543086791729817, 0.026314876316491605, 0.011062016719370517 },
                { 0.018708278498595608, 5.102620126248635E-4, 0.022172056727430375, 0.011676500577512787, 0.0023653627721559422, 0.010111784687892774,
                        0.011413981480112105, 0.016067993316525725, 9.99836487243132E-4, -0.0017767031333220554, 0.010817362138441864, 0.015089857373055519,
                        0.010921756285095256, 0.0032216445566134854, -0.005447252162487134, 0.009570477634144378, 0.004804472087725519, 0.006079609281707624,
                        0.010505179493922634, -7.022514159837804E-4, 0.012657723333581242, 0.013251448776828687, 0.011823034999153572, 0.002630950519838368,
                        0.010266879081743186, 0.010072776019056788, 0.017202930818738107, 0.013014158776799862, 0.009608066929318962, 0.0030681112471197614,
                        0.0065458665277700525, 0.012756226513091464, 0.023744101266213354, -0.0012075668724275753, 0.003832015196023262, -0.007802916851730192,
                        0.007955798950017464, -7.355199892693958E-4, 0.013800176677905836, 0.015143162635439546 },
                { 0.014970831633909321, 0.02662182422108269, 0.011051612599715517, 0.014860670675792154, 0.015667535931036404, 0.019244763913463154,
                        0.027673301958123585, 0.029543146387987093, 0.0046899020588221825, 0.0012174122098295894, 0.013664531694155244, -0.04393937633609057,
                        0.027126013615866855, 0.0034831771677052673, 0.016218934763394425, 0.004804472087725519, 0.03482315077432396, 0.014633466527603497,
                        0.007103269007426555, 0.010524763247820862, 0.013534639682284383, 0.02897187025560692, 0.01461764604251186, 0.015361319072107086,
                        0.0052730880819263325, 0.024881525212763954, 0.00897269528469963, 0.013309234827118165, 0.013135328653796036, -0.0037348958091225513,
                        0.012037957120993491, 0.0280043926493969, 0.025106286927439832, 0.03752496593459457, 0.006213468129765748, -0.013538436993519823,
                        0.007285828908618468, 0.002345777973282968, 0.013672895663618182, 0.016927638972885278 },
                { 0.022127475397152634, 0.0380112081398036, 0.05191110385554176, 0.025227968994455438, 0.0035879025637721535, 0.024523939577038523,
                        0.03699300103325232, 0.05371355782763075, 0.021048522934857813, 0.017456903839315654, 0.0251581236481739, 0.03341458154478393,
                        0.02417891891903383, 0.02627852457795584, 0.02828947837846918, 0.006079609281707624, 0.014633466527603497, 0.07863254279953086,
                        0.05987009426631798, 0.027707807274184808, 0.018521433339591744, 0.059149801137816364, -0.018709327232539708, 0.005806497343654885,
                        0.021080316256436523, 0.042821485352849444, 0.029445210393548124, 0.029912299249735312, 0.00232277856868693, 0.0056705762435304,
                        0.011021861648712052, 0.015593709119116565, 0.058935868471591005, 0.0026846346068200994, 0.03016725898030232, 0.003409853434532739,
                        0.005036536481907754, 0.02838986839135397, 0.04376711103334599, 0.016282358655623597 },
                { 0.026275696337464093, -0.010373359603696125, 0.04851472012499791, 0.02689039716603544, 0.021124443568772616, 0.015059945219934912,
                        0.025276675459190344, 0.038639360683430386, 0.022773641899567723, 0.018093861340230766, 0.01847182921430062, 0.03700948692516811,
                        0.017695739321827428, 0.037027205388486056, 0.028095754067927206, 0.010505179493922634, 0.007103269007426555, 0.05987009426631798,
                        0.0944808683306195, 0.009444078124966943, 0.023775332465795034, 0.07760794306637651, 0.03382366117232883, 0.018424255077429706,
                        0.02984711275851978, 0.04589582002339423, 0.025004946982346586, 0.031794296897754565, 0.02233281714793146, 0.013362331201211771,
                        0.004444450772757115, 0.005126717367653533, 0.04002509206842832, -0.0668837918846804, 0.039152838789894506, 0.014378676001947505,
                        0.02171965224597812, 0.03994361402837142, 0.05642160206076487, 0.026405512172605443 },
                { -0.003948643803924303, 0.016422990483992875, 0.01259368060321454, -1.3704601315645373E-4, -0.005177333134917284, 0.01211590943139201,
                        0.014949904162634543, 0.007256886182511498, 0.009031513125518007, 0.010696158215762626, 0.011158410922471993, -0.03905290023401894,
                        0.009969434625075786, 0.00815934195455226, 0.011772444267577753, -7.022514159837804E-4, 0.010524763247820862, 0.027707807274184808,
                        0.009444078124966943, 0.019442823563713515, -0.0026099850069375274, 0.006279111528365076, -0.025387881410626582, 8.647800510829043E-4,
                        -0.004151949549051522, 0.005502048628703777, 0.006301015620888986, 0.006518207758125864, -0.006609744666688612, -3.866105188219878E-4,
                        -0.003305715885100047, 1.4071781971422778E-4, 0.020650660503406002, 0.008806861319781468, 0.0094830278157249, -0.004909126464269095,
                        -0.014062473080685273, 0.010518913547028441, 0.006529385467707148, -0.0035483503724218694 },
                { 0.07082107462033305, 0.022117239596419914, 0.03366019334221206, 0.037928855695355236, 0.02696541575882995, 0.0273412600094722,
                        0.01724399352386856, 0.06019215129895734, 0.006855539071104791, -0.003868916637659621, 0.01832696130668024, -0.003902960427117434,
                        0.06343703165601249, 0.022279015199860724, 0.01826502460851273, 0.012657723333581242, 0.013534639682284383, 0.018521433339591744,
                        0.023775332465795034, -0.0026099850069375274, 0.05472625482659504, 0.04839765064403019, 0.04001215717082934, 0.027558522407069845,
                        0.026616527271625418, 0.04799923417205339, 0.014332441759093022, 0.027406695649538824, 0.0389890712465041, 0.008999200383767558,
                        0.026185092995007157, 0.03444725620587878, 0.05501622756495268, 0.016405750708487994, 0.0272334240813123, -0.006235660626244104,
                        0.031563415572385484, 9.003138068029192E-4, 0.030197299030138534, 0.031734391733064984 },
                { 0.0658366601288068, 0.009691181453777868, 0.05972331400841309, 0.057113120935497204, 0.0511518908670498, 0.04424546873330098,
                        0.03410530227596977, 0.09207647340633393, 0.014213745488325092, -0.002846668680941308, 0.03160108447203802, 0.06380416681071675,
                        0.0752993121482678, 0.031227880493592326, 0.05603513553141615, 0.013251448776828687, 0.02897187025560692, 0.059149801137816364,
                        0.07760794306637651, 0.006279111528365076, 0.04839765064403019, 0.1561106223600582, 0.05966964132803325, 0.037764218354188424,
                        0.05762083634500659, 0.09613865110455612, 0.02464628687331595, 0.04736246368637771, 0.07795792845537551, 0.02499120061074222,
                        0.03761971735742673, 0.07382605270946345, 0.053809807285986735, 0.05131261611764763, 0.033760357638912476, 0.025399438295868847,
                        0.08005296991573395, 0.026156870854088944, 0.0691461377325768, 0.051246727643268056 },
                { 0.050725663259258466, -0.021945635303864786, 0.0064452206192276516, 0.03338769930328975, 0.03671713255675548, 0.010071328652315727,
                        -0.003111499046483724, 0.03936523989223399, -6.185255981476717E-4, -0.012004263678554493, 0.002123815787306502, 0.016192821874924254,
                        0.036411698797323123, 0.011005581644748368, 0.0168853555878979, 0.011823034999153572, 0.01461764604251186, -0.018709327232539708,
                        0.03382366117232883, -0.025387881410626582, 0.04001215717082934, 0.05966964132803325, 0.10049757509059384, 0.033113533504320176,
                        0.026037095096607038, 0.04327851965565181, 0.0022887455199398932, 0.016577238077342505, 0.05117944711152918, 0.006849986104849175,
                        0.024082976029043277, 0.036485681763442884, 0.004642180782191108, -0.021917607456965926, 0.01347929697145816, -0.006666706988886162,
                        0.05191366029383879, 0.0011789191816211748, 0.02982277931100131, 0.041044461086146004 },
                { 0.024822645154161157, 0.01361349121480444, 0.02166929892310226, 0.017778504617330073, 0.03781242949121847, 0.018016028767881283,
                        0.008884900822379032, 0.025004689946515425, -0.0019163874246696656, -0.004608203002229871, -0.004142434862006321, -0.03741018548156246,
                        0.04299566574394415, 0.013849551914424496, 0.01190117824636687, 0.002630950519838368, 0.015361319072107086, 0.005806497343654885,
                        0.018424255077429706, 8.647800510829043E-4, 0.027558522407069845, 0.037764218354188424, 0.033113533504320176, 0.043830947771290346,
                        0.014325250310812172, 0.029382124011393992, -0.013016959208124144, 0.009766801497389746, 0.047071170142424415, 0.012669159733859502,
                        0.00461089895259566, 0.01700878550749971, 0.022083891436107447, 0.02234713767423641, 0.015218387862358209, 0.01937923345588322,
                        0.019862180013331718, 8.118836793493425E-4, 0.01233503496856383, 0.016758942009716286 },
                { 0.041683611229193514, 0.011958692354888122, 0.03814907665959447, 0.0301435066162404, 0.02548023892891345, 0.019137865144769835,
                        0.019778318946012053, 0.04826198605225175, 0.002563815173634479, -0.006039491561324153, 0.01257632906392454, 0.0796393026031978,
                        0.03492515752114659, 0.012025107348250064, 0.012448851168357599, 0.010266879081743186, 0.0052730880819263325, 0.021080316256436523,
                        0.02984711275851978, -0.004151949549051522, 0.026616527271625418, 0.05762083634500659, 0.026037095096607038, 0.014325250310812172,
                        0.034772396215495834, 0.03841841889427053, 0.01986972834524124, 0.01963608640344849, 0.03655827108182681, 0.015490322324714333,
                        0.0215929280529298, 0.03441917602007141, 0.03539354436426657, 0.021582354107778035, 0.014245794404883372, 0.010950732984901916,
                        0.0405849797907682, 0.004023758328516705, 0.03208273302181777, 0.029917763443526812 },
                { 0.06458034419192545, 0.02269693257165875, 0.039275477054221644, 0.04569030910463451, 0.03587739845307688, 0.03428254115459666,
                        0.029133246246129167, 0.07921658230544397, 0.01289690129752545, -6.432747059442158E-4, 0.026080599162536168, 0.029078751836244143,
                        0.06951048406658122, 0.026168684242878694, 0.042977505590523005, 0.010072776019056788, 0.024881525212763954, 0.042821485352849444,
                        0.04589582002339423, 0.005502048628703777, 0.04799923417205339, 0.09613865110455612, 0.04327851965565181, 0.029382124011393992,
                        0.03841841889427053, 0.07233558306079778, 0.020202948526286374, 0.03395858257309253, 0.049832135544160786, 0.013379801260914656,
                        0.03440017633632742, 0.05531508934940564, 0.05590728711226825, 0.03882738944239594, 0.031149893503581955, 0.0033026735256666134,
                        0.05294818779253418, 0.014629860589575615, 0.047200840625246265, 0.038990193758073534 },
                { 0.028530735004718547, 2.9719693177579937E-4, 0.04144855090481069, 0.01953583159695776, -0.007814837930681249, 0.016409659886959575,
                        0.03314910985426561, 0.03505198729563819, 0.009619424247633328, 0.004515530436644883, 0.03138050677615261, 0.06096022712407209,
                        0.012518304206139278, 0.007932758553023606, -0.0036728753981352014, 0.017202930818738107, 0.00897269528469963, 0.029445210393548124,
                        0.025004946982346586, 0.006301015620888986, 0.014332441759093022, 0.02464628687331595, 0.0022887455199398932, -0.013016959208124144,
                        0.01986972834524124, 0.020202948526286374, 0.04878909999296695, 0.026223067203438324, -0.0016698560746309724, 0.0025910424425008164,
                        0.01580359508394169, 0.025198651252298483, 0.05499594471605172, -0.007000304867496034, 0.009915041593472165, -0.022669016432465588,
                        0.012935155092187942, 0.00713553810847091, 0.03153230242214822, 0.025860683409828725 },
                { 0.03298807719882751, 0.0036152679315530544, 0.03877637999156372, 0.0233627729259551, 0.009109405263370285, 0.020344238302234554,
                        0.02348556639689422, 0.03681000640478329, 0.0075196145097158685, 0.0019374524570577875, 0.022835396931805, 0.010293849793353162,
                        0.027912603176972292, 0.01414799689219929, 0.0063170689833977264, 0.013014158776799862, 0.013309234827118165, 0.029912299249735312,
                        0.031794296897754565, 0.006518207758125864, 0.027406695649538824, 0.04736246368637771, 0.016577238077342505, 0.009766801497389746,
                        0.01963608640344849, 0.03395858257309253, 0.026223067203438324, 0.02936875343034468, 0.020078647247856157, 0.0054668714273115775,
                        0.012578610044099048, 0.02488808789857587, 0.046597465762590794, 0.0074481373387791285, 0.014984329764399649, -0.002783834610891941,
                        0.018163144395079402, 0.008499743552179553, 0.030305732141753784, 0.023028898745785755 },
                { 0.05200394751545241, 0.0020106538529165888, 0.038962578010259076, 0.0341760184116546, 0.050414064203654185, 0.03255614116521642,
                        0.011183285326402405, 0.0472229871395588, -0.0042892129468472965, -0.01403139789236411, 0.006896364814518656, 0.009920987588152234,
                        0.06924284991304819, 0.015736794045720804, 0.019710635235747717, 0.009608066929318962, 0.013135328653796036, 0.00232277856868693,
                        0.02233281714793146, -0.006609744666688612, 0.0389890712465041, 0.07795792845537551, 0.05117944711152918, 0.047071170142424415,
                        0.03655827108182681, 0.049832135544160786, -0.0016698560746309724, 0.020078647247856157, 0.07871705531603881, 0.026483728852549494,
                        0.019515843069521425, 0.04744512673809885, 0.033243350730701865, 0.048894060080586096, 0.01684830038911021, 0.024924855281312542,
                        0.05488406720800994, -0.0019511262677113295, 0.027284508498335316, 0.034521631158377176 },
                { 0.016728980309436047, -0.0056283741407999145, 0.01898917950869441, 0.009285185941754532, 0.016115077305646222, 0.011358229160216487,
                        0.004432874083613435, 0.012227707641585279, 0.0012104835302798936, -0.0013429726308514967, 0.003954023369421305, 0.018205666934178303,
                        0.023900966493056203, 0.010089318888050378, 0.007514771380440481, 0.0030681112471197614, -0.0037348958091225513, 0.0056705762435304,
                        0.013362331201211771, -3.866105188219878E-4, 0.008999200383767558, 0.02499120061074222, 0.006849986104849175, 0.012669159733859502,
                        0.015490322324714333, 0.013379801260914656, 0.0025910424425008164, 0.0054668714273115775, 0.026483728852549494, 0.014782170736792144,
                        0.003388436345598674, 0.010397665097994948, 0.014994139810391683, 0.004468396941413408, 0.010443502263986352, 0.013620616499308262,
                        0.018030513325208438, 0.004124254775079791, 0.011630864741561763, 0.009186624910400865 },
                { 0.046616864765327205, 0.02656977111291494, 0.013716128297221158, 0.02912326270484259, 0.011847508709543146, 0.019240907644382715,
                        0.012701210048847559, 0.05318046481542839, 0.0038502214814615894, -0.006494477730471548, 0.014786838068377418, 0.045367848450588895,
                        0.03894217923450521, 0.004944710258536939, 0.02199261774265877, 0.0065458665277700525, 0.012037957120993491, 0.011021861648712052,
                        0.004444450772757115, -0.003305715885100047, 0.026185092995007157, 0.03761971735742673, 0.024082976029043277, 0.00461089895259566,
                        0.0215929280529298, 0.03440017633632742, 0.01580359508394169, 0.012578610044099048, 0.019515843069521425, 0.003388436345598674,
                        0.030106053675675953, 0.040453583441481436, 0.026588144434426253, 0.035694663500834584, 0.009971198671878552, -0.016489748920360963,
                        0.032413449915873624, -0.00297510355827624, 0.021294878042261687, 0.025970841850782325 },
                { 0.05941143749220734, 0.03095384346196263, 0.03351501230894848, 0.04235564016214425, 0.024987183164790004, 0.03764622060389949,
                        0.026776226361946077, 0.07490700029477473, 6.698329372330263E-4, -0.013567632726882516, 0.02571166691262626, 0.047933809556820486,
                        0.06345282855424528, 0.0031139719744871343, 0.0273686806342315, 0.012756226513091464, 0.0280043926493969, 0.015593709119116565,
                        0.005126717367653533, 1.4071781971422778E-4, 0.03444725620587878, 0.07382605270946345, 0.036485681763442884, 0.01700878550749971,
                        0.03441917602007141, 0.05531508934940564, 0.025198651252298483, 0.02488808789857587, 0.04744512673809885, 0.010397665097994948,
                        0.040453583441481436, 0.07245948941370249, 0.045466353313958575, 0.08148076318597057, 0.007780115617141403, -0.015476145162468782,
                        0.05389283558685681, -0.006801782784101082, 0.032611326036758745, 0.04261669060816519 },
                { 0.07483002326526854, 0.03115847494092476, 0.07942808649836595, 0.042682166709852816, 0.019413844326668876, 0.04611298150742911,
                        0.05936641819148798, 0.07824501788687714, 0.016949689501183234, 0.00802580760344967, 0.04657305585151127, 0.012640488728025725,
                        0.07749673009125944, 0.03249021987345137, 0.011713627068513541, 0.023744101266213354, 0.025106286927439832, 0.058935868471591005,
                        0.04002509206842832, 0.020650660503406002, 0.05501622756495268, 0.053809807285986735, 0.004642180782191108, 0.022083891436107447,
                        0.03539354436426657, 0.05590728711226825, 0.05499594471605172, 0.046597465762590794, 0.033243350730701865, 0.014994139810391683,
                        0.026588144434426253, 0.045466353313958575, 0.1198984737712395, 0.02423075251573894, 0.038545676064494036, -0.016817359670602446,
                        0.02299219447798801, 0.010660358182798438, 0.04930827812095789, 0.041877509291861584 },
                { 0.021470119593532108, 0.09138587802305595, 0.02287207176388129, 0.027263527278438802, 0.028922306654112627, 0.03534667295223163,
                        0.02588484889486488, 0.06378515655245237, -0.02676579377758475, -0.03510199799206229, 0.0013339754567227866, 0.045037701272249174,
                        0.052334130471718726, -0.032678729706193614, 0.003893499050943835, -0.0012075668724275753, 0.03752496593459457, 0.0026846346068200994,
                        -0.0668837918846804, 0.008806861319781468, 0.016405750708487994, 0.05131261611764763, -0.021917607456965926, 0.02234713767423641,
                        0.021582354107778035, 0.03882738944239594, -0.007000304867496034, 0.0074481373387791285, 0.048894060080586096, 0.004468396941413408,
                        0.035694663500834584, 0.08148076318597057, 0.02423075251573894, 0.22653810401963947, -0.032512052696747284, 0.017900871277740232,
                        0.04175353369266785, -0.04329437387906437, -0.01115043557427735, 0.01870380458771736 },
                { 0.042083419362192044, 0.0028115584151911454, 0.017620534332066423, 0.019129899885334672, 0.016220919717171108, 0.019946492797574274,
                        0.014954715120237137, 0.03299631120353101, 0.020272571503783324, 0.014751089010379536, 0.01910808939777914, -0.03402902753784182,
                        0.04795648465578124, 0.03287123082218088, 0.035554412597114886, 0.003832015196023262, 0.006213468129765748, 0.03016725898030232,
                        0.039152838789894506, 0.0094830278157249, 0.0272334240813123, 0.033760357638912476, 0.01347929697145816, 0.015218387862358209,
                        0.014245794404883372, 0.031149893503581955, 0.009915041593472165, 0.014984329764399649, 0.01684830038911021, 0.010443502263986352,
                        0.009971198671878552, 0.007780115617141403, 0.038545676064494036, -0.032512052696747284, 0.03826771777515098, -0.006287335831449384,
                        0.011307768186952345, 0.023070679419163076, 0.028248598049344036, 0.012960211889898801 },
                { -0.029111890890972332, -0.01655623912849832, 0.007845232991027098, -0.00971908852928216, 0.027085690893309367, -0.015322402803329367,
                        -0.004683203605893482, -0.021181826438642196, -0.011402675902423573, -0.006284093973439056, -0.022295310362234862, 0.045855164114116886,
                        -0.01597501305300018, 6.177358493974885E-4, -0.013242610060189314, -0.007802916851730192, -0.013538436993519823, 0.003409853434532739,
                        0.014378676001947505, -0.004909126464269095, -0.006235660626244104, 0.025399438295868847, -0.006666706988886162, 0.01937923345588322,
                        0.010950732984901916, 0.0033026735256666134, -0.022669016432465588, -0.002783834610891941, 0.024924855281312542, 0.013620616499308262,
                        -0.016489748920360963, -0.015476145162468782, -0.016817359670602446, 0.017900871277740232, -0.006287335831449384, 0.06667472176142819,
                        0.014177583885336423, 0.0016839733929161443, -0.006650164751187066, -0.012637399781958382 },
                { 0.05021019316054823, -0.005296165729409121, 0.02173381687935026, 0.03516100894179494, 0.036466036425669146, 0.01847377124541714,
                        0.010936576822281642, 0.05543620660483181, -8.29949259254447E-4, -0.013763809666256146, 0.012160025490758715, 0.08843349481632931,
                        0.0491476311627496, 0.009566137463423413, 0.025189473493481492, 0.007955798950017464, 0.007285828908618468, 0.005036536481907754,
                        0.02171965224597812, -0.014062473080685273, 0.031563415572385484, 0.08005296991573395, 0.05191366029383879, 0.019862180013331718,
                        0.0405849797907682, 0.05294818779253418, 0.012935155092187942, 0.018163144395079402, 0.05488406720800994, 0.018030513325208438,
                        0.032413449915873624, 0.05389283558685681, 0.02299219447798801, 0.04175353369266785, 0.011307768186952345, 0.014177583885336423,
                        0.06721526461943476, 4.3721296044137777E-4, 0.031058452199708653, 0.03536238282023866 },
                { 0.002656326252966717, -0.014922728428734586, 0.005282021074004339, 0.00254392172776378, 0.0028567251966046554, 0.0042783428957608935,
                        0.008813737055580913, 0.005086821279949997, 0.017348921987971493, 0.016781781533798255, 0.012070779844287963, -0.02007244730552257,
                        0.00818064867347355, 0.021657475109794103, 0.02543086791729817, -7.355199892693958E-4, 0.002345777973282968, 0.02838986839135397,
                        0.03994361402837142, 0.010518913547028441, 9.003138068029192E-4, 0.026156870854088944, 0.0011789191816211748, 8.118836793493425E-4,
                        0.004023758328516705, 0.014629860589575615, 0.00713553810847091, 0.008499743552179553, -0.0019511262677113295, 0.004124254775079791,
                        -0.00297510355827624, -0.006801782784101082, 0.010660358182798438, -0.04329437387906437, 0.023070679419163076, 0.0016839733929161443,
                        4.3721296044137777E-4, 0.02654361008380024, 0.020087366258058816, 1.7410648515830268E-4 },
                { 0.04725639883263942, 0.011638614972424643, 0.046825194752539, 0.03531404919000957, 0.016849486672585515, 0.02718932296363568,
                        0.027572153989607752, 0.05746904155057801, 0.015570583821289027, 0.006264469295454043, 0.02563875987292235, 0.04561031106664142,
                        0.03835617178471043, 0.023965293037087054, 0.026314876316491605, 0.013800176677905836, 0.013672895663618182, 0.04376711103334599,
                        0.05642160206076487, 0.006529385467707148, 0.030197299030138534, 0.0691461377325768, 0.02982277931100131, 0.01233503496856383,
                        0.03208273302181777, 0.047200840625246265, 0.03153230242214822, 0.030305732141753784, 0.027284508498335316, 0.011630864741561763,
                        0.021294878042261687, 0.032611326036758745, 0.04930827812095789, -0.01115043557427735, 0.028248598049344036, -0.006650164751187066,
                        0.031058452199708653, 0.020087366258058816, 0.04868175449915442, 0.0347751496870258 },
                { 0.049567735395679144, 0.01922773249088486, 0.03953802848278956, 0.03527967997107477, 0.020488434921039467, 0.02577509127226036,
                        0.02282236486394741, 0.05591004503153245, 0.0030120166682332753, -0.006771266334106357, 0.017301210250680762, 0.050596875757000295,
                        0.03936815173706582, 0.008907385037788441, 0.011062016719370517, 0.015143162635439546, 0.016927638972885278, 0.016282358655623597,
                        0.026405512172605443, -0.0035483503724218694, 0.031734391733064984, 0.051246727643268056, 0.041044461086146004, 0.016758942009716286,
                        0.029917763443526812, 0.038990193758073534, 0.025860683409828725, 0.023028898745785755, 0.034521631158377176, 0.009186624910400865,
                        0.025970841850782325, 0.04261669060816519, 0.041877509291861584, 0.01870380458771736, 0.012960211889898801, -0.012637399781958382,
                        0.03536238282023866, 1.7410648515830268E-4, 0.0347751496870258, 0.03920793115629167 } };
        public static final double[][] dataSetMtrx = {
                { 0.05, 0.05, 0.07, 0.04, 0.07, 0.09, 0.04, 0.14, 0.05, 0.05, 0.11, 0.09, 0.12, 0.06, 0.1, 0.05, 0.05, 0.08, 0.11, 0.11, 0.06, 0.09, 0.09, 0.14,
                        0.07, 0.15, 0.07, 0.06, 0.06, 0.05, 0.05, 0.09, 0.1, 0.04, 0.05, 0.08, 0.07, 0.04, 0.09, 0.12 },
                { 0.11, 0.63, 0.67, 0.21, 0.0, 0.35, 0.11, 0.72, 0.37, 0.49, 0.34, 0.5, 0.32, 0.54, 0.5, 0.06, 0.19, 0.01, 0.61, 0.55, 0.68, 0.01, 0.64, 0.56,
                        0.65, 0.55, 0.31, 0.57, 0.02, 0.03, 0.25, 0.23, 0.24, 0.26, 0.07, 0.55, 0.44, 0.22, 0.5, 0.31 },
                { 1.04, 1.13, 1.12, 1.1, 1.24, 0.98, 0.91, 1.88, 0.31, 0.17, 1.03, 0.78, 1.43, 0.71, 0.61, 0.87, 0.69, 1.6, 0.88, 0.82, 1.21, 1.56, 1.69, 1.42,
                        1.21, 1.89, 0.95, 0.93, 1.37, 0.85, 0.87, 1.81, 1.48, 0.92, 0.62, 0.72, 1.36, 0.25, 1.12, 1.45 } };
        public static final double[] optimisationSolution = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.45812725759818357, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.29767030499290004, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.06713941593467697, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.07963530212652209,
                0.0, 0.09742771934771734, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0 };

        Case040B() {
            super(covarianceMtrx, dataSetMtrx, optimisationSolution, -0.046208831442687004, -0.046208831442687010);
        }

    }
    public static final class Case050B extends CaseData {

        public static final double[][] covarianceMtrx = {
                { 0.14199030086211206, 0.06322267117779498, 0.05589722842371056, 0.08092409419978439, 0.03601695516722446, 0.06341423886760034,
                        0.022103960435791076, 0.13326872423757297, 0.031506141489438366, 0.0055921127727310985, 0.0406100587684226, 0.025946447741214906,
                        0.11947198318927393, 0.05077998079118238, 0.0816689949194069, 0.01827495627023137, 0.019257623272038657, 0.05736422711496372,
                        0.07185168949490216, 0.002532075036186415, 0.08346157982652175, 0.11131859328267968, 0.06760715083358035, 0.029470299100444058,
                        0.05558439313482155, 0.09272095375985491, 0.029233548846136864, 0.04146431678532974, 0.05827968731163341, 0.02024881721137161,
                        0.05987535887602487, 0.06833527370134565, 0.07510131226243622, -0.0034610303544831884, 0.0668336982118146, -0.03720551901508286,
                        0.060929539640763974, 0.025000355192702817, 0.07911682136469031, 0.06345414054903112, 0.042067438934267495, 0.07386102107101723,
                        0.022327898989038406, 0.08606120434337522, 0.060893122812509706, 0.026546698989913166, -0.017418404248637314, 0.06820397632512222,
                        0.07145051793685725, 0.07488933945702014 },
                { 0.06322267117779498, 0.1810655076329292, 0.05334275357636329, 0.04692310653977936, 0.016347368660472594, 0.05961533860803157,
                        0.04942345614333045, 0.10805439860008778, 0.006165184742381723, -0.002505200297125928, 0.011325295205237295, -0.01157515102700365,
                        0.07643219708715858, 0.0032066653389200553, 0.037023243241299186, 0.001628559089516492, 0.04692161945594038, 0.06480658158745242,
                        -0.017511710151868906, 0.03727219786400344, 0.03658218514016751, 0.0170129992110056, -0.04844605700882658, 0.024862643782799655,
                        0.016286529242773753, 0.03882674238717706, 0.005808335064625088, 0.009736332491718087, 0.00992513269127665, -0.002041142410520545,
                        0.03857111312346192, 0.05132914660906421, 0.07099624607549501, 0.14230302989226046, 0.01819949089098011, -0.0314052003041181,
                        -0.01175789159428585, -0.014921922876607185, 0.019613724599524256, 0.026661091708265507, 0.023964258241647315, 0.09472958537687795,
                        0.010921251146816234, 0.05873848188653275, -0.023685780975292402, -0.017486821187736125, 0.024363803663659582, -0.0012162700737118814,
                        0.06854255541487272, 0.034670324121890486 },
                { 0.05589722842371056, 0.05334275357636329, 0.09654089959069134, 0.04428479094667642, 0.029263486761936526, 0.04000573185340004,
                        0.04215954811740116, 0.07034365963776537, 0.012345002799823661, 0.007975929601203018, 0.018542710236303266, 0.008693933217183068,
                        0.050153924826700697, 0.03549665917687473, 0.0038456579098830705, 0.0196642371608017, 0.018063195401167596, 0.07676063345552421,
                        0.07805265348270753, 0.021147399844983532, 0.05652375650338057, 0.06946814862997584, 0.015237057579642395, 0.04534040085118005,
                        0.034623855387281285, 0.054936555452588506, 0.02708987410948711, 0.04721913230108773, 0.04196781961498533, 0.016019218970349393,
                        0.008294996667309703, 0.016857130293880886, 0.0931130325113982, -0.002376306446747096, 0.03929274211546738, 0.017111756371728126,
                        0.00786229681839231, 0.01789436213800264, 0.055585709836948514, 0.037333713645423545, 0.05536234155443809, 0.06864217828508612,
                        0.04172971348303587, 0.05852433263683626, 0.03678886162124004, 0.012535051844670613, 0.021558548648576707, 0.009587912563895358,
                        0.04882426463077484, 0.09610536340475627 },
                { 0.08092409419978439, 0.04692310653977936, 0.04428479094667642, 0.05301252151684236, 0.03116241168067785, 0.04227583841584239,
                        0.025612825402290935, 0.0907667647534229, 0.017709456902719443, 0.0025242542306910427, 0.02550949433635801, 0.01656884391489779,
                        0.07669194182196726, 0.030765547930930743, 0.05022346749733704, 0.011482747000572807, 0.023801862409117004, 0.04758019899429915,
                        0.05085578918543725, 0.007379161161471313, 0.05394220385138819, 0.0859495668671807, 0.04361050470212717, 0.029284956297791043,
                        0.03740823982999819, 0.06928950159324, 0.019233709253406273, 0.03190542530555318, 0.04448026111666185, 0.012362736864734786,
                        0.037583168521902646, 0.05112415966816555, 0.0575256825642667, 0.02156320707178575, 0.03970887872943984, -0.01333522917159062,
                        0.04239868460640808, 0.016129277667228197, 0.05276665293939613, 0.04321141547652689, 0.03165505385396111, 0.055528117288422246,
                        0.023943297575598703, 0.06304401538295411, 0.04481212196001175, 0.017521982040632544, -0.006765293855494431, 0.04016487958361241,
                        0.055289365700728135, 0.05723838827961253 },
                { 0.03601695516722446, 0.016347368660472594, 0.029263486761936526, 0.03116241168067785, 0.07044908975438953, 0.021595753353705185,
                        0.024660841752789125, 0.04660447745186112, 7.590023335203115E-4, -0.006854861418870664, -0.00425980824125894, 0.028954391758296533,
                        0.057398188986273, 0.022177315394533312, 0.0326283474837754, 0.0019186743579239483, 0.022508945852992547, 0.01528422712131023,
                        0.04434563109656779, -0.005103023686480352, 0.03167231278397138, 0.08698557056272502, 0.0517881055145056, 0.05382084693674011,
                        0.040692005026666485, 0.05443615089315773, -0.0116382507299386, 0.012630902416388595, 0.07328167166471125, 0.025419950047336628,
                        0.015972563956055362, 0.03554712416533306, 0.020593816787798494, 0.03752778234111384, 0.024286105490069175, 0.04532052601540752,
                        0.055109086655764354, 0.010851575069873512, 0.03120649507961279, 0.030650769411466612, 0.014062668145234797, 0.0359415224733912,
                        0.031258368682786544, 0.03699394538887053, 0.07534667993947596, 0.027236598545917534, -0.03842428850400834, 0.036436158995993465,
                        0.00863332250828171, 0.057612618094148096 },
                { 0.06341423886760034, 0.05961533860803157, 0.04000573185340004, 0.04227583841584239, 0.021595753353705185, 0.037822458313863244,
                        0.021655727182440115, 0.0763969227780546, 0.011917517837183512, 8.263036094592882E-4, 0.01748903131071005, 0.009387940459024482,
                        0.06293698807857667, 0.020940461345369508, 0.037487202744069234, 0.007701861141537793, 0.019990688099678158, 0.04245735832358022,
                        0.030495393467626305, 0.011179011467022701, 0.04185455547912513, 0.058514499570549106, 0.019207163290216652, 0.024260772565887795,
                        0.026837542613741792, 0.05109264635188824, 0.01223709128880002, 0.02261965494842901, 0.03212977978831147, 0.008861849172155503,
                        0.029759499287531833, 0.04037072069359275, 0.04961387510654315, 0.036541293860140364, 0.02900022014049821, -0.012193410934990946,
                        0.025069104652403, 0.007617889780041749, 0.0372827752145765, 0.03175404698799519, 0.024137621397914238, 0.05324076131132853,
                        0.01772750700613978, 0.050333259940161676, 0.02712682305922319, 0.008408173206506412, -0.0016489167342999417, 0.023919868130552267,
                        0.04648771327495464, 0.045564958659100915 },
                { 0.022103960435791076, 0.04942345614333045, 0.04215954811740116, 0.025612825402290935, 0.024660841752789125, 0.021655727182440115,
                        0.05736909751963818, 0.05048761154507203, 0.01103275174563427, 0.007993569121582889, 0.020255091234345594, -0.0036255326984672466,
                        0.02357417852660373, 0.013599439019096486, 0.01017293878231807, 0.011550956942541307, 0.04493816244535945, 0.048479134016584186,
                        0.04024200783535551, 0.01770506449186173, 0.025812698271660776, 0.04358810069837829, 0.010817413991423537, 0.019567881664452235,
                        0.01834928713662144, 0.03885619284599652, 0.028671687841984827, 0.02873764763689312, 0.008198931568664743, -0.004233061739110836,
                        0.01384033519505982, 0.0259238226874429, 0.05943095402731848, 0.02641899806396522, 0.017448403751967528, -0.00556763946329843,
                        0.004652879105268365, 0.011874781536065483, 0.03399811228132037, 0.02807274711585795, 0.03907832126098715, 0.035158626924776074,
                        0.025430471163154902, 0.042617052086210916, -7.120166755056671E-4, 0.004158695865907189, 0.03290478939550384, 0.015390742748574793,
                        0.04606648865075327, 0.03608010348517658 },
                { 0.13326872423757297, 0.10805439860008778, 0.07034365963776537, 0.0907667647534229, 0.04660447745186112, 0.0763969227780546,
                        0.05048761154507203, 0.16514514382582915, 0.028988265585153918, 0.0031075525072491277, 0.044944763538417205, 0.020906119358012034,
                        0.1324307283604126, 0.04392220869568371, 0.08993336530611888, 0.016626526502406536, 0.052225853063807684, 0.08663464179709535,
                        0.06862112273773749, 0.020525908043417255, 0.08732025297040837, 0.13892362179334494, 0.05905987369784156, 0.045957328306873775,
                        0.05757027944958483, 0.11846758660430416, 0.033018131820394245, 0.05113894564142611, 0.06555590344191373, 0.01367217439094696,
                        0.07018524492324246, 0.09741458821208013, 0.10081329808561068, 0.07405493590857007, 0.06086515210357006, -0.03495189242523619,
                        0.06574262289330716, 0.0211572038771267, 0.08381941739889459, 0.0721603971651063, 0.05143648206330431, 0.10265113631167797,
                        0.03795642722556948, 0.11180928914709924, 0.05954074594780366, 0.02170176560458393, -0.0021242473539373396, 0.06252769197217897,
                        0.10739084538421556, 0.08599748217024838 },
                { 0.031506141489438366, 0.006165184742381723, 0.012345002799823661, 0.017709456902719443, 7.590023335203115E-4, 0.011917517837183512,
                        0.01103275174563427, 0.028988265585153918, 0.01137609515879618, 0.005642719217845866, 0.01637844838699655, 0.0016125669410293098,
                        0.019550228022236507, 0.012853481029847482, 0.016042928746541882, 0.007534325645958795, 0.008295639927543282, 0.01635680772504871,
                        0.02326437797184387, 0.0013775769465331484, 0.01884706689371629, 0.021423728553515167, 0.01928222835395192, -8.06392242197756E-4,
                        0.01002299130965817, 0.020192288671497027, 0.017817122381597734, 0.014251372364943424, 0.0016862982525222361, -8.116635505841154E-4,
                        0.013621124466734644, 0.013477770974060568, 0.02325532057282806, -0.02059668928311294, 0.01665134330109888, -0.0203282709219231,
                        0.008907803366441557, 0.009849714425995422, 0.022575555940660567, 0.016765228593236293, 0.016775086630961195, 0.0104116838298043,
                        0.004750003818887418, 0.020785017783699532, 0.0026031549700070633, 0.005399496137037798, 0.009031437303592226, 0.018697635154368855,
                        0.0213878714946937, 0.012403124600014092 },
                { 0.0055921127727310985, -0.002505200297125928, 0.007975929601203018, 0.0025242542306910427, -0.006854861418870664, 8.263036094592882E-4,
                        0.007993569121582889, 0.0031075525072491277, 0.005642719217845866, 0.005625383923737505, 0.008051919444965404, -0.004316718103132654,
                        -0.004002488791980509, 0.004748343908175596, -0.003514536526479584, 0.005380308129467679, 0.002983430652236838, 0.008076882167494213,
                        0.012700351500838633, 0.0020119927777515726, 0.004337817502488467, -0.004491236641663855, 0.0037410562351007986, -0.00505002571813387,
                        -5.706762937385157E-4, -7.234734741959695E-5, 0.01351845803109445, 0.0071968606553414645, -0.010260982134956567, -0.0038515335277121672,
                        -1.2633525910660172E-4, -0.003905004060000495, 0.013623560183945407, -0.02689328515040802, 0.005703856525319462, -0.013207289693610909,
                        -0.007509352705505668, 0.005320337817062969, 0.008613006949979723, 0.005035014758696098, 0.012288914660975524, -0.0013787685695321153,
                        0.0014670242358673325, 0.003969861280262249, -0.011668580460918447, -1.8215889427383527E-4, 0.015615921525066984, 0.004357209095811046,
                        0.005936139751894294, 0.0023039285511616487 },
                { 0.0406100587684226, 0.011325295205237295, 0.018542710236303266, 0.02550949433635801, -0.00425980824125894, 0.01748903131071005,
                        0.020255091234345594, 0.044944763538417205, 0.01637844838699655, 0.008051919444965404, 0.02882736263614381, -4.909362966315817E-4,
                        0.024642133189686746, 0.015513964690980984, 0.022770309979572505, 0.01077344219923191, 0.017196972024076995, 0.03247470202669713,
                        0.03138303923514309, 0.006711181268196022, 0.02730391948472595, 0.03985284515969248, 0.01979877371626601, -0.005834253445046313,
                        0.01274092312440569, 0.03489620671117007, 0.029755799945204775, 0.02684274890081021, -0.002671917610741493, -0.00576927672509321,
                        0.021169589859883514, 0.024890193147797328, 0.03785288949272611, -0.010427763248974558, 0.01968015751170073, -0.02834535553804242,
                        0.012522791915173278, 0.013973593227236196, 0.03269484928919321, 0.02199813202574972, 0.02312243932014181, 0.01700202421575278,
                        0.007795125133258283, 0.034639218102963956, 0.0023920388519321836, 0.004396840151698384, 0.019028325977903757, 0.01840757797427165,
                        0.045250297004903885, 0.017038474480294552 },
                { 0.025946447741214906, -0.01157515102700365, 0.008693933217183068, 0.01656884391489779, 0.028954391758296533, 0.009387940459024482,
                        -0.0036255326984672466, 0.020906119358012034, 0.0016125669410293098, -0.004316718103132654, -4.909362966315817E-4, 0.01904623401174641,
                        0.03159855191715423, 0.013919651050998572, 0.021254683451706714, 7.30074729535098E-4, -0.0015262367041356446, 0.001629943396187395,
                        0.023587602337711636, -0.00814494746093247, 0.018526814800052166, 0.04931594499330462, 0.033907730622954074, 0.02207595545260519,
                        0.021178144050204836, 0.028859337100037824, -0.007738282398952333, 0.006801906176264385, 0.03997003516695978, 0.01542363476844373,
                        0.00972253959728318, 0.017436609437178254, 0.0019351686149680601, 0.0032459583986996316, 0.014447497023192188, 0.021320999837870717,
                        0.03521491277736523, 0.007303387852771121, 0.01683760680695703, 0.01391053481398849, 9.148458964523834E-4, 0.012289055945384417,
                        0.011695479955406755, 0.01633385450858602, 0.04814308530736648, 0.01652901805730591, -0.029772354336997466, 0.020326593104194674,
                        0.0019097156376099392, 0.027931897285779175 },
                { 0.11947198318927393, 0.07643219708715858, 0.050153924826700697, 0.07669194182196726, 0.057398188986273, 0.06293698807857667,
                        0.02357417852660373, 0.1324307283604126, 0.019550228022236507, -0.004002488791980509, 0.024642133189686746, 0.03159855191715423,
                        0.12486122243921483, 0.040334557461103995, 0.08416408977093864, 0.009146485362516452, 0.029107429064739045, 0.05224415961490875,
                        0.05520588160869793, 0.00501452971325641, 0.0739972874476517, 0.12587692827706393, 0.06482110173822006, 0.05085581833587291,
                        0.05680982107099359, 0.09946877873302404, 0.007571368616614454, 0.031536835280706915, 0.08112111080252754, 0.025355838343969088,
                        0.05852836227260248, 0.08101853893850161, 0.062408771724197724, 0.05965864324659605, 0.054274012127438535, -0.008944456842580273,
                        0.07402817066391493, 0.015938711551377147, 0.06602660346539199, 0.059470535534724325, 0.029503844292306264, 0.08347529462617091,
                        0.030377220858001912, 0.08606408263693746, 0.07994050420061365, 0.02756086217281535, -0.03550132264263489, 0.06178301233393403,
                        0.0664426411808189, 0.07761625501776233 },
                { 0.05077998079118238, 0.0032066653389200553, 0.03549665917687473, 0.030765547930930743, 0.022177315394533312, 0.020940461345369508,
                        0.013599439019096486, 0.04392220869568371, 0.012853481029847482, 0.004748343908175596, 0.015513964690980984, 0.013919651050998572,
                        0.040334557461103995, 0.026869792232516482, 0.021840281320577748, 0.011411228720486565, 0.005672701226372396, 0.026925388892373222,
                        0.05093255040513831, -0.00102381644047353, 0.03646253660382098, 0.05354898898892093, 0.03924945126007171, 0.020534011447493214,
                        0.02661771026212076, 0.03915573784738208, 0.01595304589678686, 0.02408631006150489, 0.031326252614737655, 0.012338006834057048,
                        0.01498787810635313, 0.018085302336096808, 0.03781286709806883, -0.029875806274627256, 0.030783670480620133, -5.130554836833814E-4,
                        0.025876836637010632, 0.016689951919338957, 0.03896741261289548, 0.02834954727982214, 0.026652881830119175, 0.025716523316895404,
                        0.01834112289309377, 0.0341513924059251, 0.03549715777261623, 0.017066735312504376, -0.004699251236597635, 0.029297047014579004,
                        0.02197184983106578, 0.045802363757645 },
                { 0.0816689949194069, 0.037023243241299186, 0.0038456579098830705, 0.05022346749733704, 0.0326283474837754, 0.037487202744069234,
                        0.01017293878231807, 0.08993336530611888, 0.016042928746541882, -0.003514536526479584, 0.022770309979572505, 0.021254683451706714,
                        0.08416408977093864, 0.021840281320577748, 0.0730480957313974, 0.0027696895709753496, 0.02427756575264085, 0.02065815163452022,
                        0.023541034620333877, -0.0026372891910335676, 0.04328334184410909, 0.08828500650069693, 0.05508900275284541, 0.01932156930651524,
                        0.03475347697473241, 0.06998247561771959, 0.006327457873487034, 0.015770981425701998, 0.04566331021372601, 0.010218661417610972,
                        0.05023148163552049, 0.06751073487668302, 0.0232202386531493, 0.04534144646458047, 0.03215505218267876, -0.02325583018669146,
                        0.061852982507602974, 0.011034207767589725, 0.04090001339249181, 0.0395699444216102, 0.008691276405743263, 0.040965576105266625,
                        0.010667977660445388, 0.056452394144513834, 0.04972193712895255, 0.01889061843749912, -0.029411932918713697, 0.05221184280142008,
                        0.049609529647590765, 0.026024036979239613 },
                { 0.01827495627023137, 0.001628559089516492, 0.0196642371608017, 0.011482747000572807, 0.0019186743579239483, 0.007701861141537793,
                        0.011550956942541307, 0.016626526502406536, 0.007534325645958795, 0.005380308129467679, 0.01077344219923191, 7.30074729535098E-4,
                        0.009146485362516452, 0.011411228720486565, 0.0027696895709753496, 0.007521756650085169, 0.004887307792753128, 0.017345316005908128,
                        0.025411285398938877, 0.002783404801598309, 0.014918062445420485, 0.01521862901588271, 0.01209598582071392, 0.003713485021926183,
                        0.007869250286936571, 0.013226262779901984, 0.014961991268619884, 0.014246815539288016, 0.0026373342534143913, 8.485278114806951E-4,
                        0.003907891331712349, 0.0030955416784430676, 0.024055484678794783, -0.023603699599673775, 0.012945535501979823, -0.007356656885097572,
                        0.0017043969379770204, 0.008721608069530552, 0.018419015986484044, 0.01201653617206273, 0.017756982692169056, 0.009187359500872124,
                        0.00807530091495141, 0.014671670231619846, 0.0030531780230906302, 0.004443525289398188, 0.011220480067533355, 0.009404379471913882,
                        0.013592438160621766, 0.01825292123322788 },
                { 0.019257623272038657, 0.04692161945594038, 0.018063195401167596, 0.023801862409117004, 0.022508945852992547, 0.019990688099678158,
                        0.04493816244535945, 0.052225853063807684, 0.008295639927543282, 0.002983430652236838, 0.017196972024076995, -0.0015262367041356446,
                        0.029107429064739045, 0.005672701226372396, 0.02427756575264085, 0.004887307792753128, 0.04497826602868016, 0.0343100568163147,
                        0.01914884823971065, 0.014052580623718353, 0.01816342405736051, 0.0452522219659566, 0.015149460735935547, 0.015620848402734165,
                        0.014427760134330862, 0.040614928704672225, 0.018559066725491832, 0.018244100766286835, 0.010005710484596798, -0.006092929644243973,
                        0.022065200114442587, 0.038898625048217866, 0.039124687945477674, 0.05170982783468267, 0.01024634369187441, -0.012173176733372173,
                        0.016227044286925552, 0.006560596391855907, 0.024523224498329367, 0.02489872321762207, 0.02359111531605281, 0.030169169634378656,
                        0.018318342989655956, 0.039986684242463835, 0.002048135897920381, 0.00333287955412777, 0.019206430235579428, 0.019317476202761502,
                        0.045551778582298336, 0.01717867893975092 },
                { 0.05736422711496372, 0.06480658158745242, 0.07676063345552421, 0.04758019899429915, 0.01528422712131023, 0.04245735832358022,
                        0.048479134016584186, 0.08663464179709535, 0.01635680772504871, 0.008076882167494213, 0.03247470202669713, 0.001629943396187395,
                        0.05224415961490875, 0.026925388892373222, 0.02065815163452022, 0.017345316005908128, 0.0343100568163147, 0.08232643056922598,
                        0.06074310596079131, 0.026649211500845267, 0.05384845987807167, 0.07914813984380803, 0.009589320827537821, 0.02730996475442342,
                        0.028901240155661145, 0.06640513497053599, 0.03620861485241317, 0.050632417161405915, 0.02425657629644978, 0.0022568717011138143,
                        0.02385072306151171, 0.039346890237893624, 0.09147016157760011, 0.035890463579133836, 0.03212562753870899, -0.0066540489241117586,
                        0.012815173067601713, 0.01618627678326274, 0.054642635459336994, 0.03759235646984377, 0.04901120931557741, 0.06732426120758324,
                        0.03465264830251983, 0.0688845289168456, 0.022083027967365925, 0.00546947833749213, 0.031603402764409066, 0.009011668071378253,
                        0.07839741290189277, 0.07297587897441953 },
                { 0.07185168949490216, -0.017511710151868906, 0.07805265348270753, 0.05085578918543725, 0.04434563109656779, 0.030495393467626305,
                        0.04024200783535551, 0.06862112273773749, 0.02326437797184387, 0.012700351500838633, 0.03138303923514309, 0.023587602337711636,
                        0.05520588160869793, 0.05093255040513831, 0.023541034620333877, 0.025411285398938877, 0.01914884823971065, 0.06074310596079131,
                        0.1153714783404077, 0.001554245755673789, 0.0649943723910234, 0.10915743421507734, 0.07987042623650074, 0.04394545360243046,
                        0.04846799234504677, 0.07588606259684225, 0.04002110594408505, 0.05641032338868085, 0.056733343188292415, 0.01988097545284504,
                        0.01663562091438381, 0.026608320640220708, 0.0827304829380246, -0.07162830628464219, 0.053993330797205884, 0.012325903312289645,
                        0.044575392698314335, 0.037746004417918394, 0.07772648516747582, 0.05373742712210337, 0.06231629833142585, 0.04110217977213719,
                        0.04660867830127618, 0.06476829368778655, 0.0672698006549415, 0.034478393601607336, 0.007079765524602155, 0.049684254920940744,
                        0.04396700152240977, 0.09564667778749408 },
                { 0.002532075036186415, 0.03727219786400344, 0.021147399844983532, 0.007379161161471313, -0.005103023686480352, 0.011179011467022701,
                        0.01770506449186173, 0.020525908043417255, 0.0013775769465331484, 0.0020119927777515726, 0.006711181268196022, -0.00814494746093247,
                        0.00501452971325641, -0.00102381644047353, -0.0026372891910335676, 0.002783404801598309, 0.014052580623718353, 0.026649211500845267,
                        0.001554245755673789, 0.014730597105717301, 0.007013672923182388, 0.0035710559374745356, -0.018672096149585156, 0.0032338749172665647,
                        -8.670204442702457E-4, 0.00888298405520044, 0.009531005572636296, 0.01042133741917844, -0.006579717536556402, -0.005548127069262061,
                        0.0034677908765077294, 0.007944169205500909, 0.027275654524805154, 0.031431760281881965, 3.448239985529497E-4, -0.006590151748620935,
                        -0.011705253067395333, -0.0018899368535158626, 0.0057198262606141205, 0.004021000747169842, 0.011968905503713104, 0.021730012223907698,
                        0.00684142190703042, 0.015280124236799732, -0.013272197915063367, -0.007567005997181369, 0.020797051761852023, -0.010692079232214057,
                        0.02509848265404834, 0.011461561228721894 },
                { 0.08346157982652175, 0.03658218514016751, 0.05652375650338057, 0.05394220385138819, 0.03167231278397138, 0.04185455547912513,
                        0.025812698271660776, 0.08732025297040837, 0.01884706689371629, 0.004337817502488467, 0.02730391948472595, 0.018526814800052166,
                        0.0739972874476517, 0.03646253660382098, 0.04328334184410909, 0.014918062445420485, 0.01816342405736051, 0.05384845987807167,
                        0.0649943723910234, 0.007013672923182388, 0.05950567506377508, 0.09216729029533383, 0.04506436711026969, 0.031285574739739995,
                        0.04086266085659154, 0.07075544289746474, 0.023065754347590353, 0.039319039964212, 0.04777033167851654, 0.015444241281637978,
                        0.032493121399251056, 0.043826247726673165, 0.06510119823632007, 0.0037059547835040247, 0.04391961522423012, -0.0047533669257983375,
                        0.04116391683895166, 0.020238669714503593, 0.05866264741129006, 0.043773642849631186, 0.037092116362247754, 0.055963804570557565,
                        0.027949497049585964, 0.06384349309120325, 0.05228616641256083, 0.020033763532908824, -0.005358901183943695, 0.03711792583088948,
                        0.05486613162833511, 0.07020416946484041 },
                { 0.11131859328267968, 0.0170129992110056, 0.06946814862997584, 0.0859495668671807, 0.08698557056272502, 0.058514499570549106,
                        0.04358810069837829, 0.13892362179334494, 0.021423728553515167, -0.004491236641663855, 0.03985284515969248, 0.04931594499330462,
                        0.12587692827706393, 0.05354898898892093, 0.08828500650069693, 0.01521862901588271, 0.0452522219659566, 0.07914813984380803,
                        0.10915743421507734, 0.0035710559374745356, 0.09216729029533383, 0.2236110732297143, 0.10470287603571035, 0.06985740983958316,
                        0.07962258082700355, 0.15059035937359816, 0.01996818445176931, 0.06884961508766513, 0.11767126970854752, 0.03134605424945715,
                        0.05801618383719265, 0.10225650117191916, 0.08017102203062727, 0.07177755747629333, 0.05836670675873523, 0.041742925099958236,
                        0.11829241365316617, 0.03597387096564924, 0.09380720875270296, 0.07090004374513742, 0.039896917463908096, 0.08274271765974067,
                        0.06003521018086544, 0.11293795089121894, 0.14541022871151163, 0.04736321361091196, -0.045701468358177, 0.06435540908768751,
                        0.09890510499267771, 0.11958113266783735 },
                { 0.06760715083358035, -0.04844605700882658, 0.015237057579642395, 0.04361050470212717, 0.0517881055145056, 0.019207163290216652,
                        0.010817413991423537, 0.05905987369784156, 0.01928222835395192, 0.0037410562351007986, 0.01979877371626601, 0.033907730622954074,
                        0.06482110173822006, 0.03924945126007171, 0.05508900275284541, 0.01209598582071392, 0.015149460735935547, 0.009589320827537821,
                        0.07987042623650074, -0.018672096149585156, 0.04506436711026969, 0.10470287603571035, 0.11444306359999318, 0.03853710369078282,
                        0.04364116202559194, 0.07175083626389725, 0.016277428689696147, 0.025405375885681383, 0.06630142058791469, 0.020978484086922475,
                        0.031217736971209248, 0.045731214226522286, 0.02582862006670298, -0.06273598135259235, 0.044606847598282644, -0.0025022033601932027,
                        0.07623196375142574, 0.03140320586188841, 0.05911831646176029, 0.05132229226734475, 0.03096658748050608, 0.01361494997926399,
                        0.02988179809617699, 0.04922365096825167, 0.07772584474908925, 0.0420516619250631, -0.0316489759353913, 0.07866961738675295,
                        0.017612308286427958, 0.04957234984180621 },
                { 0.029470299100444058, 0.024862643782799655, 0.04534040085118005, 0.029284956297791043, 0.05382084693674011, 0.024260772565887795,
                        0.019567881664452235, 0.045957328306873775, -8.06392242197756E-4, -0.00505002571813387, -0.005834253445046313, 0.02207595545260519,
                        0.05085581833587291, 0.020534011447493214, 0.01932156930651524, 0.003713485021926183, 0.015620848402734165, 0.02730996475442342,
                        0.04394545360243046, 0.0032338749172665647, 0.031285574739739995, 0.06985740983958316, 0.03853710369078282, 0.055970903692063874,
                        0.03227915785477202, 0.04731617021479265, -0.008501104980650476, 0.015112143270958939, 0.06561894351491046, 0.023351581742516586,
                        0.009512284032177826, 0.02713683292672038, 0.034743833292875254, 0.03034200305610554, 0.022566537060477933, 0.03794216601706281,
                        0.037601829443135976, 0.007994605777405992, 0.029398925729516866, 0.028218066509836687, 0.02149649053579329, 0.04520155171751658,
                        0.03409906093413169, 0.03688187779892209, 0.061225094348345, 0.020889288168698082, -0.023618761077837096, 0.024033306686366745,
                        0.011316630462083147, 0.06466631042915777 },
                { 0.05558439313482155, 0.016286529242773753, 0.034623855387281285, 0.03740823982999819, 0.040692005026666485, 0.026837542613741792,
                        0.01834928713662144, 0.05757027944958483, 0.01002299130965817, -5.706762937385157E-4, 0.01274092312440569, 0.021178144050204836,
                        0.05680982107099359, 0.02661771026212076, 0.03475347697473241, 0.007869250286936571, 0.014427760134330862, 0.028901240155661145,
                        0.04846799234504677, -8.670204442702457E-4, 0.04086266085659154, 0.07962258082700355, 0.04364116202559194, 0.03227915785477202,
                        0.03557240539732746, 0.054890721292477365, 0.006764063963353467, 0.024041747233671872, 0.05048658165534706, 0.017283195895727264,
                        0.022205073901894093, 0.03424627590171993, 0.03526926271863265, 0.011158958849404294, 0.030637927889692274, 0.01503205956571215,
                        0.043395513364138896, 0.014846844638696847, 0.04029392331842219, 0.03181013475818602, 0.02072251736776718, 0.03668254527204581,
                        0.02340838673178454, 0.043402556194393524, 0.056598620507129556, 0.020922198988173032, -0.02022803552319418, 0.03245926941714936,
                        0.029004434950017314, 0.05365272248309082 },
                { 0.09272095375985491, 0.03882674238717706, 0.054936555452588506, 0.06928950159324, 0.05443615089315773, 0.05109264635188824,
                        0.03885619284599652, 0.11846758660430416, 0.020192288671497027, -7.234734741959695E-5, 0.03489620671117007, 0.028859337100037824,
                        0.09946877873302404, 0.03915573784738208, 0.06998247561771959, 0.013226262779901984, 0.040614928704672225, 0.06640513497053599,
                        0.07588606259684225, 0.00888298405520044, 0.07075544289746474, 0.15059035937359816, 0.07175083626389725, 0.04731617021479265,
                        0.054890721292477365, 0.11017302468333334, 0.023062509407191262, 0.050315847480930986, 0.07442957300102515, 0.017381698676083468,
                        0.04996308919372786, 0.08045469893280316, 0.072226176585183, 0.053530132078449824, 0.04675330071813151, 0.00523332702537055,
                        0.07670715279950097, 0.025025666113132738, 0.07247723116919025, 0.057913018860102805, 0.037921736192343, 0.0703520271775932,
                        0.04182689808913473, 0.09034384293031988, 0.08518578770951486, 0.02956603640897039, -0.01817814200088095, 0.05207857536090549,
                        0.08315006419021534, 0.08281085333066818 },
                { 0.029233548846136864, 0.005808335064625088, 0.02708987410948711, 0.019233709253406273, -0.0116382507299386, 0.01223709128880002,
                        0.028671687841984827, 0.033018131820394245, 0.017817122381597734, 0.01351845803109445, 0.029755799945204775, -0.007738282398952333,
                        0.007571368616614454, 0.01595304589678686, 0.006327457873487034, 0.014961991268619884, 0.018559066725491832, 0.03620861485241317,
                        0.04002110594408505, 0.009531005572636296, 0.023065754347590353, 0.01996818445176931, 0.016277428689696147, -0.008501104980650476,
                        0.006764063963353467, 0.023062509407191262, 0.03958289400311162, 0.029210966543480883, -0.01655463216914924, -0.010040535282356166,
                        0.01167681622305489, 0.01024333000074405, 0.04777504123962145, -0.04108492132512165, 0.019480984278262753, -0.03464423111066974,
                        -0.004734254410359639, 0.0165603323735002, 0.03346489057509839, 0.021491834010285274, 0.035013142959900295, 0.010805198449125107,
                        0.009955528136101987, 0.02917598112096329, -0.01648539799300491, 0.001926096039882604, 0.03856854258899345, 0.015681998540460332,
                        0.0395650297035591, 0.016758434473834218 },
                { 0.04146431678532974, 0.009736332491718087, 0.04721913230108773, 0.03190542530555318, 0.012630902416388595, 0.02261965494842901,
                        0.02873764763689312, 0.05113894564142611, 0.014251372364943424, 0.0071968606553414645, 0.02684274890081021, 0.006801906176264385,
                        0.031536835280706915, 0.02408631006150489, 0.015770981425701998, 0.014246815539288016, 0.018244100766286835, 0.050632417161405915,
                        0.05641032338868085, 0.01042133741917844, 0.039319039964212, 0.06884961508766513, 0.025405375885681383, 0.015112143270958939,
                        0.024041747233671872, 0.050315847480930986, 0.029210966543480883, 0.040169142810912994, 0.020509585396682954, 0.003424053406410403,
                        0.015172426080978282, 0.02480304717436915, 0.05676461572119647, -0.005076191990466836, 0.025663180787566672, 2.742545260877072E-4,
                        0.02014164828137556, 0.018708134384208076, 0.04397422863059344, 0.027495314352214043, 0.033673949840165196, 0.032459512669923814,
                        0.024616259186111086, 0.04570296576490284, 0.02905562721442751, 0.011139277407840622, 0.015373528869895127, 0.013985598656701015,
                        0.05041660997844717, 0.05149742362471869 },
                { 0.05827968731163341, 0.00992513269127665, 0.04196781961498533, 0.04448026111666185, 0.07328167166471125, 0.03212977978831147,
                        0.008198931568664743, 0.06555590344191373, 0.0016862982525222361, -0.010260982134956567, -0.002671917610741493, 0.03997003516695978,
                        0.08112111080252754, 0.031326252614737655, 0.04566331021372601, 0.0026373342534143913, 0.010005710484596798, 0.02425657629644978,
                        0.056733343188292415, -0.006579717536556402, 0.04777033167851654, 0.11767126970854752, 0.06630142058791469, 0.06561894351491046,
                        0.05048658165534706, 0.07442957300102515, -0.01655463216914924, 0.020509585396682954, 0.09753085427707674, 0.03542459813999699,
                        0.023104357315398016, 0.04702292666651953, 0.02721857061205004, 0.04028845574294887, 0.03380763064555947, 0.05276373451876118,
                        0.07394173637719686, 0.013639728493196052, 0.042235360456987385, 0.03727607935304138, 0.013241362915346663, 0.05164176763373245,
                        0.037566026857732936, 0.050353745392564306, 0.10636666401849473, 0.03429787348646608, -0.05661228056545248, 0.04054973959123313,
                        0.01769854337404704, 0.08064609382386137 },
                { 0.02024881721137161, -0.002041142410520545, 0.016019218970349393, 0.012362736864734786, 0.025419950047336628, 0.008861849172155503,
                        -0.004233061739110836, 0.01367217439094696, -8.116635505841154E-4, -0.0038515335277121672, -0.00576927672509321, 0.01542363476844373,
                        0.025355838343969088, 0.012338006834057048, 0.010218661417610972, 8.485278114806951E-4, -0.006092929644243973, 0.0022568717011138143,
                        0.01988097545284504, -0.005548127069262061, 0.015444241281637978, 0.03134605424945715, 0.020978484086922475, 0.023351581742516586,
                        0.017283195895727264, 0.017381698676083468, -0.010040535282356166, 0.003424053406410403, 0.03542459813999699, 0.01608338971467386,
                        0.0027243130230694716, 0.0063352917340651575, 0.004497178959862465, -0.0010453148506623044, 0.012783572772323734, 0.0233898912616186,
                        0.02189513170990342, 0.004002945186717686, 0.011771328062747385, 0.009754333918462213, 0.0025474959424185767, 0.014680819899953122,
                        0.010840517722816728, 0.009955956738649328, 0.03858207424377999, 0.012619336502716363, -0.024609112880508082, 0.012268410154407515,
                        -0.006232420244878353, 0.029872753619924725 },
                { 0.05987535887602487, 0.03857111312346192, 0.008294996667309703, 0.037583168521902646, 0.015972563956055362, 0.029759499287531833,
                        0.01384033519505982, 0.07018524492324246, 0.013621124466734644, -1.2633525910660172E-4, 0.021169589859883514, 0.00972253959728318,
                        0.05852836227260248, 0.01498787810635313, 0.05023148163552049, 0.003907891331712349, 0.022065200114442587, 0.02385072306151171,
                        0.01663562091438381, 0.0034677908765077294, 0.032493121399251056, 0.05801618383719265, 0.031217736971209248, 0.009512284032177826,
                        0.022205073901894093, 0.04996308919372786, 0.01167681622305489, 0.015172426080978282, 0.023104357315398016, 0.0027243130230694716,
                        0.037275509800053994, 0.04893424362319797, 0.026857749206552865, 0.03530192110330622, 0.023238922171878475, -0.02619471080628478,
                        0.03589209249123036, 0.007784593629349519, 0.031346773143603825, 0.029287759190042666, 0.011703082437405182, 0.03379395893861452,
                        0.0075054548702090315, 0.044419397103499736, 0.023272913577621283, 0.009259649775743773, -0.008551613873494155, 0.033735580441681894,
                        0.04516982398417219, 0.017975561143856063 },
                { 0.06833527370134565, 0.05132914660906421, 0.016857130293880886, 0.05112415966816555, 0.03554712416533306, 0.04037072069359275,
                        0.0259238226874429, 0.09741458821208013, 0.013477770974060568, -0.003905004060000495, 0.024890193147797328, 0.017436609437178254,
                        0.08101853893850161, 0.018085302336096808, 0.06751073487668302, 0.0030955416784430676, 0.038898625048217866, 0.039346890237893624,
                        0.026608320640220708, 0.007944169205500909, 0.043826247726673165, 0.10225650117191916, 0.045731214226522286, 0.02713683292672038,
                        0.03424627590171993, 0.08045469893280316, 0.01024333000074405, 0.02480304717436915, 0.04702292666651953, 0.0063352917340651575,
                        0.04893424362319797, 0.07538093081682562, 0.03900436086539902, 0.0797275511650033, 0.026632981157894482, -0.013939200229593418,
                        0.059555565493639454, 0.010152193867776636, 0.042626868113337775, 0.04050487283244285, 0.015086688692272449, 0.05285539756333113,
                        0.02098651385437475, 0.0667925361541019, 0.04947470974762157, 0.015574684120585276, -0.017193066237124143, 0.04122951622759873,
                        0.06753820543930868, 0.035999031659639635 },
                { 0.07510131226243622, 0.07099624607549501, 0.0931130325113982, 0.0575256825642667, 0.020593816787798494, 0.04961387510654315,
                        0.05943095402731848, 0.10081329808561068, 0.02325532057282806, 0.013623560183945407, 0.03785288949272611, 0.0019351686149680601,
                        0.062408771724197724, 0.03781286709806883, 0.0232202386531493, 0.024055484678794783, 0.039124687945477674, 0.09147016157760011,
                        0.0827304829380246, 0.027275654524805154, 0.06510119823632007, 0.08017102203062727, 0.02582862006670298, 0.034743833292875254,
                        0.03526926271863265, 0.072226176585183, 0.04777504123962145, 0.05676461572119647, 0.02721857061205004, 0.004497178959862465,
                        0.026857749206552865, 0.03900436086539902, 0.11176670415513645, 0.0020940383134273005, 0.046224172739342845, -0.01946624972701054,
                        0.011316167701513098, 0.023243387223156168, 0.07015468607422427, 0.05123575392817562, 0.0681063031185501, 0.07506391875244056,
                        0.04139488248564352, 0.07897476045031361, 0.018495536656020446, 0.010658158278717597, 0.04202718286985202, 0.025658383844895348,
                        0.07994824224740979, 0.08668847925774631 },
                { -0.0034610303544831884, 0.14230302989226046, -0.002376306446747096, 0.02156320707178575, 0.03752778234111384, 0.036541293860140364,
                        0.02641899806396522, 0.07405493590857007, -0.02059668928311294, -0.02689328515040802, -0.010427763248974558, 0.0032459583986996316,
                        0.05965864324659605, -0.029875806274627256, 0.04534144646458047, -0.023603699599673775, 0.05170982783468267, 0.035890463579133836,
                        -0.07162830628464219, 0.031431760281881965, 0.0037059547835040247, 0.07177755747629333, -0.06273598135259235, 0.03034200305610554,
                        0.011158958849404294, 0.053530132078449824, -0.04108492132512165, -0.005076191990466836, 0.04028845574294887, -0.0010453148506623044,
                        0.03530192110330622, 0.0797275511650033, 0.0020940383134273005, 0.28953566450031704, -0.027933437982339465, 0.04190837512968599,
                        0.03313397654550686, -0.032337308272150325, -0.02175498208725106, -0.005367564444732144, -0.03807320243473582, 0.07520104780595395,
                        0.00938045601913131, 0.042666885566231286, 0.033536757064443326, -0.018324564705520144, -0.03179472978069248, -0.03743223787386253,
                        0.06986979680350897, 0.007740533499578341 },
                { 0.0668336982118146, 0.01819949089098011, 0.03929274211546738, 0.03970887872943984, 0.024286105490069175, 0.02900022014049821,
                        0.017448403751967528, 0.06086515210357006, 0.01665134330109888, 0.005703856525319462, 0.01968015751170073, 0.014447497023192188,
                        0.054274012127438535, 0.030783670480620133, 0.03215505218267876, 0.012945535501979823, 0.01024634369187441, 0.03212562753870899,
                        0.053993330797205884, 3.448239985529497E-4, 0.04391961522423012, 0.05836670675873523, 0.044606847598282644, 0.022566537060477933,
                        0.030637927889692274, 0.04675330071813151, 0.019480984278262753, 0.025663180787566672, 0.03380763064555947, 0.012783572772323734,
                        0.023238922171878475, 0.026632981157894482, 0.046224172739342845, -0.027933437982339465, 0.03766863024780391, -0.01172413546619409,
                        0.0292802545702044, 0.01784566534805942, 0.046048353787981194, 0.035893780203764214, 0.031806125600385565, 0.03483351485005763,
                        0.01900657235933252, 0.04336783053057241, 0.03447921441728823, 0.018127681645465587, -0.003474539960621243, 0.03844334464754725,
                        0.02965802603239724, 0.04917159564434252 },
                { -0.03720551901508286, -0.0314052003041181, 0.017111756371728126, -0.01333522917159062, 0.04532052601540752, -0.012193410934990946,
                        -0.00556763946329843, -0.03495189242523619, -0.0203282709219231, -0.013207289693610909, -0.02834535553804242, 0.021320999837870717,
                        -0.008944456842580273, -5.130554836833814E-4, -0.02325583018669146, -0.007356656885097572, -0.012173176733372173,
                        -0.0066540489241117586, 0.012325903312289645, -0.006590151748620935, -0.0047533669257983375, 0.041742925099958236,
                        -0.0025022033601932027, 0.03794216601706281, 0.01503205956571215, 0.00523332702537055, -0.03464423111066974, 2.742545260877072E-4,
                        0.05276373451876118, 0.0233898912616186, -0.02619471080628478, -0.013939200229593418, -0.01946624972701054, 0.04190837512968599,
                        -0.01172413546619409, 0.09533491298405161, 0.02220301204793979, -0.002974488665739473, -0.011955304887437247, -0.01452027155414225,
                        -0.017357526941778927, 0.0016286977796600554, 0.02002788679096782, -0.013149265043071349, 0.06808063805171294, 0.012521660197467813,
                        -0.04313578439690932, -0.024874056661572726, -0.02932197299311745, 0.0402317400152294 },
                { 0.060929539640763974, -0.01175789159428585, 0.00786229681839231, 0.04239868460640808, 0.055109086655764354, 0.025069104652403,
                        0.004652879105268365, 0.06574262289330716, 0.008907803366441557, -0.007509352705505668, 0.012522791915173278, 0.03521491277736523,
                        0.07402817066391493, 0.025876836637010632, 0.061852982507602974, 0.0017043969379770204, 0.016227044286925552, 0.012815173067601713,
                        0.044575392698314335, -0.011705253067395333, 0.04116391683895166, 0.11829241365316617, 0.07623196375142574, 0.037601829443135976,
                        0.043395513364138896, 0.07670715279950097, -0.004734254410359639, 0.02014164828137556, 0.07394173637719686, 0.02189513170990342,
                        0.03589209249123036, 0.059555565493639454, 0.011316167701513098, 0.03313397654550686, 0.0292802545702044, 0.02220301204793979,
                        0.08141168815907611, 0.016886954501034113, 0.041456776077280474, 0.036385384583715694, 0.004491655790013703, 0.02937827313877628,
                        0.023813741889244844, 0.049650076806876844, 0.09085754324229704, 0.03198000522058489, -0.05106520320215872, 0.05042579461303887,
                        0.031686320199504264, 0.04600959200835599 },
                { 0.025000355192702817, -0.014921922876607185, 0.01789436213800264, 0.016129277667228197, 0.010851575069873512, 0.007617889780041749,
                        0.011874781536065483, 0.0211572038771267, 0.009849714425995422, 0.005320337817062969, 0.013973593227236196, 0.007303387852771121,
                        0.015938711551377147, 0.016689951919338957, 0.011034207767589725, 0.008721608069530552, 0.006560596391855907, 0.01618627678326274,
                        0.037746004417918394, -0.0018899368535158626, 0.020238669714503593, 0.03597387096564924, 0.03140320586188841, 0.007994605777405992,
                        0.014846844638696847, 0.025025666113132738, 0.0165603323735002, 0.018708134384208076, 0.013639728493196052, 0.004002945186717686,
                        0.007784593629349519, 0.010152193867776636, 0.023243387223156168, -0.032337308272150325, 0.01784566534805942, -0.002974488665739473,
                        0.016886954501034113, 0.014252975486536576, 0.02641636819598983, 0.018033382060008943, 0.019360006333242027, 0.006615632005219953,
                        0.01230007803997319, 0.02044100035227757, 0.019472382757094415, 0.011923372533215718, 0.0032182794022651345, 0.020376786610218713,
                        0.015699497963643227, 0.02406768527593103 },
                { 0.07911682136469031, 0.019613724599524256, 0.055585709836948514, 0.05276665293939613, 0.03120649507961279, 0.0372827752145765,
                        0.03399811228132037, 0.08381941739889459, 0.022575555940660567, 0.008613006949979723, 0.03269484928919321, 0.01683760680695703,
                        0.06602660346539199, 0.03896741261289548, 0.04090001339249181, 0.018419015986484044, 0.024523224498329367, 0.054642635459336994,
                        0.07772648516747582, 0.0057198262606141205, 0.05866264741129006, 0.09380720875270296, 0.05911831646176029, 0.029398925729516866,
                        0.04029392331842219, 0.07247723116919025, 0.03346489057509839, 0.04397422863059344, 0.042235360456987385, 0.011771328062747385,
                        0.031346773143603825, 0.042626868113337775, 0.07015468607422427, -0.02175498208725106, 0.046048353787981194, -0.011955304887437247,
                        0.041456776077280474, 0.02641636819598983, 0.06496388363545633, 0.048665668700581706, 0.04634146438615371, 0.04712394107819101,
                        0.031453262230779766, 0.06538104087029223, 0.047035687289814505, 0.023246544180397097, 0.005138736094799597, 0.045815821432056036,
                        0.05594747496550899, 0.06807609191253781 },
                { 0.06345414054903112, 0.026661091708265507, 0.037333713645423545, 0.04321141547652689, 0.030650769411466612, 0.03175404698799519,
                        0.02807274711585795, 0.0721603971651063, 0.016765228593236293, 0.005035014758696098, 0.02199813202574972, 0.01391053481398849,
                        0.059470535534724325, 0.02834954727982214, 0.0395699444216102, 0.01201653617206273, 0.02489872321762207, 0.03759235646984377,
                        0.05373742712210337, 0.004021000747169842, 0.043773642849631186, 0.07090004374513742, 0.05132229226734475, 0.028218066509836687,
                        0.03181013475818602, 0.057913018860102805, 0.021491834010285274, 0.027495314352214043, 0.03727607935304138, 0.009754333918462213,
                        0.029287759190042666, 0.04050487283244285, 0.05123575392817562, -0.005367564444732144, 0.035893780203764214, -0.01452027155414225,
                        0.036385384583715694, 0.018033382060008943, 0.048665668700581706, 0.04136204861756126, 0.03491372008726366, 0.04026916468744444,
                        0.024411365643698824, 0.05274387016939227, 0.03507983651492849, 0.018898191260714407, 1.5652860364954703E-4, 0.04384463585170688,
                        0.040909517127887624, 0.048209565982487435 },
                { 0.042067438934267495, 0.023964258241647315, 0.05536234155443809, 0.03165505385396111, 0.014062668145234797, 0.024137621397914238,
                        0.03907832126098715, 0.05143648206330431, 0.016775086630961195, 0.012288914660975524, 0.02312243932014181, 9.148458964523834E-4,
                        0.029503844292306264, 0.026652881830119175, 0.008691276405743263, 0.017756982692169056, 0.02359111531605281, 0.04901120931557741,
                        0.06231629833142585, 0.011968905503713104, 0.037092116362247754, 0.039896917463908096, 0.03096658748050608, 0.02149649053579329,
                        0.02072251736776718, 0.037921736192343, 0.035013142959900295, 0.033673949840165196, 0.013241362915346663, 0.0025474959424185767,
                        0.011703082437405182, 0.015086688692272449, 0.0681063031185501, -0.03807320243473582, 0.031806125600385565, -0.017357526941778927,
                        0.004491655790013703, 0.019360006333242027, 0.04634146438615371, 0.03491372008726366, 0.049676262865583694, 0.03434053472193331,
                        0.026848445126848402, 0.0431857774737521, 0.005872816365294107, 0.010909158677848852, 0.03144826078116081, 0.026542003952573043,
                        0.03683214201110552, 0.05065130246810593 },
                { 0.07386102107101723, 0.09472958537687795, 0.06864217828508612, 0.055528117288422246, 0.0359415224733912, 0.05324076131132853,
                        0.035158626924776074, 0.10265113631167797, 0.0104116838298043, -0.0013787685695321153, 0.01700202421575278, 0.012289055945384417,
                        0.08347529462617091, 0.025716523316895404, 0.040965576105266625, 0.009187359500872124, 0.030169169634378656, 0.06732426120758324,
                        0.04110217977213719, 0.021730012223907698, 0.055963804570557565, 0.08274271765974067, 0.01361494997926399, 0.04520155171751658,
                        0.03668254527204581, 0.0703520271775932, 0.010805198449125107, 0.032459512669923814, 0.05164176763373245, 0.014680819899953122,
                        0.03379395893861452, 0.05285539756333113, 0.07506391875244056, 0.07520104780595395, 0.03483351485005763, 0.0016286977796600554,
                        0.02937827313877628, 0.006615632005219953, 0.04712394107819101, 0.04026916468744444, 0.03434053472193331, 0.08375780262448103,
                        0.0319310268687214, 0.069882831355347, 0.04182588812333326, 0.00920913923108069, -8.380898281261503E-4, 0.019765928925993145,
                        0.06420140775392918, 0.07512769631140155 },
                { 0.022327898989038406, 0.010921251146816234, 0.04172971348303587, 0.023943297575598703, 0.031258368682786544, 0.01772750700613978,
                        0.025430471163154902, 0.03795642722556948, 0.004750003818887418, 0.0014670242358673325, 0.007795125133258283, 0.011695479955406755,
                        0.030377220858001912, 0.01834112289309377, 0.010667977660445388, 0.00807530091495141, 0.018318342989655956, 0.03465264830251983,
                        0.04660867830127618, 0.00684142190703042, 0.027949497049585964, 0.06003521018086544, 0.02988179809617699, 0.03409906093413169,
                        0.02340838673178454, 0.04182689808913473, 0.009955528136101987, 0.024616259186111086, 0.037566026857732936, 0.010840517722816728,
                        0.0075054548702090315, 0.02098651385437475, 0.04139488248564352, 0.00938045601913131, 0.01900657235933252, 0.02002788679096782,
                        0.023813741889244844, 0.01230007803997319, 0.031453262230779766, 0.024411365643698824, 0.026848445126848402, 0.0319310268687214,
                        0.02892520049251327, 0.03515610002608455, 0.03813541438217001, 0.014496608339908932, 1.8553111681012723E-4, 0.01615861416251288,
                        0.025194762816263844, 0.05144271760390398 },
                { 0.08606120434337522, 0.05873848188653275, 0.05852433263683626, 0.06304401538295411, 0.03699394538887053, 0.050333259940161676,
                        0.042617052086210916, 0.11180928914709924, 0.020785017783699532, 0.003969861280262249, 0.034639218102963956, 0.01633385450858602,
                        0.08606408263693746, 0.0341513924059251, 0.056452394144513834, 0.014671670231619846, 0.039986684242463835, 0.0688845289168456,
                        0.06476829368778655, 0.015280124236799732, 0.06384349309120325, 0.11293795089121894, 0.04922365096825167, 0.03688187779892209,
                        0.043402556194393524, 0.09034384293031988, 0.02917598112096329, 0.04570296576490284, 0.050353745392564306, 0.009955956738649328,
                        0.044419397103499736, 0.0667925361541019, 0.07897476045031361, 0.042666885566231286, 0.04336783053057241, -0.013149265043071349,
                        0.049650076806876844, 0.02044100035227757, 0.06538104087029223, 0.05274387016939227, 0.0431857774737521, 0.069882831355347,
                        0.03515610002608455, 0.0822926850377143, 0.05106860749224654, 0.019075634816033964, 0.003210603476580604, 0.042415834956967206,
                        0.07993440843712808, 0.07163565691765367 },
                { 0.060893122812509706, -0.023685780975292402, 0.03678886162124004, 0.04481212196001175, 0.07534667993947596, 0.02712682305922319,
                        -7.120166755056671E-4, 0.05954074594780366, 0.0026031549700070633, -0.011668580460918447, 0.0023920388519321836, 0.04814308530736648,
                        0.07994050420061365, 0.03549715777261623, 0.04972193712895255, 0.0030531780230906302, 0.002048135897920381, 0.022083027967365925,
                        0.0672698006549415, -0.013272197915063367, 0.05228616641256083, 0.14541022871151163, 0.07772584474908925, 0.061225094348345,
                        0.056598620507129556, 0.08518578770951486, -0.01648539799300491, 0.02905562721442751, 0.10636666401849473, 0.03858207424377999,
                        0.023272913577621283, 0.04947470974762157, 0.018495536656020446, 0.033536757064443326, 0.03447921441728823, 0.06808063805171294,
                        0.09085754324229704, 0.019472382757094415, 0.047035687289814505, 0.03507983651492849, 0.005872816365294107, 0.04182588812333326,
                        0.03813541438217001, 0.05106860749224654, 0.1314217624007696, 0.040622531431670654, -0.07052607030663664, 0.03944530558145821,
                        0.02186551144434759, 0.0857886191723465 },
                { 0.026546698989913166, -0.017486821187736125, 0.012535051844670613, 0.017521982040632544, 0.027236598545917534, 0.008408173206506412,
                        0.004158695865907189, 0.02170176560458393, 0.005399496137037798, -1.8215889427383527E-4, 0.004396840151698384, 0.01652901805730591,
                        0.02756086217281535, 0.017066735312504376, 0.01889061843749912, 0.004443525289398188, 0.00333287955412777, 0.00546947833749213,
                        0.034478393601607336, -0.007567005997181369, 0.020033763532908824, 0.04736321361091196, 0.0420516619250631, 0.020889288168698082,
                        0.020922198988173032, 0.02956603640897039, 0.001926096039882604, 0.011139277407840622, 0.03429787348646608, 0.012619336502716363,
                        0.009259649775743773, 0.015574684120585276, 0.010658158278717597, -0.018324564705520144, 0.018127681645465587, 0.012521660197467813,
                        0.03198000522058489, 0.011923372533215718, 0.023246544180397097, 0.018898191260714407, 0.010909158677848852, 0.00920913923108069,
                        0.014496608339908932, 0.019075634816033964, 0.040622531431670654, 0.018005257895419482, -0.018558860537040932, 0.026879575045495624,
                        0.004092890868880355, 0.02872057100802301 },
                { -0.017418404248637314, 0.024363803663659582, 0.021558548648576707, -0.006765293855494431, -0.03842428850400834, -0.0016489167342999417,
                        0.03290478939550384, -0.0021242473539373396, 0.009031437303592226, 0.015615921525066984, 0.019028325977903757, -0.029772354336997466,
                        -0.03550132264263489, -0.004699251236597635, -0.029411932918713697, 0.011220480067533355, 0.019206430235579428, 0.031603402764409066,
                        0.007079765524602155, 0.020797051761852023, -0.005358901183943695, -0.045701468358177, -0.0316489759353913, -0.023618761077837096,
                        -0.02022803552319418, -0.01817814200088095, 0.03856854258899345, 0.015373528869895127, -0.05661228056545248, -0.024609112880508082,
                        -0.008551613873494155, -0.017193066237124143, 0.04202718286985202, -0.03179472978069248, -0.003474539960621243, -0.04313578439690932,
                        -0.05106520320215872, 0.0032182794022651345, 0.005138736094799597, 1.5652860364954703E-4, 0.03144826078116081, -8.380898281261503E-4,
                        1.8553111681012723E-4, 0.003210603476580604, -0.07052607030663664, -0.018558860537040932, 0.07009054671511898, -0.01661973558442202,
                        0.025380879336874274, -0.011343803871738604 },
                { 0.06820397632512222, -0.0012162700737118814, 0.009587912563895358, 0.04016487958361241, 0.036436158995993465, 0.023919868130552267,
                        0.015390742748574793, 0.06252769197217897, 0.018697635154368855, 0.004357209095811046, 0.01840757797427165, 0.020326593104194674,
                        0.06178301233393403, 0.029297047014579004, 0.05221184280142008, 0.009404379471913882, 0.019317476202761502, 0.009011668071378253,
                        0.049684254920940744, -0.010692079232214057, 0.03711792583088948, 0.06435540908768751, 0.07866961738675295, 0.024033306686366745,
                        0.03245926941714936, 0.05207857536090549, 0.015681998540460332, 0.013985598656701015, 0.04054973959123313, 0.012268410154407515,
                        0.033735580441681894, 0.04122951622759873, 0.025658383844895348, -0.03743223787386253, 0.03844334464754725, -0.024874056661572726,
                        0.05042579461303887, 0.020376786610218713, 0.045815821432056036, 0.04384463585170688, 0.026542003952573043, 0.019765928925993145,
                        0.01615861416251288, 0.042415834956967206, 0.03944530558145821, 0.026879575045495624, -0.01661973558442202, 0.06693218183609918,
                        0.01940252472705867, 0.028642068457574388 },
                { 0.07145051793685725, 0.06854255541487272, 0.04882426463077484, 0.055289365700728135, 0.00863332250828171, 0.04648771327495464,
                        0.04606648865075327, 0.10739084538421556, 0.0213878714946937, 0.005936139751894294, 0.045250297004903885, 0.0019097156376099392,
                        0.0664426411808189, 0.02197184983106578, 0.049609529647590765, 0.013592438160621766, 0.045551778582298336, 0.07839741290189277,
                        0.04396700152240977, 0.02509848265404834, 0.05486613162833511, 0.09890510499267771, 0.017612308286427958, 0.011316630462083147,
                        0.029004434950017314, 0.08315006419021534, 0.0395650297035591, 0.05041660997844717, 0.01769854337404704, -0.006232420244878353,
                        0.04516982398417219, 0.06753820543930868, 0.07994824224740979, 0.06986979680350897, 0.02965802603239724, -0.02932197299311745,
                        0.031686320199504264, 0.015699497963643227, 0.05594747496550899, 0.040909517127887624, 0.03683214201110552, 0.06420140775392918,
                        0.025194762816263844, 0.07993440843712808, 0.02186551144434759, 0.004092890868880355, 0.025380879336874274, 0.01940252472705867,
                        0.10258895126688856, 0.049542045175289165 },
                { 0.07488933945702014, 0.034670324121890486, 0.09610536340475627, 0.05723838827961253, 0.057612618094148096, 0.045564958659100915,
                        0.03608010348517658, 0.08599748217024838, 0.012403124600014092, 0.0023039285511616487, 0.017038474480294552, 0.027931897285779175,
                        0.07761625501776233, 0.045802363757645, 0.026024036979239613, 0.01825292123322788, 0.01717867893975092, 0.07297587897441953,
                        0.09564667778749408, 0.011461561228721894, 0.07020416946484041, 0.11958113266783735, 0.04957234984180621, 0.06466631042915777,
                        0.05365272248309082, 0.08281085333066818, 0.016758434473834218, 0.05149742362471869, 0.08064609382386137, 0.029872753619924725,
                        0.017975561143856063, 0.035999031659639635, 0.08668847925774631, 0.007740533499578341, 0.04917159564434252, 0.0402317400152294,
                        0.04600959200835599, 0.02406768527593103, 0.06807609191253781, 0.048209565982487435, 0.05065130246810593, 0.07512769631140155,
                        0.05144271760390398, 0.07163565691765367, 0.0857886191723465, 0.02872057100802301, -0.011343803871738604, 0.028642068457574388,
                        0.049542045175289165, 0.11660793295215936 } };
        public static final double[][] dataSetMtrx = {
                { 0.05, 0.05, 0.07, 0.04, 0.07, 0.09, 0.04, 0.14, 0.05, 0.05, 0.11, 0.09, 0.12, 0.06, 0.1, 0.05, 0.05, 0.08, 0.11, 0.11, 0.06, 0.09, 0.09, 0.14,
                        0.07, 0.15, 0.07, 0.06, 0.06, 0.05, 0.05, 0.09, 0.1, 0.04, 0.05, 0.08, 0.07, 0.04, 0.09, 0.12, 0.1, 0.07, 0.12, 0.14, 0.1, 0.08, 0.05,
                        0.09, 0.13, 0.08 },
                { 0.11, 0.63, 0.67, 0.21, 0.0, 0.35, 0.11, 0.72, 0.37, 0.49, 0.34, 0.5, 0.32, 0.54, 0.5, 0.06, 0.19, 0.01, 0.61, 0.55, 0.68, 0.01, 0.64, 0.56,
                        0.65, 0.55, 0.31, 0.57, 0.02, 0.03, 0.25, 0.23, 0.24, 0.26, 0.07, 0.55, 0.44, 0.22, 0.5, 0.31, 0.36, 0.62, 0.37, 0.33, 0.56, 0.38, 0.61,
                        0.11, 0.88, 0.31 },
                { 1.04, 1.13, 1.12, 1.1, 1.24, 0.98, 0.91, 1.88, 0.31, 0.17, 1.03, 0.78, 1.43, 0.71, 0.61, 0.87, 0.69, 1.6, 0.88, 0.82, 1.21, 1.56, 1.69, 1.42,
                        1.21, 1.89, 0.95, 0.93, 1.37, 0.85, 0.87, 1.81, 1.48, 0.92, 0.62, 0.72, 1.36, 0.25, 1.12, 1.45, 1.12, 0.94, 1.79, 1.67, 1.46, 0.82,
                        0.22, 1.69, 1.32, 1.25 } };
        public static final double[] optimisationSolution = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.5, 0.0, 0.27927998649368463, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.05376189760925540, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.05, 0.0, 0.05, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 0.0, 0.0, 0.06695811589705995, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                0.0, 1.0, 0.0, 0.0, 0.0 };

        Case050B() {
            super(covarianceMtrx, dataSetMtrx, optimisationSolution, -0.009423725847017339, -0.008263637132839593);
        }

    }

    public static abstract class CaseData {

        private final double[][] myCovarianceMtrx;
        private final double[][] myDataSetMtrx;
        private final double myEstimatedValue;
        private final double myOptimalValue;
        private final double[] myOptimisationSolution;

        CaseData(final double[][] covarianceMtrx, final double[][] dataSetMtrx, final double[] optimisationSolution, final double estimatedValue,
                final double optimalValue) {
            super();
            myCovarianceMtrx = covarianceMtrx;
            myDataSetMtrx = dataSetMtrx;
            myOptimisationSolution = optimisationSolution;
            myEstimatedValue = estimatedValue;
            myOptimalValue = optimalValue;
        }

        public double getBeta(final int asset) {
            return myDataSetMtrx[2][asset];
        }

        public double[] getBetaVctr() {
            return myDataSetMtrx[2];
        }

        public double getCovariance(final int assetR, final int assetC) {
            return myCovarianceMtrx[assetR][assetC];
        }

        public double[][] getCovarianceMtrx() {
            return myCovarianceMtrx;
        }

        public double[][] getDataSetMtrx() {
            return myDataSetMtrx;
        }

        /**
         * @return Optimal value from the sequential model
         */
        public double getEstimatedValue() {
            return myEstimatedValue;
        }

        public double getMargin(final int asset) {
            return myDataSetMtrx[1][asset];
        }

        public double[] getMarginVctr() {
            return myDataSetMtrx[1];
        }

        /**
         * @return Optimal value from the all-in-one model
         */
        public double getOptimalValue() {
            return myOptimalValue;
        }

        /**
         * @return Optimal solution from the all-in-one model
         */
        public double[] getOptimisationSolution() {
            return myOptimisationSolution;
        }

        public double getOptimisationSolution(final int asset) {
            return myOptimisationSolution[asset];
        }

        public double getReturn(final int asset) {
            return myDataSetMtrx[0][asset];
        }

        public double[] getReturnVctr() {
            return myDataSetMtrx[0];
        }

        public int numberOfAssets() {
            return myCovarianceMtrx.length;
        }

        public MatrixR064 toCleanedCovariances() {
            return NextGenSysModTest.toCovariances(this.toVolatilities(), this.toCorrelations());
        }

        public MatrixR064 toCorrelations() {
            return NextGenSysModTest.toCorrelations(MatrixR064.FACTORY.copy(RawStore.wrap(myCovarianceMtrx)), true);
        }

        public MatrixR064 toVolatilities() {
            return NextGenSysModTest.toVolatilities(MatrixR064.FACTORY.copy(RawStore.wrap(myCovarianceMtrx)), true);
        }

    }

    public static final Case010A CASE_010A = new Case010A();
    public static final Case020A CASE_020A = new Case020A();
    public static final Case030B CASE_030B = new Case030B();
    public static final Case040B CASE_040B = new Case040B();
    public static final Case050B CASE_050B = new Case050B();

    private static final NumberContext SOLUTION_ACCURACY = NumberContext.of(7, 6).withMode(RoundingMode.HALF_DOWN);
    private static final NumberContext VALIDATION_ACCURACY = SOLUTION_ACCURACY.withPrecision(5);

    public static void assertSolution(final ExpressionsBasedModel model, final double[] expected, final Optimisation.Result actual,
            final NumberContext accuracy) {

        // Check that the solver (thinks) it found the optimal solution
        TestUtils.assertStateNotLessThanOptimal(actual);

        // Check that the solution is valid
        TestUtils.assertTrue(model.validate(actual, accuracy, BasicLogger.ERROR));

        // Check that it is the expected solution
        TestUtils.assertEquals(Access1D.wrap(expected), actual, accuracy);
    }

    public static ExpressionsBasedModel buildModel(final CaseData data) {

        MatrixR064 covar = data.toCleanedCovariances();

        ExpressionsBasedModel retVal = new ExpressionsBasedModel();

        int numberOfAssets = data.numberOfAssets();

        double[] returnVctr = data.getReturnVctr();
        double[] marginVctr = data.getMarginVctr();
        double[] betaVctr = data.getBetaVctr();

        SampleSet marginSamples = SampleSet.wrap(marginVctr);
        SampleSet betaSamples = SampleSet.wrap(betaVctr);

        double marginLimit = marginSamples.getQuartile1();
        double betaLimit = betaSamples.getQuartile3();

        Expression marginExpr = retVal.newExpression("Margin").lower(marginLimit);
        Expression betaExpr = retVal.newExpression("Beta").upper(betaLimit);
        Expression totalExpr = retVal.newExpression("100%").level(BigMath.ONE);
        Expression budgetExpr = retVal.newExpression("Budget").upper(Math.toIntExact(Math.round(Math.sqrt(numberOfAssets))));
        Expression varianceExpr = retVal.newExpression("Variance").weight(BigMath.DIVIDE.invoke(BigMath.HUNDRED, BigMath.TWO).negate());

        for (int j = 0; j < numberOfAssets; j++) {

            Variable weightVar = retVal.newVariable("X" + j).weight(returnVctr[j]).lower(BigMath.ZERO).upper(BigMath.HALF);

            marginExpr.set(weightVar, marginVctr[j]);
            betaExpr.set(weightVar, betaVctr[j]);
            totalExpr.set(weightVar, BigMath.ONE);

            for (int i = 0; i < numberOfAssets; i++) {
                varianceExpr.set(i, j, covar.doubleValue(i, j));
            }
        }

        for (int j = 0; j < numberOfAssets; j++) {

            Variable weightVar = retVal.getVariable(j);
            Variable activationVar = retVal.newVariable(weightVar.getName() + "_Activator").binary();

            budgetExpr.set(activationVar, BigMath.ONE);

            retVal.newExpression("Trigger_" + weightVar.getName()).set(weightVar, BigMath.ONE).set(activationVar, BigDecimal.valueOf(-0.05))
                    .lower(BigMath.ZERO);
            retVal.newExpression("Active__" + weightVar.getName()).set(weightVar, BigMath.ONE).set(activationVar, BigMath.NEG).upper(BigMath.ZERO);
        }

        // retVal.options.debug(ConvexSolver.class);
        retVal.options.solution = SOLUTION_ACCURACY;

        return retVal;
    }

    public static Optimisation.Result solveSequentially(final CaseData data) {

        MatrixR064 covar = data.toCleanedCovariances();

        int numberOfAssets = data.numberOfAssets();

        ExpressionsBasedModel model = new ExpressionsBasedModel();
        // model.options.debug(ConvexSolver.class);
        model.options.solution = SOLUTION_ACCURACY;

        double[] returnVctr = data.getReturnVctr();
        double[] marginVctr = data.getMarginVctr();
        double[] betaVctr = data.getBetaVctr();

        SampleSet marginSamples = SampleSet.wrap(marginVctr);
        SampleSet betaSamples = SampleSet.wrap(betaVctr);

        double marginLimit = marginSamples.getQuartile1();
        double betaLimit = betaSamples.getQuartile3();

        Expression marginExpr = model.newExpression("Margin").lower(marginLimit);
        Expression betaExpr = model.newExpression("Beta").upper(betaLimit);
        Expression totalExpr = model.newExpression("100%").level(BigMath.ONE);

        for (int j = 0; j < numberOfAssets; j++) {

            Variable weightVar = model.newVariable("X" + j).weight(returnVctr[j]).lower(BigMath.ZERO).upper(BigMath.HALF);

            marginExpr.set(weightVar, marginVctr[j]);
            betaExpr.set(weightVar, betaVctr[j]);
            totalExpr.set(weightVar, BigMath.ONE);
        }

        Optimisation.Result linRes = model.maximise();
        if (!linRes.getState().isFeasible()) {
            return linRes;
        }

        Expression varianceExpr = model.newExpression("Variance").weight(BigMath.DIVIDE.invoke(BigMath.HUNDRED, BigMath.TWO).negate());
        for (int j = 0; j < numberOfAssets; j++) {
            for (int i = 0; i < numberOfAssets; i++) {
                varianceExpr.set(i, j, covar.get(i, j));
            }
        }

        Optimisation.Result quadRes = model.maximise();
        if (!quadRes.getState().isFeasible()) {
            // This should not happen, but if it does return the linear solution since it was atleast feasible
            return linRes;
        }

        Expression budgetExpr = model.newExpression("Budget").upper(Math.toIntExact(Math.round(Math.sqrt(numberOfAssets))));
        for (int j = 0; j < numberOfAssets; j++) {

            Variable weightVar = model.getVariable(j);

            if (weightVar.getValue().compareTo(BigMath.ZERO) == 0) {

                weightVar.level(BigMath.ZERO);

            } else {

                Variable activationVar = model.newVariable(weightVar.getName() + "_Activator").binary();

                budgetExpr.set(activationVar, BigMath.ONE);

                model.newExpression("Trigger_" + weightVar.getName()).set(weightVar, BigMath.ONE).set(activationVar, BigDecimal.valueOf(-0.05))
                        .lower(BigMath.ZERO);
                model.newExpression("Active__" + weightVar.getName()).set(weightVar, BigMath.ONE).set(activationVar, BigMath.NEG).upper(BigMath.ZERO);
            }
        }

        return model.maximise();
    }

    /**
     * <p>
     * Will extract the correlation coefficients from the input covariance matrix. If "cleaning" is enabled
     * small and negative eigenvalues of the covariance matrix will be replaced with a new minimal value.
     * </p>
     * <p>
     * Copied from ojAlgo-finance v2.1.1-SNAPSHOT (2019-05-23) org.ojalgo.finance.FinanceUtils.
     * </p>
     */
    static MatrixR064 toCorrelations(final Access2D<?> covariances, final boolean clean) {

        int size = Math.toIntExact(Math.min(covariances.countRows(), covariances.countColumns()));

        MatrixStore<Double> covarianceMtrx = R064Store.FACTORY.makeWrapper(covariances);

        if (clean) {

            Eigenvalue<Double> evd = Eigenvalue.R064.make(covarianceMtrx, true);
            evd.decompose(covarianceMtrx);

            MatrixStore<Double> mtrxV = evd.getV();
            PhysicalStore<Double> mtrxD = evd.getD().copy();

            double largest = evd.getEigenvalues().get(0).norm();
            double limit = largest * size * PrimitiveMath.RELATIVELY_SMALL;

            for (int ij = 0; ij < size; ij++) {
                if (mtrxD.doubleValue(ij, ij) < limit) {
                    mtrxD.set(ij, ij, limit);
                }
            }

            covarianceMtrx = mtrxV.multiply(mtrxD).multiply(mtrxV.transpose());
        }

        MatrixR064.DenseReceiver retVal = MatrixR064.FACTORY.newDenseBuilder(size, size);

        double[] volatilities = new double[size];
        for (int ij = 0; ij < size; ij++) {
            volatilities[ij] = PrimitiveMath.SQRT.invoke(covarianceMtrx.doubleValue(ij, ij));
        }

        for (int j = 0; j < size; j++) {
            double colVol = volatilities[j];

            retVal.set(j, j, PrimitiveMath.ONE);

            for (int i = j + 1; i < size; i++) {
                double rowVol = volatilities[i];

                if (rowVol <= PrimitiveMath.ZERO || colVol <= PrimitiveMath.ZERO) {

                    retVal.set(i, j, PrimitiveMath.ZERO);
                    retVal.set(j, i, PrimitiveMath.ZERO);

                } else {

                    double covariance = covarianceMtrx.doubleValue(i, j);
                    double correlation = covariance / (rowVol * colVol);

                    retVal.set(i, j, correlation);
                    retVal.set(j, i, correlation);
                }
            }
        }

        return retVal.get();
    }

    /**
     * <p>
     * Vill constract a covariance matrix from the standard deviations (volatilities) and correlation
     * coefficient,
     * </p>
     * <p>
     * Copied from ojAlgo-finance v2.1.1-SNAPSHOT (2019-05-23) org.ojalgo.finance.FinanceUtils.
     * </p>
     */
    static MatrixR064 toCovariances(final Access1D<?> volatilities, final Access2D<?> correlations) {

        int tmpSize = (int) volatilities.count();

        MatrixR064.DenseReceiver retVal = MatrixR064.FACTORY.newDenseBuilder(tmpSize, tmpSize);

        for (int j = 0; j < tmpSize; j++) {
            double tmpColumnVolatility = volatilities.doubleValue(j);
            retVal.set(j, j, tmpColumnVolatility * tmpColumnVolatility);
            for (int i = j + 1; i < tmpSize; i++) {
                double tmpCovariance = volatilities.doubleValue(i) * correlations.doubleValue(i, j) * tmpColumnVolatility;
                retVal.set(i, j, tmpCovariance);
                retVal.set(j, i, tmpCovariance);
            }
        }

        return retVal.get();
    }

    /**
     * <p>
     * Will extract the standard deviations (volatilities) from the input covariance matrix. If "cleaning" is
     * enabled small variances will be replaced with a new minimal value.
     * </p>
     * <p>
     * Copied from ojAlgo-finance v2.1.1-SNAPSHOT (2019-05-23) org.ojalgo.finance.FinanceUtils.
     * </p>
     */
    static MatrixR064 toVolatilities(final Access2D<?> covariances, final boolean clean) {

        int size = Math.toIntExact(Math.min(covariances.countRows(), covariances.countColumns()));

        MatrixR064.DenseReceiver retVal = MatrixR064.FACTORY.makeDense(size);

        if (clean) {

            MatrixStore<Double> covarianceMtrx = R064Store.FACTORY.makeWrapper(covariances);

            double largest = covarianceMtrx.aggregateDiagonal(Aggregator.LARGEST);
            double limit = largest * size * PrimitiveMath.RELATIVELY_SMALL;
            double smallest = PrimitiveMath.SQRT.invoke(limit);

            for (int ij = 0; ij < size; ij++) {
                double variance = covariances.doubleValue(ij, ij);

                if (variance < limit) {
                    retVal.set(ij, smallest);
                } else {
                    retVal.set(ij, PrimitiveMath.SQRT.invoke(variance));
                }
            }

        } else {

            for (int ij = 0; ij < size; ij++) {
                double variance = covariances.doubleValue(ij, ij);

                if (variance <= PrimitiveMath.ZERO) {
                    retVal.set(ij, PrimitiveMath.ZERO);
                } else {
                    retVal.set(ij, PrimitiveMath.SQRT.invoke(variance));
                }
            }
        }

        return retVal.get();
    }

    @Test
    @Tag("unstable")
    public void testSequentialCase010A() {
        this.doTestSequential(CASE_010A);
    }

    @Test
    @Tag("unstable")
    public void testSequentialCase020A() {
        this.doTestSequential(CASE_020A);
    }

    @Test
    @Tag("unstable")
    public void testSequentialCase030B() {
        this.doTestSequential(CASE_030B);
    }

    @Test
    public void testSequentialCase040B() {
        this.doTestSequential(CASE_040B);
    }

    @Test
    public void testSequentialCase050B() {
        this.doTestSequential(CASE_050B);
    }

    protected void doTestAllInOne(final CaseData testCase) {

        ExpressionsBasedModel model = NextGenSysModTest.buildModel(testCase);

        Result result = model.maximise();

        if (OptimisationIntegerTests.DEBUG) {
            BasicLogger.debug("  Result: {}", result);
        }

        NextGenSysModTest.assertSolution(model, testCase.getOptimisationSolution(), result, VALIDATION_ACCURACY);
    }

    protected void doTestSequential(final CaseData testCase) {

        Result estimate = NextGenSysModTest.solveSequentially(testCase);

        if (OptimisationIntegerTests.DEBUG) {
            BasicLogger.debug("Estimate: {}", estimate);
        }

        TestUtils.assertEquals(testCase.getEstimatedValue(), estimate.getValue(), 0.0001);
    }

}
