# Sorting utilities for musiclibrarian.
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import treemodel
import types

# like cmp, but case-insensitive for strings.  Could be more efficient?
def caseicmp(a, b):
    if type(a)==types.StringType:
        a=a.upper()
    if type(b)==types.StringType:
        b=b.upper()

    return cmp(a,b)

# Returns a function to sort using casecmp(), given a function on
# iterators.  The function should take two arguments (model, iter) and
# return the data used for comparison.
def sort_by(f, invert=0):
    if not invert:
        return lambda model, a, b, *rest:caseicmp(f(model, a), f(model, b))
    else:
        return lambda model, a, b, *rest:-caseicmp(f(model, a), f(model, b))

# data fetcher for the above; reads the raw column data
def column(column):
    def f(model, iter):
        val=model.get_value(iter, column)
        if type(val)==types.StringType:
            return val.lower()
        else:
            return val

    return f

# data fetcher to read a tag
def tag(tag):
    def get_tag(model, iter):
        val=model.get_value(iter, treemodel.COLUMN_PYOBJ).get_tag(tag)
        if len(val)==0:
            return None
        else:
            return val[0].lower()

    return get_tag

# helper to convert a string to an integer, passing None through.
def safeint(val):
    if val==None:
        return None
    else:
        return int(val)

# Given some number of procedures with the signature of a tree sorter,
# returns a new sorter which uses them to sort lexicographically.
def make_lexicographic_sorter(*sorters):
    def sorter(model,a,b,*rest):
        for f in sorters:
            val=apply(f, (model, a, b)+rest)
            if val<>0:
                return val

        return 0

    return sorter

# Returns a sorter to sort by track number if available, then by
# label.  The track number will only be used if both iterators are at
# depth "max_depth".
def make_basic_sort(max_depth):
    doublesort=make_lexicographic_sorter(sort_by(lambda model, iter:safeint(tag('tracknumber')(model, iter))), sort_by(column(treemodel.COLUMN_LABEL)))
    singlesort=sort_by(column(treemodel.COLUMN_LABEL))

    def sorter(model, a, b):
        if model.iter_depth(a)==max_depth and model.iter_depth(b)==max_depth:
            return doublesort(model, a, b)
        else:
            return singlesort(model, a, b)

    return sorter

