/*
 * SPDX-FileCopyrightText: 2017~2017 CSSlayer <wengxt@gmail.com>
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 */

#include "jyutpingdata.h"

namespace libime {
namespace jyutping {

const std::vector<bool> &getEncodedInitialFinal() {
    static const auto encodedInitialFinal = []() {
        std::vector<bool> a;
        std::unordered_set<int16_t> encodedInitialFinalSet;
        int16_t maxCode = 0;
        for (auto &p : getJyutpingMap()) {

            auto initial = p.initial();
            auto final = p.final();

            if (p.fuzzy()) {
                continue;
            }

            int16_t encode =
                ((static_cast<int16_t>(initial) -
                  JyutpingEncoder::firstInitial) *
                 (JyutpingEncoder::lastFinal - JyutpingEncoder::firstFinal +
                  1)) +
                (static_cast<int16_t>(final) - JyutpingEncoder::firstFinal);
            encodedInitialFinalSet.insert(encode);

            maxCode = std::max(encode, maxCode);
        }
        a.resize(maxCode + 1);
        std::fill(a.begin(), a.end(), false);
        for (auto i : encodedInitialFinalSet) {
            a[i] = true;
        }
        return a;
    }();

    return encodedInitialFinal;
}

const std::unordered_map<std::string, std::pair<std::string, std::string>> &
getInnerSegment() {
    static const std::unordered_map<std::string,
                                    std::pair<std::string, std::string>>
        innerSegment = {
            {"ceoi", {"ce", "oi"}}, {"ceon", {"ce", "on"}},
            {"deoi", {"de", "oi"}}, {"deon", {"de", "on"}},
            {"geoi", {"ge", "oi"}}, {"jeoi", {"je", "oi"}},
            {"jeon", {"je", "on"}}, {"keoi", {"ke", "oi"}},
            {"leoi", {"le", "oi"}}, {"leon", {"le", "on"}},
            {"loei", {"lo", "ei"}}, {"naai", {"na", "ai"}},
            {"naam", {"na", "am"}}, {"naau", {"na", "au"}},
            {"neoi", {"ne", "oi"}}, {"seoi", {"se", "oi"}},
            {"seon", {"se", "on"}}, {"zeoi", {"ze", "oi"}},
            {"zeon", {"ze", "on"}},
        };

    return innerSegment;
}

const JyutpingMap &getJyutpingMap() {
    static const JyutpingMap jyutpingMap = {
        {"aa", JyutpingInitial::Zero, JyutpingFinal::AA},
        {"aai", JyutpingInitial::Zero, JyutpingFinal::AAI},
        {"aak", JyutpingInitial::Zero, JyutpingFinal::AAK},
        {"aam", JyutpingInitial::Zero, JyutpingFinal::AAM},
        {"aan", JyutpingInitial::Zero, JyutpingFinal::AAN},
        {"aang", JyutpingInitial::Zero, JyutpingFinal::AANG},
        {"aap", JyutpingInitial::Zero, JyutpingFinal::AAP},
        {"aat", JyutpingInitial::Zero, JyutpingFinal::AAT},
        {"aau", JyutpingInitial::Zero, JyutpingFinal::AAU},
        {"ai", JyutpingInitial::Zero, JyutpingFinal::AI},
        {"ak", JyutpingInitial::Zero, JyutpingFinal::AK},
        {"am", JyutpingInitial::Zero, JyutpingFinal::AM},
        {"an", JyutpingInitial::Zero, JyutpingFinal::AN},
        {"ang", JyutpingInitial::Zero, JyutpingFinal::ANG},
        {"ap", JyutpingInitial::Zero, JyutpingFinal::AP},
        {"at", JyutpingInitial::Zero, JyutpingFinal::AT},
        {"au", JyutpingInitial::Zero, JyutpingFinal::AU},
        {"baa", JyutpingInitial::B, JyutpingFinal::AA},
        {"baai", JyutpingInitial::B, JyutpingFinal::AAI},
        {"baak", JyutpingInitial::B, JyutpingFinal::AAK},
        {"baan", JyutpingInitial::B, JyutpingFinal::AAN},
        {"baang", JyutpingInitial::B, JyutpingFinal::AANG},
        {"baat", JyutpingInitial::B, JyutpingFinal::AAT},
        {"baau", JyutpingInitial::B, JyutpingFinal::AAU},
        {"bai", JyutpingInitial::B, JyutpingFinal::AI},
        {"bak", JyutpingInitial::B, JyutpingFinal::AK},
        {"bam", JyutpingInitial::B, JyutpingFinal::AM},
        {"ban", JyutpingInitial::B, JyutpingFinal::AN},
        {"bang", JyutpingInitial::B, JyutpingFinal::ANG},
        {"bat", JyutpingInitial::B, JyutpingFinal::AT},
        {"bau", JyutpingInitial::B, JyutpingFinal::AU},
        {"be", JyutpingInitial::B, JyutpingFinal::E},
        {"bei", JyutpingInitial::B, JyutpingFinal::EI},
        {"bek", JyutpingInitial::B, JyutpingFinal::EK},
        {"beng", JyutpingInitial::B, JyutpingFinal::ENG},
        {"bi", JyutpingInitial::B, JyutpingFinal::I},
        {"bik", JyutpingInitial::B, JyutpingFinal::IK},
        {"bin", JyutpingInitial::B, JyutpingFinal::IN},
        {"bing", JyutpingInitial::B, JyutpingFinal::ING},
        {"bit", JyutpingInitial::B, JyutpingFinal::IT},
        {"biu", JyutpingInitial::B, JyutpingFinal::IU},
        {"bo", JyutpingInitial::B, JyutpingFinal::O},
        {"bok", JyutpingInitial::B, JyutpingFinal::OK},
        {"bong", JyutpingInitial::B, JyutpingFinal::ONG},
        {"bou", JyutpingInitial::B, JyutpingFinal::OU},
        {"bu", JyutpingInitial::B, JyutpingFinal::U},
        {"bui", JyutpingInitial::B, JyutpingFinal::UI},
        {"buk", JyutpingInitial::B, JyutpingFinal::UK},
        {"bun", JyutpingInitial::B, JyutpingFinal::UN},
        {"bung", JyutpingInitial::B, JyutpingFinal::UNG},
        {"but", JyutpingInitial::B, JyutpingFinal::UT},
        {"caa", JyutpingInitial::C, JyutpingFinal::AA},
        {"caai", JyutpingInitial::C, JyutpingFinal::AAI},
        {"caak", JyutpingInitial::C, JyutpingFinal::AAK},
        {"caam", JyutpingInitial::C, JyutpingFinal::AAM},
        {"caan", JyutpingInitial::C, JyutpingFinal::AAN},
        {"caang", JyutpingInitial::C, JyutpingFinal::AANG},
        {"caap", JyutpingInitial::C, JyutpingFinal::AAP},
        {"caat", JyutpingInitial::C, JyutpingFinal::AAT},
        {"caau", JyutpingInitial::C, JyutpingFinal::AAU},
        {"cai", JyutpingInitial::C, JyutpingFinal::AI},
        {"cak", JyutpingInitial::C, JyutpingFinal::AK},
        {"cam", JyutpingInitial::C, JyutpingFinal::AM},
        {"can", JyutpingInitial::C, JyutpingFinal::AN},
        {"cang", JyutpingInitial::C, JyutpingFinal::ANG},
        {"cap", JyutpingInitial::C, JyutpingFinal::AP},
        {"cat", JyutpingInitial::C, JyutpingFinal::AT},
        {"cau", JyutpingInitial::C, JyutpingFinal::AU},
        {"ce", JyutpingInitial::C, JyutpingFinal::E},
        {"cek", JyutpingInitial::C, JyutpingFinal::EK},
        {"ceng", JyutpingInitial::C, JyutpingFinal::ENG},
        {"ceoi", JyutpingInitial::C, JyutpingFinal::EOI},
        {"ceon", JyutpingInitial::C, JyutpingFinal::EON},
        {"ceot", JyutpingInitial::C, JyutpingFinal::EOT},
        {"ci", JyutpingInitial::C, JyutpingFinal::I},
        {"cik", JyutpingInitial::C, JyutpingFinal::IK},
        {"cim", JyutpingInitial::C, JyutpingFinal::IM},
        {"cin", JyutpingInitial::C, JyutpingFinal::IN},
        {"cing", JyutpingInitial::C, JyutpingFinal::ING},
        {"cip", JyutpingInitial::C, JyutpingFinal::IP},
        {"cit", JyutpingInitial::C, JyutpingFinal::IT},
        {"ciu", JyutpingInitial::C, JyutpingFinal::IU},
        {"co", JyutpingInitial::C, JyutpingFinal::O},
        {"coek", JyutpingInitial::C, JyutpingFinal::OEK},
        {"coeng", JyutpingInitial::C, JyutpingFinal::OENG},
        {"coi", JyutpingInitial::C, JyutpingFinal::OI},
        {"cok", JyutpingInitial::C, JyutpingFinal::OK},
        {"cong", JyutpingInitial::C, JyutpingFinal::ONG},
        {"cou", JyutpingInitial::C, JyutpingFinal::OU},
        {"cuk", JyutpingInitial::C, JyutpingFinal::UK},
        {"cun", JyutpingInitial::C, JyutpingFinal::UN},
        {"cung", JyutpingInitial::C, JyutpingFinal::UNG},
        {"cyu", JyutpingInitial::C, JyutpingFinal::YU},
        {"cyun", JyutpingInitial::C, JyutpingFinal::YUN},
        {"cyut", JyutpingInitial::C, JyutpingFinal::YUT},
        {"daa", JyutpingInitial::D, JyutpingFinal::AA},
        {"daai", JyutpingInitial::D, JyutpingFinal::AAI},
        {"daak", JyutpingInitial::D, JyutpingFinal::AAK},
        {"daam", JyutpingInitial::D, JyutpingFinal::AAM},
        {"daan", JyutpingInitial::D, JyutpingFinal::AAN},
        {"daang", JyutpingInitial::D, JyutpingFinal::AANG},
        {"daap", JyutpingInitial::D, JyutpingFinal::AAP},
        {"daat", JyutpingInitial::D, JyutpingFinal::AAT},
        {"daau", JyutpingInitial::D, JyutpingFinal::AAU},
        {"dai", JyutpingInitial::D, JyutpingFinal::AI},
        {"dak", JyutpingInitial::D, JyutpingFinal::AK},
        {"dam", JyutpingInitial::D, JyutpingFinal::AM},
        {"dan", JyutpingInitial::D, JyutpingFinal::AN},
        {"dang", JyutpingInitial::D, JyutpingFinal::ANG},
        {"dap", JyutpingInitial::D, JyutpingFinal::AP},
        {"dat", JyutpingInitial::D, JyutpingFinal::AT},
        {"dau", JyutpingInitial::D, JyutpingFinal::AU},
        {"de", JyutpingInitial::D, JyutpingFinal::E},
        {"dei", JyutpingInitial::D, JyutpingFinal::EI},
        {"dek", JyutpingInitial::D, JyutpingFinal::EK},
        {"deng", JyutpingInitial::D, JyutpingFinal::ENG},
        {"deoi", JyutpingInitial::D, JyutpingFinal::EOI},
        {"deon", JyutpingInitial::D, JyutpingFinal::EON},
        {"deot", JyutpingInitial::D, JyutpingFinal::EOT},
        {"deu", JyutpingInitial::D, JyutpingFinal::EU},
        {"di", JyutpingInitial::D, JyutpingFinal::I},
        {"dik", JyutpingInitial::D, JyutpingFinal::IK},
        {"dim", JyutpingInitial::D, JyutpingFinal::IM},
        {"din", JyutpingInitial::D, JyutpingFinal::IN},
        {"ding", JyutpingInitial::D, JyutpingFinal::ING},
        {"dip", JyutpingInitial::D, JyutpingFinal::IP},
        {"dit", JyutpingInitial::D, JyutpingFinal::IT},
        {"diu", JyutpingInitial::D, JyutpingFinal::IU},
        {"do", JyutpingInitial::D, JyutpingFinal::O},
        {"doe", JyutpingInitial::D, JyutpingFinal::OE},
        {"doek", JyutpingInitial::D, JyutpingFinal::OEK},
        {"doeng", JyutpingInitial::D, JyutpingFinal::OENG},
        {"doi", JyutpingInitial::D, JyutpingFinal::OI},
        {"dok", JyutpingInitial::D, JyutpingFinal::OK},
        {"dong", JyutpingInitial::D, JyutpingFinal::ONG},
        {"dou", JyutpingInitial::D, JyutpingFinal::OU},
        {"duk", JyutpingInitial::D, JyutpingFinal::UK},
        {"dung", JyutpingInitial::D, JyutpingFinal::UNG},
        {"dyun", JyutpingInitial::D, JyutpingFinal::YUN},
        {"dyut", JyutpingInitial::D, JyutpingFinal::YUT},
        {"e", JyutpingInitial::Zero, JyutpingFinal::E},
        {"ei", JyutpingInitial::Zero, JyutpingFinal::EI},
        {"eot", JyutpingInitial::Zero, JyutpingFinal::EOT},
        {"faa", JyutpingInitial::F, JyutpingFinal::AA},
        {"faai", JyutpingInitial::F, JyutpingFinal::AAI},
        {"faak", JyutpingInitial::F, JyutpingFinal::AAK},
        {"faan", JyutpingInitial::F, JyutpingFinal::AAN},
        {"faat", JyutpingInitial::F, JyutpingFinal::AAT},
        {"faau", JyutpingInitial::F, JyutpingFinal::AAU},
        {"fai", JyutpingInitial::F, JyutpingFinal::AI},
        {"fan", JyutpingInitial::F, JyutpingFinal::AN},
        {"fang", JyutpingInitial::F, JyutpingFinal::ANG},
        {"fat", JyutpingInitial::F, JyutpingFinal::AT},
        {"fau", JyutpingInitial::F, JyutpingFinal::AU},
        {"fe", JyutpingInitial::F, JyutpingFinal::E},
        {"fen", JyutpingInitial::F, JyutpingFinal::EN},
        {"fei", JyutpingInitial::F, JyutpingFinal::EI},
        {"fet", JyutpingInitial::F, JyutpingFinal::ET},
        {"fik", JyutpingInitial::F, JyutpingFinal::IK},
        {"fing", JyutpingInitial::F, JyutpingFinal::ING},
        {"fit", JyutpingInitial::F, JyutpingFinal::IT},
        {"fiu", JyutpingInitial::F, JyutpingFinal::IU},
        {"fo", JyutpingInitial::F, JyutpingFinal::O},
        {"fok", JyutpingInitial::F, JyutpingFinal::OK},
        {"fong", JyutpingInitial::F, JyutpingFinal::ONG},
        {"fu", JyutpingInitial::F, JyutpingFinal::U},
        {"fui", JyutpingInitial::F, JyutpingFinal::UI},
        {"fuk", JyutpingInitial::F, JyutpingFinal::UK},
        {"fun", JyutpingInitial::F, JyutpingFinal::UN},
        {"fung", JyutpingInitial::F, JyutpingFinal::UNG},
        {"fut", JyutpingInitial::F, JyutpingFinal::UT},
        {"gaa", JyutpingInitial::G, JyutpingFinal::AA},
        {"gaai", JyutpingInitial::G, JyutpingFinal::AAI},
        {"gaak", JyutpingInitial::G, JyutpingFinal::AAK},
        {"gaam", JyutpingInitial::G, JyutpingFinal::AAM},
        {"gaan", JyutpingInitial::G, JyutpingFinal::AAN},
        {"gaang", JyutpingInitial::G, JyutpingFinal::AANG},
        {"gaap", JyutpingInitial::G, JyutpingFinal::AAP},
        {"gaat", JyutpingInitial::G, JyutpingFinal::AAT},
        {"gaau", JyutpingInitial::G, JyutpingFinal::AAU},
        {"gai", JyutpingInitial::G, JyutpingFinal::AI},
        {"gak", JyutpingInitial::G, JyutpingFinal::AK},
        {"gam", JyutpingInitial::G, JyutpingFinal::AM},
        {"gan", JyutpingInitial::G, JyutpingFinal::AN},
        {"gang", JyutpingInitial::G, JyutpingFinal::ANG},
        {"gap", JyutpingInitial::G, JyutpingFinal::AP},
        {"gat", JyutpingInitial::G, JyutpingFinal::AT},
        {"gau", JyutpingInitial::G, JyutpingFinal::AU},
        {"ge", JyutpingInitial::G, JyutpingFinal::E},
        {"gei", JyutpingInitial::G, JyutpingFinal::EI},
        {"gek", JyutpingInitial::G, JyutpingFinal::EK},
        {"geng", JyutpingInitial::G, JyutpingFinal::ENG},
        {"geoi", JyutpingInitial::G, JyutpingFinal::EOI},
        {"gep", JyutpingInitial::G, JyutpingFinal::EP},
        {"gi", JyutpingInitial::G, JyutpingFinal::I},
        {"gik", JyutpingInitial::G, JyutpingFinal::IK},
        {"gim", JyutpingInitial::G, JyutpingFinal::IM},
        {"gin", JyutpingInitial::G, JyutpingFinal::IN},
        {"ging", JyutpingInitial::G, JyutpingFinal::ING},
        {"gip", JyutpingInitial::G, JyutpingFinal::IP},
        {"git", JyutpingInitial::G, JyutpingFinal::IT},
        {"giu", JyutpingInitial::G, JyutpingFinal::IU},
        {"go", JyutpingInitial::G, JyutpingFinal::O},
        {"goe", JyutpingInitial::G, JyutpingFinal::OE},
        {"goek", JyutpingInitial::G, JyutpingFinal::OEK},
        {"goeng", JyutpingInitial::G, JyutpingFinal::OENG},
        {"goi", JyutpingInitial::G, JyutpingFinal::OI},
        {"gok", JyutpingInitial::G, JyutpingFinal::OK},
        {"gon", JyutpingInitial::G, JyutpingFinal::ON},
        {"gong", JyutpingInitial::G, JyutpingFinal::ONG},
        {"got", JyutpingInitial::G, JyutpingFinal::OT},
        {"gou", JyutpingInitial::G, JyutpingFinal::OU},
        {"gu", JyutpingInitial::G, JyutpingFinal::U},
        {"gui", JyutpingInitial::G, JyutpingFinal::UI},
        {"guk", JyutpingInitial::G, JyutpingFinal::UK},
        {"gut", JyutpingInitial::G, JyutpingFinal::UT},
        {"gun", JyutpingInitial::G, JyutpingFinal::UN},
        {"gung", JyutpingInitial::G, JyutpingFinal::UNG},
        {"gwaa", JyutpingInitial::GW, JyutpingFinal::AA},
        {"gwaai", JyutpingInitial::GW, JyutpingFinal::AAI},
        {"gwaak", JyutpingInitial::GW, JyutpingFinal::AAK},
        {"gwaan", JyutpingInitial::GW, JyutpingFinal::AAN},
        {"gwaang", JyutpingInitial::GW, JyutpingFinal::AANG},
        {"gwaat", JyutpingInitial::GW, JyutpingFinal::AAT},
        {"gwai", JyutpingInitial::GW, JyutpingFinal::AI},
        {"gwan", JyutpingInitial::GW, JyutpingFinal::AN},
        {"gwang", JyutpingInitial::GW, JyutpingFinal::ANG},
        {"gwat", JyutpingInitial::GW, JyutpingFinal::AT},
        {"gwe", JyutpingInitial::GW, JyutpingFinal::E},
        {"gwi", JyutpingInitial::GW, JyutpingFinal::I},
        {"gwik", JyutpingInitial::GW, JyutpingFinal::IK},
        {"gwing", JyutpingInitial::GW, JyutpingFinal::ING},
        {"gwit", JyutpingInitial::GW, JyutpingFinal::IT},
        {"gwo", JyutpingInitial::GW, JyutpingFinal::O},
        {"gwok", JyutpingInitial::GW, JyutpingFinal::OK},
        {"gwong", JyutpingInitial::GW, JyutpingFinal::ONG},
        {"gyun", JyutpingInitial::G, JyutpingFinal::YUN},
        {"gyut", JyutpingInitial::G, JyutpingFinal::YUT},
        {"haa", JyutpingInitial::H, JyutpingFinal::AA},
        {"haai", JyutpingInitial::H, JyutpingFinal::AAI},
        {"haak", JyutpingInitial::H, JyutpingFinal::AAK},
        {"haam", JyutpingInitial::H, JyutpingFinal::AAM},
        {"haan", JyutpingInitial::H, JyutpingFinal::AAN},
        {"haang", JyutpingInitial::H, JyutpingFinal::AANG},
        {"haap", JyutpingInitial::H, JyutpingFinal::AAP},
        {"haat", JyutpingInitial::H, JyutpingFinal::AAT},
        {"haau", JyutpingInitial::H, JyutpingFinal::AAU},
        {"hai", JyutpingInitial::H, JyutpingFinal::AI},
        {"hak", JyutpingInitial::H, JyutpingFinal::AK},
        {"ham", JyutpingInitial::H, JyutpingFinal::AM},
        {"han", JyutpingInitial::H, JyutpingFinal::AN},
        {"hang", JyutpingInitial::H, JyutpingFinal::ANG},
        {"hap", JyutpingInitial::H, JyutpingFinal::AP},
        {"hat", JyutpingInitial::H, JyutpingFinal::AT},
        {"hau", JyutpingInitial::H, JyutpingFinal::AU},
        {"he", JyutpingInitial::H, JyutpingFinal::E},
        {"hei", JyutpingInitial::H, JyutpingFinal::EI},
        {"hek", JyutpingInitial::H, JyutpingFinal::EK},
        {"heng", JyutpingInitial::H, JyutpingFinal::ENG},
        {"heoi", JyutpingInitial::H, JyutpingFinal::EOI},
        {"hi", JyutpingInitial::H, JyutpingFinal::I},
        {"hik", JyutpingInitial::H, JyutpingFinal::IK},
        {"him", JyutpingInitial::H, JyutpingFinal::IM},
        {"hin", JyutpingInitial::H, JyutpingFinal::IN},
        {"hing", JyutpingInitial::H, JyutpingFinal::ING},
        {"hip", JyutpingInitial::H, JyutpingFinal::IP},
        {"hit", JyutpingInitial::H, JyutpingFinal::IT},
        {"hiu", JyutpingInitial::H, JyutpingFinal::IU},
        {"hm", JyutpingInitial::H, JyutpingFinal::M},
        {"hng", JyutpingInitial::H, JyutpingFinal::NG},
        {"ho", JyutpingInitial::H, JyutpingFinal::O},
        {"hoe", JyutpingInitial::H, JyutpingFinal::OE},
        {"hoeng", JyutpingInitial::H, JyutpingFinal::OENG},
        {"hoi", JyutpingInitial::H, JyutpingFinal::OI},
        {"hok", JyutpingInitial::H, JyutpingFinal::OK},
        {"hon", JyutpingInitial::H, JyutpingFinal::ON},
        {"hong", JyutpingInitial::H, JyutpingFinal::ONG},
        {"hot", JyutpingInitial::H, JyutpingFinal::OT},
        {"hou", JyutpingInitial::H, JyutpingFinal::OU},
        {"huk", JyutpingInitial::H, JyutpingFinal::UK},
        {"hung", JyutpingInitial::H, JyutpingFinal::UNG},
        {"hyun", JyutpingInitial::H, JyutpingFinal::YUN},
        {"hyut", JyutpingInitial::H, JyutpingFinal::YUT},
        {"jaa", JyutpingInitial::J, JyutpingFinal::AA},
        {"jaai", JyutpingInitial::J, JyutpingFinal::AAI},
        {"jaang", JyutpingInitial::J, JyutpingFinal::AANG},
        {"jaak", JyutpingInitial::J, JyutpingFinal::AAK},
        {"jaam", JyutpingInitial::J, JyutpingFinal::AAM},
        {"jaap", JyutpingInitial::J, JyutpingFinal::AAP},
        {"jaau", JyutpingInitial::J, JyutpingFinal::AAU},
        {"jai", JyutpingInitial::J, JyutpingFinal::AI},
        {"jam", JyutpingInitial::J, JyutpingFinal::AM},
        {"jan", JyutpingInitial::J, JyutpingFinal::AN},
        {"jap", JyutpingInitial::J, JyutpingFinal::AP},
        {"jat", JyutpingInitial::J, JyutpingFinal::AT},
        {"jau", JyutpingInitial::J, JyutpingFinal::AU},
        {"je", JyutpingInitial::J, JyutpingFinal::E},
        {"jeng", JyutpingInitial::J, JyutpingFinal::ENG},
        {"jeoi", JyutpingInitial::J, JyutpingFinal::EOI},
        {"jeon", JyutpingInitial::J, JyutpingFinal::EON},
        {"ji", JyutpingInitial::J, JyutpingFinal::I},
        {"jik", JyutpingInitial::J, JyutpingFinal::IK},
        {"jim", JyutpingInitial::J, JyutpingFinal::IM},
        {"jin", JyutpingInitial::J, JyutpingFinal::IN},
        {"jing", JyutpingInitial::J, JyutpingFinal::ING},
        {"jip", JyutpingInitial::J, JyutpingFinal::IP},
        {"jit", JyutpingInitial::J, JyutpingFinal::IT},
        {"jiu", JyutpingInitial::J, JyutpingFinal::IU},
        {"jo", JyutpingInitial::J, JyutpingFinal::O},
        {"joek", JyutpingInitial::J, JyutpingFinal::OEK},
        {"joeng", JyutpingInitial::J, JyutpingFinal::OENG},
        {"joi", JyutpingInitial::J, JyutpingFinal::OI},
        {"juk", JyutpingInitial::J, JyutpingFinal::UK},
        {"jung", JyutpingInitial::J, JyutpingFinal::UNG},
        {"jyu", JyutpingInitial::J, JyutpingFinal::YU},
        {"jyun", JyutpingInitial::J, JyutpingFinal::YUN},
        {"jyut", JyutpingInitial::J, JyutpingFinal::YUT},
        {"kaa", JyutpingInitial::K, JyutpingFinal::AA},
        {"kaai", JyutpingInitial::K, JyutpingFinal::AAI},
        {"kaak", JyutpingInitial::K, JyutpingFinal::AAK},
        {"kaam", JyutpingInitial::K, JyutpingFinal::AAM},
        {"kaan", JyutpingInitial::K, JyutpingFinal::AAN},
        {"kaap", JyutpingInitial::K, JyutpingFinal::AAP},
        {"kaat", JyutpingInitial::K, JyutpingFinal::AAT},
        {"kaau", JyutpingInitial::K, JyutpingFinal::AAU},
        {"kai", JyutpingInitial::K, JyutpingFinal::AI},
        {"kak", JyutpingInitial::K, JyutpingFinal::AK},
        {"kam", JyutpingInitial::K, JyutpingFinal::AM},
        {"kan", JyutpingInitial::K, JyutpingFinal::AN},
        {"kang", JyutpingInitial::K, JyutpingFinal::ANG},
        {"kap", JyutpingInitial::K, JyutpingFinal::AP},
        {"kat", JyutpingInitial::K, JyutpingFinal::AT},
        {"kau", JyutpingInitial::K, JyutpingFinal::AU},
        {"ke", JyutpingInitial::K, JyutpingFinal::E},
        {"kei", JyutpingInitial::K, JyutpingFinal::EI},
        {"kek", JyutpingInitial::K, JyutpingFinal::EK},
        {"kem", JyutpingInitial::K, JyutpingFinal::EM},
        {"ken", JyutpingInitial::K, JyutpingFinal::EN},
        {"kep", JyutpingInitial::K, JyutpingFinal::EP},
        {"keoi", JyutpingInitial::K, JyutpingFinal::EOI},
        {"ki", JyutpingInitial::K, JyutpingFinal::I},
        {"kik", JyutpingInitial::K, JyutpingFinal::IK},
        {"kim", JyutpingInitial::K, JyutpingFinal::IM},
        {"kin", JyutpingInitial::K, JyutpingFinal::IN},
        {"king", JyutpingInitial::K, JyutpingFinal::ING},
        {"kip", JyutpingInitial::K, JyutpingFinal::IP},
        {"kit", JyutpingInitial::K, JyutpingFinal::IT},
        {"kiu", JyutpingInitial::K, JyutpingFinal::IU},
        {"ko", JyutpingInitial::K, JyutpingFinal::O},
        {"koe", JyutpingInitial::K, JyutpingFinal::OE},
        {"koek", JyutpingInitial::K, JyutpingFinal::OEK},
        {"koeng", JyutpingInitial::K, JyutpingFinal::OENG},
        {"koi", JyutpingInitial::K, JyutpingFinal::OI},
        {"kok", JyutpingInitial::K, JyutpingFinal::OK},
        {"kong", JyutpingInitial::K, JyutpingFinal::ONG},
        {"ku", JyutpingInitial::K, JyutpingFinal::U},
        {"kui", JyutpingInitial::K, JyutpingFinal::UI},
        {"kuk", JyutpingInitial::K, JyutpingFinal::UK},
        {"kung", JyutpingInitial::K, JyutpingFinal::UNG},
        {"kut", JyutpingInitial::K, JyutpingFinal::UT},
        {"kwaa", JyutpingInitial::KW, JyutpingFinal::AA},
        {"kwaai", JyutpingInitial::KW, JyutpingFinal::AAI},
        {"kwaak", JyutpingInitial::KW, JyutpingFinal::AAK},
        {"kwaan", JyutpingInitial::KW, JyutpingFinal::AAN},
        {"kwaang", JyutpingInitial::KW, JyutpingFinal::AANG},
        {"kwaat", JyutpingInitial::KW, JyutpingFinal::AAT},
        {"kwai", JyutpingInitial::KW, JyutpingFinal::AI},
        {"kwan", JyutpingInitial::KW, JyutpingFinal::AN},
        {"kwang", JyutpingInitial::KW, JyutpingFinal::ANG},
        {"kwik", JyutpingInitial::KW, JyutpingFinal::IK},
        {"kwok", JyutpingInitial::KW, JyutpingFinal::OK},
        {"kwong", JyutpingInitial::KW, JyutpingFinal::ONG},
        {"kwui", JyutpingInitial::KW, JyutpingFinal::UI},
        {"kwun", JyutpingInitial::KW, JyutpingFinal::UN},
        {"kwut", JyutpingInitial::KW, JyutpingFinal::UT},
        {"kyun", JyutpingInitial::K, JyutpingFinal::YUN},
        {"kyut", JyutpingInitial::K, JyutpingFinal::YUT},
        {"laa", JyutpingInitial::L, JyutpingFinal::AA},
        {"laai", JyutpingInitial::L, JyutpingFinal::AAI},
        {"laak", JyutpingInitial::L, JyutpingFinal::AAK},
        {"laam", JyutpingInitial::L, JyutpingFinal::AAM},
        {"laan", JyutpingInitial::L, JyutpingFinal::AAN},
        {"laang", JyutpingInitial::L, JyutpingFinal::AANG},
        {"laap", JyutpingInitial::L, JyutpingFinal::AAP},
        {"laat", JyutpingInitial::L, JyutpingFinal::AAT},
        {"laau", JyutpingInitial::L, JyutpingFinal::AAU},
        {"lai", JyutpingInitial::L, JyutpingFinal::AI},
        {"lak", JyutpingInitial::L, JyutpingFinal::AK},
        {"lam", JyutpingInitial::L, JyutpingFinal::AM},
        {"lan", JyutpingInitial::L, JyutpingFinal::AN},
        {"lang", JyutpingInitial::L, JyutpingFinal::ANG},
        {"lap", JyutpingInitial::L, JyutpingFinal::AP},
        {"lat", JyutpingInitial::L, JyutpingFinal::AT},
        {"lau", JyutpingInitial::L, JyutpingFinal::AU},
        {"le", JyutpingInitial::L, JyutpingFinal::E},
        {"lei", JyutpingInitial::L, JyutpingFinal::EI},
        {"lek", JyutpingInitial::L, JyutpingFinal::EK},
        {"lem", JyutpingInitial::L, JyutpingFinal::EM},
        {"leng", JyutpingInitial::L, JyutpingFinal::ENG},
        {"leoi", JyutpingInitial::L, JyutpingFinal::EOI},
        {"leon", JyutpingInitial::L, JyutpingFinal::EON},
        {"leot", JyutpingInitial::L, JyutpingFinal::EOT},
        {"li", JyutpingInitial::L, JyutpingFinal::I},
        {"lik", JyutpingInitial::L, JyutpingFinal::IK},
        {"lim", JyutpingInitial::L, JyutpingFinal::IM},
        {"lin", JyutpingInitial::L, JyutpingFinal::IN},
        {"ling", JyutpingInitial::L, JyutpingFinal::ING},
        {"lip", JyutpingInitial::L, JyutpingFinal::IP},
        {"lit", JyutpingInitial::L, JyutpingFinal::IT},
        {"liu", JyutpingInitial::L, JyutpingFinal::IU},
        {"lo", JyutpingInitial::L, JyutpingFinal::O},
        {"loe", JyutpingInitial::L, JyutpingFinal::OE},
        {"loek", JyutpingInitial::L, JyutpingFinal::OEK},
        {"loeng", JyutpingInitial::L, JyutpingFinal::OENG},
        {"loi", JyutpingInitial::L, JyutpingFinal::OI},
        {"lok", JyutpingInitial::L, JyutpingFinal::OK},
        {"long", JyutpingInitial::L, JyutpingFinal::ONG},
        {"lou", JyutpingInitial::L, JyutpingFinal::OU},
        {"luk", JyutpingInitial::L, JyutpingFinal::UK},
        {"lung", JyutpingInitial::L, JyutpingFinal::UNG},
        {"lyun", JyutpingInitial::L, JyutpingFinal::YUN},
        {"lyut", JyutpingInitial::L, JyutpingFinal::YUT},
        {"m", JyutpingInitial::M, JyutpingFinal::Zero},
        {"maa", JyutpingInitial::M, JyutpingFinal::AA},
        {"maai", JyutpingInitial::M, JyutpingFinal::AAI},
        {"maak", JyutpingInitial::M, JyutpingFinal::AAK},
        {"maan", JyutpingInitial::M, JyutpingFinal::AAN},
        {"maang", JyutpingInitial::M, JyutpingFinal::AANG},
        {"maat", JyutpingInitial::M, JyutpingFinal::AAT},
        {"maau", JyutpingInitial::M, JyutpingFinal::AAU},
        {"mai", JyutpingInitial::M, JyutpingFinal::AI},
        {"mak", JyutpingInitial::M, JyutpingFinal::AK},
        {"mam", JyutpingInitial::M, JyutpingFinal::AM},
        {"man", JyutpingInitial::M, JyutpingFinal::AN},
        {"mang", JyutpingInitial::M, JyutpingFinal::ANG},
        {"mat", JyutpingInitial::M, JyutpingFinal::AT},
        {"mau", JyutpingInitial::M, JyutpingFinal::AU},
        {"me", JyutpingInitial::M, JyutpingFinal::E},
        {"mei", JyutpingInitial::M, JyutpingFinal::EI},
        {"meng", JyutpingInitial::M, JyutpingFinal::ENG},
        {"mi", JyutpingInitial::M, JyutpingFinal::I},
        {"mik", JyutpingInitial::M, JyutpingFinal::IK},
        {"min", JyutpingInitial::M, JyutpingFinal::IN},
        {"ming", JyutpingInitial::M, JyutpingFinal::ING},
        {"mit", JyutpingInitial::M, JyutpingFinal::IT},
        {"miu", JyutpingInitial::M, JyutpingFinal::IU},
        {"mo", JyutpingInitial::M, JyutpingFinal::O},
        {"moi", JyutpingInitial::M, JyutpingFinal::OI},
        {"mok", JyutpingInitial::M, JyutpingFinal::OK},
        {"mong", JyutpingInitial::M, JyutpingFinal::ONG},
        {"mou", JyutpingInitial::M, JyutpingFinal::OU},
        {"mui", JyutpingInitial::M, JyutpingFinal::UI},
        {"muk", JyutpingInitial::M, JyutpingFinal::UK},
        {"mun", JyutpingInitial::M, JyutpingFinal::UN},
        {"mung", JyutpingInitial::M, JyutpingFinal::UNG},
        {"mut", JyutpingInitial::M, JyutpingFinal::UT},
        {"naa", JyutpingInitial::N, JyutpingFinal::AA},
        {"naai", JyutpingInitial::N, JyutpingFinal::AAI},
        {"naam", JyutpingInitial::N, JyutpingFinal::AAM},
        {"naan", JyutpingInitial::N, JyutpingFinal::AAN},
        {"naang", JyutpingInitial::N, JyutpingFinal::AANG},
        {"naap", JyutpingInitial::N, JyutpingFinal::AAP},
        {"naat", JyutpingInitial::N, JyutpingFinal::AAT},
        {"naau", JyutpingInitial::N, JyutpingFinal::AAU},
        {"nai", JyutpingInitial::N, JyutpingFinal::AI},
        {"nak", JyutpingInitial::N, JyutpingFinal::AK},
        {"nam", JyutpingInitial::N, JyutpingFinal::AM},
        {"nan", JyutpingInitial::N, JyutpingFinal::AN},
        {"nang", JyutpingInitial::N, JyutpingFinal::ANG},
        {"nap", JyutpingInitial::N, JyutpingFinal::AP},
        {"nat", JyutpingInitial::N, JyutpingFinal::AT},
        {"nau", JyutpingInitial::N, JyutpingFinal::AU},
        {"ne", JyutpingInitial::N, JyutpingFinal::E},
        {"nei", JyutpingInitial::N, JyutpingFinal::EI},
        {"neng", JyutpingInitial::N, JyutpingFinal::ENG},
        {"neoi", JyutpingInitial::N, JyutpingFinal::EOI},
        {"neot", JyutpingInitial::N, JyutpingFinal::EOT},
        {"ng", JyutpingInitial::NG, JyutpingFinal::Zero},
        {"ngaa", JyutpingInitial::NG, JyutpingFinal::AA},
        {"ngaai", JyutpingInitial::NG, JyutpingFinal::AAI},
        {"ngaak", JyutpingInitial::NG, JyutpingFinal::AAK},
        {"ngaam", JyutpingInitial::NG, JyutpingFinal::AAM},
        {"ngaan", JyutpingInitial::NG, JyutpingFinal::AAN},
        {"ngaang", JyutpingInitial::NG, JyutpingFinal::AANG},
        {"ngaap", JyutpingInitial::NG, JyutpingFinal::AAP},
        {"ngaat", JyutpingInitial::NG, JyutpingFinal::AAT},
        {"ngaau", JyutpingInitial::NG, JyutpingFinal::AAU},
        {"ngai", JyutpingInitial::NG, JyutpingFinal::AI},
        {"ngak", JyutpingInitial::NG, JyutpingFinal::AK},
        {"ngam", JyutpingInitial::NG, JyutpingFinal::AM},
        {"ngan", JyutpingInitial::NG, JyutpingFinal::AN},
        {"ngang", JyutpingInitial::NG, JyutpingFinal::ANG},
        {"ngap", JyutpingInitial::NG, JyutpingFinal::AP},
        {"ngat", JyutpingInitial::NG, JyutpingFinal::AT},
        {"ngau", JyutpingInitial::NG, JyutpingFinal::AU},
        {"nge", JyutpingInitial::NG, JyutpingFinal::E},
        {"ngi", JyutpingInitial::NG, JyutpingFinal::I},
        {"ngit", JyutpingInitial::NG, JyutpingFinal::IT},
        {"ngo", JyutpingInitial::NG, JyutpingFinal::O},
        {"ngoi", JyutpingInitial::NG, JyutpingFinal::OI},
        {"ngok", JyutpingInitial::NG, JyutpingFinal::OK},
        {"ngon", JyutpingInitial::NG, JyutpingFinal::ON},
        {"ngong", JyutpingInitial::NG, JyutpingFinal::ONG},
        {"ngou", JyutpingInitial::NG, JyutpingFinal::OU},
        {"nguk", JyutpingInitial::NG, JyutpingFinal::UK},
        {"ngung", JyutpingInitial::NG, JyutpingFinal::UNG},
        {"ni", JyutpingInitial::N, JyutpingFinal::I},
        {"nik", JyutpingInitial::N, JyutpingFinal::IK},
        {"nim", JyutpingInitial::N, JyutpingFinal::IM},
        {"nin", JyutpingInitial::N, JyutpingFinal::IN},
        {"ning", JyutpingInitial::N, JyutpingFinal::ING},
        {"nip", JyutpingInitial::N, JyutpingFinal::IP},
        {"nit", JyutpingInitial::N, JyutpingFinal::IT},
        {"niu", JyutpingInitial::N, JyutpingFinal::IU},
        {"no", JyutpingInitial::N, JyutpingFinal::O},
        {"noeng", JyutpingInitial::N, JyutpingFinal::OENG},
        {"noek", JyutpingInitial::N, JyutpingFinal::OEK},
        {"noi", JyutpingInitial::N, JyutpingFinal::OI},
        {"nok", JyutpingInitial::N, JyutpingFinal::OK},
        {"nong", JyutpingInitial::N, JyutpingFinal::ONG},
        {"nou", JyutpingInitial::N, JyutpingFinal::OU},
        {"nuk", JyutpingInitial::N, JyutpingFinal::UK},
        {"nung", JyutpingInitial::N, JyutpingFinal::UNG},
        {"nyun", JyutpingInitial::N, JyutpingFinal::YUN},
        {"o", JyutpingInitial::Zero, JyutpingFinal::O},
        {"oe", JyutpingInitial::Zero, JyutpingFinal::OE},
        {"oi", JyutpingInitial::Zero, JyutpingFinal::OI},
        {"ok", JyutpingInitial::Zero, JyutpingFinal::OK},
        {"om", JyutpingInitial::Zero, JyutpingFinal::OM},
        {"on", JyutpingInitial::Zero, JyutpingFinal::ON},
        {"ong", JyutpingInitial::Zero, JyutpingFinal::ONG},
        {"ou", JyutpingInitial::Zero, JyutpingFinal::OU},
        {"paa", JyutpingInitial::P, JyutpingFinal::AA},
        {"paai", JyutpingInitial::P, JyutpingFinal::AAI},
        {"paak", JyutpingInitial::P, JyutpingFinal::AAK},
        {"paan", JyutpingInitial::P, JyutpingFinal::AAN},
        {"paang", JyutpingInitial::P, JyutpingFinal::AANG},
        {"paat", JyutpingInitial::P, JyutpingFinal::AAT},
        {"paau", JyutpingInitial::P, JyutpingFinal::AAU},
        {"pai", JyutpingInitial::P, JyutpingFinal::AI},
        {"pak", JyutpingInitial::P, JyutpingFinal::AK},
        {"pan", JyutpingInitial::P, JyutpingFinal::AN},
        {"pang", JyutpingInitial::P, JyutpingFinal::ANG},
        {"pat", JyutpingInitial::P, JyutpingFinal::AT},
        {"pau", JyutpingInitial::P, JyutpingFinal::AU},
        {"pe", JyutpingInitial::P, JyutpingFinal::E},
        {"pei", JyutpingInitial::P, JyutpingFinal::EI},
        {"pek", JyutpingInitial::P, JyutpingFinal::EK},
        {"pen", JyutpingInitial::P, JyutpingFinal::EN},
        {"peng", JyutpingInitial::P, JyutpingFinal::ENG},
        {"pet", JyutpingInitial::P, JyutpingFinal::ET},
        {"pi", JyutpingInitial::P, JyutpingFinal::I},
        {"pik", JyutpingInitial::P, JyutpingFinal::IK},
        {"pin", JyutpingInitial::P, JyutpingFinal::IN},
        {"ping", JyutpingInitial::P, JyutpingFinal::ING},
        {"pit", JyutpingInitial::P, JyutpingFinal::IT},
        {"piu", JyutpingInitial::P, JyutpingFinal::IU},
        {"po", JyutpingInitial::P, JyutpingFinal::O},
        {"pok", JyutpingInitial::P, JyutpingFinal::OK},
        {"pong", JyutpingInitial::P, JyutpingFinal::ONG},
        {"pou", JyutpingInitial::P, JyutpingFinal::OU},
        {"pui", JyutpingInitial::P, JyutpingFinal::UI},
        {"puk", JyutpingInitial::P, JyutpingFinal::UK},
        {"pun", JyutpingInitial::P, JyutpingFinal::UN},
        {"pung", JyutpingInitial::P, JyutpingFinal::UNG},
        {"put", JyutpingInitial::P, JyutpingFinal::UT},
        {"saa", JyutpingInitial::S, JyutpingFinal::AA},
        {"saai", JyutpingInitial::S, JyutpingFinal::AAI},
        {"saak", JyutpingInitial::S, JyutpingFinal::AAK},
        {"saam", JyutpingInitial::S, JyutpingFinal::AAM},
        {"saan", JyutpingInitial::S, JyutpingFinal::AAN},
        {"saang", JyutpingInitial::S, JyutpingFinal::AANG},
        {"saap", JyutpingInitial::S, JyutpingFinal::AAP},
        {"saat", JyutpingInitial::S, JyutpingFinal::AAT},
        {"saau", JyutpingInitial::S, JyutpingFinal::AAU},
        {"sai", JyutpingInitial::S, JyutpingFinal::AI},
        {"sak", JyutpingInitial::S, JyutpingFinal::AK},
        {"sam", JyutpingInitial::S, JyutpingFinal::AM},
        {"san", JyutpingInitial::S, JyutpingFinal::AN},
        {"sang", JyutpingInitial::S, JyutpingFinal::ANG},
        {"sap", JyutpingInitial::S, JyutpingFinal::AP},
        {"sat", JyutpingInitial::S, JyutpingFinal::AT},
        {"sau", JyutpingInitial::S, JyutpingFinal::AU},
        {"se", JyutpingInitial::S, JyutpingFinal::E},
        {"sei", JyutpingInitial::S, JyutpingFinal::EI},
        {"sek", JyutpingInitial::S, JyutpingFinal::EK},
        {"seng", JyutpingInitial::S, JyutpingFinal::ENG},
        {"seoi", JyutpingInitial::S, JyutpingFinal::EOI},
        {"seon", JyutpingInitial::S, JyutpingFinal::EON},
        {"seot", JyutpingInitial::S, JyutpingFinal::EOT},
        {"si", JyutpingInitial::S, JyutpingFinal::I},
        {"sik", JyutpingInitial::S, JyutpingFinal::IK},
        {"sim", JyutpingInitial::S, JyutpingFinal::IM},
        {"sin", JyutpingInitial::S, JyutpingFinal::IN},
        {"sing", JyutpingInitial::S, JyutpingFinal::ING},
        {"sip", JyutpingInitial::S, JyutpingFinal::IP},
        {"sit", JyutpingInitial::S, JyutpingFinal::IT},
        {"siu", JyutpingInitial::S, JyutpingFinal::IU},
        {"so", JyutpingInitial::S, JyutpingFinal::O},
        {"soe", JyutpingInitial::S, JyutpingFinal::OE},
        {"soek", JyutpingInitial::S, JyutpingFinal::OEK},
        {"soeng", JyutpingInitial::S, JyutpingFinal::OENG},
        {"soi", JyutpingInitial::S, JyutpingFinal::OI},
        {"sok", JyutpingInitial::S, JyutpingFinal::OK},
        {"song", JyutpingInitial::S, JyutpingFinal::ONG},
        {"sot", JyutpingInitial::S, JyutpingFinal::OT},
        {"sou", JyutpingInitial::S, JyutpingFinal::OU},
        {"suk", JyutpingInitial::S, JyutpingFinal::UK},
        {"sung", JyutpingInitial::S, JyutpingFinal::UNG},
        {"syu", JyutpingInitial::S, JyutpingFinal::YU},
        {"syun", JyutpingInitial::S, JyutpingFinal::YUN},
        {"syut", JyutpingInitial::S, JyutpingFinal::YUT},
        {"taa", JyutpingInitial::T, JyutpingFinal::AA},
        {"taai", JyutpingInitial::T, JyutpingFinal::AAI},
        {"taam", JyutpingInitial::T, JyutpingFinal::AAM},
        {"taan", JyutpingInitial::T, JyutpingFinal::AAN},
        {"taang", JyutpingInitial::T, JyutpingFinal::AANG},
        {"taap", JyutpingInitial::T, JyutpingFinal::AAP},
        {"taat", JyutpingInitial::T, JyutpingFinal::AAT},
        {"taau", JyutpingInitial::T, JyutpingFinal::AAU},
        {"tai", JyutpingInitial::T, JyutpingFinal::AI},
        {"tam", JyutpingInitial::T, JyutpingFinal::AM},
        {"tan", JyutpingInitial::T, JyutpingFinal::AN},
        {"tang", JyutpingInitial::T, JyutpingFinal::ANG},
        {"tap", JyutpingInitial::T, JyutpingFinal::AP},
        {"tat", JyutpingInitial::T, JyutpingFinal::AT},
        {"tau", JyutpingInitial::T, JyutpingFinal::AU},
        {"tek", JyutpingInitial::T, JyutpingFinal::EK},
        {"teng", JyutpingInitial::T, JyutpingFinal::ENG},
        {"teoi", JyutpingInitial::T, JyutpingFinal::EOI},
        {"teon", JyutpingInitial::T, JyutpingFinal::EON},
        {"teot", JyutpingInitial::T, JyutpingFinal::EOT},
        {"ti", JyutpingInitial::T, JyutpingFinal::I},
        {"tik", JyutpingInitial::T, JyutpingFinal::IK},
        {"tim", JyutpingInitial::T, JyutpingFinal::IM},
        {"tin", JyutpingInitial::T, JyutpingFinal::IN},
        {"ting", JyutpingInitial::T, JyutpingFinal::ING},
        {"tip", JyutpingInitial::T, JyutpingFinal::IP},
        {"tit", JyutpingInitial::T, JyutpingFinal::IT},
        {"tiu", JyutpingInitial::T, JyutpingFinal::IU},
        {"to", JyutpingInitial::T, JyutpingFinal::O},
        {"toe", JyutpingInitial::T, JyutpingFinal::OE},
        {"toi", JyutpingInitial::T, JyutpingFinal::OI},
        {"tok", JyutpingInitial::T, JyutpingFinal::OK},
        {"tong", JyutpingInitial::T, JyutpingFinal::ONG},
        {"tou", JyutpingInitial::T, JyutpingFinal::OU},
        {"tuk", JyutpingInitial::T, JyutpingFinal::UK},
        {"tung", JyutpingInitial::T, JyutpingFinal::UNG},
        {"tyun", JyutpingInitial::T, JyutpingFinal::YUN},
        {"tyut", JyutpingInitial::T, JyutpingFinal::YUT},
        {"uk", JyutpingInitial::Zero, JyutpingFinal::UK},
        {"ung", JyutpingInitial::Zero, JyutpingFinal::UNG},
        {"waa", JyutpingInitial::W, JyutpingFinal::AA},
        {"waai", JyutpingInitial::W, JyutpingFinal::AAI},
        {"waak", JyutpingInitial::W, JyutpingFinal::AAK},
        {"waan", JyutpingInitial::W, JyutpingFinal::AAN},
        {"waang", JyutpingInitial::W, JyutpingFinal::AANG},
        {"waat", JyutpingInitial::W, JyutpingFinal::AAT},
        {"wai", JyutpingInitial::W, JyutpingFinal::AI},
        {"wan", JyutpingInitial::W, JyutpingFinal::AN},
        {"wang", JyutpingInitial::W, JyutpingFinal::ANG},
        {"wak", JyutpingInitial::W, JyutpingFinal::AK},
        {"wat", JyutpingInitial::W, JyutpingFinal::AT},
        {"wau", JyutpingInitial::W, JyutpingFinal::AU},
        {"we", JyutpingInitial::W, JyutpingFinal::E},
        {"wen", JyutpingInitial::W, JyutpingFinal::EN},
        {"weng", JyutpingInitial::W, JyutpingFinal::ENG},
        {"wet", JyutpingInitial::W, JyutpingFinal::ET},
        {"wi", JyutpingInitial::W, JyutpingFinal::I},
        {"wik", JyutpingInitial::W, JyutpingFinal::IK},
        {"wing", JyutpingInitial::W, JyutpingFinal::ING},
        {"wo", JyutpingInitial::W, JyutpingFinal::O},
        {"wok", JyutpingInitial::W, JyutpingFinal::OK},
        {"wong", JyutpingInitial::W, JyutpingFinal::ONG},
        {"wu", JyutpingInitial::W, JyutpingFinal::U},
        {"wui", JyutpingInitial::W, JyutpingFinal::UI},
        {"wun", JyutpingInitial::W, JyutpingFinal::UN},
        {"wut", JyutpingInitial::W, JyutpingFinal::UT},
        {"zaa", JyutpingInitial::Z, JyutpingFinal::AA},
        {"zaai", JyutpingInitial::Z, JyutpingFinal::AAI},
        {"zaak", JyutpingInitial::Z, JyutpingFinal::AAK},
        {"zaam", JyutpingInitial::Z, JyutpingFinal::AAM},
        {"zaan", JyutpingInitial::Z, JyutpingFinal::AAN},
        {"zaang", JyutpingInitial::Z, JyutpingFinal::AANG},
        {"zaap", JyutpingInitial::Z, JyutpingFinal::AAP},
        {"zaat", JyutpingInitial::Z, JyutpingFinal::AAT},
        {"zaau", JyutpingInitial::Z, JyutpingFinal::AAU},
        {"zai", JyutpingInitial::Z, JyutpingFinal::AI},
        {"zak", JyutpingInitial::Z, JyutpingFinal::AK},
        {"zam", JyutpingInitial::Z, JyutpingFinal::AM},
        {"zan", JyutpingInitial::Z, JyutpingFinal::AN},
        {"zang", JyutpingInitial::Z, JyutpingFinal::ANG},
        {"zap", JyutpingInitial::Z, JyutpingFinal::AP},
        {"zat", JyutpingInitial::Z, JyutpingFinal::AT},
        {"zau", JyutpingInitial::Z, JyutpingFinal::AU},
        {"ze", JyutpingInitial::Z, JyutpingFinal::E},
        {"zem", JyutpingInitial::Z, JyutpingFinal::EM},
        {"zek", JyutpingInitial::Z, JyutpingFinal::EK},
        {"zeng", JyutpingInitial::Z, JyutpingFinal::ENG},
        {"zeoi", JyutpingInitial::Z, JyutpingFinal::EOI},
        {"zeon", JyutpingInitial::Z, JyutpingFinal::EON},
        {"zeot", JyutpingInitial::Z, JyutpingFinal::EOT},
        {"zi", JyutpingInitial::Z, JyutpingFinal::I},
        {"zik", JyutpingInitial::Z, JyutpingFinal::IK},
        {"zim", JyutpingInitial::Z, JyutpingFinal::IM},
        {"zin", JyutpingInitial::Z, JyutpingFinal::IN},
        {"zing", JyutpingInitial::Z, JyutpingFinal::ING},
        {"zip", JyutpingInitial::Z, JyutpingFinal::IP},
        {"zit", JyutpingInitial::Z, JyutpingFinal::IT},
        {"ziu", JyutpingInitial::Z, JyutpingFinal::IU},
        {"zo", JyutpingInitial::Z, JyutpingFinal::O},
        {"zoe", JyutpingInitial::Z, JyutpingFinal::OE},
        {"zoek", JyutpingInitial::Z, JyutpingFinal::OEK},
        {"zoeng", JyutpingInitial::Z, JyutpingFinal::OENG},
        {"zoi", JyutpingInitial::Z, JyutpingFinal::OI},
        {"zok", JyutpingInitial::Z, JyutpingFinal::OK},
        {"zong", JyutpingInitial::Z, JyutpingFinal::ONG},
        {"zou", JyutpingInitial::Z, JyutpingFinal::OU},
        {"zui", JyutpingInitial::Z, JyutpingFinal::UI},
        {"zuk", JyutpingInitial::Z, JyutpingFinal::UK},
        {"zung", JyutpingInitial::Z, JyutpingFinal::UNG},
        {"zyu", JyutpingInitial::Z, JyutpingFinal::YU},
        {"zyun", JyutpingInitial::Z, JyutpingFinal::YUN},
        {"zyut", JyutpingInitial::Z, JyutpingFinal::YUT},

        // Fuzzy for "a" -> "aa".
        {"a", JyutpingInitial::Zero, JyutpingFinal::AA, true},
        {"ba", JyutpingInitial::B, JyutpingFinal::AA, true},
        {"ca", JyutpingInitial::C, JyutpingFinal::AA, true},
        {"da", JyutpingInitial::D, JyutpingFinal::AA, true},
        {"fa", JyutpingInitial::F, JyutpingFinal::AA, true},
        {"ga", JyutpingInitial::G, JyutpingFinal::AA, true},
        {"gwa", JyutpingInitial::GW, JyutpingFinal::AA, true},
        {"ha", JyutpingInitial::H, JyutpingFinal::AA, true},
        {"ja", JyutpingInitial::J, JyutpingFinal::AA, true},
        {"ka", JyutpingInitial::K, JyutpingFinal::AA, true},
        {"kwa", JyutpingInitial::KW, JyutpingFinal::AA, true},
        {"la", JyutpingInitial::L, JyutpingFinal::AA, true},
        {"ma", JyutpingInitial::M, JyutpingFinal::AA, true},
        {"na", JyutpingInitial::N, JyutpingFinal::AA, true},
        {"nga", JyutpingInitial::NG, JyutpingFinal::AA, true},
        {"pa", JyutpingInitial::P, JyutpingFinal::AA, true},
        {"sa", JyutpingInitial::S, JyutpingFinal::AA, true},
        {"ta", JyutpingInitial::T, JyutpingFinal::AA, true},
        {"wa", JyutpingInitial::W, JyutpingFinal::AA, true},
        {"za", JyutpingInitial::Z, JyutpingFinal::AA, true},
    };
    return jyutpingMap;
}

} // namespace jyutping
} // namespace libime
