/****************************************************************************/
/* ORB manager v. 2.0 - CORBA management with Log forwarders                */
/*                                                                          */
/*  Author(s):                                                              */
/*    - Gaël Le Mahec (gael.le.mahec@ens-lyon.fr)                           */
/*                                                                          */
/*  This file is part of DIET .                                             */
/*                                                                          */
/*  Copyright (C) 2000-2003 ENS Lyon, LIFC, INSA, INRIA and SysFera (2000)  */
/*                                                                          */
/*  - Frederic.Desprez@ens-lyon.fr (Project Manager)                        */
/*  - Eddy.Caron@ens-lyon.fr (Technical Manager)                            */
/*  - Tech@sysfera.com (Maintainer and Technical Support)                   */
/*                                                                          */
/*  This software is a computer program whose purpose is to provide an      */
/*  distributed logging services.                                           */
/*                                                                          */
/*                                                                          */
/*  This software is governed by the CeCILL license under French law and    */
/*  abiding by the rules of distribution of free software.  You can  use,   */
/*  modify and/ or redistribute the software under the terms of the CeCILL  */
/*  license as circulated by CEA, CNRS and INRIA at the following URL       */
/*  "http://www.cecill.info".                                               */
/*                                                                          */
/*  As a counterpart to the access to the source code and  rights to copy,  */
/*  modify and redistribute granted by the license, users are provided      */
/*  only with a limited warranty  and the software's author,  the holder    */
/*  of the economic rights,  and the successive licensors  have only        */
/*  limited liability.                                                      */
/*                                                                          */
/*  In this respect, the user's attention is drawn to the risks             */
/*  associated with loading,  using,  modifying and/or developing or        */
/*  reproducing the software by the user in light of its specific status    */
/*  of free software, that may mean  that it is complicated to              */
/*  manipulate, and  that  also therefore means  that it is reserved for    */
/*  developers and experienced professionals having in-depth computer       */
/*  knowledge. Users are therefore encouraged to load and test the          */
/*  software's suitability as regards their requirements in conditions      */
/*  enabling the security of their systems and/or data to be ensured and,   */
/*  more generally, to use and operate it in the same conditions as         */
/*  regards security.                                                       */
/*                                                                          */
/*  The fact that you are presently reading this means that you have had    */
/*  knowledge of the CeCILL license and that you accept its terms.          */
/*                                                                          */
/****************************************************************************/
/* $Id$
 * $Log$
 * Revision 1.9  2011/05/13 12:39:36  bdepardo
 * Fixed a bug with multiple initialization
 *
 * Revision 1.8  2011/05/13 08:17:51  bdepardo
 * Update ORB manager with changes made in DIET ORB manager.
 *
 * Revision 1.7  2011/05/13 06:42:44  bdepardo
 * Const methods whenever possible
 *
 * Revision 1.6  2011/04/28 07:45:30  bdepardo
 * Change POA name so that DIET and LogService do not use the same one.
 *
 * Revision 1.5  2011/04/22 14:08:57  kcoulomb
 * Fix bug if POA name already exist, use it instead of throwing an exception
 *
 * Revision 1.4  2011/04/22 11:44:21  bdepardo
 * Use a signal handler to handle background option.
 * This handler catches SIGINT and SIGTERM.
 *
 * Revision 1.3  2011/02/04 15:35:00  bdepardo
 * Code indentation.
 * Removed unused variables.
 *
 * Revision 1.2  2010/12/17 15:18:20  kcoulomb
 * update log
 *
 * Revision 1.1  2010/12/03 12:40:25  kcoulomb
 * MAJ log to use forwarders
 *
 * Revision 1.1  2010/11/10 02:27:43  kcoulomb
 * Update the log to use the forwarder.
 * Programm run without launching forwarders but fails with forwarder.
 *
 * Revision 1.29  2010/07/27 16:16:48  glemahec
 * Forwarders robustness
 *
 * Revision 1.28  2010/07/27 10:24:32  glemahec
 * Improve robustness & general performance
 *
 * Revision 1.27  2010/07/14 23:45:30  bdepardo
 * Header corrections
 *
 *
 ****************************************************************************/

#include <string>
#include <list>
#include <map>
#include <sstream>
#include <stdexcept>
#include <iostream>
#include <csignal>

#include <omniORB4/CORBA.h>
#include <omniORB4/omniURI.h>

#include "LogORBMgr.hh"
#include "CorbaLogForwarder.hh"

using namespace std;

LogORBMgr* LogORBMgr::theMgr = NULL;

#ifndef __cygwin__
omni_mutex LogORBMgr::waitLock;
#else
sem_t LogORBMgr::waitLock;
#endif


/* Manager initialization. */
void LogORBMgr::init(CORBA::ORB_ptr ORB) {
  CORBA::Object_var object;
  PortableServer::POA_var initPOA;
  PortableServer::POAManager_var manager;
  CORBA::PolicyList policies;
  CORBA::Any policy;
  CosNaming::NamingContext_var rootContext, context;
  CosNaming::Name cosName;
  
  if (CORBA::is_nil(ORB)) {
    throw runtime_error("ORB not initialized");
  }
  object = ORB->resolve_initial_references("RootPOA");
  initPOA = PortableServer::POA::_narrow(object);
  manager = initPOA->the_POAManager();
  policies.length(1);
  policy <<= BiDirPolicy::BOTH;
  policies[0] = ORB->create_policy(BiDirPolicy::BIDIRECTIONAL_POLICY_TYPE,
                                   policy);

  // If poa already exist, use the same
  try {
    POA = initPOA->create_POA("bidirLogService", manager, policies);
  } catch (PortableServer::POA::AdapterAlreadyExists &e) {
    POA = initPOA->find_POA("bidirLogService", false);    
  }
  
  manager->activate();
}

LogORBMgr::LogORBMgr(int argc, char* argv[]) {
  const char* opts[][2]= {{0,0}};
  
  ORB = CORBA::ORB_init(argc, argv, "omniORB4", opts);
  init(ORB);
  down = false;
}

LogORBMgr::LogORBMgr(CORBA::ORB_ptr ORB) {
  this->ORB = ORB;
  init(ORB);
  down = false;
}

LogORBMgr::LogORBMgr(CORBA::ORB_ptr ORB, PortableServer::POA_var POA) {
  this->ORB=ORB;
  this->POA=POA;
  down = false;
}

LogORBMgr::~LogORBMgr() {
  shutdown(true);
  ORB->destroy();
  theMgr = NULL;
}

void LogORBMgr::bind(const string& ctxt, const string& name,
                     CORBA::Object_ptr object,
                     const bool rebind) const {
  CORBA::Object_var obj;
  CosNaming::NamingContext_var rootContext, context;
  CosNaming::Name cosName;
	
  obj = ORB->resolve_initial_references("NameService");
  if (CORBA::is_nil(obj)) {
    throw runtime_error("Error resolving initial references");
  }  
  rootContext = CosNaming::NamingContext::_narrow(obj);
  
  if (CORBA::is_nil(rootContext)) {
    throw runtime_error("Error initializing root context");
  }

  cosName.length(1);
  cosName[0].id = ctxt.c_str();
  cosName[0].kind = "";
  try {
    context = rootContext->bind_new_context(cosName);
  } catch (CosNaming::NamingContext::AlreadyBound& err) {
    obj = rootContext->resolve(cosName);
    context = CosNaming::NamingContext::_narrow(obj);
    if (CORBA::is_nil(context)) {
      throw runtime_error(string("Error creating context ")+ctxt);
    }
  }
  cosName[0].id = name.c_str();
  cosName[0].kind = "";
  try {
    context->bind(cosName, object);
  } catch (CosNaming::NamingContext::AlreadyBound& err) {
    if (rebind) {
      context->rebind(cosName, object);
    } else{
      throw runtime_error("Already bound!");
    }
  }
}

void LogORBMgr::bind(const string& ctxt, const string& name,
                     const string& IOR, const bool rebind) const {
  CORBA::Object_ptr object = ORB->string_to_object(IOR.c_str());
	
  bind(ctxt, name, object, rebind);
}

void LogORBMgr::rebind(const string& ctxt, const string& name,
                       CORBA::Object_ptr object) const {
  bind(ctxt, name, object, true);
}

void LogORBMgr::rebind(const string& ctxt, const string& name,
                       const string& IOR) const {
  CORBA::Object_ptr object = ORB->string_to_object(IOR.c_str());
  
  rebind(ctxt, name, object);
}

void LogORBMgr::unbind(const string& ctxt, const string& name) const {
  CORBA::Object_var obj;
  CosNaming::NamingContext_var rootContext, context;
  CosNaming::Name cosName;
  std::list<pair<string,string> >::iterator it;
  
  obj = ORB->resolve_initial_references("NameService");
  rootContext = CosNaming::NamingContext::_narrow(obj);
  
  cosName.length(1);
  cosName[0].id = ctxt.c_str();
  cosName[0].kind = "";
  
  obj = rootContext->resolve(cosName);
  context = CosNaming::NamingContext::_narrow(obj);
  if (CORBA::is_nil(context))
    throw runtime_error(string("Error retrieving context ")+ctxt);
  cosName[0].id = name.c_str();
  cosName[0].kind = "";
  try {
    removeObjectFromCache(ctxt, name);
    context->unbind(cosName);
  } catch (CosNaming::NamingContext::NotFound& err) {
    throw runtime_error("Object "+name+" not found in " + ctxt +" context");
  }
}

void LogORBMgr::fwdsBind(const string& ctxt, const string& name,
                         const string& ior, const string& fwName) const {
  std::list<string> forwarders = LogORBMgr::list(LOGFWRDCTXT);
  std::list<string>::const_iterator it;
	
  for (it=forwarders.begin(); it!=forwarders.end(); ++it) {
    if (fwName==*it) continue;
    CorbaLogForwarder_var fwd = resolve<CorbaLogForwarder, CorbaLogForwarder_var>(LOGFWRDCTXT, *it);
    string objName = ctxt+"/"+name;
    try {
      fwd->bind(objName.c_str(), ior.c_str());
    } catch (const CORBA::TRANSIENT& err) {
      continue;
    } catch (LogBadNameException& err) {
    }
  }
}

void LogORBMgr::fwdsUnbind(const string& ctxt, const string& name,
                           const string& fwName) const {
  std::list<string> forwarders = LogORBMgr::list(LOGFWRDCTXT);
  std::list<string>::const_iterator it;
	
  for (it=forwarders.begin(); it!=forwarders.end(); ++it) {
    if (fwName==*it) continue;
    CorbaLogForwarder_var fwd = resolve<CorbaLogForwarder, CorbaLogForwarder_var>(LOGFWRDCTXT, *it);
    string objName = ctxt+"/"+name;
    try {
      fwd->unbind(objName.c_str());
    } catch (const CORBA::TRANSIENT& err) {
      cerr << "Unable to contact LOG forwarder " << *it << endl;
      continue;
    }
  }
}

CORBA::Object_ptr LogORBMgr::resolveObject(const string& IOR) const {
  return ORB->string_to_object(IOR.c_str());
}

CORBA::Object_ptr LogORBMgr::resolveObject(const string& context, const string& name,
                                           const string& fwdName) const {
  string ctxt = context;
  string ctxt2 = context;
  if (ctxt2 == LOGCOMPCONFCTXT) {
    ctxt = LOGCOMPCTXT;
  } else if (ctxt2==LOGTOOLMSGCTXT) {
    ctxt = LOGTOOLCTXT;
  }
  cacheMutex.lock();

  /* Use object cache. */
  if (cache.find(ctxt+"/"+name)!=cache.end()) {
    CORBA::Object_ptr ptr = cache[ctxt+"/"+name];
    cacheMutex.unlock();
    try {
      cout << "Check if the object is still present" << endl;
      if (ptr->_non_existent()) {
	cout << "Remove non existing object from cache (" << ctxt
	     << "/" << name << ")" << endl;
	removeObjectFromCache(name);
      } else {
	cout << "Use object from cache (" << ctxt
	     << "/" << name << ")" << endl;
	return CORBA::Object::_duplicate(ptr);
      }
    } catch (const CORBA::OBJECT_NOT_EXIST& err) {
      cout << "Remove non existing object from cache (" << ctxt
	   << "/" << name << ")" << endl;
      removeObjectFromCache(name);
    } catch (const CORBA::TRANSIENT& err) {
      cout << "Remove unreachable object from cache (" << ctxt
	   << "/" << name << ")" << endl;
      removeObjectFromCache(name);
    } catch (const CORBA::COMM_FAILURE& err) {
      cout << "Remove unreachable object from cache (" << ctxt
	   << "/" << name << ")" << endl;
      removeObjectFromCache(name);
    } catch (...) {
      cout << "Remove unreachable object from cache (" << ctxt
           << "/" << name << ")" << endl;
      removeObjectFromCache(name);
    }

  }
  cacheMutex.unlock();
  CORBA::Object_ptr object = ORB->resolve_initial_references("NameService");
  CosNaming::NamingContext_var rootContext =
    CosNaming::NamingContext::_narrow(object);
  CosNaming::Name cosName;
  
  cosName.length(2);
  cosName[0].id   = ctxt.c_str();
  cosName[0].kind = "";
  cosName[1].id   = name.c_str();
  cosName[1].kind = "";
  
  try {
    object = rootContext->resolve(cosName);
    /* If the object is not a forwarder object, then
     * search if we need to use a forwarder to reach it.
     */
    if (ctxt!=LOGFWRDCTXT && fwdName!="no-Forwarder") { // MODIF
      string objIOR = getIOR(object);
      IOP::IOR ior;
      makeIOR(objIOR, ior);
      std::list<string> forwarders = LogORBMgr::list(LOGFWRDCTXT);
      std::list<string>::const_iterator it;
      for (it=forwarders.begin(); it!=forwarders.end(); ++it) {
	// This is the same forwarder...
	if (*it==fwdName) continue;
	CorbaLogForwarder_var fwd = LogORBMgr::getMgr()->resolve<CorbaLogForwarder, CorbaLogForwarder_var>(LOGFWRDCTXT, *it);
	string objName = ctxt+"/"+name;
	string ior = getIOR(object);
	string objHost = getHost(ior);
	try {
	  if (fwd->manage(objHost.c_str())) {
	    cout << "Object " << ctxt << "/" << name << "."
		 << "is reachable through forwarder " << *it << endl;
	    if (ctxt2==LOGCOMPCTXT) {
	      object = fwd->getLogCentralComponent(name.c_str());
	      string objIOR = getIOR(object);
	      break;
	    }
	    if (ctxt2==LOGTOOLCTXT) {
	      object = fwd->getLogCentralTool(name.c_str());
	      break;
	    }
	    if (ctxt2==LOGTOOLMSGCTXT) {
	      object = fwd->getToolMsgReceiver(name.c_str());
	      break;
	    }
	    if (ctxt2==LOGCOMPCONFCTXT) {
	      object = fwd->getCompoConf(name.c_str());
	      break;
	    }
	  } else {
	    cout << "Direct access to object " << ctxt << "/" << name << endl;
	  }
	} catch (const CORBA::TRANSIENT& err) {
	  cerr << "Unable to contact LOG forwarder \"" << *it << "\"" << endl;
	  continue;
	}
      }
    }
  } catch (CosNaming::NamingContext::NotFound& err) {
    cerr << "Error resolving " << ctxt << "/" << name << endl;
    throw runtime_error("Error resolving "+ctxt+"/"+name);
  }
  cacheMutex.lock();
  cache[ctxt+"/"+name] = CORBA::Object::_duplicate(object);
  cacheMutex.unlock();

  return CORBA::Object::_duplicate(object);
}

std::list<string> LogORBMgr::list(CosNaming::NamingContext_var& ctxt) const {
  std::list<string> result;
  CosNaming::BindingList_var ctxtList;
  CosNaming::BindingIterator_var it;
  CosNaming::Binding_var bv;

  ctxt->list(256, ctxtList, it);
  for (unsigned int i=0; i<ctxtList->length(); ++i)
    if (ctxtList[i].binding_type==CosNaming::nobject)
      for (unsigned int j=0; j<ctxtList[i].binding_name.length(); ++j) {
        result.push_back(string(ctxtList[i].binding_name[j].id));
      }
  if (CORBA::is_nil(it)) return result;
  while (it->next_one(bv)) {
    if (bv->binding_type==CosNaming::nobject)
      for (unsigned int j=0; j<bv->binding_name.length(); ++j) {
        result.push_back(string(bv->binding_name[j].id));
      }
  }
  return result;
}

std::list<string> LogORBMgr::list(const std::string& ctxtName) const {
  std::list<string> result;
	
  CORBA::Object_ptr object = ORB->resolve_initial_references("NameService");
  CosNaming::NamingContext_var rootContext =
    CosNaming::NamingContext::_narrow(object);
  CosNaming::BindingList_var bindingList;
  CosNaming::BindingIterator_var it;
	
  rootContext->list(256, bindingList, it);

  for (unsigned int i=0; i<bindingList->length(); ++i) {
    if (bindingList[i].binding_type==CosNaming::ncontext)
      if (string(bindingList[i].binding_name[0].id)==ctxtName) {
        std::list<string> tmpRes;
        CORBA::Object_ptr ctxtObj = rootContext->resolve(bindingList[i].binding_name);
        CosNaming::NamingContext_var ctxt =
          CosNaming::NamingContext::_narrow(ctxtObj);
			
        tmpRes = list(ctxt);
        result.insert(result.end(), tmpRes.begin(), tmpRes.end());
      }
  }
  if (CORBA::is_nil(it)) return result;
  CosNaming::Binding_var bv;
  while (it->next_one(bv))
    if (bv->binding_type==CosNaming::ncontext)
      if (string(bv->binding_name[0].id)==ctxtName) {
        std::list<string> tmpRes;
        CORBA::Object_ptr ctxtObj = rootContext->resolve(bv->binding_name);
        CosNaming::NamingContext_var ctxt =
          CosNaming::NamingContext::_narrow(ctxtObj);
				
        tmpRes = list(ctxt);
        result.insert(result.end(), tmpRes.begin(), tmpRes.end());
      }
  return result;
}


string LogORBMgr::getIOR(CORBA::Object_ptr object) const {
  return ORB->object_to_string(object);
}

string LogORBMgr::getIOR(const string& ctxt, const string& name) const {
  return ORB->object_to_string(resolveObject(ctxt, name));
}

void LogORBMgr::activate(PortableServer::ServantBase* object) const {
  POA->activate_object(object);
  object->_remove_ref();
}

void LogORBMgr::deactivate(PortableServer::ServantBase* object) const {
  PortableServer::ObjectId* id = POA->servant_to_id(object);
  POA->deactivate_object(*id);
}

void
LogORBMgr::shutdown(bool waitForCompletion) {
  if (!down) {
    ORB->shutdown(waitForCompletion);
    down = true;
  }
}


void
LogORBMgr::sigIntHandler(int sig) {
  /* Prevent from raising a new SIGINT handler */
  signal(SIGINT, SIG_IGN);
  signal(SIGTERM, SIG_IGN);
#ifndef __cygwin__  
  LogORBMgr::getMgr()->waitLock.unlock();
#else
  sem_post(&(LogORBMgr::getMgr()->waitLock));
#endif
  signal(SIGINT, SIG_DFL);
  signal(SIGTERM, SIG_DFL);
}


void
LogORBMgr::wait() const {
  /* FIXME: this is pretty ugly,
   * but currently I do not see how to do so properly
   */
  signal(SIGINT, LogORBMgr::sigIntHandler);
  signal(SIGTERM, LogORBMgr::sigIntHandler);
  try {
    cout << "Press CTRL+C to exit" << std::endl;
#ifdef __cygwin__
    sem_init(&waitLock,0,1);
    sem_wait(&waitLock);
    sem_wait(&waitLock);
    sem_post(&waitLock);
#else
    waitLock.lock();
    waitLock.lock();
    waitLock.unlock();
#endif
  } catch (...) {
  }
}


LogORBMgr* LogORBMgr::getMgr() {
  if (theMgr==NULL) {
    throw runtime_error("ORB manager not initialized!");
  } else {
    return theMgr;
  }
}

void LogORBMgr::init(int argc, char* argv[]) {
  if (!theMgr) {
    //delete theMgr;
    theMgr = new LogORBMgr(argc, argv);
  }
}


/* Translate the string passed as first argument in bytes and
 * record them into the buffer.
 */
void LogORBMgr::hexStringToBuffer(const char* ptr, const size_t size,
                                  cdrMemoryStream& buffer) {
  stringstream ss;
  int value;
  CORBA::Octet c;
	
  for (unsigned int i=0; i<size; i+=2) {
    ss << ptr[i] << ptr[i+1];
    ss >> hex >> value;
    c = value;
    buffer.marshalOctet(c);
    ss.flush();
    ss.clear();
  }
}

/* Make an IOP::IOR object using a stringified IOR. */
void LogORBMgr::makeIOR(const string& strIOR, IOP::IOR& ior) {
  /* An IOR must start with "IOR:" or "ior:" */
  if (strIOR.find("IOR:")!=0 && strIOR.find("ior:")!=0)
    throw runtime_error("Bad IOR: "+strIOR);

  const char* tab = strIOR.c_str();
  size_t size = (strIOR.length()-4);
  cdrMemoryStream buffer(size, false);
  CORBA::Boolean byteOrder;
	
  /* Convert the hex bytes string into buffer. */
  hexStringToBuffer(tab+4, size, buffer);
  buffer.rewindInputPtr();
  /* Get the endianness and init the buffer flag. */
  byteOrder = buffer.unmarshalBoolean();
  buffer.setByteSwapFlag(byteOrder);
	
  /* Get the object type id. */
  ior.type_id = IOP::IOR::unmarshaltype_id(buffer);
  /* Get the IOR profiles. */
  ior.profiles <<= buffer;
}

/* Convert IOP::IOR to a stringified IOR. */
void LogORBMgr::makeString(const IOP::IOR& ior, string& strIOR) {
  strIOR = "IOR:";
  cdrMemoryStream buffer(0, true);
  stringstream ss;
  unsigned char* ptr;
		
  buffer.marshalBoolean(omni::myByteOrder);
  buffer.marshalRawString(ior.type_id);
  ior.profiles >>= buffer;
	
  buffer.rewindInputPtr();
  ptr = static_cast<unsigned char*>(buffer.bufPtr());
	
  for (unsigned long i=0; i < buffer.bufSize(); ++ptr, ++i)
    {
      string str;
      unsigned char c = *ptr;
      if (c<16) ss << '0';
      ss << (unsigned short) c;
      ss >> hex  >> str;
      ss.flush();
      ss.clear();
      strIOR+=str;
    }
}

/* Get the hostname of the first profile in this IOR. */
string LogORBMgr::getHost(IOP::IOR& ior) {
  IIOP::ProfileBody body;
	
  if (ior.profiles.length()==0)
    return "nohost";
	
  IIOP::unmarshalProfile(ior.profiles[0], body);
  return string(body.address.host);
}

/* Get the hostname of the first profile in IOR passed
 * as a string.
 */
string LogORBMgr::getHost(const string& strIOR) {
  IOP::IOR ior;
  makeIOR(strIOR, ior);
  return getHost(ior);
}

/* Get the port of the first profile in this IOR. */
unsigned int LogORBMgr::getPort(IOP::IOR& ior) {
  IIOP::ProfileBody body;
	
  if (ior.profiles.length()==0)
    return 0;
	
  IIOP::unmarshalProfile(ior.profiles[0], body);
  return body.address.port;
}

/* Get the port of the first profile in IOR passed
 * as a string.
 */
unsigned int LogORBMgr::getPort(const string& strIOR) {
  IOP::IOR ior;
  makeIOR(strIOR, ior);
  return getPort(ior);
}

/* Get the type id of the IOR. */
string LogORBMgr::getTypeID(IOP::IOR& ior) {
  return string(ior.type_id);
}

/* Get the type id of the IOR passed as a string. */
std::string LogORBMgr::getTypeID(const string& strIOR) {
  IOP::IOR ior;
  makeIOR(strIOR, ior);
  return getTypeID(ior);
}

std::string LogORBMgr::convertIOR(IOP::IOR& ior, const std::string& host,
                                  const unsigned int port)
{
  IIOP::ProfileBody body;
  IOP::TaggedProfile profile;
  CORBA::ULong max_data;
  CORBA::ULong nb_data;
  CORBA::Octet* buffer;
  std::string result;

  if (ior.profiles.length()==0)
    throw runtime_error("Invalid IOR");
	
  for (unsigned int i=0; i<ior.profiles.length(); ++i)
    if (ior.profiles[i].tag==IOP::TAG_INTERNET_IOP) {
      IIOP::unmarshalProfile(ior.profiles[i], body);
			
      body.address.host = host.c_str();
      body.address.port = port;
			
      IIOP::encodeProfile(body, profile);
			
      max_data = profile.profile_data.maximum();
      nb_data = profile.profile_data.length();
      buffer = profile.profile_data.get_buffer(true);
      ior.profiles[i].profile_data.replace(max_data, nb_data, buffer, true);
    }
  makeString(ior, result);
  return result;
}

std::string LogORBMgr::convertIOR(const std::string& strIOR, const std::string& host,
                                  const unsigned int port)
{
  IOP::IOR ior;
  makeIOR(strIOR, ior);
  return convertIOR(ior, host, port);
}

/* Object cache management functions. */
void LogORBMgr::resetCache() const {
  cacheMutex.lock();
  cache.clear();
  cacheMutex.unlock();
}

void LogORBMgr::removeObjectFromCache(const string& name) const {
  map<string, CORBA::Object_ptr>::iterator it;
  cacheMutex.lock();
  if ((it=cache.find(name))!=cache.end())	cache.erase(it);
  cacheMutex.unlock();
}

void LogORBMgr::removeObjectFromCache(const string& ctxt,
                                      const string& name) const {
  removeObjectFromCache(ctxt+"/"+name);
}

void LogORBMgr::cleanCache() const {
  map<string, CORBA::Object_ptr>::iterator it;
  std::list<string> toRemove;
  std::list<string>::const_iterator jt;

  cacheMutex.lock();
  for (it = cache.begin(); it != cache.end(); ++it) {
    try {
      if (it->second->_non_existent()) {
        toRemove.push_back(it->first);
      }
    } catch (const CORBA::OBJECT_NOT_EXIST& err) {
      toRemove.push_back(it->first);
    } catch (...) {
      toRemove.push_back(it->first);
    }
  }
  cacheMutex.unlock();
  for (jt = toRemove.begin(); jt != toRemove.end(); ++jt) {
    removeObjectFromCache(*jt);
  }
}
