/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
 *  \file evaluationmanager.c
 *
 *  \brief Implementation of the evaluation manager interface.
 *
 *  Besides the implementation of the interface of the evaluation
 *  manager some additional functions are implemented here. Please
 *  note that these are \b not part of the interface and hence should
 *  not be used. Look in evaluationmanager.h for the functions that
 *  can be called by external sources.
 *
 *
 *  \author  G.M. Venekamp  (venekamp@nikhef.nl)
 *  \version $Revision: 15826 $
 *  \date    $Date: 2012-01-04 15:32:25 +0100 (Wed, 04 Jan 2012) $
 *
 */

#include <stdlib.h>
#include <string.h>

#include "_lcmaps_pluginmanager.h"
#include "lcmaps_log.h"
#include "evaluationmanager.h"

/*!
 *  When the lcmaps_getPluginNameAndArgs() function has been called, the
 *  global_plugin_list variable get initialized with the first element
 *  of the list. This variable is later used to free the resources held
 *  by the list. In addition, multiple calls to lcmaps_getPluginNameAndArgs()
 *  result in returning the value of this pointer.
 *
 */
static lcmaps_db_entry_t* global_plugin_list = NULL;

int free_lcmaps_db_entry(void);


/*!
 *  Start the evaluation manager.
 *
 *  \param  name Name of the configure script.
 *  \param  argc number of policies
 *  \param  argv list of policies
 *  \retval 0 when the call is successful,
 *  \retval 1 otherwise.
 *
 */
int lcmaps_startEvaluationManager(const char* name, int argc, char*argv[])
{
  if (lcmaps_pdl_init(name) < 0) {
    lcmaps_stopEvaluationManager();
    return -1;
  }

  lcmaps_SetSetOfRules(argc, argv);

  /*  Let's see if the config file makes sence.  */
  yyparse();

  /*
   *  Check if there were any errors. Ifo so, there is not much point
   *  on continuing.
   */
  if (yyparse_errors()) {
    lcmaps_stopEvaluationManager();
    return -1;
  }

  /*
   *  It might be that we have allocated policy rulez without any rules
   *  attachted to it. Before we can continue we must clean up the tree
   *  a free the allocated policy rules with an empty rule set.
   */
  lcmaps_cleanup_policies();

  /*
   *  Check to see if there are any recustions in the config file.
   *  If so, continuing might/will cause an infinite loop.
   */
  if (lcmaps_check_policies_for_recursion())
    return -1;


  /*
   *  We need to replace the variables found in the policy rules by
   *  their respective values.
   */
  lcmaps_reduce_policies();

  return 0;
}


/*!
 *  Get a list of plugins and their arguments based on the
 *  configuration file. The memory that is allocted is freed during
 *  the lcmaps_stopEvaluationManager() call.
 *
 *  \param  plugins Pointer to be intialized with the first entry
 *                  of the plugin list.
 *  \retval 0 when the call is successful,
 *  \retval 1 otherwise.
 *
 */
int lcmaps_getPluginNameAndArgs(lcmaps_db_entry_t** plugins)
{
  const plugin_t* p_list, *tmp_p_list;
  lcmaps_db_entry_t* p=0;
  BOOL string_too_long;

  string_too_long = FALSE;

  /*  Check if the plugins have been requested before.  */
  if (global_plugin_list) {
    *plugins = global_plugin_list;
    return 0;
  }

  /*  Set to a safe default value.  */
  *plugins = 0;

  p_list = lcmaps_get_plugins();

  while (p_list) {
    if (!*plugins) {
      *plugins = (lcmaps_db_entry_t*)malloc(sizeof(lcmaps_db_entry_t));
      p = *plugins;
    } else {
      p->next = (lcmaps_db_entry_t*)malloc(sizeof(lcmaps_db_entry_t));
      p = p->next;
    }

    /*  Copy the name and arguments while respecting max. lengths.  */
    strncpy(p->pluginname, p_list->name, LCMAPS_MAXPATHLEN);
    if (strlen(p_list->name)>=LCMAPS_MAXPATHLEN) {
      lcmaps_log(LOG_ERR, "String too long to copy. Max length = %d\n", LCMAPS_MAXPATHLEN);
      string_too_long = TRUE;
    }

    if (p_list->args) {
      strncpy(p->pluginargs, p_list->args, LCMAPS_MAXARGSTRING);
      if (strlen(p_list->args)>LCMAPS_MAXARGSTRING) {
        lcmaps_log(LOG_ERR, "String too long to copy. Max length = %d\n", LCMAPS_MAXARGSTRING);
        string_too_long = TRUE;
      }
    }
    else
      *p->pluginargs = '\0';
    p->next = 0;

    tmp_p_list = p_list->next;

    /* if (p_list->name)  free(p_list->name); */
    /* if (p_list->args)  free(p_list->args); */
    /* free((plugin_t*)p_list); */

    p_list = tmp_p_list;

    /*  Output some debug info.  */
    /* lcmaps_log_debug(1, "%s\n", p->pluginname); */
    /* lcmaps_log_debug(1, "%s\n", p->pluginargs); */
  }

  global_plugin_list = *plugins;

  return string_too_long ? -1 : 0;
}


/*!
 *  Run the evaluation manager. The evaluation manager has to be
 *  initialized by calling statrEvaluation Manager first.
 *
 *  \retval 0 when the call is successful,
 *  \retval 1 otherwise.
 *
 */
int lcmaps_runEvaluationManager(int argc, char *argv[])
{
    char* plugin_name      = NULL;
    plugin_status_t result;
    policy_t * policy      = NULL;
    policy_t * prev_policy = NULL;
    int i = 0;
    int found = 0;
    int rc = 0;

    result = EVALUATION_START;
    /* Note:  returns the short pluginname,
     * lcmaps_runPlugin will figure out the long one */
    while ((plugin_name = lcmaps_pdl_next_plugin(result)))
    {
        /* get active policy */
        policy = lcmaps_get_current_policy();

        /*
         *  Check for explicitly demanded policies to be executed
         *  If it is not the case, the old behaviour is unaffected.
         */
        if (argc > 0) 
        {
            /* Is the current policy selected to be executed? */
            for (i = 0, found = 0; policy && (i<argc); ++i) 
            {
                if (strcmp(policy->name, argv[i])==0) 
                {
                    found = 1;
                    break;
                }
            }

            /* Don't execute current policy as it is NOT selected to be executed */
            if (found != 1)
            {
                /* Clean plug_name (malloc'ed output from lcmaps_pdl_next_plugin) */
                if (plugin_name) { free(plugin_name); plugin_name = NULL; }

                /*
                 *  Set the result to EVALUATION_FAILURE in order to fall through
                 *  the next policy rule. This might take a few steps though!
                 */
                result = EVALUATION_FAILURE;
                continue;
            }
        }

        /* If the execution policy has switched to a new one:
         * First we need to clean the credential data structs to avoid
         * intermediate results of a failed policy execution to contaminate the next */
        if (policy != prev_policy)
        {
	    lcmaps_log(LOG_INFO, "Starting policy: %s\n",policy->name);
            lcmaps_log_debug(4, "evaluationmanager: Resetting credential data.\n");

            /* This must not be reset on the final policy */
            rc = lcmaps_resetCredentialData();
            if (rc)
            {
                lcmaps_log_debug (5, "Resetting credential data failed: rc = %d", rc);
            }

            /* Update the prev_policy to the current to detect the change */
            prev_policy = policy;
        }

        /* Execute selected plug-in */
        result = (lcmaps_runPlugin(plugin_name) ? EVALUATION_FAILURE : EVALUATION_SUCCESS);

        lcmaps_log_debug(1, "Execution of plugin \"%s\". Result: %s.\n", plugin_name, (result==EVALUATION_SUCCESS) ? "Success" : "Failed");

        if (plugin_name) { free(plugin_name); plugin_name = NULL; }
    }

    if (result==EVALUATION_START)
        lcmaps_log(LOG_ERR, "Initialization of the EvaluationManager either failed or was not done.\n");

    return result==EVALUATION_SUCCESS ? 0 : 1;
}


/*!
 *  Stop the evaluation manager after is has run successfully.
 *  Strictly speaking, the evalauation manager needs no stopping.
 *  This call is a good point to clean up the resources used by
 *  the evaluation manager.
 *
 *  \retval 0 when the call is successful,
 *  \retval 1 otherwise.
 *
 */
int lcmaps_stopEvaluationManager(void)
{
  lcmaps_log_debug(5, "lcmaps_stopEvaluationManager: cleaning up!\n");

  lcmaps_free_resources();

  free_lcmaps_db_entry();

  return 0;
}


/*!
 *  During the getPluginsAndArgs() call, a list structure is created.
 *  This structure is never cleaned automatically, nor can it be. When
 *  it is necessay and safe to free the resources, call this function
 *
 *  \retval 0 when the call is successful,
 *  \retval 1 otherwise.
 *
 */
int free_lcmaps_db_entry(void)
{
  lcmaps_db_entry_t* plugin = global_plugin_list;

  while (plugin) {
    lcmaps_db_entry_t* tmp = plugin->next;
    free(plugin);
    plugin = tmp;
  }

  global_plugin_list = NULL;

  return 0;
}
