/***************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as               *
*   published by the Free Software Foundation; either version 2.0 of      *
*   the License, or (at your option) any later version.                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
*   General Public License for more details.                              *
*                                                                         *
*   You should have received a copy of the GNU General Public             *
*   License along with this program; if not, write to                     *
*   the Free Software Foundation, Inc.,                                   *
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
***************************************************************************/

#include "client_main_window.h"

ClientMainWindow::ClientMainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    setWindowIcon(QIcon(":/images/kontrolpack.png"));
    settings = new QSettings("OpenNetwork", "KontrolPack");

    int width = settings->value("startup/width", QString::number(this->size().width())).toInt();
    int height = settings->value("startup/height", QString::number(this->size().height())).toInt();
    this->resize(QSize(width, height));

    QString title = tr("KontrolPack ") + QString(VERSION) + tr(" : Secure remote access [TCP/IP client mode] ");
    setWindowTitle(title);
#ifdef WIN32
    QFont font("Arial", 8, true);
#elif defined __APPLE__
    QFont font = QFont();
#else
    QFont font("Arial", 10, true);
#endif
    setUnifiedTitleAndToolBarOnMac(true);
    // Menus creation
    QMenu *menu = new QMenu;
    addComputerAction = menu->addAction(tr("&Connect to remote computer"));
    addComputerAction->setShortcut(QKeySequence("Ctrl+A"));
    addComputerAction->setIcon(QIcon(":/images/add.png"));
    menu->addSeparator();
    switchModeAction = menu->addAction(tr("&Switch KontrolPack Mode Client/Server"));
    switchModeAction->setShortcut(QKeySequence("Ctrl+S"));
    switchModeAction->setIcon(QIcon(":/images/go-jump.png"));
    menu->addSeparator();
    QAction *dropListAction = menu->addAction(tr("&Empty computers database"));
    menu->addSeparator();
    updateAction = menu->addAction(tr("&Check for updates"));
    updateAction->setShortcut(QKeySequence("Ctrl+U"));
    updateAction->setIcon(QIcon(":/images/update.png"));
    menu->addSeparator();
    QAction *userModeAction = menu->addAction(tr("&User mode"));
    userModeAction->setShortcut(QKeySequence("Ctrl+M"));
    userModeAction->setIcon(QIcon(":/images/go-jump.png"));
    menu->addSeparator();
    QAction *fullScreenAction = menu->addAction(tr("&Full Screen"));
    fullScreenAction->setShortcut(QKeySequence("Ctrl+Shift+F11"));
    fullScreenAction->setIcon(QIcon(":/images/video-display.png"));
    menu->addSeparator();
    aboutAction = menu->addAction(tr("&About KontrolPack"));
    aboutAction->setIcon(QIcon(":/images/about.png"));
    menu->addSeparator();
    quitAction = menu->addAction(tr("&Quit"));
    quitAction->setShortcut(QKeySequence("Ctrl+Q"));
    quitAction->setIcon(QIcon(":/images/exit.png"));

    QToolButton *menuButton = new QToolButton;
    menuButton->setToolTip(tr("Menu settings"));
    menuButton->setPopupMode(QToolButton::InstantPopup);
    menuButton->setContextMenuPolicy(Qt::DefaultContextMenu);
    menuButton->setIcon(QIcon(":/images/settings.png"));
    menuButton->setMenu(menu);

#ifdef __APPLE__
    // Main toolBar
    QToolBar *fileToolBar = addToolBar(tr("Shell"));
    fileToolBar->addAction(addComputerAction);
    fileToolBar->addAction(switchModeAction);
    fileToolBar->addAction(aboutAction);
    fileToolBar->addAction(updateAction);
    fileToolBar->addAction(quitAction);
#endif

    //StatusBar creation
    toolBar = statusBar();
    toolBar->setFont(font);
    toolBar->setAttribute(Qt::WA_TranslucentBackground, true);
    toolBar->showMessage(tr("RSA | AES encryption over TCP/IP"));
    QLabel *security = new QLabel(this);
    security->setPixmap(QPixmap(":/images/security.png"));
    toolBar->addWidget(security);

    // Central zone creation
    centralWindow = new ClientCentralWindow(menuButton);
    centralWindow->setAttribute(Qt::WA_TranslucentBackground, false);
    centralWindow->setAttribute(Qt::WA_NoSystemBackground, false);
    setCentralWidget(centralWindow);

    centerOnScreen();

    // Tray icon
    createActions();
    createTrayIcon();
    trayIcon->setIcon(QIcon(":/images/kontrolpack.png"));
    trayIcon->setToolTip(tr("KontrolPack : Secure remote access."));
    trayIcon->show();

    //Initialisation
    aboutDialog = new AboutDialog();

    //signals and slots connections
    QObject::connect(addComputerAction, SIGNAL(triggered()), centralWindow, SLOT(show_connect_dialog()));
    QObject::connect(userModeAction, SIGNAL(triggered()), this, SLOT(switchUserMode()));
    QObject::connect(switchModeAction, SIGNAL(triggered()), this, SLOT(switchMode()));
    QObject::connect(dropListAction, SIGNAL(triggered()), centralWindow, SLOT(drop_database()));
    QObject::connect(updateAction, SIGNAL(triggered()), this, SLOT(displayUpdateManager()));
    QObject::connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason)));
    QObject::connect(quitAction, SIGNAL(triggered()), qApp, SLOT(quit()));
    QObject::connect(aboutAction, SIGNAL(triggered()), this, SLOT(displayAbout()));
    QObject::connect(centralWindow, SIGNAL(showMessage(QString,QString,int)), this, SLOT(showMessage(QString,QString,int)));
    QObject::connect(fullScreenAction, SIGNAL(triggered()), this, SLOT(fullScreenMode()));
}

void ClientMainWindow::createActions()
{
    minimizeAction = new QAction(tr("Mi&nimize KontrolPack"), this);
    connect(minimizeAction, SIGNAL(triggered()), this, SLOT(hide()));

    maximizeAction = new QAction(tr("Ma&ximize KontrolPack"), this);
    connect(maximizeAction, SIGNAL(triggered()), this, SLOT(showMaximized()));
}

void ClientMainWindow::createTrayIcon()
{
    trayIconMenu = new QMenu(this);
    trayIconMenu->addAction(addComputerAction);
    trayIconMenu->addSeparator();
    trayIconMenu->addAction(switchModeAction);
    trayIconMenu->addSeparator();
    trayIconMenu->addAction(minimizeAction);
    trayIconMenu->addAction(maximizeAction);
    trayIconMenu->addSeparator();
    trayIconMenu->addAction(updateAction);
    trayIconMenu->addAction(aboutAction);
    trayIconMenu->addSeparator();
    trayIconMenu->addAction(quitAction);

    trayIcon = new QSystemTrayIcon(this);
    trayIcon->setContextMenu(trayIconMenu);
}

void ClientMainWindow::iconActivated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
    case QSystemTrayIcon::Trigger:
        if (this->isVisible()) {
            hide();
        } else {
            show();
        }
        break;
        case QSystemTrayIcon::DoubleClick:
        break;
    case QSystemTrayIcon::MiddleClick:
        QMessageBox::information(this, tr("About KontrolPack"), tr("KontrolPack  : Secure remote access."));
        break;
    default:
        ;
    }
}

void ClientMainWindow::closeEvent(QCloseEvent *event)
{
    if (trayIcon->isVisible()) {
        showMessage(tr("Information"), tr("KontrolPack will keep running into the "
                                          "system tray. To terminate the program, "
                                          "choose \"Quit\" in the context menu "
                                          "that pops up when clicking this program's "
                                          "entry in the system tray."), 4000);
        hide();
        event->ignore();
    } else {
        // Free ressources
        centralWindow->database->freeDatabase();
        trayIcon->hide();
        trayIcon->setVisible(false);
    }
}

void ClientMainWindow::centerOnScreen()
{
    QDesktopWidget screen;
    QRect screenGeom = screen.screenGeometry(this);
    int screenCenterX = screenGeom.center().x();
    int screenCenterY = screenGeom.center().y();
    move(screenCenterX - width () / 2, screenCenterY - height() / 2);
}

void ClientMainWindow::switchMode()
{
    settings->setValue("startup/mode", "NONE");
    showMessage(tr("Switch KontrolPack Mode warning"), tr("Please, restart KontrolPack in order to choose another mode."), 4000);
}

void ClientMainWindow::displayAbout()
{
    aboutDialog->exec();
}

void ClientMainWindow::showMessage(QString title, QString content, int duration)
{
    trayIcon->showMessage(title, content, QSystemTrayIcon::Information, duration);
}

void ClientMainWindow::displayUpdateManager()
{
    UpdateManager *updateManager = new UpdateManager();
    updateManager->updateManager();
}

void ClientMainWindow::resizeEvent(QResizeEvent* event)
{
    settings->setValue("startup/width", QString::number(this->size().width()));
    settings->setValue("startup/height", QString::number(this->size().height()));
}

void ClientMainWindow::fullScreenMode()
{
    if (this->windowState() != Qt::WindowFullScreen)
        showFullScreen();
    else
        showMaximized();
}

void ClientMainWindow::switchUserMode()
{
    UserModeWidGet *mode_widget = new UserModeWidGet;
    mode_widget->exec();
}

