/* Copyright 2018-present Samsung Electronics Co., Ltd. and other contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var assert = require('assert');
var fs = require('fs');
var http = require('http');
var https = require('https');

var response_message = 'DATA';
var response_got = 'NOT THIS';

/*
 * Test if 'subclassing' the IncomingMessage
 * and ServerResponse is correctly propagated.
 */
function newIncomingMessage() {
  http.IncomingMessage.apply(this, arguments);
};
newIncomingMessage.prototype = Object.create(http.IncomingMessage.prototype);
newIncomingMessage.prototype.extra_field = 'found';
newIncomingMessage.prototype.waitForBody = function() {
  var chunks = [];
  this.on('data', function(chunk) {
    chunks.push(chunk);
  });
  this.on('end', function() {
    this.emit('full_body', Buffer.concat(chunks));
  });
};


function newServerResponse() {
  http.ServerResponse.apply(this, arguments);
};
newServerResponse.prototype = Object.create(http.ServerResponse.prototype);
newServerResponse.prototype.sendJSON = function (obj) {
  var message_body = JSON.stringify(obj);
  this.setHeader('Content-Length', message_body.length);
  this.setHeader('Content-Type', 'application/json');
  this.writeHead(200);
  this.end(message_body);
};

var serveropts = {
  IncomingMessage: newIncomingMessage,
  ServerResponse: newServerResponse,

  key: fs.readFileSync(process.cwd() + '/resources/my_key.key'),
  cert: fs.readFileSync(process.cwd() + '/resources/my_crt.crt')
};

var server = https.createServer(serveropts, function (req, res) {
  assert.equal(req.method, 'POST', 'Incorrect request method detected');
  assert.equal(req.url, '/put_msg', 'Incorrect request url detected');
  assert.assert(req instanceof http.IncomingMessage,
                'Request is not instance of http.IncomingMessage');
  assert.assert(req instanceof newIncomingMessage,
                'Request is not instance of the new Request object');
  assert.equal(req.extra_field, 'found',
               'No "extra_field" property on the request instance');

  req.waitForBody();
  req.on('full_body', function(body) {
    assert.assert(body instanceof Buffer);
    assert.equal(body.toString(), 'DEMO');

    res.sendJSON({message: response_message});
  });
});

server.listen(3001);

var demo_msg = 'DEMO';
var options = {
  method : 'POST',
  port : 3001,
  path : '/put_msg',
  headers: {
    'Content-Length': demo_msg.length,
  },
  rejectUnauthorized: false
};
var req = https.request(options, function (response){
  var content_type =
    response.headers['Content-Type'] || response.headers['content-type']
  assert.equal(content_type, 'application/json',
               'Incorrect content type returned by the server');

  var chunks = []
  response.on('data', function(chunk) {
    chunks.push(chunk);
  });
  response.on('end', function() {
    var body = JSON.parse(Buffer.concat(chunks).toString());
    assert.assert('message' in body, 'No "message" key in response JSON');
    response_got = body.message;

    server.close();
  });
});

req.end(demo_msg);

process.on('exit', function() {
  assert.equal(response_got, response_message,
               'Invalid response returned from the demo server');
});
