from __future__ import absolute_import, print_function

import sys

try:
    import dulwich
except ImportError:
    print("skipped: missing feature: dulwich")
    sys.exit(80)

import os, tempfile, unittest, shutil
from mercurial import ui, hg, commands, pycompat

sys.path.append(os.path.join(os.path.dirname(__file__), os.path.pardir))

from hggit.git_handler import GitHandler


class TestUrlParsing(object):
    def setUp(self):
        # create a test repo location.
        self.tmpdir = tempfile.mkdtemp('hg-git_url-test').encode('utf-8')
        commands.init(ui.ui(), self.tmpdir)
        repo = hg.repository(ui.ui(), self.tmpdir)
        self.handler = GitHandler(repo, ui.ui())

    def tearDown(self):
        # remove the temp repo
        shutil.rmtree(self.tmpdir)

    def assertEquals(self, l, r):
        """assert equality of l and r, converting bytes to str

        This is so we don't have to adapt the whole .t output.
        """
        ls = pycompat.strurl(l)
        rs = pycompat.strurl(r)
        print('%% expect %r' % (r, ))
        print(ls)
        assert ls == rs, (l, r)

    def test_ssh_github_style_slash(self):
        url = b"git+ssh://git@github.com/webjam/webjam.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, '/webjam/webjam.git')
        self.assertEquals(client.host, 'git@github.com')

    def test_ssh_github_style_colon_number_starting_username(self):
        url = b"git+ssh://git@github.com:42qu/vps.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, '42qu/vps.git')
        self.assertEquals(client.host, 'git@github.com')

    def test_ssh_github_style_colon(self):
        url = b"git+ssh://git@github.com:webjam/webjam.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, 'webjam/webjam.git')
        self.assertEquals(client.host, 'git@github.com')

    def test_ssh_heroku_style(self):
        url = b"git+ssh://git@heroku.com:webjam.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, 'webjam.git')
        self.assertEquals(client.host, 'git@heroku.com')
        # also test that it works even if heroku isn't in the name
        url = b"git+ssh://git@compatible.com:webjam.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, 'webjam.git')
        self.assertEquals(client.host, 'git@compatible.com')

    def test_ssh_heroku_style_with_trailing_slash(self):
        # some versions of mercurial add a trailing slash even if
        #  the user didn't supply one.
        url = b"git+ssh://git@heroku.com:webjam.git/"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, 'webjam.git')
        self.assertEquals(client.host, 'git@heroku.com')

    def test_heroku_style_with_port(self):
        url = b"git+ssh://git@heroku.com:999:webjam.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, 'webjam.git')
        self.assertEquals(client.host, 'git@heroku.com')
        self.assertEquals(client.port, '999')

    def test_gitdaemon_style(self):
        url = b"git://github.com/webjam/webjam.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, '/webjam/webjam.git')
        try:
            self.assertEquals(client._host, 'github.com')
        except AttributeError:
            self.assertEquals(client.host, 'github.com')

    def test_ssh_github_style_slash_with_port(self):
        url = b"git+ssh://git@github.com:10022/webjam/webjam.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, '/webjam/webjam.git')
        self.assertEquals(client.host, 'git@github.com')
        self.assertEquals(client.port, '10022')

    def test_gitdaemon_style_with_port(self):
        url = b"git://github.com:19418/webjam/webjam.git"
        client, path = self.handler._get_transport_and_path(url)
        self.assertEquals(path, '/webjam/webjam.git')
        try:
            self.assertEquals(client._host, 'github.com')
        except AttributeError:
            self.assertEquals(client.host, 'github.com')
        self.assertEquals(client._port, '19418')

if __name__ == '__main__':
    tc = TestUrlParsing()
    for test in sorted(t for t in dir(tc) if t.startswith('test_')):
        tc.setUp()
        getattr(tc, test)()
        tc.tearDown()
