/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file ab_thunderbird.c
 * \brief Thunderbird address book plugin
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include "ffgtk.h"
#include "addressbook.h"

#define MORK_COLUMN_META	"<(a=c)>"
#define DEFAULT_SCOPE		0x80

#define MAX_VAL				0x7FFFFFFF

enum {
	PARSE_VALUES,
	PARSE_ROWS,
	PARSE_COLUMNS
};

static gchar *pnMorkData = NULL;
static int nMorkPos = 0;
static int nMorkNowParsing = PARSE_VALUES;
static long nMorkNextAddValueId = MAX_VAL;
static GHashTable *psMorkValues = NULL;
static GHashTable *psMorkColumns = NULL;

static GHashTable *psCurrentCells = NULL;

static GHashTable *psTableScopeMap = NULL;
static gchar *pnThunderbirdDir = NULL;
static int nNumPossible = 0;
static int nNumPersons = 0;
static int nMorkSize = 0;

/**
 * \brief Get selected thunderbird addressbook
 * \return thunderbird addressbook
 */
static const gchar *thunderbirdGetSelectedBook( void ) {
	return prefsGetString( getActiveProfile(), "/plugins/thunderbird/book" );
}

/**
 * \brief Set selected thunderbird addressbook
 * \param pnUri thunderbird addressbook
 */
static void thunderbirdSetSelectedBook( gchar *pnUri ) {
	prefsSetString( getActiveProfile(), "/plugins/thunderbird/book", pnUri );
}

/**
 * \brief Destroy hashtable
 * \param pData hashtable widget
 */
void hashDestroy( void *pData ) {
	g_hash_table_destroy( pData );
}

/**
 * \brief Create map structure
 * \param FreeData free data function
 * \return new hash table
 */
static GHashTable *CreateMap( GDestroyNotify notify ) {
	GHashTable *psTable;

	psTable = g_hash_table_new_full( g_direct_hash, g_direct_equal, NULL, notify );

	return psTable;
}

/**
 * \brief Insert key and value into map structure, check for double entries
 * \param psTable pointer to hash table
 * \param nKey key value
 * \param pValue value pointer
 */
static inline void InsertMap( GHashTable *psTable, long nKey, void *pValue ) {
	g_hash_table_insert( psTable, GINT_TO_POINTER( nKey ), pValue );
}

/**
 * \brief Find map entry by key
 * \param psTable pointer to hash table
 * \param nKey key id
 * \return pValue of key or NULL on error
 */
static inline void *FindMapEntry( GHashTable *psTable, int nKey ) {
	return g_hash_table_lookup( psTable, GINT_TO_POINTER( nKey ) );
}

/**
 * \brief Get thunderbird directory
 * \return string to directory
 */
static gchar *getThunderbirdDir( void ) {
	gchar *pnBuffer;
	gchar anFile[ 256 ];
	gchar *pnPath;
	gchar *pnRelative;
	GString *psResult;
	gboolean bVersion3 = FALSE;
	gboolean bRelative = TRUE;

	psResult = g_string_new( NULL );
	snprintf( anFile, sizeof( anFile ), "%s/.mozilla-thunderbird/profiles.ini", getHomeDir() );

	pnBuffer = ( gchar * ) loadData( anFile, NULL );
	if ( pnBuffer == NULL ) {
		snprintf( anFile, sizeof( anFile ), "%s/.thunderbird/profiles.ini", getHomeDir() );
		pnBuffer = ( gchar * ) loadData( anFile, NULL );
		bVersion3 = TRUE;
	}

	if ( pnBuffer != NULL ) {
		pnRelative = strstr( pnBuffer, "IsRelative=" );
		if ( pnRelative != NULL ) {
			bRelative = pnRelative[ 11 ] == '1';
		}

		pnPath = strstr( pnBuffer, "Path" );
		if ( pnPath != NULL ) {
			pnPath += 5;

			if ( bRelative == TRUE ) {
				psResult = g_string_append( psResult, getHomeDir() );
				if ( bVersion3 == FALSE ) {
					psResult = g_string_append( psResult, "/.mozilla-thunderbird/" );
				} else {
					psResult = g_string_append( psResult, "/.thunderbird/" );
				}
			}

			while ( pnPath != NULL && *pnPath != '\n' ) {
				psResult = g_string_append_c( psResult, ( gchar ) *pnPath++ );
			}
	
			while ( psResult -> str[ strlen( psResult -> str ) - 1 ] == '\n' ) {
				psResult -> str[ strlen( psResult -> str ) - 1 ] = '\0';
			}
			psResult -> str[ strlen( psResult -> str ) ] = '\0';
		}
		g_free( pnBuffer );
	}

	return g_string_free( psResult, FALSE );
}

/**
 * \brief Get next gchar of buffer
 * \return next gchar
 */
static gchar nextChar( void ) {
	gchar cCur = 0;

	if ( pnMorkData != NULL && nMorkPos < nMorkSize ) {
		cCur = pnMorkData[ nMorkPos ];
		nMorkPos++;
	}

	return cCur;
}

/**
 * \brief Check if gchar is whitespace
 * \param cChar gchar to check
 * \return 1 or 0
 */
static gchar isWhiteSpace( gchar cChar ) {
	switch ( cChar ) {
		case ' ':
		case '\t':
		case '\r':
		case '\n':
		case '\f':
			return 1;
		default:
			return 0;
	}
}

/**
 * \brief Parse comment section
 * \return 1 on success, else error
 */
static gchar parseComment( void ) {
	gchar cCur = nextChar();

	if ( cCur != '/' ) {
		return 0;
	}

	while ( cCur != '\r' && cCur != '\n' && cCur ) {
		cCur = nextChar();
	}

	return 1;
}

/**
 * \brief Parse cell section
 * \return 1 on success, else error
 */
static gchar parseCell( void ) {
	gchar bResult = 1;
	gchar bColumn = 1;
	gchar cCur = nextChar();
	GString *psColumn = g_string_new( NULL );
	GString *psText = g_string_new( NULL );
	int nCorners = 0;
	gchar bValue0id = 0;
	int bHex = 0;
	GString *psHexString = g_string_new( NULL );

	while ( bResult && cCur != ')' && cCur ) {
		switch ( cCur ) {
			case '=':
				/* From column to value */
				if ( bColumn ) {
					bColumn = 0;
				} else {
					psText = g_string_append_c( psText, cCur );
				}
				break;
			case '$': {
				gchar anHexChar[ 3 ];
				gchar cNext1, cNext2;
				int nX;

				cNext1 = nextChar();
				cNext2 = nextChar();
				snprintf( anHexChar, sizeof( anHexChar ), "%c%c", cNext1, cNext2 );
				nX = strtoul( anHexChar, 0, 16 );
				psHexString = g_string_append_c( psHexString, nX );
				bHex = 1;
				break;
			}
			case '\\': {
				gchar cNextChar = nextChar();
				if ( cNextChar != '\r' && cNextChar != '\n' ) {
					psText = g_string_append_c( psText, cNextChar );
				} else {
					nextChar();
				}
				break;
			}
			case '^':
				nCorners++;
				if ( nCorners == 1 ) {
				} else if ( nCorners == 2 ) {
					bColumn = 0;
					bValue0id = 1;
				} else {
					psText = g_string_append_c( psText, cCur );
				}
				break;
			default:
				if ( bColumn ) {
					psColumn = g_string_append_c( psColumn, cCur );
				} else {
					psText = g_string_append_c( psText, cCur );
				}
				break;
		}

		cCur = nextChar();
		if ( bHex && cCur != '$' ) {
			psText = g_string_append( psText, psHexString -> str );
			g_string_set_size( psHexString, 0 );
			bHex = 0;
		}
	}

	int nColumnId = strtoul( psColumn -> str, 0, 16 );
	if ( nMorkNowParsing != PARSE_ROWS ) {
		/* Dicts */
		if ( psText && strlen( psText -> str ) ) {
			//Debug( KERN_DEBUG, "Text: %s, %lx, %d\n", psText -> str, nColumnId, nMorkNowParsing == PARSE_COLUMNS );
			if ( nMorkNowParsing == PARSE_COLUMNS ) {
				InsertMap( psMorkColumns, nColumnId, strdup( psText -> str ) );
			} else {
				InsertMap( psMorkValues, nColumnId, strdup( psText -> str ) );
			}
		}
	} else {
		if ( psText && strlen( psText -> str ) ) {
			long nValueId = strtoul( psText -> str, 0, 16 );

			//Debug( KERN_DEBUG, "bValue0id: %d\n", bValue0id );
			/* Rows */
			if ( bValue0id ) {
				InsertMap( psCurrentCells, nColumnId, ( void * ) nValueId );
			} else {
				nMorkNextAddValueId--;
				InsertMap( psMorkValues, nMorkNextAddValueId, strdup( psText -> str ) );
				InsertMap( psCurrentCells, nColumnId, ( void * ) nMorkNextAddValueId );
				//Debug( KERN_DEBUG, "psText: %s\n", psText -> str );
			}
		}
	}

	g_string_free( psHexString, TRUE );
	g_string_free( psColumn, TRUE );
	g_string_free( psText, TRUE );

	return bResult;
}

/**
 * \brief Parse dictionary section
 * \return 1 on success, else error
 */
static gchar parseDict( void ) {
	gchar cCur = nextChar();
	gchar bResult = 1;

	nMorkNowParsing = PARSE_VALUES;

	while ( bResult && cCur != '>' && cCur ) {
		if ( !isWhiteSpace( cCur ) ) {
			switch ( cCur ) {
				case '<':
					if ( !strncmp( pnMorkData + nMorkPos - 1, MORK_COLUMN_META, strlen( MORK_COLUMN_META ) ) ) {
						nMorkNowParsing = PARSE_COLUMNS;
						nMorkPos += strlen( MORK_COLUMN_META ) - 1;
					}
					break;
				case '/':
					bResult = parseComment();
					break;
				case '(':
					bResult = parseCell();
					break;
				default:
					printf( "[%s]: error '%c'\n", __FUNCTION__, cCur );
					bResult = 0;
					break;
			}
		}
		cCur = nextChar();
	}

	return bResult;
}

/**
 * \brief Parse scope id
 * \param pnText while text
 * \param pnId pointer to save id
 * \param pnScope pointer to save scope
 */
static void parseScopeId( GString *psText, int *pnId, int *pnScope ) {
	gchar *pnPos;

	if ( ( pnPos = strchr( psText -> str, ':' ) ) != NULL ) {
		int nSize = pnPos - psText -> str;
		gchar *pnIdStr = NULL;
		gchar *pnScStr = NULL;
		int nPos = nSize;

		pnIdStr = g_malloc( nSize + 1 );
		strncpy( pnIdStr, psText -> str, nPos );
		pnIdStr[ nSize ] = '\0';

		nSize = strlen( psText -> str ) - nPos;
		pnScStr = g_malloc( nSize );
		strncpy( pnScStr, psText -> str + nPos + 1, nSize );
		pnScStr[ strlen( pnScStr ) ] = '\0';
		if ( strlen( pnScStr ) > 1 && pnScStr[ 0 ] == '^' ) {
			gchar *pnTmp = g_malloc( strlen( pnScStr ) );
			strncpy( pnTmp, pnScStr + 1, strlen( pnScStr ) );
			g_free( pnScStr );
			pnScStr = pnTmp;
		}

		*pnId = strtoul( pnIdStr, 0, 16 );
		g_free( pnIdStr );
		*pnScope = strtoul( pnScStr, 0, 16 );
		g_free( pnScStr );
	} else {
		*pnId = strtoul( psText -> str, 0, 16 );
		*pnScope = 0;
	}
}

/**
 * \brief Parse meta section
 * \param cChar current gchar
 * \return 1 on success, else error
 */
static gchar parseMeta( gchar cChar ) {
	gchar cCur = nextChar();

	while ( cCur != cChar && cCur ) {
		cCur = nextChar();
	}

	return 1;
}

/**
 * \brief Set current row
 * \param nTableScope table scope id
 * \param nTableId table id
 * \param nRowScope row scope id
 * \param nRowId row id
 */
static void setCurrentRow( int nTableScope, int nTableId, int nRowScope, int nRowId ) {
	GHashTable *psMap;
	GHashTable *psTmp;

	if ( !nRowScope ) {
		nRowScope = DEFAULT_SCOPE;
	}

	if ( !nTableScope ) {
		nTableScope = DEFAULT_SCOPE;
	}

	//Debug( KERN_DEBUG, "Set to %lx/%lx/%lx/%lx\n", nTableScope, nTableId, nRowScope, nRowId );

	/* First: Get table scope map */
	psTmp = FindMapEntry( psTableScopeMap, abs( nTableScope ) );
	if ( psTmp == NULL ) {
		InsertMap( psTableScopeMap, abs( nTableScope ), CreateMap( hashDestroy ) );
		psTmp = FindMapEntry( psTableScopeMap, abs( nTableScope ) );
		if ( psTmp == NULL ) {
			printf( "[%s]: Could not create table scope map!!\n", __FUNCTION__ );
			return;
		}
	}
	psMap = psTmp;

	/* Second: Get table id map */
	psTmp = FindMapEntry( psMap, abs( nTableId ) );
	if ( psTmp == NULL ) {
		InsertMap( psMap, abs( nTableId ), CreateMap( hashDestroy ) );
		psTmp = FindMapEntry( psMap, abs( nTableId ) );
		if ( psTmp == NULL ) {
			printf( "[%s]: Could not create table id map!!\n", __FUNCTION__ );
			return;
		}
	}
	psMap = psTmp;

	/* Third: Get row scope map */
	psTmp = FindMapEntry( psMap, abs( nRowScope ) );
	if ( psTmp == NULL ) {
		InsertMap( psMap, abs( nRowScope ), CreateMap( hashDestroy ) );
		psTmp = FindMapEntry( psMap, abs( nRowScope ) );
		if ( psTmp == NULL ) {
			printf( "[%s]: Could not create row scope map!!\n", __FUNCTION__ );
			return;
		}
	}
	psMap = psTmp;

	/* Fourth: Get row id map */
	psTmp = FindMapEntry( psMap, abs( nRowId ) );
	if ( psTmp == NULL ) {
		InsertMap( psMap, abs( nRowId ), CreateMap( NULL ) );
		psTmp = FindMapEntry( psMap, abs( nRowId ) );
		if ( psTmp == NULL ) {
			printf( "[%s]: Could not create row id map!!\n", __FUNCTION__ );
			return;
		}
	}
	psMap = psTmp;

	psCurrentCells = psMap;
}

/**
 * \brief Parse row
 * \param nTableId table id
 * \param nTableScope table scope id
 * \return 1 on success, else error
 */
static gchar parseRow( int nTableId, int nTableScope ) {
	gchar bResult = 1;
	gchar cCur = nextChar();
	GString *psText = g_string_new( NULL );
	int nId, nScope;

	nMorkNowParsing = PARSE_ROWS;

	while ( cCur != '(' && cCur != ']' && cCur != '[' && cCur ) {
		if ( !isWhiteSpace( cCur ) ) {
			psText = g_string_append_c( psText, cCur );
		}
		cCur = nextChar();
	}

	parseScopeId( psText, &nId, &nScope );
	setCurrentRow( nTableScope, nTableId, nScope, nId );

	while ( bResult && cCur != ']' && cCur ) {
		if ( !isWhiteSpace( cCur ) ) {
			switch ( cCur ) {
				case '(':
					bResult = parseCell();
					break;
				case '[':
					bResult = parseMeta( ']' );
					break;
				default:
					bResult = 0;
					break;
			}
		}
		cCur = nextChar();
	}

	g_string_free( psText, TRUE );

	return bResult;
}

/**
 * \brief Parse table section
 * \return 1 on success, else error
 */
static gchar parseTable( void ) {
	gchar bResult = 1;
	gchar cCur = nextChar();
	GString *psTextId = g_string_new( NULL );
	int nId = 0, nScope = 0;

	while ( cCur != '{' && cCur != '[' && cCur != '}' && cCur ) {
		if ( !isWhiteSpace( cCur ) ) {
			psTextId = g_string_append_c( psTextId, cCur );
		}
		cCur = nextChar();
	}

	parseScopeId( psTextId, &nId, &nScope );

	while ( bResult && cCur != '}' && cCur ) {
		if ( !isWhiteSpace( cCur ) ) {
			switch ( cCur ) {
				case '{':
					bResult = parseMeta( '}' );
					break;
				case '[':
					bResult = parseRow( nId, nScope );
					break;
				case '-':
				case '+':
					break;
				default: {
					GString *psJustId = g_string_new( NULL );

					while ( !isWhiteSpace( cCur ) && cCur ) {
						psJustId = g_string_append_c( psJustId, cCur );
						cCur = nextChar();

						if ( cCur == '}' ) {
							return bResult;
						}
					}

					int nJustIdNum = 0, nJustScopeNum = 0;
					parseScopeId( psJustId, &nJustIdNum, &nJustScopeNum );
					setCurrentRow( nScope, nId, nJustScopeNum, nJustIdNum );
					g_string_free( psJustId, TRUE );
					break;
				}
			}
		}
		cCur = nextChar();
	}

	g_string_free( psTextId, TRUE );

	return bResult;
}

/**
 * \brief Parse group section
 * \return 1 on success, else error
 */
static gchar parseGroup( void ) {
	return parseMeta( '@' );
}

/**
 * \brief Parse mork code
 * \return 1 on success, else error
 */
static gchar parseMork( void ) {
	gchar bResult = 1;
	gchar cCur = 0;

	cCur = nextChar();

	while ( bResult && cCur ) {
		if ( !isWhiteSpace( cCur ) ) {
			switch ( cCur ) {
				case '/':
					/* Comment */
					bResult = parseComment();
					break;
				case '<':
					/* Dict */
					bResult = parseDict();
					break;
				case '{':
					/* Table */
					bResult = parseTable();
					break;
				case '@':
					/* Group */
					bResult = parseGroup();
					break;
				case '[':
					/* Row */
					bResult = parseRow( 0, 0 );
					break;
				default:
					printf( "[%s]: Error: %c\n", __FUNCTION__, cCur );
					bResult = 0;
					break;
			}
		}
		cCur = nextChar();
	}

	g_free( pnMorkData );
	pnMorkData = NULL;

	return bResult;
}

/**
 * \brief Get column entry by key
 * \param nKey key id
 * \return column entry
 */
static inline gchar *getColumn( int nKey ) {
	return g_hash_table_lookup( psMorkColumns, GINT_TO_POINTER( nKey ) );
}

/**
 * \brief Get value entry by key
 * \param nKey key id
 * \return value entry
 */
static inline gchar *getValue( int nKey ) {
	return g_hash_table_lookup( psMorkValues, GINT_TO_POINTER( nKey ) );
}

/**
 * \brief Parse person data
 * \param psMap pointer to map structure holding person informations
 * \param pId id
 */
static void parsePerson( GHashTable *psMap, gpointer pId ) {
	const gchar *pnFirstName = NULL;
	const gchar *pnLastName = NULL;
	const gchar *pnHome = NULL;
	const gchar *pnWork = NULL;
	const gchar *pnFax = NULL;
	const gchar *pnMobile = NULL;
	const gchar *pnCheck = NULL;
	const gchar *pnCompany = NULL;
	const gchar *pnPrimaryEmail = NULL;
	const gchar *pnDisplayName = NULL;
	const gchar *pnTitle = NULL;
	const gchar *pnHomeAddress = NULL;
	const gchar *pnHomeCity = NULL;
	const gchar *pnHomeZipCode = NULL;
	const gchar *pnHomeCountry = NULL;
	const gchar *pnWorkAddress = NULL;
	const gchar *pnWorkCity = NULL;
	const gchar *pnWorkZipCode = NULL;
	const gchar *pnWorkCountry = NULL;
	const gchar *pnPhotoName = NULL;
	gchar *pnCity = NULL;
	GdkPixbuf *psImage = NULL;
	GHashTableIter sIter5;
	gpointer pKey5, pValue5;
	GHashTable *psTable = NULL;
	struct sPerson *psPerson = NULL;

	nNumPossible++;
	g_hash_table_iter_init( &sIter5, psMap );
	while ( g_hash_table_iter_next( &sIter5, &pKey5, &pValue5 ) )  {
		if ( GPOINTER_TO_INT( pKey5 ) == 0 ) {
			continue;
		}
		const gchar *pnColumn = getColumn( GPOINTER_TO_INT( pKey5 ) );
		//Debug( KERN_DEBUG, "pnColumn = '%s'\n", pnColumn );
		const gchar *pnValue = getValue( GPOINTER_TO_INT( pValue5 ) );
		//Debug( KERN_DEBUG, "pnValue = '%s'\n", pnValue );

		if ( !strcmp( pnColumn, "FirstName" ) ) {
			pnFirstName = pnValue;
		} else if ( !strcmp( pnColumn, "LastName" ) ) {
			pnLastName = pnValue;
		} else if ( !strcmp( pnColumn, "HomePhone" ) ) {
			pnHome = pnValue;
		} else if ( !strcmp( pnColumn, "WorkPhone" ) ) {
			pnWork = pnValue;
		} else if ( !strcmp( pnColumn, "FaxNumber" ) ) {
			pnFax = pnValue;
		} else if ( !strcmp( pnColumn, "CellularNumber" ) ) {
			pnMobile = pnValue;
		} else if ( !strcmp( pnColumn, "Company" ) ) {
			pnCompany = pnValue;
		} else if ( !strcmp( pnColumn, "PrimaryEmail" ) ) {
			pnPrimaryEmail = pnValue;
		} else if ( !strcmp( pnColumn, "DisplayName" ) ) {
			pnDisplayName = pnValue;
		} else if ( !strcmp( pnColumn, "HomeAddress" ) ) {
			pnHomeAddress = pnValue;
		} else if ( !strcmp( pnColumn, "HomeCity" ) ) {
			pnHomeCity = pnValue;
		} else if ( !strcmp( pnColumn, "HomeZipCode" ) ) {
			pnHomeZipCode = pnValue;
		} else if ( !strcmp( pnColumn, "HomeCountry" ) ) {
			pnHomeCountry = pnValue;
		} else if ( !strcmp( pnColumn, "WorkAddress" ) ) {
			pnWorkAddress = pnValue;
		} else if ( !strcmp( pnColumn, "WorkCity" ) ) {
			pnWorkCity = pnValue;
		} else if ( !strcmp( pnColumn, "WorkZipCode" ) ) {
			pnWorkZipCode = pnValue;
		} else if ( !strcmp( pnColumn, "WorkCountry" ) ) {
			pnWorkCountry = pnValue;
		} else if ( !strcmp( pnColumn, "JobTitle" ) ) {
			pnTitle = pnValue;
		} else if ( !strcmp( pnColumn, "PhotoName" ) ) {
			pnPhotoName = pnValue;
		}
	}

	/* Do not add entries without name */
	if ( pnFirstName == NULL && pnLastName == NULL && pnCompany == NULL ) {
		return;
	}

	if ( pnDisplayName == NULL ) {
		if ( pnLastName != NULL ) {
			pnDisplayName = pnLastName;
		} else if ( pnFirstName != NULL ) {
			pnDisplayName = pnFirstName;
		} else if ( pnCompany != NULL ) {
			pnDisplayName = pnCompany;
		} else if ( pnPrimaryEmail != NULL ) {
			pnDisplayName = pnPrimaryEmail;
		} else {
			Debug( KERN_WARNING, "No display name set and no alternative found!\n" );
			return;
		}
	}

	if ( pnPrimaryEmail != NULL ) {
		pnCheck = pnPrimaryEmail;
	} else {
		pnCheck = pnDisplayName;
	}

	if ( pnCheck != NULL && strlen( pnCheck ) > 0 ) {
		gchar *pnFileName = NULL;
		gchar *pnFileNameExt = NULL;
		gchar *pnBase = NULL;
		gchar *pnDest = NULL;
		gchar *pnSrc = NULL;

		pnBase = g_base64_encode( ( const guchar * ) pnCheck, strlen( pnCheck ) );

		pnSrc = pnBase;
		pnDest = pnBase;
		do {
			while ( *pnSrc == '=' ) {
				pnSrc++;
			}
		} while ( ( *pnDest++ = *pnSrc++ ) != '\0' );

		pnFileName = g_strdup_printf( "%s/ABphotos/%s", pnThunderbirdDir, pnBase );
		g_free( pnBase );

		pnFileNameExt = g_strdup_printf( "%s.jpg", pnFileName );
		psImage = gdk_pixbuf_new_from_file( pnFileNameExt, NULL );
		if ( psImage == NULL ) {
			g_free( pnFileNameExt );
			pnFileNameExt = g_strdup_printf( "%s.png", pnFileName );
			psImage = gdk_pixbuf_new_from_file( pnFileNameExt, NULL );
			if ( psImage == NULL ) {
				g_free( pnFileNameExt );
				pnFileNameExt = g_strdup_printf( "%s.gif", pnFileName );
				psImage = gdk_pixbuf_new_from_file( pnFileNameExt, NULL );
			}
		}

		g_free( pnFileNameExt );
		g_free( pnFileName );
	}

	if ( pnHomeCity != NULL ) {
		if ( pnHomeZipCode != NULL ) {
			pnCity = g_strdup_printf( "%s %s", pnHomeZipCode, pnHomeCity );
		} else {
			pnCity = g_strdup_printf( "%s", pnHomeCity );
		}
	}

	if ( psImage == NULL && pnPhotoName != NULL ) {
		gchar *pnFileName = NULL;

		pnFileName = g_strdup_printf( "%s/Photos/%s", pnThunderbirdDir, pnPhotoName );
		psImage = gdk_pixbuf_new_from_file( pnFileName, NULL );
		g_free( pnFileName );
	}

	psPerson = findPerson( pnDisplayName );
	if ( psPerson != NULL ) {
		removePerson( pnDisplayName );
	}

	psTable = g_hash_table_new( NULL, NULL );

	gchar *pnTmp = g_strdup_printf( "%d", nNumPersons );
	AddInfo( psTable, PERSON_ID, pnTmp );
	AddInfo( psTable, PERSON_FIRST_NAME, pnFirstName );
	AddInfo( psTable, PERSON_LAST_NAME, pnLastName );
	AddInfo( psTable, PERSON_DISPLAY_NAME, pnDisplayName );
	AddInfo( psTable, PERSON_COMPANY, pnCompany );
	AddInfo( psTable, PERSON_TITLE, pnTitle );
	AddInfo( psTable, PERSON_BUSINESS_PHONE, pnWork );
	/* BUSINESS FAX not supported */
	AddInfo( psTable, PERSON_BUSINESS_FAX, NULL );
	AddInfo( psTable, PERSON_PRIVATE_PHONE, pnHome );
	AddInfo( psTable, PERSON_PRIVATE_FAX, pnFax );
	AddInfo( psTable, PERSON_PRIVATE_MOBILE, pnMobile );

	AddInfo( psTable, PERSON_PRIVATE_STREET, pnHomeAddress );
	AddInfo( psTable, PERSON_PRIVATE_CITY, pnHomeCity );
	AddInfo( psTable, PERSON_PRIVATE_ZIPCODE, pnHomeZipCode );
	AddInfo( psTable, PERSON_PRIVATE_COUNTRY, pnHomeCountry );
	AddInfo( psTable, PERSON_BUSINESS_STREET, pnWorkAddress );
	AddInfo( psTable, PERSON_BUSINESS_CITY, pnWorkCity );
	AddInfo( psTable, PERSON_BUSINESS_ZIPCODE, pnWorkZipCode );
	AddInfo( psTable, PERSON_BUSINESS_COUNTRY, pnWorkCountry );
	AddInfo( psTable, PERSON_IMAGE, ( const gchar * ) psImage );

	AddPerson( psTable, FALSE );
	g_free( pnTmp );

	if ( pnCity != NULL ) {
		g_free( pnCity );
	}

	g_hash_table_destroy( psTable );

	nNumPersons++;
}

/**
 * \brief Parse tables for important informations
 */
static void parseTables( void ) {
	GHashTable *psTables = psTableScopeMap;

	if ( psTables != NULL ) {
		GHashTableIter sIter1;
		gpointer pKey1, pValue1;

		g_hash_table_iter_init( &sIter1, psTables );
		while ( g_hash_table_iter_next( &sIter1, &pKey1, &pValue1 ) )  {
			if ( GPOINTER_TO_INT( pKey1 ) == 0 ) {
				//continue;
			}

			GHashTable *psRows = pValue1;
			if ( psRows != NULL ) {
				GHashTableIter sIter2;
				gpointer pKey2, pValue2;

				g_hash_table_iter_init( &sIter2, psRows );
				while ( g_hash_table_iter_next( &sIter2, &pKey2, &pValue2 ) )  {
					if ( GPOINTER_TO_INT( pKey2 ) == 0 ) {
						//continue;
					}

					GHashTable *psRows2 = pValue2;
					if ( psRows2 != NULL ) {
						GHashTableIter sIter3;
						gpointer pKey3, pValue3;

						g_hash_table_iter_init( &sIter3, psRows2 );
						while ( g_hash_table_iter_next( &sIter3, &pKey3, &pValue3 ) )  {
							if ( GPOINTER_TO_INT( pKey3 ) == 0 ) {
								//continue;
							}

							GHashTable *psRows3 = pValue3;
							if ( psRows3 != NULL ) {
								GHashTableIter sIter4;
								gpointer pKey4, pValue4;

								g_hash_table_iter_init( &sIter4, psRows3 );
								while ( g_hash_table_iter_next( &sIter4, &pKey4, &pValue4 ) )  {
									if ( GPOINTER_TO_INT( pKey4 ) == 0 ) {
										//continue;
									}

									GHashTable *psRows4 = pValue4;
									if ( psRows4 != NULL ) {
										parsePerson( psRows4, pKey4 );
									}			
								}
							}			
						}
					}			
				}
			}			
		}
	}
}

/**
 * \brief Open thunderbird address book
 * \param pnBook address book file name
 */
static void openThunderbirdBook( gchar *pnBook ) {
	int nFile, nSize;

	nFile = open( pnBook, O_RDONLY );
	if ( nFile == -1 ) {
		return;
	}

	nSize = lseek( nFile, 0, SEEK_END );
	lseek( nFile, 0, SEEK_SET );

	pnMorkData = g_malloc( nSize );
	if ( pnMorkData != NULL ) {
		nMorkSize = nSize;
		if ( read( nFile, pnMorkData, nSize ) == nSize ) {
			Debug( KERN_DEBUG, "Parsing mork\n" );
			parseMork();
			Debug( KERN_DEBUG, "Parsing tables\n" );
			parseTables();
			Debug( KERN_DEBUG, "Done\n" );
		}
	}

	close( nFile );
}

/**
 * \brief Read thunderbird book
 * \return error code
 */
static int thunderbirdReadBook( void ) {
	const gchar *pnBook;
	gchar anFile[ 256 ];

	nNumPersons = 0;
	nNumPossible = 0;
	Debug( KERN_DEBUG, "Start\n" );
	pnMorkData = NULL;
	nMorkPos = 0;
	nMorkNowParsing = PARSE_VALUES;
	nMorkNextAddValueId = MAX_VAL;
	psMorkValues = NULL;
	psMorkColumns = NULL;
	psCurrentCells = NULL;
	psTableScopeMap = NULL;

	Debug( KERN_DEBUG, "Create maps\n" );
	psMorkValues = CreateMap( free );
	psMorkColumns = CreateMap( free );
	psTableScopeMap = CreateMap( hashDestroy );

	Debug( KERN_DEBUG, "get thunderbird book\n" );

	pnThunderbirdDir = getThunderbirdDir();
	if ( pnThunderbirdDir == NULL ) {
		return -1;
	}

	pnBook = thunderbirdGetSelectedBook();
	if ( ( pnBook == NULL ) || ( strlen( pnBook ) <= 0 ) ) {
		snprintf( anFile, sizeof( anFile ), "%s/abook.mab", pnThunderbirdDir );
	} else {
		strncpy( anFile, pnBook, sizeof( anFile ) );
	}


	Debug( KERN_DEBUG, "Thunderbird book (%s)\n", anFile );
	openThunderbirdBook( anFile );
	Debug( KERN_DEBUG, "free structure\n" );

	g_hash_table_destroy( psTableScopeMap );
	g_hash_table_destroy( psMorkColumns );
	g_hash_table_destroy( psMorkValues );
	Debug( KERN_DEBUG, "Done\n" );
	Debug( KERN_DEBUG, "%d possible entries!\n", nNumPossible );
	Debug( KERN_DEBUG, "%d persons imported!\n", nNumPersons );

	return 0;
}

/**
 * \brief Display thunderbird preferences window
 */
static void thunderbirdPreferences( void ) {
	GtkWidget *psDialog;
	GtkFileFilter *psFilter;
	gint nResult;
	gchar *pnDir;
	gchar anFile[ 256 ];
	const gchar *pnBook;

	psDialog = gtk_file_chooser_dialog_new( _( "Select Thunderbird addressbook" ), NULL, GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL );
	psFilter = gtk_file_filter_new();
	gtk_file_filter_add_pattern( psFilter, "*.mab" );

	gtk_file_chooser_set_filter( GTK_FILE_CHOOSER( psDialog ), psFilter );

	pnBook = thunderbirdGetSelectedBook();
	if ( pnBook != NULL && strlen( pnBook ) > 0 ) {
		gtk_file_chooser_set_filename( GTK_FILE_CHOOSER( psDialog ), pnBook );
	} else {
		pnDir = getThunderbirdDir();
		snprintf( anFile, sizeof( anFile ), "%s/abook.mab", pnDir );

		gtk_file_chooser_set_filename( GTK_FILE_CHOOSER( psDialog ), anFile );
		g_free( pnDir );
	}

	nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );

	if ( nResult == GTK_RESPONSE_OK ) {
		gchar *pnFileName = NULL;

		prefsAddNone( getActiveProfile(), "/plugins/thunderbird" );

		pnFileName = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( psDialog ) );
		thunderbirdSetSelectedBook( pnFileName );
		g_free( pnFileName );

		SavePreferences( getActiveProfile() );
		freePersons();

		thunderbirdReadBook();
	}

	gtk_widget_destroy( psDialog );
}

/** book definition */
static struct sAddressBook sThunderbirdBook = {
	thunderbirdReadBook,
	NULL,
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME | PERSON_COMPANY |
	PERSON_TITLE | PERSON_IMAGE	| PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_PRIVATE_FAX | PERSON_PRIVATE_STREET | PERSON_PRIVATE_CITY |
	PERSON_PRIVATE_ZIPCODE | PERSON_PRIVATE_COUNTRY | PERSON_BUSINESS_PHONE |
	PERSON_BUSINESS_FAX | PERSON_BUSINESS_STREET | PERSON_BUSINESS_CITY |
	PERSON_BUSINESS_ZIPCODE | PERSON_BUSINESS_COUNTRY,
	0,
	thunderbirdPreferences
};

MODULE_INIT( PLUGIN_TYPE_BOOK, _( "Thunderbird Addressbook" ), &sThunderbirdBook, NULL, NULL );
