/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file plugins.c
 * \brief Plugins (load/unload/handling...)
 */

#include <ffgtk.h>

/** global plugin list */
GList *psPluginList = NULL;

/**
 * \brief Get default book plugin
 * \return book plugin pointer or NULL on error
 */
struct sAddressBook *getDefaultBookPlugin( struct sProfile *psProfile ) {
	if ( psProfile != NULL && psProfile -> psDefaultBook != NULL ) {
		return psProfile -> psDefaultBook;
	}

	return NULL;
}

/**
 * \brief Get default fax plugin
 * \return fax plugin pointer or NULL on error
 */
struct sFax *getDefaultFaxPlugin( struct sProfile *psProfile ) {
	if ( psProfile != NULL && psProfile -> psDefaultFax != NULL ) {
		return psProfile -> psDefaultFax;
	}

	return NULL;
}

/**
 * \brief Get default password plugin
 * \return password plugin pointer or NULL on error
 */
struct sPassword *getDefaultPasswordPlugin( struct sProfile *psProfile ) {
	if ( psProfile != NULL && psProfile -> psDefaultPassword != NULL ) {
		return psProfile -> psDefaultPassword;
	}

	return NULL;
}

/**
 * \brief Get default audio plugin
 * \return audio plugin pointer or NULL on error
 */
struct sAudio *getDefaultAudioPlugin( struct sProfile *psProfile ) {
	if ( psProfile != NULL && psProfile -> psDefaultAudio != NULL ) {
		return psProfile -> psDefaultAudio;
	}

	return NULL;
}

/**
 * \brief Find plugin by name
 * \param nType plugin type
 * \param pnName plugin name
 * \return plugin structure
 */
struct sPlugin *findPlugin( gint nType, const gchar *pnName ) {
	GList *psList = NULL;
	struct sPlugin *psPlugin = NULL;

	if ( pnName == NULL ) {
		return NULL;
	}

	for ( psList = psPluginList; psList != NULL; psList = psList -> next ) {
		psPlugin = psList -> data;
		if ( psPlugin && psPlugin -> nType == nType ) {
			if ( !strcmp( psPlugin -> pnName, pnName ) ) {
				return psPlugin;
			}
		}
	}

	return NULL;
}

/**
 * \brief Transfer password from one password plugin to another
 * \param psProfile profile structure
 * \param pnPlugin new password plugin
 */
void transferPassword( struct sProfile *psProfile, gchar *pnPlugin ) {
	GList *psList = NULL;
	struct sPassword *psPasswordNew = NULL;
	struct sPlugin *psPlugin = NULL;

	for ( psList = psPluginList; psList != NULL; psList = psList -> next ) {
		psPlugin = psList -> data;
		if ( psPlugin && psPlugin -> nType == PLUGIN_TYPE_PASSWORD ) {
			if ( !strcmp( psPlugin -> pnName, pnPlugin ) ) {
				psPasswordNew = psPlugin -> pData;
				break;
			}
		}
	}

	if ( psProfile -> psDefaultPassword != psPasswordNew ) {
		/* We switched the password plugin, remove old entry and set new one!! */
		struct sPassword *psPasswordOld = psProfile -> psDefaultPassword;
		const gchar *pnPassword = NULL;

		pnPassword = routerGetPassword( psProfile );
		psProfile -> psDefaultPassword = psPasswordNew;
		routerSetPassword( psProfile, pnPassword );
		psProfile -> psDefaultPassword = psPasswordOld;
		routerRemovePassword( psProfile );
	}
}

/**
 * \brief Set default plugin of type nType
 * \param psProfile profile structure
 * \param nType plugin type
 * \param pnName plugin name
 */
void setDefaultPlugin( struct sProfile *psProfile, int nType, const char *pnName ) {
	GList *psList;
	struct sPlugin *psPlugin;

	//Debug( KERN_DEBUG, "psProfile: %p, nType %d, pnName %p\n", psProfile, nType, pnName );
	if ( pnName == NULL ) {
		return;
	}
	//Debug( KERN_DEBUG, "psProfile: %p, nType %d, pnName %s\n", psProfile, nType, pnName );

	if ( nType == PLUGIN_TYPE_LOOKUP ) {
		struct sLookup *psLookup = NULL;

		for ( psList = getLookupList( routerGetCountryCode( psProfile ) ); psList != NULL && psList -> data != NULL; psList = psList -> next ) {
			psLookup = psList -> data;
			if ( strcmp( psLookup -> pnService, pnName ) == 0 ) {
				psProfile -> psDefaultLookup = psLookup;
				return;
			}
		}
	} else {
		for ( psList = psPluginList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
			psPlugin = psList -> data;
			//Debug( KERN_DEBUG, "Type check %d <-> %d\n", psPlugin -> nType, nType );
			if ( psPlugin && psPlugin -> nType == nType ) {
				//Debug( KERN_DEBUG, "Name check '%s' <-> '%s'\n", psPlugin -> pnName, pnName );
				if ( !strcmp( psPlugin -> pnName, pnName ) ) {
					switch ( nType ) {
						case PLUGIN_TYPE_BOOK:
							if ( psProfile -> psDefaultBook != psPlugin -> pData ) {
								freePersons();
								psProfile -> psDefaultBook = psPlugin -> pData;
								Debug( KERN_DEBUG, "ReadBook (%s)\n", psPlugin -> pnName );
								psProfile -> psDefaultBook -> ReadBook();
								Debug( KERN_DEBUG, "ReadBook done\n" );
							}
							return;
						case PLUGIN_TYPE_FAX:
							psProfile -> psDefaultFax = psPlugin -> pData;
							return;
						case PLUGIN_TYPE_PASSWORD:
							psProfile -> psDefaultPassword = psPlugin -> pData;
							psProfile -> psDefaultPassword -> GetPassword( psProfile );
							return;
						case PLUGIN_TYPE_AUDIO:
							if ( psProfile -> psDefaultAudio != psPlugin -> pData ) {
								if ( psProfile -> psDefaultAudio != NULL ) {
									psProfile -> psDefaultAudio -> Deinit();
								}
								psProfile -> psDefaultAudio = psPlugin -> pData;
								psProfile -> psDefaultAudio -> Init( 1, 8000, 16 );
							}
							return;
						default:
							Debug( KERN_WARNING, "Unknown type %d\n", nType );
					}
				}
			}
		}
	}

	//Debug( KERN_DEBUG, "Error: Plugin not found, setting NULL!\n" );

	switch ( nType ) {
		case PLUGIN_TYPE_LOOKUP:
			psProfile -> psDefaultLookup = NULL;
			break;
		case PLUGIN_TYPE_BOOK:
			psProfile -> psDefaultBook = NULL;
			break;
		case PLUGIN_TYPE_FAX:
			psProfile -> psDefaultFax = NULL;
			break;
		case PLUGIN_TYPE_PASSWORD:
			psProfile -> psDefaultPassword = NULL;
			break;
		case PLUGIN_TYPE_AUDIO:
			psProfile -> psDefaultAudio = NULL;
			break;
		default:
			Debug( KERN_WARNING, "Unknown type %d\n", nType );
	}
}

/**
 * \brief Register plugin
 * \param psPlugin plugin structure pointer
 * \return error code
 */
static int RegisterPlugin( struct sPlugin *psPlugin ) {
	if ( psPlugin == NULL || psPlugin -> pnName == NULL || psPlugin -> nVersion != PLUGIN_LOADER_VERSION ) {
		g_free( psPlugin -> pnName );
		g_free( psPlugin );		
		return -1;
	}

	psPluginList = g_list_append( psPluginList, psPlugin );

	Debug( KERN_DEBUG, "Adding '%s'\n", psPlugin -> pnName );

	return 0;
}

/**
 * \brief load plugin by filename
 * \param pnName file name
 * \return error code
 */
static int loadPlugin( char *pnName ) {
	struct sPlugin *psPlugin;
	GModule *psModule;
	typedef int ( *InitPlugin )( struct sPlugin *psPlugin );
	InitPlugin initPlugin;
	gpointer pTmpPointer;

	Debug( KERN_DEBUG, "Trying to load '%s'\n", pnName );
	psModule = g_module_open( pnName, G_MODULE_BIND_LAZY );
	if ( psModule == NULL ) {
		Debug( KERN_WARNING, "Cannot load plugin %s\n%s\n", pnName, g_module_error() );
		return -1;
	}

	if ( !g_module_symbol( psModule, "InitPlugin", &pTmpPointer ) ) {
		Debug( KERN_WARNING, "Cannot load symbol 'InitPlugin' : %s\n", g_module_error() );
		g_module_close( psModule );
		return -2;
	}
	initPlugin = pTmpPointer;

	psPlugin = g_malloc( sizeof( struct sPlugin ) );

	if ( initPlugin( psPlugin ) != 0 ) {
		g_free( psPlugin );
		return -3;
	}

	psPlugin -> psModule = psModule;
	return RegisterPlugin( psPlugin );
}

/**
 * \brief Check if file has extension
 * \param pnFile filename
 * \param pnExt extension
 * \return TRUE or FALSE
 */
gboolean has_file_extension( const char *pnFile, const char *pnExt ) {
	int nLen, nExtLen;

	if ( pnFile == NULL || *pnFile == '\0' || pnExt == NULL ) {
		return FALSE;
	}

	nExtLen = strlen( pnExt );
	nLen = strlen( pnFile ) - nExtLen;

	if ( nLen < 0 ) {
		return FALSE;
	}

	return ( strncmp( pnFile + nLen, pnExt, nExtLen ) == 0 );
}

/**
 * \brief Find and load plugins from pnPluginDir
 * \param pnPluginDir plugin directory
 */
void PluginsLoad( char *pnPluginDir ) {
	GDir *dir;
	const gchar *file;
	gchar *path;
	char *pnExt = G_MODULE_SUFFIX;

	dir = g_dir_open( pnPluginDir, 0, NULL );
	if ( dir != NULL ) {
		while ( ( file = g_dir_read_name( dir ) ) != NULL ) {
			path = g_build_filename( pnPluginDir, file, NULL );
			if ( pnExt == NULL || has_file_extension( file, pnExt ) ) {
				loadPlugin( path );
			}
			g_free( path );
		}
		g_dir_close( dir );
	} else {
		Debug( KERN_WARNING, "Could not open plugin directory '%s'\n", pnPluginDir );
	}
}

/**
 * \brief Unload plugins
 */
void PluginsUnload( void ) {
	GList *psList;
	struct sPlugin *psPlugin;

	while ( psPluginList != NULL ) {
		psList = psPluginList;
		psPlugin = psList -> data;
		if ( psPlugin != NULL ) {
			Debug( KERN_DEBUG, "Plugin: %s\n", psPlugin -> pnName );
			if ( psPlugin -> Close != NULL ) {
				psPlugin -> Close();
			}
			//g_module_close( psPlugin -> psModule );
			g_free( psPlugin -> pnName );
			psPlugin -> pData = NULL;
			psPluginList = g_list_delete_link( psPluginList, psList );
			g_free( psPlugin );
			Debug( KERN_DEBUG, "Plugin: done\n" );
		}
	}
}
