# SPDX-License-Identifier: MIT
# Copyright (C) 2004-2008 Tristan Seligmann and Jonathan Jacobs
# Copyright (C) 2012-2014 Bastian Kleineidam
# Copyright (C) 2015-2022 Tobias Gruetzmacher
# Copyright (C) 2019-2020 Daniel Ring
import os

from ..scraper import ParserScraper
from ..helpers import bounceStarter

XPATH_LINK = '//a[d:class("%s") and contains(text(), "%s")]'
XPATH_IMG = '//div[d:class("comicnav")]//a[img[contains(@alt, "%s")]]'


class ComicFury(ParserScraper):
    imageSearch = (
        '//img[@id="comicimage"]',
        '//div[@id="comicimagewrap"]//embed',
        '//div[@id="comicimagewrap"]//img',
    )
    prevSearch = (
        '//link[@rel="prev"]',
        # 137 (needs to be before the generic a@rel, because layout is wrong)
        '//a[contains(@title, "previous")]',
        '//a[@rel="prev"]',
        XPATH_LINK % ('comicnavlink', 'Previous'),
        XPATH_IMG % ('Previous'),
        # Art, ConsolersDLC, etc.
        u'//nav//a[contains(text(), "\u2039")]',
        # LatchkeyKingdom
        '//a[d:class("navi") and img[contains(@src, "Previous")]]',
        # KATRAN
        '//a[contains(text(), "Previous")]',
        # MansionofE
        '//a[img[contains(@alt, "PREVIOUS")]]',
        # RedSpot
        '//a[contains(text(), "Back")]',
    )
    nextSearch = (
        '//link[@rel="next"]',
        # 137 (see above)
        '//a[contains(@title, "next")]',
        '//a[@rel="next"]',
        XPATH_LINK % ('comicnavlink', 'Next'),
        XPATH_IMG % ('Next'),
        # Art, ConsolersDLC, etc.
        u'//nav//a[contains(text(), "\u203A")]',
        # LatchkeyKingdom
        '//a[d:class("navi") and img[contains(@src, "Next")]]',
        # RedSpot, KATRAN
        '//a[contains(text(), "Next")]',
        # MansionofE
        '//a[img[contains(@alt, "NEXT")]]',
    )
    help = 'Index format: n'
    starter = bounceStarter

    def __init__(self, name, sub, lang=None, adult=False, endOfLife=False, segmented=False):
        super().__init__('ComicFury/' + name)
        self.prefix = name
        self.url = 'https://%s.webcomic.ws/comics/' % sub
        self.stripUrl = self.url + '%s'
        self.firstStripUrl = self.stripUrl % '1'
        if lang:
            self.lang = lang
        if adult:
            self.adult = adult
        if endOfLife:
            self.endOfLife = endOfLife
        if segmented:
            self.multipleImagesPerStrip = True
            self.imageSearch = self.imageSearch + (
                '//img[d:class("comicsegmentimage")]',
            )

    def namer(self, image_url, page_url):
        parts = page_url.split('/')
        path, ext = os.path.splitext(image_url)
        num = parts[-1]
        return "%s_%s%s" % (self.prefix, num, ext)

    def shouldSkipUrl(self, url, data):
        """Skip pages without images."""
        # Videos on Underverse
        return (data.xpath('//div[@id="comicimagewrap"]//video') and
            not data.xpath('//div[@id="comicimagewrap"]//img'))

    @classmethod
    def getmodules(cls):  # noqa: Allowed to be long
        return (
            # These were once in the list below, but fell out from the index...
            cls('BadassologyByMichaelBay', 'strudelology'),
            cls('DandyAndCompany', 'dandyandcompany'),
            cls('DeadAtNight', 'deadnight'),
            cls('Shatterrealm', 'shatterrealm'),

            # do not edit anything below since these entries are generated from
            # scripts/comicfury.py
            # START AUTOUPDATE
            cls('0Eight', '0eight'),
            cls('1000', '1000'),
            cls('12YearsLater', '12yearslater'),
            cls('137', '137'),
            cls('20', 'two-over-zero'),
            cls('20QuidAmusements', 'twentyquidamusements'),
            cls('30', '30years'),
            cls('30DaysOfCharacters', '30days'),
            cls('3DGlasses', '3dglasses'),
            cls('60SecondComics', '6tsc'),
            cls('6ColorStories', '6colorstories'),
            cls('6Tales', 'sixtales'),
            cls('933Dollars', '933dollars'),
            cls('_Thetest_', 'thetest'),
            cls('AbbyComics', 'abbycomics'),
            cls('ABrickishSpaceComic', 'abrickishspacecomic'),
            cls('AbsentMindedTheatre', 'amtheatre'),
            cls('Absurd', 'absurd'),
            cls('ACannonadeOfHogwash', 'cannonadeofhogwash'),
            cls('AccidentallyOnPurpose', 'accidentally-on-purpose'),
            cls('ACelestialStory', 'acelestialstory'),
            cls('AComicExistense', 'acomicexistense'),
            cls('Acroalis', 'acroalis'),
            cls('ActingOut', 'actingout'),
            cls('ActionLand', 'actionland'),
            cls('Advent', 'advent'),
            cls('AdventuresInJetpacks', 'adventuresinjetpacks'),
            cls('AdventuresInTanoshii', 'adventuresintanoshii'),
            cls('AdventuresInTrueLove', 'advtl'),
            cls('Aerosol', 'aerosol'),
            cls('AetherEarthAndSun', 'aether'),
            cls('AForeverQuest', 'aforeverquest'),
            cls('Afterdead', 'afterdead'),
            cls('AGame', 'kirahitogame'),
            cls('Agency', 'agency-comic'),
            cls('AgentBishop', 'agentbishop'),
            cls('AHappierKindOfSad', 'ahappierkindofsad'),
            cls('AlbinoBrothers', 'albinobros'),
            cls('Alderwood', 'alderwood'),
            cls('AlexanderAndLucasRebooted', 'alexanderandlucas'),
            cls('AliaTerra', 'alia-terra'),
            cls('AlienIrony', 'alien-irony'),
            cls('AlienSpike', 'alienspike'),
            cls('Alignment', 'alignment'),
            cls('AllTheBbqSauce', 'allthebbqsauce'),
            cls('Alone', 'alone'),
            cls('ALoonaticsTale', 'aloonaticstale'),
            cls('ALoveStorydraft', 'alovestory'),
            cls('AlyaTheLastChildOfLight', 'alya'),
            cls('Amara', 'amara'),
            cls('Ampre', 'ampere'),
            cls('AmyOok', 'amyook'),
            cls('AndroidFiles', 'androidfiles'),
            cls('AngelGuardianEnEspanol', 'angelguardianespanol', 'es'),
            cls('AngelsOfIblis', 'angelsofiblis'),
            cls('AngryFaerie', 'angryfaerie'),
            cls('AnimalInstinct', 'fur-realanimalinstinct'),
            cls('Animangitis', 'animangitis'),
            cls('AnK', 'ank'),
            cls('Anne', 'anne'),
            cls('AntarcticBroadcasting', 'antarcticbroadcasting'),
            cls('AntaresComplex', 'antarescomplex'),
            cls('Antcomics', 'antcomics'),
            cls('Anthology', 'strudelology'),
            cls('AnthologyOfAnfer', 'anfer'),
            cls('AnthrosAndDungeons', 'anthrosanddungeons'),
            cls('AntiqueTimeMachine', 'atm'),
            cls('APiratesLife', 'pirateslife'),
            cls('ApocalypsoAdventure', 'thewriter13'),
            cls('ApplepineMonkeyAndFriends', 'applepine'),
            cls('AquazoneBreakfastNews', 'aqbn'),
            cls('ArachnidGoddess', 'arachnidgoddess'),
            cls('Arcane', 'rbsarcane'),
            cls('Archibald', 'archibald'),
            cls('ArchiNinja', 'archininja'),
            cls('AreYouDoneYet', 'areyoudoneyet'),
            cls('ArmlessAmy', 'armlessamy'),
            cls('ArmlessAmyExtraEdition', 'armlessamyextraedition'),
            cls('ArmyBrat', 'armybrat'),
            cls('Art', 'art'),
            cls('ArtificialStorm', 'artificialstorm'),
            cls('ArtisticAdventuresInBoredom', 'aab'),
            cls('ARVEYToonz', 'arveytoonz'),
            cls('Ashes', 'ashescomic'),
            cls('Asperchu', 'asperchu'),
            cls('AsperitasAstraalia', 'asperitasastraalia'),
            cls('AssholeAndDouchebag', 'aaanddb'),
            cls('AstralAves', 'astralaves'),
            cls('ASTRAYCATS', 'astraycats'),
            cls('Astronautical', 'astronautical'),
            cls('AtomicMonkeyComics', 'atomicmonkey'),
            cls('ATownCalledAlandale', 'atowncalledalandale'),
            cls('AttackOfTheRobofemoids', 'attack-of-the-robofemoids'),
            cls('AugustosClassic', 'augustos-classic'),
            cls('AuntieClara', 'auntieclara'),
            cls('Auriga', 'auriga'),
            cls('Auster', 'auster'),
            cls('AutumnBay', 'autumnbay'),
            cls('AutumnBayExtraEdition', 'autumnbayextra'),
            cls('Avatars', 'avatars'),
            cls('AvengersRollInitiative', 'avengersrollinitiative'),
            cls('AwkwardPaws', 'awkwardpaws'),
            cls('AwkwardShelby', 'awkwardshelby'),
            cls('BabesOfDongaria', 'dongaria'),
            cls('Baby001', 'baby001'),
            cls('BabyBatman', 'babybatman'),
            cls('BackToTheRefridgerator', 'bttf'),
            cls('BadAdjectives', 'badadjectives'),
            cls('BananaCreamCake', 'bananacreamcake'),
            cls('BarkingCrayon', 'barkingcrayon'),
            cls('BASKERVILLE', 'baskerville'),
            cls('BASO', 'baso'),
            cls('BattleOfTheRobofemoids', 'battle-of-the-robofemoids'),
            cls('BeatStuffUpMan', 'beatstuffupman'),
            cls('BeepClub', 'beepclub'),
            cls('BeePolice', 'beepolice'),
            cls('Beezwax', 'beezwax'),
            cls('BeforeAndAfter', 'beforeandafter'),
            cls('Being', 'being'),
            cls('BELECOMICS', 'belecomics'),
            cls('BentElbows', 'bentelbows'),
            cls('BetaParticles', 'betaparticles'),
            cls('BetweenTheFrames', 'betweentheframes'),
            cls('BetweenTheInterval', 'betweentheinterval'),
            cls('BibleBelt', 'biblebelt'),
            cls('BilateralComics', 'bilateralcomics'),
            cls('BionicleTales', 'bionicletales'),
            cls('BioSyte', 'biosyte'),
            cls('Birdman', 'birdman'),
            cls('BlankLifeInsertPlayerRokulily', 'blanklife'),
            cls('BlackTapestries', 'blacktapestries', adult=True),
            cls('BlitzPhoenix', 'blinix'),
            cls('BlobWorld', 'blobworld'),
            cls('BlueBloodHeroes', 'bluebloodheroes'),
            cls('BoatcrashChronicles', 'boatcrash'),
            cls('BobbyTheFetus', 'bobbythefetus'),
            cls('Boobgirl', 'boobgirl'),
            cls('BookOfThree', 'bookofthree'),
            cls('BooksDontWorkHere', 'booksdontworkhere'),
            cls('BorisAndBjorn', 'borisandbjorn'),
            cls('Boritom', 'boritom'),
            cls('BrainFood', 'brainfood'),
            cls('BrainTeaser', 'brainteaser'),
            cls('BritarsesHashHymnal', 'hashhymnal'),
            cls('BroadoakPeople', 'broadoakpeople'),
            cls('BrokenWings', 'brokenwingscomic'),
            cls('BromosWorld', 'bromosworld'),
            cls('Brujagh', 'brujagh'),
            cls('BubbleFox', 'bubblefox'),
            cls('Bulletproof', 'bulletproof'),
            cls('BunnyGoreJustice', 'bunny-gore-justice'),
            cls('BustySolar', 'bustysolar'),
            cls('ButterflyAFortuitousTale', 'butterfly'),
            cls('ButterflyEffect', 'thebutterflyeffect'),
            cls('BUXYAndDave', 'buxy'),
            cls('BuyingTime', 'buyingtime'),
            cls('CACKLENCOMICS', 'cacklencomics'),
            cls('CactusCanyon', 'cactuscanyon'),
            cls('CAFEGRUESOME', 'cafegruesome'),
            cls('Cagegirl', 'cagegirl'),
            cls('CastOfMadness', 'castofmadness'),
            cls('CatHerosEpicCatventuresAsAnHero', 'cathero'),
            cls('CatosApprenticeship', 'cato'),
            cls('CattDogg', 'cattdogg'),
            cls('Cattic', 'cattic'),
            cls('CattusesChristmasCalendar', 'xmascattuses'),
            cls('CatWithGoggles', 'catwithgoggles'),
            cls('CautionaryTales', 'cautionarytales'),
            cls('CazTheComicStrip', 'cazthecomicstrip'),
            cls('CelticShaman', 'celticshaman'),
            cls('Chainbreaker', 'chainbreaker'),
            cls('ChamberOfTheArcanum', 'cofthea'),
            cls('ChampionOfKatara', 'championofkatara'),
            cls('ChanpuruSaga', 'chanpuru'),
            cls('CharacterBattleBetweenRounds', 'between-rounds'),
            cls('CHLOE', 'chloe'),
            cls('ChocoLavaCOMICScom', 'chocolava'),
            cls('Chosen', 'chosentheultimatecliche'),
            cls('CHRISTMASEVETheFirstLadyOfYuletideCheer', 'coolyulecomics'),
            cls('ChristmasWithMadDog', 'christmas-with-maddog'),
            cls('ChronoRedux', 'chronoredux'),
            cls('Cinder', 'cinder'),
            cls('CircusJaxs', 'circusjaxs', segmented=True),
            cls('CityFolk', 'cityfolkwebcomics'),
            cls('CityOfDream', 'cityofdream'),
            cls('CKarrus', 'ckarrus'),
            cls('ClassicElsewhere', 'classicelsewhere'),
            cls('ClassicMissJAndTheAmComics19842006', 'missjandtheam'),
            cls('ClydeNOwen', 'clydenowen'),
            cls('COCHLEAAndEUSTACHIA', 'chromefetus'),
            cls('CockeyedComix', 'cockeyed'),
            cls('Code', 'code'),
            cls('CollegeMunchies', 'collegemunchies'),
            cls('Colorforce', 'colorforce'),
            cls('ComicFuryFanArtExchanges', 'cfexchanges'),
            cls('ComicShopOfHorror', 'comicshop'),
            cls('ComicShortsTheMainSeries', 'comicshortsmain'),
            cls('ComingApartments', 'comingapartments'),
            cls('COMIXTURE', 'comixture'),
            cls('CommonReadComicAdaptions', 'slucommonread'),
            cls('CompanyManComic', 'companyman'),
            cls('ConcerningJustice', 'concerningjustice'),
            cls('CONIES', 'conies'),
            cls('ConradTheCaterpillar', 'conradthecaterpillar'),
            cls('Consolers', 'consolers'),
            cls('ConsolersDLC', 'consolers-dlc'),
            cls('ContestedTerritory', 'contestedterritory'),
            cls('CoolstarComicsMasterFiles', 'coolstarcomicsmasterfiles'),
            cls('CopyPasteAndMrBenjy', 'copypasteandmrbenjy'),
            cls('Corpses', 'corpses'),
            cls('Cosmos', 'planetcosmos'),
            # CourageousManAdventures has a duplicate in ComicSherpa/CourageousManAdventures
            cls('CowboysAndCrossovers', 'cowboysandcrossovers'),
            cls('Cowtoon', 'cowtoon'),
            cls('CrackPutty', 'crackputty'),
            cls('CRashCourse', 'crashcourse'),
            cls('Crawlers', 'crawlers'),
            cls('CrimsonPixelComics', 'crimsonpixel'),
            cls('Critters', 'critters'),
            cls('CrossoverChampionship', 'crossoverchampionship'),
            cls('CrossoverExchange', 'crossoverexchange'),
            cls('CrossoverlordAndCrossoverkill', 'crossoverlordkill'),
            cls('CrossWorld', 'crossworld'),
            cls('CrowbarASciFiAdventure', 'crowbar'),
            cls('CrowbarsDontKillPeopleCROWBARSDo', 'crowbars'),
            cls('Cryptida', 'cryptida', 'de'),
            cls('CryptidaEnglish', 'cryptida-eng'),
            cls('CrystalBall', 'crystalball'),
            cls('CtrlZ', 'ctrlz'),
            cls('CubeCows', 'cubecows'),
            cls('CupcakeGraffiti', 'cupcakegraffiti'),
            cls('CYXLOSISM', 'cyxlocistic'),
            cls('DailyDoodle', 'dailydoodle'),
            cls('DailyOneLiner', 'daily1l'),
            cls('DamaclesAndKenjall', 'wowwithatwist-damaclesandkejallcomic'),
            cls('DamnHipsters', 'damnhipsters'),
            cls('DAndDAangvanced', 'danddaangvanced'),
            cls('Daredoers', 'daredoers'),
            cls('DarkHorse', 'darkhorse'),
            cls('Darklings', 'darklings'),
            cls('DarkSisters', 'darksisters'),
            cls('DarVal', 'murghcomics'),
            cls('Datachasers', 'datachasers'),
            cls('DaughterOfDarkness', 'honeyvenom'),
            cls('DaxTapu', 'daxtapu'),
            cls('DDSR', 'ddsr'),
            cls('DEAD', 'dead'),
            cls('DeadDucks', 'deadducks'),
            cls('DeadFingers', 'deadfingers'),
            cls('DeadRabbitCa', 'afairtrade'),
            cls('DeepBlue', 'deepblue'),
            cls('DefineHero', 'definehero'),
            cls('DELIA', 'delia'),
            cls('DemasPokmonAdventure', 'nuzlocke-dema'),
            cls('DesertGrey', 'desertgrey'),
            cls('DesertShark', 'desertshark'),
            cls('Dictatorship', 'dictatorship'),
            cls('DieRabbitDie', 'dierabbitdie'),
            cls('DimensioNoir', 'dimensionoir'),
            cls('DivinaFortuna', 'divinafortuna'),
            cls('DNA', 'd-n-a'),
            cls('DoffeEllende', 'doffeellende'),
            cls('Dogstar', 'dogstar'),
            cls('Domain', 'domain'),
            cls('DonutsForSharks', 'donutsforsharks'),
            cls('DotComic', 'dotcomic'),
            cls('DotX', 'dotx'),
            cls('DoubleJumpGameComics', 'doublejump'),
            cls('Draginbeard', 'draginbeard'),
            cls('DragonballZElsewhere', 'dbzelsewhere'),
            cls('DragonCity', 'dragoncity'),
            cls('DragonsAndSilk', 'dragonsandsilk'),
            cls('DragonsOfAzuma', 'dragonsofazuma'),
            cls('DrApocalyptosSurvivorama', 'docapoc'),
            cls('DressedForSuccess', 'dressedforsuccess'),
            cls('Drettaville', 'drettaville'),
            cls('DrifterJournalsOfAHero', 'drifterjournalsofahero'),
            cls('Drifting', 'drifting'),
            cls('Droned', 'droned'),
            cls('DRouggs', 'drouggs'),
            cls('DrugsAndKisses', 'd-and-k'),
            cls('Druids', 'druids', adult=True),
            cls('DubCity', 'dubcity'),
            cls('DueEast', 'dueeast'),
            cls('DuelingHeroes', 'duelingheroes'),
            # DungeonHordes has a duplicate in ComicSherpa/DungeonHordes
            cls('DungeonMasterEffect', 'dungeonmastereffect'),
            cls('DyerinsLine', 'dyerinsline'),
            cls('EclipseLegend', 'eclipselegend'),
            cls('Educomix', 'educomix'),
            cls('EffinguKookoo', 'effingukookoo'),
            cls('EightBitAdventuresOfCaptainA', 'eightbitadventures'),
            cls('ElektrosComicAnthology', 'elektroanthology'),
            cls('Element8', 'element8'),
            cls('ElementsOfEve', 'elementsofeve'),
            cls('Elf', 'elf-comic'),
            cls('Elsewhere', 'elsewhere'),
            cls('EmpiresOfSteam', 'empiresofsteam'),
            cls('Energize', 'energize'),
            cls('enoZone', 'xenozone'),
            cls('EpicsOfNoche', 'epicsofnoche'),
            cls('Equilibrium', 'equilibrists'),
            cls('Ergosphere', 'ergosphereworld'),
            cls('Eros', 'eros'),
            cls('ErraticElegance', 'erratice'),
            cls('EscapeVelocity', 'escapevelocity'),
            cls('EternalNight', 'eternalnight'),
            cls('EternityComplex', 'eternityc'),
            cls('EverydayAbnormal', 'everydayabnormal'),
            cls('EvilRising', 'evilrising'),
            cls('EWMIC', 'ewmic'),
            cls('ExperiMentalTheatre', 'emt'),
            cls('FacesOfFire', 'facesofire'),
            cls('Fallacy', 'fallacy-harha'),
            cls('Fannicklas', 'fannicklas'),
            cls('FatalExpression', 'fexpression'),
            cls('FBHNKAG', 'fbhnk-ag'),
            cls('FeliciaSorceressOfKatara', 'felicia'),
            cls('FEZ', 'fez'),
            cls('FiendishFellowship', 'fiendishfellowship'),
            cls('FighterDan', 'fighterdan'),
            cls('FingerPuppetShow', 'fingerpuppetshow'),
            cls('FireBorn', 'fireborn2'),
            cls('Fishbowl', 'fishbowl'),
            cls('FishfaceAndBirdbrain', 'ahtiventures'),
            cls('Flickwit', 'flickwit'),
            cls('FlintlockesGuideToAzeroth', 'flintlocke'),
            cls('FlintlockeVsTheHorde', 'flintlockevshorde'),
            cls('ForeignTerritory', 'foreignterritory'),
            cls('ForNathaniel', 'fornathaniel'),
            cls('FoxyFlavoredCookie', 'pobrepucho'),
            cls('FracturedTea', 'fracturedtea'),
            cls('Frames', 'frames'),
            cls('FraterniT', 'fraterni-t'),
            cls('FraternityOfEvil', 'foe'),
            cls('FreeLancer', 'freelancer'),
            cls('FreQuency', 'frequency'),
            cls('FridayAndGrover', 'fridayandgrover'),
            cls('FriendshipIsDragons', 'friendshipisdragons'),
            cls('FromDustToRuination', 'fromdust2ruination'),
            cls('Frontier2170', 'frontier2170'),
            cls('FrostFire', 'frostfire'),
            cls('FullmetalBrothers', 'fullmetalbrothers', 'es'),
            cls('FurAndN3rdy', 'furnerdy'),
            cls('FurryExperience', 'furryexperience'),
            cls('Fusion', 'fusion'),
            cls('FutureRegrets', 'futureregrets'),
            cls('FuzzballAndScuzzball', 'fuzzballandscuzzball'),
            cls('GalbertOfBruges', 'galbertofbruges'),
            cls('GarfieldMinusJon', 'garfieldminusjon'),
            cls('Gatito', 'gatito'),
            cls('GenjiGami', 'genjigami'),
            cls('Ghelis', 'ghelis'),
            cls('GhostGirlsClubZero', 'ghostgirlsclubzero'),
            cls('GhostSyndrome', 'ghostsyndrome'),
            cls('GiantQueenSakura', 'giantqueensakura'),
            cls('GillimurphyStories', 'gillimurphy'),
            cls('GillimurphyStoriesorig', 'gillimurphy-orig'),
            cls('GlomshireKnights', 'glomshire'),
            cls('Glorianna', 'glorianna'),
            cls('GnomereganForever', 'gnomereganforever'),
            cls('GODHATESDADS', 'godhatesdads'),
            cls('GoldBlood', 'goldblood'),
            cls('Goldrush', 'goldrush-dynllewcomics'),
            cls('GrandfathersTale', 'grandfatherstale'),
            cls('Grandify', 'grandify'),
            cls('Gratz', 'gratz'),
            cls('Grayling', 'grayling'),
            cls('GreenEyes', 'greeneyes'),
            cls('GreysterJemp', 'greysterjemp'),
            cls('GrimReaperSchool', 'grimreaperschool'),
            cls('GrippsBrain', 'grippsbrain'),
            cls('GrokBoop', 'grokboop'),
            cls('GrowingTroubles', 'growingtroubles'),
            cls('Guardia', 'guardia-tales-of-halgeis'),
            cls('GUS', 'gus'),
            cls('HalloweenCameoCaper2012', 'halloween2012'),
            cls('HalloweenCameoCaper2013', 'halloween2013'),
            cls('HalloweenCameoCaper2014', 'halloween2014'),
            cls('HARDLUCK', 'hardluck'),
            cls('HAYWIRE', 'haywire'),
            cls('HazardousScience', 'hazsci'),
            cls('HazardsWake', 'hazardswake'),
            cls('HazyDaze', 'hazydaze'),
            cls('HeadRoom', 'headroom'),
            cls('HeadWound', 'headwound'),
            cls('HeartOfKeol', 'keol', segmented=True),
            cls('HeavyLittlePeople', 'heavylittlepeople'),
            cls('HeavyMetalSailorMoon', 'hmsm'),
            cls('Hellbent', 'hellbent'),
            cls('Hellbound', 'hellboundarchive'),
            cls('HellCar', 'hellcar'),
            cls('HenriettaLamb', 'henriettalamb'),
            cls('HeraclesKnot', 'heraclesknot'),
            cls('HeroesAtWork', 'heroesatwork'),
            cls('HeroesOfPower', 'myhorriblesite'),
            cls('HitmanPiranha', 'hitmanpiranha'),
            cls('HitmenForDestiny', 'hitmen'),
            cls('HobGoblinAdventure', 'hobgoblin'),
            cls('Holon', 'holon'),
            cls('HolyBibble', 'holy-bibble'),
            cls('HolyCowComics', 'holycowcomics'),
            cls('HomeOfTheSpaceWalnut', 'hotsw'),
            cls('HoodzAndCaperz', 'hoodzandcaperz'),
            cls('HorizonGakuen', 'horizongakuen'),
            cls('HourlyKelly', 'hourlykelly'),
            cls('HouseOnWritersBlock', 'houseonwritersblock'),
            cls('Housepets1X', 'housepets1x'),
            cls('HowIRememberIt', 'hiri'),
            cls('HowToRaiseYourTeenageDragon', 'teenagedragon'),
            cls('HowWeStaySaneAtWork', 'howwestaysaneatwork'),
            cls('HumanCookies', 'humancookies'),
            cls('HurfanosOrphans', 'huerfanos'),
            cls('HUSH', 'hush'),
            cls('HyperactiveComics', 'hyperactivecomics'),
            cls('ICryWhileYouSleep', 'icrywhileusleep'),
            cls('IDGet', 'idget'),
            cls('IFSU', 'ifsused'),
            cls('IgnitionZero', 'ignitionzero'),
            cls('IlusionOfTime', 'illusionoftime'),
            cls('Immigrant', 'immigrant'),
            cls('ImNotYourFriend', 'imnotyourfriend'),
            cls('ImperialEntanglements', 'imperialentanglements'),
            cls('Imperium', 'imperium'),
            cls('IMPERIVM', 'imperivmgalactica'),
            cls('Impisha', 'impisha'),
            cls('InBloodOfColour', 'inbloodofcolour'),
            cls('Indexmancave', 'indexmancave'),
            cls('InfraCityTheComic', 'infracity'),
            cls('InkLaRue', 'inkalarue'),
            cls('Inorganic', 'disturbingcomics'),
            cls('InsanityCorpV22', 'insanitycorp'),
            cls('Insectia', 'insectia'),
            cls('InsideOuT', 'insideout'),
            cls('InstantGraphicNovel', 'ign'),
            cls('IntergalacticTruckstop', 'its'),
            cls('InternetSuperbuddies', 'isb'),
            cls('Invicta', 'invicta'),
            cls('IsaacAndFriends', 'isaacandfriends'),
            cls('IslandOfTheMoths', 'moths'),
            cls('Isonacia', 'isonacia'),
            cls('ItsComplicated', 'itscomplicated'),
            cls('ItsJustAnotherDay', 'itsjustanotherday'),
            cls('ItsNEWDAY', 'itsnewday'),
            cls('Jack', 'jackrabbit', adult=True),
            cls('JackAndTheBeanstalk', 'jackandthebeanstalk'),
            cls('JackFrostDoujin', 'jfdoujin'),
            cls('JackitAndFriends', 'jackitandfriends'),
            cls('JakeBone', 'jakebone'),
            cls('JamieJupiter', 'jamiejupiter'),
            cls('JaquieNovemberAndTheSpookiness', 'november-spookiness'),
            cls('JaysInternetFightClub', 'jaysinternetfightclub'),
            cls('JellyfishStew', 'yppcomic'),
            cls('JenffersShowsFanArtPage', 'jenffersshowsfanartpage'),
            cls('JenffersShowsMissJAndJensPhotoAlbum', 'missjandjensphotoalbum'),
            cls('JenffersShowTheNewStoriesOfMissJAndJen', 'thenewstoriesofmissjandjen'),
            cls('Jericho', 'jericho'),
            cls('JillpokeBohemia', 'jillpokebohemia'),
            cls('Jix', 'jix'),
            cls('JohnnyBullet', 'johnnybullet'),
            cls('JonathinQuackupOfThePlanetWeralt', 'quackup'),
            cls('JoostsDailyDealings', 'joostdailies'),
            cls('JournalComics', 'jordansjournal'),
            cls('JourneyToRaifina', 'journeytoraifina'),
            cls('JudeAndMaria', 'judeandmaria'),
            cls('Jump', 'jump2'),
            cls('Junk', 'junk'),
            cls('Jupiter', 'jupiter'),
            cls('JustPeachy', 'justpeachy'),
            cls('KarensEdge', 'karensedge'),
            cls('Katastrophe', 'katastrophe'),
            cls('KATRAN', 'katran'),
            cls('KayAndP', 'kayandp'),
            cls('KazasMateGwenna', 'kaza-and-gwenna'),
            cls('KAZE', 'kaze'),
            cls('KeepingThePeace', 'keepingthepeace'),
            cls('KeepingUpWithThursday', 'keepingupwiththursday'),
            cls('KetsuekiDoku', 'ketsuekidoku'),
            cls('KevinWatch', 'kevinwatch'),
            cls('KevinWatchTheMovie', 'kevinwatchthemovie'),
            cls('KiasComic', 'kiascomic'),
            cls('KiasOTHERComic', 'kiasothercomic'),
            cls('KiLAILO', 'kilailo'),
            cls('KingdomOfTheDinosaurs', 'dinosaurkingdom'),
            cls('KingdomPrettyCure', 'kingdomprettycure'),
            cls('KirbyVsShyGuy', 'kvsg'),
            cls('KMLsSticks', 'kmlssticks'),
            cls('KnavesEnd', 'knavesend'),
            cls('KnightGuy', 'knightguy'),
            cls('Kordinar25000', 'kordinar'),
            cls('KougarStreetTheHumiliationOfLisaRumpson', 'kougarstreet'),
            cls('KronosWoWComics', 'kronoswowcomics'),
            cls('KyoniWanderer', 'kyoniwanderer'),
            cls('LaceyInvestigations', 'lacey-investigations'),
            cls('LadySpectraAndSparky', 'ladyspectra'),
            cls('Lambo', 'lambo'),
            cls('LandOfTheEverYoung', 'landoftheeveryoung'),
            cls('LaserBrigade', 'laserbrigade'),
            cls('LastCall', 'lastcallcomic'),
            cls('LastTaxi', 'lasttaxi'),
            cls('Latchkey', 'latchkey'),
            cls('LatchkeyKingdom', 'latchkeykingdom'),
            cls('Lately', 'lately'),
            cls('Lauras24HourComics', 'lauras24hourcomics'),
            cls('LazyComics', 'lazy'),
            cls('LeahClearwaterFancomic', 'leahclearwaterfancomic'),
            cls('LegendOfPaean', 'legend-of-paean'),
            cls('LegendOfTheRedPhantom', 'legendoftheredphantom'),
            cls('LegendOfZeldaOcarinaOfTim', 'ocarinaoftim'),
            cls('LethargicMisanthropy', 'lethargicmisanthropy'),
            cls('LetsCelebrate', 'letscelebrate'),
            cls('Level30Psychiatry', 'lvl30psy'),
            cls('LifeExplained', 'lifeexplained'),
            cls('LightBulbs', 'lightbulbs'),
            cls('LightningProphetess', 'lp'),
            cls('LilHeroArtists', 'lilheroartists'),
            # LimboRoad has a duplicate in ComicSherpa/LimboRoad
            cls('Lint', 'lint'),
            cls('Lintier', 'lintier'),
            cls('LiquidLunch', 'liquidlunch'),
            cls('LiteBites', 'litebites'),
            cls('LittleBlackDress', 'little-black-dress'),
            cls('LittleJacquie', 'littlejacquie'),
            cls('LittleRedRobo', 'littleredrobo'),
            cls('LivingInACloud', 'livinginacloud'),
            # Lola has a duplicate in GoComics/Lola
            cls('LongDistanceChargesApply', 'zacharybinks'),
            cls('Longhike', 'longhike'),
            cls('LookStraightAhead', 'lookstraightahead'),
            cls('LOSTLOVE', 'lostlove'),
            cls('LoveIsConplicated', 'conplicated'),
            cls('LoveKillsSlowly', 'lovekillsslowly'),
            cls('LOVETriologyExtraArt', 'mlextralove'),
            cls('LuckyHazard', 'luckyhazard'),
            cls('Lukewarm', 'lukewarm'),
            cls('LunaStar', 'lunastar'),
            cls('LustAndIre', 'lustandire', adult=True),
            cls('MadGirl', 'madgirl'),
            cls('MagicElDesencuentro', 'magiceldesencuentro', 'es'),
            cls('MagicTheScattering', 'magicthescattering'),
            cls('Magience', 'magience'),
            cls('MAGISAPARASAYOupdatesMonFri', 'mag-isa'),
            cls('MagnaComica', 'magnacomica'),
            cls('ManChildren', 'manchildren'),
            cls('MariosCastleTales', 'mariocastletales', 'it'),
            cls('MarriedToATransformersFan', 'marriedtoatransformersfan'),
            cls('MARS', 'mars'),
            cls('MaskOfTheAryans', 'mask-of-the-aryans'),
            cls('MassEffectMinarga', 'minarga'),
            cls('Mateys', 'mateys'),
            cls('MaxFuture', 'maxfuture'),
            cls('MAYBELOVE', 'emmacomics'),
            cls('MayonakaDensha', 'mayonakadensha'),
            cls('MayTheRainCome', 'maytheraincome', endOfLife=True),
            cls('MegaMaidenVSTheChopChopPrincess', 'megamaiden'),
            cls('MeganKearneysBeautyAndTheBeast', 'batb'),
            cls('MelancholyGoRound', 'melancholygoround'),
            cls('MerelyMortal', 'merelymortal'),
            cls('Messenger', 'messenger'),
            cls('MichaelTDesingsArmyAnts', 'armyants'),
            cls('MichellesUniverseScrapbook', 'michellesuniversescrapbook'),
            cls('MidnightMoon', 'midnightmoonrp'),
            cls('MidnightRUN', 'midnight-run'),
            cls('MIGHTYRACCOON', 'starraccoon'),
            cls('MildlyAmusing', 'mildlyamusing'),
            cls('Minecraft2b2tnet', 'minecraft2b2t'),
            cls('MiraclesOfNeksenziPoint', 'neksenzi-miracles'),
            cls('MirroredConversations', 'mirroredconversations'),
            cls('MiscellaneousMadness', 'rangerrandom'),
            cls('MissingDream', 'missingdream'),
            cls('MissionMars', 'missionmars'),
            cls('MithrilRavens', 'mithril-ravens'),
            cls('MiVidaSinUnJetpack', 'sinjetpack', 'es'),
            cls('MobiusAdventures', 'mobiusadventures'),
            cls('Mohyla', 'mohyla'),
            cls('Molasses', 'molasses'),
            cls('MondayMonday', 'mondaymonday'),
            cls('MonochromeRainbow', 'monobow'),
            cls('MonsterBait', 'deadnight'),
            cls('MonsterInTheKingdom', 'monster'),
            cls('MonstersWithBenefits', 'failmonsters'),
            cls('MonstroniverseAdventures', 'monstroniverse'),
            cls('MoonlitBrew', 'moonlitbrew'),
            cls('MoonWraith', 'moonwraith'),
            cls('MorningSquirtz', 'morningsquirtz'),
            cls('MotherOfAllMonsters', 'moam'),
            cls('MousebearComedy', 'mousebearcomedy'),
            cls('MrCow', 'mrcow'),
            cls('MrPunchAndProfRatbaggyEmeritus', 'punch'),
            cls('MudCompany', 'mudcompany'),
            cls('Mudskipper', 'mudskipper'),
            cls('Muscleheart', 'muscleheart'),
            cls('MushroomGo', 'mushroomgo'),
            cls('MutantElf', 'mutantelf'),
            cls('Mutigenx', 'mutigenx'),
            cls('MVPL', 'mvpl'),
            cls('MyForgottenPast', 'myforgottenpast'),
            cls('MyGirlfriendTheSecretAgent', 'mygfthesecagent'),
            cls('MyLifeWithoutAJetpack', 'nojetpack'),
            cls('MyLittlePonyFriendshipIsBetrayal', 'mlp-fib'),
            cls('MysteriousManOfSkull', 'mysteriousmanofskull'),
            cls('MyTVIsEvil', 'mytvisevil'),
            cls('NA', 'noche'),
            cls('NamcoWars', 'namcowars'),
            cls('NarutoJutsuAndJinchuriki', 'jutsuandjinchuriki'),
            cls('NatureDEEP', 'naturedeep'),
            cls('Necreshaw', 'nartopia'),
            cls('Neighbors', 'neighborscomic'),
            cls('NeverMindTheGap', 'nmg'),
            cls('Newheimburg', 'newheimburg'),
            cls('NEXGEN', 'nexgentheseries'),
            cls('NightmareNauts', 'nightmarenauts'),
            cls('NightshadeTheMerryWidow', 'lorddarke'),
            cls('NinthLife', 'ninthlife'),
            cls('Nocturne21', 'nocturne21'),
            cls('NoFuture', 'nofuturevit'),
            cls('NoKeys', 'nokeys'),
            cls('Noprrkele', 'noprrkele'),
            cls('NothingMen', 'nothing-men'),
            cls('NoTitleRequired', 'ntr'),
            cls('NotSinceYou', 'notsinceyou'),
            cls('NyxInTheOverworld', 'nyx'),
            cls('Oeight', 'oeight'),
            cls('OffCentaured', 'offcentaured'),
            cls('OfficeLogic', 'office-logic'),
            cls('OffSeason', 'offseasoncomic'),
            cls('OffWorldTheCrease', 'thecrease'),
            cls('OldFiyoraNya', 'retrofiyora'),
            cls('OldHumanCookies', 'oldhumancookies'),
            cls('OmegaChronicles', 'omegachronicles', 'es'),
            cls('OnceStung', 'oncestung'),
            cls('OnePageComicCollection', 'onepagecomiccollection'),
            cls('OnePieceGrandLine3Point5', 'grandline3point5'),
            cls('OneSided', 'one-sided'),
            cls('OrbFragmentSlim', 'orbfragment'),
            cls('OrganizedMess', 'organizedmess'),
            cls('Otherworldly', 'otherworldly-comics'),
            cls('OutFerASmoke', 'outferasmoke'),
            cls('Outletting', 'outletting'),
            cls('OutsideIn', 'outside-in'),
            cls('Palindrome', 'palindrome'),
            cls('PANAPANSTRAKOVI', 'strakovi'),
            cls('PaperStreamerAtDefCon5', 'paperstreamer'),
            cls('ParaFrenic', 'parafrenic'),
            cls('ParasiteGalaxy', 'parasitegalaxy'),
            cls('Parisel313', 'parisel313'),
            cls('PARKER', 'parker', segmented=True),
            cls('Parmeshen', 'parmeshen'),
            cls('ParoxysmTemporal', 'pt'),
            cls('PateEmpire', 'pateempire'),
            cls('PCMS20', 'pcms'),
            cls('PeeInTheMorningREBOOTED', 'holy-hecking-balls-rebooted', 'pt'),
            cls('PeepsAndPerks', 'peepsnperks'),
            cls('Pegwarmers', 'pegwarmers'),
            cls('PenguinCapers', 'penguin-capers'),
            cls('PerceivablyHuman', 'perceivablyhuman'),
            cls('PersonaForTheWin', 'personaftw'),
            cls('Perspectives', 'perspectives'),
            cls('PhantomsTrail', 'phantomstrail'),
            cls('Phoenix', 'phoenix'),
            cls('Pilgrim', 'pilgrimsprogress'),
            cls('PilgrimEnEspanol', 'pilgrimenespanol', 'es'),
            cls('PITCHBLACK', 'pitchblack'),
            cls('PlasticBulletsMayhemUnloaded', 'plasticbulletsmayhemunloaded'),
            cls('Poharex', 'poharex'),
            cls('PokemonWarpers', 'pokemonwarpers'),
            cls('PokmonOurStory', 'pokemonos'),
            cls('PokmonShadowStories', 'shadowstories'),
            cls('PoldaAPolda', 'poldove'),
            cls('PopCulturesKids', 'pop-cultures-kids'),
            cls('Powertrip', 'powertrip'),
            cls('POWRightInTheNostalgia', 'powrightinthenostalgia'),
            cls('PrimalWarsAftermath', 'primalwars'),
            cls('PrinceOfCats', 'princeofcats'),
            cls('ProfessorAstonishing', 'professorastonishing'),
            cls('ProfessorAmazingAndTheIncredibleGoldenFox', 'paigf'),
            cls('ProjectArc', 'projectarc'),
            cls('ProjectGTH', 'projectgth'),
            cls('ProjectJikoku', 'projectjikoku'),
            cls('ProjectSternenlicht', 'projectsternenlicht'),
            cls('PromiseList', 'promiselist'),
            cls('ProportionalExcitability', 'proportionalexcitability'),
            cls('Prosopopoeia', 'prosopopoeia'),
            cls('Pulse', 'pulse'),
            cls('PureHavoc', 'pure-havoc'),
            cls('Queenie', 'queenie'),
            cls('QuestCorporeal', 'questcorporeal'),
            cls('Rain', 'rain'),
            cls('RandomlyAssembled', 'randomlyassembled'),
            cls('RandomThoughts', 'randomthoughts'),
            cls('RapturousArcane', 'rapturousarcane'),
            cls('RawLatex', 'rawlatex'),
            cls('ReadershipOfOne', 'readershipofone'),
            cls('RebelYell', 'rebelyell'),
            cls('RebuildOfGenericMangaShippuden', 'rebuildofgenericmanga'),
            cls('RecklessComix', 'recklesscomix'),
            cls('RedSpot', 'redspot'),
            cls('RegardingDandelions', 'regardingdandelions'),
            cls('Remedy', 'remedy'),
            cls('RememberBedlam', 'bedlam'),
            cls('RequiemsGate', 'requiemsgate'),
            cls('ReSetArt', 'resetfanarts'),
            cls('ResidentWeirdo', 'residentweirdo'),
            cls('ReturnOfWonderland', 'returnofwonderland'),
            cls('Revive', 'revive'),
            cls('RexAfterDark', 'rexafterdark'),
            cls('RexfordAvenue', 'rexfordavenue'),
            # Ringers has a duplicate in ComicSherpa/Ringers
            cls('RockGardenComics', 'rockgardencomics'),
            cls('RoguesOfClwydRhan', 'rocr'),
            cls('RoleplayingPartyTales', 'rpt'),
            cls('RoomOfMirrors', 'room-of-mirrors'),
            cls('RootBeers', 'root-beers'),
            cls('Rozak', 'rozak'),
            cls('RPSLARPComic', 'rps'),
            cls('RumfAdventures', 'rumfadventures'),
            cls('RunningRiot', 'runningriot'),
            cls('SailorMoonTheEnemyNextDoor', 'sailormoontheenemynextdoor'),
            cls('Saluna', 'saluna'),
            cls('SanctaTerra', 'sanctaterra'),
            cls('SanityProtectionFactor', 'spf1337'),
            cls('SaraAndKleeyo', 'sarakleeyo'),
            cls('SaveMeGebus', 'savemegebus'),
            cls('SawbladersBlackNuzlockeChallenge', 'sawbladersblacknuzlocke'),
            cls('ScottieRoad', 'scottieroad'),
            cls('Scoundrels', 'scoundrels'),
            cls('ScrubDiving', 'scrubdiving'),
            cls('Scuvener', 'scuvener'),
            cls('SEAAOMSagaArchive', 'seaaom'),
            cls('SECRETLOVE', 'secretlove'),
            cls('SecretSanta2013', 'secretsanta2013'),
            cls('SeeYourFeels', 'seeyourfeels'),
            cls('SenatorSurprise', 'senatorsurprise'),
            cls('Sentiments', 'sentiments'),
            cls('SerengettiDreams', 'serengetti'),
            cls('SeriousEngineering', 'seriousengineering'),
            cls('SerpamiaFlare', 'serpamiaflare'),
            cls('SerpentsOfOld', 'serpentsofold'),
            cls('SerpentsOfOldFanArt', 'soofans'),
            cls('Shades', 'shades'),
            cls('ShadesOfGray', 'fuzzylittleninjas'),
            cls('ShaiAway', 'shaiaway'),
            cls('ShakingOffSorcery', 'shakingoffsorcery'),
            cls('ShakingOffSorceryPL', 'shakingoffsorcery-pl'),
            cls('ShamanQuest', 'shamanquest'),
            cls('ShatteredSkies', 'shatteredskies'),
            cls('Sharak', 'sharak'),
            cls('Shenanigans', 's'),
            cls('ShenaniganSquares', 'ss-comic'),
            cls('ShikuTheFirstAndFinal', 'shiku'),
            cls('ShiroAndKuro', 'shiroandkuro'),
            cls('ShutUpDiarybyBarbaraHolm', 'shutupdiary'),
            cls('Sigh', 'sigh'),
            cls('Silver', 'sil-ver'),
            cls('SilverNights', 'silvernights'),
            cls('Skeeter', 'herecomesskeeter'),
            cls('Sketchy', 'sketchy'),
            cls('Skylords', 'skylords'),
            cls('SlugMan', 'slug-man'),
            cls('SmallTownValues', 'smalltownvalues'),
            cls('SmitheeZombieHunter', 'smitheezombiehunter'),
            cls('SneakersUForce', 'sneakers'),
            cls('Snowfall', 'snowfall'),
            cls('SoFunnyIForgotToLaugh', 'sofunnyiforgottolaugh'),
            cls('SonichuREDone', 'sonichuredone'),
            cls('SonichuREDoneJ', 'sonichuredonejapanese', 'ja'),
            cls('Soulsworn', 'soulsworn'),
            cls('SpacedOutTheBeginning', 'spacedoutthebeginning'),
            cls('SpaceFarmer', 'spacefarmer'),
            cls('SpacePiratesOfTheBlackQuarter', 'spacepirates'),
            cls('SpacePulp', 'spacepulp'),
            cls('Spades', 'spades'),
            cls('SpicyDesu', 'desu'),
            cls('SpiderManShadowsOfNight', 'shadowsofnight'),
            cls('SpiritSquireTheQuestForTheUltimateKnight', 'spiritsquire-1'),
            cls('Spooky', 'spooky'),
            cls('SPOON', 'spooncomic'),
            cls('StampedeJessicasStory', 'stampedegirl'),
            cls('Starcrossed', 'starcrossed'),
            cls('StarPunchGirl', 'starpunchgirl'),
            cls('STARWARSXWingAlliance', 'x-wingalliance'),
            cls('STASonicTheAdventure', 'sta'),
            cls('StereotyPixs', 'stereotypixs'),
            cls('StevenAndTheCrystalGMs', 'crystalgms'),
            cls('StickLife', 'sticklife'),
            cls('StickMisadventures', 'stick-misadventures'),
            cls('StinkomanFatChickenQuest', 'stinkoman'),
            cls('StonedHeroes', 'stonedheroes'),
            cls('StrangeAttractors', 'strangeattractors', segmented=True),
            cls('Streamo', 'streamo'),
            cls('SundaySmash', 'sundaysmash'),
            cls('Sunray', 'sunray'),
            cls('SuperGalaxyKnightsDeluxeR', 'sgkdr'),
            cls('SuperheroTales', 'superherobeingsuper'),
            cls('SuperShashi', 'supershashi'),
            cls('Supervillainous', 'supervillainous'),
            cls('SurrealScience', 'surrealscience'),
            cls('Swashbuckled', 'swashbuckled'),
            cls('Swazzyknocks', 'swazzyknocks'),
            cls('Synapticisms', 'synapticisms'),
            cls('TalesFromRiota', 'ganold'),
            cls('TalesOfBrickland', 'brickland'),
            cls('TalesOfMiddar', 'talesofmiddar'),
            cls('TalesOfSpoons', 'talesofspoons'),
            cls('TalesOfTheGalli', 'totg-mirror'),
            cls('TamTeamAdventures', 'tamteam'),
            cls('TangledMessTheGirlyNerdyTerriblyStrangeJournalComi', 'tangledmess'),
            cls('TangledRiver', 'tangled-river', segmented=True),
            cls('TBA', 'tba'),
            cls('TBAold', 'tba-old'),
            cls('TerwilligersCafe', 'terwilligers'),
            cls('TheAccidentalSpaceSpy', 'spacespy'),
            cls('TheAccidentalWitch', 'theaccidentalwitch'),
            cls('TheAdventuresOfAquilaAndTeren', 'aquilateren'),
            cls('TheAdventuresOfBaldy', 'adventuresofbaldy'),
            cls('TheAdventuresOfBidoof', 'bidoof'),
            cls('TheAdventuresOfCarrotKnight', 'carrotknight'),
            cls('TheAdventuresOfGrumpyBearAndMrGoose', 'grumpyandgoose'),
            cls('TheAdventuresOfMechaSmiles', 'mechasmiles'),
            cls('TheAdventuresOfSherilynAndEmma', 'taosae'),
            cls('TheAdventuresOfTheLadySkylark', 'ladyskylark'),
            cls('TheBarrowHill', 'thebarrowhill'),
            cls('TheBellInTheOcean', 'bellintheocean'),
            cls('TheBend', 'thebend'),
            cls('TheBigFoldy', 'bigfoldy'),
            cls('THEBIGSCIFIMISHMASH', 'thebigsci-fimish-mash'),
            cls('TheBlackPrincess', 'theblackprincess'),
            cls('THEBOOKOFLIES', 'bookofliescomic'),
            cls('TheBookOfThree', 'thebookofthree', segmented=True),
            cls('TheChanterelleAndMayLife', 'chanterelleandmay'),
            cls('TheChroniclesOfBuckyONeill', 'buckyoneill'),
            cls('TheChroniclesOfDrew', 'thechroniclesofdrew'),
            cls('TheChroniclesOfLillian', 'chroniclesoflillian'),
            cls('TheChroniclesOfLoth', 'chroniclesofloth'),
            cls('TheCompozerz', 'compozerz'),
            cls('TheContinentals', 'continentals'),
            cls('TheCrepusculars', 'crepusculars'),
            cls('TheCrumpletonExperiments', 'thecrumpletonexperiments'),
            cls('TheDailyDoodle', 'tdd'),
            cls('TheDevilsHorn', 'thedevilshorn'),
            cls('TheDragonFistsOfSmortySmythe', 'thedragonfistsofsmortysmythe'),
            cls('TheDrongos', 'thedrongos'),
            cls('TheEpicEpic', 'theepicepic'),
            cls('TheFaithful', 'thefaithful'),
            cls('TheFeloranChronicles', 'felora'),
            cls('TheFunnyZone', 'thefunnyzone'),
            cls('TheGalleryOfFreaks', 'galleryoffreaks'),
            cls('TheGarage', 'thegarage'),
            cls('TheGarden', 'thegarden'),
            cls('TheGingerbreadManChronicles', 'gingerbreadmanchronicles'),
            cls('TheGrazingMongrel', 'grazingmongrel'),
            cls('TheGuardian', 'theguardian'),
            cls('TheHarriopulate', 'theharriopulate'),
            cls('TheHighestBet', 'thehighestbet'),
            cls('TheHighestBetITA', 'thehighestbet-ita', 'it'),
            cls('TheHobbit', 'hobbit'),
            cls('TheHolidayDoctor', 'holidaydoctor'),
            cls('TheHorrifyingExperimentsOfDrPleasant', 'thehorrifyingexperimentsofdrpleasant'),
            cls('TheHub', 'cbbrthehub'),
            cls('TheHubBook', 'thehubbook'),
            cls('TheHumanBattery', 'thehumanbattery'),
            cls('TheHundredsUprising', 'thehundredsuprising'),
            cls('TheILL', 'theill'),
            cls('TheIntrovertManifesto', 'introvert'),
            cls('TheJabbercrow', 'jabbercrow'),
            cls('TheKeepOnTheBorderlands', 'thekeepontheborderlands'),
            cls('TheLamp', 'thelamp'),
            cls('TheLastHope', 'tlhcomic'),
            cls('TheLawOfPurple', 'lawofpurple'),
            cls('TheLeagueOfExtraordinaryRoleplayers', 'lxgrpg'),
            cls('TheLeapfrogTeam', 'leapfrogteam'),
            cls('TheLegendaryPixelCrew', 'thelegendarypixelcrew'),
            cls('TheLegendOfLink', 'legendoflink'),
            cls('TheLozoyas', 'thelozoyas'),
            cls('TheMansionofE', 'mansionofe'),
            cls('TheMates', 'themates'),
            cls('TheMatesPortugus', 'matespt', 'pt'),
            cls('TheMeaningOfLife', 'themeaningoflife'),
            cls('TheMightyBlue', 'themightyblue'),
            cls('TheMightyMeteorite', 'mightymeteorite'),
            cls('TheMisadventuresOfDexterTheAlien', 'dexterthealien'),
            cls('TheMisadventuresOfTheTrailerParkTrio', 'tmaottpt'),
            cls('TheMitchellEffect', 'themitchelleffect'),
            cls('TheMoonValley', 'moonvalley'),
            cls('TheNew30DaysOfCharacters', '30l30characters'),
            cls('TheNewAdventuresOfFelicity', 'felicity'),
            cls('TheNineteenthCenturyIndustrialist', 'thebaron'),
            cls('TheNonesuchTales', 'thenonesuchtales'),
            cls('TheORIGINALShonenPunk', 'shonenpunk'),
            cls('TheOtherGreyMeat', 'togm'),
            cls('TheOverture', 'theoverture'),
            cls('ThePresident', 'president'),
            cls('TheQuantumKid', 'thequantumkid'),
            cls('TheQuestForCoitus', 'acomicstudios'),
            cls('TheRathNexus', 'rath'),
            cls('TheRealmOfKaerwyn', 'kaerwyn'),
            cls('TheRebels', 'rebels'),
            cls('TheRedeemers', 'theredeemers'),
            cls('TheRestlessDead', 'therestlessdead'),
            cls('TheRidiculousPushyReeder', 'pushy'),
            cls('TheRoseKiller', 'therosekiller'),
            cls('TheRubyNation', 'rubynation'),
            cls('TheScienceOfCookies', 'cookiescience', 'fr'),
            cls('TheSecondCrimeanWar', 'secondcrimeanwar'),
            cls('TheSettlers', 'thesettlers'),
            cls('TheSketchyAdventuresOfKyoAndMatt', 'kyoandmatt'),
            cls('TheSkybox', 'skybox'),
            cls('TheSpecialCASE', 'thespecialcase'),
            cls('THESTORMRUNNERS', 'thestormrunners'),
            cls('TheStoryOfBobChapter1Part1', 'thebobstory'),
            cls('TheStoryOfSaliria', 'saliria'),
            cls('TheSupernaturalsEpisode4', 'thesupernaturals4'),
            cls('TheSurface', 'thesurface'),
            cls('TheTempleAtFiftyFathoms', 'the-temple-at-fifty-fathoms'),
            cls('TheTenTailorsOfWestonCourt', 'tentailors'),
            cls('TheTrialsOfMannack', 'mannack'),
            cls('TheUnclean', 'theunclean'),
            cls('TheWayOfTheMetagamer', 'wayofthemetagamer'),
            cls('TheWellkeeper', 'thewellkeeper'),
            cls('TheWesternGang', 'thewesterngang'),
            cls('TheWhizzkids', 'whizzkids'),
            cls('TheWolfAtWestonCourt', 'thewolfatwestoncourt'),
            cls('TheWorldJumper', 'theworldjumper'),
            cls('TheWorldOfUh', 'theworldofuh'),
            cls('TheWrongTree', 'thewrongtree'),
            cls('TheWWord', 'thewword'),
            cls('ThisIsNormal', 'thisisnormal'),
            cls('ThisIsTheLife', 'thisisthelifecomic'),
            cls('ThomasAndZacharyArchives', 'thomasandzachary'),
            cls('Thornwar', 'thornwar'),
            cls('ThoseUnknowableTheShadowsOverInnsmouth', 'tsoi'),
            cls('Threan', 'threan'),
            cls('ThreeFreeFrikis', 'tff', 'es'),
            cls('TickTock', 'tick-tock'),
            cls('TigerWrestling', 'anybodythere'),
            cls('Timezone', 'timezone'),
            cls('Tinytown', 'tinytown'),
            cls('TM47', 'tm47'),
            cls('TohvelinTuhinoita', 'tuhinaloota'),
            cls('TOLVA', 'tolva'),
            cls('TombOfTheKing', 'tomboftheking'),
            cls('TomorrowsGirls', 'tomorrowsgirls'),
            cls('ToneOutComics', 'toneout'),
            cls('TonyComics', 'tonycomics'),
            cls('Toontown', 'toontowncomics'),
            cls('TotalChaos', 'totalchaos'),
            cls('TotallyKaimera', 'totallykaimera'),
            cls('TotallyKaimeraBackgroundStory2', 'totallykaimerabackgroundstory2'),
            cls('TotallyKaimeraPart2', 'totallykaimerapart2'),
            cls('TotallyKaimeraPart3', 'totallykaimerapart3'),
            cls('TracyAndTristan', 'tandt'),
            cls('TradScribbles', 'tradscribbles'),
            cls('TrAgEdY', 'tragedy'),
            cls('TransdimensionalBrainChip', 'brainchip'),
            cls('TransformersNexus', 'tfnexus'),
            cls('TransientPulseNotIntentionallyObsessive', 'niotp'),
            cls('Transmission', 'transmission'),
            cls('TransUmanSUbterran', 'sub-terran'),
            cls('Traveler', 'clioyorokobi'),
            cls('TreeScratches', 'treescratches'),
            cls('Treeville', 'treeville'),
            cls('TriforceOfPower', 'triforceofpower'),
            cls('Trigonometry', 'trigonometry'),
            cls('Trinity', 'trinity'),
            cls('TrixieSlaughteraxeForPresident', 'trixie'),
            cls('TrollGirl', 'trollgirl'),
            cls('TrueFist', 'true-fist'),
            cls('TruFax', 'trufax'),
            cls('TSAndTJ', 'tsandtj'),
            cls('TsuyuSociety', 'tsuyusociety'),
            cls('TurnerAndHercules', 'turnerandhercules'),
            cls('TussenKatersEnSpraakwater', 'tussenkatersenspraakwater'),
            cls('TvQuest', 'tvquest'),
            cls('TwilightTrust', 'twilighttrust'),
            cls('TwinsAgony', 'twinsagony'),
            cls('TwistedPeel', 'twistedpeel'),
            cls('TwoFaced', 'twofaced'),
            cls('TwoHearts', 'twohearts'),
            cls('TWTWE', 'twtwe'),
            cls('TypicalStrange', 'typicalstrange'),
            cls('UglyBookCovers', 'uglybookcovers'),
            cls('UltimateSwordsSummoner', 'uss'),
            cls('UltraViresEnglish', 'ultravires-eng'),
            cls('UltraViresesky', 'ultravires'),
            cls('Unconventional', 'unconventional', adult=True),
            cls('Underverse', 'underverse'),
            cls('UnfortunateCircumstances', 'unfortunatecircumstances'),
            cls('UniversityOfSpeed', 'u-speed'),
            cls('UnknownLands', 'unknownlands'),
            cls('UNPROFESSIONAL', 'unprofessional'),
            cls('Unreliable', 'unreliable'),
            cls('V4', 'v4'),
            cls('ValeOfDemons', 'valeofdemons'),
            cls('VampireBites', 'vampirebites'),
            cls('VampireCatgirlPart2', 'vampirecatgirl2'),
            cls('VeldaGirlDetective', 'veldagirldetective'),
            cls('Verboten', 'verboten'),
            cls('VHV', 'vhv'),
            cls('Victory', 'victoryadventures'),
            cls('ViewHub', 'viewhub'),
            cls('ViolentBlue', 'violentblue'),
            cls('Virtrena', 'virtrena'),
            cls('VisualDiaryOfMyLife', 'visualdiary'),
            cls('VOE', 'voe'),
            cls('Voidchild', 'voidchild'),
            cls('WaitWhat', 'waitwhatcomic'),
            cls('WARG', 'warg'),
            cls('Wargyrl', 'wargyrl'),
            cls('WarriorTwentySeven', 'warrior27'),
            cls('WastedAway', 'wastedaway'),
            cls('WastedPotential', 'wastedpotential'),
            cls('WastelandersAnonymous', 'wastelanders'),
            cls('WasteOfTime', 'wasteoftime'),
            cls('WayTooOffensive', 'waytooffensive'),
            cls('WeAreTheLosers', 'thelosers'),
            cls('WeeabooIsland', 'weeabooisland'),
            cls('WestTreeAcademyOfHeroes', 'westtree'),
            cls('WhatIDontEven', 'idonteven'),
            cls('WHATSERP', 'whatserp'),
            cls('WhiskeyAndMelancholy', 'whiskeyandmelancholy'),
            cls('WhiteOut', 'whiteout'),
            cls('WhiteSpace', 'whitespace'),
            cls('WhoseLineIsItAnyhoo', 'whoseline'),
            cls('WilfordTheWalrus', 'wilfordthewalrus'),
            cls('Willem', 'willem'),
            cls('WindRiders', 'windriders'),
            cls('WinstonsWorld', 'winstonsworld'),
            cls('WitchesTeaParty', 'witchesteaparty'),
            cls('WithoutMoonlight', 'withoutmoonlight'),
            cls('WonderTeam', 'wonderteam'),
            cls('WoodsForTheTrees', 'woodsforthetrees'),
            cls('WoodsOfEvil', 'woodsofevil'),
            cls('Woohooligan', 'woohooligan'),
            cls('WordsToLiveBy', 'wordstoliveby'),
            cls('WORMCURSE', 'wormcurse'),
            cls('WrightAsRayne', 'wrightasrayne'),
            cls('WrongNumber', 'wrongnumber'),
            cls('WYIHN', 'wyihn'),
            cls('Xibalba', 'xibalba'),
            cls('Xit', 'x-it'),
            cls('YesterdayBound', 'yesterdaybound'),
            cls('YouAreNow', 'yan'),
            cls('YOURCHOICE', 'yourchoice'),
            cls('ZackDragonbladeAndTheExcalites', 'zackdragonblade'),
            cls('ZebraGirl', 'zebragirl'),
            cls('Zelfia', 'zelfia'),
            cls('ZeroEffortFantasy', 'zeroeffort'),
            cls('ZombieZoup', 'zombiezoup'),
            cls('ZwergElf', 'zwergelf', 'de'),
            # END AUTOUPDATE
        )
