
/******************************************************************************
 *
 *  This file is part of canu, a software program that assembles whole-genome
 *  sequencing reads into contigs.
 *
 *  This software is based on:
 *    'Celera Assembler' (http://wgs-assembler.sourceforge.net)
 *    the 'kmer package' (http://kmer.sourceforge.net)
 *  both originally distributed by Applera Corporation under the GNU General
 *  Public License, version 2.
 *
 *  Canu branched from Celera Assembler at its revision 4587.
 *  Canu branched from the kmer project at its revision 1994.
 *
 *  This file is derived from:
 *
 *    src/AS_OVS/overlapStoreBuild.C
 *
 *  Modifications by:
 *
 *    Brian P. Walenz from 2012-APR-02 to 2013-AUG-01
 *      are Copyright 2012-2013 J. Craig Venter Institute, and
 *      are subject to the GNU General Public License version 2
 *
 *    Brian P. Walenz from 2014-JUL-31 to 2015-SEP-21
 *      are Copyright 2014-2015 Battelle National Biodefense Institute, and
 *      are subject to the BSD 3-Clause License
 *
 *    Brian P. Walenz beginning on 2015-NOV-03
 *      are a 'United States Government Work', and
 *      are released in the public domain
 *
 *    Sergey Koren beginning on 2016-FEB-20
 *      are a 'United States Government Work', and
 *      are released in the public domain
 *
 *  File 'README.licenses' in the root directory of this distribution contains
 *  full conditions and disclaimers for each license.
 */

#include "AS_global.H"

#include "sqStore.H"
#include "ovStore.H"
#include "ovStoreConfig.H"

#include <vector>
#include <algorithm>

using namespace std;


static
void
writeToDumpFile(sqStore          *seq,
                ovOverlap       *overlap,
                ovFile          **dumpFile,
                uint64           *dumpLength,
                uint32           *iidToBucket,
                char             *ovlName) {
  uint32 df = iidToBucket[overlap->a_iid];

  //  If the dump file isn't opened, open it.

  if (dumpFile[df] == NULL) {
    char name[FILENAME_MAX];
    snprintf(name, FILENAME_MAX, "%s/tmp.sort.%04d", ovlName, df);
    fprintf(stderr, "-- Create bucket '%s'\n", name);
    dumpFile[df]   = new ovFile(seq, name, ovFileFullWriteNoCounts);
    dumpLength[df] = 0;
  }

  //  And write the overlap.

  dumpFile[df]->writeOverlap(overlap);
  dumpLength[df]++;
}



int
main(int argc, char **argv) {
  char           *ovlName        = NULL;
  char           *seqName        = NULL;
  char           *cfgName        = NULL;

  double          maxErrorRate   = 1.0;

  bool            eValues        = false;
  char           *configOut      = NULL;

  bool            beVerbose      = false;

  argc = AS_configure(argc, argv);

  vector<char *>  err;
  int             arg=1;
  while (arg < argc) {
    if        (strcmp(argv[arg], "-O") == 0) {
      ovlName = argv[++arg];

    } else if (strcmp(argv[arg], "-S") == 0) {
      seqName = argv[++arg];

    } else if (strcmp(argv[arg], "-C") == 0) {
      cfgName = argv[++arg];

    } else if (strcmp(argv[arg], "-e") == 0) {
      maxErrorRate = atof(argv[++arg]);

    } else if (strcmp(argv[arg], "-v") == 0) {
      beVerbose = true;

    } else {
      char *s = new char [1024];
      snprintf(s, 1024, "%s: unknown option '%s'.\n", argv[0], argv[arg]);
      err.push_back(s);
    }

    arg++;
  }

  if (ovlName == NULL)
    err.push_back("ERROR: No overlap store (-O) supplied.\n");

  if (seqName == NULL)
    err.push_back("ERROR: No sequence store (-S) supplied.\n");

  if (err.size() > 0) {
    fprintf(stderr, "usage: %s -O asm.ovlStore -S asm.seqStore -C ovStoreConfig [opts]\n", argv[0]);
    fprintf(stderr, "  -O asm.ovlStore       path to overlap store to create\n");
    fprintf(stderr, "  -S asm.seqStore       path to a sequence store\n");
    fprintf(stderr, "  -C config             path to ovStoreConfig configuration file\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "  -e e                  filter overlaps above e fraction error\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "  -v                    be overly verbose\n");
    fprintf(stderr, "\n");

    for (uint32 ii=0; ii<err.size(); ii++)
      if (err[ii])
        fputs(err[ii], stderr);

    exit(1);
  }

  //  Load the config, open the store, create a filter.

  ovStoreConfig    *config = new ovStoreConfig(cfgName);
  sqStore          *seq    = sqStore::sqStore_open(seqName);
  ovStoreFilter    *filter = new ovStoreFilter(seq, maxErrorRate, beVerbose);

  //  Figure out how many overlaps there are, quit if too many.

  uint32  maxID       = seq->sqStore_getNumReads();
  uint64  ovlsTotal   = 0;  //  Total in inputs.
  uint32  numInputs   = 0;

  fprintf(stderr, "\n");
  fprintf(stderr, "-- SCANNING INPUTS --\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "   Moverlaps\n");
  fprintf(stderr, "------------ ----------------------------------------\n");

  for (uint32 bb=1; bb<=config->numBuckets(); bb++) {
    for (uint32 ii=0; ii<config->numInputs(bb); ii++) {
      char              *inputName = config->getInput(bb, ii);
      ovFile            *inputFile = new ovFile(seq, inputName, ovFileFull);

      ovlsTotal += inputFile->getCounts()->numOverlaps() * 2;
      numInputs += 1;

      fprintf(stderr, "%12.3f %40s\n",
              inputFile->getCounts()->numOverlaps() / 1000000.0,
              inputName);

      delete inputFile;
    }
  }

  fprintf(stderr, "------------ ----------------------------------------\n");
  fprintf(stderr, "%12.3f Moverlaps in inputs\n", ovlsTotal / 2 / 1000000.0);
  fprintf(stderr, "%12.3f Moverlaps to sort\n",   ovlsTotal     / 1000000.0);
  fprintf(stderr, "\n");

  if (ovlsTotal == 0)
    fprintf(stderr, "Found no overlaps to sort.\n");

  //  Load overlaps into memory.

  fprintf(stderr, "\n");
  fprintf(stderr, "Allocating space for " F_U64 " overlaps.\n", ovlsTotal);
  fprintf(stderr, "\n");

  ovOverlap      *ovls       = new ovOverlap [ovlsTotal];
  uint64          ovlsInput  = 0;
  uint64          ovlsLoaded = 0;

  fprintf(stderr, "\n");
  fprintf(stderr, "-- LOADING OVERLAPS --\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "       Input       Loaded  Percent  Percent\n");
  fprintf(stderr, "   Moverlaps    Moverlaps   Loaded Complete\n");
  fprintf(stderr, "------------ ------------ -------- -------- ----------------------------------------\n");

  for (uint32 bb=1; bb<=config->numBuckets(); bb++) {
    for (uint32 ii=0; ii<config->numInputs(bb); ii++) {
      char     *inputName = config->getInput(bb, ii);

      fprintf(stderr, "%12.3f %12.3f %7.2f%% %7.2f%% %40s\n",
              ovlsInput   / 1000000.0,
              ovlsLoaded  / 1000000.0,
              100.0 * ovlsInput   / ovlsTotal,
              (ovlsInput == 0) ? (100.0) : (100.0 * ovlsLoaded / ovlsInput),
              inputName);

      ovOverlap foverlap;
      ovOverlap roverlap;

      ovFile   *inputFile = new ovFile(seq, inputName, ovFileFull);

      while (inputFile->readOverlap(&foverlap)) {
        filter->filterOverlap(foverlap, roverlap);  //  The filter copies f into r, and checks IDs

        ovlsInput += 2;

        //  Write the overlap if anything requests it.  These can be non-symmetric; e.g., if
        //  we only want to trim reads 1-1000, we'll not output any overlaps for a_iid > 1000.

        if ((foverlap.dat.ovl.forUTG == true) ||
            (foverlap.dat.ovl.forOBT == true) ||
            (foverlap.dat.ovl.forDUP == true))
          ovls[ovlsLoaded++] = foverlap;

        if ((roverlap.dat.ovl.forUTG == true) ||
            (roverlap.dat.ovl.forOBT == true) ||
            (roverlap.dat.ovl.forDUP == true))
          ovls[ovlsLoaded++] = roverlap;

        //  Report every 15.5 million overlaps (it's the millionth prime, why not).

        if ((ovlsLoaded % 15485863) == 0)
          fprintf(stderr, "%12.3f %12.3f %7.2f%% %7.2f%%\n",
                  ovlsInput   / 1000000.0,
                  ovlsLoaded  / 1000000.0,
                  100.0 * ovlsInput   / ovlsTotal,
                  (ovlsInput == 0) ? (100.0) : (100.0 * ovlsLoaded / ovlsInput));

        //  Make sure we didn't blow our space.

        assert(ovlsLoaded <= ovlsTotal);
      }

      delete inputFile;
    }
  }

  fprintf(stderr, "------------ ------------ -------- -------- ----------------------------------------\n");
  fprintf(stderr, "%12.3f %12.3f %7.2f%% %7.2f%%\n",
          ovlsInput   / 1000000.0,
          ovlsLoaded  / 1000000.0,
          100.0 * ovlsInput   / ovlsTotal,
          (ovlsInput == 0) ? (100.0) : (100.0 * ovlsLoaded / ovlsInput));

  //  Report what was filtered and loaded.

  fprintf(stderr, "\n");
  fprintf(stderr, "-- OVERLAP FILTERING --\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "DEDUPE OVERLAPS\n");
  fprintf(stderr, "Saved      " F_U64 " dedupe overlaps\n",          filter->savedDedupe());
  fprintf(stderr, "Discarded  " F_U64 " don't care\n",               filter->filteredNoDedupe());
  fprintf(stderr, "Discarded  " F_U64 " different library\n",        filter->filteredNotDupe());
  fprintf(stderr, "Discarded  " F_U64 " obviously not duplicates\n", filter->filteredDiffLib());
  fprintf(stderr, "\n");
  fprintf(stderr, "TRIMMING OVERLAPS\n");
  fprintf(stderr, "Saved      " F_U64 " trimming overlaps\n", filter->savedTrimming());
  fprintf(stderr, "Discarded  " F_U64 " don't care\n",        filter->filteredNoTrim());
  fprintf(stderr, "Discarded  " F_U64 " too similar\n",       filter->filteredBadTrim());
  fprintf(stderr, "Discarded  " F_U64 " too short\n",         filter->filteredShortTrim());
  fprintf(stderr, "\n");
  fprintf(stderr, "UNITIGGING OVERLAPS\n");
  fprintf(stderr, "Saved      " F_U64 " unitigging overlaps\n", filter->savedUnitigging());
  fprintf(stderr, "\n");
  fprintf(stderr, "Discarded  " F_U64 " low quality, more than %.4f fraction error\n", filter->filteredErate(), maxErrorRate);
  fprintf(stderr, "Discarded  " F_U64 " opposite orientation\n", filter->filteredFlipped());
  fprintf(stderr, "\n");

  delete filter;

  //  Sort the assorted overlaps.

  fprintf(stderr, "\n");
  fprintf(stderr, "-- SORT OVERLAPS --\n");
  fprintf(stderr, "\n");

#ifdef _GLIBCXX_PARALLEL
  //  If we have the parallel STL, don't use it!  Sort is not inplace!
  __gnu_sequential::
#endif
  sort(ovls, ovls + ovlsLoaded);

  //  Write.

  fprintf(stderr, "\n");
  fprintf(stderr, "-- OUTPUT OVERLAPS --\n");
  fprintf(stderr, "\n");

  ovStoreWriter  *store = new ovStoreWriter(ovlName, seq);

  for (uint64 oo=0; oo<ovlsLoaded; oo++)
    store->writeOverlap(ovls + oo);

  delete    store;
  delete [] ovls;

  seq->sqStore_close();

  //  And we have a store.

  fprintf(stderr, "Bye.\n");

  exit(0);
}
