# Copyright (C) 2015 Red Hat, Inc. Neependra Khare <nkhare@redhat.com>
# Copyright (C) 2015 Red Hat, Inc. Bryn M. Reeves <bmr@redhat.com>

# This file is part of the sos project: https://github.com/sosreport/sos
#
# This copyrighted material is made available to anyone wishing to use,
# modify, copy, or redistribute it subject to the terms and conditions of
# version 2 of the GNU General Public License.
#
# See the LICENSE file in the source distribution for further information.

from sos.report.plugins import Plugin, RedHatPlugin


class etcd(Plugin, RedHatPlugin):

    short_desc = 'etcd plugin'

    plugin_name = 'etcd'
    packages = ('etcd',)
    profiles = ('container', 'system', 'services', 'cluster')
    files = ('/etc/origin/node/pods/etcd.yaml',)
    services = ('etcd',)

    def setup(self):
        if self.path_exists('/etc/origin/node/pods/etcd.yaml'):
            etcd_cmd = 'master-exec etcd etcd etcdctl'
        else:
            etcd_cmd = 'etcdctl'

        self.add_file_tags({
            '/etc/etcd/etcd.conf': 'etcd_conf'
        })

        etcd_url = self.get_etcd_url()

        self.add_forbidden_path([
            '/etc/etcd/ca',
            '/etc/etcd/*.key'
        ])
        self.add_copy_spec('/etc/etcd')

        subcmds = [
           '--version',
           'member list',
           'cluster-health',
           'ls --recursive',
        ]

        self.add_cmd_output(['%s %s' % (etcd_cmd, sub) for sub in subcmds])

        urls = [
            '/v2/stats/leader',
            '/v2/stats/self',
            '/v2/stats/store',
        ]

        if etcd_url:
            self.add_cmd_output(['curl -s %s%s' % (etcd_url, u) for u in urls])

        self.add_cmd_output("ls -lR /var/lib/etcd/")

    def get_etcd_url(self):
        try:
            with open(self.path_join('/etc/etcd/etcd.conf'), 'r') as ef:
                for line in ef:
                    if line.startswith('ETCD_LISTEN_CLIENT_URLS'):
                        return line.split('=')[1].replace('"', '').strip()
        # If we can't read etcd.conf, assume defaults by etcd version
        except IOError:
            # assume v3 is the default
            url = 'http://localhost:2379'
            try:
                ver = self.policy.package_manager.packages['etcd']
                ver = ver['version'][0]
                if ver == '2':
                    url = 'http://localhost:4001'
            except Exception:
                # fallback when etcd is not installed
                pass
            return url

# vim: et ts=5 sw=4
