/*
 * apstat-like print routines
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "common.h"
#include <time.h>
#include <pwd.h>

/**
 * print_hex_placement_list  -  print all placement entries, apstat-style
 * @buf_start: start of appinfo buffer area
 * @ai_info:   current appinfo entry
 */
void print_hex_placement_list(const uint8_t *buf_start, appInfo_t *ai_info)
{
	placeList_t *entry = (placeList_t *)(buf_start + ai_info->places);
	int j;
#if CLE > 2
	int idx = -1, nid = -1;
	uint32_t mask = 0;

	for (j = 0; j < ai_info->numPlaces; j++, entry++) {
 		/*
		 * Each placement entry corresponds to one core (CPU) of a node.
		 * If all have the same bitmask of all 1's, and if no bit is set
		 * in any of the bitmasks, the node has not been claimed.
		 */
		if (entry->cmdIx != idx || entry->nid != nid) {
			printf("    cmd %u, nid %u, CPU %#06x, map %#x\n",
					entry->cmdIx,
					entry->nid,
					(unsigned)entry->procMask,
					entry->nodeMap);
			idx   = entry->cmdIx;
			nid   = entry->nid;
			mask  = entry->procMask;
		}
	}
#else
	for (j = 0; j < ai_info->numPlaces; j++, entry++)
		printf("    PE %d, cmd %u, nid %u, CPU %#x\n", j, entry->cmdIx,
			entry->nid, (unsigned)entry->procMask);
#endif
}

/**
 * print_cmd_detail  -  print details of individual appinfo entry
 * @buf_start: start of appinfo buffer area
 * @ai_info:   current appinfo entry
 */
void print_cmd_detail(const uint8_t *buf_start, appInfo_t *ai_info)
{
	cmdDetail_t *det = (cmdDetail_t *)(buf_start + ai_info->cmdDetail);
	int j;
	/*
	 *--------------------------------------------------------------
	 * cmdDetail_t
	 *--------------------------------------------------------------
	 * int   flags		- command-specific flags (often 0)
	 * int   width		- number of PEs for this command
	 * int   depth		- processors per PE
	 * int   fixedPerNode	- user set per-node PE count
	 * int   memory		- per PE memory limit in megabytes
	 * alps_archType_t arch	- architecture type
	 * char  cmd[32]	- a.out name ("BASIL" for reservation)
	 *
	 * int   nodeCnt	- number of nodes allocated
	 *
	 * uint16_t pesPerSeg	-S value
	 * uint16_t nodeSegCnt	-sn value
	 * uint32_t segBits	-sl <0-3> - each bit is a segment number
	 *
	 * uint16_t rvalue	-r value - 0 or 1
	 */
	for (j = 0; j < ai_info->numCmds; j++, det++) {
		printf("  Cmd[%d]: %-.13s -n %d", j,
			ai_info->timePlaced ? det->cmd : "BASIL", det->width);
#if CLE <= 2
		printf(" -d %u -N %u -S %u -sn %u -sl %u",
			det->depth, det->fixedPerNode,
			det->pesPerSeg, det->nodeSegCnt,
			det->segBits);
#else
		if (det->depth != 1)
			printf(" -d %u", det->depth);
		if (det->fixedPerNode)
			printf(" -N %u", det->fixedPerNode);
		if (det->flags & 0x800)
			printf(" -ss");
#endif
		printf(", %dMB, %s, nodes %d\n", det->memory,
			appinfo_arch(det->arch), det->nodeCnt);
	}
}

/**
 * print_resinfo  -  print single appinfo entry
 * @ai_buf:   start of appinfo buffer
 * @ai_info:  entry to print
 * @inv:      recent ALPS Inventory
 * @num:      number of @ai_info
 * @verbose:  verbosity level
 */
static void print_resinfo(const uint8_t *ai_buf, appInfo_t *ai_info,
			  struct basil_inventory *inv, int num, uint8_t verbose)
{
	cmdDetail_t *cmd = (cmdDetail_t *)(ai_buf + ai_info->cmdDetail);
	const struct basil_rsvn *rsvn = basil_rsvn_by_id(inv, ai_info->resId);

	if (rsvn == NULL)
		fatal("can not find reservation entry in INVENTORY");
	/*
	 * Reservation source: this can be
	 * - "batch", for reservations made by the batch system
	 * - "aprun", for interactive submissions on a login node
	 * Due to also displaying the batch_id, only the first alternative is
	 * currently supported.
	 *
	 * State information:
	 * - "conf":  reservation has been confirmed (batch job has started),
	 *            but an aprun command is not currently executing
	 * - "claim": appears additionally once aprun is executing
	 * - "NID list": present if user has supplied a candidate node list
	 */
	if (verbose == 0) {
		if (ai_info->timePlaced)
			/*     ResId  ApId From      Arch  PEs N d Memory State */
			printf("A%5u %8llu batch:%-10s %s  %4d - - %6u conf,claim\n",
				ai_info->resId,
				(unsigned long long)ai_info->apid,
				rsvn->batch_id,
				nam_arch[rsvn->app_head->cmd_head->arch],
				ai_info->numPlaces,	/* number of PEs requested */
				rsvn->app_head->cmd_head->memory /* per-PE memory  */
			      );
		else
			/*    ResId  ApId From      Arch  PEs N d Memory State */
			printf("%6u %8llu batch:%-10s %s  %4d %d %d %6u %sconf%s\n",
				ai_info->resId,
				(unsigned long long)ai_info->apid,
				rsvn->batch_id,
				nam_arch[rsvn->app_head->cmd_head->arch],
				ai_info->numPlaces,	/* number of PEs requested */
				cmd->fixedPerNode,	/* PEs per node (nppn)	   */
				cmd->depth,		/* number of CPUs per PE   */
				cmd->memory,		/* per-PE memory size	   */
				ai_info->flags & 0x40 ? "NID list," : "",
				num_placed_applications(ai_buf, ai_info->resId) ? ",claim" : ""
				//is_any_node_claimed(ai_buf, ai_info) ? ",claim" : ""
			      );
	} else {
		struct nodespec *ns;
		/*
		 * apid:    ALPS application ID
		 * pagg:    process aggregate job ID (shell SID or SGI container ID)
		 * account: user account number (often UID)
		 * time:    CPU time limit for the application
		 */
		printf("Res[%d]: apid %llu, pagg %llu, resId %u, user %s,%s\n"
		       "       gid %u, account %llu, time %d, normal\n", num,
			(unsigned long long)ai_info->apid,
			(unsigned long long)ai_info->pagg,
			ai_info->resId, rsvn->user_name,
			ai_info->flags & 0x40 ? " NID list," : "",
			ai_info->gid,
			(unsigned long long)ai_info->account,
			ai_info->timeLim);

		printf("  Batch System ID = %s\n", rsvn->batch_id);
		printf("  Created at %s", ctime(&rsvn->timestamp));
		printf("  Number of commands %d, control network fanout %d\n",
			ai_info->numCmds, ai_info->fanout);

		print_cmd_detail(ai_buf, ai_info);
		ns = appinfo_nidlist(ai_buf, ai_info);
#if CLE <= 2
		printf("  Reservation list entries: %d\n", ai_info->numPlaces);
#else
		printf("  Reservation list entries: %d\n", ns_count_nodes(ns));
#endif
		if (verbose > 2)
			print_hex_placement_list(ai_buf, ai_info);
		else if (verbose > 1)
			printf("  Reservation list: %s\n", ns_to_string(ns));
		free_nodespec(ns);
	}
}

/**
 * apstat_r  -  print all or select reservation information
 * @ai_buf:    start of appinfo buffer area
 * @inv:       Basil INVENTORY response
 * @resid:     set to > 0 to select specific resId
 * @verbosity: set degree of output detail
 */
void apstat_r(const uint8_t *ai_buf, struct basil_inventory *inv,
	      uint32_t resid, unsigned verbosity)
{
	appInfoHdr_t *ai_hdr = (appInfoHdr_t *)ai_buf;
	size_t offset;
	int i;

	if (ai_hdr->apNum == 0) {
		printf("No resource reservations are present\n");
		return;
	}

	printf(" ResId     ApId From           Arch   PEs N d Memory State\n");
	for (i = 0, offset = ai_hdr->apStart; i < ai_hdr->apNum; i++) {
		appInfo_t  *ai_info = (appInfo_t *)(ai_buf + offset);

		if (resid == 0 || ai_info->resId == resid)
			print_resinfo(ai_buf, ai_info, inv, i, 0);

		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}

	if (verbosity == 0)
		return;
	else if (resid)
		printf("\nReservation detail for resid %u\n", resid);
	else
		printf("\nReservation detail\n");

	for (i = 0, offset = ai_hdr->apStart; i < ai_hdr->apNum; i++) {
		appInfo_t *ai_info = (appInfo_t *)(ai_buf + offset);

		/* timePlaced is > 0 whenever command runs in reservation */
		if (!ai_info->timePlaced && (!resid || ai_info->resId == resid))
			print_resinfo(ai_buf, ai_info, inv, i, verbosity);

		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}
}
