# Minimizer thresholds

This directory contains code to compute thresholds for minimizer-based searches.
It also provides a random data generator that can be used to reproduce the results presented in the paper.

## Usage

### Prerequisites
* gcc >=9
* cmake >= 3.8
* git

For more information, refer to [the SeqAn3 documentation](https://docs.seqan.de/seqan/3-master-user/setup.html).

### Building
```bash
mkdir build
cd build
cmake .. -DCMAKE_BUILD_TYPE=Release
make
```
In the following, we will reproduce the results for p={100, 150, 250} and 0 to 3 errors using varying tau.
You can either follow the step-by-step guide, or use the bash script provided under `script/run.sh`. You can place this
file in the `build` directory and execute it from within the `build` directory. This will run all command consecutively.
Not that you can adjust the pattern sizes and tau in the `run.sh`.

We recommend following directory structure:
```
.
├── test_data
│   ├── 100
│   ├── 150
│   └── 250
└── test_out
    ├── negative
    │   ├── 100
    │   ├── 150
    │   └── 250
    ├── positive
    │   ├── 100
    │   ├── 150
    │   └── 250
    └── with_precompute
        ├── negative
        │   ├── 100
        │   ├── 150
        │   └── 250
        └── positive
            ├── 100
            ├── 150
            └── 250
```
This tree can be generated by, e.g.:
```bash
for p in 100 150 250
do
    mkdir -p test_data/$p
    mkdir -p test_out/negative/$p
    mkdir -p test_out/positive/$p
    mkdir -p test_out/with_precompute/negative/$p
    mkdir -p test_out/with_precompute/positive/$p
done
```
In the following, we will refer to the binaries by `./random_data` and `./query`.  Please adjust this path if the binaries are not in the current working directory.

### Generating random data
See `./random_data --help` for information.
To generate a dataset for reads of length 100, 150 and 250 with 3 errors, you can use this command:
```bash
for p in 100 150 250
do
	./random_data --query_length $p --min_error 3 --max_error 3 --out test_data/$p
done
```
This will generate a reference of size 10'000'000 and 4 read sets containing 100'000 reads of length `$p` with 3 errors.

### Computing thresholds and searching
See `test --help` for information.
To use all methods to search for 3 errors, you may use:

To query for random reads, i.e. reads that should not be found in the reference data set, use:
```bash
for p in 100 150 250
do
    for tau in $(seq 0 0.05 1.01) 0.99 0.999
    do ./query --reference test_data/$p/reference.fasta \
               --query test_data/$p/query_random.fasta \
               --out test_out/negative/$p \
               --error 3 \
               --method all \
               --tau $tau
    done
done
```
After running this command, you copy the precomputed thresholds to the `with_precompute` directory and run a adjusted command to use the precomputed thresholds:
```bash
for p in 100 150 250
do
	cp test_out/negative/$p/binary_* test_out/with_precompute/negative/$p
done
```
```bash
for p in 100 150 250
do
    for tau in $(seq 0 0.05 1.01) 0.99 0.999
    do ./query --reference test_data/$p/reference.fasta \
               --query test_data/$p/query_random.fasta \
               --out test_out/with_precompute/negative/$p \
               --error 3 \
               --method all \
               --from_file \
               --tau $tau
    done
done
```
In a similar manner, the queries containing 3 errors can be searched:

```bash
for p in 100 150 250
do
    for tau in $(seq 0 0.05 1.01) 0.99 0.999
    do ./query --reference test_data/$p/reference.fasta \
               --query test_data/$p/query_e3.fasta \
               --out test_out/positive/$p \
               --error 3 \
               --method all \
               --tau $tau
    done
done
```
```bash
for p in 100 150 250
do
	cp test_out/positive/$p/binary_* test_out/with_precompute/positive/$p
done
```
```bash
for p in 100 150 250
do
    for tau in $(seq 0 0.05 1.01) 0.99 0.999
    do ./query --reference test_data/$p/reference.fasta \
               --query test_data/$p/query_random.fasta \
               --out test_out/with_precompute/positive/$p \
               --error 3 \
               --method all \
               --from_file \
               --tau $tau
    done
done
```
### Results
Different files are generated as an result:
* `binary_{METHOD}_w{W}_k{K}_e{E}_tau{TAU}`: The precomupted thresholds for METHOD, using window size W, k-mer size K, E many errors and a tau of TAU.
* `result_binary_{METHOD}_w{W}_k{K}_e{E}_tau{TAU}.csv` A csv file containing statistics for METHOD, using window size W, k-mer size K, E many errors and a tau of TAU.

Additionally, the results (including the threshold vector) are also printed to cerr.

### Scripts
The `script` folder contains scripts used to generate the plots.
`jupyter_notebooks` hosts python scripts/jupyter notebooks used to parse the result files.
The R scripts in `Rscripts` are used to plot the parsed results.
`svg_plots` shows the resulting plots.
