/*
 * CRC-32 algorithm implemented with the z/Architecture Vector Extension
 * Facility.
 *
 * Copyright 2024 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License.  You may obtain a copy
 * of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 *
 * Author(s): Hendrik Brueckner <brueckner@linux.vnet.ibm.com>
 *            Anton Blanchard <anton@au.ibm.com>
 *            Bryan Chan <bryan.chan@ca.ibm.com>
 *            Chris Zou <chriszou@ca.ibm.com>
 *            Aliaksei Makarau <aliaksei.makarau@ibm.com>
 */

#include <sys/types.h>
#include <endian.h>
#include "crc32c_s390x.h"

#define VX_MIN_LEN		64
#define VX_ALIGNMENT		16L
#define VX_ALIGN_MASK		(VX_ALIGNMENT - 1)

/* CRC-32C slicing-by-8 constants, for use on big-endian systems */
static const unsigned int __attribute__((aligned(128))) crc32ctable_le[8][256] = {
    {
    0x00000000, 0x03836bf2, 0xf7703be1, 0xf4f35013,
    0x1f979ac7, 0x1c14f135, 0xe8e7a126, 0xeb64cad4,
    0xcf58d98a, 0xccdbb278, 0x3828e26b, 0x3bab8999,
    0xd0cf434d, 0xd34c28bf, 0x27bf78ac, 0x243c135e,
    0x6fc75e10, 0x6c4435e2, 0x98b765f1, 0x9b340e03,
    0x7050c4d7, 0x73d3af25, 0x8720ff36, 0x84a394c4,
    0xa09f879a, 0xa31cec68, 0x57efbc7b, 0x546cd789,
    0xbf081d5d, 0xbc8b76af, 0x487826bc, 0x4bfb4d4e,
    0xde8ebd20, 0xdd0dd6d2, 0x29fe86c1, 0x2a7ded33,
    0xc11927e7, 0xc29a4c15, 0x36691c06, 0x35ea77f4,
    0x11d664aa, 0x12550f58, 0xe6a65f4b, 0xe52534b9,
    0x0e41fe6d, 0x0dc2959f, 0xf931c58c, 0xfab2ae7e,
    0xb149e330, 0xb2ca88c2, 0x4639d8d1, 0x45bab323,
    0xaede79f7, 0xad5d1205, 0x59ae4216, 0x5a2d29e4,
    0x7e113aba, 0x7d925148, 0x8961015b, 0x8ae26aa9,
    0x6186a07d, 0x6205cb8f, 0x96f69b9c, 0x9575f06e,
    0xbc1d7b41, 0xbf9e10b3, 0x4b6d40a0, 0x48ee2b52,
    0xa38ae186, 0xa0098a74, 0x54fada67, 0x5779b195,
    0x7345a2cb, 0x70c6c939, 0x8435992a, 0x87b6f2d8,
    0x6cd2380c, 0x6f5153fe, 0x9ba203ed, 0x9821681f,
    0xd3da2551, 0xd0594ea3, 0x24aa1eb0, 0x27297542,
    0xcc4dbf96, 0xcfced464, 0x3b3d8477, 0x38beef85,
    0x1c82fcdb, 0x1f019729, 0xebf2c73a, 0xe871acc8,
    0x0315661c, 0x00960dee, 0xf4655dfd, 0xf7e6360f,
    0x6293c661, 0x6110ad93, 0x95e3fd80, 0x96609672,
    0x7d045ca6, 0x7e873754, 0x8a746747, 0x89f70cb5,
    0xadcb1feb, 0xae487419, 0x5abb240a, 0x59384ff8,
    0xb25c852c, 0xb1dfeede, 0x452cbecd, 0x46afd53f,
    0x0d549871, 0x0ed7f383, 0xfa24a390, 0xf9a7c862,
    0x12c302b6, 0x11406944, 0xe5b33957, 0xe63052a5,
    0xc20c41fb, 0xc18f2a09, 0x357c7a1a, 0x36ff11e8,
    0xdd9bdb3c, 0xde18b0ce, 0x2aebe0dd, 0x29688b2f,
    0x783bf682, 0x7bb89d70, 0x8f4bcd63, 0x8cc8a691,
    0x67ac6c45, 0x642f07b7, 0x90dc57a4, 0x935f3c56,
    0xb7632f08, 0xb4e044fa, 0x401314e9, 0x43907f1b,
    0xa8f4b5cf, 0xab77de3d, 0x5f848e2e, 0x5c07e5dc,
    0x17fca892, 0x147fc360, 0xe08c9373, 0xe30ff881,
    0x086b3255, 0x0be859a7, 0xff1b09b4, 0xfc986246,
    0xd8a47118, 0xdb271aea, 0x2fd44af9, 0x2c57210b,
    0xc733ebdf, 0xc4b0802d, 0x3043d03e, 0x33c0bbcc,
    0xa6b54ba2, 0xa5362050, 0x51c57043, 0x52461bb1,
    0xb922d165, 0xbaa1ba97, 0x4e52ea84, 0x4dd18176,
    0x69ed9228, 0x6a6ef9da, 0x9e9da9c9, 0x9d1ec23b,
    0x767a08ef, 0x75f9631d, 0x810a330e, 0x828958fc,
    0xc97215b2, 0xcaf17e40, 0x3e022e53, 0x3d8145a1,
    0xd6e58f75, 0xd566e487, 0x2195b494, 0x2216df66,
    0x062acc38, 0x05a9a7ca, 0xf15af7d9, 0xf2d99c2b,
    0x19bd56ff, 0x1a3e3d0d, 0xeecd6d1e, 0xed4e06ec,
    0xc4268dc3, 0xc7a5e631, 0x3356b622, 0x30d5ddd0,
    0xdbb11704, 0xd8327cf6, 0x2cc12ce5, 0x2f424717,
    0x0b7e5449, 0x08fd3fbb, 0xfc0e6fa8, 0xff8d045a,
    0x14e9ce8e, 0x176aa57c, 0xe399f56f, 0xe01a9e9d,
    0xabe1d3d3, 0xa862b821, 0x5c91e832, 0x5f1283c0,
    0xb4764914, 0xb7f522e6, 0x430672f5, 0x40851907,
    0x64b90a59, 0x673a61ab, 0x93c931b8, 0x904a5a4a,
    0x7b2e909e, 0x78adfb6c, 0x8c5eab7f, 0x8fddc08d,
    0x1aa830e3, 0x192b5b11, 0xedd80b02, 0xee5b60f0,
    0x053faa24, 0x06bcc1d6, 0xf24f91c5, 0xf1ccfa37,
    0xd5f0e969, 0xd673829b, 0x2280d288, 0x2103b97a,
    0xca6773ae, 0xc9e4185c, 0x3d17484f, 0x3e9423bd,
    0x756f6ef3, 0x76ec0501, 0x821f5512, 0x819c3ee0,
    0x6af8f434, 0x697b9fc6, 0x9d88cfd5, 0x9e0ba427,
    0xba37b779, 0xb9b4dc8b, 0x4d478c98, 0x4ec4e76a,
    0xa5a02dbe, 0xa623464c, 0x52d0165f, 0x51537dad
    },{
    0x00000000, 0x7798a213, 0xee304527, 0x99a8e734,
    0xdc618a4e, 0xabf9285d, 0x3251cf69, 0x45c96d7a,
    0xb8c3149d, 0xcf5bb68e, 0x56f351ba, 0x216bf3a9,
    0x64a29ed3, 0x133a3cc0, 0x8a92dbf4, 0xfd0a79e7,
    0x81f1c53f, 0xf669672c, 0x6fc18018, 0x1859220b,
    0x5d904f71, 0x2a08ed62, 0xb3a00a56, 0xc438a845,
    0x3932d1a2, 0x4eaa73b1, 0xd7029485, 0xa09a3696,
    0xe5535bec, 0x92cbf9ff, 0x0b631ecb, 0x7cfbbcd8,
    0x02e38b7f, 0x757b296c, 0xecd3ce58, 0x9b4b6c4b,
    0xde820131, 0xa91aa322, 0x30b24416, 0x472ae605,
    0xba209fe2, 0xcdb83df1, 0x5410dac5, 0x238878d6,
    0x664115ac, 0x11d9b7bf, 0x8871508b, 0xffe9f298,
    0x83124e40, 0xf48aec53, 0x6d220b67, 0x1abaa974,
    0x5f73c40e, 0x28eb661d, 0xb1438129, 0xc6db233a,
    0x3bd15add, 0x4c49f8ce, 0xd5e11ffa, 0xa279bde9,
    0xe7b0d093, 0x90287280, 0x098095b4, 0x7e1837a7,
    0x04c617ff, 0x735eb5ec, 0xeaf652d8, 0x9d6ef0cb,
    0xd8a79db1, 0xaf3f3fa2, 0x3697d896, 0x410f7a85,
    0xbc050362, 0xcb9da171, 0x52354645, 0x25ade456,
    0x6064892c, 0x17fc2b3f, 0x8e54cc0b, 0xf9cc6e18,
    0x8537d2c0, 0xf2af70d3, 0x6b0797e7, 0x1c9f35f4,
    0x5956588e, 0x2ecefa9d, 0xb7661da9, 0xc0febfba,
    0x3df4c65d, 0x4a6c644e, 0xd3c4837a, 0xa45c2169,
    0xe1954c13, 0x960dee00, 0x0fa50934, 0x783dab27,
    0x06259c80, 0x71bd3e93, 0xe815d9a7, 0x9f8d7bb4,
    0xda4416ce, 0xaddcb4dd, 0x347453e9, 0x43ecf1fa,
    0xbee6881d, 0xc97e2a0e, 0x50d6cd3a, 0x274e6f29,
    0x62870253, 0x151fa040, 0x8cb74774, 0xfb2fe567,
    0x87d459bf, 0xf04cfbac, 0x69e41c98, 0x1e7cbe8b,
    0x5bb5d3f1, 0x2c2d71e2, 0xb58596d6, 0xc21d34c5,
    0x3f174d22, 0x488fef31, 0xd1270805, 0xa6bfaa16,
    0xe376c76c, 0x94ee657f, 0x0d46824b, 0x7ade2058,
    0xf9fac3fb, 0x8e6261e8, 0x17ca86dc, 0x605224cf,
    0x259b49b5, 0x5203eba6, 0xcbab0c92, 0xbc33ae81,
    0x4139d766, 0x36a17575, 0xaf099241, 0xd8913052,
    0x9d585d28, 0xeac0ff3b, 0x7368180f, 0x04f0ba1c,
    0x780b06c4, 0x0f93a4d7, 0x963b43e3, 0xe1a3e1f0,
    0xa46a8c8a, 0xd3f22e99, 0x4a5ac9ad, 0x3dc26bbe,
    0xc0c81259, 0xb750b04a, 0x2ef8577e, 0x5960f56d,
    0x1ca99817, 0x6b313a04, 0xf299dd30, 0x85017f23,
    0xfb194884, 0x8c81ea97, 0x15290da3, 0x62b1afb0,
    0x2778c2ca, 0x50e060d9, 0xc94887ed, 0xbed025fe,
    0x43da5c19, 0x3442fe0a, 0xadea193e, 0xda72bb2d,
    0x9fbbd657, 0xe8237444, 0x718b9370, 0x06133163,
    0x7ae88dbb, 0x0d702fa8, 0x94d8c89c, 0xe3406a8f,
    0xa68907f5, 0xd111a5e6, 0x48b942d2, 0x3f21e0c1,
    0xc22b9926, 0xb5b33b35, 0x2c1bdc01, 0x5b837e12,
    0x1e4a1368, 0x69d2b17b, 0xf07a564f, 0x87e2f45c,
    0xfd3cd404, 0x8aa47617, 0x130c9123, 0x64943330,
    0x215d5e4a, 0x56c5fc59, 0xcf6d1b6d, 0xb8f5b97e,
    0x45ffc099, 0x3267628a, 0xabcf85be, 0xdc5727ad,
    0x999e4ad7, 0xee06e8c4, 0x77ae0ff0, 0x0036ade3,
    0x7ccd113b, 0x0b55b328, 0x92fd541c, 0xe565f60f,
    0xa0ac9b75, 0xd7343966, 0x4e9cde52, 0x39047c41,
    0xc40e05a6, 0xb396a7b5, 0x2a3e4081, 0x5da6e292,
    0x186f8fe8, 0x6ff72dfb, 0xf65fcacf, 0x81c768dc,
    0xffdf5f7b, 0x8847fd68, 0x11ef1a5c, 0x6677b84f,
    0x23bed535, 0x54267726, 0xcd8e9012, 0xba163201,
    0x471c4be6, 0x3084e9f5, 0xa92c0ec1, 0xdeb4acd2,
    0x9b7dc1a8, 0xece563bb, 0x754d848f, 0x02d5269c,
    0x7e2e9a44, 0x09b63857, 0x901edf63, 0xe7867d70,
    0xa24f100a, 0xd5d7b219, 0x4c7f552d, 0x3be7f73e,
    0xc6ed8ed9, 0xb1752cca, 0x28ddcbfe, 0x5f4569ed,
    0x1a8c0497, 0x6d14a684, 0xf4bc41b0, 0x8324e3a3
    },{
    0x00000000, 0x7e9241a5, 0x0d526f4f, 0x73c02eea,
    0x1aa4de9e, 0x64369f3b, 0x17f6b1d1, 0x6964f074,
    0xc53e5138, 0xbbac109d, 0xc86c3e77, 0xb6fe7fd2,
    0xdf9a8fa6, 0xa108ce03, 0xd2c8e0e9, 0xac5aa14c,
    0x8a7da270, 0xf4efe3d5, 0x872fcd3f, 0xf9bd8c9a,
    0x90d97cee, 0xee4b3d4b, 0x9d8b13a1, 0xe3195204,
    0x4f43f348, 0x31d1b2ed, 0x42119c07, 0x3c83dda2,
    0x55e72dd6, 0x2b756c73, 0x58b54299, 0x2627033c,
    0x14fb44e1, 0x6a690544, 0x19a92bae, 0x673b6a0b,
    0x0e5f9a7f, 0x70cddbda, 0x030df530, 0x7d9fb495,
    0xd1c515d9, 0xaf57547c, 0xdc977a96, 0xa2053b33,
    0xcb61cb47, 0xb5f38ae2, 0xc633a408, 0xb8a1e5ad,
    0x9e86e691, 0xe014a734, 0x93d489de, 0xed46c87b,
    0x8422380f, 0xfab079aa, 0x89705740, 0xf7e216e5,
    0x5bb8b7a9, 0x252af60c, 0x56ead8e6, 0x28789943,
    0x411c6937, 0x3f8e2892, 0x4c4e0678, 0x32dc47dd,
    0xd98065c7, 0xa7122462, 0xd4d20a88, 0xaa404b2d,
    0xc324bb59, 0xbdb6fafc, 0xce76d416, 0xb0e495b3,
    0x1cbe34ff, 0x622c755a, 0x11ec5bb0, 0x6f7e1a15,
    0x061aea61, 0x7888abc4, 0x0b48852e, 0x75dac48b,
    0x53fdc7b7, 0x2d6f8612, 0x5eafa8f8, 0x203de95d,
    0x49591929, 0x37cb588c, 0x440b7666, 0x3a9937c3,
    0x96c3968f, 0xe851d72a, 0x9b91f9c0, 0xe503b865,
    0x8c674811, 0xf2f509b4, 0x8135275e, 0xffa766fb,
    0xcd7b2126, 0xb3e96083, 0xc0294e69, 0xbebb0fcc,
    0xd7dfffb8, 0xa94dbe1d, 0xda8d90f7, 0xa41fd152,
    0x0845701e, 0x76d731bb, 0x05171f51, 0x7b855ef4,
    0x12e1ae80, 0x6c73ef25, 0x1fb3c1cf, 0x6121806a,
    0x47068356, 0x3994c2f3, 0x4a54ec19, 0x34c6adbc,
    0x5da25dc8, 0x23301c6d, 0x50f03287, 0x2e627322,
    0x8238d26e, 0xfcaa93cb, 0x8f6abd21, 0xf1f8fc84,
    0x989c0cf0, 0xe60e4d55, 0x95ce63bf, 0xeb5c221a,
    0x4377278b, 0x3de5662e, 0x4e2548c4, 0x30b70961,
    0x59d3f915, 0x2741b8b0, 0x5481965a, 0x2a13d7ff,
    0x864976b3, 0xf8db3716, 0x8b1b19fc, 0xf5895859,
    0x9ceda82d, 0xe27fe988, 0x91bfc762, 0xef2d86c7,
    0xc90a85fb, 0xb798c45e, 0xc458eab4, 0xbacaab11,
    0xd3ae5b65, 0xad3c1ac0, 0xdefc342a, 0xa06e758f,
    0x0c34d4c3, 0x72a69566, 0x0166bb8c, 0x7ff4fa29,
    0x16900a5d, 0x68024bf8, 0x1bc26512, 0x655024b7,
    0x578c636a, 0x291e22cf, 0x5ade0c25, 0x244c4d80,
    0x4d28bdf4, 0x33bafc51, 0x407ad2bb, 0x3ee8931e,
    0x92b23252, 0xec2073f7, 0x9fe05d1d, 0xe1721cb8,
    0x8816eccc, 0xf684ad69, 0x85448383, 0xfbd6c226,
    0xddf1c11a, 0xa36380bf, 0xd0a3ae55, 0xae31eff0,
    0xc7551f84, 0xb9c75e21, 0xca0770cb, 0xb495316e,
    0x18cf9022, 0x665dd187, 0x159dff6d, 0x6b0fbec8,
    0x026b4ebc, 0x7cf90f19, 0x0f3921f3, 0x71ab6056,
    0x9af7424c, 0xe46503e9, 0x97a52d03, 0xe9376ca6,
    0x80539cd2, 0xfec1dd77, 0x8d01f39d, 0xf393b238,
    0x5fc91374, 0x215b52d1, 0x529b7c3b, 0x2c093d9e,
    0x456dcdea, 0x3bff8c4f, 0x483fa2a5, 0x36ade300,
    0x108ae03c, 0x6e18a199, 0x1dd88f73, 0x634aced6,
    0x0a2e3ea2, 0x74bc7f07, 0x077c51ed, 0x79ee1048,
    0xd5b4b104, 0xab26f0a1, 0xd8e6de4b, 0xa6749fee,
    0xcf106f9a, 0xb1822e3f, 0xc24200d5, 0xbcd04170,
    0x8e0c06ad, 0xf09e4708, 0x835e69e2, 0xfdcc2847,
    0x94a8d833, 0xea3a9996, 0x99fab77c, 0xe768f6d9,
    0x4b325795, 0x35a01630, 0x466038da, 0x38f2797f,
    0x5196890b, 0x2f04c8ae, 0x5cc4e644, 0x2256a7e1,
    0x0471a4dd, 0x7ae3e578, 0x0923cb92, 0x77b18a37,
    0x1ed57a43, 0x60473be6, 0x1387150c, 0x6d1554a9,
    0xc14ff5e5, 0xbfddb440, 0xcc1d9aaa, 0xb28fdb0f,
    0xdbeb2b7b, 0xa5796ade, 0xd6b94434, 0xa82b0591
    },{
    0x00000000, 0xb8aa45dd, 0x812367bf, 0x39892262,
    0xf331227b, 0x4b9b67a6, 0x721245c4, 0xcab80019,
    0xe66344f6, 0x5ec9012b, 0x67402349, 0xdfea6694,
    0x1552668d, 0xadf82350, 0x94710132, 0x2cdb44ef,
    0x3db164e9, 0x851b2134, 0xbc920356, 0x0438468b,
    0xce804692, 0x762a034f, 0x4fa3212d, 0xf70964f0,
    0xdbd2201f, 0x637865c2, 0x5af147a0, 0xe25b027d,
    0x28e30264, 0x904947b9, 0xa9c065db, 0x116a2006,
    0x8b1425d7, 0x33be600a, 0x0a374268, 0xb29d07b5,
    0x782507ac, 0xc08f4271, 0xf9066013, 0x41ac25ce,
    0x6d776121, 0xd5dd24fc, 0xec54069e, 0x54fe4343,
    0x9e46435a, 0x26ec0687, 0x1f6524e5, 0xa7cf6138,
    0xb6a5413e, 0x0e0f04e3, 0x37862681, 0x8f2c635c,
    0x45946345, 0xfd3e2698, 0xc4b704fa, 0x7c1d4127,
    0x50c605c8, 0xe86c4015, 0xd1e56277, 0x694f27aa,
    0xa3f727b3, 0x1b5d626e, 0x22d4400c, 0x9a7e05d1,
    0xe75fa6ab, 0x5ff5e376, 0x667cc114, 0xded684c9,
    0x146e84d0, 0xacc4c10d, 0x954de36f, 0x2de7a6b2,
    0x013ce25d, 0xb996a780, 0x801f85e2, 0x38b5c03f,
    0xf20dc026, 0x4aa785fb, 0x732ea799, 0xcb84e244,
    0xdaeec242, 0x6244879f, 0x5bcda5fd, 0xe367e020,
    0x29dfe039, 0x9175a5e4, 0xa8fc8786, 0x1056c25b,
    0x3c8d86b4, 0x8427c369, 0xbdaee10b, 0x0504a4d6,
    0xcfbca4cf, 0x7716e112, 0x4e9fc370, 0xf63586ad,
    0x6c4b837c, 0xd4e1c6a1, 0xed68e4c3, 0x55c2a11e,
    0x9f7aa107, 0x27d0e4da, 0x1e59c6b8, 0xa6f38365,
    0x8a28c78a, 0x32828257, 0x0b0ba035, 0xb3a1e5e8,
    0x7919e5f1, 0xc1b3a02c, 0xf83a824e, 0x4090c793,
    0x51fae795, 0xe950a248, 0xd0d9802a, 0x6873c5f7,
    0xa2cbc5ee, 0x1a618033, 0x23e8a251, 0x9b42e78c,
    0xb799a363, 0x0f33e6be, 0x36bac4dc, 0x8e108101,
    0x44a88118, 0xfc02c4c5, 0xc58be6a7, 0x7d21a37a,
    0x3fc9a052, 0x8763e58f, 0xbeeac7ed, 0x06408230,
    0xccf88229, 0x7452c7f4, 0x4ddbe596, 0xf571a04b,
    0xd9aae4a4, 0x6100a179, 0x5889831b, 0xe023c6c6,
    0x2a9bc6df, 0x92318302, 0xabb8a160, 0x1312e4bd,
    0x0278c4bb, 0xbad28166, 0x835ba304, 0x3bf1e6d9,
    0xf149e6c0, 0x49e3a31d, 0x706a817f, 0xc8c0c4a2,
    0xe41b804d, 0x5cb1c590, 0x6538e7f2, 0xdd92a22f,
    0x172aa236, 0xaf80e7eb, 0x9609c589, 0x2ea38054,
    0xb4dd8585, 0x0c77c058, 0x35fee23a, 0x8d54a7e7,
    0x47eca7fe, 0xff46e223, 0xc6cfc041, 0x7e65859c,
    0x52bec173, 0xea1484ae, 0xd39da6cc, 0x6b37e311,
    0xa18fe308, 0x1925a6d5, 0x20ac84b7, 0x9806c16a,
    0x896ce16c, 0x31c6a4b1, 0x084f86d3, 0xb0e5c30e,
    0x7a5dc317, 0xc2f786ca, 0xfb7ea4a8, 0x43d4e175,
    0x6f0fa59a, 0xd7a5e047, 0xee2cc225, 0x568687f8,
    0x9c3e87e1, 0x2494c23c, 0x1d1de05e, 0xa5b7a583,
    0xd89606f9, 0x603c4324, 0x59b56146, 0xe11f249b,
    0x2ba72482, 0x930d615f, 0xaa84433d, 0x122e06e0,
    0x3ef5420f, 0x865f07d2, 0xbfd625b0, 0x077c606d,
    0xcdc46074, 0x756e25a9, 0x4ce707cb, 0xf44d4216,
    0xe5276210, 0x5d8d27cd, 0x640405af, 0xdcae4072,
    0x1616406b, 0xaebc05b6, 0x973527d4, 0x2f9f6209,
    0x034426e6, 0xbbee633b, 0x82674159, 0x3acd0484,
    0xf075049d, 0x48df4140, 0x71566322, 0xc9fc26ff,
    0x5382232e, 0xeb2866f3, 0xd2a14491, 0x6a0b014c,
    0xa0b30155, 0x18194488, 0x219066ea, 0x993a2337,
    0xb5e167d8, 0x0d4b2205, 0x34c20067, 0x8c6845ba,
    0x46d045a3, 0xfe7a007e, 0xc7f3221c, 0x7f5967c1,
    0x6e3347c7, 0xd699021a, 0xef102078, 0x57ba65a5,
    0x9d0265bc, 0x25a82061, 0x1c210203, 0xa48b47de,
    0x88500331, 0x30fa46ec, 0x0973648e, 0xb1d92153,
    0x7b61214a, 0xc3cb6497, 0xfa4246f5, 0x42e80328
    },{
    0x00000000, 0xac6f1138, 0x58df2270, 0xf4b03348,
    0xb0be45e0, 0x1cd154d8, 0xe8616790, 0x440e76a8,
    0x910b67c5, 0x3d6476fd, 0xc9d445b5, 0x65bb548d,
    0x21b52225, 0x8dda331d, 0x796a0055, 0xd505116d,
    0xd361228f, 0x7f0e33b7, 0x8bbe00ff, 0x27d111c7,
    0x63df676f, 0xcfb07657, 0x3b00451f, 0x976f5427,
    0x426a454a, 0xee055472, 0x1ab5673a, 0xb6da7602,
    0xf2d400aa, 0x5ebb1192, 0xaa0b22da, 0x066433e2,
    0x57b5a81b, 0xfbdab923, 0x0f6a8a6b, 0xa3059b53,
    0xe70bedfb, 0x4b64fcc3, 0xbfd4cf8b, 0x13bbdeb3,
    0xc6becfde, 0x6ad1dee6, 0x9e61edae, 0x320efc96,
    0x76008a3e, 0xda6f9b06, 0x2edfa84e, 0x82b0b976,
    0x84d48a94, 0x28bb9bac, 0xdc0ba8e4, 0x7064b9dc,
    0x346acf74, 0x9805de4c, 0x6cb5ed04, 0xc0dafc3c,
    0x15dfed51, 0xb9b0fc69, 0x4d00cf21, 0xe16fde19,
    0xa561a8b1, 0x090eb989, 0xfdbe8ac1, 0x51d19bf9,
    0xae6a5137, 0x0205400f, 0xf6b57347, 0x5ada627f,
    0x1ed414d7, 0xb2bb05ef, 0x460b36a7, 0xea64279f,
    0x3f6136f2, 0x930e27ca, 0x67be1482, 0xcbd105ba,
    0x8fdf7312, 0x23b0622a, 0xd7005162, 0x7b6f405a,
    0x7d0b73b8, 0xd1646280, 0x25d451c8, 0x89bb40f0,
    0xcdb53658, 0x61da2760, 0x956a1428, 0x39050510,
    0xec00147d, 0x406f0545, 0xb4df360d, 0x18b02735,
    0x5cbe519d, 0xf0d140a5, 0x046173ed, 0xa80e62d5,
    0xf9dff92c, 0x55b0e814, 0xa100db5c, 0x0d6fca64,
    0x4961bccc, 0xe50eadf4, 0x11be9ebc, 0xbdd18f84,
    0x68d49ee9, 0xc4bb8fd1, 0x300bbc99, 0x9c64ada1,
    0xd86adb09, 0x7405ca31, 0x80b5f979, 0x2cdae841,
    0x2abedba3, 0x86d1ca9b, 0x7261f9d3, 0xde0ee8eb,
    0x9a009e43, 0x366f8f7b, 0xc2dfbc33, 0x6eb0ad0b,
    0xbbb5bc66, 0x17daad5e, 0xe36a9e16, 0x4f058f2e,
    0x0b0bf986, 0xa764e8be, 0x53d4dbf6, 0xffbbcace,
    0x5cd5a26e, 0xf0bab356, 0x040a801e, 0xa8659126,
    0xec6be78e, 0x4004f6b6, 0xb4b4c5fe, 0x18dbd4c6,
    0xcddec5ab, 0x61b1d493, 0x9501e7db, 0x396ef6e3,
    0x7d60804b, 0xd10f9173, 0x25bfa23b, 0x89d0b303,
    0x8fb480e1, 0x23db91d9, 0xd76ba291, 0x7b04b3a9,
    0x3f0ac501, 0x9365d439, 0x67d5e771, 0xcbbaf649,
    0x1ebfe724, 0xb2d0f61c, 0x4660c554, 0xea0fd46c,
    0xae01a2c4, 0x026eb3fc, 0xf6de80b4, 0x5ab1918c,
    0x0b600a75, 0xa70f1b4d, 0x53bf2805, 0xffd0393d,
    0xbbde4f95, 0x17b15ead, 0xe3016de5, 0x4f6e7cdd,
    0x9a6b6db0, 0x36047c88, 0xc2b44fc0, 0x6edb5ef8,
    0x2ad52850, 0x86ba3968, 0x720a0a20, 0xde651b18,
    0xd80128fa, 0x746e39c2, 0x80de0a8a, 0x2cb11bb2,
    0x68bf6d1a, 0xc4d07c22, 0x30604f6a, 0x9c0f5e52,
    0x490a4f3f, 0xe5655e07, 0x11d56d4f, 0xbdba7c77,
    0xf9b40adf, 0x55db1be7, 0xa16b28af, 0x0d043997,
    0xf2bff359, 0x5ed0e261, 0xaa60d129, 0x060fc011,
    0x4201b6b9, 0xee6ea781, 0x1ade94c9, 0xb6b185f1,
    0x63b4949c, 0xcfdb85a4, 0x3b6bb6ec, 0x9704a7d4,
    0xd30ad17c, 0x7f65c044, 0x8bd5f30c, 0x27bae234,
    0x21ded1d6, 0x8db1c0ee, 0x7901f3a6, 0xd56ee29e,
    0x91609436, 0x3d0f850e, 0xc9bfb646, 0x65d0a77e,
    0xb0d5b613, 0x1cbaa72b, 0xe80a9463, 0x4465855b,
    0x006bf3f3, 0xac04e2cb, 0x58b4d183, 0xf4dbc0bb,
    0xa50a5b42, 0x09654a7a, 0xfdd57932, 0x51ba680a,
    0x15b41ea2, 0xb9db0f9a, 0x4d6b3cd2, 0xe1042dea,
    0x34013c87, 0x986e2dbf, 0x6cde1ef7, 0xc0b10fcf,
    0x84bf7967, 0x28d0685f, 0xdc605b17, 0x700f4a2f,
    0x766b79cd, 0xda0468f5, 0x2eb45bbd, 0x82db4a85,
    0xc6d53c2d, 0x6aba2d15, 0x9e0a1e5d, 0x32650f65,
    0xe7601e08, 0x4b0f0f30, 0xbfbf3c78, 0x13d02d40,
    0x57de5be8, 0xfbb14ad0, 0x0f017998, 0xa36e68a0
    },{
    0x00000000, 0x196b30ef, 0xc3a08cdb, 0xdacbbc34,
    0x7737f5b2, 0x6e5cc55d, 0xb4977969, 0xadfc4986,
    0x1f180660, 0x0673368f, 0xdcb88abb, 0xc5d3ba54,
    0x682ff3d2, 0x7144c33d, 0xab8f7f09, 0xb2e44fe6,
    0x3e300cc0, 0x275b3c2f, 0xfd90801b, 0xe4fbb0f4,
    0x4907f972, 0x506cc99d, 0x8aa775a9, 0x93cc4546,
    0x21280aa0, 0x38433a4f, 0xe288867b, 0xfbe3b694,
    0x561fff12, 0x4f74cffd, 0x95bf73c9, 0x8cd44326,
    0x8d16f485, 0x947dc46a, 0x4eb6785e, 0x57dd48b1,
    0xfa210137, 0xe34a31d8, 0x39818dec, 0x20eabd03,
    0x920ef2e5, 0x8b65c20a, 0x51ae7e3e, 0x48c54ed1,
    0xe5390757, 0xfc5237b8, 0x26998b8c, 0x3ff2bb63,
    0xb326f845, 0xaa4dc8aa, 0x7086749e, 0x69ed4471,
    0xc4110df7, 0xdd7a3d18, 0x07b1812c, 0x1edab1c3,
    0xac3efe25, 0xb555ceca, 0x6f9e72fe, 0x76f54211,
    0xdb090b97, 0xc2623b78, 0x18a9874c, 0x01c2b7a3,
    0xeb5b040e, 0xf23034e1, 0x28fb88d5, 0x3190b83a,
    0x9c6cf1bc, 0x8507c153, 0x5fcc7d67, 0x46a74d88,
    0xf443026e, 0xed283281, 0x37e38eb5, 0x2e88be5a,
    0x8374f7dc, 0x9a1fc733, 0x40d47b07, 0x59bf4be8,
    0xd56b08ce, 0xcc003821, 0x16cb8415, 0x0fa0b4fa,
    0xa25cfd7c, 0xbb37cd93, 0x61fc71a7, 0x78974148,
    0xca730eae, 0xd3183e41, 0x09d38275, 0x10b8b29a,
    0xbd44fb1c, 0xa42fcbf3, 0x7ee477c7, 0x678f4728,
    0x664df08b, 0x7f26c064, 0xa5ed7c50, 0xbc864cbf,
    0x117a0539, 0x081135d6, 0xd2da89e2, 0xcbb1b90d,
    0x7955f6eb, 0x603ec604, 0xbaf57a30, 0xa39e4adf,
    0x0e620359, 0x170933b6, 0xcdc28f82, 0xd4a9bf6d,
    0x587dfc4b, 0x4116cca4, 0x9bdd7090, 0x82b6407f,
    0x2f4a09f9, 0x36213916, 0xecea8522, 0xf581b5cd,
    0x4765fa2b, 0x5e0ecac4, 0x84c576f0, 0x9dae461f,
    0x30520f99, 0x29393f76, 0xf3f28342, 0xea99b3ad,
    0xd6b7081c, 0xcfdc38f3, 0x151784c7, 0x0c7cb428,
    0xa180fdae, 0xb8ebcd41, 0x62207175, 0x7b4b419a,
    0xc9af0e7c, 0xd0c43e93, 0x0a0f82a7, 0x1364b248,
    0xbe98fbce, 0xa7f3cb21, 0x7d387715, 0x645347fa,
    0xe88704dc, 0xf1ec3433, 0x2b278807, 0x324cb8e8,
    0x9fb0f16e, 0x86dbc181, 0x5c107db5, 0x457b4d5a,
    0xf79f02bc, 0xeef43253, 0x343f8e67, 0x2d54be88,
    0x80a8f70e, 0x99c3c7e1, 0x43087bd5, 0x5a634b3a,
    0x5ba1fc99, 0x42cacc76, 0x98017042, 0x816a40ad,
    0x2c96092b, 0x35fd39c4, 0xef3685f0, 0xf65db51f,
    0x44b9faf9, 0x5dd2ca16, 0x87197622, 0x9e7246cd,
    0x338e0f4b, 0x2ae53fa4, 0xf02e8390, 0xe945b37f,
    0x6591f059, 0x7cfac0b6, 0xa6317c82, 0xbf5a4c6d,
    0x12a605eb, 0x0bcd3504, 0xd1068930, 0xc86db9df,
    0x7a89f639, 0x63e2c6d6, 0xb9297ae2, 0xa0424a0d,
    0x0dbe038b, 0x14d53364, 0xce1e8f50, 0xd775bfbf,
    0x3dec0c12, 0x24873cfd, 0xfe4c80c9, 0xe727b026,
    0x4adbf9a0, 0x53b0c94f, 0x897b757b, 0x90104594,
    0x22f40a72, 0x3b9f3a9d, 0xe15486a9, 0xf83fb646,
    0x55c3ffc0, 0x4ca8cf2f, 0x9663731b, 0x8f0843f4,
    0x03dc00d2, 0x1ab7303d, 0xc07c8c09, 0xd917bce6,
    0x74ebf560, 0x6d80c58f, 0xb74b79bb, 0xae204954,
    0x1cc406b2, 0x05af365d, 0xdf648a69, 0xc60fba86,
    0x6bf3f300, 0x7298c3ef, 0xa8537fdb, 0xb1384f34,
    0xb0faf897, 0xa991c878, 0x735a744c, 0x6a3144a3,
    0xc7cd0d25, 0xdea63dca, 0x046d81fe, 0x1d06b111,
    0xafe2fef7, 0xb689ce18, 0x6c42722c, 0x752942c3,
    0xd8d50b45, 0xc1be3baa, 0x1b75879e, 0x021eb771,
    0x8ecaf457, 0x97a1c4b8, 0x4d6a788c, 0x54014863,
    0xf9fd01e5, 0xe096310a, 0x3a5d8d3e, 0x2336bdd1,
    0x91d2f237, 0x88b9c2d8, 0x52727eec, 0x4b194e03,
    0xe6e50785, 0xff8e376a, 0x25458b5e, 0x3c2ebbb1
    },{
    0x00000000, 0xc82c0368, 0x905906d0, 0x587505b8,
    0xd1c5e0a5, 0x19e9e3cd, 0x419ce675, 0x89b0e51d,
    0x53fd2d4e, 0x9bd12e26, 0xc3a42b9e, 0x0b8828f6,
    0x8238cdeb, 0x4a14ce83, 0x1261cb3b, 0xda4dc853,
    0xa6fa5b9c, 0x6ed658f4, 0x36a35d4c, 0xfe8f5e24,
    0x773fbb39, 0xbf13b851, 0xe766bde9, 0x2f4abe81,
    0xf50776d2, 0x3d2b75ba, 0x655e7002, 0xad72736a,
    0x24c29677, 0xecee951f, 0xb49b90a7, 0x7cb793cf,
    0xbd835b3d, 0x75af5855, 0x2dda5ded, 0xe5f65e85,
    0x6c46bb98, 0xa46ab8f0, 0xfc1fbd48, 0x3433be20,
    0xee7e7673, 0x2652751b, 0x7e2770a3, 0xb60b73cb,
    0x3fbb96d6, 0xf79795be, 0xafe29006, 0x67ce936e,
    0x1b7900a1, 0xd35503c9, 0x8b200671, 0x430c0519,
    0xcabce004, 0x0290e36c, 0x5ae5e6d4, 0x92c9e5bc,
    0x48842def, 0x80a82e87, 0xd8dd2b3f, 0x10f12857,
    0x9941cd4a, 0x516dce22, 0x0918cb9a, 0xc134c8f2,
    0x7a07b77a, 0xb22bb412, 0xea5eb1aa, 0x2272b2c2,
    0xabc257df, 0x63ee54b7, 0x3b9b510f, 0xf3b75267,
    0x29fa9a34, 0xe1d6995c, 0xb9a39ce4, 0x718f9f8c,
    0xf83f7a91, 0x301379f9, 0x68667c41, 0xa04a7f29,
    0xdcfdece6, 0x14d1ef8e, 0x4ca4ea36, 0x8488e95e,
    0x0d380c43, 0xc5140f2b, 0x9d610a93, 0x554d09fb,
    0x8f00c1a8, 0x472cc2c0, 0x1f59c778, 0xd775c410,
    0x5ec5210d, 0x96e92265, 0xce9c27dd, 0x06b024b5,
    0xc784ec47, 0x0fa8ef2f, 0x57ddea97, 0x9ff1e9ff,
    0x16410ce2, 0xde6d0f8a, 0x86180a32, 0x4e34095a,
    0x9479c109, 0x5c55c261, 0x0420c7d9, 0xcc0cc4b1,
    0x45bc21ac, 0x8d9022c4, 0xd5e5277c, 0x1dc92414,
    0x617eb7db, 0xa952b4b3, 0xf127b10b, 0x390bb263,
    0xb0bb577e, 0x78975416, 0x20e251ae, 0xe8ce52c6,
    0x32839a95, 0xfaaf99fd, 0xa2da9c45, 0x6af69f2d,
    0xe3467a30, 0x2b6a7958, 0x731f7ce0, 0xbb337f88,
    0xf40e6ef5, 0x3c226d9d, 0x64576825, 0xac7b6b4d,
    0x25cb8e50, 0xede78d38, 0xb5928880, 0x7dbe8be8,
    0xa7f343bb, 0x6fdf40d3, 0x37aa456b, 0xff864603,
    0x7636a31e, 0xbe1aa076, 0xe66fa5ce, 0x2e43a6a6,
    0x52f43569, 0x9ad83601, 0xc2ad33b9, 0x0a8130d1,
    0x8331d5cc, 0x4b1dd6a4, 0x1368d31c, 0xdb44d074,
    0x01091827, 0xc9251b4f, 0x91501ef7, 0x597c1d9f,
    0xd0ccf882, 0x18e0fbea, 0x4095fe52, 0x88b9fd3a,
    0x498d35c8, 0x81a136a0, 0xd9d43318, 0x11f83070,
    0x9848d56d, 0x5064d605, 0x0811d3bd, 0xc03dd0d5,
    0x1a701886, 0xd25c1bee, 0x8a291e56, 0x42051d3e,
    0xcbb5f823, 0x0399fb4b, 0x5becfef3, 0x93c0fd9b,
    0xef776e54, 0x275b6d3c, 0x7f2e6884, 0xb7026bec,
    0x3eb28ef1, 0xf69e8d99, 0xaeeb8821, 0x66c78b49,
    0xbc8a431a, 0x74a64072, 0x2cd345ca, 0xe4ff46a2,
    0x6d4fa3bf, 0xa563a0d7, 0xfd16a56f, 0x353aa607,
    0x8e09d98f, 0x4625dae7, 0x1e50df5f, 0xd67cdc37,
    0x5fcc392a, 0x97e03a42, 0xcf953ffa, 0x07b93c92,
    0xddf4f4c1, 0x15d8f7a9, 0x4dadf211, 0x8581f179,
    0x0c311464, 0xc41d170c, 0x9c6812b4, 0x544411dc,
    0x28f38213, 0xe0df817b, 0xb8aa84c3, 0x708687ab,
    0xf93662b6, 0x311a61de, 0x696f6466, 0xa143670e,
    0x7b0eaf5d, 0xb322ac35, 0xeb57a98d, 0x237baae5,
    0xaacb4ff8, 0x62e74c90, 0x3a924928, 0xf2be4a40,
    0x338a82b2, 0xfba681da, 0xa3d38462, 0x6bff870a,
    0xe24f6217, 0x2a63617f, 0x721664c7, 0xba3a67af,
    0x6077affc, 0xa85bac94, 0xf02ea92c, 0x3802aa44,
    0xb1b24f59, 0x799e4c31, 0x21eb4989, 0xe9c74ae1,
    0x9570d92e, 0x5d5cda46, 0x0529dffe, 0xcd05dc96,
    0x44b5398b, 0x8c993ae3, 0xd4ec3f5b, 0x1cc03c33,
    0xc68df460, 0x0ea1f708, 0x56d4f2b0, 0x9ef8f1d8,
    0x174814c5, 0xdf6417ad, 0x87111215, 0x4f3d117d
    },{
    0x00000000, 0x277d3c49, 0x4efa7892, 0x698744db,
    0x6d821d21, 0x4aff2168, 0x237865b3, 0x040559fa,
    0xda043b42, 0xfd79070b, 0x94fe43d0, 0xb3837f99,
    0xb7862663, 0x90fb1a2a, 0xf97c5ef1, 0xde0162b8,
    0xb4097684, 0x93744acd, 0xfaf30e16, 0xdd8e325f,
    0xd98b6ba5, 0xfef657ec, 0x97711337, 0xb00c2f7e,
    0x6e0d4dc6, 0x4970718f, 0x20f73554, 0x078a091d,
    0x038f50e7, 0x24f26cae, 0x4d752875, 0x6a08143c,
    0x9965000d, 0xbe183c44, 0xd79f789f, 0xf0e244d6,
    0xf4e71d2c, 0xd39a2165, 0xba1d65be, 0x9d6059f7,
    0x43613b4f, 0x641c0706, 0x0d9b43dd, 0x2ae67f94,
    0x2ee3266e, 0x099e1a27, 0x60195efc, 0x476462b5,
    0x2d6c7689, 0x0a114ac0, 0x63960e1b, 0x44eb3252,
    0x40ee6ba8, 0x679357e1, 0x0e14133a, 0x29692f73,
    0xf7684dcb, 0xd0157182, 0xb9923559, 0x9eef0910,
    0x9aea50ea, 0xbd976ca3, 0xd4102878, 0xf36d1431,
    0x32cb001a, 0x15b63c53, 0x7c317888, 0x5b4c44c1,
    0x5f491d3b, 0x78342172, 0x11b365a9, 0x36ce59e0,
    0xe8cf3b58, 0xcfb20711, 0xa63543ca, 0x81487f83,
    0x854d2679, 0xa2301a30, 0xcbb75eeb, 0xecca62a2,
    0x86c2769e, 0xa1bf4ad7, 0xc8380e0c, 0xef453245,
    0xeb406bbf, 0xcc3d57f6, 0xa5ba132d, 0x82c72f64,
    0x5cc64ddc, 0x7bbb7195, 0x123c354e, 0x35410907,
    0x314450fd, 0x16396cb4, 0x7fbe286f, 0x58c31426,
    0xabae0017, 0x8cd33c5e, 0xe5547885, 0xc22944cc,
    0xc62c1d36, 0xe151217f, 0x88d665a4, 0xafab59ed,
    0x71aa3b55, 0x56d7071c, 0x3f5043c7, 0x182d7f8e,
    0x1c282674, 0x3b551a3d, 0x52d25ee6, 0x75af62af,
    0x1fa77693, 0x38da4ada, 0x515d0e01, 0x76203248,
    0x72256bb2, 0x555857fb, 0x3cdf1320, 0x1ba22f69,
    0xc5a34dd1, 0xe2de7198, 0x8b593543, 0xac24090a,
    0xa82150f0, 0x8f5c6cb9, 0xe6db2862, 0xc1a6142b,
    0x64960134, 0x43eb3d7d, 0x2a6c79a6, 0x0d1145ef,
    0x09141c15, 0x2e69205c, 0x47ee6487, 0x609358ce,
    0xbe923a76, 0x99ef063f, 0xf06842e4, 0xd7157ead,
    0xd3102757, 0xf46d1b1e, 0x9dea5fc5, 0xba97638c,
    0xd09f77b0, 0xf7e24bf9, 0x9e650f22, 0xb918336b,
    0xbd1d6a91, 0x9a6056d8, 0xf3e71203, 0xd49a2e4a,
    0x0a9b4cf2, 0x2de670bb, 0x44613460, 0x631c0829,
    0x671951d3, 0x40646d9a, 0x29e32941, 0x0e9e1508,
    0xfdf30139, 0xda8e3d70, 0xb30979ab, 0x947445e2,
    0x90711c18, 0xb70c2051, 0xde8b648a, 0xf9f658c3,
    0x27f73a7b, 0x008a0632, 0x690d42e9, 0x4e707ea0,
    0x4a75275a, 0x6d081b13, 0x048f5fc8, 0x23f26381,
    0x49fa77bd, 0x6e874bf4, 0x07000f2f, 0x207d3366,
    0x24786a9c, 0x030556d5, 0x6a82120e, 0x4dff2e47,
    0x93fe4cff, 0xb48370b6, 0xdd04346d, 0xfa790824,
    0xfe7c51de, 0xd9016d97, 0xb086294c, 0x97fb1505,
    0x565d012e, 0x71203d67, 0x18a779bc, 0x3fda45f5,
    0x3bdf1c0f, 0x1ca22046, 0x7525649d, 0x525858d4,
    0x8c593a6c, 0xab240625, 0xc2a342fe, 0xe5de7eb7,
    0xe1db274d, 0xc6a61b04, 0xaf215fdf, 0x885c6396,
    0xe25477aa, 0xc5294be3, 0xacae0f38, 0x8bd33371,
    0x8fd66a8b, 0xa8ab56c2, 0xc12c1219, 0xe6512e50,
    0x38504ce8, 0x1f2d70a1, 0x76aa347a, 0x51d70833,
    0x55d251c9, 0x72af6d80, 0x1b28295b, 0x3c551512,
    0xcf380123, 0xe8453d6a, 0x81c279b1, 0xa6bf45f8,
    0xa2ba1c02, 0x85c7204b, 0xec406490, 0xcb3d58d9,
    0x153c3a61, 0x32410628, 0x5bc642f3, 0x7cbb7eba,
    0x78be2740, 0x5fc31b09, 0x36445fd2, 0x1139639b,
    0x7b3177a7, 0x5c4c4bee, 0x35cb0f35, 0x12b6337c,
    0x16b36a86, 0x31ce56cf, 0x58491214, 0x7f342e5d,
    0xa1354ce5, 0x864870ac, 0xefcf3477, 0xc8b2083e,
    0xccb751c4, 0xebca6d8d, 0x824d2956, 0xa530151f
    }
};

/* Prototypes for functions in assembly files */
unsigned int crc32c_le_vgfm_16(uint32_t crc, unsigned char const*buf, unsigned size);

/* Pure C implementations of CRC, one byte at a time */
unsigned int crc32c_le(uint32_t crc, unsigned char const *buf, unsigned len){
	crc = htole32(crc);
	if(buf != 0)
		while (len--)
			crc = crc32ctable_le[0][((crc >> 24) ^ *buf++) & 0xFF] ^ (crc << 8);
	else
		while (len--)
			crc = crc32ctable_le[0][((crc >> 24)) & 0xFF] ^ (crc << 8);
	crc = le32toh(crc);
	return crc;
}

unsigned int ceph_crc32c_s390x(uint32_t crc, unsigned char const *data, unsigned datalen)
{
	unsigned long prealign, aligned, remaining;

	if(data == 0)
		return crc32c_le(crc, data, datalen);

	if(datalen < VX_MIN_LEN + VX_ALIGN_MASK)
		return crc32c_le(crc, data, datalen);

	if ((unsigned long)data & VX_ALIGN_MASK) {
		prealign = VX_ALIGNMENT - ((unsigned long)data & VX_ALIGN_MASK);
		datalen -= prealign;
		crc = crc32c_le(crc, data, prealign);
		data = data + prealign;
	}

	if (datalen < VX_MIN_LEN)
		return crc32c_le(crc, data, datalen);

	aligned = datalen & ~VX_ALIGN_MASK;
	remaining = datalen & VX_ALIGN_MASK;

	crc = crc32c_le_vgfm_16(crc, data, aligned);
	data = data + aligned;

	if (remaining)
		crc = crc32c_le(crc, data, remaining);

	return crc;
}
