/**********************************************************************
  Copyright(c) 2011-2016 Intel Corporation All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**********************************************************************/

#include "aes_xts.h"

#define NVEC 14

// struct to hold pointers to the key, plaintext and ciphertext vectors
struct xts_vector {
	uint64_t ptlen;		// length of our plaintext
	uint8_t *key1;		// dimension 16 for 128 bit aes
	uint8_t *key2;		// dimension 16 for 128 bit aes
	uint8_t *TW;		// dimension 16 for both 128 and 256 bit
	uint8_t *PTX;		// min. dimension 16
	uint8_t *CTX;		// same dimension as PTX
};

/* Define our test vectors statically here. Test vectors are from the standard:
 * "IEEE Standard for Cryptographic Protection of Data on Block-Oriented
 * Storage Devices"
 * http://ieeexplore.ieee.org/stamp/stamp.jsp?tp=&arnumber=4493450
 *
 * Vector 1
 * Key1 00000000000000000000000000000000
 * Key2 00000000000000000000000000000000
 * Data Unit Sequence number 0
 * PTX 0000000000000000000000000000000000000000000000000000000000000000 /128bit
 * TWK 66e94bd4ef8a2c3b884cfa59ca342b2eccd297a8df1559761099f4b39469565c
 * CTX 917cf69ebd68b2ec9b9fe9a3eadda692cd43d2f59598ed858c02c2652fbf922e
 * Plaintext length (bytes): 32
 */

static uint8_t v1_key1[16] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v1_key2[16] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v1_TW[16] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v1_PTX[32] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v1_CTX[32] = {
	0x91, 0x7c, 0xf6, 0x9e, 0xbd, 0x68, 0xb2, 0xec,
	0x9b, 0x9f, 0xe9, 0xa3, 0xea, 0xdd, 0xa6, 0x92,
	0xcd, 0x43, 0xd2, 0xf5, 0x95, 0x98, 0xed, 0x85,
	0x8c, 0x02, 0xc2, 0x65, 0x2f, 0xbf, 0x92, 0x2e
};

/*
 * Vector 2
 * Key1 11111111111111111111111111111111
 * Key2 22222222222222222222222222222222
 * Data Unit Sequence number 3333333333
 * PTX 4444444444444444444444444444444444444444444444444444444444444444
 * TWK 3f803bcd0d7fd2b37558419f59d5cda6f900779a1bfea467ebb0823eb3aa9b4d
 * CTX c454185e6a16936e39334038acef838bfb186fff7480adc4289382ecd6d394f0
 * Plaintext length (bytes): 32
 */

static uint8_t v2_key1[16] = {
	0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
	0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11
};

static uint8_t v2_key2[16] = {
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
};

static uint8_t v2_TW[16] = {
	0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v2_PTX[32] = {
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44
};

static uint8_t v2_CTX[32] = {
	0xc4, 0x54, 0x18, 0x5e, 0x6a, 0x16, 0x93, 0x6e,
	0x39, 0x33, 0x40, 0x38, 0xac, 0xef, 0x83, 0x8b,
	0xfb, 0x18, 0x6f, 0xff, 0x74, 0x80, 0xad, 0xc4,
	0x28, 0x93, 0x82, 0xec, 0xd6, 0xd3, 0x94, 0xf0
};

/*
 * Vector 3
 * Key1 fffefdfcfbfaf9f8f7f6f5f4f3f2f1f0
 * Key2 22222222222222222222222222222222
 * Data Unit Sequence number 3333333333
 * PTX 4444444444444444444444444444444444444444444444444444444444444444
 * TWK 3f803bcd0d7fd2b37558419f59d5cda6f900779a1bfea467ebb0823eb3aa9b4d
 * CTX af85336b597afc1a900b2eb21ec949d292df4c047e0b21532186a5971a227a89
 * Plaintext length (bytes): 32
 */

static uint8_t v3_key1[16] = {
	0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
	0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0
};

static uint8_t v3_key2[16] = {
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
};

static uint8_t v3_TW[16] = {
	0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v3_PTX[32] = {
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44
};

static uint8_t v3_CTX[32] = {
	0xaf, 0x85, 0x33, 0x6b, 0x59, 0x7a, 0xfc, 0x1a,
	0x90, 0x0b, 0x2e, 0xb2, 0x1e, 0xc9, 0x49, 0xd2,
	0x92, 0xdf, 0x4c, 0x04, 0x7e, 0x0b, 0x21, 0x53,
	0x21, 0x86, 0xa5, 0x97, 0x1a, 0x22, 0x7a, 0x89
};

/*
 * Vector 4
 * Key1 27182818284590452353602874713526
 * Key2 31415926535897932384626433832795
 * Data Unit Sequence number 0
 * PTX 000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f
 * PTX 202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f
 * PTX 404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f
 * PTX 606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f
 * PTX 808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f
 * PTX a0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebf
 * PTX c0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedf
 * PTX e0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff
 * PTX 000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f
 * PTX 202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f
 * PTX 404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f
 * PTX 606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f
 * PTX 808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f
 * PTX a0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebf
 * PTX c0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedf
 * PTX e0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff
 * CTX 27a7479befa1d476489f308cd4cfa6e2a96e4bbe3208ff25287dd3819616e89c
 * CTX c78cf7f5e543445f8333d8fa7f56000005279fa5d8b5e4ad40e736ddb4d35412
 * CTX 328063fd2aab53e5ea1e0a9f332500a5df9487d07a5c92cc512c8866c7e860ce
 * CTX 93fdf166a24912b422976146ae20ce846bb7dc9ba94a767aaef20c0d61ad0265
 * CTX 5ea92dc4c4e41a8952c651d33174be51a10c421110e6d81588ede82103a252d8
 * CTX a750e8768defffed9122810aaeb99f9172af82b604dc4b8e51bcb08235a6f434
 * CTX 1332e4ca60482a4ba1a03b3e65008fc5da76b70bf1690db4eae29c5f1badd03c
 * CTX 5ccf2a55d705ddcd86d449511ceb7ec30bf12b1fa35b913f9f747a8afd1b130e
 * CTX 94bff94effd01a91735ca1726acd0b197c4e5b03393697e126826fb6bbde8ecc
 * CTX 1e08298516e2c9ed03ff3c1b7860f6de76d4cecd94c8119855ef5297ca67e9f3
 * CTX e7ff72b1e99785ca0a7e7720c5b36dc6d72cac9574c8cbbc2f801e23e56fd344
 * CTX b07f22154beba0f08ce8891e643ed995c94d9a69c9f1b5f499027a78572aeebd
 * CTX 74d20cc39881c213ee770b1010e4bea718846977ae119f7a023ab58cca0ad752
 * CTX afe656bb3c17256a9f6e9bf19fdd5a38fc82bbe872c5539edb609ef4f79c203e
 * CTX bb140f2e583cb2ad15b4aa5b655016a8449277dbd477ef2c8d6c017db738b18d
 * CTX eb4a427d1923ce3ff262735779a418f20a282df920147beabe421ee5319d0568
 * Plaintext length (bytes): 512
 */
static uint8_t v4_key1[16] = {
	0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
	0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26
};

static uint8_t v4_key2[16] = {
	0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
	0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95
};

static uint8_t v4_TW[16] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v4_PTX[512] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

static uint8_t v4_CTX[512] = {
	0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76,
	0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2,
	0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25,
	0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c,
	0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f,
	0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00,
	0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad,
	0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12,
	0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5,
	0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5,
	0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc,
	0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce,
	0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4,
	0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84,
	0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a,
	0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65,
	0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89,
	0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51,
	0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15,
	0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8,
	0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed,
	0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91,
	0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e,
	0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34,
	0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b,
	0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5,
	0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4,
	0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c,
	0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd,
	0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3,
	0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f,
	0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e,
	0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91,
	0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19,
	0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1,
	0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc,
	0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed,
	0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde,
	0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98,
	0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3,
	0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca,
	0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6,
	0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc,
	0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44,
	0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0,
	0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95,
	0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4,
	0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd,
	0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13,
	0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7,
	0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a,
	0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52,
	0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a,
	0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38,
	0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e,
	0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e,
	0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad,
	0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8,
	0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c,
	0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d,
	0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f,
	0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2,
	0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea,
	0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68
};

/*
 * Vector 5
 * Key1 27182818284590452353602874713526
 * Key2 31415926535897932384626433832795
 * Data Unit Sequence Number 01
 * PTX 27a7479befa1d476489f308cd4cfa6e2a96e4bbe3208ff25287dd3819616e89c
 * PTX c78cf7f5e543445f8333d8fa7f56000005279fa5d8b5e4ad40e736ddb4d35412
 * PTX 328063fd2aab53e5ea1e0a9f332500a5df9487d07a5c92cc512c8866c7e860ce
 * PTX 93fdf166a24912b422976146ae20ce846bb7dc9ba94a767aaef20c0d61ad0265
 * PTX 5ea92dc4c4e41a8952c651d33174be51a10c421110e6d81588ede82103a252d8
 * PTX a750e8768defffed9122810aaeb99f9172af82b604dc4b8e51bcb08235a6f434
 * PTX 1332e4ca60482a4ba1a03b3e65008fc5da76b70bf1690db4eae29c5f1badd03c
 * PTX 5ccf2a55d705ddcd86d449511ceb7ec30bf12b1fa35b913f9f747a8afd1b130e
 * PTX 94bff94effd01a91735ca1726acd0b197c4e5b03393697e126826fb6bbde8ecc
 * PTX 1e08298516e2c9ed03ff3c1b7860f6de76d4cecd94c8119855ef5297ca67e9f3
 * PTX e7ff72b1e99785ca0a7e7720c5b36dc6d72cac9574c8cbbc2f801e23e56fd344
 * PTX b07f22154beba0f08ce8891e643ed995c94d9a69c9f1b5f499027a78572aeebd
 * PTX 74d20cc39881c213ee770b1010e4bea718846977ae119f7a023ab58cca0ad752
 * PTX afe656bb3c17256a9f6e9bf19fdd5a38fc82bbe872c5539edb609ef4f79c203e
 * PTX bb140f2e583cb2ad15b4aa5b655016a8449277dbd477ef2c8d6c017db738b18d
 * PTX eb4a427d1923ce3ff262735779a418f20a282df920147beabe421ee5319d0568
 * CTX 264d3ca8512194fec312c8c9891f279fefdd608d0c027b60483a3fa811d65ee5
 * CTX 9d52d9e40ec5672d81532b38b6b089ce951f0f9c35590b8b978d175213f329bb
 * CTX 1c2fd30f2f7f30492a61a532a79f51d36f5e31a7c9a12c286082ff7d2394d18f
 * CTX 783e1a8e72c722caaaa52d8f065657d2631fd25bfd8e5baad6e527d763517501
 * CTX c68c5edc3cdd55435c532d7125c8614deed9adaa3acade5888b87bef641c4c99
 * CTX 4c8091b5bcd387f3963fb5bc37aa922fbfe3df4e5b915e6eb514717bdd2a7407
 * CTX 9a5073f5c4bfd46adf7d282e7a393a52579d11a028da4d9cd9c77124f9648ee3
 * CTX 83b1ac763930e7162a8d37f350b2f74b8472cf09902063c6b32e8c2d9290cefb
 * CTX d7346d1c779a0df50edcde4531da07b099c638e83a755944df2aef1aa31752fd
 * CTX 323dcb710fb4bfbb9d22b925bc3577e1b8949e729a90bbafeacf7f7879e7b114
 * CTX 7e28ba0bae940db795a61b15ecf4df8db07b824bb062802cc98a9545bb2aaeed
 * CTX 77cb3fc6db15dcd7d80d7d5bc406c4970a3478ada8899b329198eb61c193fb62
 * CTX 75aa8ca340344a75a862aebe92eee1ce032fd950b47d7704a3876923b4ad6284
 * CTX 4bf4a09c4dbe8b4397184b7471360c9564880aedddb9baa4af2e75394b08cd32
 * CTX ff479c57a07d3eab5d54de5f9738b8d27f27a9f0ab11799d7b7ffefb2704c95c
 * CTX 6ad12c39f1e867a4b7b1d7818a4b753dfd2a89ccb45e001a03a867b187f225dd
 * Plaintext length (bytes): 512
 */

static uint8_t v5_key1[16] = {
	0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
	0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26
};

static uint8_t v5_key2[16] = {
	0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
	0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95
};

static uint8_t v5_TW[16] = {
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v5_PTX[512] = {
	0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76,
	0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2,
	0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25,
	0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c,
	0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f,
	0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00,
	0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad,
	0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12,
	0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5,
	0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5,
	0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc,
	0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce,
	0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4,
	0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84,
	0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a,
	0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65,
	0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89,
	0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51,
	0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15,
	0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8,
	0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed,
	0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91,
	0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e,
	0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34,
	0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b,
	0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5,
	0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4,
	0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c,
	0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd,
	0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3,
	0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f,
	0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e,
	0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91,
	0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19,
	0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1,
	0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc,
	0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed,
	0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde,
	0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98,
	0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3,
	0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca,
	0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6,
	0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc,
	0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44,
	0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0,
	0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95,
	0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4,
	0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd,
	0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13,
	0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7,
	0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a,
	0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52,
	0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a,
	0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38,
	0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e,
	0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e,
	0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad,
	0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8,
	0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c,
	0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d,
	0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f,
	0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2,
	0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea,
	0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68
};

static uint8_t v5_CTX[512] = {
	0x26, 0x4d, 0x3c, 0xa8, 0x51, 0x21, 0x94, 0xfe,
	0xc3, 0x12, 0xc8, 0xc9, 0x89, 0x1f, 0x27, 0x9f,
	0xef, 0xdd, 0x60, 0x8d, 0x0c, 0x02, 0x7b, 0x60,
	0x48, 0x3a, 0x3f, 0xa8, 0x11, 0xd6, 0x5e, 0xe5,
	0x9d, 0x52, 0xd9, 0xe4, 0x0e, 0xc5, 0x67, 0x2d,
	0x81, 0x53, 0x2b, 0x38, 0xb6, 0xb0, 0x89, 0xce,
	0x95, 0x1f, 0x0f, 0x9c, 0x35, 0x59, 0x0b, 0x8b,
	0x97, 0x8d, 0x17, 0x52, 0x13, 0xf3, 0x29, 0xbb,
	0x1c, 0x2f, 0xd3, 0x0f, 0x2f, 0x7f, 0x30, 0x49,
	0x2a, 0x61, 0xa5, 0x32, 0xa7, 0x9f, 0x51, 0xd3,
	0x6f, 0x5e, 0x31, 0xa7, 0xc9, 0xa1, 0x2c, 0x28,
	0x60, 0x82, 0xff, 0x7d, 0x23, 0x94, 0xd1, 0x8f,
	0x78, 0x3e, 0x1a, 0x8e, 0x72, 0xc7, 0x22, 0xca,
	0xaa, 0xa5, 0x2d, 0x8f, 0x06, 0x56, 0x57, 0xd2,
	0x63, 0x1f, 0xd2, 0x5b, 0xfd, 0x8e, 0x5b, 0xaa,
	0xd6, 0xe5, 0x27, 0xd7, 0x63, 0x51, 0x75, 0x01,
	0xc6, 0x8c, 0x5e, 0xdc, 0x3c, 0xdd, 0x55, 0x43,
	0x5c, 0x53, 0x2d, 0x71, 0x25, 0xc8, 0x61, 0x4d,
	0xee, 0xd9, 0xad, 0xaa, 0x3a, 0xca, 0xde, 0x58,
	0x88, 0xb8, 0x7b, 0xef, 0x64, 0x1c, 0x4c, 0x99,
	0x4c, 0x80, 0x91, 0xb5, 0xbc, 0xd3, 0x87, 0xf3,
	0x96, 0x3f, 0xb5, 0xbc, 0x37, 0xaa, 0x92, 0x2f,
	0xbf, 0xe3, 0xdf, 0x4e, 0x5b, 0x91, 0x5e, 0x6e,
	0xb5, 0x14, 0x71, 0x7b, 0xdd, 0x2a, 0x74, 0x07,
	0x9a, 0x50, 0x73, 0xf5, 0xc4, 0xbf, 0xd4, 0x6a,
	0xdf, 0x7d, 0x28, 0x2e, 0x7a, 0x39, 0x3a, 0x52,
	0x57, 0x9d, 0x11, 0xa0, 0x28, 0xda, 0x4d, 0x9c,
	0xd9, 0xc7, 0x71, 0x24, 0xf9, 0x64, 0x8e, 0xe3,
	0x83, 0xb1, 0xac, 0x76, 0x39, 0x30, 0xe7, 0x16,
	0x2a, 0x8d, 0x37, 0xf3, 0x50, 0xb2, 0xf7, 0x4b,
	0x84, 0x72, 0xcf, 0x09, 0x90, 0x20, 0x63, 0xc6,
	0xb3, 0x2e, 0x8c, 0x2d, 0x92, 0x90, 0xce, 0xfb,
	0xd7, 0x34, 0x6d, 0x1c, 0x77, 0x9a, 0x0d, 0xf5,
	0x0e, 0xdc, 0xde, 0x45, 0x31, 0xda, 0x07, 0xb0,
	0x99, 0xc6, 0x38, 0xe8, 0x3a, 0x75, 0x59, 0x44,
	0xdf, 0x2a, 0xef, 0x1a, 0xa3, 0x17, 0x52, 0xfd,
	0x32, 0x3d, 0xcb, 0x71, 0x0f, 0xb4, 0xbf, 0xbb,
	0x9d, 0x22, 0xb9, 0x25, 0xbc, 0x35, 0x77, 0xe1,
	0xb8, 0x94, 0x9e, 0x72, 0x9a, 0x90, 0xbb, 0xaf,
	0xea, 0xcf, 0x7f, 0x78, 0x79, 0xe7, 0xb1, 0x14,
	0x7e, 0x28, 0xba, 0x0b, 0xae, 0x94, 0x0d, 0xb7,
	0x95, 0xa6, 0x1b, 0x15, 0xec, 0xf4, 0xdf, 0x8d,
	0xb0, 0x7b, 0x82, 0x4b, 0xb0, 0x62, 0x80, 0x2c,
	0xc9, 0x8a, 0x95, 0x45, 0xbb, 0x2a, 0xae, 0xed,
	0x77, 0xcb, 0x3f, 0xc6, 0xdb, 0x15, 0xdc, 0xd7,
	0xd8, 0x0d, 0x7d, 0x5b, 0xc4, 0x06, 0xc4, 0x97,
	0x0a, 0x34, 0x78, 0xad, 0xa8, 0x89, 0x9b, 0x32,
	0x91, 0x98, 0xeb, 0x61, 0xc1, 0x93, 0xfb, 0x62,
	0x75, 0xaa, 0x8c, 0xa3, 0x40, 0x34, 0x4a, 0x75,
	0xa8, 0x62, 0xae, 0xbe, 0x92, 0xee, 0xe1, 0xce,
	0x03, 0x2f, 0xd9, 0x50, 0xb4, 0x7d, 0x77, 0x04,
	0xa3, 0x87, 0x69, 0x23, 0xb4, 0xad, 0x62, 0x84,
	0x4b, 0xf4, 0xa0, 0x9c, 0x4d, 0xbe, 0x8b, 0x43,
	0x97, 0x18, 0x4b, 0x74, 0x71, 0x36, 0x0c, 0x95,
	0x64, 0x88, 0x0a, 0xed, 0xdd, 0xb9, 0xba, 0xa4,
	0xaf, 0x2e, 0x75, 0x39, 0x4b, 0x08, 0xcd, 0x32,
	0xff, 0x47, 0x9c, 0x57, 0xa0, 0x7d, 0x3e, 0xab,
	0x5d, 0x54, 0xde, 0x5f, 0x97, 0x38, 0xb8, 0xd2,
	0x7f, 0x27, 0xa9, 0xf0, 0xab, 0x11, 0x79, 0x9d,
	0x7b, 0x7f, 0xfe, 0xfb, 0x27, 0x04, 0xc9, 0x5c,
	0x6a, 0xd1, 0x2c, 0x39, 0xf1, 0xe8, 0x67, 0xa4,
	0xb7, 0xb1, 0xd7, 0x81, 0x8a, 0x4b, 0x75, 0x3d,
	0xfd, 0x2a, 0x89, 0xcc, 0xb4, 0x5e, 0x00, 0x1a,
	0x03, 0xa8, 0x67, 0xb1, 0x87, 0xf2, 0x25, 0xdd
};

/*
 * Vector 6
 * Key1 27182818284590452353602874713526
 * Key2 31415926535897932384626433832795
 * Data Unit Sequence Number 02
 * PTX 264d3ca8512194fec312c8c9891f279fefdd608d0c027b60483a3fa811d65ee5
 * PTX 9d52d9e40ec5672d81532b38b6b089ce951f0f9c35590b8b978d175213f329bb
 * PTX 1c2fd30f2f7f30492a61a532a79f51d36f5e31a7c9a12c286082ff7d2394d18f
 * PTX 783e1a8e72c722caaaa52d8f065657d2631fd25bfd8e5baad6e527d763517501
 * PTX c68c5edc3cdd55435c532d7125c8614deed9adaa3acade5888b87bef641c4c99
 * PTX 4c8091b5bcd387f3963fb5bc37aa922fbfe3df4e5b915e6eb514717bdd2a7407
 * PTX 9a5073f5c4bfd46adf7d282e7a393a52579d11a028da4d9cd9c77124f9648ee3
 * PTX 83b1ac763930e7162a8d37f350b2f74b8472cf09902063c6b32e8c2d9290cefb
 * PTX d7346d1c779a0df50edcde4531da07b099c638e83a755944df2aef1aa31752fd
 * PTX 323dcb710fb4bfbb9d22b925bc3577e1b8949e729a90bbafeacf7f7879e7b114
 * PTX 7e28ba0bae940db795a61b15ecf4df8db07b824bb062802cc98a9545bb2aaeed
 * PTX 77cb3fc6db15dcd7d80d7d5bc406c4970a3478ada8899b329198eb61c193fb62
 * PTX 75aa8ca340344a75a862aebe92eee1ce032fd950b47d7704a3876923b4ad6284
 * PTX 4bf4a09c4dbe8b4397184b7471360c9564880aedddb9baa4af2e75394b08cd32
 * PTX ff479c57a07d3eab5d54de5f9738b8d27f27a9f0ab11799d7b7ffefb2704c95c
 * PTX 6ad12c39f1e867a4b7b1d7818a4b753dfd2a89ccb45e001a03a867b187f225dd
 * CTX fa762a3680b76007928ed4a4f49a9456031b704782e65e16cecb54ed7d017b5e
 * CTX 18abd67b338e81078f21edb7868d901ebe9c731a7c18b5e6dec1d6a72e078ac9
 * CTX a4262f860beefa14f4e821018272e411a951502b6e79066e84252c3346f3aa62
 * CTX 344351a291d4bedc7a07618bdea2af63145cc7a4b8d4070691ae890cd65733e7
 * CTX 946e9021a1dffc4c59f159425ee6d50ca9b135fa6162cea18a939838dc000fb3
 * CTX 86fad086acce5ac07cb2ece7fd580b00cfa5e98589631dc25e8e2a3daf2ffdec
 * CTX 26531659912c9d8f7a15e5865ea8fb5816d6207052bd7128cd743c12c8118791
 * CTX a4736811935eb982a532349e31dd401e0b660a568cb1a4711f552f55ded59f1f
 * CTX 15bf7196b3ca12a91e488ef59d64f3a02bf45239499ac6176ae321c4a211ec54
 * CTX 5365971c5d3f4f09d4eb139bfdf2073d33180b21002b65cc9865e76cb24cd92c
 * CTX 874c24c18350399a936ab3637079295d76c417776b94efce3a0ef7206b151105
 * CTX 19655c956cbd8b2489405ee2b09a6b6eebe0c53790a12a8998378b33a5b71159
 * CTX 625f4ba49d2a2fdba59fbf0897bc7aabd8d707dc140a80f0f309f835d3da54ab
 * CTX 584e501dfa0ee977fec543f74186a802b9a37adb3e8291eca04d66520d229e60
 * CTX 401e7282bef486ae059aa70696e0e305d777140a7a883ecdcb69b9ff938e8a42
 * CTX 31864c69ca2c2043bed007ff3e605e014bcf518138dc3a25c5e236171a2d01d6
 * Plaintext length (bytes): 512
 */
static uint8_t v6_key1[16] = {
	0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
	0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26
};

static uint8_t v6_key2[16] = {
	0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
	0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95
};

static uint8_t v6_TW[16] = {
	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v6_PTX[512] = {

	0x26, 0x4d, 0x3c, 0xa8, 0x51, 0x21, 0x94, 0xfe,
	0xc3, 0x12, 0xc8, 0xc9, 0x89, 0x1f, 0x27, 0x9f,
	0xef, 0xdd, 0x60, 0x8d, 0x0c, 0x02, 0x7b, 0x60,
	0x48, 0x3a, 0x3f, 0xa8, 0x11, 0xd6, 0x5e, 0xe5,
	0x9d, 0x52, 0xd9, 0xe4, 0x0e, 0xc5, 0x67, 0x2d,
	0x81, 0x53, 0x2b, 0x38, 0xb6, 0xb0, 0x89, 0xce,
	0x95, 0x1f, 0x0f, 0x9c, 0x35, 0x59, 0x0b, 0x8b,
	0x97, 0x8d, 0x17, 0x52, 0x13, 0xf3, 0x29, 0xbb,
	0x1c, 0x2f, 0xd3, 0x0f, 0x2f, 0x7f, 0x30, 0x49,
	0x2a, 0x61, 0xa5, 0x32, 0xa7, 0x9f, 0x51, 0xd3,
	0x6f, 0x5e, 0x31, 0xa7, 0xc9, 0xa1, 0x2c, 0x28,
	0x60, 0x82, 0xff, 0x7d, 0x23, 0x94, 0xd1, 0x8f,
	0x78, 0x3e, 0x1a, 0x8e, 0x72, 0xc7, 0x22, 0xca,
	0xaa, 0xa5, 0x2d, 0x8f, 0x06, 0x56, 0x57, 0xd2,
	0x63, 0x1f, 0xd2, 0x5b, 0xfd, 0x8e, 0x5b, 0xaa,
	0xd6, 0xe5, 0x27, 0xd7, 0x63, 0x51, 0x75, 0x01,
	0xc6, 0x8c, 0x5e, 0xdc, 0x3c, 0xdd, 0x55, 0x43,
	0x5c, 0x53, 0x2d, 0x71, 0x25, 0xc8, 0x61, 0x4d,
	0xee, 0xd9, 0xad, 0xaa, 0x3a, 0xca, 0xde, 0x58,
	0x88, 0xb8, 0x7b, 0xef, 0x64, 0x1c, 0x4c, 0x99,
	0x4c, 0x80, 0x91, 0xb5, 0xbc, 0xd3, 0x87, 0xf3,
	0x96, 0x3f, 0xb5, 0xbc, 0x37, 0xaa, 0x92, 0x2f,
	0xbf, 0xe3, 0xdf, 0x4e, 0x5b, 0x91, 0x5e, 0x6e,
	0xb5, 0x14, 0x71, 0x7b, 0xdd, 0x2a, 0x74, 0x07,
	0x9a, 0x50, 0x73, 0xf5, 0xc4, 0xbf, 0xd4, 0x6a,
	0xdf, 0x7d, 0x28, 0x2e, 0x7a, 0x39, 0x3a, 0x52,
	0x57, 0x9d, 0x11, 0xa0, 0x28, 0xda, 0x4d, 0x9c,
	0xd9, 0xc7, 0x71, 0x24, 0xf9, 0x64, 0x8e, 0xe3,
	0x83, 0xb1, 0xac, 0x76, 0x39, 0x30, 0xe7, 0x16,
	0x2a, 0x8d, 0x37, 0xf3, 0x50, 0xb2, 0xf7, 0x4b,
	0x84, 0x72, 0xcf, 0x09, 0x90, 0x20, 0x63, 0xc6,
	0xb3, 0x2e, 0x8c, 0x2d, 0x92, 0x90, 0xce, 0xfb,
	0xd7, 0x34, 0x6d, 0x1c, 0x77, 0x9a, 0x0d, 0xf5,
	0x0e, 0xdc, 0xde, 0x45, 0x31, 0xda, 0x07, 0xb0,
	0x99, 0xc6, 0x38, 0xe8, 0x3a, 0x75, 0x59, 0x44,
	0xdf, 0x2a, 0xef, 0x1a, 0xa3, 0x17, 0x52, 0xfd,
	0x32, 0x3d, 0xcb, 0x71, 0x0f, 0xb4, 0xbf, 0xbb,
	0x9d, 0x22, 0xb9, 0x25, 0xbc, 0x35, 0x77, 0xe1,
	0xb8, 0x94, 0x9e, 0x72, 0x9a, 0x90, 0xbb, 0xaf,
	0xea, 0xcf, 0x7f, 0x78, 0x79, 0xe7, 0xb1, 0x14,
	0x7e, 0x28, 0xba, 0x0b, 0xae, 0x94, 0x0d, 0xb7,
	0x95, 0xa6, 0x1b, 0x15, 0xec, 0xf4, 0xdf, 0x8d,
	0xb0, 0x7b, 0x82, 0x4b, 0xb0, 0x62, 0x80, 0x2c,
	0xc9, 0x8a, 0x95, 0x45, 0xbb, 0x2a, 0xae, 0xed,
	0x77, 0xcb, 0x3f, 0xc6, 0xdb, 0x15, 0xdc, 0xd7,
	0xd8, 0x0d, 0x7d, 0x5b, 0xc4, 0x06, 0xc4, 0x97,
	0x0a, 0x34, 0x78, 0xad, 0xa8, 0x89, 0x9b, 0x32,
	0x91, 0x98, 0xeb, 0x61, 0xc1, 0x93, 0xfb, 0x62,
	0x75, 0xaa, 0x8c, 0xa3, 0x40, 0x34, 0x4a, 0x75,
	0xa8, 0x62, 0xae, 0xbe, 0x92, 0xee, 0xe1, 0xce,
	0x03, 0x2f, 0xd9, 0x50, 0xb4, 0x7d, 0x77, 0x04,
	0xa3, 0x87, 0x69, 0x23, 0xb4, 0xad, 0x62, 0x84,
	0x4b, 0xf4, 0xa0, 0x9c, 0x4d, 0xbe, 0x8b, 0x43,
	0x97, 0x18, 0x4b, 0x74, 0x71, 0x36, 0x0c, 0x95,
	0x64, 0x88, 0x0a, 0xed, 0xdd, 0xb9, 0xba, 0xa4,
	0xaf, 0x2e, 0x75, 0x39, 0x4b, 0x08, 0xcd, 0x32,
	0xff, 0x47, 0x9c, 0x57, 0xa0, 0x7d, 0x3e, 0xab,
	0x5d, 0x54, 0xde, 0x5f, 0x97, 0x38, 0xb8, 0xd2,
	0x7f, 0x27, 0xa9, 0xf0, 0xab, 0x11, 0x79, 0x9d,
	0x7b, 0x7f, 0xfe, 0xfb, 0x27, 0x04, 0xc9, 0x5c,
	0x6a, 0xd1, 0x2c, 0x39, 0xf1, 0xe8, 0x67, 0xa4,
	0xb7, 0xb1, 0xd7, 0x81, 0x8a, 0x4b, 0x75, 0x3d,
	0xfd, 0x2a, 0x89, 0xcc, 0xb4, 0x5e, 0x00, 0x1a,
	0x03, 0xa8, 0x67, 0xb1, 0x87, 0xf2, 0x25, 0xdd
};

static uint8_t v6_CTX[512] = {

	0xfa, 0x76, 0x2a, 0x36, 0x80, 0xb7, 0x60, 0x07,
	0x92, 0x8e, 0xd4, 0xa4, 0xf4, 0x9a, 0x94, 0x56,
	0x03, 0x1b, 0x70, 0x47, 0x82, 0xe6, 0x5e, 0x16,
	0xce, 0xcb, 0x54, 0xed, 0x7d, 0x01, 0x7b, 0x5e,
	0x18, 0xab, 0xd6, 0x7b, 0x33, 0x8e, 0x81, 0x07,
	0x8f, 0x21, 0xed, 0xb7, 0x86, 0x8d, 0x90, 0x1e,
	0xbe, 0x9c, 0x73, 0x1a, 0x7c, 0x18, 0xb5, 0xe6,
	0xde, 0xc1, 0xd6, 0xa7, 0x2e, 0x07, 0x8a, 0xc9,
	0xa4, 0x26, 0x2f, 0x86, 0x0b, 0xee, 0xfa, 0x14,
	0xf4, 0xe8, 0x21, 0x01, 0x82, 0x72, 0xe4, 0x11,
	0xa9, 0x51, 0x50, 0x2b, 0x6e, 0x79, 0x06, 0x6e,
	0x84, 0x25, 0x2c, 0x33, 0x46, 0xf3, 0xaa, 0x62,
	0x34, 0x43, 0x51, 0xa2, 0x91, 0xd4, 0xbe, 0xdc,
	0x7a, 0x07, 0x61, 0x8b, 0xde, 0xa2, 0xaf, 0x63,
	0x14, 0x5c, 0xc7, 0xa4, 0xb8, 0xd4, 0x07, 0x06,
	0x91, 0xae, 0x89, 0x0c, 0xd6, 0x57, 0x33, 0xe7,
	0x94, 0x6e, 0x90, 0x21, 0xa1, 0xdf, 0xfc, 0x4c,
	0x59, 0xf1, 0x59, 0x42, 0x5e, 0xe6, 0xd5, 0x0c,
	0xa9, 0xb1, 0x35, 0xfa, 0x61, 0x62, 0xce, 0xa1,
	0x8a, 0x93, 0x98, 0x38, 0xdc, 0x00, 0x0f, 0xb3,
	0x86, 0xfa, 0xd0, 0x86, 0xac, 0xce, 0x5a, 0xc0,
	0x7c, 0xb2, 0xec, 0xe7, 0xfd, 0x58, 0x0b, 0x00,
	0xcf, 0xa5, 0xe9, 0x85, 0x89, 0x63, 0x1d, 0xc2,
	0x5e, 0x8e, 0x2a, 0x3d, 0xaf, 0x2f, 0xfd, 0xec,
	0x26, 0x53, 0x16, 0x59, 0x91, 0x2c, 0x9d, 0x8f,
	0x7a, 0x15, 0xe5, 0x86, 0x5e, 0xa8, 0xfb, 0x58,
	0x16, 0xd6, 0x20, 0x70, 0x52, 0xbd, 0x71, 0x28,
	0xcd, 0x74, 0x3c, 0x12, 0xc8, 0x11, 0x87, 0x91,
	0xa4, 0x73, 0x68, 0x11, 0x93, 0x5e, 0xb9, 0x82,
	0xa5, 0x32, 0x34, 0x9e, 0x31, 0xdd, 0x40, 0x1e,
	0x0b, 0x66, 0x0a, 0x56, 0x8c, 0xb1, 0xa4, 0x71,
	0x1f, 0x55, 0x2f, 0x55, 0xde, 0xd5, 0x9f, 0x1f,
	0x15, 0xbf, 0x71, 0x96, 0xb3, 0xca, 0x12, 0xa9,
	0x1e, 0x48, 0x8e, 0xf5, 0x9d, 0x64, 0xf3, 0xa0,
	0x2b, 0xf4, 0x52, 0x39, 0x49, 0x9a, 0xc6, 0x17,
	0x6a, 0xe3, 0x21, 0xc4, 0xa2, 0x11, 0xec, 0x54,
	0x53, 0x65, 0x97, 0x1c, 0x5d, 0x3f, 0x4f, 0x09,
	0xd4, 0xeb, 0x13, 0x9b, 0xfd, 0xf2, 0x07, 0x3d,
	0x33, 0x18, 0x0b, 0x21, 0x00, 0x2b, 0x65, 0xcc,
	0x98, 0x65, 0xe7, 0x6c, 0xb2, 0x4c, 0xd9, 0x2c,
	0x87, 0x4c, 0x24, 0xc1, 0x83, 0x50, 0x39, 0x9a,
	0x93, 0x6a, 0xb3, 0x63, 0x70, 0x79, 0x29, 0x5d,
	0x76, 0xc4, 0x17, 0x77, 0x6b, 0x94, 0xef, 0xce,
	0x3a, 0x0e, 0xf7, 0x20, 0x6b, 0x15, 0x11, 0x05,
	0x19, 0x65, 0x5c, 0x95, 0x6c, 0xbd, 0x8b, 0x24,
	0x89, 0x40, 0x5e, 0xe2, 0xb0, 0x9a, 0x6b, 0x6e,
	0xeb, 0xe0, 0xc5, 0x37, 0x90, 0xa1, 0x2a, 0x89,
	0x98, 0x37, 0x8b, 0x33, 0xa5, 0xb7, 0x11, 0x59,
	0x62, 0x5f, 0x4b, 0xa4, 0x9d, 0x2a, 0x2f, 0xdb,
	0xa5, 0x9f, 0xbf, 0x08, 0x97, 0xbc, 0x7a, 0xab,
	0xd8, 0xd7, 0x07, 0xdc, 0x14, 0x0a, 0x80, 0xf0,
	0xf3, 0x09, 0xf8, 0x35, 0xd3, 0xda, 0x54, 0xab,
	0x58, 0x4e, 0x50, 0x1d, 0xfa, 0x0e, 0xe9, 0x77,
	0xfe, 0xc5, 0x43, 0xf7, 0x41, 0x86, 0xa8, 0x02,
	0xb9, 0xa3, 0x7a, 0xdb, 0x3e, 0x82, 0x91, 0xec,
	0xa0, 0x4d, 0x66, 0x52, 0x0d, 0x22, 0x9e, 0x60,
	0x40, 0x1e, 0x72, 0x82, 0xbe, 0xf4, 0x86, 0xae,
	0x05, 0x9a, 0xa7, 0x06, 0x96, 0xe0, 0xe3, 0x05,
	0xd7, 0x77, 0x14, 0x0a, 0x7a, 0x88, 0x3e, 0xcd,
	0xcb, 0x69, 0xb9, 0xff, 0x93, 0x8e, 0x8a, 0x42,
	0x31, 0x86, 0x4c, 0x69, 0xca, 0x2c, 0x20, 0x43,
	0xbe, 0xd0, 0x07, 0xff, 0x3e, 0x60, 0x5e, 0x01,
	0x4b, 0xcf, 0x51, 0x81, 0x38, 0xdc, 0x3a, 0x25,
	0xc5, 0xe2, 0x36, 0x17, 0x1a, 0x2d, 0x01, 0xd6
};

/*
 * Vector 7
 * Key1 27182818284590452353602874713526
 * Key2 31415926535897932384626433832795
 * Data Unit Sequence Number fd
 * PTX 8e41b78c390b5af9d758bb214a67e9f6bf7727b09ac6124084c37611398fa45d
 * PTX aad94868600ed391fb1acd4857a95b466e62ef9f4b377244d1c152e7b30d731a
 * PTX ad30c716d214b707aed99eb5b5e580b3e887cf7497465651d4b60e6042051da3
 * PTX 693c3b78c14489543be8b6ad0ba629565bba202313ba7b0d0c94a3252b676f46
 * PTX cc02ce0f8a7d34c0ed229129673c1f61aed579d08a9203a25aac3a77e9db6026
 * PTX 7996db38df637356d9dcd1632e369939f2a29d89345c66e05066f1a3677aef18
 * PTX dea4113faeb629e46721a66d0a7e785d3e29af2594eb67dfa982affe0aac058f
 * PTX 6e15864269b135418261fc3afb089472cf68c45dd7f231c6249ba0255e1e0338
 * PTX 33fc4d00a3fe02132d7bc3873614b8aee34273581ea0325c81f0270affa13641
 * PTX d052d36f0757d484014354d02d6883ca15c24d8c3956b1bd027bcf41f151fd80
 * PTX 23c5340e5606f37e90fdb87c86fb4fa634b3718a30bace06a66eaf8f63c4aa3b
 * PTX 637826a87fe8cfa44282e92cb1615af3a28e53bc74c7cba1a0977be9065d0c1a
 * PTX 5dec6c54ae38d37f37aa35283e048e5530a85c4e7a29d7b92ec0c3169cdf2a80
 * PTX 5c7604bce60049b9fb7b8eaac10f51ae23794ceba68bb58112e293b9b692ca72
 * PTX 1b37c662f8574ed4dba6f88e170881c82cddc1034a0ca7e284bf0962b6b26292
 * PTX d836fa9f73c1ac770eef0f2d3a1eaf61d3e03555fd424eedd67e18a18094f888
 * CTX d55f684f81f4426e9fde92a5ff02df2ac896af63962888a97910c1379e20b0a3
 * CTX b1db613fb7fe2e07004329ea5c22bfd33e3dbe4cf58cc608c2c26c19a2e2fe22
 * CTX f98732c2b5cb844cc6c0702d91e1d50fc4382a7eba5635cd602432a2306ac4ce
 * CTX 82f8d70c8d9bc15f918fe71e74c622d5cf71178bf6e0b9cc9f2b41dd8dbe441c
 * CTX 41cd0c73a6dc47a348f6702f9d0e9b1b1431e948e299b9ec2272ab2c5f0c7be8
 * CTX 6affa5dec87a0bee81d3d50007edaa2bcfccb35605155ff36ed8edd4a40dcd4b
 * CTX 243acd11b2b987bdbfaf91a7cac27e9c5aea525ee53de7b2d3332c8644402b82
 * CTX 3e94a7db26276d2d23aa07180f76b4fd29b9c0823099c9d62c519880aee7e969
 * CTX 7617c1497d47bf3e571950311421b6b734d38b0db91eb85331b91ea9f61530f5
 * CTX 4512a5a52a4bad589eb69781d537f23297bb459bdad2948a29e1550bf4787e0b
 * CTX e95bb173cf5fab17dab7a13a052a63453d97ccec1a321954886b7a1299faaeec
 * CTX ae35c6eaaca753b041b5e5f093bf83397fd21dd6b3012066fcc058cc32c3b09d
 * CTX 7562dee29509b5839392c9ff05f51f3166aaac4ac5f238038a3045e6f72e48ef
 * CTX 0fe8bc675e82c318a268e43970271bf119b81bf6a982746554f84e72b9f00280
 * CTX a320a08142923c23c883423ff949827f29bbacdc1ccdb04938ce6098c95ba6b3
 * CTX 2528f4ef78eed778b2e122ddfd1cbdd11d1c0a6783e011fc536d63d053260637
 * Plaintext length (bytes): 512
 */
static uint8_t v7_key1[16] = {

	0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
	0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26
};

static uint8_t v7_key2[16] = {

	0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
	0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95
};

static uint8_t v7_TW[16] = {

	0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v7_PTX[512] = {

	0x8e, 0x41, 0xb7, 0x8c, 0x39, 0x0b, 0x5a, 0xf9,
	0xd7, 0x58, 0xbb, 0x21, 0x4a, 0x67, 0xe9, 0xf6,
	0xbf, 0x77, 0x27, 0xb0, 0x9a, 0xc6, 0x12, 0x40,
	0x84, 0xc3, 0x76, 0x11, 0x39, 0x8f, 0xa4, 0x5d,
	0xaa, 0xd9, 0x48, 0x68, 0x60, 0x0e, 0xd3, 0x91,
	0xfb, 0x1a, 0xcd, 0x48, 0x57, 0xa9, 0x5b, 0x46,
	0x6e, 0x62, 0xef, 0x9f, 0x4b, 0x37, 0x72, 0x44,
	0xd1, 0xc1, 0x52, 0xe7, 0xb3, 0x0d, 0x73, 0x1a,
	0xad, 0x30, 0xc7, 0x16, 0xd2, 0x14, 0xb7, 0x07,
	0xae, 0xd9, 0x9e, 0xb5, 0xb5, 0xe5, 0x80, 0xb3,
	0xe8, 0x87, 0xcf, 0x74, 0x97, 0x46, 0x56, 0x51,
	0xd4, 0xb6, 0x0e, 0x60, 0x42, 0x05, 0x1d, 0xa3,
	0x69, 0x3c, 0x3b, 0x78, 0xc1, 0x44, 0x89, 0x54,
	0x3b, 0xe8, 0xb6, 0xad, 0x0b, 0xa6, 0x29, 0x56,
	0x5b, 0xba, 0x20, 0x23, 0x13, 0xba, 0x7b, 0x0d,
	0x0c, 0x94, 0xa3, 0x25, 0x2b, 0x67, 0x6f, 0x46,
	0xcc, 0x02, 0xce, 0x0f, 0x8a, 0x7d, 0x34, 0xc0,
	0xed, 0x22, 0x91, 0x29, 0x67, 0x3c, 0x1f, 0x61,
	0xae, 0xd5, 0x79, 0xd0, 0x8a, 0x92, 0x03, 0xa2,
	0x5a, 0xac, 0x3a, 0x77, 0xe9, 0xdb, 0x60, 0x26,
	0x79, 0x96, 0xdb, 0x38, 0xdf, 0x63, 0x73, 0x56,
	0xd9, 0xdc, 0xd1, 0x63, 0x2e, 0x36, 0x99, 0x39,
	0xf2, 0xa2, 0x9d, 0x89, 0x34, 0x5c, 0x66, 0xe0,
	0x50, 0x66, 0xf1, 0xa3, 0x67, 0x7a, 0xef, 0x18,
	0xde, 0xa4, 0x11, 0x3f, 0xae, 0xb6, 0x29, 0xe4,
	0x67, 0x21, 0xa6, 0x6d, 0x0a, 0x7e, 0x78, 0x5d,
	0x3e, 0x29, 0xaf, 0x25, 0x94, 0xeb, 0x67, 0xdf,
	0xa9, 0x82, 0xaf, 0xfe, 0x0a, 0xac, 0x05, 0x8f,
	0x6e, 0x15, 0x86, 0x42, 0x69, 0xb1, 0x35, 0x41,
	0x82, 0x61, 0xfc, 0x3a, 0xfb, 0x08, 0x94, 0x72,
	0xcf, 0x68, 0xc4, 0x5d, 0xd7, 0xf2, 0x31, 0xc6,
	0x24, 0x9b, 0xa0, 0x25, 0x5e, 0x1e, 0x03, 0x38,
	0x33, 0xfc, 0x4d, 0x00, 0xa3, 0xfe, 0x02, 0x13,
	0x2d, 0x7b, 0xc3, 0x87, 0x36, 0x14, 0xb8, 0xae,
	0xe3, 0x42, 0x73, 0x58, 0x1e, 0xa0, 0x32, 0x5c,
	0x81, 0xf0, 0x27, 0x0a, 0xff, 0xa1, 0x36, 0x41,
	0xd0, 0x52, 0xd3, 0x6f, 0x07, 0x57, 0xd4, 0x84,
	0x01, 0x43, 0x54, 0xd0, 0x2d, 0x68, 0x83, 0xca,
	0x15, 0xc2, 0x4d, 0x8c, 0x39, 0x56, 0xb1, 0xbd,
	0x02, 0x7b, 0xcf, 0x41, 0xf1, 0x51, 0xfd, 0x80,
	0x23, 0xc5, 0x34, 0x0e, 0x56, 0x06, 0xf3, 0x7e,
	0x90, 0xfd, 0xb8, 0x7c, 0x86, 0xfb, 0x4f, 0xa6,
	0x34, 0xb3, 0x71, 0x8a, 0x30, 0xba, 0xce, 0x06,
	0xa6, 0x6e, 0xaf, 0x8f, 0x63, 0xc4, 0xaa, 0x3b,
	0x63, 0x78, 0x26, 0xa8, 0x7f, 0xe8, 0xcf, 0xa4,
	0x42, 0x82, 0xe9, 0x2c, 0xb1, 0x61, 0x5a, 0xf3,
	0xa2, 0x8e, 0x53, 0xbc, 0x74, 0xc7, 0xcb, 0xa1,
	0xa0, 0x97, 0x7b, 0xe9, 0x06, 0x5d, 0x0c, 0x1a,
	0x5d, 0xec, 0x6c, 0x54, 0xae, 0x38, 0xd3, 0x7f,
	0x37, 0xaa, 0x35, 0x28, 0x3e, 0x04, 0x8e, 0x55,
	0x30, 0xa8, 0x5c, 0x4e, 0x7a, 0x29, 0xd7, 0xb9,
	0x2e, 0xc0, 0xc3, 0x16, 0x9c, 0xdf, 0x2a, 0x80,
	0x5c, 0x76, 0x04, 0xbc, 0xe6, 0x00, 0x49, 0xb9,
	0xfb, 0x7b, 0x8e, 0xaa, 0xc1, 0x0f, 0x51, 0xae,
	0x23, 0x79, 0x4c, 0xeb, 0xa6, 0x8b, 0xb5, 0x81,
	0x12, 0xe2, 0x93, 0xb9, 0xb6, 0x92, 0xca, 0x72,
	0x1b, 0x37, 0xc6, 0x62, 0xf8, 0x57, 0x4e, 0xd4,
	0xdb, 0xa6, 0xf8, 0x8e, 0x17, 0x08, 0x81, 0xc8,
	0x2c, 0xdd, 0xc1, 0x03, 0x4a, 0x0c, 0xa7, 0xe2,
	0x84, 0xbf, 0x09, 0x62, 0xb6, 0xb2, 0x62, 0x92,
	0xd8, 0x36, 0xfa, 0x9f, 0x73, 0xc1, 0xac, 0x77,
	0x0e, 0xef, 0x0f, 0x2d, 0x3a, 0x1e, 0xaf, 0x61,
	0xd3, 0xe0, 0x35, 0x55, 0xfd, 0x42, 0x4e, 0xed,
	0xd6, 0x7e, 0x18, 0xa1, 0x80, 0x94, 0xf8, 0x88
};

static uint8_t v7_CTX[512] = {

	0xd5, 0x5f, 0x68, 0x4f, 0x81, 0xf4, 0x42, 0x6e,
	0x9f, 0xde, 0x92, 0xa5, 0xff, 0x02, 0xdf, 0x2a,
	0xc8, 0x96, 0xaf, 0x63, 0x96, 0x28, 0x88, 0xa9,
	0x79, 0x10, 0xc1, 0x37, 0x9e, 0x20, 0xb0, 0xa3,
	0xb1, 0xdb, 0x61, 0x3f, 0xb7, 0xfe, 0x2e, 0x07,
	0x00, 0x43, 0x29, 0xea, 0x5c, 0x22, 0xbf, 0xd3,
	0x3e, 0x3d, 0xbe, 0x4c, 0xf5, 0x8c, 0xc6, 0x08,
	0xc2, 0xc2, 0x6c, 0x19, 0xa2, 0xe2, 0xfe, 0x22,
	0xf9, 0x87, 0x32, 0xc2, 0xb5, 0xcb, 0x84, 0x4c,
	0xc6, 0xc0, 0x70, 0x2d, 0x91, 0xe1, 0xd5, 0x0f,
	0xc4, 0x38, 0x2a, 0x7e, 0xba, 0x56, 0x35, 0xcd,
	0x60, 0x24, 0x32, 0xa2, 0x30, 0x6a, 0xc4, 0xce,
	0x82, 0xf8, 0xd7, 0x0c, 0x8d, 0x9b, 0xc1, 0x5f,
	0x91, 0x8f, 0xe7, 0x1e, 0x74, 0xc6, 0x22, 0xd5,
	0xcf, 0x71, 0x17, 0x8b, 0xf6, 0xe0, 0xb9, 0xcc,
	0x9f, 0x2b, 0x41, 0xdd, 0x8d, 0xbe, 0x44, 0x1c,
	0x41, 0xcd, 0x0c, 0x73, 0xa6, 0xdc, 0x47, 0xa3,
	0x48, 0xf6, 0x70, 0x2f, 0x9d, 0x0e, 0x9b, 0x1b,
	0x14, 0x31, 0xe9, 0x48, 0xe2, 0x99, 0xb9, 0xec,
	0x22, 0x72, 0xab, 0x2c, 0x5f, 0x0c, 0x7b, 0xe8,
	0x6a, 0xff, 0xa5, 0xde, 0xc8, 0x7a, 0x0b, 0xee,
	0x81, 0xd3, 0xd5, 0x00, 0x07, 0xed, 0xaa, 0x2b,
	0xcf, 0xcc, 0xb3, 0x56, 0x05, 0x15, 0x5f, 0xf3,
	0x6e, 0xd8, 0xed, 0xd4, 0xa4, 0x0d, 0xcd, 0x4b,
	0x24, 0x3a, 0xcd, 0x11, 0xb2, 0xb9, 0x87, 0xbd,
	0xbf, 0xaf, 0x91, 0xa7, 0xca, 0xc2, 0x7e, 0x9c,
	0x5a, 0xea, 0x52, 0x5e, 0xe5, 0x3d, 0xe7, 0xb2,
	0xd3, 0x33, 0x2c, 0x86, 0x44, 0x40, 0x2b, 0x82,
	0x3e, 0x94, 0xa7, 0xdb, 0x26, 0x27, 0x6d, 0x2d,
	0x23, 0xaa, 0x07, 0x18, 0x0f, 0x76, 0xb4, 0xfd,
	0x29, 0xb9, 0xc0, 0x82, 0x30, 0x99, 0xc9, 0xd6,
	0x2c, 0x51, 0x98, 0x80, 0xae, 0xe7, 0xe9, 0x69,
	0x76, 0x17, 0xc1, 0x49, 0x7d, 0x47, 0xbf, 0x3e,
	0x57, 0x19, 0x50, 0x31, 0x14, 0x21, 0xb6, 0xb7,
	0x34, 0xd3, 0x8b, 0x0d, 0xb9, 0x1e, 0xb8, 0x53,
	0x31, 0xb9, 0x1e, 0xa9, 0xf6, 0x15, 0x30, 0xf5,
	0x45, 0x12, 0xa5, 0xa5, 0x2a, 0x4b, 0xad, 0x58,
	0x9e, 0xb6, 0x97, 0x81, 0xd5, 0x37, 0xf2, 0x32,
	0x97, 0xbb, 0x45, 0x9b, 0xda, 0xd2, 0x94, 0x8a,
	0x29, 0xe1, 0x55, 0x0b, 0xf4, 0x78, 0x7e, 0x0b,
	0xe9, 0x5b, 0xb1, 0x73, 0xcf, 0x5f, 0xab, 0x17,
	0xda, 0xb7, 0xa1, 0x3a, 0x05, 0x2a, 0x63, 0x45,
	0x3d, 0x97, 0xcc, 0xec, 0x1a, 0x32, 0x19, 0x54,
	0x88, 0x6b, 0x7a, 0x12, 0x99, 0xfa, 0xae, 0xec,
	0xae, 0x35, 0xc6, 0xea, 0xac, 0xa7, 0x53, 0xb0,
	0x41, 0xb5, 0xe5, 0xf0, 0x93, 0xbf, 0x83, 0x39,
	0x7f, 0xd2, 0x1d, 0xd6, 0xb3, 0x01, 0x20, 0x66,
	0xfc, 0xc0, 0x58, 0xcc, 0x32, 0xc3, 0xb0, 0x9d,
	0x75, 0x62, 0xde, 0xe2, 0x95, 0x09, 0xb5, 0x83,
	0x93, 0x92, 0xc9, 0xff, 0x05, 0xf5, 0x1f, 0x31,
	0x66, 0xaa, 0xac, 0x4a, 0xc5, 0xf2, 0x38, 0x03,
	0x8a, 0x30, 0x45, 0xe6, 0xf7, 0x2e, 0x48, 0xef,
	0x0f, 0xe8, 0xbc, 0x67, 0x5e, 0x82, 0xc3, 0x18,
	0xa2, 0x68, 0xe4, 0x39, 0x70, 0x27, 0x1b, 0xf1,
	0x19, 0xb8, 0x1b, 0xf6, 0xa9, 0x82, 0x74, 0x65,
	0x54, 0xf8, 0x4e, 0x72, 0xb9, 0xf0, 0x02, 0x80,
	0xa3, 0x20, 0xa0, 0x81, 0x42, 0x92, 0x3c, 0x23,
	0xc8, 0x83, 0x42, 0x3f, 0xf9, 0x49, 0x82, 0x7f,
	0x29, 0xbb, 0xac, 0xdc, 0x1c, 0xcd, 0xb0, 0x49,
	0x38, 0xce, 0x60, 0x98, 0xc9, 0x5b, 0xa6, 0xb3,
	0x25, 0x28, 0xf4, 0xef, 0x78, 0xee, 0xd7, 0x78,
	0xb2, 0xe1, 0x22, 0xdd, 0xfd, 0x1c, 0xbd, 0xd1,
	0x1d, 0x1c, 0x0a, 0x67, 0x83, 0xe0, 0x11, 0xfc,
	0x53, 0x6d, 0x63, 0xd0, 0x53, 0x26, 0x06, 0x37
};

/*
 * Vector 8
 * Key1 27182818284590452353602874713526
 * Key2 31415926535897932384626433832795
 * Data Unit Sequence Number fe
 * PTX d55f684f81f4426e9fde92a5ff02df2ac896af63962888a97910c1379e20b0a3
 * PTX b1db613fb7fe2e07004329ea5c22bfd33e3dbe4cf58cc608c2c26c19a2e2fe22
 * PTX f98732c2b5cb844cc6c0702d91e1d50fc4382a7eba5635cd602432a2306ac4ce
 * PTX 82f8d70c8d9bc15f918fe71e74c622d5cf71178bf6e0b9cc9f2b41dd8dbe441c
 * PTX 41cd0c73a6dc47a348f6702f9d0e9b1b1431e948e299b9ec2272ab2c5f0c7be8
 * PTX 6affa5dec87a0bee81d3d50007edaa2bcfccb35605155ff36ed8edd4a40dcd4b
 * PTX 243acd11b2b987bdbfaf91a7cac27e9c5aea525ee53de7b2d3332c8644402b82
 * PTX 3e94a7db26276d2d23aa07180f76b4fd29b9c0823099c9d62c519880aee7e969
 * PTX 7617c1497d47bf3e571950311421b6b734d38b0db91eb85331b91ea9f61530f5
 * PTX 4512a5a52a4bad589eb69781d537f23297bb459bdad2948a29e1550bf4787e0b
 * PTX e95bb173cf5fab17dab7a13a052a63453d97ccec1a321954886b7a1299faaeec
 * PTX ae35c6eaaca753b041b5e5f093bf83397fd21dd6b3012066fcc058cc32c3b09d
 * PTX 7562dee29509b5839392c9ff05f51f3166aaac4ac5f238038a3045e6f72e48ef
 * PTX 0fe8bc675e82c318a268e43970271bf119b81bf6a982746554f84e72b9f00280
 * PTX a320a08142923c23c883423ff949827f29bbacdc1ccdb04938ce6098c95ba6b3
 * PTX 2528f4ef78eed778b2e122ddfd1cbdd11d1c0a6783e011fc536d63d053260637
 * CTX 72efc1ebfe1ee25975a6eb3aa8589dda2b261f1c85bdab442a9e5b2dd1d7c395
 * CTX 7a16fc08e526d4b1223f1b1232a11af274c3d70dac57f83e0983c498f1a6f1ae
 * CTX cb021c3e70085a1e527f1ce41ee5911a82020161529cd82773762daf5459de94
 * CTX a0a82adae7e1703c808543c29ed6fb32d9e004327c1355180c995a07741493a0
 * CTX 9c21ba01a387882da4f62534b87bb15d60d197201c0fd3bf30c1500a3ecfecdd
 * CTX 66d8721f90bcc4c17ee925c61b0a03727a9c0d5f5ca462fbfa0af1c2513a9d9d
 * CTX 4b5345bd27a5f6e653f751693e6b6a2b8ead57d511e00e58c45b7b8d005af792
 * CTX 88f5c7c22fd4f1bf7a898b03a5634c6a1ae3f9fae5de4f296a2896b23e7ed43e
 * CTX d14fa5a2803f4d28f0d3ffcf24757677aebdb47bb388378708948a8d4126ed18
 * CTX 39e0da29a537a8c198b3c66ab00712dd261674bf45a73d67f76914f830ca014b
 * CTX 65596f27e4cf62de66125a5566df9975155628b400fbfb3a29040ed50faffdbb
 * CTX 18aece7c5c44693260aab386c0a37b11b114f1c415aebb653be468179428d43a
 * CTX 4d8bc3ec38813eca30a13cf1bb18d524f1992d44d8b1a42ea30b22e6c95b199d
 * CTX 8d182f8840b09d059585c31ad691fa0619ff038aca2c39a943421157361717c4
 * CTX 9d322028a74648113bd8c9d7ec77cf3c89c1ec8718ceff8516d96b34c3c614f1
 * CTX 0699c9abc4ed0411506223bea16af35c883accdbe1104eef0cfdb54e12fb230a
 * Plaintext length (bytes): 512
 */
static uint8_t v8_key1[16] = {
	0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
	0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26
};

static uint8_t v8_key2[16] = {
	0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
	0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95
};

static uint8_t v8_TW[16] = {
	0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v8_PTX[512] = {
	0xd5, 0x5f, 0x68, 0x4f, 0x81, 0xf4, 0x42, 0x6e,
	0x9f, 0xde, 0x92, 0xa5, 0xff, 0x02, 0xdf, 0x2a,
	0xc8, 0x96, 0xaf, 0x63, 0x96, 0x28, 0x88, 0xa9,
	0x79, 0x10, 0xc1, 0x37, 0x9e, 0x20, 0xb0, 0xa3,
	0xb1, 0xdb, 0x61, 0x3f, 0xb7, 0xfe, 0x2e, 0x07,
	0x00, 0x43, 0x29, 0xea, 0x5c, 0x22, 0xbf, 0xd3,
	0x3e, 0x3d, 0xbe, 0x4c, 0xf5, 0x8c, 0xc6, 0x08,
	0xc2, 0xc2, 0x6c, 0x19, 0xa2, 0xe2, 0xfe, 0x22,
	0xf9, 0x87, 0x32, 0xc2, 0xb5, 0xcb, 0x84, 0x4c,
	0xc6, 0xc0, 0x70, 0x2d, 0x91, 0xe1, 0xd5, 0x0f,
	0xc4, 0x38, 0x2a, 0x7e, 0xba, 0x56, 0x35, 0xcd,
	0x60, 0x24, 0x32, 0xa2, 0x30, 0x6a, 0xc4, 0xce,
	0x82, 0xf8, 0xd7, 0x0c, 0x8d, 0x9b, 0xc1, 0x5f,
	0x91, 0x8f, 0xe7, 0x1e, 0x74, 0xc6, 0x22, 0xd5,
	0xcf, 0x71, 0x17, 0x8b, 0xf6, 0xe0, 0xb9, 0xcc,
	0x9f, 0x2b, 0x41, 0xdd, 0x8d, 0xbe, 0x44, 0x1c,
	0x41, 0xcd, 0x0c, 0x73, 0xa6, 0xdc, 0x47, 0xa3,
	0x48, 0xf6, 0x70, 0x2f, 0x9d, 0x0e, 0x9b, 0x1b,
	0x14, 0x31, 0xe9, 0x48, 0xe2, 0x99, 0xb9, 0xec,
	0x22, 0x72, 0xab, 0x2c, 0x5f, 0x0c, 0x7b, 0xe8,
	0x6a, 0xff, 0xa5, 0xde, 0xc8, 0x7a, 0x0b, 0xee,
	0x81, 0xd3, 0xd5, 0x00, 0x07, 0xed, 0xaa, 0x2b,
	0xcf, 0xcc, 0xb3, 0x56, 0x05, 0x15, 0x5f, 0xf3,
	0x6e, 0xd8, 0xed, 0xd4, 0xa4, 0x0d, 0xcd, 0x4b,
	0x24, 0x3a, 0xcd, 0x11, 0xb2, 0xb9, 0x87, 0xbd,
	0xbf, 0xaf, 0x91, 0xa7, 0xca, 0xc2, 0x7e, 0x9c,
	0x5a, 0xea, 0x52, 0x5e, 0xe5, 0x3d, 0xe7, 0xb2,
	0xd3, 0x33, 0x2c, 0x86, 0x44, 0x40, 0x2b, 0x82,
	0x3e, 0x94, 0xa7, 0xdb, 0x26, 0x27, 0x6d, 0x2d,
	0x23, 0xaa, 0x07, 0x18, 0x0f, 0x76, 0xb4, 0xfd,
	0x29, 0xb9, 0xc0, 0x82, 0x30, 0x99, 0xc9, 0xd6,
	0x2c, 0x51, 0x98, 0x80, 0xae, 0xe7, 0xe9, 0x69,
	0x76, 0x17, 0xc1, 0x49, 0x7d, 0x47, 0xbf, 0x3e,
	0x57, 0x19, 0x50, 0x31, 0x14, 0x21, 0xb6, 0xb7,
	0x34, 0xd3, 0x8b, 0x0d, 0xb9, 0x1e, 0xb8, 0x53,
	0x31, 0xb9, 0x1e, 0xa9, 0xf6, 0x15, 0x30, 0xf5,
	0x45, 0x12, 0xa5, 0xa5, 0x2a, 0x4b, 0xad, 0x58,
	0x9e, 0xb6, 0x97, 0x81, 0xd5, 0x37, 0xf2, 0x32,
	0x97, 0xbb, 0x45, 0x9b, 0xda, 0xd2, 0x94, 0x8a,
	0x29, 0xe1, 0x55, 0x0b, 0xf4, 0x78, 0x7e, 0x0b,
	0xe9, 0x5b, 0xb1, 0x73, 0xcf, 0x5f, 0xab, 0x17,
	0xda, 0xb7, 0xa1, 0x3a, 0x05, 0x2a, 0x63, 0x45,
	0x3d, 0x97, 0xcc, 0xec, 0x1a, 0x32, 0x19, 0x54,
	0x88, 0x6b, 0x7a, 0x12, 0x99, 0xfa, 0xae, 0xec,
	0xae, 0x35, 0xc6, 0xea, 0xac, 0xa7, 0x53, 0xb0,
	0x41, 0xb5, 0xe5, 0xf0, 0x93, 0xbf, 0x83, 0x39,
	0x7f, 0xd2, 0x1d, 0xd6, 0xb3, 0x01, 0x20, 0x66,
	0xfc, 0xc0, 0x58, 0xcc, 0x32, 0xc3, 0xb0, 0x9d,
	0x75, 0x62, 0xde, 0xe2, 0x95, 0x09, 0xb5, 0x83,
	0x93, 0x92, 0xc9, 0xff, 0x05, 0xf5, 0x1f, 0x31,
	0x66, 0xaa, 0xac, 0x4a, 0xc5, 0xf2, 0x38, 0x03,
	0x8a, 0x30, 0x45, 0xe6, 0xf7, 0x2e, 0x48, 0xef,
	0x0f, 0xe8, 0xbc, 0x67, 0x5e, 0x82, 0xc3, 0x18,
	0xa2, 0x68, 0xe4, 0x39, 0x70, 0x27, 0x1b, 0xf1,
	0x19, 0xb8, 0x1b, 0xf6, 0xa9, 0x82, 0x74, 0x65,
	0x54, 0xf8, 0x4e, 0x72, 0xb9, 0xf0, 0x02, 0x80,
	0xa3, 0x20, 0xa0, 0x81, 0x42, 0x92, 0x3c, 0x23,
	0xc8, 0x83, 0x42, 0x3f, 0xf9, 0x49, 0x82, 0x7f,
	0x29, 0xbb, 0xac, 0xdc, 0x1c, 0xcd, 0xb0, 0x49,
	0x38, 0xce, 0x60, 0x98, 0xc9, 0x5b, 0xa6, 0xb3,
	0x25, 0x28, 0xf4, 0xef, 0x78, 0xee, 0xd7, 0x78,
	0xb2, 0xe1, 0x22, 0xdd, 0xfd, 0x1c, 0xbd, 0xd1,
	0x1d, 0x1c, 0x0a, 0x67, 0x83, 0xe0, 0x11, 0xfc,
	0x53, 0x6d, 0x63, 0xd0, 0x53, 0x26, 0x06, 0x37
};

static uint8_t v8_CTX[512] = {
	0x72, 0xef, 0xc1, 0xeb, 0xfe, 0x1e, 0xe2, 0x59,
	0x75, 0xa6, 0xeb, 0x3a, 0xa8, 0x58, 0x9d, 0xda,
	0x2b, 0x26, 0x1f, 0x1c, 0x85, 0xbd, 0xab, 0x44,
	0x2a, 0x9e, 0x5b, 0x2d, 0xd1, 0xd7, 0xc3, 0x95,
	0x7a, 0x16, 0xfc, 0x08, 0xe5, 0x26, 0xd4, 0xb1,
	0x22, 0x3f, 0x1b, 0x12, 0x32, 0xa1, 0x1a, 0xf2,
	0x74, 0xc3, 0xd7, 0x0d, 0xac, 0x57, 0xf8, 0x3e,
	0x09, 0x83, 0xc4, 0x98, 0xf1, 0xa6, 0xf1, 0xae,
	0xcb, 0x02, 0x1c, 0x3e, 0x70, 0x08, 0x5a, 0x1e,
	0x52, 0x7f, 0x1c, 0xe4, 0x1e, 0xe5, 0x91, 0x1a,
	0x82, 0x02, 0x01, 0x61, 0x52, 0x9c, 0xd8, 0x27,
	0x73, 0x76, 0x2d, 0xaf, 0x54, 0x59, 0xde, 0x94,
	0xa0, 0xa8, 0x2a, 0xda, 0xe7, 0xe1, 0x70, 0x3c,
	0x80, 0x85, 0x43, 0xc2, 0x9e, 0xd6, 0xfb, 0x32,
	0xd9, 0xe0, 0x04, 0x32, 0x7c, 0x13, 0x55, 0x18,
	0x0c, 0x99, 0x5a, 0x07, 0x74, 0x14, 0x93, 0xa0,
	0x9c, 0x21, 0xba, 0x01, 0xa3, 0x87, 0x88, 0x2d,
	0xa4, 0xf6, 0x25, 0x34, 0xb8, 0x7b, 0xb1, 0x5d,
	0x60, 0xd1, 0x97, 0x20, 0x1c, 0x0f, 0xd3, 0xbf,
	0x30, 0xc1, 0x50, 0x0a, 0x3e, 0xcf, 0xec, 0xdd,
	0x66, 0xd8, 0x72, 0x1f, 0x90, 0xbc, 0xc4, 0xc1,
	0x7e, 0xe9, 0x25, 0xc6, 0x1b, 0x0a, 0x03, 0x72,
	0x7a, 0x9c, 0x0d, 0x5f, 0x5c, 0xa4, 0x62, 0xfb,
	0xfa, 0x0a, 0xf1, 0xc2, 0x51, 0x3a, 0x9d, 0x9d,
	0x4b, 0x53, 0x45, 0xbd, 0x27, 0xa5, 0xf6, 0xe6,
	0x53, 0xf7, 0x51, 0x69, 0x3e, 0x6b, 0x6a, 0x2b,
	0x8e, 0xad, 0x57, 0xd5, 0x11, 0xe0, 0x0e, 0x58,
	0xc4, 0x5b, 0x7b, 0x8d, 0x00, 0x5a, 0xf7, 0x92,
	0x88, 0xf5, 0xc7, 0xc2, 0x2f, 0xd4, 0xf1, 0xbf,
	0x7a, 0x89, 0x8b, 0x03, 0xa5, 0x63, 0x4c, 0x6a,
	0x1a, 0xe3, 0xf9, 0xfa, 0xe5, 0xde, 0x4f, 0x29,
	0x6a, 0x28, 0x96, 0xb2, 0x3e, 0x7e, 0xd4, 0x3e,
	0xd1, 0x4f, 0xa5, 0xa2, 0x80, 0x3f, 0x4d, 0x28,
	0xf0, 0xd3, 0xff, 0xcf, 0x24, 0x75, 0x76, 0x77,
	0xae, 0xbd, 0xb4, 0x7b, 0xb3, 0x88, 0x37, 0x87,
	0x08, 0x94, 0x8a, 0x8d, 0x41, 0x26, 0xed, 0x18,
	0x39, 0xe0, 0xda, 0x29, 0xa5, 0x37, 0xa8, 0xc1,
	0x98, 0xb3, 0xc6, 0x6a, 0xb0, 0x07, 0x12, 0xdd,
	0x26, 0x16, 0x74, 0xbf, 0x45, 0xa7, 0x3d, 0x67,
	0xf7, 0x69, 0x14, 0xf8, 0x30, 0xca, 0x01, 0x4b,
	0x65, 0x59, 0x6f, 0x27, 0xe4, 0xcf, 0x62, 0xde,
	0x66, 0x12, 0x5a, 0x55, 0x66, 0xdf, 0x99, 0x75,
	0x15, 0x56, 0x28, 0xb4, 0x00, 0xfb, 0xfb, 0x3a,
	0x29, 0x04, 0x0e, 0xd5, 0x0f, 0xaf, 0xfd, 0xbb,
	0x18, 0xae, 0xce, 0x7c, 0x5c, 0x44, 0x69, 0x32,
	0x60, 0xaa, 0xb3, 0x86, 0xc0, 0xa3, 0x7b, 0x11,
	0xb1, 0x14, 0xf1, 0xc4, 0x15, 0xae, 0xbb, 0x65,
	0x3b, 0xe4, 0x68, 0x17, 0x94, 0x28, 0xd4, 0x3a,
	0x4d, 0x8b, 0xc3, 0xec, 0x38, 0x81, 0x3e, 0xca,
	0x30, 0xa1, 0x3c, 0xf1, 0xbb, 0x18, 0xd5, 0x24,
	0xf1, 0x99, 0x2d, 0x44, 0xd8, 0xb1, 0xa4, 0x2e,
	0xa3, 0x0b, 0x22, 0xe6, 0xc9, 0x5b, 0x19, 0x9d,
	0x8d, 0x18, 0x2f, 0x88, 0x40, 0xb0, 0x9d, 0x05,
	0x95, 0x85, 0xc3, 0x1a, 0xd6, 0x91, 0xfa, 0x06,
	0x19, 0xff, 0x03, 0x8a, 0xca, 0x2c, 0x39, 0xa9,
	0x43, 0x42, 0x11, 0x57, 0x36, 0x17, 0x17, 0xc4,
	0x9d, 0x32, 0x20, 0x28, 0xa7, 0x46, 0x48, 0x11,
	0x3b, 0xd8, 0xc9, 0xd7, 0xec, 0x77, 0xcf, 0x3c,
	0x89, 0xc1, 0xec, 0x87, 0x18, 0xce, 0xff, 0x85,
	0x16, 0xd9, 0x6b, 0x34, 0xc3, 0xc6, 0x14, 0xf1,
	0x06, 0x99, 0xc9, 0xab, 0xc4, 0xed, 0x04, 0x11,
	0x50, 0x62, 0x23, 0xbe, 0xa1, 0x6a, 0xf3, 0x5c,
	0x88, 0x3a, 0xcc, 0xdb, 0xe1, 0x10, 0x4e, 0xef,
	0x0c, 0xfd, 0xb5, 0x4e, 0x12, 0xfb, 0x23, 0x0a
};

/*
 * Vector 9
 * Key1 27182818284590452353602874713526
 * Key2 31415926535897932384626433832795
 * Data Unit Sequence Number ff
 * PTX 72efc1ebfe1ee25975a6eb3aa8589dda2b261f1c85bdab442a9e5b2dd1d7c395
 * PTX 7a16fc08e526d4b1223f1b1232a11af274c3d70dac57f83e0983c498f1a6f1ae
 * PTX cb021c3e70085a1e527f1ce41ee5911a82020161529cd82773762daf5459de94
 * PTX a0a82adae7e1703c808543c29ed6fb32d9e004327c1355180c995a07741493a0
 * PTX 9c21ba01a387882da4f62534b87bb15d60d197201c0fd3bf30c1500a3ecfecdd
 * PTX 66d8721f90bcc4c17ee925c61b0a03727a9c0d5f5ca462fbfa0af1c2513a9d9d
 * PTX 4b5345bd27a5f6e653f751693e6b6a2b8ead57d511e00e58c45b7b8d005af792
 * PTX 88f5c7c22fd4f1bf7a898b03a5634c6a1ae3f9fae5de4f296a2896b23e7ed43e
 * PTX d14fa5a2803f4d28f0d3ffcf24757677aebdb47bb388378708948a8d4126ed18
 * PTX 39e0da29a537a8c198b3c66ab00712dd261674bf45a73d67f76914f830ca014b
 * PTX 65596f27e4cf62de66125a5566df9975155628b400fbfb3a29040ed50faffdbb
 * PTX 18aece7c5c44693260aab386c0a37b11b114f1c415aebb653be468179428d43a
 * PTX 4d8bc3ec38813eca30a13cf1bb18d524f1992d44d8b1a42ea30b22e6c95b199d
 * PTX 8d182f8840b09d059585c31ad691fa0619ff038aca2c39a943421157361717c4
 * PTX 9d322028a74648113bd8c9d7ec77cf3c89c1ec8718ceff8516d96b34c3c614f1
 * PTX 0699c9abc4ed0411506223bea16af35c883accdbe1104eef0cfdb54e12fb230a
 * CTX 3260ae8dad1f4a32c5cafe3ab0eb95549d461a67ceb9e5aa2d3afb62dece0553
 * CTX 193ba50c75be251e08d1d08f1088576c7efdfaaf3f459559571e12511753b07a
 * CTX f073f35da06af0ce0bbf6b8f5ccc5cea500ec1b211bd51f63b606bf6528796ca
 * CTX 12173ba39b8935ee44ccce646f90a45bf9ccc567f0ace13dc2d53ebeedc81f58
 * CTX b2e41179dddf0d5a5c42f5d8506c1a5d2f8f59f3ea873cbcd0eec19acbf32542
 * CTX 3bd3dcb8c2b1bf1d1eaed0eba7f0698e4314fbeb2f1566d1b9253008cbccf45a
 * CTX 2b0d9c5c9c21474f4076e02be26050b99dee4fd68a4cf890e496e4fcae7b70f9
 * CTX 4ea5a9062da0daeba1993d2ccd1dd3c244b8428801495a58b216547e7e847c46
 * CTX d1d756377b6242d2e5fb83bf752b54e0df71e889f3a2bb0f4c10805bf3c59037
 * CTX 6e3c24e22ff57f7fa965577375325cea5d920db94b9c336b455f6e894c01866f
 * CTX e9fbb8c8d3f70a2957285f6dfb5dcd8cbf54782f8fe7766d4723819913ac7734
 * CTX 21e3a31095866bad22c86a6036b2518b2059b4229d18c8c2ccbdf906c6cc6e82
 * CTX 464ee57bddb0bebcb1dc645325bfb3e665ef7251082c88ebb1cf203bd779fdd3
 * CTX 8675713c8daadd17e1cabee432b09787b6ddf3304e38b731b45df5df51b78fcf
 * CTX b3d32466028d0ba36555e7e11ab0ee0666061d1645d962444bc47a38188930a8
 * CTX 4b4d561395c73c087021927ca638b7afc8a8679ccb84c26555440ec7f10445cd
 * Plaintext length (bytes): 512
 */
static uint8_t v9_key1[16] = {
	0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
	0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26
};

static uint8_t v9_key2[16] = {
	0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
	0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95
};

static uint8_t v9_TW[16] = {
	0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v9_PTX[512] = {
	0x72, 0xef, 0xc1, 0xeb, 0xfe, 0x1e, 0xe2, 0x59,
	0x75, 0xa6, 0xeb, 0x3a, 0xa8, 0x58, 0x9d, 0xda,
	0x2b, 0x26, 0x1f, 0x1c, 0x85, 0xbd, 0xab, 0x44,
	0x2a, 0x9e, 0x5b, 0x2d, 0xd1, 0xd7, 0xc3, 0x95,
	0x7a, 0x16, 0xfc, 0x08, 0xe5, 0x26, 0xd4, 0xb1,
	0x22, 0x3f, 0x1b, 0x12, 0x32, 0xa1, 0x1a, 0xf2,
	0x74, 0xc3, 0xd7, 0x0d, 0xac, 0x57, 0xf8, 0x3e,
	0x09, 0x83, 0xc4, 0x98, 0xf1, 0xa6, 0xf1, 0xae,
	0xcb, 0x02, 0x1c, 0x3e, 0x70, 0x08, 0x5a, 0x1e,
	0x52, 0x7f, 0x1c, 0xe4, 0x1e, 0xe5, 0x91, 0x1a,
	0x82, 0x02, 0x01, 0x61, 0x52, 0x9c, 0xd8, 0x27,
	0x73, 0x76, 0x2d, 0xaf, 0x54, 0x59, 0xde, 0x94,
	0xa0, 0xa8, 0x2a, 0xda, 0xe7, 0xe1, 0x70, 0x3c,
	0x80, 0x85, 0x43, 0xc2, 0x9e, 0xd6, 0xfb, 0x32,
	0xd9, 0xe0, 0x04, 0x32, 0x7c, 0x13, 0x55, 0x18,
	0x0c, 0x99, 0x5a, 0x07, 0x74, 0x14, 0x93, 0xa0,
	0x9c, 0x21, 0xba, 0x01, 0xa3, 0x87, 0x88, 0x2d,
	0xa4, 0xf6, 0x25, 0x34, 0xb8, 0x7b, 0xb1, 0x5d,
	0x60, 0xd1, 0x97, 0x20, 0x1c, 0x0f, 0xd3, 0xbf,
	0x30, 0xc1, 0x50, 0x0a, 0x3e, 0xcf, 0xec, 0xdd,
	0x66, 0xd8, 0x72, 0x1f, 0x90, 0xbc, 0xc4, 0xc1,
	0x7e, 0xe9, 0x25, 0xc6, 0x1b, 0x0a, 0x03, 0x72,
	0x7a, 0x9c, 0x0d, 0x5f, 0x5c, 0xa4, 0x62, 0xfb,
	0xfa, 0x0a, 0xf1, 0xc2, 0x51, 0x3a, 0x9d, 0x9d,
	0x4b, 0x53, 0x45, 0xbd, 0x27, 0xa5, 0xf6, 0xe6,
	0x53, 0xf7, 0x51, 0x69, 0x3e, 0x6b, 0x6a, 0x2b,
	0x8e, 0xad, 0x57, 0xd5, 0x11, 0xe0, 0x0e, 0x58,
	0xc4, 0x5b, 0x7b, 0x8d, 0x00, 0x5a, 0xf7, 0x92,
	0x88, 0xf5, 0xc7, 0xc2, 0x2f, 0xd4, 0xf1, 0xbf,
	0x7a, 0x89, 0x8b, 0x03, 0xa5, 0x63, 0x4c, 0x6a,
	0x1a, 0xe3, 0xf9, 0xfa, 0xe5, 0xde, 0x4f, 0x29,
	0x6a, 0x28, 0x96, 0xb2, 0x3e, 0x7e, 0xd4, 0x3e,
	0xd1, 0x4f, 0xa5, 0xa2, 0x80, 0x3f, 0x4d, 0x28,
	0xf0, 0xd3, 0xff, 0xcf, 0x24, 0x75, 0x76, 0x77,
	0xae, 0xbd, 0xb4, 0x7b, 0xb3, 0x88, 0x37, 0x87,
	0x08, 0x94, 0x8a, 0x8d, 0x41, 0x26, 0xed, 0x18,
	0x39, 0xe0, 0xda, 0x29, 0xa5, 0x37, 0xa8, 0xc1,
	0x98, 0xb3, 0xc6, 0x6a, 0xb0, 0x07, 0x12, 0xdd,
	0x26, 0x16, 0x74, 0xbf, 0x45, 0xa7, 0x3d, 0x67,
	0xf7, 0x69, 0x14, 0xf8, 0x30, 0xca, 0x01, 0x4b,
	0x65, 0x59, 0x6f, 0x27, 0xe4, 0xcf, 0x62, 0xde,
	0x66, 0x12, 0x5a, 0x55, 0x66, 0xdf, 0x99, 0x75,
	0x15, 0x56, 0x28, 0xb4, 0x00, 0xfb, 0xfb, 0x3a,
	0x29, 0x04, 0x0e, 0xd5, 0x0f, 0xaf, 0xfd, 0xbb,
	0x18, 0xae, 0xce, 0x7c, 0x5c, 0x44, 0x69, 0x32,
	0x60, 0xaa, 0xb3, 0x86, 0xc0, 0xa3, 0x7b, 0x11,
	0xb1, 0x14, 0xf1, 0xc4, 0x15, 0xae, 0xbb, 0x65,
	0x3b, 0xe4, 0x68, 0x17, 0x94, 0x28, 0xd4, 0x3a,
	0x4d, 0x8b, 0xc3, 0xec, 0x38, 0x81, 0x3e, 0xca,
	0x30, 0xa1, 0x3c, 0xf1, 0xbb, 0x18, 0xd5, 0x24,
	0xf1, 0x99, 0x2d, 0x44, 0xd8, 0xb1, 0xa4, 0x2e,
	0xa3, 0x0b, 0x22, 0xe6, 0xc9, 0x5b, 0x19, 0x9d,
	0x8d, 0x18, 0x2f, 0x88, 0x40, 0xb0, 0x9d, 0x05,
	0x95, 0x85, 0xc3, 0x1a, 0xd6, 0x91, 0xfa, 0x06,
	0x19, 0xff, 0x03, 0x8a, 0xca, 0x2c, 0x39, 0xa9,
	0x43, 0x42, 0x11, 0x57, 0x36, 0x17, 0x17, 0xc4,
	0x9d, 0x32, 0x20, 0x28, 0xa7, 0x46, 0x48, 0x11,
	0x3b, 0xd8, 0xc9, 0xd7, 0xec, 0x77, 0xcf, 0x3c,
	0x89, 0xc1, 0xec, 0x87, 0x18, 0xce, 0xff, 0x85,
	0x16, 0xd9, 0x6b, 0x34, 0xc3, 0xc6, 0x14, 0xf1,
	0x06, 0x99, 0xc9, 0xab, 0xc4, 0xed, 0x04, 0x11,
	0x50, 0x62, 0x23, 0xbe, 0xa1, 0x6a, 0xf3, 0x5c,
	0x88, 0x3a, 0xcc, 0xdb, 0xe1, 0x10, 0x4e, 0xef,
	0x0c, 0xfd, 0xb5, 0x4e, 0x12, 0xfb, 0x23, 0x0a
};

static uint8_t v9_CTX[512] = {
	0x32, 0x60, 0xae, 0x8d, 0xad, 0x1f, 0x4a, 0x32,
	0xc5, 0xca, 0xfe, 0x3a, 0xb0, 0xeb, 0x95, 0x54,
	0x9d, 0x46, 0x1a, 0x67, 0xce, 0xb9, 0xe5, 0xaa,
	0x2d, 0x3a, 0xfb, 0x62, 0xde, 0xce, 0x05, 0x53,
	0x19, 0x3b, 0xa5, 0x0c, 0x75, 0xbe, 0x25, 0x1e,
	0x08, 0xd1, 0xd0, 0x8f, 0x10, 0x88, 0x57, 0x6c,
	0x7e, 0xfd, 0xfa, 0xaf, 0x3f, 0x45, 0x95, 0x59,
	0x57, 0x1e, 0x12, 0x51, 0x17, 0x53, 0xb0, 0x7a,
	0xf0, 0x73, 0xf3, 0x5d, 0xa0, 0x6a, 0xf0, 0xce,
	0x0b, 0xbf, 0x6b, 0x8f, 0x5c, 0xcc, 0x5c, 0xea,
	0x50, 0x0e, 0xc1, 0xb2, 0x11, 0xbd, 0x51, 0xf6,
	0x3b, 0x60, 0x6b, 0xf6, 0x52, 0x87, 0x96, 0xca,
	0x12, 0x17, 0x3b, 0xa3, 0x9b, 0x89, 0x35, 0xee,
	0x44, 0xcc, 0xce, 0x64, 0x6f, 0x90, 0xa4, 0x5b,
	0xf9, 0xcc, 0xc5, 0x67, 0xf0, 0xac, 0xe1, 0x3d,
	0xc2, 0xd5, 0x3e, 0xbe, 0xed, 0xc8, 0x1f, 0x58,
	0xb2, 0xe4, 0x11, 0x79, 0xdd, 0xdf, 0x0d, 0x5a,
	0x5c, 0x42, 0xf5, 0xd8, 0x50, 0x6c, 0x1a, 0x5d,
	0x2f, 0x8f, 0x59, 0xf3, 0xea, 0x87, 0x3c, 0xbc,
	0xd0, 0xee, 0xc1, 0x9a, 0xcb, 0xf3, 0x25, 0x42,
	0x3b, 0xd3, 0xdc, 0xb8, 0xc2, 0xb1, 0xbf, 0x1d,
	0x1e, 0xae, 0xd0, 0xeb, 0xa7, 0xf0, 0x69, 0x8e,
	0x43, 0x14, 0xfb, 0xeb, 0x2f, 0x15, 0x66, 0xd1,
	0xb9, 0x25, 0x30, 0x08, 0xcb, 0xcc, 0xf4, 0x5a,
	0x2b, 0x0d, 0x9c, 0x5c, 0x9c, 0x21, 0x47, 0x4f,
	0x40, 0x76, 0xe0, 0x2b, 0xe2, 0x60, 0x50, 0xb9,
	0x9d, 0xee, 0x4f, 0xd6, 0x8a, 0x4c, 0xf8, 0x90,
	0xe4, 0x96, 0xe4, 0xfc, 0xae, 0x7b, 0x70, 0xf9,
	0x4e, 0xa5, 0xa9, 0x06, 0x2d, 0xa0, 0xda, 0xeb,
	0xa1, 0x99, 0x3d, 0x2c, 0xcd, 0x1d, 0xd3, 0xc2,
	0x44, 0xb8, 0x42, 0x88, 0x01, 0x49, 0x5a, 0x58,
	0xb2, 0x16, 0x54, 0x7e, 0x7e, 0x84, 0x7c, 0x46,
	0xd1, 0xd7, 0x56, 0x37, 0x7b, 0x62, 0x42, 0xd2,
	0xe5, 0xfb, 0x83, 0xbf, 0x75, 0x2b, 0x54, 0xe0,
	0xdf, 0x71, 0xe8, 0x89, 0xf3, 0xa2, 0xbb, 0x0f,
	0x4c, 0x10, 0x80, 0x5b, 0xf3, 0xc5, 0x90, 0x37,
	0x6e, 0x3c, 0x24, 0xe2, 0x2f, 0xf5, 0x7f, 0x7f,
	0xa9, 0x65, 0x57, 0x73, 0x75, 0x32, 0x5c, 0xea,
	0x5d, 0x92, 0x0d, 0xb9, 0x4b, 0x9c, 0x33, 0x6b,
	0x45, 0x5f, 0x6e, 0x89, 0x4c, 0x01, 0x86, 0x6f,
	0xe9, 0xfb, 0xb8, 0xc8, 0xd3, 0xf7, 0x0a, 0x29,
	0x57, 0x28, 0x5f, 0x6d, 0xfb, 0x5d, 0xcd, 0x8c,
	0xbf, 0x54, 0x78, 0x2f, 0x8f, 0xe7, 0x76, 0x6d,
	0x47, 0x23, 0x81, 0x99, 0x13, 0xac, 0x77, 0x34,
	0x21, 0xe3, 0xa3, 0x10, 0x95, 0x86, 0x6b, 0xad,
	0x22, 0xc8, 0x6a, 0x60, 0x36, 0xb2, 0x51, 0x8b,
	0x20, 0x59, 0xb4, 0x22, 0x9d, 0x18, 0xc8, 0xc2,
	0xcc, 0xbd, 0xf9, 0x06, 0xc6, 0xcc, 0x6e, 0x82,
	0x46, 0x4e, 0xe5, 0x7b, 0xdd, 0xb0, 0xbe, 0xbc,
	0xb1, 0xdc, 0x64, 0x53, 0x25, 0xbf, 0xb3, 0xe6,
	0x65, 0xef, 0x72, 0x51, 0x08, 0x2c, 0x88, 0xeb,
	0xb1, 0xcf, 0x20, 0x3b, 0xd7, 0x79, 0xfd, 0xd3,
	0x86, 0x75, 0x71, 0x3c, 0x8d, 0xaa, 0xdd, 0x17,
	0xe1, 0xca, 0xbe, 0xe4, 0x32, 0xb0, 0x97, 0x87,
	0xb6, 0xdd, 0xf3, 0x30, 0x4e, 0x38, 0xb7, 0x31,
	0xb4, 0x5d, 0xf5, 0xdf, 0x51, 0xb7, 0x8f, 0xcf,
	0xb3, 0xd3, 0x24, 0x66, 0x02, 0x8d, 0x0b, 0xa3,
	0x65, 0x55, 0xe7, 0xe1, 0x1a, 0xb0, 0xee, 0x06,
	0x66, 0x06, 0x1d, 0x16, 0x45, 0xd9, 0x62, 0x44,
	0x4b, 0xc4, 0x7a, 0x38, 0x18, 0x89, 0x30, 0xa8,
	0x4b, 0x4d, 0x56, 0x13, 0x95, 0xc7, 0x3c, 0x08,
	0x70, 0x21, 0x92, 0x7c, 0xa6, 0x38, 0xb7, 0xaf,
	0xc8, 0xa8, 0x67, 0x9c, 0xcb, 0x84, 0xc2, 0x65,
	0x55, 0x44, 0x0e, 0xc7, 0xf1, 0x04, 0x45, 0xcd
};

/*
 * Vector 15
 * Key1 fffefdfcfbfaf9f8f7f6f5f4f3f2f1f0
 * Key2 bfbebdbcbbbab9b8b7b6b5b4b3b2b1b0
 * Data unit sequence number 9a78563412
 * PTX 000102030405060708090a0b0c0d0e0f10
 * CTX 6c1625db4671522d3d7599601de7ca09ed
 * Plaintext length (bytes): 17
 */

static uint8_t v15_key1[16] = {
	0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
	0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0
};

static uint8_t v15_key2[16] = {
	0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8,
	0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0
};

static uint8_t v15_TW[16] = {
	0x9a, 0x78, 0x56, 0x34, 0x12, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v15_PTX[17] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10
};

static uint8_t v15_CTX[17] = {
	0x6c, 0x16, 0x25, 0xdb, 0x46, 0x71, 0x52, 0x2d,
	0x3d, 0x75, 0x99, 0x60, 0x1d, 0xe7, 0xca, 0x09,
	0xed
};

/*
 * Vector 16
 * Key1 fffefdfcfbfaf9f8f7f6f5f4f3f2f1f0
 * Key2 bfbebdbcbbbab9b8b7b6b5b4b3b2b1b0
 * Data unit sequence number 9a78563412
 * PTX 000102030405060708090a0b0c0d0e0f1011
 * CTX d069444b7a7e0cab09e24447d24deb1fedbf
 * Plaintext length (bytes): 18
 */
static uint8_t v16_key1[16] = {
	0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
	0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0
};

static uint8_t v16_key2[16] = {
	0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8,
	0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0
};

static uint8_t v16_TW[16] = {
	0x9a, 0x78, 0x56, 0x34, 0x12, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v16_PTX[18] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11
};

static uint8_t v16_CTX[18] = {
	0xd0, 0x69, 0x44, 0x4b, 0x7a, 0x7e, 0x0c, 0xab,
	0x09, 0xe2, 0x44, 0x47, 0xd2, 0x4d, 0xeb, 0x1f,
	0xed, 0xbf
};

/*
 * Vector 17
 * Key1 fffefdfcfbfaf9f8f7f6f5f4f3f2f1f0
 * Key2 bfbebdbcbbbab9b8b7b6b5b4b3b2b1b0
 * Data unit sequence number 9a78563412
 * PTX 000102030405060708090a0b0c0d0e0f101112
 * CTX e5df1351c0544ba1350b3363cd8ef4beedbf9d
 * Plaintext length (bytes): 19
 */

static uint8_t v17_key1[16] = {
	0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
	0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0
};

static uint8_t v17_key2[16] = {
	0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8,
	0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0
};

static uint8_t v17_TW[16] = {
	0x9a, 0x78, 0x56, 0x34, 0x12, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v17_PTX[19] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12
};

static uint8_t v17_CTX[19] = {
	0xe5, 0xdf, 0x13, 0x51, 0xc0, 0x54, 0x4b, 0xa1,
	0x35, 0x0b, 0x33, 0x63, 0xcd, 0x8e, 0xf4, 0xbe,
	0xed, 0xbf, 0x9d
};

/*
 * Vector 18
 * Key1 fffefdfcfbfaf9f8f7f6f5f4f3f2f1f0
 * Key2 bfbebdbcbbbab9b8b7b6b5b4b3b2b1b0
 * Data unit sequence number 9a78563412
 * PTX 000102030405060708090a0b0c0d0e0f10111213
 * CTX 9d84c813f719aa2c7be3f66171c7c5c2edbf9dac
 * Plaintext length (bytes): 20
 */

static uint8_t v18_key1[16] = {
	0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
	0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0
};

static uint8_t v18_key2[16] = {
	0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8,
	0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0
};

static uint8_t v18_TW[16] = {
	0x9a, 0x78, 0x56, 0x34, 0x12, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v18_PTX[20] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13
};

static uint8_t v18_CTX[20] = {
	0x9d, 0x84, 0xc8, 0x13, 0xf7, 0x19, 0xaa, 0x2c,
	0x7b, 0xe3, 0xf6, 0x61, 0x71, 0xc7, 0xc5, 0xc2,
	0xed, 0xbf, 0x9d, 0xac
};

/*
 * Vector 19
 * Key1 e0e1e2e3e4e5e6e7e8e9eaebecedeeef
 * Key2 c0c1c2c3c4c5c6c7c8c9cacbcccdcecf
 * Data unit sequence number 21436587a9
 * PTX 000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f
 * PTX 202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f
 * PTX 404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f
 * PTX 606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f
 * PTX 808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f
 * PTX a0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebf
 * PTX c0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedf
 * PTX e0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff
 * PTX 000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f
 * PTX 202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f
 * PTX 404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f
 * PTX 606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f
 * PTX 808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f
 * PTX a0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebf
 * PTX c0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedf
 * PTX e0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff
 * CTX 38b45812ef43a05bd957e545907e223b954ab4aaf088303ad910eadf14b42be6
 * CTX 8b2461149d8c8ba85f992be970bc621f1b06573f63e867bf5875acafa04e42cc
 * CTX bd7bd3c2a0fb1fff791ec5ec36c66ae4ac1e806d81fbf709dbe29e471fad3854
 * CTX 9c8e66f5345d7c1eb94f405d1ec785cc6f6a68f6254dd8339f9d84057e01a177
 * CTX 41990482999516b5611a38f41bb6478e6f173f320805dd71b1932fc333cb9ee3
 * CTX 9936beea9ad96fa10fb4112b901734ddad40bc1878995f8e11aee7d141a2f5d4
 * CTX 8b7a4e1e7f0b2c04830e69a4fd1378411c2f287edf48c6c4e5c247a19680f7fe
 * CTX 41cefbd49b582106e3616cbbe4dfb2344b2ae9519391f3e0fb4922254b1d6d2d
 * CTX 19c6d4d537b3a26f3bcc51588b32f3eca0829b6a5ac72578fb814fb43cf80d64
 * CTX a233e3f997a3f02683342f2b33d25b492536b93becb2f5e1a8b82f5b88334272
 * CTX 9e8ae09d16938841a21a97fb543eea3bbff59f13c1a18449e398701c1ad51648
 * CTX 346cbc04c27bb2da3b93a1372ccae548fb53bee476f9e9c91773b1bb19828394
 * CTX d55d3e1a20ed69113a860b6829ffa847224604435070221b257e8dff783615d2
 * CTX cae4803a93aa4334ab482a0afac9c0aeda70b45a481df5dec5df8cc0f423c77a
 * CTX 5fd46cd312021d4b438862419a791be03bb4d97c0e59578542531ba466a83baf
 * CTX 92cefc151b5cc1611a167893819b63fb8a6b18e86de60290fa72b797b0ce59f3
 * Plaintext length (bytes): 512
 */
static uint8_t v19_key1[16] = {

	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef
};

static uint8_t v19_key2[16] = {

	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf
};

static uint8_t v19_TW[16] = {

	0x21, 0x43, 0x65, 0x87, 0xa9, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t v19_PTX[512] = {

	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

static uint8_t v19_CTX[512] = {
	0x38, 0xb4, 0x58, 0x12, 0xef, 0x43, 0xa0, 0x5b,
	0xd9, 0x57, 0xe5, 0x45, 0x90, 0x7e, 0x22, 0x3b,
	0x95, 0x4a, 0xb4, 0xaa, 0xf0, 0x88, 0x30, 0x3a,
	0xd9, 0x10, 0xea, 0xdf, 0x14, 0xb4, 0x2b, 0xe6,
	0x8b, 0x24, 0x61, 0x14, 0x9d, 0x8c, 0x8b, 0xa8,
	0x5f, 0x99, 0x2b, 0xe9, 0x70, 0xbc, 0x62, 0x1f,
	0x1b, 0x06, 0x57, 0x3f, 0x63, 0xe8, 0x67, 0xbf,
	0x58, 0x75, 0xac, 0xaf, 0xa0, 0x4e, 0x42, 0xcc,
	0xbd, 0x7b, 0xd3, 0xc2, 0xa0, 0xfb, 0x1f, 0xff,
	0x79, 0x1e, 0xc5, 0xec, 0x36, 0xc6, 0x6a, 0xe4,
	0xac, 0x1e, 0x80, 0x6d, 0x81, 0xfb, 0xf7, 0x09,
	0xdb, 0xe2, 0x9e, 0x47, 0x1f, 0xad, 0x38, 0x54,
	0x9c, 0x8e, 0x66, 0xf5, 0x34, 0x5d, 0x7c, 0x1e,
	0xb9, 0x4f, 0x40, 0x5d, 0x1e, 0xc7, 0x85, 0xcc,
	0x6f, 0x6a, 0x68, 0xf6, 0x25, 0x4d, 0xd8, 0x33,
	0x9f, 0x9d, 0x84, 0x05, 0x7e, 0x01, 0xa1, 0x77,
	0x41, 0x99, 0x04, 0x82, 0x99, 0x95, 0x16, 0xb5,
	0x61, 0x1a, 0x38, 0xf4, 0x1b, 0xb6, 0x47, 0x8e,
	0x6f, 0x17, 0x3f, 0x32, 0x08, 0x05, 0xdd, 0x71,
	0xb1, 0x93, 0x2f, 0xc3, 0x33, 0xcb, 0x9e, 0xe3,
	0x99, 0x36, 0xbe, 0xea, 0x9a, 0xd9, 0x6f, 0xa1,
	0x0f, 0xb4, 0x11, 0x2b, 0x90, 0x17, 0x34, 0xdd,
	0xad, 0x40, 0xbc, 0x18, 0x78, 0x99, 0x5f, 0x8e,
	0x11, 0xae, 0xe7, 0xd1, 0x41, 0xa2, 0xf5, 0xd4,
	0x8b, 0x7a, 0x4e, 0x1e, 0x7f, 0x0b, 0x2c, 0x04,
	0x83, 0x0e, 0x69, 0xa4, 0xfd, 0x13, 0x78, 0x41,
	0x1c, 0x2f, 0x28, 0x7e, 0xdf, 0x48, 0xc6, 0xc4,
	0xe5, 0xc2, 0x47, 0xa1, 0x96, 0x80, 0xf7, 0xfe,
	0x41, 0xce, 0xfb, 0xd4, 0x9b, 0x58, 0x21, 0x06,
	0xe3, 0x61, 0x6c, 0xbb, 0xe4, 0xdf, 0xb2, 0x34,
	0x4b, 0x2a, 0xe9, 0x51, 0x93, 0x91, 0xf3, 0xe0,
	0xfb, 0x49, 0x22, 0x25, 0x4b, 0x1d, 0x6d, 0x2d,
	0x19, 0xc6, 0xd4, 0xd5, 0x37, 0xb3, 0xa2, 0x6f,
	0x3b, 0xcc, 0x51, 0x58, 0x8b, 0x32, 0xf3, 0xec,
	0xa0, 0x82, 0x9b, 0x6a, 0x5a, 0xc7, 0x25, 0x78,
	0xfb, 0x81, 0x4f, 0xb4, 0x3c, 0xf8, 0x0d, 0x64,
	0xa2, 0x33, 0xe3, 0xf9, 0x97, 0xa3, 0xf0, 0x26,
	0x83, 0x34, 0x2f, 0x2b, 0x33, 0xd2, 0x5b, 0x49,
	0x25, 0x36, 0xb9, 0x3b, 0xec, 0xb2, 0xf5, 0xe1,
	0xa8, 0xb8, 0x2f, 0x5b, 0x88, 0x33, 0x42, 0x72,
	0x9e, 0x8a, 0xe0, 0x9d, 0x16, 0x93, 0x88, 0x41,
	0xa2, 0x1a, 0x97, 0xfb, 0x54, 0x3e, 0xea, 0x3b,
	0xbf, 0xf5, 0x9f, 0x13, 0xc1, 0xa1, 0x84, 0x49,
	0xe3, 0x98, 0x70, 0x1c, 0x1a, 0xd5, 0x16, 0x48,
	0x34, 0x6c, 0xbc, 0x04, 0xc2, 0x7b, 0xb2, 0xda,
	0x3b, 0x93, 0xa1, 0x37, 0x2c, 0xca, 0xe5, 0x48,
	0xfb, 0x53, 0xbe, 0xe4, 0x76, 0xf9, 0xe9, 0xc9,
	0x17, 0x73, 0xb1, 0xbb, 0x19, 0x82, 0x83, 0x94,
	0xd5, 0x5d, 0x3e, 0x1a, 0x20, 0xed, 0x69, 0x11,
	0x3a, 0x86, 0x0b, 0x68, 0x29, 0xff, 0xa8, 0x47,
	0x22, 0x46, 0x04, 0x43, 0x50, 0x70, 0x22, 0x1b,
	0x25, 0x7e, 0x8d, 0xff, 0x78, 0x36, 0x15, 0xd2,
	0xca, 0xe4, 0x80, 0x3a, 0x93, 0xaa, 0x43, 0x34,
	0xab, 0x48, 0x2a, 0x0a, 0xfa, 0xc9, 0xc0, 0xae,
	0xda, 0x70, 0xb4, 0x5a, 0x48, 0x1d, 0xf5, 0xde,
	0xc5, 0xdf, 0x8c, 0xc0, 0xf4, 0x23, 0xc7, 0x7a,
	0x5f, 0xd4, 0x6c, 0xd3, 0x12, 0x02, 0x1d, 0x4b,
	0x43, 0x88, 0x62, 0x41, 0x9a, 0x79, 0x1b, 0xe0,
	0x3b, 0xb4, 0xd9, 0x7c, 0x0e, 0x59, 0x57, 0x85,
	0x42, 0x53, 0x1b, 0xa4, 0x66, 0xa8, 0x3b, 0xaf,
	0x92, 0xce, 0xfc, 0x15, 0x1b, 0x5c, 0xc1, 0x61,
	0x1a, 0x16, 0x78, 0x93, 0x81, 0x9b, 0x63, 0xfb,
	0x8a, 0x6b, 0x18, 0xe8, 0x6d, 0xe6, 0x02, 0x90,
	0xfa, 0x72, 0xb7, 0x97, 0xb0, 0xce, 0x59, 0xf3
};

// Define vector of structs, with pointers to the statically defined vectors

struct xts_vector vlist[NVEC] = {

	// pointers to the statically defined vectors here

	// Vector 1
	{sizeof(v1_CTX), v1_key1, v1_key2, v1_TW, v1_PTX, v1_CTX}
	,
	// Vector 2
	{sizeof(v2_CTX), v2_key1, v2_key2, v2_TW, v2_PTX, v2_CTX}
	,
	// Vector 3
	{sizeof(v3_CTX), v3_key1, v3_key2, v3_TW, v3_PTX, v3_CTX}
	,
	// Vector 4
	{sizeof(v4_CTX), v4_key1, v4_key2, v4_TW, v4_PTX, v4_CTX}
	,
	// Vector 5
	{sizeof(v5_CTX), v5_key1, v5_key2, v5_TW, v5_PTX, v5_CTX}
	,
	// Vector 6
	{sizeof(v6_CTX), v6_key1, v6_key2, v6_TW, v6_PTX, v6_CTX}
	,
	// Vector 7
	{sizeof(v7_CTX), v7_key1, v7_key2, v7_TW, v7_PTX, v7_CTX}
	,
	// Vector 8
	{sizeof(v8_CTX), v8_key1, v8_key2, v8_TW, v8_PTX, v8_CTX}
	,
	// Vector 9
	{sizeof(v9_CTX), v9_key1, v9_key2, v9_TW, v9_PTX, v9_CTX}
	,
	// Vector 15
	{sizeof(v15_CTX), v15_key1, v15_key2, v15_TW, v15_PTX, v15_CTX}
	,
	// Vector 16
	{sizeof(v16_CTX), v16_key1, v16_key2, v16_TW, v16_PTX, v16_CTX}
	,
	// Vector 17
	{sizeof(v17_CTX), v17_key1, v17_key2, v17_TW, v17_PTX, v17_CTX}
	,
	// Vector 18
	{sizeof(v18_CTX), v18_key1, v18_key2, v18_TW, v18_PTX, v18_CTX}
	,
	// Vector 19
	{sizeof(v19_CTX), v19_key1, v19_key2, v19_TW, v19_PTX, v19_CTX}

};
